
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Dimensions,
  Alert,
  Modal,
  Animated,
  KeyboardAvoidingView,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import DateTimePicker from '@react-native-community/datetimepicker';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';

const { width } = Dimensions.get('window');

// Date helpers to avoid timezone shifts when parsing/storing plain dates (YYYY-MM-DD)
const parseDateStringToLocalDate = (dateString) => {
  if (!dateString) return new Date();
  const parts = dateString.split('-');
  if (parts.length !== 3) return new Date(dateString);
  const year = parseInt(parts[0], 10);
  const month = parseInt(parts[1], 10) - 1;
  const day = parseInt(parts[2], 10);
  return new Date(year, month, day);
};

const formatDateToISO = (date) => {
  if (!(date instanceof Date) || isNaN(date.getTime())) return '';
  const y = date.getFullYear();
  const m = `${date.getMonth() + 1}`.padStart(2, '0');
  const d = `${date.getDate()}`.padStart(2, '0');
  return `${y}-${m}-${d}`;
};

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, message = 'Processing' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_020Step2 = ({ setActiveStep }) => {
  console.log('HSE_020Step2 component rendered');
  const navigation = useNavigation();
  
  const [formData, setFormData] = useState({
    name_of_person_under_observation: '',
    date: new Date().toISOString().split('T')[0],
    department: '',
    task_under_observation: '',
    type_of_observation: {
      planned_task_observation: false,
      visible_felt_leadership: false
    },
    reason_for_pto_vfl: {
      task_procedure_review: false,
      behavioural: false,
      incident_occurred: false,
      training_follow_up: false,
      new_employee: false,
      equipment_suitability: false
    },
    evaluation_criteria: {
      property_damage_or_personal_injury: {
        yes: false,
        no: false
      },
      efficient_and_productive: {
        yes: false,
        no: false
      },
      comply_with_standards: {
        yes: false,
        no: false
      },
      detrimental_effect_on_quality: {
        yes: false,
        no: false
      }
    },
    commendation_section: {
      description: 'Describe clearly below any practices or conditions related to observation that deserve commendation',
      details: ''
    },
    correction_section: {
      description: 'Describe clearly below any practices or conditions related to observation that deserve correction',
      deviations_corrections: [
        {
          deviation: '',
          correction: ''
        }
      ]
    },
    action_items: {
      employee_complimented_instructed: {
        yes: false,
        no: false
      },
      follow_up_observation_needed: {
        yes: false,
        no: false
      }
    },
    signatures: {
      observed_person: '',
      conducting_observation_and_leadership: ''
    }
  });

  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState(null);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [tempDateValue, setTempDateValue] = useState(new Date());
  const [currentPickerType, setCurrentPickerType] = useState(null); // 'date'

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      setIsLoading(true);
      setError(null);
      const savedData = await AsyncStorage.getItem('hseStep2');
      if (savedData) {
        const data = JSON.parse(savedData);
        // Merge saved data with default structure to ensure all properties exist
        const mergedData = {
          name_of_person_under_observation: data.name_of_person_under_observation || '',
          date: data.date || new Date().toISOString().split('T')[0],
          department: data.department || '',
          task_under_observation: data.task_under_observation || '',
          type_of_observation: {
            planned_task_observation: data.type_of_observation?.planned_task_observation || false,
            visible_felt_leadership: data.type_of_observation?.visible_felt_leadership || false
          },
          reason_for_pto_vfl: {
            task_procedure_review: data.reason_for_pto_vfl?.task_procedure_review || false,
            behavioural: data.reason_for_pto_vfl?.behavioural || false,
            incident_occurred: data.reason_for_pto_vfl?.incident_occurred || false,
            training_follow_up: data.reason_for_pto_vfl?.training_follow_up || false,
            new_employee: data.reason_for_pto_vfl?.new_employee || false,
            equipment_suitability: data.reason_for_pto_vfl?.equipment_suitability || false
          },
          evaluation_criteria: {
            property_damage_or_personal_injury: {
              yes: data.evaluation_criteria?.property_damage_or_personal_injury?.yes || false,
              no: data.evaluation_criteria?.property_damage_or_personal_injury?.no || false
            },
            efficient_and_productive: {
              yes: data.evaluation_criteria?.efficient_and_productive?.yes || false,
              no: data.evaluation_criteria?.efficient_and_productive?.no || false
            },
            comply_with_standards: {
              yes: data.evaluation_criteria?.comply_with_standards?.yes || false,
              no: data.evaluation_criteria?.comply_with_standards?.no || false
            },
            detrimental_effect_on_quality: {
              yes: data.evaluation_criteria?.detrimental_effect_on_quality?.yes || false,
              no: data.evaluation_criteria?.detrimental_effect_on_quality?.no || false
            }
          },
          commendation_section: {
            description: 'Describe clearly below any practices or conditions related to observation that deserve commendation',
            details: data.commendation_section?.details || ''
          },
          correction_section: {
            description: 'Describe clearly below any practices or conditions related to observation that deserve correction',
            deviations_corrections: data.correction_section?.deviations_corrections || [
              {
                deviation: '',
                correction: ''
              }
            ]
          },
          action_items: {
            employee_complimented_instructed: {
              yes: data.action_items?.employee_complimented_instructed?.yes || false,
              no: data.action_items?.employee_complimented_instructed?.no || false
            },
            follow_up_observation_needed: {
              yes: data.action_items?.follow_up_observation_needed?.yes || false,
              no: data.action_items?.follow_up_observation_needed?.no || false
            }
          },
          signatures: {
            observed_person: data.signatures?.observed_person || '',
            conducting_observation_and_leadership: data.signatures?.conducting_observation_and_leadership || ''
          }
        };
        setFormData(mergedData);
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
      setError('Failed to load saved data');
    } finally {
      setIsLoading(false);
    }
  };

  const saveData = async () => {
    try {
      await AsyncStorage.setItem('hseStep2', JSON.stringify(formData));
    } catch (error) {
      console.error('Error saving data:', error);
      Alert.alert('Error', 'Failed to save data');
    }
  };

  const handleInputChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleNestedChange = (parentField, childField, value) => {
    setFormData(prev => ({
      ...prev,
      [parentField]: {
        ...prev[parentField],
        [childField]: value
      }
    }));
  };

  const handleEvaluationChange = (field, option) => {
    setFormData(prev => ({
      ...prev,
      evaluation_criteria: {
        ...prev.evaluation_criteria,
        [field]: {
          yes: option === 'yes',
          no: option === 'no'
        }
      }
    }));
  };

  const handleActionItemChange = (field, option) => {
    setFormData(prev => ({
      ...prev,
      action_items: {
        ...prev.action_items,
        [field]: {
          yes: option === 'yes',
          no: option === 'no'
        }
      }
    }));
  };

  const addDeviationCorrection = () => {
    if (formData.correction_section.deviations_corrections.length < 15) {
      setFormData(prev => ({
        ...prev,
        correction_section: {
          ...prev.correction_section,
          deviations_corrections: [
            ...prev.correction_section.deviations_corrections,
            { deviation: '', correction: '' }
          ]
        }
      }));
    } else {
      Alert.alert('Limit Reached', 'Maximum 15 entries allowed');
    }
  };

  const removeDeviationCorrection = (index) => {
    if (formData.correction_section.deviations_corrections.length > 1) {
      setFormData(prev => ({
        ...prev,
        correction_section: {
          ...prev.correction_section,
          deviations_corrections: prev.correction_section.deviations_corrections.filter((_, i) => i !== index)
        }
      }));
    }
  };

  const updateDeviationCorrection = (index, field, value) => {
    setFormData(prev => ({
      ...prev,
      correction_section: {
        ...prev.correction_section,
        deviations_corrections: prev.correction_section.deviations_corrections.map((item, i) => 
          i === index ? { ...item, [field]: value } : item
        )
      }
    }));
  };

  const validateForm = () => {
    // Check required fields
    if (!formData.name_of_person_under_observation.trim()) {
      Alert.alert('Required Field', 'Please enter the name of person under observation.');
      return false;
    }
    
    if (!formData.date.trim()) {
      Alert.alert('Required Field', 'Please enter the date.');
      return false;
    }
    
    if (!formData.department.trim()) {
      Alert.alert('Required Field', 'Please enter the department.');
      return false;
    }
    
    if (!formData.task_under_observation.trim()) {
      Alert.alert('Required Field', 'Please enter the task under observation.');
      return false;
    }
    
    // Check if at least one type of observation is selected
    if (!formData.type_of_observation.planned_task_observation && !formData.type_of_observation.visible_felt_leadership) {
      Alert.alert('Required Field', 'Please select at least one type of observation.');
      return false;
    }
    
    // Check if at least one reason for PTO/VFL is selected
    const reasons = formData.reason_for_pto_vfl;
    const hasReason = reasons.task_procedure_review || reasons.behavioural || reasons.incident_occurred || 
                     reasons.training_follow_up || reasons.new_employee || reasons.equipment_suitability;
    if (!hasReason) {
      Alert.alert('Required Field', 'Please select at least one reason for PTO/VFL.');
      return false;
    }
    
    // Check if all evaluation criteria are answered
    const evaluationCriteria = formData.evaluation_criteria;
    const criteriaFields = ['property_damage_or_personal_injury', 'efficient_and_productive', 'comply_with_standards', 'detrimental_effect_on_quality'];
    for (const field of criteriaFields) {
      if (!evaluationCriteria[field].yes && !evaluationCriteria[field].no) {
        Alert.alert('Required Field', `Please answer the evaluation criteria: ${field.replace(/_/g, ' ')}`);
        return false;
      }
    }
    
    // Check if action items are answered
    const actionItems = formData.action_items;
    if (!actionItems.employee_complimented_instructed.yes && !actionItems.employee_complimented_instructed.no) {
      Alert.alert('Required Field', 'Please answer whether the employee has been complimented/instructed.');
      return false;
    }
    
    if (!actionItems.follow_up_observation_needed.yes && !actionItems.follow_up_observation_needed.no) {
      Alert.alert('Required Field', 'Please answer whether a follow-up observation is needed.');
      return false;
    }
    
    return true;
  };

  const handleNext = async () => {
    if (!validateForm()) {
      return;
    }
    
    try {
      await saveData();
      if (setActiveStep && typeof setActiveStep === 'function') {
        setActiveStep(3);
      } else {
        console.warn('setActiveStep is not available');
        navigation.navigate('HSEInspectionHome');
      }
    } catch (error) {
      console.error('Error in handleNext:', error);
      Alert.alert('Error', 'Failed to proceed to next step');
    }
  };

  const handleBack = () => {
    try {
      if (setActiveStep && typeof setActiveStep === 'function') {
        setActiveStep(1);
      } else {
        console.warn('setActiveStep is not available');
        navigation.goBack();
      }
    } catch (error) {
      console.error('Error in handleBack:', error);
      navigation.goBack();
    }
  };

  const renderCheckbox = (label, checked, onPress) => (
    <TouchableOpacity style={styles.checkboxContainer} onPress={onPress}>
      <View style={[styles.checkbox, checked && styles.checkboxChecked]}>
        {checked && <MaterialIcons name="check" size={16} color="#fff" />}
      </View>
      <Text style={styles.checkboxLabel}>{label}</Text>
    </TouchableOpacity>
  );

  const renderEvaluationQuestion = (question, field) => (
    <View style={styles.evaluationQuestion}>
      <Text style={styles.questionText}>{question}</Text>
      <View style={styles.yesNoContainer}>
        <TouchableOpacity
          style={[
            styles.yesNoButton,
            formData.evaluation_criteria?.[field]?.yes && styles.yesNoButtonSelected
          ]}
          onPress={() => handleEvaluationChange(field, 'yes')}
        >
          <Text style={[
            styles.yesNoButtonText,
            formData.evaluation_criteria?.[field]?.yes && styles.yesNoButtonTextSelected
          ]}>YES</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[
            styles.yesNoButton,
            formData.evaluation_criteria?.[field]?.no && styles.yesNoButtonSelected
          ]}
          onPress={() => handleEvaluationChange(field, 'no')}
        >
          <Text style={[
            styles.yesNoButtonText,
            formData.evaluation_criteria?.[field]?.no && styles.yesNoButtonTextSelected
          ]}>NO</Text>
        </TouchableOpacity>
      </View>
    </View>
  );

  // Show loading state
  if (isLoading) {
    return (
      <View style={styles.container}>
        <SyncModal visible={isLoading} message="Loading Form" />
      </View>
    );
  }

  // Show error state
  if (error) {
    return (
      <View style={styles.errorContainer}>
        <Text style={styles.errorText}>{error}</Text>
        <TouchableOpacity style={styles.retryButton} onPress={loadSavedData}>
          <Text style={styles.retryButtonText}>Retry</Text>
        </TouchableOpacity>
      </View>
    );
  }

  console.log('Rendering HSE_020Step2 form');

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>

        {/* Observation Details */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Observation Details</Text>
          
          <View style={styles.inputContainer}>
            <Text style={styles.label}>Name of Person Under Observation *</Text>
            <TextInput
              style={styles.textInput}
              value={formData.name_of_person_under_observation}
              onChangeText={(text) => handleInputChange('name_of_person_under_observation', text)}
              placeholder="Enter name"
            />
          </View>

          <View style={styles.inputContainer}>
            <Text style={styles.label}>Date *</Text>
            <TouchableOpacity onPress={() => {
              // parse stored YYYY-MM-DD to a local Date to avoid timezone shifts on iOS
              const parsed = parseDateStringToLocalDate(formData.date);
              setTempDateValue(parsed instanceof Date && !isNaN(parsed.getTime()) ? parsed : new Date());
              setCurrentPickerType('date');
              setShowDatePicker(true);
            }} activeOpacity={0.8}>
              <View pointerEvents="none">
                <TextInput
                  style={styles.textInput}
                  value={formData.date}
                  editable={false}
                  placeholder="YYYY-MM-DD"
                />
              </View>
            </TouchableOpacity>

            {/* iOS modal picker */}
            {Platform.OS === 'ios' && showDatePicker && currentPickerType === 'date' && (
              <Modal
                visible={showDatePicker}
                animationType="slide"
                transparent={true}
                onRequestClose={() => {
                  setShowDatePicker(false);
                  setCurrentPickerType(null);
                }}
                presentationStyle="overFullScreen"
              >
                <View style={styles.iosDatePickerBackground}>
                  <View style={styles.iosDatePickerContainer}>
                    <View style={styles.iosDatePickerHeader}>
                      <TouchableOpacity onPress={() => { setShowDatePicker(false); setCurrentPickerType(null); }}>
                        <Text style={styles.iosDatePickerCancelText}>Cancel</Text>
                      </TouchableOpacity>
                      <Text style={styles.iosDatePickerTitle}>Select Date</Text>
                      <TouchableOpacity onPress={() => {
                        // commit temp value (format to YYYY-MM-DD using local date parts)
                        try {
                          if (tempDateValue && tempDateValue instanceof Date && !isNaN(tempDateValue.getTime())) {
                            const iso = formatDateToISO(tempDateValue);
                            handleInputChange('date', iso);
                          }
                        } catch (e) {
                          console.error('Error setting iOS date:', e);
                        }
                        setShowDatePicker(false);
                        setCurrentPickerType(null);
                      }}>
                        <Text style={styles.iosDatePickerDoneText}>Done</Text>
                      </TouchableOpacity>
                    </View>
                    <View style={styles.iosDatePickerContent}>
                      <DateTimePicker
                        value={tempDateValue instanceof Date && !isNaN(tempDateValue.getTime()) ? tempDateValue : new Date()}
                        mode="date"
                        display="spinner"
                        onChange={(event, selectedDate) => {
                          // On iOS the picker returns a Date in local timezone already - store locally in temp until Done
                          if (selectedDate && selectedDate instanceof Date && !isNaN(selectedDate.getTime())) {
                            setTempDateValue(selectedDate);
                          }
                        }}
                        textColor="#333"
                      />
                    </View>
                  </View>
                </View>
              </Modal>
            )}

            {/* Android inline picker */}
            {Platform.OS === 'android' && showDatePicker && (
              <DateTimePicker
                // parse stored YYYY-MM-DD into local Date for Android as well
                value={parseDateStringToLocalDate(formData.date)}
                mode="date"
                display="default"
                onChange={(event, selectedDate) => {
                  setShowDatePicker(false);
                  if (selectedDate) {
                    const isoDate = formatDateToISO(selectedDate);
                    handleInputChange('date', isoDate);
                  }
                }}
              />
            )}
          </View>

          <View style={styles.inputContainer}>
            <Text style={styles.label}>Department *</Text>
            <TextInput
              style={styles.textInput}
              value={formData.department}
              onChangeText={(text) => handleInputChange('department', text)}
              placeholder="Enter department"
            />
          </View>

          <View style={styles.inputContainer}>
            <Text style={styles.label}>Task Under Observation *</Text>
            <TextInput
              style={styles.textInput}
              value={formData.task_under_observation}
              onChangeText={(text) => handleInputChange('task_under_observation', text)}
              placeholder="Describe the task"
              multiline
            />
          </View>

          <Text style={styles.label}>Type of Observation *</Text>
          {renderCheckbox(
            'Planned Task Observation',
            formData.type_of_observation?.planned_task_observation || false,
            () => handleNestedChange('type_of_observation', 'planned_task_observation', !(formData.type_of_observation?.planned_task_observation || false))
          )}
          {renderCheckbox(
            'Visible Felt Leadership',
            formData.type_of_observation?.visible_felt_leadership || false,
            () => handleNestedChange('type_of_observation', 'visible_felt_leadership', !(formData.type_of_observation?.visible_felt_leadership || false))
          )}
        </View>

        {/* Reason for PTO/VFL */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Reason for PTO/VFL *</Text>
          {renderCheckbox(
            'TASK PROCEDURE REVIEW',
            formData.reason_for_pto_vfl?.task_procedure_review || false,
            () => handleNestedChange('reason_for_pto_vfl', 'task_procedure_review', !(formData.reason_for_pto_vfl?.task_procedure_review || false))
          )}
          {renderCheckbox(
            'BEHAVIOURAL',
            formData.reason_for_pto_vfl?.behavioural || false,
            () => handleNestedChange('reason_for_pto_vfl', 'behavioural', !(formData.reason_for_pto_vfl?.behavioural || false))
          )}
          {renderCheckbox(
            'INCIDENT OCCURRED',
            formData.reason_for_pto_vfl?.incident_occurred || false,
            () => handleNestedChange('reason_for_pto_vfl', 'incident_occurred', !(formData.reason_for_pto_vfl?.incident_occurred || false))
          )}
          {renderCheckbox(
            'TRAINING FOLLOW UP',
            formData.reason_for_pto_vfl?.training_follow_up || false,
            () => handleNestedChange('reason_for_pto_vfl', 'training_follow_up', !(formData.reason_for_pto_vfl?.training_follow_up || false))
          )}
          {renderCheckbox(
            'NEW EMPLOYEE',
            formData.reason_for_pto_vfl?.new_employee || false,
            () => handleNestedChange('reason_for_pto_vfl', 'new_employee', !(formData.reason_for_pto_vfl?.new_employee || false))
          )}
          {renderCheckbox(
            'EQUIPMENT SUITABILITY',
            formData.reason_for_pto_vfl?.equipment_suitability || false,
            () => handleNestedChange('reason_for_pto_vfl', 'equipment_suitability', !(formData.reason_for_pto_vfl?.equipment_suitability || false))
          )}
        </View>

        {/* Evaluation Criteria */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Evaluation Criteria *</Text>
          {renderEvaluationQuestion(
            'Could any of the practices or conditions observed result in property damage or personal injury?',
            'property_damage_or_personal_injury'
          )}
          {renderEvaluationQuestion(
            'Are the methods and practices observed the most efficient and productive for the task under observation?',
            'efficient_and_productive'
          )}
          {renderEvaluationQuestion(
            'Did the practices observed comply with all of the applicable standards that exist for the task or job?',
            'comply_with_standards'
          )}
          {renderEvaluationQuestion(
            'Could any of the practices observed have a detrimental effect upon the quality of the work undertaken?',
            'detrimental_effect_on_quality'
          )}
        </View>

        {/* Commendation Section */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Commendation Section</Text>
          <Text style={styles.descriptionText}>
            {formData.commendation_section.description}
          </Text>
          <TextInput
            style={[styles.textInput, styles.textArea]}
            value={formData.commendation_section.details}
            onChangeText={(text) => handleNestedChange('commendation_section', 'details', text)}
            placeholder="Enter commendation details..."
            multiline
            numberOfLines={5}
          />
        </View>

        {/* Correction Section */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Correction Section</Text>
          <Text style={styles.descriptionText}>
            {formData.correction_section.description}
          </Text>
          
          {formData.correction_section.deviations_corrections.map((item, index) => (
            <View key={index} style={styles.deviationContainer}>
              <View style={styles.deviationRow}>
                <Text style={styles.columnHeader}>Deviation</Text>
                {formData.correction_section.deviations_corrections.length > 1 && (
                  <TouchableOpacity 
                    style={styles.removeButton}
                    onPress={() => removeDeviationCorrection(index)}
                  >
                    <MaterialIcons name="delete" size={20} color="#dc3545" />
                  </TouchableOpacity>
                )}
              </View>
              <TextInput
                style={[styles.textInput, styles.textArea]}
                value={item.deviation}
                onChangeText={(text) => updateDeviationCorrection(index, 'deviation', text)}
                placeholder="Describe the deviation..."
                multiline
                numberOfLines={3}
              />
              
              <View style={styles.correctionRow}>
                <Text style={styles.columnHeader}>Correction</Text>
              </View>
              <TextInput
                style={[styles.textInput, styles.textArea]}
                value={item.correction}
                onChangeText={(text) => updateDeviationCorrection(index, 'correction', text)}
                placeholder="Describe the correction..."
                multiline
                numberOfLines={3}
              />
            </View>
          ))}
          
          <TouchableOpacity style={styles.addButton} onPress={addDeviationCorrection}>
            <MaterialIcons name="add" size={20} color="#fff" />
            <Text style={styles.addButtonText}>Add More</Text>
          </TouchableOpacity>
        </View>

        {/* Action Items */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Follow-up Actions *</Text>
          
          <View style={styles.actionItemsContainer}>
            <View style={styles.actionItem}>
              <Text style={styles.questionText}>Has the employee been complimented / instructed based on the observations:</Text>
              <View style={styles.yesNoContainer}>
                <TouchableOpacity
                  style={[
                    styles.yesNoButton,
                    formData.action_items?.employee_complimented_instructed?.yes && styles.yesNoButtonSelected
                  ]}
                  onPress={() => handleActionItemChange('employee_complimented_instructed', 'yes')}
                >
                  <Text style={[
                    styles.yesNoButtonText,
                    formData.action_items?.employee_complimented_instructed?.yes && styles.yesNoButtonTextSelected
                  ]}>YES</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[
                    styles.yesNoButton,
                    formData.action_items?.employee_complimented_instructed?.no && styles.yesNoButtonSelected
                  ]}
                  onPress={() => handleActionItemChange('employee_complimented_instructed', 'no')}
                >
                  <Text style={[
                    styles.yesNoButtonText,
                    formData.action_items?.employee_complimented_instructed?.no && styles.yesNoButtonTextSelected
                  ]}>NO</Text>
                </TouchableOpacity>
              </View>
            </View>

            <View style={styles.actionItem}>
              <Text style={styles.questionText}>Should a follow up observation be conducted on the person(s) conducting this particular task:</Text>
              <View style={styles.yesNoContainer}>
                <TouchableOpacity
                  style={[
                    styles.yesNoButton,
                    formData.action_items?.follow_up_observation_needed?.yes && styles.yesNoButtonSelected
                  ]}
                  onPress={() => handleActionItemChange('follow_up_observation_needed', 'yes')}
                >
                  <Text style={[
                    styles.yesNoButtonText,
                    formData.action_items?.follow_up_observation_needed?.yes && styles.yesNoButtonTextSelected
                  ]}>YES</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[
                    styles.yesNoButton,
                    formData.action_items?.follow_up_observation_needed?.no && styles.yesNoButtonSelected
                  ]}
                  onPress={() => handleActionItemChange('follow_up_observation_needed', 'no')}
                >
                  <Text style={[
                    styles.yesNoButtonText,
                    formData.action_items?.follow_up_observation_needed?.no && styles.yesNoButtonTextSelected
                  ]}>NO</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </View>


      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={handleNext}
        >
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      
      {/* SyncModal for Loading */}
      <SyncModal visible={isLoading} message="Loading Form" />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  header: {
    backgroundColor: '#015185',
    padding: 20,
    alignItems: 'center',
  },
  headerTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#fff',
    marginBottom: 5,
  },
  headerSubtitle: {
    fontSize: 16,
    color: '#fff',
    opacity: 0.8,
  },
  section: {
    marginBottom: width * 0.04,
    paddingHorizontal: width * 0.04,
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 8,
  },
  inputContainer: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  textArea: {
    minHeight: 80,
    textAlignVertical: 'top',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 0,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '60%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
  },
  errorContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
    padding: 20,
  },
  errorText: {
    fontSize: 16,
    color: '#dc3545',
    textAlign: 'center',
    marginBottom: 20,
  },
  retryButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 25,
    borderRadius: 5,
  },
  retryButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  checkboxContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 10,
  },
  checkbox: {
    width: 20,
    height: 20,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 4,
    justifyContent: 'center',
    alignItems: 'center',
    marginRight: 10,
    backgroundColor: '#fff',
  },
  checkboxChecked: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  checkboxLabel: {
    fontSize: 16,
    color: '#333',
  },
  evaluationQuestion: {
    marginBottom: 15,
  },
  questionText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  yesNoContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    backgroundColor: '#f0f0f0',
    borderRadius: 5,
    padding: 5,
  },
  yesNoButton: {
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
  },
  yesNoButtonSelected: {
    backgroundColor: '#015185',
  },
  yesNoButtonText: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
  },
  yesNoButtonTextSelected: {
    color: '#fff',
  },

  deviationContainer: {
    marginBottom: 15,
    backgroundColor: '#f9f9f9',
    borderRadius: 5,
    padding: 10,
    borderWidth: 1,
    borderColor: '#eee',
  },
  deviationRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 5,
  },
  correctionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginTop: 10,
    marginBottom: 5,
  },
  columnHeader: {
    fontSize: 14,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 5,
  },
  removeButton: {
    padding: 5,
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 5,
    marginTop: 10,
  },
  addButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 5,
  },
  actionItemsContainer: {
    marginTop: 10,
  },
  actionItem: {
    marginBottom: 15,
  },
  descriptionText: {
    fontSize: 14,
    color: '#666',
    marginBottom: 10,
    fontStyle: 'italic',
  },
  iosDatePickerBackground: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  iosDatePickerContainer: {
    width: '90%',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    overflow: 'hidden',
  },
  iosDatePickerHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: width * 0.03,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    backgroundColor: '#f8f9fa',
  },
  iosDatePickerCancelText: {
    color: '#015185',
    fontSize: width * 0.037,
    fontWeight: '600',
  },
  iosDatePickerTitle: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    flex: 1,
  },
  iosDatePickerDoneText: {
    color: '#015185',
    fontSize: width * 0.037,
    fontWeight: '600',
  },
  iosDatePickerContent: {
    padding: width * 0.04,
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default HSE_020Step2;