/*
  Batch update Firebase Authentication passwords using Admin SDK.
  - Sets each user's password to their employeeNumber from Userdatajson.json
  - Updates existing Auth users (by email) or creates them if missing

  Usage:
    1) Install deps in project root:
       npm init -y && npm i firebase-admin
    2) Create a Firebase Service Account key (JSON) and set env var:
       export GOOGLE_APPLICATION_CREDENTIALS="/absolute/path/to/serviceAccountKey.json"
    3) Run:
       node scripts/update-auth-passwords.js
*/

const path = require('path');
const admin = require('firebase-admin');

// Initialize Admin SDK (uses GOOGLE_APPLICATION_CREDENTIALS)
admin.initializeApp({
  credential: admin.credential.applicationDefault(),
});

// Load admins JSON (relative to repo root)
const dataPath = path.resolve(
  __dirname,
  '../components/MainHome/AForTesting/Uploaduserdata/Userdatajson.json'
);

/** @type {{ admins: Record<string, any> }} */
const payload = require(dataPath);

async function upsertUserByEmail(email, displayName, password) {
  try {
    const userRecord = await admin.auth().getUserByEmail(email);
    await admin.auth().updateUser(userRecord.uid, { password, displayName });
    return { status: 'updated', uid: userRecord.uid };
  } catch (err) {
    if (err && err.code === 'auth/user-not-found') {
      const created = await admin.auth().createUser({ email, password, displayName });
      return { status: 'created', uid: created.uid };
    }
    throw err;
  }
}

async function main() {
  if (!payload || !payload.admins || typeof payload.admins !== 'object') {
    console.error('Invalid JSON: expected { admins: { ... } }');
    process.exit(1);
  }

  const entries = Object.entries(payload.admins);
  console.log(`Processing ${entries.length} admins...`);

  let updated = 0;
  let created = 0;
  let skipped = 0;
  const errors = [];

  for (const [key, value] of entries) {
    const email = (value && value.email && String(value.email).trim()) || '';
    const employeeNumber = (value && value.employeeNumber && String(value.employeeNumber).trim()) || '';
    const displayName = (value && value.name) || employeeNumber || '';

    if (!email || !employeeNumber) {
      skipped++;
      console.log(`SKIP [${key}]: Missing email or employeeNumber`);
      continue;
    }

    try {
      const res = await upsertUserByEmail(email, displayName, employeeNumber);
      if (res.status === 'updated') {
        updated++;
        console.log(`UPDATED [${key}] -> uid=${res.uid}`);
      } else if (res.status === 'created') {
        created++;
        console.log(`CREATED [${key}] -> uid=${res.uid}`);
      }
    } catch (e) {
      errors.push({ key, email, message: e?.message || String(e) });
      console.error(`ERROR [${key}] email=${email}:`, e?.message || e);
    }
  }

  console.log('Done. Summary:');
  console.log({ updated, created, skipped, errors: errors.length });
  if (errors.length) {
    console.log('Errors detail:', errors);
  }
  process.exit(0);
}

main().catch((e) => {
  console.error('Fatal error:', e);
  process.exit(1);
});


