const express = require('express');
const mongoose = require('mongoose');
const cors = require('cors');
const nodemailer = require('nodemailer');
const http = require('http');

const app = express();

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB configuration
const MONGO_HOST = process.env.MONGO_HOST || '168.231.113.219';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'Titandrilling_@2024';
const MONGO_DB = process.env.MONGO_DB || 'titan_drilling';

// MongoDB connection - use existing connection if available
if (mongoose.connection.readyState === 0) {
  const mongoURI = `mongodb://${MONGO_USER}:${MONGO_PASSWORD}@${MONGO_HOST}:${MONGO_PORT}/${MONGO_DB}?authSource=admin`;

  mongoose.connect(mongoURI, {
    useNewUrlParser: true,
    useUnifiedTopology: true,
  })
  .then(() => {
    console.log('✅ MongoDB connected successfully for Inspection Notifications');
  })
  .catch((err) => {
    console.error('❌ MongoDB connection error:', err);
    process.exit(1);
  });
} else {
  console.log('✅ Inspection Notifications: Using existing MongoDB connection');
}

// GlobalUser Schema
const GlobalUserSchema = new mongoose.Schema({
  _id: String,
  name: String,
  email: String,
  role: String,
  countries: [String],
  country: String,
  project: String,
  fcmTokens: [{
    token: String,
    device: String,
    updatedAt: Date
  }],
  fcmToken: String,
}, { strict: false, collection: 'GlobalUsers' });

// Check if model exists to prevent OverwriteModelError
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [InspectionNotifications] Using existing GlobalUser model');
} catch (error) {
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [InspectionNotifications] Created new GlobalUser model');
}

// Email transporter configuration
const emailTransporter = nodemailer.createTransport({
  service: 'gmail',
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'sonia.owninfotech@gmail.com',
    pass: 'wvpv dmsv ytxa vdgd',
  },
});

// Helper function to send push notifications
const sendPushNotifications = async (inspectionData, createdByName, recipients) => {
  try {
    console.log('📱 Sending push notifications for inspection report...');
    
    const notificationData = {
      userIds: recipients.map(r => r._id),
      title: '📋 New Inspection Report',
      body: `${inspectionData.requestNumber}\nEquipment: ${inspectionData.equipmentName || 'N/A'}\nProject: ${inspectionData.project || 'N/A'}\n\nInspected by: ${createdByName}`,
      data: {
        type: 'inspection_report',
        action: 'view_inspection',
        inspectionId: inspectionData._id || inspectionData.id,
        requestNumber: inspectionData.requestNumber,
        equipmentName: inspectionData.equipmentName || '',
        project: inspectionData.project || '',
        country: inspectionData.country || '',
        inspector: createdByName,
        timestamp: new Date().toISOString(),
      },
      // iOS specific settings
      apns: {
        payload: {
          aps: {
            alert: {
              title: '📋 New Inspection Report',
              body: `${inspectionData.requestNumber}\nEquipment: ${inspectionData.equipmentName || 'N/A'}\n\nInspected by: ${createdByName}`
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'inspection_notifications',
        }
      }
    };
    
    // Send push via PushNotificationHandler (port 5016)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(notificationData);
      const options = {
        hostname: '168.231.113.219',
        port: 5016,
        path: '/push-notifications/send-to-users',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Push notifications sent to ${result.successCount || 0} device(s)`);
    return result;
  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to save notifications to MongoDB
const saveNotificationsToMongoDB = async (inspectionData, createdByName, recipients) => {
  try {
    console.log('💾 Saving inspection notifications to MongoDB...');
    
    const notifications = recipients.map(recipient => ({
      notificationId: `inspection_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${recipient._id}`,
      userId: recipient._id,
      title: '📋 New Inspection Report',
      body: `${inspectionData.requestNumber}\nEquipment: ${inspectionData.equipmentName || 'N/A'}\nProject: ${inspectionData.project || 'N/A'}\n\nInspected by: ${createdByName}`,
      type: 'inspection_report',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'inspection_report',
        action: 'view_inspection',
        inspectionId: inspectionData._id || inspectionData.id,
        requestNumber: inspectionData.requestNumber,
        equipmentName: inspectionData.equipmentName || '',
        project: inspectionData.project || '',
        country: inspectionData.country || '',
        inspector: createdByName,
        faultyItems: inspectionData.faultyItems || 0,
        timestamp: new Date().toISOString(),
      }
    }));
    
    // Save to MongoDB via Notifications API (port 5017)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: '168.231.113.219',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Saved ${result.count || notifications.length} notification(s) to MongoDB`);
    return result;
  } catch (error) {
    console.error('❌ Error saving notifications to MongoDB:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send email to inspector with PDF attachment
const sendInspectorEmail = async (inspectionData, createdByName, inspectorEmail, pdfUrl) => {
  try {
    if (!inspectorEmail) {
      console.log('⚠️ No email address for inspector, skipping inspector email');
      return { success: false, error: 'No email address' };
    }
    
    console.log(`📧 Sending confirmation email to inspector: ${createdByName} (${inspectorEmail})`);
    
    // Prepare attachments if PDF URL is provided
    const attachments = [];
    if (pdfUrl) {
      try {
        // Download PDF from URL and attach
        const https = require('https');
        const http = require('http');
        
        const pdfBuffer = await new Promise((resolve, reject) => {
          const protocol = pdfUrl.startsWith('https') ? https : http;
          protocol.get(pdfUrl, (response) => {
            const chunks = [];
            response.on('data', (chunk) => chunks.push(chunk));
            response.on('end', () => resolve(Buffer.concat(chunks)));
            response.on('error', reject);
          }).on('error', reject);
        });
        
        attachments.push({
          filename: `Inspection_Report_${inspectionData.requestNumber}.pdf`,
          content: pdfBuffer,
          contentType: 'application/pdf'
        });
        
        console.log(`📎 PDF attachment prepared for inspector: ${inspectionData.requestNumber}.pdf`);
      } catch (pdfError) {
        console.error('⚠️ Failed to download PDF for inspector email:', pdfError.message);
        // Continue without attachment
      }
    }
    
    const emailSubject = `Inspection Report Submitted: ${inspectionData.requestNumber}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Inspection Report Submitted - Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">✅ Inspection Report Submitted Successfully</h2>

      <p style="line-height: 1.6; margin-bottom: 10px;">Dear ${createdByName},</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">Your inspection report has been successfully submitted and saved in the system.</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Request Number:</th>
            <td style="padding: 10px; text-align: left;">${inspectionData.requestNumber}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Equipment:</th>
            <td style="padding: 10px; text-align: left;">${inspectionData.equipmentName || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Country:</th>
            <td style="padding: 10px; text-align: left;">${inspectionData.country || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Project:</th>
            <td style="padding: 10px; text-align: left;">${inspectionData.project || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Inspector:</th>
            <td style="padding: 10px; text-align: left;">${createdByName}</td>
          </tr>
          ${inspectionData.faultyItems > 0 ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Faulty Items:</th>
            <td style="padding: 10px; text-align: left;"><strong style="color:#d9534f;">${inspectionData.faultyItems}</strong></td>
          </tr>` : ''}
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Date:</th>
            <td style="padding: 10px; text-align: left;">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        Thank you for completing this inspection. The report has been sent to all relevant supervisors and managers.
      </p>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    const mailOptions = {
      from: '"Titan Drilling" <mandeep.owninfotech@gmail.com>',
      to: inspectorEmail,
      subject: emailSubject,
      html: emailHtml,
    };
    
    // Add PDF attachment if available
    if (attachments.length > 0) {
      mailOptions.attachments = attachments;
    }
    
    await emailTransporter.sendMail(mailOptions);
    
    console.log(`✅ Confirmation email sent to inspector ${createdByName} (${inspectorEmail})${attachments.length > 0 ? ' with PDF' : ''}`);
    return { success: true };
  } catch (error) {
    console.error('❌ Error sending inspector email:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send emails with PDF attachment
const sendEmails = async (inspectionData, createdByName, recipients, pdfUrl) => {
  try {
    console.log('📧 Sending email notifications for inspection report...');
    
    let successCount = 0;
    let failCount = 0;
    
    // Prepare attachments if PDF URL is provided
    const attachments = [];
    if (pdfUrl) {
      try {
        // Download PDF from URL and attach
        const https = require('https');
        const http = require('http');
        
        const pdfBuffer = await new Promise((resolve, reject) => {
          const protocol = pdfUrl.startsWith('https') ? https : http;
          protocol.get(pdfUrl, (response) => {
            const chunks = [];
            response.on('data', (chunk) => chunks.push(chunk));
            response.on('end', () => resolve(Buffer.concat(chunks)));
            response.on('error', reject);
          }).on('error', reject);
        });
        
        attachments.push({
          filename: `Inspection_Report_${inspectionData.requestNumber}.pdf`,
          content: pdfBuffer,
          contentType: 'application/pdf'
        });
        
        console.log(`📎 PDF attachment prepared: ${inspectionData.requestNumber}.pdf`);
      } catch (pdfError) {
        console.error('⚠️ Failed to download PDF for attachment:', pdfError.message);
        // Continue without attachment
      }
    }
    
    // Email template
    const emailSubject = `New Inspection Report: ${inspectionData.requestNumber}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>New Inspection Report - Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">📋 New Inspection Report</h2>

      <p style="line-height: 1.6; margin-bottom: 10px;">Dear Recipient,</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">A new inspection report has been submitted in the Titan Drilling system.</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Request Number:</th>
            <td style="padding: 10px; text-align: left;">${inspectionData.requestNumber}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Equipment:</th>
            <td style="padding: 10px; text-align: left;">${inspectionData.equipmentName || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Country:</th>
            <td style="padding: 10px; text-align: left;">${inspectionData.country || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Project:</th>
            <td style="padding: 10px; text-align: left;">${inspectionData.project || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Inspector:</th>
            <td style="padding: 10px; text-align: left;">${createdByName}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Status:</th>
            <td style="padding: 10px; text-align: left;"><strong style="color:#4CAF50;">Completed</strong></td>
          </tr>
          ${inspectionData.faultyItems > 0 ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Faulty Items:</th>
            <td style="padding: 10px; text-align: left;"><strong style="color:#d9534f;">${inspectionData.faultyItems}</strong></td>
          </tr>` : ''}
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Date:</th>
            <td style="padding: 10px; text-align: left;">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        Please review the inspection report in the Titan Drilling application.
      </p>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    // Send email to each recipient
    for (const recipient of recipients) {
      if (recipient.email) {
        try {
          const mailOptions = {
            from: '"Titan Drilling" <mandeep.owninfotech@gmail.com>',
            to: recipient.email,
            subject: emailSubject,
            html: emailHtml,
          };
          
          // Add PDF attachment if available
          if (attachments.length > 0) {
            mailOptions.attachments = attachments;
          }
          
          await emailTransporter.sendMail(mailOptions);
          
          console.log(`✅ Email sent to ${recipient.name} (${recipient.email})${attachments.length > 0 ? ' with PDF' : ''}`);
          successCount++;
        } catch (emailError) {
          console.error(`❌ Failed to send email to ${recipient.name}:`, emailError.message);
          failCount++;
        }
      }
    }
    
    console.log(`📧 Email summary: ${successCount} sent, ${failCount} failed`);
    return { success: true, sent: successCount, failed: failCount, total: recipients.length };
  } catch (error) {
    console.error('❌ Error sending emails:', error);
    return { success: false, error: error.message };
  }
};

/**
 * POST /api/inspection-notifications/new-inspection
 * Handle notifications when a new inspection report is created
 */
app.post('/api/inspection-notifications/new-inspection', async (req, res) => {
  try {
    const { inspectionData, createdByName, inspectorEmail, pdfUrl } = req.body;
    
    if (!inspectionData || !inspectionData.requestNumber) {
      return res.status(400).json({
        success: false,
        error: 'Inspection data is required'
      });
    }
    
    console.log(`📎 PDF URL received: ${pdfUrl || 'Not provided'}`);
    
    console.log(`\n🔔 Processing notifications for inspection: ${inspectionData.requestNumber}`);
    console.log(`📋 Country: ${inspectionData.country}, Project: ${inspectionData.project}`);
    console.log(`👤 Inspector: ${createdByName}`);
    
    // Manager roles to notify
    const managerRoles = [
      'HrManager',
      'Workshop Manager',
      'Maintenance Manager',
      'FinanceManager',
      'Project Manager',
      'Site Manager'
    ];
    
    // Get all ProAdmins (no filtering)
    const proAdmins = await GlobalUser.find({
      role: 'ProAdmin'
    }).select('_id name email fcmToken fcmTokens');
    
    console.log(`📋 Found ${proAdmins.length} ProAdmin(s)`);
    
    // Get Supervisors filtered by country AND project
    // Note: countries and projects are stored as ARRAYS in MongoDB
    const supervisors = await GlobalUser.find({
      role: 'Supervisor',
      countries: inspectionData.country,  // MongoDB matches if array contains value
      projects: inspectionData.project
    }).select('_id name email fcmToken fcmTokens role countries projects');
    
    console.log(`📋 Found ${supervisors.length} Supervisor(s) for ${inspectionData.country} - ${inspectionData.project}`);
    
    // Get Managers filtered by country AND project
    // Note: countries and projects are stored as ARRAYS in MongoDB
    const managers = await GlobalUser.find({
      role: { $in: managerRoles },
      countries: inspectionData.country,  // MongoDB matches if array contains value
      projects: inspectionData.project
    }).select('_id name email fcmToken fcmTokens role countries projects');
    
    console.log(`📋 Found ${managers.length} Manager(s) for ${inspectionData.country} - ${inspectionData.project}`);
    
    // Combine all recipients
    const allRecipients = [...proAdmins, ...supervisors, ...managers];
    
    // Remove duplicates based on _id
    const uniqueRecipients = allRecipients.filter((recipient, index, self) =>
      index === self.findIndex((r) => r._id === recipient._id)
    );
    
    console.log(`📋 Total unique recipients: ${uniqueRecipients.length}`);
    
    if (uniqueRecipients.length === 0) {
      console.log('⚠️ No recipients found');
      return res.json({
        success: true,
        message: 'No recipients to notify',
        pushSent: 0,
        emailsSent: 0,
        notificationsSaved: 0,
      });
    }
    
    // Execute all notification tasks in parallel
    const [mongoResult, pushResult, emailResult, inspectorEmailResult] = await Promise.all([
      saveNotificationsToMongoDB(inspectionData, createdByName, uniqueRecipients),
      sendPushNotifications(inspectionData, createdByName, uniqueRecipients),
      sendEmails(inspectionData, createdByName, uniqueRecipients, pdfUrl),
      sendInspectorEmail(inspectionData, createdByName, inspectorEmail, pdfUrl),
    ]);
    
    // Log results
    console.log('\n📊 Notification Results:');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 Emails to Recipients: ${emailResult.success ? '✅' : '❌'} (${emailResult.sent || 0} sent)`);
    console.log(`  📧 Inspector Email: ${inspectorEmailResult.success ? '✅' : '❌'}\n`);
    
    res.json({
      success: true,
      message: `Notifications sent for inspection: ${inspectionData.requestNumber}`,
      results: {
        mongodb: {
          success: mongoResult.success,
          saved: mongoResult.count || 0,
          error: mongoResult.error || null,
        },
        push: {
          success: pushResult.success,
          sent: pushResult.successCount || 0,
          failed: pushResult.failureCount || 0,
          error: pushResult.error || null,
        },
        emails: {
          success: emailResult.success,
          sent: emailResult.sent || 0,
          failed: emailResult.failed || 0,
          total: emailResult.total || 0,
          error: emailResult.error || null,
        },
        inspectorEmail: {
          success: inspectorEmailResult.success,
          error: inspectorEmailResult.error || null,
        },
      },
      recipients: {
        proAdmins: proAdmins.length,
        supervisors: supervisors.length,
        managers: managers.length,
        total: uniqueRecipients.length,
      }
    });
  } catch (error) {
    console.error('❌ Error processing inspection notifications:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'Inspection Notifications Service',
    mongodb: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
    timestamp: new Date().toISOString(),
  });
});

// Start server
const PORT = process.env.INSPECTION_NOTIFICATIONS_PORT || 5019;
app.listen(PORT, () => {
  console.log(`\n🚀 Inspection Notifications Service running on port ${PORT}`);
  console.log(`📊 MongoDB: titan_drilling`);
  console.log(`🔗 Health check: http://168.231.113.219:${PORT}/health\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  app.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});

module.exports = app;

