const express = require('express');
const mongoose = require('mongoose');
const cors = require('cors');
const nodemailer = require('nodemailer');
const http = require('http');

const app = express();

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB configuration
const MONGO_HOST = process.env.MONGO_HOST || '168.231.113.219';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'Titandrilling_@2024';
const MONGO_DB = process.env.MONGO_DB || 'titan_drilling';

// MongoDB connection - use existing connection if available
if (mongoose.connection.readyState === 0) {
  const mongoURI = `mongodb://${MONGO_USER}:${MONGO_PASSWORD}@${MONGO_HOST}:${MONGO_PORT}/${MONGO_DB}?authSource=admin`;

  mongoose.connect(mongoURI, {
    useNewUrlParser: true,
    useUnifiedTopology: true,
  })
  .then(() => {
    console.log('✅ MongoDB connected successfully for HSE Inspection Notifications');
  })
  .catch((err) => {
    console.error('❌ MongoDB connection error:', err);
    process.exit(1);
  });
} else {
  console.log('✅ HSE Inspection Notifications: Using existing MongoDB connection');
}

// GlobalUser Schema
const GlobalUserSchema = new mongoose.Schema({
  _id: String,
  name: String,
  email: String,
  role: String,
  countries: [String],
  country: String,
  projects: [String],
  project: String,
  fcmTokens: [{
    token: String,
    device: String,
    updatedAt: Date
  }],
  fcmToken: String,
}, { strict: false, collection: 'GlobalUsers' });

// Check if model exists to prevent OverwriteModelError
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [HSEInspectionNotifications] Using existing GlobalUser model');
} catch (error) {
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [HSEInspectionNotifications] Created new GlobalUser model');
}

// Email transporter configuration
const emailTransporter = nodemailer.createTransport({
  service: 'gmail',
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'sonia.owninfotech@gmail.com',
    pass: 'wvpv dmsv ytxa vdgd',
  },
});

// Helper function to send push notifications
const sendPushNotifications = async (hseInspectionData, createdByName, recipients) => {
  try {
    console.log('📱 Sending push notifications for HSE inspection report...');
    
    const notificationData = {
      userIds: recipients.map(r => r._id),
      title: '🏥 New HSE Inspection Report',
      body: `${hseInspectionData.requestNumber}\nChecklist: ${hseInspectionData.checklistName || 'N/A'}\nProject: ${hseInspectionData.project || 'N/A'}\n\nInspected by: ${createdByName}`,
      data: {
        type: 'hse_inspection_report',
        action: 'view_hse_inspection',
        inspectionId: hseInspectionData._id || hseInspectionData.id,
        requestNumber: hseInspectionData.requestNumber,
        checklistName: hseInspectionData.checklistName || '',
        equipmentName: hseInspectionData.equipmentName || '',
        project: hseInspectionData.project || '',
        country: hseInspectionData.country || '',
        inspector: createdByName,
        safePercentage: hseInspectionData.percentageSafe || 0,
        timestamp: new Date().toISOString(),
      },
      // iOS specific settings
      apns: {
        payload: {
          aps: {
            alert: {
              title: '🏥 New HSE Inspection Report',
              body: `${hseInspectionData.requestNumber}\nChecklist: ${hseInspectionData.checklistName || 'N/A'}\n\nInspected by: ${createdByName}`
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'hse_inspection_notifications',
        }
      }
    };
    
    // Send push via PushNotificationHandler (port 5016)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(notificationData);
      const options = {
        hostname: '168.231.113.219',
        port: 5016,
        path: '/push-notifications/send-to-users',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Push notifications sent to ${result.successCount || 0} device(s)`);
    return result;
  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to save notifications to MongoDB
const saveNotificationsToMongoDB = async (hseInspectionData, createdByName, recipients) => {
  try {
    console.log('💾 Saving HSE inspection notifications to MongoDB...');
    
    const notifications = recipients.map(recipient => ({
      notificationId: `hse_insp_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${recipient._id}`,
      userId: recipient._id,
      title: '🏥 New HSE Inspection Report',
      body: `${hseInspectionData.requestNumber}\nChecklist: ${hseInspectionData.checklistName || 'N/A'}\nProject: ${hseInspectionData.project || 'N/A'}\n\nInspected by: ${createdByName}`,
      type: 'hse_inspection_report',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'hse_inspection_report',
        action: 'view_hse_inspection',
        inspectionId: hseInspectionData._id || hseInspectionData.id,
        requestNumber: hseInspectionData.requestNumber,
        checklistName: hseInspectionData.checklistName || '',
        equipmentName: hseInspectionData.equipmentName || '',
        project: hseInspectionData.project || '',
        country: hseInspectionData.country || '',
        inspector: createdByName,
        safePercentage: hseInspectionData.percentageSafe || 0,
        timestamp: new Date().toISOString(),
      }
    }));
    
    // Save to MongoDB via Notifications API (port 5017)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: '168.231.113.219',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Saved ${result.count || notifications.length} notification(s) to MongoDB`);
    return result;
  } catch (error) {
    console.error('❌ Error saving notifications to MongoDB:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send emails
const sendEmails = async (hseInspectionData, createdByName, recipients, pdfUrl) => {
  try {
    console.log('📧 Sending HSE inspection emails...');
    console.log(`📧 Recipients: ${recipients.length}`);
    console.log(`📧 PDF URL: ${pdfUrl}`);
    
    let successCount = 0;
    let failCount = 0;
    
    for (const recipient of recipients) {
      if (recipient.email && recipient.email.includes('@')) {
        try {
          const mailOptions = {
            from: '"Titan Drilling" <sonia.owninfotech@gmail.com>',
            to: recipient.email,
            subject: `New HSE Inspection Report: ${hseInspectionData.requestNumber}`,
            html: `
<!DOCTYPE html>
<html>
<head>
  <style>
    body { font-family: Arial, sans-serif; background-color: #f4f4f4; margin: 0; padding: 20px; }
    .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
    .header { background: linear-gradient(135deg, #FF9800 0%, #F57C00 100%); color: #ffffff; padding: 30px 20px; text-align: center; }
    .header h1 { margin: 0; font-size: 24px; font-weight: bold; }
    .content { padding: 30px 20px; }
    .inspection-info { background-color: #fff3cd; border-left: 4px solid #FF9800; padding: 15px; margin: 20px 0; border-radius: 4px; }
    .info-row { margin: 10px 0; }
    .info-label { font-weight: bold; color: #333; display: inline-block; width: 150px; }
    .info-value { color: #666; }
    .status-badge { display: inline-block; padding: 5px 15px; border-radius: 20px; font-weight: bold; margin: 10px 0; }
    .status-completed { background-color: #4CAF50; color: white; }
    .pdf-button { display: inline-block; background: linear-gradient(135deg, #FF9800 0%, #F57C00 100%); color: white; padding: 12px 30px; text-decoration: none; border-radius: 25px; font-weight: bold; margin: 20px 0; }
    .footer { background-color: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #999; }
    .icon { font-size: 48px; margin-bottom: 10px; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <div class="icon">🏥</div>
      <h1>New HSE Inspection Report</h1>
    </div>
    <div class="content">
      <p>Hello ${recipient.name || 'Team Member'},</p>
      <p>A new HSE inspection report has been submitted and is ready for your review.</p>
      
      <div class="inspection-info">
        <h3 style="margin-top: 0; color: #FF9800;">HSE Inspection Details:</h3>
        <div class="info-row">
          <span class="info-label">Request Number:</span>
          <span class="info-value">${hseInspectionData.requestNumber}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Checklist:</span>
          <span class="info-value">${hseInspectionData.checklistName || 'N/A'}</span>
        </div>
        ${hseInspectionData.equipmentName ? `
        <div class="info-row">
          <span class="info-label">Equipment:</span>
          <span class="info-value">${hseInspectionData.equipmentName}</span>
        </div>` : ''}
        <div class="info-row">
          <span class="info-label">Country:</span>
          <span class="info-value">${hseInspectionData.country || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Project:</span>
          <span class="info-value">${hseInspectionData.project || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Inspected By:</span>
          <span class="info-value">${createdByName}</span>
        </div>
        ${hseInspectionData.percentageSafe !== undefined ? `
        <div class="info-row">
          <span class="info-label">Safety Score:</span>
          <span class="info-value" style="color: ${hseInspectionData.percentageSafe >= 80 ? '#4CAF50' : hseInspectionData.percentageSafe >= 60 ? '#FF9800' : '#F44336'}; font-weight: bold;">${hseInspectionData.percentageSafe}% Safe</span>
        </div>` : ''}
        <div class="info-row">
          <span class="info-label">Date:</span>
          <span class="info-value">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Status:</span>
          <span class="status-badge status-completed">Completed</span>
        </div>
      </div>
      
      ${pdfUrl ? `
      <div style="text-align: center; margin: 30px 0;">
        <a href="${pdfUrl}" class="pdf-button">📄 Download PDF Report</a>
      </div>` : ''}
      
      <p style="color: #666;">Please review this HSE inspection report in the Titan Drilling app or download the PDF above.</p>
    </div>
    <div class="footer">
      <p>Titan Drilling - HSE Management System</p>
      <p>&copy; ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
    </div>
  </div>
</body>
</html>
            `,
          };
          
          await emailTransporter.sendMail(mailOptions);
          console.log(`✅ Email sent to ${recipient.name} (${recipient.email})`);
          successCount++;
        } catch (emailError) {
          console.error(`❌ Failed to send email to ${recipient.name}:`, emailError);
          failCount++;
        }
      }
    }
    
    console.log(`📧 Email Summary: ${successCount} sent, ${failCount} failed`);
    return { success: true, sent: successCount, failed: failCount };
  } catch (error) {
    console.error('❌ Error sending emails:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send confirmation email to inspector
const sendInspectorConfirmationEmail = async (hseInspectionData, inspectorName, inspectorEmail, pdfUrl) => {
  try {
    if (!inspectorEmail || !inspectorEmail.includes('@')) {
      console.log('⚠️ No valid inspector email provided');
      return { success: false, error: 'No valid email' };
    }
    
    console.log(`📧 Sending confirmation email to inspector: ${inspectorEmail}`);
    
    const mailOptions = {
      from: '"Titan Drilling" <sonia.owninfotech@gmail.com>',
      to: inspectorEmail,
      subject: `HSE Inspection Report Submitted: ${hseInspectionData.requestNumber}`,
      html: `
<!DOCTYPE html>
<html>
<head>
  <style>
    body { font-family: Arial, sans-serif; background-color: #f4f4f4; margin: 0; padding: 20px; }
    .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
    .header { background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%); color: #ffffff; padding: 40px 20px; text-align: center; }
    .header h1 { margin: 0; font-size: 28px; font-weight: bold; }
    .success-box { background-color: #f1f8f4; border-left: 4px solid #4CAF50; padding: 20px; margin: 20px 0; border-radius: 4px; }
    .info-box { background-color: #f8f9fa; padding: 15px; margin: 20px 0; border-radius: 4px; }
    .info-row { margin: 8px 0; }
    .info-label { font-weight: bold; color: #333; }
    .info-value { color: #666; }
    .pdf-button { display: inline-block; background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%); color: white; padding: 12px 30px; text-decoration: none; border-radius: 25px; font-weight: bold; margin: 20px 0; }
    .footer { background-color: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #999; }
    .icon { font-size: 64px; margin-bottom: 10px; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <div class="icon">✅</div>
      <h1>HSE Inspection Report Submitted Successfully</h1>
    </div>
    <div class="content" style="padding: 30px 20px;">
      <div class="success-box">
        <h2 style="margin-top: 0; color: #4CAF50;">Submission Confirmed!</h2>
        <p>Your HSE inspection report has been successfully submitted and all relevant team members have been notified.</p>
      </div>
      
      <h3 style="color: #333;">HSE Inspection Summary:</h3>
      <div class="info-box">
        <div class="info-row">
          <span class="info-label">Request Number:</span>
          <span class="info-value">${hseInspectionData.requestNumber}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Checklist:</span>
          <span class="info-value">${hseInspectionData.checklistName || 'N/A'}</span>
        </div>
        ${hseInspectionData.equipmentName ? `
        <div class="info-row">
          <span class="info-label">Equipment:</span>
          <span class="info-value">${hseInspectionData.equipmentName}</span>
        </div>` : ''}
        <div class="info-row">
          <span class="info-label">Country:</span>
          <span class="info-value">${hseInspectionData.country || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Project:</span>
          <span class="info-value">${hseInspectionData.project || 'N/A'}</span>
        </div>
        ${hseInspectionData.percentageSafe !== undefined ? `
        <div class="info-row">
          <span class="info-label">Safety Score:</span>
          <span class="info-value" style="color: ${hseInspectionData.percentageSafe >= 80 ? '#4CAF50' : hseInspectionData.percentageSafe >= 60 ? '#FF9800' : '#F44336'}; font-weight: bold;">${hseInspectionData.percentageSafe}% Safe</span>
        </div>` : ''}
        <div class="info-row">
          <span class="info-label">Submitted:</span>
          <span class="info-value">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</span>
        </div>
      </div>
      
      ${pdfUrl ? `
      <div style="text-align: center; margin: 30px 0;">
        <a href="${pdfUrl}" class="pdf-button">📄 Download Your Report PDF</a>
      </div>` : ''}
      
      <p style="color: #666;">Thank you for maintaining high safety standards. Your supervisors and management team have been notified of this inspection.</p>
    </div>
    <div class="footer">
      <p>Titan Drilling - HSE Management System</p>
      <p>&copy; ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
    </div>
  </div>
</body>
</html>
      `,
    };
    
    await emailTransporter.sendMail(mailOptions);
    console.log(`✅ Confirmation email sent to inspector: ${inspectorEmail}`);
    return { success: true };
  } catch (error) {
    console.error('❌ Error sending inspector confirmation email:', error);
    return { success: false, error: error.message };
  }
};

/**
 * POST /api/hse-inspection-notifications/new-inspection
 * Handle notifications when a new HSE inspection report is created
 */
app.post('/api/hse-inspection-notifications/new-inspection', async (req, res) => {
  try {
    console.log('\n🔔 ========== NEW HSE INSPECTION NOTIFICATION REQUEST ==========');
    console.log('📥 Received payload:', JSON.stringify(req.body, null, 2));
    
    const { hseInspectionData, createdByName, inspectorEmail, pdfUrl } = req.body;
    
    if (!hseInspectionData || !hseInspectionData.requestNumber) {
      console.error('❌ Missing HSE inspection data or request number');
      return res.status(400).json({
        success: false,
        error: 'HSE inspection data with request number is required'
      });
    }
    
    console.log(`\n🔔 Processing notifications for HSE inspection: ${hseInspectionData.requestNumber}`);
    console.log(`📋 Country: ${hseInspectionData.country}, Project: ${hseInspectionData.project}`);
    console.log(`👤 Inspector: ${createdByName}`);
    console.log(`📧 Inspector Email: ${inspectorEmail}`);
    console.log(`📄 PDF URL: ${pdfUrl}`);
    
    // Get all ProAdmins (no filtering)
    const proAdmins = await GlobalUser.find({
      role: 'ProAdmin'
    }).select('_id name email fcmToken fcmTokens');
    
    console.log(`📋 Found ${proAdmins.length} ProAdmin(s)`);
    
    // Get Supervisors filtered by country AND project
    const supervisors = await GlobalUser.find({
      role: 'Supervisor',
      $or: [
        { country: hseInspectionData.country, project: hseInspectionData.project },
        { countries: hseInspectionData.country, projects: hseInspectionData.project }
      ]
    }).select('_id name email fcmToken fcmTokens role country countries project projects');
    
    console.log(`📋 Found ${supervisors.length} Supervisor(s) for ${hseInspectionData.country} - ${hseInspectionData.project}`);
    
    // Get Managers filtered by country AND project
    const managerRoles = ['HrManager', 'Workshop Manager', 'Maintenance Manager', 'FinanceManager', 'Project Manager', 'Site Manager'];
    const managers = await GlobalUser.find({
      role: { $in: managerRoles },
      $or: [
        { country: hseInspectionData.country, project: hseInspectionData.project },
        { countries: hseInspectionData.country, projects: hseInspectionData.project }
      ]
    }).select('_id name email fcmToken fcmTokens role country countries project projects');
    
    console.log(`📋 Found ${managers.length} Manager(s) for ${hseInspectionData.country} - ${hseInspectionData.project}`);
    
    // Combine all recipients
    const allRecipients = [...proAdmins, ...supervisors, ...managers];
    
    // Remove duplicates based on _id
    const uniqueRecipients = allRecipients.filter((recipient, index, self) =>
      index === self.findIndex((r) => r._id === recipient._id)
    );
    
    console.log(`📋 Total unique recipients: ${uniqueRecipients.length}`);
    
    if (uniqueRecipients.length === 0) {
      console.log('⚠️ No recipients found for notifications');
    }
    
    // Execute all notification tasks in parallel
    console.log('🚀 Starting parallel execution of notifications...');
    
    const results = await Promise.all([
      saveNotificationsToMongoDB(hseInspectionData, createdByName, uniqueRecipients),
      sendPushNotifications(hseInspectionData, createdByName, uniqueRecipients),
      sendEmails(hseInspectionData, createdByName, uniqueRecipients, pdfUrl),
      sendInspectorConfirmationEmail(hseInspectionData, createdByName, inspectorEmail, pdfUrl)
    ]);
    
    const [mongoResult, pushResult, emailResult, inspectorEmailResult] = results;
    
    // Log results
    console.log('\n📊 ========== NOTIFICATION RESULTS ==========');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 Emails: ${emailResult.sent || 0} sent, ${emailResult.failed || 0} failed`);
    console.log(`  📧 Inspector Email: ${inspectorEmailResult.success ? '✅' : '❌'}`);
    console.log('========================================\n');
    
    res.json({
      success: true,
      message: `Notifications sent for HSE inspection: ${hseInspectionData.requestNumber}`,
      results: {
        mongodb: mongoResult,
        push: pushResult,
        emails: emailResult,
        inspectorEmail: inspectorEmailResult
      },
      recipients: {
        proAdmins: proAdmins.length,
        supervisors: supervisors.length,
        managers: managers.length,
        total: uniqueRecipients.length
      }
    });
  } catch (error) {
    console.error('❌ Error processing HSE inspection notifications:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'HSE Inspection Notifications Service',
    mongodb: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
    timestamp: new Date().toISOString(),
  });
});

// Start server
const PORT = process.env.HSE_INSPECTION_NOTIFICATIONS_PORT || 5028;
app.listen(PORT, () => {
  console.log(`\n🚀 HSE Inspection Notifications Service running on port ${PORT}`);
  console.log(`📊 MongoDB: titan_drilling`);
  console.log(`🔗 Health check: http://168.231.113.219:${PORT}/health\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  app.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});

module.exports = app;

