const express = require('express');
const mongoose = require('mongoose');
const cors = require('cors');
const nodemailer = require('nodemailer');
const http = require('http');

const app = express();

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB configuration
const MONGO_HOST = process.env.MONGO_HOST || '168.231.113.219';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'Titandrilling_@2024';
const MONGO_DB = process.env.MONGO_DB || 'titan_drilling';

// MongoDB connection - use existing connection if available
if (mongoose.connection.readyState === 0) {
  const mongoURI = `mongodb://${MONGO_USER}:${MONGO_PASSWORD}@${MONGO_HOST}:${MONGO_PORT}/${MONGO_DB}?authSource=admin`;

  mongoose.connect(mongoURI, {
    useNewUrlParser: true,
    useUnifiedTopology: true,
  })
  .then(() => {
    console.log('✅ MongoDB connected successfully for HR Application Notifications');
  })
  .catch((err) => {
    console.error('❌ MongoDB connection error:', err);
    process.exit(1);
  });
} else {
  console.log('✅ HR Application Notifications: Using existing MongoDB connection');
}

// GlobalUser Schema
const GlobalUserSchema = new mongoose.Schema({
  _id: String,
  name: String,
  email: String,
  role: String,
  countries: [String],
  country: String,
  projects: [String],
  project: String,
  fcmTokens: [{
    token: String,
    device: String,
    updatedAt: Date
  }],
  fcmToken: String,
}, { strict: false, collection: 'GlobalUsers' });

// Check if model exists to prevent OverwriteModelError
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [HRApplicationNotifications] Using existing GlobalUser model');
} catch (error) {
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [HRApplicationNotifications] Created new GlobalUser model');
}

// Email transporter configuration with connection pooling
const emailTransporter = nodemailer.createTransport({
  service: 'gmail',
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  pool: true, // Enable connection pooling
  maxConnections: 1, // Use single connection
  maxMessages: 100, // Reuse connection for 100 messages
  rateDelta: 1000, // 1 second between messages
  rateLimit: 5, // Max 5 messages per rateDelta
  auth: {
    user: 'sonia.owninfotech@gmail.com',
    pass: 'wvpv dmsv ytxa vdgd',
  },
  tls: {
    rejectUnauthorized: false
  }
});

// Helper function to get application type display name
const getApplicationTypeDisplay = (type) => {
  const typeMap = {
    'leave_application': 'Leave Application',
    'funeral_grant_form': 'Funeral Grant Form',
    'salary_advance_application': 'Salary Advance Application',
    'application_for_appeal': 'Application for Appeal',
    'grievance_record': 'Grievance Record',
    'hospital_attendance_form': 'Hospital Attendance Form'
  };
  return typeMap[type] || type?.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase()) || 'Application';
};

// Helper function to send push notifications
const sendPushNotifications = async (applicationData, message, recipients) => {
  try {
    console.log('📱 Sending push notifications for HR application...');
    
    const notificationData = {
      userIds: recipients.map(r => r._id),
      title: message.title,
      body: message.body,
      data: {
        type: 'hr_application',
        action: 'view_hr_application',
        applicationId: applicationData._id || applicationData.id,
        applicationNumber: applicationData.applicationNumber,
        applicationType: applicationData.applicationType,
        applicantName: applicationData.fullName,
        project: applicationData.project || '',
        country: applicationData.country || '',
        status: applicationData.status || '',
        timestamp: new Date().toISOString(),
      },
      // iOS specific settings
      apns: {
        payload: {
          aps: {
            alert: {
              title: message.title,
              body: message.body
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'hr_application_notifications',
        }
      }
    };
    
    // Send push via PushNotificationHandler (port 5016)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(notificationData);
      const options = {
        hostname: '168.231.113.219',
        port: 5016,
        path: '/push-notifications/send-to-users',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Push notifications sent to ${result.successCount || 0} device(s)`);
    return result;
  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to save notifications to MongoDB
const saveNotificationsToMongoDB = async (applicationData, message, recipients) => {
  try {
    console.log('💾 ========== SAVING NOTIFICATIONS TO MONGODB ==========');
    console.log(`💾 Recipients count: ${recipients.length}`);
    console.log(`💾 Message:`, message);
    
    const notifications = recipients.map(recipient => ({
      notificationId: `hr_app_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${recipient._id}`,
      userId: recipient._id,
      title: message.title,
      body: message.body,
      type: 'hr_application',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'hr_application',
        action: 'view_hr_application',
        applicationId: applicationData._id || applicationData.id,
        applicationNumber: applicationData.applicationNumber,
        applicationType: applicationData.applicationType,
        applicantName: applicationData.fullName,
        project: applicationData.project || '',
        country: applicationData.country || '',
        status: applicationData.status || '',
        timestamp: new Date().toISOString(),
      }
    }));
    
    console.log(`💾 Created ${notifications.length} notification objects`);
    console.log(`💾 Sample notification:`, notifications[0]);
    
    // Save to MongoDB via Notifications API (port 5017)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: '168.231.113.219',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      console.log(`💾 Sending ${notifications.length} notifications to port 5017...`);
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            console.log(`💾 Response from notification service:`, data);
            const response = JSON.parse(data);
            console.log(`💾 Parsed response:`, response);
            resolve(response);
          } catch (e) {
            console.error('❌ Failed to parse response:', e.message);
            console.error('❌ Raw data:', data);
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => {
        console.error('❌ HTTP request error to notification service:', error.message);
        console.error('❌ Error code:', error.code);
        console.error('❌ Full error:', JSON.stringify(error, null, 2));
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ MongoDB save result:`, result);
    console.log(`✅ Saved ${result.count || notifications.length} notification(s) to MongoDB`);
    return result;
  } catch (error) {
    console.error('❌ Error saving notifications to MongoDB:', error.message);
    console.error('❌ Error stack:', error.stack);
    return { success: false, error: error.message };
  }
};

// Helper function to send emails
const sendEmails = async (applicationData, subject, htmlContent, recipients) => {
  try {
    console.log('📧 ========== SENDING EMAILS ==========');
    console.log(`📧 Recipients count: ${recipients.length}`);
    console.log(`📧 Recipients:`, recipients.map(r => ({ name: r.name, email: r.email })));
    
    let successCount = 0;
    let failCount = 0;
    let noEmailCount = 0;
    
    // Send email to each recipient with delay to prevent rate limiting
    for (let i = 0; i < recipients.length; i++) {
      const recipient = recipients[i];
      console.log(`📧 Processing recipient ${i + 1}/${recipients.length}: ${recipient.name || 'Unknown'}`);
      console.log(`📧   Email: ${recipient.email || 'NO EMAIL'}`);
      
      if (recipient.email && recipient.email.includes('@')) {
        try {
          console.log(`📧   Attempting to send email to ${recipient.email}...`);
          
          const mailOptions = {
            from: '"Titan Drilling" <sonia.owninfotech@gmail.com>',
            to: recipient.email,
            subject: subject,
            html: htmlContent,
          };
          
          const info = await emailTransporter.sendMail(mailOptions);
          
          console.log(`✅   Email sent successfully to ${recipient.name} (${recipient.email})`);
          console.log(`✅   Message ID: ${info.messageId}`);
          successCount++;
          
          // Add small delay between emails to prevent rate limiting (except for last email)
          if (i < recipients.length - 1) {
            await new Promise(resolve => setTimeout(resolve, 200)); // 200ms delay
          }
        } catch (emailError) {
          console.error(`❌   Failed to send email to ${recipient.name}:`, emailError.message);
          console.error(`❌   Error code:`, emailError.code);
          failCount++;
        }
      } else {
        console.log(`⚠️   Skipping ${recipient.name || 'Unknown'} - no valid email`);
        noEmailCount++;
      }
    }
    
    console.log(`📧 ========== EMAIL SUMMARY ==========`);
    console.log(`✅ Sent: ${successCount}`);
    console.log(`❌ Failed: ${failCount}`);
    console.log(`⚠️  No email: ${noEmailCount}`);
    console.log(`📊 Total: ${recipients.length}`);
    
    return { success: true, sent: successCount, failed: failCount, total: recipients.length };
  } catch (error) {
    console.error('❌ Error sending emails:', error);
    console.error('❌ Error stack:', error.stack);
    return { success: false, error: error.message };
  }
};

/**
 * POST /api/hr-notifications/new-application
 * Handle notifications when a new HR application is created
 */
app.post('/api/hr-notifications/new-application', async (req, res) => {
  try {
    console.log('\n🔔 ========== NEW HR APPLICATION NOTIFICATION REQUEST ==========');
    console.log('📥 Received payload:', JSON.stringify(req.body, null, 2));
    
    const { applicationData, createdByName, creatorEmail } = req.body;
    
    if (!applicationData || !applicationData.applicationType) {
      console.error('❌ Missing application data or type');
      return res.status(400).json({
        success: false,
        error: 'Application data is required'
      });
    }
    
    console.log(`\n🔔 Processing notifications for NEW HR application`);
    console.log(`📋 Type: ${applicationData.applicationType}`);
    console.log(`📋 Country: ${applicationData.country}, Project: ${applicationData.project}`);
    console.log(`👤 Creator: ${createdByName}`);
    console.log(`📧 Creator Email: ${creatorEmail}`);
    
    // Get all ProAdmins (no filtering)
    const proAdmins = await GlobalUser.find({
      role: 'ProAdmin'
    }).select('_id name email fcmToken fcmTokens');
    
    console.log(`📋 Found ${proAdmins.length} ProAdmin(s)`);
    
    // Get Supervisors filtered by country AND project
    const supervisors = await GlobalUser.find({
      role: 'Supervisor',
      countries: applicationData.country,
      projects: applicationData.project
    }).select('_id name email fcmToken fcmTokens role countries projects');
    
    console.log(`📋 Found ${supervisors.length} Supervisor(s) for ${applicationData.country} - ${applicationData.project}`);
    
    // Combine all recipients
    const allRecipients = [...proAdmins, ...supervisors];
    
    // Remove duplicates based on _id
    const uniqueRecipients = allRecipients.filter((recipient, index, self) =>
      index === self.findIndex((r) => r._id === recipient._id)
    );
    
    console.log(`📋 Total unique recipients: ${uniqueRecipients.length}`);
    
    if (uniqueRecipients.length === 0) {
      console.log('⚠️ No recipients found');
    }
    
    // Create message
    const message = {
      title: '📝 New HR Application Submitted',
      body: `${getApplicationTypeDisplay(applicationData.applicationType)}\nEmployee: ${applicationData.fullName}\nEmployee #: ${applicationData.employeeNo}\nProject: ${applicationData.project}\n\nSubmitted by: ${createdByName}`
    };
    
    // Email content for recipients
    const recipientEmailHtml = `
<!DOCTYPE html>
<html>
<head>
  <style>
    body { font-family: Arial, sans-serif; background-color: #f4f4f4; margin: 0; padding: 20px; }
    .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
    .header { background: linear-gradient(135deg, #015185 0%, #0078D4 100%); color: #ffffff; padding: 30px 20px; text-align: center; }
    .header h1 { margin: 0; font-size: 24px; font-weight: bold; }
    .content { padding: 30px 20px; }
    .application-info { background-color: #f8f9fa; border-left: 4px solid #015185; padding: 15px; margin: 20px 0; border-radius: 4px; }
    .info-row { margin: 10px 0; }
    .info-label { font-weight: bold; color: #333; display: inline-block; width: 150px; }
    .info-value { color: #666; }
    .footer { background-color: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #999; }
    .icon { font-size: 48px; margin-bottom: 10px; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <div class="icon">📝</div>
      <h1>New HR Application</h1>
    </div>
    <div class="content">
      <p>Hello,</p>
      <p>A new HR application has been submitted and requires your review.</p>
      
      <div class="application-info">
        <h3 style="margin-top: 0; color: #015185;">Application Details:</h3>
        <div class="info-row">
          <span class="info-label">Application Type:</span>
          <span class="info-value">${getApplicationTypeDisplay(applicationData.applicationType)}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Employee Name:</span>
          <span class="info-value">${applicationData.fullName}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Employee Number:</span>
          <span class="info-value">${applicationData.employeeNo}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Department:</span>
          <span class="info-value">${applicationData.department || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Country:</span>
          <span class="info-value">${applicationData.country}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Project:</span>
          <span class="info-value">${applicationData.project}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Date:</span>
          <span class="info-value">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</span>
        </div>
      </div>
      
      <p style="color: #666;">Please review this application in the Titan Drilling app.</p>
    </div>
    <div class="footer">
      <p>Titan Drilling - HR Management System</p>
      <p>&copy; ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
    </div>
  </div>
</body>
</html>
    `;
    
    // Confirmation email to creator
    const creatorEmailHtml = `
<!DOCTYPE html>
<html>
<head>
  <style>
    body { font-family: Arial, sans-serif; background-color: #f4f4f4; margin: 0; padding: 20px; }
    .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
    .header { background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%); color: #ffffff; padding: 40px 20px; text-align: center; }
    .header h1 { margin: 0; font-size: 28px; font-weight: bold; }
    .success-box { background-color: #f1f8f4; border-left: 4px solid #4CAF50; padding: 20px; margin: 20px 0; border-radius: 4px; }
    .info-box { background-color: #f8f9fa; padding: 15px; margin: 20px 0; border-radius: 4px; }
    .info-row { margin: 8px 0; }
    .info-label { font-weight: bold; color: #333; }
    .info-value { color: #666; }
    .footer { background-color: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #999; }
    .icon { font-size: 64px; margin-bottom: 10px; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <div class="icon">✅</div>
      <h1>Application Submitted Successfully</h1>
    </div>
    <div class="content" style="padding: 30px 20px;">
      <div class="success-box">
        <h2 style="margin-top: 0; color: #4CAF50;">Submission Confirmed!</h2>
        <p>Your ${getApplicationTypeDisplay(applicationData.applicationType)} has been successfully submitted.</p>
      </div>
      
      <h3 style="color: #333;">Application Details:</h3>
      <div class="info-box">
        <div class="info-row">
          <span class="info-label">Application Type:</span>
          <span class="info-value">${getApplicationTypeDisplay(applicationData.applicationType)}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Your Name:</span>
          <span class="info-value">${applicationData.fullName}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Employee Number:</span>
          <span class="info-value">${applicationData.employeeNo}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Status:</span>
          <span class="info-value" style="color: #ffa500; font-weight: bold;">Pending Review</span>
        </div>
        <div class="info-row">
          <span class="info-label">Date:</span>
          <span class="info-value">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</span>
        </div>
      </div>
      
      <p style="color: #666;">Your application has been sent to supervisors for review. You will be notified once a decision is made.</p>
    </div>
    <div class="footer">
      <p>Titan Drilling - HR Management System</p>
      <p>&copy; ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
    </div>
  </div>
</body>
</html>
    `;
    
    // Execute all notification tasks in parallel
    console.log('🚀 Starting parallel execution of all notifications...');
    
    let recipientEmailResult = { success: true, sent: 0 };
    let creatorEmailResult = { success: true, sent: 0 };
    let mongoResult = { success: true, count: 0 };
    let pushResult = { success: true, successCount: 0 };
    
    if (uniqueRecipients.length > 0) {
      console.log(`📤 Processing ${uniqueRecipients.length} recipients...`);
      
      [mongoResult, pushResult, recipientEmailResult] = await Promise.all([
        saveNotificationsToMongoDB(applicationData, message, uniqueRecipients),
        sendPushNotifications(applicationData, message, uniqueRecipients),
        sendEmails(applicationData, `New HR Application: ${getApplicationTypeDisplay(applicationData.applicationType)}`, recipientEmailHtml, uniqueRecipients),
      ]);
    }
    
    if (creatorEmail && creatorEmail.includes('@')) {
      console.log(`📤 Sending confirmation email to creator: ${creatorEmail}`);
      creatorEmailResult = await sendEmails(applicationData, `Application Submitted: ${getApplicationTypeDisplay(applicationData.applicationType)}`, creatorEmailHtml, [{ email: creatorEmail, name: createdByName }]);
    } else {
      console.log(`⚠️ No valid creator email: ${creatorEmail}`);
    }
    
    // Log results
    console.log('\n📊 ========== NOTIFICATION RESULTS ==========');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 Emails to Recipients: ${recipientEmailResult.sent || 0} sent`);
    console.log(`  📧 Creator Email: ${creatorEmailResult.sent || 0} sent`);
    console.log('========================================\n');
    
    res.json({
      success: true,
      message: `Notifications sent for new HR application`,
      results: {
        mongodb: { success: mongoResult.success, saved: mongoResult.count || 0 },
        push: { success: pushResult.success, sent: pushResult.successCount || 0 },
        emails: { recipientsSent: recipientEmailResult.sent || 0, creatorSent: creatorEmailResult.sent || 0 },
      },
      recipients: {
        proAdmins: proAdmins.length,
        supervisors: supervisors.length,
        total: uniqueRecipients.length,
      }
    });
  } catch (error) {
    console.error('❌ Error processing new application notifications:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

/**
 * POST /api/hr-notifications/supervisor-approval
 * Handle notifications when Supervisor approves application (notify HR Managers)
 */
app.post('/api/hr-notifications/supervisor-approval', async (req, res) => {
  try {
    const { applicationData, approverName } = req.body;
    
    console.log(`\n🔔 Supervisor APPROVED - Notifying HR Managers`);
    console.log(`📋 Application: ${applicationData.applicationType}`);
    
    // Get HR Managers filtered by country AND project
    const hrManagers = await GlobalUser.find({
      role: 'HrManager',
      countries: applicationData.country,
      projects: applicationData.project
    }).select('_id name email fcmToken fcmTokens');
    
    console.log(`📋 Found ${hrManagers.length} HR Manager(s) for ${applicationData.country} - ${applicationData.project}`);
    
    if (hrManagers.length === 0) {
      return res.json({ success: true, message: 'No HR Managers to notify', recipients: { hrManagers: 0 } });
    }
    
    const message = {
      title: '✅ Application Approved by Supervisor',
      body: `${getApplicationTypeDisplay(applicationData.applicationType)}\nEmployee: ${applicationData.fullName}\n\nApproved by: ${approverName}\n\n⚠️ Awaiting your approval`
    };
    
    // Create notifications with supervisor_approved type
    const notifications = hrManagers.map(recipient => ({
      notificationId: `hr_app_supervisor_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${recipient._id}`,
      userId: recipient._id,
      title: message.title,
      body: message.body,
      type: 'hr_application_supervisor_approved',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'hr_application_supervisor_approved',
        action: 'view_hr_application',
        applicationId: applicationData._id || applicationData.id,
        applicationNumber: applicationData.applicationNumber,
        applicationType: applicationData.applicationType,
        applicantName: applicationData.fullName,
        project: applicationData.project || '',
        country: applicationData.country || '',
        status: applicationData.status || '',
        timestamp: new Date().toISOString(),
      }
    }));
    
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Application Requires Approval - Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">✅ Application Requires Your Approval</h2>

      <div style="background-color: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; margin: 20px 0; border-radius: 4px;">
        <h3 style="margin-top: 0; color: #f57c00;">⚠️ Action Required</h3>
        <p style="margin: 0; line-height: 1.6;">A supervisor has approved this application. It now requires your approval as HR Manager.</p>
      </div>

      <p style="line-height: 1.6; margin-bottom: 20px;">Dear HR Manager,</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Application Type:</th>
            <td style="padding: 10px; text-align: left;">${getApplicationTypeDisplay(applicationData.applicationType)}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Employee:</th>
            <td style="padding: 10px; text-align: left;">${applicationData.fullName} (#${applicationData.employeeNo})</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Approved By:</th>
            <td style="padding: 10px; text-align: left;">${approverName} (Supervisor)</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        Please review and approve/reject this application in the Titan Drilling application.
      </p>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    // Save notifications to MongoDB
    const mongoResult = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: '168.231.113.219',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => reject(error));
      req.write(postData);
      req.end();
    });
    
    // Send push notifications with supervisor_approved type
    const pushNotificationData = {
      userIds: hrManagers.map(r => r._id),
      title: message.title,
      body: message.body,
      data: {
        type: 'hr_application_supervisor_approved',
        action: 'view_hr_application',
        applicationId: applicationData._id || applicationData.id,
        applicationNumber: applicationData.applicationNumber,
        applicationType: applicationData.applicationType,
        applicantName: applicationData.fullName,
        project: applicationData.project || '',
        country: applicationData.country || '',
        status: applicationData.status || '',
        timestamp: new Date().toISOString(),
      },
      apns: {
        payload: {
          aps: {
            alert: { title: message.title, body: message.body },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'hr_application_notifications',
        }
      }
    };
    
    const pushResult = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(pushNotificationData);
      const options = {
        hostname: '168.231.113.219',
        port: 5016,
        path: '/push-notifications/send-to-users',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => reject(error));
      req.write(postData);
      req.end();
    });
    
    // Send emails
    const emailResult = await sendEmails(applicationData, `Application Approved by Supervisor - Requires Your Action`, emailHtml, hrManagers);
    
    console.log('\n📊 Notification Results:');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 Emails: ${emailResult.sent || 0} sent\n`);
    
    res.json({
      success: true,
      message: 'HR Managers notified',
      results: { mongodb: mongoResult, push: pushResult, emails: emailResult },
      recipients: { hrManagers: hrManagers.length }
    });
  } catch (error) {
    console.error('❌ Error processing supervisor approval notifications:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

/**
 * POST /api/hr-notifications/application-decision
 * Handle notifications when application is approved/rejected (notify applicant)
 */
app.post('/api/hr-notifications/application-decision', async (req, res) => {
  try {
    const { applicationData, status, approverName, approverRole, reason, applicantId } = req.body;
    
    console.log(`\n🔔 Application ${status} - Notifying Applicant`);
    console.log(`📋 Applicant ID: ${applicantId}`);
    console.log(`👤 Approver: ${approverName} (${approverRole})`);
    
    // Get the applicant
    const applicant = await GlobalUser.findOne({ _id: applicantId }).select('_id name email fcmToken fcmTokens');
    
    if (!applicant) {
      console.log('⚠️ Applicant not found');
      return res.json({ success: true, message: 'Applicant not found' });
    }
    
    console.log(`📋 Found applicant: ${applicant.name} (${applicant.email})`);
    
    const isApproved = status === 'Approved';
    const message = {
      title: isApproved ? '✅ Application Approved' : '❌ Application Rejected',
      body: `Your ${getApplicationTypeDisplay(applicationData.applicationType)} has been ${status.toLowerCase()}\n\nDecided by: ${approverName} (${approverRole})`
    };
    
    // Create notification with approval_rejection type
    const notifications = [{
      notificationId: `hr_app_decision_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${applicant._id}`,
      userId: applicant._id,
      title: message.title,
      body: message.body,
      type: 'hr_application_approval_rejection',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'hr_application_approval_rejection',
        action: 'view_hr_application',
        applicationId: applicationData._id || applicationData.id,
        applicationNumber: applicationData.applicationNumber,
        applicationType: applicationData.applicationType,
        applicantName: applicationData.fullName,
        project: applicationData.project || '',
        country: applicationData.country || '',
        status: status,
        approverName: approverName,
        approverRole: approverRole,
        reason: reason || '',
        timestamp: new Date().toISOString(),
      }
    }];
    
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Application ${status} - Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">${isApproved ? '✅' : '❌'} Application ${status}</h2>

      <div style="background-color: ${isApproved ? '#f1f8f4' : '#ffebee'}; border-left: 4px solid ${isApproved ? '#4CAF50' : '#f44336'}; padding: 15px; margin: 20px 0; border-radius: 4px;">
        <h3 style="margin-top: 0; color: ${isApproved ? '#4CAF50' : '#f44336'};">${isApproved ? 'Approved' : 'Rejected'}</h3>
        <p style="margin: 0; line-height: 1.6;">Your ${getApplicationTypeDisplay(applicationData.applicationType)} has been ${status.toLowerCase()} by ${approverRole}.</p>
      </div>

      <p style="line-height: 1.6; margin-bottom: 20px;">Dear ${applicationData.fullName},</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Application Type:</th>
            <td style="padding: 10px; text-align: left;">${getApplicationTypeDisplay(applicationData.applicationType)}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Your Name:</th>
            <td style="padding: 10px; text-align: left;">${applicationData.fullName}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Status:</th>
            <td style="padding: 10px; text-align: left;"><strong style="color:${isApproved ? '#4CAF50' : '#f44336'};">${status}</strong></td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Decided By:</th>
            <td style="padding: 10px; text-align: left;">${approverName} (${approverRole})</td>
          </tr>
          ${reason ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Reason:</th>
            <td style="padding: 10px; text-align: left;">${reason}</td>
          </tr>` : ''}
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Date:</th>
            <td style="padding: 10px; text-align: left;">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    // Save notification to MongoDB
    const mongoResult = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: '168.231.113.219',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => reject(error));
      req.write(postData);
      req.end();
    });
    
    // Send push notification with approval_rejection type
    const pushNotificationData = {
      userIds: [applicant._id],
      title: message.title,
      body: message.body,
      data: {
        type: 'hr_application_approval_rejection',
        action: 'view_hr_application',
        applicationId: applicationData._id || applicationData.id,
        applicationNumber: applicationData.applicationNumber,
        applicationType: applicationData.applicationType,
        applicantName: applicationData.fullName,
        project: applicationData.project || '',
        country: applicationData.country || '',
        status: status,
        approverName: approverName,
        approverRole: approverRole,
        reason: reason || '',
        timestamp: new Date().toISOString(),
      },
      apns: {
        payload: {
          aps: {
            alert: { title: message.title, body: message.body },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'hr_application_notifications',
        }
      }
    };
    
    const pushResult = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(pushNotificationData);
      const options = {
        hostname: '168.231.113.219',
        port: 5016,
        path: '/push-notifications/send-to-users',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => reject(error));
      req.write(postData);
      req.end();
    });
    
    // Send email
    const emailResult = await sendEmails(applicationData, `Application ${status}: ${getApplicationTypeDisplay(applicationData.applicationType)}`, emailHtml, [applicant]);
    
    console.log('\n📊 Notification Results:');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 Email: ${emailResult.sent || 0} sent\n`);
    
    res.json({
      success: true,
      message: 'Applicant notified of decision',
      results: { mongodb: mongoResult, push: pushResult, emails: emailResult },
    });
  } catch (error) {
    console.error('❌ Error processing decision notifications:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'HR Application Notifications Service',
    mongodb: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
    timestamp: new Date().toISOString(),
  });
});

// Start server
const PORT = process.env.HR_APPLICATION_NOTIFICATIONS_PORT || 5027;
app.listen(PORT, () => {
  console.log(`\n🚀 HR Application Notifications Service running on port ${PORT}`);
  console.log(`📊 MongoDB: titan_drilling`);
  console.log(`🔗 Health check: http://168.231.113.219:${PORT}/health\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  app.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});

module.exports = app;

