import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Alert,
  ActivityIndicator,
  Platform,
} from 'react-native';
import equipmentData from './titan-drilling-1f8e9-default-rtdb-EquipmentsDataJSON-export.json';
import { MONGODB_API_URL } from '../../../config/api.config';

const API_BASE_URL = MONGODB_API_URL;

const UploadEquipmentToMongo = () => {
  const [databases, setDatabases] = useState([]);
  const [selectedDatabase, setSelectedDatabase] = useState('titan_drilling');
  const [loading, setLoading] = useState(false);
  const [uploading, setUploading] = useState(false);
  const [equipmentCount, setEquipmentCount] = useState(0);

  useEffect(() => {
    fetchDatabases();
    fetchEquipmentCount();
  }, []);

  // Fetch available databases
  const fetchDatabases = async () => {
    try {
      setLoading(true);
      const response = await fetch(`${API_BASE_URL}/databases`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error('Fetch databases response:', response.status, errorText);
        throw new Error(`HTTP error! Status: ${response.status}, Message: ${errorText}`);
      }
      
      const contentType = response.headers.get('content-type');
      if (!contentType || !contentType.includes('application/json')) {
        const text = await response.text();
        console.error('Server returned non-JSON response:', text);
        throw new Error('Server returned non-JSON response');
      }
      
      const data = await response.json();
      console.log('Databases fetched:', data);
      
      if (data.databases) {
        setDatabases(data.databases);
      } else {
        throw new Error('No databases found in response');
      }
    } catch (error) {
      console.error('Error fetching databases:', error);
      Alert.alert('Error', `Failed to fetch databases: ${error.message}`);
    } finally {
      setLoading(false);
    }
  };

  // Fetch equipment count
  const fetchEquipmentCount = async () => {
    try {
      // Use the new path
      const response = await fetch(`${API_BASE_URL}/EquipmentsDataJSON/GlobalEquipmentJsonData`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });
      
      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`HTTP error! Status: ${response.status}, Message: ${errorText}`);
      }
      
      const contentType = response.headers.get('content-type');
      if (!contentType || !contentType.includes('application/json')) {
        const text = await response.text();
        console.error('Server returned non-JSON response:', text);
        throw new Error('Server returned non-JSON response');
      }
      
      const data = await response.json();
      setEquipmentCount(data.count || 0);
    } catch (error) {
      console.error('Error fetching count:', error);
      Alert.alert('Error', `Failed to fetch equipment count: ${error.message}`);
    }
  };

  // Upload equipment data to MongoDB
  const handleUpload = async () => {
    try {
      const equipmentArray = equipmentData.GlobalEquipmentJsonData;
      
      if (!equipmentArray || !Array.isArray(equipmentArray) || equipmentArray.length === 0) {
        Alert.alert('Error', 'No equipment data found to upload');
        return;
      }

      Alert.alert(
        'Confirm Upload',
        `Upload ${equipmentArray.length} equipment items to ${selectedDatabase}?`,
        [
          { text: 'Cancel', style: 'cancel' },
          {
            text: 'Upload',
            onPress: async () => {
              setUploading(true);
              try {
                // Use the new path
                const response = await fetch(`${API_BASE_URL}/EquipmentsDataJSON/GlobalEquipmentJsonData`, {
                  method: 'POST',
                  headers: {
                    'Content-Type': 'application/json',
                  },
                  body: JSON.stringify({
                    equipmentData: equipmentArray,
                  }),
                });

                if (!response.ok) {
                  const errorText = await response.text();
                  throw new Error(`HTTP error! Status: ${response.status}, Message: ${errorText}`);
                }

                const contentType = response.headers.get('content-type');
                if (!contentType || !contentType.includes('application/json')) {
                  const text = await response.text();
                  console.error('Server returned non-JSON response:', text);
                  throw new Error('Server returned non-JSON response');
                }

                const result = await response.json();
                console.log('Upload result:', result);

                Alert.alert('Success', result.message);
                fetchEquipmentCount();
              } catch (error) {
                console.error('Upload error:', error);
                Alert.alert('Error', `Failed to upload equipment data: ${error.message}`);
              } finally {
                setUploading(false);
              }
            },
          },
        ]
      );
    } catch (error) {
      console.error('Error preparing upload:', error);
      Alert.alert('Error', `Failed to prepare upload: ${error.message}`);
    }
  };

  // Delete all equipment data
  const handleDelete = async () => {
    Alert.alert(
      'Confirm Delete',
      'Are you sure you want to delete ALL equipment data? This cannot be undone!',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            setLoading(true);
            try {
              // Use the new path
              const response = await fetch(`${API_BASE_URL}/EquipmentsDataJSON/GlobalEquipmentJsonData`, {
                method: 'DELETE',
                headers: {
                  'Content-Type': 'application/json',
                },
              });

              if (!response.ok) {
                const errorText = await response.text();
                throw new Error(`HTTP error! Status: ${response.status}, Message: ${errorText}`);
              }

              const result = await response.json();
              Alert.alert('Success', result.message);
              fetchEquipmentCount();
            } catch (error) {
              console.error('Delete error:', error);
              Alert.alert('Error', `Failed to delete equipment data: ${error.message}`);
            } finally {
              setLoading(false);
            }
          },
        },
      ]
    );
  };

  // Delete equipment from specific database
  const handleDeleteDatabase = async (dbName) => {
    Alert.alert(
      'Confirm Delete',
      `Delete all equipment from ${dbName}?`,
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            setLoading(true);
            try {
              const response = await fetch(`${API_BASE_URL}/clear-database/${dbName}`, {
                method: 'DELETE',
                headers: {
                  'Content-Type': 'application/json',
                },
              });

              if (!response.ok) {
                const errorText = await response.text();
                throw new Error(`HTTP error! Status: ${response.status}, Message: ${errorText}`);
              }

              const result = await response.json();
              Alert.alert('Success', result.message);
              fetchEquipmentCount();
            } catch (error) {
              console.error('Delete error:', error);
              Alert.alert('Error', `Failed to delete database equipment: ${error.message}`);
            } finally {
              setLoading(false);
            }
          },
        },
      ]
    );
  };

  const totalEquipmentInFile = equipmentData.GlobalEquipmentJsonData?.length || 0;

  return (
    <ScrollView style={styles.container}>
      {/* Header */}
      <View style={styles.header}>
        <View style={styles.headerContent}>
          <View style={styles.iconCircle}>
            <Text style={styles.iconText}>☁️</Text>
          </View>
          <View style={styles.headerTextContainer}>
            <Text style={styles.headerTitle}>MongoDB Equipment Upload</Text>
            <Text style={styles.headerSubtitle}>
              Titan Drilling Equipment Management
            </Text>
          </View>
        </View>
        <View style={styles.headerStats}>
          <Text style={styles.iconTextSmall}>🗄️</Text>
          <Text style={styles.headerStatsText}>
            {equipmentCount} items in database
          </Text>
        </View>
      </View>

      {/* Connection Info */}
      <View style={styles.infoCard}>
        <View style={styles.infoRow}>
          <Text style={styles.iconTextInfo}>🔗</Text>
          <Text style={styles.infoLabel}>Server:</Text>
          <Text style={styles.infoValue}>168.231.113.219:27017</Text>
        </View>
        <View style={styles.infoRow}>
          <Text style={styles.iconTextInfo}>👤</Text>
          <Text style={styles.infoLabel}>User:</Text>
          <Text style={styles.infoValue}>titandrillingadminuser</Text>
        </View>
        <View style={styles.infoRow}>
          <Text style={styles.iconTextInfo}>📄</Text>
          <Text style={styles.infoLabel}>Collection:</Text>
          <Text style={styles.infoValue}>Equipment</Text>
        </View>
        <View style={styles.infoRow}>
          <Text style={styles.iconTextInfo}>📋</Text>
          <Text style={styles.infoLabel}>File Items:</Text>
          <Text style={styles.infoValue}>{totalEquipmentInFile} equipment</Text>
        </View>
      </View>

      {/* Database Selection */}
      <View style={styles.section}>
        <View style={styles.sectionHeader}>
          <Text style={styles.iconTextSection}>📁</Text>
          <Text style={styles.sectionTitle}>Select Database</Text>
        </View>

        {loading ? (
          <ActivityIndicator size="large" color="#0055A5" />
        ) : (
          <View style={styles.databaseGrid}>
            {databases.map((db, index) => (
              <TouchableOpacity
                key={index}
                style={[
                  styles.databaseButton,
                  selectedDatabase === db && styles.databaseButtonSelected,
                ]}
                onPress={() => setSelectedDatabase(db)}
              >
                <Text
                  style={[
                    styles.iconTextDb,
                    selectedDatabase === db && styles.iconTextDbSelected,
                  ]}
                >
                  💾
                </Text>
                <Text
                  style={[
                    styles.databaseButtonText,
                    selectedDatabase === db && styles.databaseButtonTextSelected,
                  ]}
                >
                  {db}
                </Text>
                {selectedDatabase === db && (
                  <Text style={styles.checkIcon}>✓</Text>
                )}
              </TouchableOpacity>
            ))}
          </View>
        )}

        {databases.length === 0 && !loading && (
          <Text style={styles.noDataText}>No databases found</Text>
        )}
      </View>

      {/* Action Buttons */}
      <View style={styles.section}>
        <View style={styles.sectionHeader}>
          <Text style={styles.iconTextSection}>⚙️</Text>
          <Text style={styles.sectionTitle}>Actions</Text>
        </View>

        {/* Upload Button */}
        <TouchableOpacity
          style={[styles.actionButton, styles.uploadButton]}
          onPress={handleUpload}
          disabled={uploading || loading}
        >
          {uploading ? (
            <ActivityIndicator color="#fff" />
          ) : (
            <>
              <Text style={styles.buttonIcon}>⬆️</Text>
              <Text style={styles.actionButtonText}>
                Upload {totalEquipmentInFile} Equipment Items
              </Text>
            </>
          )}
        </TouchableOpacity>

        {/* Refresh Button */}
        <TouchableOpacity
          style={[styles.actionButton, styles.refreshButton]}
          onPress={() => {
            fetchDatabases();
            fetchEquipmentCount();
          }}
          disabled={loading}
        >
          <Text style={styles.buttonIcon}>🔄</Text>
          <Text style={styles.actionButtonText}>Refresh Data</Text>
        </TouchableOpacity>

        {/* Delete All Button */}
        <TouchableOpacity
          style={[styles.actionButton, styles.deleteButton]}
          onPress={handleDelete}
          disabled={loading}
        >
          <Text style={styles.buttonIcon}>🗑️</Text>
          <Text style={styles.actionButtonText}>Delete All Equipment</Text>
        </TouchableOpacity>
      </View>

      {/* Footer */}
      <View style={styles.footer}>
        <Text style={styles.iconTextInfo}>ℹ️</Text>
        <Text style={styles.footerText}>
          Data will be uploaded to Equipment collection
        </Text>
      </View>
    </ScrollView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  header: {
    backgroundColor: '#0055A5',
    padding: 20,
    paddingTop: Platform.OS === 'ios' ? 50 : 20,
    borderBottomLeftRadius: 20,
    borderBottomRightRadius: 20,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 5,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 15,
  },
  iconCircle: {
    width: 50,
    height: 50,
    borderRadius: 25,
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  iconText: {
    fontSize: 28,
  },
  iconTextSmall: {
    fontSize: 18,
  },
  iconTextInfo: {
    fontSize: 20,
    marginRight: 5,
  },
  iconTextSection: {
    fontSize: 24,
  },
  iconTextDb: {
    fontSize: 18,
    marginRight: 5,
  },
  iconTextDbSelected: {
    fontSize: 18,
    marginRight: 5,
  },
  buttonIcon: {
    fontSize: 22,
    marginRight: 8,
  },
  checkIcon: {
    fontSize: 18,
    color: '#fff',
    fontWeight: 'bold',
  },
  headerTextContainer: {
    marginLeft: 15,
    flex: 1,
  },
  headerTitle: {
    fontSize: 22,
    fontWeight: 'bold',
    color: '#fff',
    fontFamily: 'Jost',
  },
  headerSubtitle: {
    fontSize: 14,
    color: '#ffd205',
    marginTop: 4,
    fontFamily: 'Jost',
  },
  headerStats: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    padding: 10,
    borderRadius: 10,
  },
  headerStatsText: {
    color: '#fff',
    marginLeft: 8,
    fontSize: 14,
    fontWeight: '600',
    fontFamily: 'Jost',
  },
  infoCard: {
    backgroundColor: '#fff',
    margin: 15,
    padding: 15,
    borderRadius: 12,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  infoRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 12,
  },
  infoLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
    width: 90,
    fontFamily: 'Jost',
  },
  infoValue: {
    fontSize: 14,
    color: '#666',
    flex: 1,
    fontFamily: 'Jost',
  },
  section: {
    margin: 15,
    marginTop: 0,
  },
  sectionHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 15,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginLeft: 10,
    fontFamily: 'Jost',
  },
  databaseGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 10,
  },
  databaseButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    padding: 12,
    paddingHorizontal: 16,
    borderRadius: 10,
    borderWidth: 2,
    borderColor: '#0055A5',
    minWidth: '45%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    elevation: 2,
  },
  databaseButtonSelected: {
    backgroundColor: '#0055A5',
    borderColor: '#ffd205',
  },
  databaseButtonText: {
    fontSize: 14,
    color: '#0055A5',
    fontWeight: '600',
    marginLeft: 8,
    flex: 1,
    fontFamily: 'Jost',
  },
  databaseButtonTextSelected: {
    color: '#fff',
  },
  noDataText: {
    textAlign: 'center',
    color: '#999',
    fontSize: 14,
    marginTop: 20,
    fontFamily: 'Jost',
  },
  actionButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    padding: 16,
    borderRadius: 12,
    marginBottom: 12,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  uploadButton: {
    backgroundColor: '#0055A5',
  },
  refreshButton: {
    backgroundColor: '#00B894',
  },
  deleteButton: {
    backgroundColor: '#d63031',
  },
  actionButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 10,
    fontFamily: 'Jost',
  },
  footer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    padding: 20,
    marginTop: 10,
  },
  footerText: {
    color: '#666',
    fontSize: 12,
    marginLeft: 8,
    textAlign: 'center',
    fontFamily: 'Jost',
  },
});

export default UploadEquipmentToMongo;