@extends('welcome')

@section('title', 'Create New Logbook')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
@endsection

@section('content')
<div class="workorder-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('logbook') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Create New Logbook Entry</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" onclick="debugFormData()">
                    <i class="fas fa-plus"></i>
                    Create
                </button>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
    <div class="form-container">
        <form id="logbookForm" class="logbook-form">
            @csrf
            
            <!-- Entry Number -->
            <div class="form-group">
                <label for="entryNumber">Entry #</label>
                <input type="text" id="entryNumber" name="entryNumber" class="form-control" readonly>
            </div>
            
            <!-- Country -->
            <div class="form-group">
                <label for="country">Country <span class="required">*</span></label>
                <select id="country" name="country" class="form-control" required>
                    <option value="">Select Country</option>
                    <option value="Zambia">Zambia</option>
                    <option value="DRC">DRC</option>
                    <option value="Namibia">Namibia</option>
                    <option value="Botswana">Botswana</option>
                </select>
            </div>
            
            <!-- Project -->
            <div class="form-group">
                <label for="project">Project <span class="required">*</span></label>
                <select id="project" name="project" class="form-control" required>
                    <option value="">Select Project</option>
                    <!-- Projects will be populated dynamically based on country selection -->
                </select>
            </div>
            
            <!-- Equipment -->
            <div class="form-group">
                <label for="equipment">Equipment <span class="required">*</span></label>
                <div class="equipment-select-container">
                    <div class="equipment-select" onclick="openEquipmentModal()">
                        <span id="equipmentText">Select Equipment</span>
                        <i class="fas fa-chevron-down"></i>
                    </div>
                    <input type="hidden" id="equipmentId" name="equipmentId">
                </div>
                <div id="selectedEquipmentInfo" class="selected-equipment-info" style="display: none;">
                    <div class="equipment-details">
                        <p><strong>Selected:</strong> <span id="selectedEquipmentName"></span></p>
                        <p><strong>Current Meter Reading:</strong> <span id="currentMeterReading"></span></p>
                    </div>
                </div>
            </div>
            
            <!-- Equipment Status -->
            <div class="form-group">
                <label for="equipmentStatus">Equipment Status <span class="required">*</span></label>
                <div class="status-segments">
                    <input type="radio" id="available" name="equipmentStatus" value="available" class="status-radio">
                    <label for="available" class="status-label">Available</label>
                    
                    <input type="radio" id="deployed" name="equipmentStatus" value="deployed" class="status-radio">
                    <label for="deployed" class="status-label">Deployed</label>
                    
                    <input type="radio" id="maintenance" name="equipmentStatus" value="maintenance" class="status-radio">
                    <label for="maintenance" class="status-label">In Maintenance</label>
                    
                    <input type="radio" id="breakdown" name="equipmentStatus" value="breakdown" class="status-radio">
                    <label for="breakdown" class="status-label">Breakdown</label>
                </div>
            </div>
            
            <!-- Start Reading & End Reading -->
            <div class="form-grid">
                <div class="form-group">
                    <label for="startReading">Start Reading <span class="required">*</span></label>
                    <input type="number" id="startReading" name="startReading" class="form-control" placeholder="Miles" required>
                </div>
                
                <div class="form-group">
                    <label for="endReading">End Reading <span class="required">*</span></label>
                    <input type="number" id="endReading" name="endReading" class="form-control" placeholder="Miles" required>
                </div>
            </div>
            
            <!-- Start Date & Time & End Date & Time -->
            <div class="form-grid">
                <div class="form-group">
                    <label for="startDateTime">Start Date & Time <span class="required">*</span></label>
                    <input type="datetime-local" id="startDateTime" name="startDateTime" class="form-control" placeholder="Select Date & Time" required>
                </div>
                
                <div class="form-group">
                    <label for="endDateTime">End Date & Time <span class="required">*</span></label>
                    <input type="datetime-local" id="endDateTime" name="endDateTime" class="form-control" placeholder="Select Date & Time" required>
                </div>
            </div>
            
            <!-- Fuel Opening & Fuel Closing -->
            <div class="form-grid">
                <div class="form-group">
                    <label for="fuelOpening">Fuel Opening</label>
                    <input type="number" id="fuelOpening" name="fuelOpening" class="form-control" placeholder="Enter fuel opening" step="0.1" min="0">
                </div>
                
                <div class="form-group">
                    <label for="fuelClosing">Fuel Closing</label>
                    <input type="number" id="fuelClosing" name="fuelClosing" class="form-control" placeholder="Enter fuel closing" step="0.1" min="0">
                </div>
            </div>
            
            <!-- Fuel Unit -->
            <div class="form-group">
                <label for="fuelUnit">Fuel Unit</label>
                <select id="fuelUnit" name="fuelUnit" class="form-control">
                    <option value="KG">KG</option>
                    <option value="Piece">Piece</option>
                    <option value="Gram">Gram</option>
                    <option value="Gallon">Gallon</option>
                    <option value="Quart">Quart</option>
                    <option value="Liter">Liter</option>
                    <option value="ML">ML</option>
                    <option value="Meter">Meter</option>
                </select>
            </div>
            
            <!-- Operator Name -->
            <div class="form-group">
                <label for="operatorName">Operator Name</label>
                <input type="text" id="operatorName" name="operatorName" class="form-control" placeholder="Enter operator name">
            </div>
            
            <!-- Work Location -->
            <div class="form-group">
                <label for="workLocation">Work Location</label>
                <input type="text" id="workLocation" name="workLocation" class="form-control" placeholder="Enter work location">
            </div>
            
            <!-- Work Type -->
            <div class="form-group">
                <label for="workType">Work Type</label>
                <input type="text" id="workType" name="workType" class="form-control" placeholder="Enter work type">
            </div>
            
            <!-- Work Description -->
            <div class="form-group">
                <label for="workDescription">Work Description</label>
                <textarea id="workDescription" name="workDescription" class="form-control" rows="3" placeholder="Enter work description"></textarea>
            </div>
            
            <!-- Material Description -->
            <div class="form-group">
                <div class="collapsible-header" onclick="toggleCollapsible('materialSection')">
                <label for="materialDescription">Material Description</label>
                    <span class="collapsible-icon" id="materialIcon">▼</span>
                </div>
                <div id="materialSection" class="collapsible-content" style="display: none;">
                    <input type="text" id="materialName" name="materialName" class="form-control" placeholder="Material Name">
                    <div class="form-grid">
                        <div class="form-group">
                            <input type="number" id="materialQuantity" name="materialQuantity" class="form-control" placeholder="Material Quantity" step="0.1" min="0">
                        </div>
                        <div class="form-group">
                            <select id="materialUnit" name="materialUnit" class="form-control">
                                <option value="Ton">Ton</option>
                                <option value="KG">KG</option>
                                <option value="Piece">Piece</option>
                                <option value="Gram">Gram</option>
                                <option value="Gallon">Gallon</option>
                                <option value="Quart">Quart</option>
                                <option value="Liter">Liter</option>
                                <option value="ML">ML</option>
                                <option value="Meter">Meter</option>
                </select>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Issue Report -->
            <div class="form-group">
                <div class="collapsible-header" onclick="toggleCollapsible('issueSection')">
                <label for="issueReport">Issue Report</label>
                    <span class="collapsible-icon" id="issueIcon">▼</span>
                </div>
                <div id="issueSection" class="collapsible-content" style="display: none;">
                    <input type="text" id="issueReport" name="issueReport" class="form-control" placeholder="Issue Report">
                    <input type="text" id="issueName" name="issueName" class="form-control" placeholder="Issue Name">
                    <textarea id="issueDetails" name="issueDetails" class="form-control" rows="3" placeholder="Issue Details"></textarea>
                </div>
            </div>

        </form>
        </div>
    </div>
</div>

<!-- Equipment Selection Modal -->
<div id="equipmentModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Equipment</h3>
            <button type="button" class="close-btn" onclick="closeEquipmentModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="equipmentSearch" class="search-input" placeholder="Search equipment..." onkeyup="filterEquipment()">
                <button type="button" class="search-btn" onclick="filterEquipment()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="equipment-list" id="equipmentList">
                <div class="loading-state">
                    <div class="loading-spinner"></div>
                    Loading equipment...
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Success Modal -->
<div id="successModal" class="success-modal-overlay" style="display: none;">
    <div class="success-modal-container">
        <div class="success-modal-header">
            <h2 class="success-title">Success</h2>
        </div>
        <div class="success-modal-body">
            <p class="success-message">Logbook Entry Created and Saved Successfully!</p>
            <div class="success-buttons">
                <button type="button" class="success-btn back-btn" onclick="goBackToLogbook()">
                    Back
                </button>
                <button type="button" class="success-btn create-new-btn" onclick="createNewEntry()">
                    Create New Entry
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Sync Modal -->
<div id="syncModal" class="sync-modal" style="display: none;">
    <div class="sync-modal-content">
        <h3 class="sync-modal-text">Please Wait</h3>
        <div class="sync-icon-container">
            <div class="sync-icon">🔄</div>
            <span class="sync-modal-subtext">Saving Logbook Entry</span>
        </div>
    </div>
</div>

<style>
/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: all 0.15s ease-in-out;
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: #0055a5;
    border: none;
    padding: 12px 24px;
    border-radius: 6px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.15s ease-in-out;
    box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
}

/* Success Modal Styles - Matching Create New Workorder File */
.success-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
}

.success-modal-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
    max-width: 500px;
    width: 90%;
    max-height: 80vh;
    overflow: hidden;
    position: relative;
}

.success-modal-header {
    background: #0055A5;
    padding: 20px 30px;
    text-align: center;
    border-radius: 12px 12px 0 0;
}

.success-title {
    color: white;
    font-size: 24px;
    font-weight: 700;
    margin: 0;
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

.success-modal-body {
    padding: 40px 30px;
    text-align: center;
}

.success-message {
    color: #333;
    font-size: 18px;
    font-weight: 500;
    margin: 0 0 30px 0;
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

.success-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
    margin-top: 20px;
}

.success-btn {
    background: #0055A5;
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: 8px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

.success-btn:hover {
    background: #003d7a;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.success-btn:active {
    transform: translateY(0);
}

.back-btn {
    background: #0055A5;
}

.create-new-btn {
    background: #0055A5;
}

.create-new-btn:hover {
    background: #003d7a;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

.form-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    overflow: hidden;
}

.logbook-form {
    padding: 40px;
}

.form-section {
    margin-bottom: 40px;
    padding-bottom: 32px;
    border-bottom: 1px solid #e9ecef;
}

.form-section:last-of-type {
    border-bottom: none;
    margin-bottom: 0;
}

.section-header {
    margin-bottom: 24px;
}

.section-header h3 {
    font-size: 1.25rem;
    font-weight: 600;
    color: #0055a5;
    margin: 0 0 8px 0;
    font-family: 'Jost', sans-serif;
}

.section-header p {
    color: #6c757d;
    margin: 0;
    font-size: 0.9rem;
}

.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 20px;
    margin-bottom: 20px;
}

.status-segments {
    display: flex;
    gap: 8px;
    flex-wrap: wrap;
}

.status-radio {
    display: none;
}

.status-label {
    padding: 8px 16px;
    border: 2px solid #ced4da;
    border-radius: 6px;
    background: white;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.9rem;
    font-weight: 500;
    color: #495057;
    text-align: center;
    min-width: 100px;
}

.status-radio:checked + .status-label {
    background: #0055a5;
    color: white;
    border-color: #0055a5;
}

.status-label:hover {
    border-color: #0055a5;
    color: #0055a5;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    font-weight: 600;
    color: #495057;
    margin-bottom: 8px;
    font-size: 0.9rem;
}

.required {
    color: #dc3545;
}

.form-control {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid #ced4da;
    border-radius: 6px;
    font-size: 0.9rem;
    transition: border-color 0.3s ease, box-shadow 0.3s ease;
    font-family: inherit;
}

.form-control:focus {
    outline: none;
    border-color: #0055a5;
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-control[type="file"] {
    padding: 8px 12px;
}

.form-text {
    display: block;
    margin-top: 4px;
    font-size: 0.8rem;
    color: #6c757d;
}

textarea.form-control {
    resize: vertical;
    min-height: 80px;
}


/* Equipment Select Styles */
.equipment-select-container {
    position: relative;
}

.equipment-select {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid #ddd;
    border-radius: 6px;
    background: white;
    cursor: pointer;
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 14px;
    color: #333;
    transition: all 0.2s;
    min-height: 48px;
}

.equipment-select:hover {
    border-color: #0055A5;
    box-shadow: 0 0 0 2px rgba(0, 85, 165, 0.1);
}

.equipment-select i {
    color: #666;
    transition: transform 0.2s;
}

.equipment-select:hover i {
    color: #0055A5;
    transform: translateY(-1px);
}

.equipment-select-btn {
    background: #0055a5;
    color: white;
    border: none;
    padding: 12px 20px;
    border-radius: 6px;
    cursor: pointer;
    font-weight: 500;
    transition: all 0.3s ease;
}

.equipment-select-btn:hover {
    background: #003d82;
    transform: translateY(-1px);
}

.selected-equipment-info {
    margin-top: 10px;
        padding: 15px;
    background: #f8f9fa;
    border-radius: 6px;
    border-left: 4px solid #0055a5;
}

.equipment-details p {
    margin: 5px 0;
    color: #495057;
}

/* Collapsible Styles */
.collapsible-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    cursor: pointer;
    padding: 10px 0;
    border-bottom: 1px solid #e9ecef;
    margin-bottom: 10px;
}

.collapsible-header:hover {
    background: #f8f9fa;
    padding: 10px;
    border-radius: 6px;
}

.collapsible-icon {
    font-size: 12px;
    color: #0055a5;
    transition: transform 0.3s ease;
}

.collapsible-content {
    padding: 15px 0;
    border-top: 1px solid #e9ecef;
}

.collapsible-content.show {
    display: block !important;
}

/* Body scroll lock when modal is open */
body.modal-open {
    overflow: hidden;
}

/* Equipment Modal Styles - Matching Create New Workorder */
.custom-modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    overflow-y: auto;
}

.custom-modal.show {
    display: block;
}

.custom-modal-content {
    background-color: #fefefe;
    margin: 0;
    padding: 0;
    border: 1px solid #888;
    width: 80%;
    max-width: 600px;
    border-radius: 8px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    position: absolute;
    left: 50%;
    transform: translateX(-50%);
    max-height: 80vh;
    overflow-y: auto;
    animation: modalSlideIn 0.3s ease;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: translateX(-50%) scale(0.9);
    }
    to {
        opacity: 1;
        transform: translateX(-50%) scale(1);
    }
}

.custom-modal-header {
    background-color: #0055A5;
    color: white;
    padding: 15px 20px;
    border-radius: 8px 8px 0 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.custom-modal-header h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 500;
}

.close-btn {
    color: white;
    font-size: 28px;
    font-weight: bold;
    background: none;
    border: none;
    cursor: pointer;
    padding: 0;
    line-height: 1;
}

.close-btn:hover {
    opacity: 0.7;
}

.custom-modal-body {
    padding: 20px;
    max-height: 60vh;
    overflow-y: auto;
}

.search-container {
    display: flex;
    gap: 10px;
    margin-bottom: 20px;
}

.search-input {
    flex: 1;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
}

.search-btn {
    padding: 10px 15px;
    background-color: #0055A5;
    color: white;
    border: none;
    border-radius: 4px;
    cursor: pointer;
}

.search-btn:hover {
    background-color: #003f7d;
}

.equipment-list {
    max-height: 400px;
    overflow-y: auto;
    border: 1px solid #eee;
    border-radius: 4px;
}

.equipment-item {
    padding: 15px;
    border-bottom: 1px solid #eee;
    cursor: pointer;
    transition: background-color 0.2s;
}

.equipment-item:hover {
    background-color: #f8f9fa;
}

.equipment-item:last-child {
    border-bottom: none;
}

.equipment-name {
    font-weight: 600;
    color: #0055A5;
    margin-bottom: 5px;
}

.equipment-details {
    font-size: 12px;
    color: #666;
    line-height: 1.4;
}

.equipment-status {
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 11px;
    font-weight: 500;
    text-transform: uppercase;
}

.equipment-status.available {
    background-color: #d4edda;
    color: #155724;
}

.equipment-status.inuse {
    background-color: #fff3cd;
    color: #856404;
}

.equipment-status.maintenance {
    background-color: #f8d7da;
    color: #721c24;
}

.loading-state {
    text-align: center;
    padding: 40px 20px;
    color: #666;
}

.loading-spinner {
    width: 20px;
    height: 20px;
    border: 2px solid #f3f3f3;
    border-top: 2px solid #0055A5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin: 0 auto 10px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.no-data {
    text-align: center;
    padding: 40px 20px;
    color: #666;
    font-style: italic;
}

/* Sync Modal Styles */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
}

.sync-modal-content {
    background: white;
    border-radius: 12px;
    padding: 30px;
    text-align: center;
    max-width: 400px;
    width: 90%;
}

.sync-modal-text {
    font-size: 18px;
    font-weight: 600;
    color: #333;
    margin-bottom: 20px;
}

.sync-icon-container {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 15px;
    margin-bottom: 15px;
}

.sync-icon {
    width: 30px;
    height: 30px;
    animation: rotate 1s linear infinite;
}

.sync-modal-subtext {
    font-size: 14px;
    color: #666;
}

@keyframes rotate {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .main-content {
        padding: 20px 15px;
    }
    
    .logbook-form {
        padding: 24px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 16px;
    }
    
    .equipment-select-container {
        flex-direction: column;
        align-items: stretch;
    }
    
    .equipment-select-btn {
        margin-top: 10px;
    }
    
    .success-modal-container {
        width: 95%;
        margin: 10px;
    }
    
    .success-modal-header {
        padding: 15px 20px;
    }
    
    .success-title {
        font-size: 20px;
    }
    
    .success-modal-body {
        padding: 30px 20px;
    }
    
    .success-message {
        font-size: 16px;
    }
    
    .success-buttons {
        flex-direction: column;
    }
    
    .success-btn {
        width: 100%;
        margin: 5px 0;
    }
    
    .custom-modal-content {
        width: 95%;
        max-width: 95%;
        margin: 10px;
    }
    
    .equipment-list {
        max-height: 300px;
    }
    
    .custom-modal-header {
        padding: 12px 15px;
    }
    
    .custom-modal-body {
        padding: 15px;
        max-height: 50vh;
    }
}

@media (max-width: 480px) {
    .header-content {
        padding: 0 10px;
    }
    
    .header-title {
        font-size: 20px;
    }
    
    .main-content {
        padding: 15px 10px;
    }
    
    .logbook-form {
        padding: 20px;
    }
    
    .form-section {
        margin-bottom: 32px;
        padding-bottom: 24px;
    }
}
</style>

<script>
// Global variables
let selectedEquipment = null;
let equipmentId = null;
let currentMeterReading = null;
let equipmentList = [];
let userData = {
    countries: [],
    projects: [],
    logBookPermissions: []
};

// Project data based on country
const projectData = {
    'Zambia': ['Kobold', 'Kalumbila', 'Kansanshi', 'Kimiteto', 'Mimosa'],
    'DRC': ['Musompo', 'IME'],
    'Namibia': ['Rosh Pinah', 'Tsumeb'],
    'Botswana': ['Jwaneng', 'Orapa']
};

document.addEventListener('DOMContentLoaded', function() {
    initializeForm();
    setupEventListeners();
    generateEntryNumber();
});
    
function initializeForm() {
    // Load equipment from Firebase
    loadEquipmentFromFirebase();
    
    // Set current date and time as default
    const now = new Date();
    const startDateTime = document.getElementById('startDateTime');
    const endDateTime = document.getElementById('endDateTime');
    
    // Format datetime for input (YYYY-MM-DDTHH:MM)
    const formatDateTime = (date) => {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        const hours = String(date.getHours()).padStart(2, '0');
        const minutes = String(date.getMinutes()).padStart(2, '0');
        return `${year}-${month}-${day}T${hours}:${minutes}`;
    };
    
    startDateTime.value = formatDateTime(now);
    endDateTime.value = formatDateTime(new Date(now.getTime() + 60 * 60 * 1000)); // 1 hour later
    
    // Set default fuel unit
    document.getElementById('fuelUnit').value = 'KG';
    document.getElementById('materialUnit').value = 'Ton';
}

function setupEventListeners() {
    const form = document.getElementById('logbookForm');
    const countrySelect = document.getElementById('country');
    const projectSelect = document.getElementById('project');
    const selectEquipmentBtn = document.getElementById('selectEquipmentBtn');
    
    // Country change handler
    countrySelect.addEventListener('change', function() {
        updateProjectItems(this.value);
        generateEntryNumber();
    });
    
    // Project change handler
    projectSelect.addEventListener('change', function() {
        generateEntryNumber();
    });
    
    // Equipment select handler
    selectEquipmentBtn.addEventListener('click', function() {
        // Simulate equipment selection (replace with actual equipment selection modal)
        showEquipmentSelectionModal();
    });
    
    // Form submission
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        saveLogBookEntry();
    });
    
    // Real-time validation
    const inputs = form.querySelectorAll('.form-control');
    inputs.forEach(input => {
        input.addEventListener('blur', function() {
            validateField(this);
        });
        
        input.addEventListener('input', function() {
            // Clear error styling on input
            if (this.style.borderColor === 'rgb(220, 53, 69)') {
                this.style.borderColor = '#ced4da';
                this.style.boxShadow = 'none';
            }
        });
    });
    
    // Add validation for equipment status radio buttons
    const statusRadios = form.querySelectorAll('input[name="equipmentStatus"]');
    statusRadios.forEach(radio => {
        radio.addEventListener('change', function() {
            // Clear any previous error styling on equipment select
            const equipmentSelect = document.querySelector('.equipment-select');
            if (equipmentSelect) {
                equipmentSelect.style.borderColor = '#ced4da';
                equipmentSelect.style.boxShadow = 'none';
            }
        });
    });
}

function updateProjectItems(country) {
    const projectSelect = document.getElementById('project');
    projectSelect.innerHTML = '<option value="">Select Project</option>';
    
    if (country && projectData[country]) {
        projectData[country].forEach(project => {
            const option = document.createElement('option');
            option.value = project;
            option.textContent = project;
            projectSelect.appendChild(option);
        });
    }
}

function generateEntryNumber() {
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    
    if (!country || !project) {
        document.getElementById('entryNumber').value = Math.floor(Math.random() * 1000000000).toString();
        return;
    }
    
    // Generate custom entry number based on country and project
    let countryCode = '';
    if (country === 'Zambia') {
        countryCode = 'ZMB';
    } else if (country === 'DRC') {
        countryCode = 'DRC';
    } else {
        countryCode = country.substring(0, 3).toUpperCase();
    }
    
    const projectCode = project.substring(0, 3).toUpperCase();
    const sequentialNumber = Math.floor(Math.random() * 999) + 1;
    const paddedNumber = sequentialNumber.toString().padStart(3, '0');
    
    document.getElementById('entryNumber').value = `${countryCode}-${projectCode}-LBE-${paddedNumber}`;
}

function showEquipmentSelectionModal() {
    // Simulate equipment selection
    const equipmentOptions = [
        { id: '1', name: 'Drill Rig 001', meterReading: '1250' },
        { id: '2', name: 'Truck 001', meterReading: '8500' },
        { id: '3', name: 'Excavator 001', meterReading: '3200' },
        { id: '4', name: 'Generator 001', meterReading: '450' }
    ];
    
    // Create modal
    const modal = document.createElement('div');
    modal.className = 'equipment-modal';
    modal.innerHTML = `
        <div class="equipment-modal-content">
            <h3>Select Equipment</h3>
            <div class="equipment-list">
                ${equipmentOptions.map(equipment => `
                    <div class="equipment-item" onclick="selectEquipment('${equipment.id}', '${equipment.name}', '${equipment.meterReading}')">
                        <strong>${equipment.name}</strong>
                        <span>Meter Reading: ${equipment.meterReading}</span>
                    </div>
                `).join('')}
            </div>
            <button onclick="closeEquipmentModal()" class="btn-cancel">Cancel</button>
        </div>
    `;
    
    // Add modal styles
    modal.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.5);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10000;
    `;
    
    const modalContent = modal.querySelector('.equipment-modal-content');
    modalContent.style.cssText = `
        background: white;
        border-radius: 12px;
        padding: 30px;
        max-width: 500px;
        width: 90%;
        max-height: 80vh;
        overflow-y: auto;
    `;
    
    const equipmentList = modal.querySelector('.equipment-list');
    equipmentList.style.cssText = `
        margin: 20px 0;
    `;
    
    const equipmentItems = modal.querySelectorAll('.equipment-item');
    equipmentItems.forEach(item => {
        item.style.cssText = `
            padding: 15px;
            border: 1px solid #e9ecef;
            border-radius: 6px;
            margin-bottom: 10px;
            cursor: pointer;
            transition: all 0.3s ease;
        `;
        
        item.addEventListener('mouseenter', function() {
            this.style.backgroundColor = '#f8f9fa';
            this.style.borderColor = '#0055a5';
        });
        
        item.addEventListener('mouseleave', function() {
            this.style.backgroundColor = 'white';
            this.style.borderColor = '#e9ecef';
        });
    });
    
    document.body.appendChild(modal);
}

// Equipment modal functions
function openEquipmentModal(event) {
    const modal = document.getElementById('equipmentModal');
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    // Position modal content with proper scrolling
    setTimeout(() => {
        const modalContent = modal.querySelector('.custom-modal-content');
        if (modalContent && event) {
            const clickY = event.clientY;
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            
            let topPosition = scrollY + clickY - (contentHeight / 2);
            const minTop = scrollY + 20;
            const maxTop = scrollY + viewportHeight - contentHeight - 20;
            
            if (topPosition < minTop) topPosition = minTop;
            if (topPosition > maxTop) topPosition = maxTop;
            
            modalContent.style.top = `${topPosition}px`;
            
            // Ensure modal is visible by scrolling to it if needed
            const modalRect = modalContent.getBoundingClientRect();
            if (modalRect.top < 0 || modalRect.bottom > viewportHeight) {
                modalContent.scrollIntoView({ 
                    behavior: 'smooth', 
                    block: 'center',
                    inline: 'center'
                });
            }
        } else if (modalContent) {
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            const topPosition = scrollY + (viewportHeight - contentHeight) / 2;
            modalContent.style.top = `${topPosition}px`;
            
            // Center the modal in viewport
            modalContent.scrollIntoView({ 
                behavior: 'smooth', 
                block: 'center',
                inline: 'center'
            });
        }
        
        // Auto-focus search input
        const searchInput = document.getElementById('equipmentSearch');
        if (searchInput) {
            searchInput.focus();
        }
    }, 10);
}

function closeEquipmentModal() {
    const modal = document.getElementById('equipmentModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function updateEquipmentDisplay() {
    const container = document.getElementById('equipmentList');
    if (!container) return;
    
    container.innerHTML = '';
    
    if (equipmentList.length === 0) {
        container.innerHTML = '<div class="no-data">No equipment available</div>';
        return;
    }
    
    equipmentList.forEach(equipment => {
        const equipmentItem = document.createElement('div');
        equipmentItem.className = 'equipment-item';
        equipmentItem.innerHTML = `
            <div class="equipment-info">
                <div class="equipment-name">${equipment.name}</div>
                <div class="equipment-details">
                    <span class="equipment-type">${equipment.type}</span>
                    <span class="equipment-location">${equipment.location}</span>
                </div>
            </div>
            <div class="equipment-status ${equipment.status.toLowerCase()}">${equipment.status}</div>
        `;
        equipmentItem.onclick = () => selectEquipment(equipment);
        container.appendChild(equipmentItem);
    });
}

function filterEquipment() {
    const searchTerm = document.getElementById('equipmentSearch').value.toLowerCase();
    const equipmentListDiv = document.getElementById('equipmentList');
    
    equipmentListDiv.innerHTML = '';
    
    if (equipmentList.length === 0) {
        equipmentListDiv.innerHTML = '<div class="no-data">Loading equipment...</div>';
        return;
    }
    
    const filteredEquipment = equipmentList.filter(equipment => 
        equipment.name.toLowerCase().includes(searchTerm) ||
        equipment.type.toLowerCase().includes(searchTerm) ||
        equipment.location.toLowerCase().includes(searchTerm) ||
        (equipment.regNum && equipment.regNum.toLowerCase().includes(searchTerm)) ||
        (equipment.model && equipment.model.toLowerCase().includes(searchTerm)) ||
        (equipment.manufacturer && equipment.manufacturer.toLowerCase().includes(searchTerm)) ||
        (equipment.serialNumber && equipment.serialNumber.toLowerCase().includes(searchTerm)) ||
        (equipment.engineNumber && equipment.engineNumber.toLowerCase().includes(searchTerm))
    );
    
    if (filteredEquipment.length === 0) {
        equipmentListDiv.innerHTML = '<div class="no-data">No equipment found matching your search</div>';
        return;
    }
    
    filteredEquipment.forEach(equipment => {
        const equipmentItem = document.createElement('div');
        equipmentItem.className = 'equipment-item';
        equipmentItem.innerHTML = `
            <div class="equipment-info">
                <div class="equipment-header">
                    <div class="equipment-name">${equipment.name}</div>
                    <div class="equipment-status ${equipment.status.toLowerCase().replace(/\s+/g, '')}">${equipment.status}</div>
                </div>
                <div class="equipment-details">
                    <div class="equipment-row">
                        <span class="equipment-label">Type:</span>
                        <span class="equipment-value">${equipment.type}</span>
                    </div>
                    <div class="equipment-row">
                        <span class="equipment-label">Location:</span>
                        <span class="equipment-value">${equipment.location}</span>
                    </div>
                    ${equipment.regNum ? `
                    <div class="equipment-row">
                        <span class="equipment-label">Registration:</span>
                        <span class="equipment-value">${equipment.regNum}</span>
                    </div>
                    ` : ''}
                    ${equipment.model ? `
                    <div class="equipment-row">
                        <span class="equipment-label">Model:</span>
                        <span class="equipment-value">${equipment.model}</span>
                    </div>
                    ` : ''}
                    ${equipment.manufacturer ? `
                    <div class="equipment-row">
                        <span class="equipment-label">Manufacturer:</span>
                        <span class="equipment-value">${equipment.manufacturer}</span>
                    </div>
                    ` : ''}
                    ${equipment.serialNumber ? `
                    <div class="equipment-row">
                        <span class="equipment-label">Serial Number:</span>
                        <span class="equipment-value">${equipment.serialNumber}</span>
                    </div>
                    ` : ''}
                    ${equipment.engineNumber ? `
                    <div class="equipment-row">
                        <span class="equipment-label">Engine Number:</span>
                        <span class="equipment-value">${equipment.engineNumber}</span>
                    </div>
                    ` : ''}
                    ${equipment.meterReading ? `
                    <div class="equipment-row">
                        <span class="equipment-label">Meter Reading:</span>
                        <span class="equipment-value">${equipment.meterReading ? parseFloat(equipment.meterReading).toLocaleString() : '0'} ${equipment.meterUnit || ''}</span>
                    </div>
                    ` : ''}
                </div>
            </div>
        `;
        equipmentItem.onclick = () => selectEquipment(equipment);
        equipmentListDiv.appendChild(equipmentItem);
    });
}

// Equipment selection function
function selectEquipment(equipment) {
    selectedEquipment = equipment;
    equipmentId = equipment.id;
    currentMeterReading = equipment.meterReading;
    equipmentStatus = equipment.status;
    
    // Update the display
    document.getElementById('equipmentText').textContent = equipment.name;
    document.getElementById('equipmentId').value = equipment.id;
    document.getElementById('selectedEquipmentName').textContent = equipment.name;
    document.getElementById('currentMeterReading').textContent = equipment.meterReading || '0';
    document.getElementById('selectedEquipmentInfo').style.display = 'block';
    
    // Set equipment status radio button
    const statusValue = equipment.status.toLowerCase().replace(/\s+/g, '');
    const statusRadio = document.querySelector(`input[name="equipmentStatus"][value="${statusValue}"]`);
    if (statusRadio) {
        statusRadio.checked = true;
    }
    
    // Close modal
    closeEquipmentModal();
}

// Load equipment from Firebase
async function loadEquipmentFromFirebase() {
    try {
        console.log('Loading equipment from Firebase...');
        const response = await fetch('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData.json');
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const data = await response.json();
        console.log('Equipment data received:', data);
        
        // Process equipment data
        if (data && typeof data === 'object') {
            equipmentList = [];
            Object.keys(data).forEach(equipmentId => {
                const equipment = data[equipmentId];
                const equipmentData = {
                    id: equipmentId,
                    name: equipment.vehicleNumber || equipment.equipmentName || equipment.name || equipment.title || 'Unknown Equipment',
                    type: equipment.vehicleType || equipment.equipmentCategory || equipment.equipmentType || equipment.type || 'Unknown',
                    location: equipment.location || equipment.site || equipment.project || 'Unknown',
                    status: equipment.status || equipment.availability || 'Available',
                    regNum: equipment.registrationNumber || equipment.regNum || equipment.vehicleNumber || '',
                    model: equipment.model || equipment.equipmentModel || '',
                    manufacturer: equipment.manufacturer || equipment.make || equipment.brand || '',
                    serialNumber: equipment.serialNumber || equipment.serial || '',
                    engineNumber: equipment.engineNumber || equipment.engine || '',
                    meterReading: equipment.meterReading || equipment.hours || equipment.odometer || '0',
                    meterUnit: equipment.meterUnit || equipment.unit || 'Hours'
                };
                equipmentList.push(equipmentData);
            });
            
            console.log('Processed equipment list:', equipmentList);
            updateEquipmentDisplay();
        } else {
            console.error('Invalid data format received from Firebase');
            document.getElementById('equipmentList').innerHTML = '<div class="no-data">No equipment data available</div>';
        }
    } catch (error) {
        console.error('Error loading equipment from Firebase:', error);
        document.getElementById('equipmentList').innerHTML = '<div class="no-data">Error loading equipment. Please try again.</div>';
    }
}

function toggleCollapsible(sectionId) {
    const section = document.getElementById(sectionId);
    const icon = document.getElementById(sectionId.replace('Section', 'Icon'));
    
    if (section.style.display === 'none') {
        section.style.display = 'block';
        icon.textContent = '▲';
    } else {
        section.style.display = 'none';
        icon.textContent = '▼';
    }
}

// Debug function to log all form data and save to Firebase
async function debugFormData() {
    console.log('=== LOGBOOK FORM DEBUG DATA ===');
    console.log('Timestamp:', new Date().toISOString());
    console.log('Form ID:', 'logbookForm');
    console.log('');
    
    // Validate required fields first
    const validationResult = validateRequiredFields();
    if (!validationResult.isValid) {
        showNotification(validationResult.message, 'error');
        return;
    }
    
    // Basic Information
    console.log('--- BASIC INFORMATION ---');
    console.log('Entry Number:', document.getElementById('entryNumber').value);
    console.log('Country:', document.getElementById('country').value);
    console.log('Project:', document.getElementById('project').value);
    console.log('');
    
    // Equipment Information
    console.log('--- EQUIPMENT INFORMATION ---');
    console.log('Equipment ID:', document.getElementById('equipmentId').value);
    console.log('Selected Equipment Object:', selectedEquipment);
    console.log('Equipment Status:', document.querySelector('input[name="equipmentStatus"]:checked')?.value || 'None selected');
    console.log('Current Meter Reading:', currentMeterReading);
    console.log('');
    
    // Reading Information
    console.log('--- READING INFORMATION ---');
    console.log('Start Reading:', document.getElementById('startReading').value);
    console.log('End Reading:', document.getElementById('endReading').value);
    console.log('Start Date/Time:', document.getElementById('startDateTime').value);
    console.log('End Date/Time:', document.getElementById('endDateTime').value);
    console.log('');
    
    // Fuel Information
    console.log('--- FUEL INFORMATION ---');
    console.log('Fuel Opening:', document.getElementById('fuelOpening').value);
    console.log('Fuel Closing:', document.getElementById('fuelClosing').value);
    console.log('Fuel Unit:', document.getElementById('fuelUnit').value);
    console.log('Calculated Fuel Consumption:', calculateFuelConsumption());
    console.log('');
    
    // Work Information
    console.log('--- WORK INFORMATION ---');
    console.log('Operator Name:', document.getElementById('operatorName').value);
    console.log('Work Location:', document.getElementById('workLocation').value);
    console.log('Work Type:', document.getElementById('workType').value);
    console.log('Work Description:', document.getElementById('workDescription').value);
    console.log('');
    
    // Material Information
    console.log('--- MATERIAL INFORMATION ---');
    console.log('Material Name:', document.getElementById('materialName').value);
    console.log('Material Quantity:', document.getElementById('materialQuantity').value);
    console.log('Material Unit:', document.getElementById('materialUnit').value);
    console.log('');
    
    // Issue Report Information
    console.log('--- ISSUE REPORT INFORMATION ---');
    console.log('Issue Report:', document.getElementById('issueReport').value);
    console.log('Issue Name:', document.getElementById('issueName').value);
    console.log('Issue Details:', document.getElementById('issueDetails').value);
    console.log('');
    
    // Calculated Fields
    console.log('--- CALCULATED FIELDS ---');
    console.log('Fuel Consumption:', calculateFuelConsumption());
    console.log('Work Duration (hours):', calculateWorkDuration());
    console.log('Distance Traveled:', calculateDistanceTraveled());
    console.log('');
    
    // Form Validation Status
    console.log('--- FORM VALIDATION STATUS ---');
    const requiredFields = [
        'country', 'project', 'equipmentId', 'equipmentStatus',
        'startReading', 'endReading', 'startDateTime', 'endDateTime'
    ];
    
    let validationStatus = {};
    requiredFields.forEach(fieldName => {
        const field = document.getElementById(fieldName);
        const value = field ? field.value : '';
        const isValid = field && value.trim() !== '';
        validationStatus[fieldName] = {
            value: value,
            isValid: isValid,
            element: field ? field.tagName : 'Not found'
        };
    });
    console.log('Validation Status:', validationStatus);
    console.log('');
    
    // Generate unique ID for the entry
    const entryId = Date.now().toString();
    const currentTime = new Date().toISOString();
    
    // Complete Form Data Object in the required format
    const completeFormData = {
        // Basic Information
        entryNumber: document.getElementById('entryNumber').value,
        country: document.getElementById('country').value,
        project: document.getElementById('project').value,
        
        // Equipment Information
        equipmentId: document.getElementById('equipmentId').value,
        selectedEquipment: selectedEquipment ? selectedEquipment.name : '',
        equipmentStatus: document.querySelector('input[name="equipmentStatus"]:checked')?.value || '',
        currentMeterReading: currentMeterReading ? parseFloat(currentMeterReading) : 0,
        
        // Reading Information
        startReading: document.getElementById('startReading').value,
        endReading: document.getElementById('endReading').value,
        startDateTime: new Date(document.getElementById('startDateTime').value).toISOString(),
        endDateTime: new Date(document.getElementById('endDateTime').value).toISOString(),
        
        // Fuel Information
        fuelOpening: document.getElementById('fuelOpening').value || '',
        fuelClosing: document.getElementById('fuelClosing').value || '',
        fuelUnit: document.getElementById('fuelUnit').value,
        fuelConsumption: calculateFuelConsumption().toString(),
        
        // Work Information
        operatorName: document.getElementById('operatorName').value || '',
        workLocation: document.getElementById('workLocation').value || '',
        workType: document.getElementById('workType').value || '',
        workDescription: document.getElementById('workDescription').value || '',
        
        // Material Information
        materialName: document.getElementById('materialName').value || '',
        materialQuantity: document.getElementById('materialQuantity').value || '',
        materialUnit: document.getElementById('materialUnit').value,
        materialDescription: '', // This field doesn't exist in the form, keeping empty
        
        // Issue Report Information
        issueReport: document.getElementById('issueReport').value || '',
        issueName: document.getElementById('issueName').value || '',
        issueDetails: document.getElementById('issueDetails').value || '',
        
        // Metadata
        id: entryId,
        createdAt: currentTime,
        updatedAt: currentTime
    };
    
    console.log('--- COMPLETE FORM DATA OBJECT ---');
    console.log(completeFormData);
    console.log('');
    console.log('=== END DEBUG DATA ===');
    
    // Show loading notification
    showNotification('Saving data to Firebase database...', 'info');
    
    try {
        // Save to Firebase Realtime Database
        console.log('Saving to Firebase:', 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalLogbook/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json');
        
        const response = await fetch('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalLogbook/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(completeFormData)
        });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('Data saved successfully to Firebase:', result);
        
        // Show success modal after 1 second
        setTimeout(() => {
            showSuccessModal();
        }, 1000);
        
    } catch (error) {
        console.error('Error saving data to Firebase:', error);
        showNotification('Error saving data to Firebase: ' + error.message, 'error');
    }
}

// Validation function for required fields
function validateRequiredFields() {
    const errors = [];
    
    // Required field mappings
    const requiredFields = [
        { id: 'country', name: 'Country', element: document.getElementById('country') },
        { id: 'project', name: 'Project', element: document.getElementById('project') },
        { id: 'equipmentId', name: 'Equipment', element: document.getElementById('equipmentId') },
        { id: 'startReading', name: 'Start Reading', element: document.getElementById('startReading') },
        { id: 'endReading', name: 'End Reading', element: document.getElementById('endReading') },
        { id: 'startDateTime', name: 'Start Date & Time', element: document.getElementById('startDateTime') },
        { id: 'endDateTime', name: 'End Date & Time', element: document.getElementById('endDateTime') }
    ];
    
    // Check equipment status (radio buttons)
    const equipmentStatus = document.querySelector('input[name="equipmentStatus"]:checked');
    if (!equipmentStatus) {
        errors.push('Equipment Status is required');
    }
    
    // Check each required field
        requiredFields.forEach(field => {
        if (!field.element || !field.element.value.trim()) {
            errors.push(`${field.name} is required`);
            // Highlight the field in red
            if (field.element) {
                field.element.style.borderColor = '#dc3545';
                field.element.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
            }
        } else {
            // Reset field styling if valid
            if (field.element) {
                field.element.style.borderColor = '#ced4da';
                field.element.style.boxShadow = 'none';
            }
        }
    });
    
    // Validate date/time logic
    const startDateTime = document.getElementById('startDateTime').value;
    const endDateTime = document.getElementById('endDateTime').value;
    
    if (startDateTime && endDateTime) {
        const startDate = new Date(startDateTime);
        const endDate = new Date(endDateTime);
        
        if (endDate <= startDate) {
            errors.push('End Date & Time must be after Start Date & Time');
            document.getElementById('endDateTime').style.borderColor = '#dc3545';
            document.getElementById('endDateTime').style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
        } else {
            document.getElementById('endDateTime').style.borderColor = '#ced4da';
            document.getElementById('endDateTime').style.boxShadow = 'none';
        }
    }
    
    // Validate reading values
    const startReading = document.getElementById('startReading').value;
    const endReading = document.getElementById('endReading').value;
    
    if (startReading && endReading) {
        const start = parseFloat(startReading);
        const end = parseFloat(endReading);
        
        if (isNaN(start) || isNaN(end)) {
            errors.push('Start Reading and End Reading must be valid numbers');
        } else if (end < start) {
            errors.push('End Reading must be greater than or equal to Start Reading');
            document.getElementById('endReading').style.borderColor = '#dc3545';
            document.getElementById('endReading').style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
        } else {
            document.getElementById('endReading').style.borderColor = '#ced4da';
            document.getElementById('endReading').style.boxShadow = 'none';
        }
    }
    
    // Validate fuel values if provided
    const fuelOpening = document.getElementById('fuelOpening').value;
    const fuelClosing = document.getElementById('fuelClosing').value;
    
    if (fuelOpening || fuelClosing) {
        const opening = parseFloat(fuelOpening) || 0;
        const closing = parseFloat(fuelClosing) || 0;
        
        if (isNaN(parseFloat(fuelOpening)) && fuelOpening) {
            errors.push('Fuel Opening must be a valid number');
            document.getElementById('fuelOpening').style.borderColor = '#dc3545';
        } else {
            document.getElementById('fuelOpening').style.borderColor = '#ced4da';
        }
        
        if (isNaN(parseFloat(fuelClosing)) && fuelClosing) {
            errors.push('Fuel Closing must be a valid number');
            document.getElementById('fuelClosing').style.borderColor = '#dc3545';
        } else {
            document.getElementById('fuelClosing').style.borderColor = '#ced4da';
        }
        
        if (closing > opening && fuelOpening && fuelClosing) {
            errors.push('Fuel Closing cannot be greater than Fuel Opening');
            document.getElementById('fuelClosing').style.borderColor = '#dc3545';
            document.getElementById('fuelClosing').style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
        }
    }
    
    // Check if equipment is selected
    if (!selectedEquipment || !equipmentId) {
        errors.push('Please select an equipment from the equipment list');
        // Highlight equipment select area
        const equipmentSelect = document.querySelector('.equipment-select');
        if (equipmentSelect) {
            equipmentSelect.style.borderColor = '#dc3545';
            equipmentSelect.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
        }
    } else {
        const equipmentSelect = document.querySelector('.equipment-select');
        if (equipmentSelect) {
            equipmentSelect.style.borderColor = '#ced4da';
            equipmentSelect.style.boxShadow = 'none';
        }
    }
    
    return {
        isValid: errors.length === 0,
        message: errors.length > 0 ? errors.join('. ') + '.' : 'All required fields are valid'
    };
}

// Real-time field validation function
function validateField(field) {
    const fieldId = field.id;
    const fieldValue = field.value.trim();
    
    // Reset styling
    field.style.borderColor = '#ced4da';
    field.style.boxShadow = 'none';
    
    // Required field validation
    const requiredFields = ['country', 'project', 'equipmentId', 'startReading', 'endReading', 'startDateTime', 'endDateTime'];
    
    if (requiredFields.includes(fieldId) && !fieldValue) {
        field.style.borderColor = '#dc3545';
        field.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
        return false;
    }
    
    // Numeric validation for reading fields
    if ((fieldId === 'startReading' || fieldId === 'endReading') && fieldValue) {
        const numValue = parseFloat(fieldValue);
        if (isNaN(numValue) || numValue < 0) {
            field.style.borderColor = '#dc3545';
            field.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
            return false;
        }
    }
    
    // Numeric validation for fuel fields
    if ((fieldId === 'fuelOpening' || fieldId === 'fuelClosing') && fieldValue) {
        const numValue = parseFloat(fieldValue);
        if (isNaN(numValue) || numValue < 0) {
            field.style.borderColor = '#dc3545';
            field.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
            return false;
        }
    }
    
    // Date validation
    if ((fieldId === 'startDateTime' || fieldId === 'endDateTime') && fieldValue) {
        const dateValue = new Date(fieldValue);
        if (isNaN(dateValue.getTime())) {
            field.style.borderColor = '#dc3545';
            field.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
            return false;
        }
    }
    
    // Cross-field validation for dates
    if (fieldId === 'endDateTime' && fieldValue) {
        const startDateTime = document.getElementById('startDateTime').value;
        if (startDateTime) {
            const startDate = new Date(startDateTime);
            const endDate = new Date(fieldValue);
            if (endDate <= startDate) {
                field.style.borderColor = '#dc3545';
                field.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
                return false;
            }
        }
    }
    
    // Cross-field validation for readings
    if (fieldId === 'endReading' && fieldValue) {
        const startReading = document.getElementById('startReading').value;
        if (startReading) {
            const start = parseFloat(startReading);
            const end = parseFloat(fieldValue);
            if (!isNaN(start) && !isNaN(end) && end < start) {
                field.style.borderColor = '#dc3545';
                field.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
                return false;
            }
        }
    }
    
    // Cross-field validation for fuel
    if (fieldId === 'fuelClosing' && fieldValue) {
        const fuelOpening = document.getElementById('fuelOpening').value;
        if (fuelOpening) {
            const opening = parseFloat(fuelOpening);
            const closing = parseFloat(fieldValue);
            if (!isNaN(opening) && !isNaN(closing) && closing > opening) {
                field.style.borderColor = '#dc3545';
                field.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
                return false;
            }
        }
    }
    
    return true;
}

// Helper functions for calculations
function calculateFuelConsumption() {
    const opening = parseFloat(document.getElementById('fuelOpening').value) || 0;
    const closing = parseFloat(document.getElementById('fuelClosing').value) || 0;
    return Math.max(0, opening - closing);
}

function calculateWorkDuration() {
    const startTime = new Date(document.getElementById('startDateTime').value);
    const endTime = new Date(document.getElementById('endDateTime').value);
    
    if (isNaN(startTime.getTime()) || isNaN(endTime.getTime())) {
        return 0;
    }
    
    const durationMs = endTime.getTime() - startTime.getTime();
    return Math.max(0, durationMs / (1000 * 60 * 60)); // Convert to hours
}

function calculateDistanceTraveled() {
    const startReading = parseFloat(document.getElementById('startReading').value) || 0;
    const endReading = parseFloat(document.getElementById('endReading').value) || 0;
    return Math.max(0, endReading - startReading);
}

function saveLogBookEntry() {
    // Validate required fields
    const requiredFields = [
        'country', 'project', 'equipment', 'equipmentStatus',
        'startReading', 'endReading', 'startDateTime', 'endDateTime'
    ];
    
    let isValid = true;
    requiredFields.forEach(fieldName => {
        const field = document.getElementById(fieldName);
        if (!field || !field.value.trim()) {
                isValid = false;
            if (field) {
                field.style.borderColor = '#dc3545';
            }
            } else {
            if (field) {
                field.style.borderColor = '#ced4da';
            }
        }
    });
    
    // Validate date/time
    const startDateTime = new Date(document.getElementById('startDateTime').value);
    const endDateTime = new Date(document.getElementById('endDateTime').value);
    
    if (endDateTime <= startDateTime) {
        showNotification('End Date and Time must be after Start Date and Time.', 'error');
        return;
    }
        
        if (!isValid) {
        showNotification('Please fill in all required fields.', 'error');
            return;
        }
        
    // Show sync modal
    showSyncModal();
    
    // Simulate saving (replace with actual API call)
        setTimeout(() => {
        hideSyncModal();
            showNotification('Logbook entry saved successfully!', 'success');
            
            // Reset form
        document.getElementById('logbookForm').reset();
        initializeForm();
        document.getElementById('selectedEquipmentInfo').style.display = 'none';
        selectedEquipment = null;
        equipmentId = null;
        currentMeterReading = null;
            
            // Redirect to logbook page after 2 seconds
            setTimeout(() => {
                window.location.href = '{{ route('logbook') }}';
            }, 2000);
    }, 2000);
}

function showSyncModal() {
    document.getElementById('syncModal').style.display = 'flex';
}

function hideSyncModal() {
    document.getElementById('syncModal').style.display = 'none';
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span class="notification-message">${message}</span>
            <button class="notification-close" onclick="this.parentElement.parentElement.remove()">×</button>
        </div>
    `;
    
    // Add styles
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#d4edda' : type === 'error' ? '#f8d7da' : '#d1ecf1'};
        color: ${type === 'success' ? '#155724' : type === 'error' ? '#721c24' : '#0c5460'};
        border: 1px solid ${type === 'success' ? '#c3e6cb' : type === 'error' ? '#f5c6cb' : '#bee5eb'};
        border-radius: 6px;
        padding: 12px 16px;
        z-index: 9999;
        max-width: 400px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        animation: slideIn 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentElement) {
            notification.remove();
        }
    }, 5000);
}

// Add animation styles
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    .notification-content {
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 12px;
    }
    
    .notification-close {
        background: none;
        border: none;
        font-size: 18px;
        cursor: pointer;
        padding: 0;
        line-height: 1;
    }
`;
document.head.appendChild(style);

// Success Modal Functions (matching Create New Workorder file)
function showSuccessModal() {
    const successModal = document.getElementById('successModal');
    if (successModal) {
        successModal.style.display = 'flex';
        successModal.style.opacity = '0';
        successModal.style.transition = 'opacity 0.3s ease';
        
        // Fade in the success modal
        setTimeout(() => {
            successModal.style.opacity = '1';
        }, 10);
    }
}

function hideSuccessModal() {
    const successModal = document.getElementById('successModal');
    if (successModal) {
        successModal.style.display = 'none';
    }
}

function closeSuccessModal() {
    document.getElementById('successModal').style.display = 'none';
}

// Go back to Logbook list
function goBackToLogbook() {
    closeSuccessModal();
    // Navigate to logbook page
    window.location.href = "{{ route('logbook') }}";
}

// Create new entry
function createNewEntry() {
    closeSuccessModal();
    // Reset form for new entry
    document.getElementById('logbookForm').reset();
    initializeForm();
    document.getElementById('selectedEquipmentInfo').style.display = 'none';
    selectedEquipment = null;
    equipmentId = null;
    currentMeterReading = null;
    // Scroll to top
    window.scrollTo({ top: 0, behavior: 'smooth' });
}
</script>
@endsection
