@extends('welcome')

@section('title', 'Appeal Applications')

@section('styles')
<meta name="csrf-token" content="{{ csrf_token() }}">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.applications-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
    text-decoration: none;
}

.header-title {
    color: white;
    font-size: 24px;
    font-weight: 600;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    text-decoration: none;
    color: var(--primary-color);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-title i {
    color: var(--primary-color);
}

/* Applications List */
.applications-list {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.application-card {
    background: white;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    cursor: pointer;
    transition: all 0.2s ease;
}

.application-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.card-content {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.applicant-info h3 {
    font-size: 18px;
    font-weight: 600;
    color: #1f2937;
    margin: 0 0 4px 0;
}

.applicant-info p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

.appeal-info p {
    font-size: 14px;
    color: #374151;
    margin: 0;
    font-weight: 500;
}

.card-footer {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-top: 8px;
}

.status {
    font-size: 14px;
    font-weight: 600;
    padding: 4px 12px;
    border-radius: 20px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status.pending {
    background: #fef3c7;
    color: #d97706;
}

.status.approved {
    background: #d1fae5;
    color: #059669;
}

.status.rejected {
    background: #fee2e2;
    color: #dc2626;
}

/* Loading State */
.loading-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 20px;
    text-align: center;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #e5e7eb;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin-bottom: 16px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-state p {
    color: #6b7280;
    font-size: 16px;
    margin: 0;
}

/* Empty State */
.empty-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 20px;
    text-align: center;
}

.empty-icon {
    color: #9ca3af;
    margin-bottom: 16px;
}

.empty-state h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0 0 8px 0;
}

.empty-state p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
}

.btn-menu {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    background: #f3f4f6;
    color: #6b7280;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-menu:hover {
    background: #e5e7eb;
    color: #374151;
}

/* Responsive Design */
@media (max-width: 768px) {
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 20px;
    }
    
    .application-card {
        padding: 16px;
    }
    
    .applicant-info h3 {
        font-size: 16px;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        gap: 15px;
        align-items: flex-start;
    }
    
    .header-right {
        width: 100%;
        justify-content: flex-end;
    }
    
    .main-content {
        padding: 15px 10px;
    }
    
    .form-section {
        padding: 15px;
    }
    
    .application-card {
        padding: 12px;
    }
    
    .applicant-info h3 {
        font-size: 15px;
    }
    
    .applicant-info p,
    .appeal-info p {
        font-size: 13px;
    }
    
    .status {
        font-size: 12px;
        padding: 3px 8px;
    }
}
</style>
@endsection

@section('content')
<div class="applications-container">
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('hr.admin.dashboard') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Appeal Applications</h1>
            </div>
            <div class="header-right">
                <a href="{{ route('hr.create.application') }}" class="create-btn">
                    <i class="fas fa-plus"></i>
                    New Appeal
                </a>
            </div>
        </div>
    </div>

    <div class="main-content">
        <div class="form-container">
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-file-alt"></i>
                    Applications
                </h2>

                <div class="applications-list" id="applicationsList">
                    <!-- Loading State -->
                    <div class="loading-state" id="loadingState">
                        <div class="loading-spinner"></div>
                        <p>Loading appeal applications...</p>
                    </div>
                    
                    <!-- Empty State -->
                    <div class="empty-state" id="emptyState" style="display: none;">
                        <div class="empty-icon">
                            <i class="fas fa-file-alt" style="font-size: 48px;"></i>
                        </div>
                        <h3>No Appeal Applications</h3>
                        <p>There are no appeal applications to display at the moment.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    loadAppealApplications();
});

// Load appeal applications from MongoDB
async function loadAppealApplications() {
    try {
        console.log('Loading appeal applications from MongoDB...');
        
        const response = await fetch('/api/hr-applications/application_for_appeal');
        
        if (!response.ok) {
            throw new Error('Failed to fetch appeal applications');
        }
        
        const result = await response.json();
        console.log('MongoDB data received:', result);
        
        // Hide loading state
        document.getElementById('loadingState').style.display = 'none';
        
        if (!result.success || !result.data || result.data.length === 0) {
            document.getElementById('emptyState').style.display = 'block';
            updateHomePageCounts([]);
            return;
        }
        
        // Update home page counts
        updateHomePageCounts(result.data);
        
        // Display applications
        displayApplications(result.data);
        
    } catch (error) {
        console.error('Error loading appeal applications:', error);
        document.getElementById('loadingState').style.display = 'none';
        showNotification('Failed to load appeal applications. Please try again.', 'error');
    }
}

// Display applications in the UI
function displayApplications(applications) {
    const applicationsList = document.getElementById('applicationsList');
    
    // Handle both array and object formats
    const applicationsArray = Array.isArray(applications) 
        ? applications 
        : Object.entries(applications).map(([key, value]) => ({ id: key, ...value }));
    
    // Store applications in sessionStorage for later use
    const applicationsObject = {};
    applicationsArray.forEach(app => {
        applicationsObject[app.id || app._id] = app;
    });
    sessionStorage.setItem('appealApplications', JSON.stringify(applicationsObject));
    
    // Clear existing content
    applicationsList.innerHTML = '';
    
    // Sort by submission date
    const sortedApplications = applicationsArray.sort((a, b) => 
        new Date(b.submittedAt || b.submissionDate || b.createdAt) - new Date(a.submittedAt || a.submissionDate || a.createdAt)
    );
    
    console.log('Applications array:', sortedApplications);
    
    sortedApplications.forEach(application => {
        const applicationCard = createApplicationCard(application);
        applicationsList.appendChild(applicationCard);
    });
    
    // Add event listeners to new cards
    addCardEventListeners();
}

// Create application card element
function createApplicationCard(application) {
    const card = document.createElement('div');
    card.className = 'application-card';
    card.dataset.applicationId = application.id || application._id;
    
    // Format submission date
    const submissionDate = new Date(application.submittedAt || application.submissionDate || application.createdAt).toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
    
    // Get appeal details
    const appealFormDate = application.appealFormDate ? new Date(application.appealFormDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'N/A';
    const appealSanction = application.appealSanction || 'N/A';
    const appealOffence = application.appealOffence || 'N/A';
    const appealGrounds = (application.appealGrounds || 'N/A').substring(0, 100) + ((application.appealGrounds && application.appealGrounds.length > 100) ? '...' : '');
    
    card.innerHTML = `
        <div class="card-content">
            <div class="applicant-info">
                <h3 class="applicant-name">${application.fullName || application.name || application.applicantName || 'N/A'}</h3>
                <p class="employee-no">Employee No: ${application.employeeNo || application.employeeNumber || 'N/A'}</p>
                <p class="submission-date">Submitted: ${submissionDate}</p>
            </div>
            <div class="appeal-info">
                <p class="appeal-details">
                    <strong>Appeal Date:</strong> ${appealFormDate}<br>
                    <strong>Sanction:</strong> ${appealSanction}<br>
                    <strong>Offence:</strong> ${appealOffence}<br>
                    <strong>Grounds:</strong> ${appealGrounds}<br>
                    <strong>Department:</strong> ${application.department || 'N/A'} | <strong>Project:</strong> ${application.project || 'N/A'}
                </p>
            </div>
            <div class="card-footer">
                <span class="status ${application.status ? application.status.toLowerCase() : 'pending'}">Status: ${application.status || 'Pending'}</span>
                <button class="btn-menu">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <circle cx="12" cy="12" r="1"/>
                        <circle cx="12" cy="5" r="1"/>
                        <circle cx="12" cy="19" r="1"/>
                    </svg>
                </button>
            </div>
        </div>
    `;
    
    return card;
}

// Add event listeners to application cards
function addCardEventListeners() {
    const applicationCards = document.querySelectorAll('.application-card');
    
    applicationCards.forEach(card => {
        card.addEventListener('click', function(e) {
            if (e.target.closest('.btn-menu')) return;
            
            const applicationId = this.dataset.applicationId;
            
            // Navigate to dedicated application detail page
            window.location.href = `/application-detail?id=${applicationId}&type=application_for_appeal`;
        });
    });

    const menuButtons = document.querySelectorAll('.btn-menu');
    
    menuButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.stopPropagation();
            const card = this.closest('.application-card');
            const applicationId = card.dataset.applicationId;
            showActionMenu(this, applicationId);
        });
    });
}

// Update home page counts
function updateHomePageCounts(applications) {
    try {
        const applicationsArray = Array.isArray(applications) ? applications : Object.values(applications);
        
        const pendingCount = applicationsArray.filter(app => 
            (app.status || 'pending').toLowerCase() === 'pending'
        ).length;
        
        const approvedCount = applicationsArray.filter(app => 
            (app.status || '').toLowerCase() === 'approved'
        ).length;
        
        const rejectedCount = applicationsArray.filter(app => 
            (app.status || '').toLowerCase() === 'rejected'
        ).length;
        
        const totalCount = applicationsArray.length;
        
        console.log('Appeal Applications Counts:', {
            total: totalCount,
            pending: pendingCount,
            approved: approvedCount,
            rejected: rejectedCount
        });
        
        localStorage.setItem('appealApplicationsCount', JSON.stringify({
            total: totalCount,
            pending: pendingCount,
            approved: approvedCount,
            rejected: rejectedCount,
            lastUpdated: new Date().toISOString()
        }));
        
        if (window.opener && !window.opener.closed) {
            window.opener.postMessage({
                type: 'appealApplicationsUpdate',
                counts: {
                    total: totalCount,
                    pending: pendingCount,
                    approved: approvedCount,
                    rejected: rejectedCount
                }
            }, '*');
        }
        
    } catch (error) {
        console.error('Error updating home page counts:', error);
    }
}

let currentOpenMenu = null;

function showActionMenu(button, applicationId) {
    if (currentOpenMenu) {
        currentOpenMenu.remove();
        currentOpenMenu = null;
    }

    const menu = document.createElement('div');
    menu.className = 'action-menu';
    menu.innerHTML = `
        <div class="action-menu-item" data-action="view"><i class="fas fa-eye"></i> View Details</div>
        <div class="action-menu-item" data-action="approve"><i class="fas fa-check"></i> Approve</div>
        <div class="action-menu-item" data-action="reject"><i class="fas fa-times"></i> Reject</div>
        <div class="action-menu-item action-menu-item-danger" data-action="delete"><i class="fas fa-trash"></i> Delete</div>
    `;

    const buttonRect = button.getBoundingClientRect();
    menu.style.position = 'fixed';
    menu.style.top = `${buttonRect.bottom + 5}px`;
    menu.style.left = `${buttonRect.left - 150}px`;
    
    document.body.appendChild(menu);
    currentOpenMenu = menu;

    menu.querySelectorAll('.action-menu-item').forEach(item => {
        item.addEventListener('click', (e) => {
            e.stopPropagation();
            handleMenuAction(item.dataset.action, applicationId);
            menu.remove();
            currentOpenMenu = null;
        });
    });

    setTimeout(() => {
        document.addEventListener('click', function closeMenu(e) {
            if (menu && !menu.contains(e.target)) {
                menu.remove();
                currentOpenMenu = null;
                document.removeEventListener('click', closeMenu);
            }
        });
    }, 10);
}

async function handleMenuAction(action, applicationId) {
    const applications = JSON.parse(sessionStorage.getItem('appealApplications') || '{}');
    const application = applications[applicationId];

    if (!application) {
        showNotification('Application not found', 'error');
        return;
    }

    switch(action) {
        case 'view':
            showApplicationDetails(application);
            break;
        case 'approve':
            showConfirmDialog('Approve Application', 'Are you sure you want to approve this appeal?', () => {
                updateApplicationStatus(applicationId, 'approved');
            });
            break;
        case 'reject':
            showConfirmDialog('Reject Application', 'Are you sure you want to reject this appeal?', () => {
                updateApplicationStatus(applicationId, 'rejected');
            });
            break;
        case 'delete':
            showConfirmDialog('Delete Application', 'Are you sure you want to delete this application? This action cannot be undone.', () => {
                deleteApplication(applicationId);
            }, true);
            break;
    }
}

async function updateApplicationStatus(applicationId, status) {
    try {
        const statusData = {
            status: status,
            comments: '',
            approverName: 'Admin',
            approverRole: 'ProAdmin',
            reason: `${status} by admin`,
            timestamp: new Date().toISOString()
        };
        
        const response = await fetch(`/api/hr-applications/${applicationId}/status`, {
            method: 'PUT',
            headers: { 
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            },
            body: JSON.stringify(statusData)
        });

        if (!response.ok) {
            throw new Error('Failed to update status');
        }
        
        const result = await response.json();
        if (result.success) {
            showNotification(`Application ${status} successfully`, 'success');
            loadAppealApplications();
        } else {
            throw new Error(result.error || 'Failed to update status');
        }
    } catch (error) {
        console.error('Error updating application status:', error);
        showNotification('Failed to update application status', 'error');
    }
}

async function deleteApplication(applicationId) {
    try {
        const response = await fetch(`/api/hr-applications/${applicationId}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            }
        });

        if (!response.ok) {
            throw new Error('Failed to delete application');
        }
        
        const result = await response.json();
        if (result.success) {
            showNotification('Application deleted successfully', 'success');
            loadAppealApplications();
        } else {
            throw new Error(result.error || 'Failed to delete application');
        }
    } catch (error) {
        console.error('Error deleting application:', error);
        showNotification('Failed to delete application', 'error');
    }
}

function showApplicationDetails(application) {
    // Get appeal details
    const appealFormDate = application.appealFormDate ? new Date(application.appealFormDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'N/A';
    const appealOffenceDate = application.appealOffenceDate ? new Date(application.appealOffenceDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'N/A';
    const appealName = application.appealName || 'N/A';
    const appealManNo = application.appealManNo || 'N/A';
    const appealSanction = application.appealSanction || 'N/A';
    const appealOffence = application.appealOffence || 'N/A';
    const appealGrounds = application.appealGrounds || 'N/A';
    
    const modal = document.createElement('div');
    modal.className = 'details-modal';
    modal.innerHTML = `
        <div class="details-modal-content">
            <div class="details-modal-header">
                <h2>Appeal Application Details</h2>
                <button class="details-modal-close">&times;</button>
            </div>
            <div class="details-modal-body">
                <div class="detail-row"><span class="detail-label">Application No:</span><span class="detail-value">${application.applicationNumber || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Full Name:</span><span class="detail-value">${application.fullName || application.name || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Employee No:</span><span class="detail-value">${application.employeeNo || application.employeeNumber || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Department:</span><span class="detail-value">${application.department || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Country:</span><span class="detail-value">${application.country || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Project:</span><span class="detail-value">${application.project || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">NRC No:</span><span class="detail-value">${application.nrcNo || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Appeal Form Date:</span><span class="detail-value">${appealFormDate}</span></div>
                <div class="detail-row"><span class="detail-label">Man No:</span><span class="detail-value">${appealManNo}</span></div>
                <div class="detail-row"><span class="detail-label">Appellant Name:</span><span class="detail-value">${appealName}</span></div>
                <div class="detail-row"><span class="detail-label">Sanction:</span><span class="detail-value">${appealSanction}</span></div>
                <div class="detail-row"><span class="detail-label">Offence:</span><span class="detail-value">${appealOffence}</span></div>
                <div class="detail-row"><span class="detail-label">Offence Date:</span><span class="detail-value">${appealOffenceDate}</span></div>
                <div class="detail-row"><span class="detail-label">Grounds for Appeal:</span><span class="detail-value" style="white-space: pre-wrap;">${appealGrounds}</span></div>
                <div class="detail-row"><span class="detail-label">Status:</span><span class="detail-value"><span class="status ${(application.status || 'pending').toLowerCase()}">${application.status || 'Pending'}</span></span></div>
                <div class="detail-row"><span class="detail-label">Submitted At:</span><span class="detail-value">${new Date(application.submittedAt || application.submissionDate || application.createdAt).toLocaleString('en-US', { year: 'numeric', month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit' })}</span></div>
            </div>
        </div>
    `;

    document.body.appendChild(modal);
    modal.querySelector('.details-modal-close').addEventListener('click', () => modal.remove());
    modal.addEventListener('click', (e) => {
        if (e.target === modal) modal.remove();
    });
}

function showConfirmDialog(title, message, onConfirm, isDanger = false) {
    const dialog = document.createElement('div');
    dialog.className = 'confirm-dialog';
    dialog.innerHTML = `
        <div class="confirm-dialog-content">
            <h3>${title}</h3>
            <p>${message}</p>
            <div class="confirm-dialog-buttons">
                <button class="btn-cancel">Cancel</button>
                <button class="btn-confirm ${isDanger ? 'btn-danger' : ''}">${isDanger ? 'Delete' : 'Confirm'}</button>
            </div>
        </div>
    `;

    document.body.appendChild(dialog);
    dialog.querySelector('.btn-cancel').addEventListener('click', () => dialog.remove());
    dialog.querySelector('.btn-confirm').addEventListener('click', () => {
        onConfirm();
        dialog.remove();
    });
    dialog.addEventListener('click', (e) => {
        if (e.target === dialog) dialog.remove();
    });
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `app-notification app-notification-${type}`;
    notification.textContent = message;
    
    notification.style.cssText = `
        position: fixed;
        top: 80px;
        right: 20px;
        padding: 12px 20px;
        border-radius: 8px;
        color: white;
        font-weight: 500;
        font-size: 14px;
        z-index: 10001;
        opacity: 0;
        transform: translateX(100%);
        transition: all 0.3s ease;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        font-family: 'Inter', sans-serif;
    `;
    
    if (type === 'success') {
        notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
    } else if (type === 'error') {
        notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
    } else if (type === 'warning') {
        notification.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
    } else {
        notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
    }
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.opacity = '1';
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    setTimeout(() => {
        notification.style.opacity = '0';
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (notification.parentNode) {
                document.body.removeChild(notification);
            }
        }, 300);
    }, 3000);
}
</script>

<style>
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

.applications-container {
    min-height: 100vh;
}

.applications-header {
    background: #0055a5;
    padding: 16px 20px;
    box-shadow: 0 2px 8px rgba(0, 85, 165, 0.3);
}

.header-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.btn-back {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: rgba(255, 255, 255, 0.1);
    color: white;
    text-decoration: none;
    border-radius: 8px;
    transition: all 0.2s ease;
    border: 1px solid rgba(255, 255, 255, 0.2);
}

.btn-back:hover {
    background: rgba(255, 255, 255, 0.2);
    color: white;
    text-decoration: none;
    transform: translateY(-1px);
}

.page-title h1 {
    color: white;
    font-size: 20px;
    font-weight: 600;
    margin: 0;
    text-align: center;
    flex: 1;
}

.btn-notification {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: rgba(255, 255, 255, 0.1);
    color: white;
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-notification:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
}

.applications-content {
    padding: 20px;
}

.section-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 20px;
}

.section-header h2 {
    color: #0055a5;
    font-size: 24px;
    font-weight: 700;
    margin: 0;
}

.btn-lock {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: #f3f4f6;
    color: #6b7280;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-lock:hover {
    background: #e5e7eb;
    color: #374151;
}

.applications-list {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.application-card {
    background: white;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    cursor: pointer;
    transition: all 0.2s ease;
}

.application-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.card-content {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.applicant-info h3 {
    font-size: 18px;
    font-weight: 600;
    color: #1f2937;
    margin: 0 0 4px 0;
}

.applicant-info p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

.appeal-info p {
    font-size: 14px;
    color: #374151;
    margin: 0;
    font-weight: 500;
}

.card-footer {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-top: 8px;
}

.status {
    font-size: 14px;
    font-weight: 600;
    padding: 4px 12px;
    border-radius: 20px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status.pending {
    background: #fef3c7;
    color: #d97706;
}

.status.approved {
    background: #d1fae5;
    color: #059669;
}

.status.rejected {
    background: #fee2e2;
    color: #dc2626;
}

/* Loading State */
.loading-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 20px;
    text-align: center;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #e5e7eb;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin-bottom: 16px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-state p {
    color: #6b7280;
    font-size: 16px;
    margin: 0;
}

/* Empty State */
.empty-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 20px;
    text-align: center;
}

.empty-icon {
    color: #9ca3af;
    margin-bottom: 16px;
}

.empty-state h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0 0 8px 0;
}

.empty-state p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
}

.btn-menu {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    background: #f3f4f6;
    color: #6b7280;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-menu:hover {
    background: #e5e7eb;
    color: #374151;
}

@media (max-width: 768px) {
    .applications-content {
        padding: 16px;
    }
    
    .section-header h2 {
        font-size: 20px;
    }
    
    .application-card {
        padding: 16px;
    }
    
    .applicant-info h3 {
        font-size: 16px;
    }
}

@media (max-width: 576px) {
    .applications-header {
        padding: 12px 16px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .applications-content {
        padding: 12px;
    }
    
    .section-header h2 {
        font-size: 18px;
    }
    
    .application-card {
        padding: 12px;
    }
    
    .applicant-info h3 {
        font-size: 15px;
    }
    
    .applicant-info p,
    .appeal-info p {
        font-size: 13px;
    }
    
    .status {
        font-size: 12px;
        padding: 3px 8px;
    }
}

/* Action Menu */
.action-menu {
    position: fixed;
    background: white;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    min-width: 180px;
    z-index: 10000;
    overflow: hidden;
    animation: slideIn 0.2s ease;
}

@keyframes slideIn {
    from { opacity: 0; transform: translateY(-10px); }
    to { opacity: 1; transform: translateY(0); }
}

.action-menu-item {
    padding: 12px 16px;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 10px;
    color: #374151;
    font-size: 14px;
    transition: all 0.2s ease;
}

.action-menu-item:hover { background: #f3f4f6; }
.action-menu-item i { width: 16px; font-size: 14px; }
.action-menu-item-danger { color: #dc2626; }
.action-menu-item-danger:hover { background: #fee2e2; }

/* Details Modal */
.details-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10002;
    padding: 20px;
}

.details-modal-content {
    background: white;
    border-radius: 12px;
    max-width: 600px;
    width: 100%;
    max-height: 80vh;
    overflow-y: auto;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
}

.details-modal-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 20px;
    border-bottom: 1px solid #e5e7eb;
}

.details-modal-header h2 {
    margin: 0;
    font-size: 20px;
    font-weight: 600;
    color: #0055a5;
}

.details-modal-close {
    background: none;
    border: none;
    font-size: 28px;
    color: #6b7280;
    cursor: pointer;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 4px;
    transition: all 0.2s ease;
}

.details-modal-close:hover { background: #f3f4f6; color: #374151; }
.details-modal-body { padding: 20px; }
.detail-row { display: flex; padding: 12px 0; border-bottom: 1px solid #f3f4f6; }
.detail-row:last-child { border-bottom: none; }
.detail-label { font-weight: 600; color: #374151; min-width: 150px; font-size: 14px; }
.detail-value { color: #6b7280; flex: 1; font-size: 14px; }

/* Confirm Dialog */
.confirm-dialog {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10003;
    padding: 20px;
}

.confirm-dialog-content {
    background: white;
    border-radius: 12px;
    padding: 24px;
    max-width: 400px;
    width: 100%;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
}

.confirm-dialog-content h3 { margin: 0 0 12px 0; font-size: 18px; font-weight: 600; color: #1f2937; }
.confirm-dialog-content p { margin: 0 0 20px 0; color: #6b7280; font-size: 14px; line-height: 1.5; }
.confirm-dialog-buttons { display: flex; gap: 12px; justify-content: flex-end; }

.btn-cancel, .btn-confirm {
    padding: 10px 20px;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    border: none;
}

.btn-cancel { background: #f3f4f6; color: #374151; }
.btn-cancel:hover { background: #e5e7eb; }
.btn-confirm { background: #0055a5; color: white; }
.btn-confirm:hover { background: #003f7d; }
.btn-confirm.btn-danger { background: #dc2626; }
.btn-confirm.btn-danger:hover { background: #b91c1c; }

@media (max-width: 640px) {
    .details-modal-content { max-width: 100%; }
    .detail-row { flex-direction: column; gap: 4px; }
    .detail-label { min-width: auto; }
}
</style>

@endsection
