@extends('welcome')

@section('title', 'Dynamic Form Demo')

@section('content')
<div class="content-box">
    <div class="row">
        <div class="col-12 col-lg-8 mx-auto">
            <div class="card">
                <div class="card-header">
                    <h4 class="mb-0">
                        <i class="fas fa-magic me-2"></i>Dynamic Form Demo
                    </h4>
                    <p class="text-muted mb-0">This page demonstrates dynamic form handling with AJAX</p>
                </div>
                <div class="card-body">
                    <!-- Dynamic Form Example -->
                    <form data-dynamic action="#" method="POST" class="mb-4" novalidate role="form" aria-label="Dynamic form demo">
                        @csrf
                        <div class="row">
                            <div class="col-12 col-md-6 mb-3">
                                <label for="name" class="form-label" aria-required="true">Name</label>
                                <input type="text" 
                                       class="form-control" 
                                       id="name" 
                                       name="name" 
                                       required 
                                       autocomplete="name"
                                       aria-describedby="name-help"
                                       placeholder="Enter your full name">
                                <div id="name-help" class="form-text">Please enter your full name</div>
                            </div>
                            <div class="col-12 col-md-6 mb-3">
                                <label for="email" class="form-label" aria-required="true">Email</label>
                                <input type="email" 
                                       class="form-control" 
                                       id="email" 
                                       name="email" 
                                       required 
                                       autocomplete="email"
                                       aria-describedby="email-help"
                                       placeholder="Enter your email address">
                                <div id="email-help" class="form-text">We'll never share your email</div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="message" class="form-label" aria-required="true">Message</label>
                            <textarea class="form-control" 
                                      id="message" 
                                      name="message" 
                                      rows="4" 
                                      required 
                                      aria-describedby="message-help"
                                      placeholder="Enter your message here"></textarea>
                            <div id="message-help" class="form-text">Please provide details about your inquiry</div>
                        </div>
                        
                        <div class="d-flex flex-column flex-md-row gap-2">
                            <button type="submit" 
                                    class="btn btn-primary" 
                                    aria-describedby="submit-help">
                                <i class="fas fa-paper-plane me-1" aria-hidden="true"></i>Submit Form
                            </button>
                            <button type="button" 
                                    class="btn btn-secondary" 
                                    onclick="this.form.reset()"
                                    aria-describedby="reset-help">
                                <i class="fas fa-undo me-1" aria-hidden="true"></i>Reset
                            </button>
                        </div>
                        <div id="submit-help" class="form-text">Click to submit the form</div>
                        <div id="reset-help" class="form-text">Click to clear all fields</div>
                    </form>

                    <!-- Dynamic Content Loading Demo -->
                    <div class="card">
                        <div class="card-header">
                            <h6 class="mb-0">Dynamic Content Loading</h6>
                        </div>
                        <div class="card-body">
                            <div class="d-flex flex-column flex-md-row gap-2 mb-3">
                                <button class="btn btn-sm btn-outline-primary" onclick="dynamicLoader.loadPage('{{ route('dashboard') }}', 'Dashboard')">
                                    Load Dashboard
                                </button>
                                <button class="btn btn-sm btn-outline-success" onclick="dynamicLoader.loadPage('{{ route('inspection.index') }}', 'Inspection')">
                                    Load Inspection
                                </button>
                                <button class="btn btn-sm btn-outline-warning" onclick="dynamicLoader.loadPage('{{ route('maintenance.workorder') }}', 'Workorder')">
                                    Load Workorder
                                </button>
                            </div>
                            
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                Click any button above to see dynamic page loading in action. 
                                The page will load without refreshing the browser!
                            </div>
                        </div>
                    </div>

                    <!-- Cache Management Demo -->
                    <div class="card mt-3">
                        <div class="card-header">
                            <h6 class="mb-0">Cache Management</h6>
                        </div>
                        <div class="card-body">
                            <div class="d-flex flex-column flex-md-row gap-2 mb-3">
                                <button class="btn btn-sm btn-outline-info" onclick="dynamicLoader.clearCache()">
                                    <i class="fas fa-broom me-1"></i>Clear Cache
                                </button>
                                <button class="btn btn-sm btn-outline-secondary" onclick="dynamicLoader.preloadPage('{{ route('vehicles.inventory') }}')">
                                    <i class="fas fa-download me-1"></i>Preload Vehicles
                                </button>
                            </div>
                            
                            <div class="alert alert-warning">
                                <i class="fas fa-lightbulb me-2"></i>
                                The system automatically caches pages for faster loading. 
                                Use these buttons to manage the cache.
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* CSS Variables for better maintainability */
:root {
    --primary-color: #0055A5;
    --primary-hover: #3b82f6;
    --text-primary: #1e293b;
    --text-secondary: #374151;
    --text-muted: #64748b;
    --text-light: #cbd5e1;
    --bg-primary: #f8fafc;
    --bg-white: #ffffff;
    --bg-hover: #f8f9fa;
    --border-color: #e2e8f0;
    --border-light: #cbd5e1;
    --shadow-sm: 0 2px 4px rgba(0, 0, 0, 0.1);
    --shadow-md: 0 4px 8px rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 8px 25px rgba(0, 0, 0, 0.1);
    --radius-sm: 0;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.2s ease;
    --transition-slow: all 0.3s ease;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-modal: 1055;
    --z-index-notification: 9999;
}

/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* Card styles */
.card {
    border: none;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.card:hover {
    box-shadow: var(--shadow-md);
}

/* Form styles */
.form-control:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 0.2rem rgba(0, 85, 165, 0.25);
}

/* Button styles */
.btn {
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.btn:hover {
    transform: translateY(-1px);
}

/* Alert styles */
.alert {
    font-family: var(--font-family);
    border-radius: var(--radius-md);
}

/* Form label styles */
.form-label {
    font-weight: 500;
    color: var(--text-secondary);
    font-family: var(--font-family);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 768px) {
    .content-box {
        padding: 0;
    }
    
    .card {
        margin-bottom: 1rem;
        border-radius: var(--radius-sm);
    }
    
    .card-header {
        padding: 1rem;
    }
    
    .card-header h4 {
        font-size: 1.25rem;
        color: var(--text-primary);
    }
    
    .card-header p {
        font-size: 0.9rem;
        color: var(--text-muted);
    }
    
    .card-body {
        padding: 1rem;
    }
    
    .form-label {
        font-size: 0.9rem;
        font-weight: 600;
        color: var(--text-secondary);
    }
    
    .form-control {
        font-size: 16px;
        padding: 0.75rem;
        min-height: 48px;
        border-color: var(--border-color);
    }
    
    .btn {
        font-size: 0.9rem;
        padding: 0.75rem 1rem;
        min-height: 48px;
    }
    
    .btn-sm {
        font-size: 0.8rem;
        padding: 0.6rem 0.8rem;
        min-height: 40px;
    }
    
    .alert {
        font-size: 0.85rem;
        padding: 0.75rem;
    }
    
    .alert i {
        font-size: 0.9rem;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
.btn:focus-visible,
.form-control:focus-visible {
    outline: 2px solid var(--primary-color);
    outline-offset: 2px;
}

/* ===== PRINT STYLES ===== */
@media print {
    .card {
        box-shadow: none;
        border: 1px solid #000;
    }
    
    .btn {
        display: none;
    }
}

@media (max-width: 576px) {
    .card-header {
        padding: 0.75rem;
    }
    
    .card-header h4 {
        font-size: 1.1rem;
    }
    
    .card-header p {
        font-size: 0.8rem;
    }
    
    .card-body {
        padding: 0.75rem;
    }
    
    .form-label {
        font-size: 0.85rem;
    }
    
    .form-control {
        font-size: 16px;
        padding: 0.6rem;
        min-height: 44px;
    }
    
    .btn {
        font-size: 0.85rem;
        padding: 0.6rem 0.8rem;
        min-height: 44px;
    }
    
    .btn-sm {
        font-size: 0.75rem;
        padding: 0.5rem 0.7rem;
        min-height: 36px;
    }
    
    .alert {
        font-size: 0.8rem;
        padding: 0.6rem;
    }
    
    .d-flex.gap-2 {
        gap: 0.5rem !important;
    }
}

@media (max-width: 480px) {
    .card-header {
        padding: 0.6rem;
    }
    
    .card-header h4 {
        font-size: 1rem;
    }
    
    .card-header p {
        font-size: 0.75rem;
    }
    
    .card-body {
        padding: 0.6rem;
    }
    
    .form-label {
        font-size: 0.8rem;
    }
    
    .form-control {
        font-size: 16px;
        padding: 0.5rem;
        min-height: 40px;
    }
    
    .btn {
        font-size: 0.8rem;
        padding: 0.5rem 0.7rem;
        min-height: 40px;
    }
    
    .btn-sm {
        font-size: 0.7rem;
        padding: 0.4rem 0.6rem;
        min-height: 32px;
    }
    
    .alert {
        font-size: 0.75rem;
        padding: 0.5rem;
    }
    
    .d-flex.gap-2 {
        gap: 0.4rem !important;
    }
}

@media (max-width: 360px) {
    .card-header {
        padding: 0.5rem;
    }
    
    .card-header h4 {
        font-size: 0.9rem;
    }
    
    .card-header p {
        font-size: 0.7rem;
    }
    
    .card-body {
        padding: 0.5rem;
    }
    
    .form-label {
        font-size: 0.75rem;
    }
    
    .form-control {
        font-size: 14px;
        padding: 0.4rem;
        min-height: 36px;
    }
    
    .btn {
        font-size: 0.75rem;
        padding: 0.4rem 0.6rem;
        min-height: 36px;
    }
    
    .btn-sm {
        font-size: 0.65rem;
        padding: 0.3rem 0.5rem;
        min-height: 28px;
    }
    
    .alert {
        font-size: 0.7rem;
        padding: 0.4rem;
    }
    
    .d-flex.gap-2 {
        gap: 0.3rem !important;
    }
}
</style>
@endsection

@section('scripts')
<script>
// Optimized Dynamic Form Functionality
(function() {
    'use strict';
    
    // Configuration object
    const CONFIG = {
        selectors: {
            form: 'form[data-dynamic]',
            submitBtn: '[type="submit"]',
            resetBtn: '[type="button"]'
        },
        classes: {
            spinner: 'fas fa-spinner fa-spin me-1',
            paperPlane: 'fas fa-paper-plane me-1',
            undo: 'fas fa-undo me-1'
        },
        messages: {
            processing: 'Processing...',
            success: 'Form submitted successfully! This is a demo - no actual data was sent.',
            error: 'An error occurred. Please try again.'
        },
        timing: {
            processingDelay: 2000
        }
    };
    
    // Cache DOM elements
    let elements = {};
    
    function initElements() {
        elements = {
            form: document.querySelector(CONFIG.selectors.form),
            submitBtn: null,
            resetBtn: null
        };
        
        if (elements.form) {
            elements.submitBtn = elements.form.querySelector(CONFIG.selectors.submitBtn);
            elements.resetBtn = elements.form.querySelector(CONFIG.selectors.resetBtn);
        }
    }
    
    // Form validation
    function validateForm(formData) {
        const errors = [];
        
        if (!formData.get('name')?.trim()) {
            errors.push('Name is required');
        }
        
        if (!formData.get('email')?.trim()) {
            errors.push('Email is required');
        } else if (!isValidEmail(formData.get('email'))) {
            errors.push('Please enter a valid email address');
        }
        
        if (!formData.get('message')?.trim()) {
            errors.push('Message is required');
        }
        
        return errors;
    }
    
    // Email validation
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    // Show loading state
    function showLoadingState() {
        if (!elements.submitBtn) return;
        
        elements.submitBtn.disabled = true;
        elements.submitBtn.innerHTML = `<i class="${CONFIG.classes.spinner}"></i>${CONFIG.messages.processing}`;
    }
    
    // Hide loading state
    function hideLoadingState() {
        if (!elements.submitBtn) return;
        
        elements.submitBtn.disabled = false;
        elements.submitBtn.innerHTML = `<i class="${CONFIG.classes.paperPlane}"></i>Submit Form`;
    }
    
    // Show success message
    function showSuccessMessage(message) {
        if (typeof dynamicLoader?.showSuccessMessage === 'function') {
            dynamicLoader.showSuccessMessage(message);
        } else {
            console.log('Success:', message);
        }
    }
    
    // Show error message
    function showErrorMessage(message) {
        if (typeof dynamicLoader?.showErrorMessage === 'function') {
            dynamicLoader.showErrorMessage(message);
        } else {
            console.error('Error:', message);
        }
    }
    
    // Handle form submission
    function handleFormSubmit(e) {
        e.preventDefault();
        
        if (!elements.form) return;
        
        const formData = new FormData(elements.form);
        const errors = validateForm(formData);
        
        if (errors.length > 0) {
            showErrorMessage(errors.join(', '));
            return;
        }
        
        showLoadingState();
        
        // Simulate form processing
        setTimeout(() => {
            try {
                showSuccessMessage(CONFIG.messages.success);
                elements.form.reset();
            } catch (error) {
                showErrorMessage(CONFIG.messages.error);
            } finally {
                hideLoadingState();
            }
        }, CONFIG.timing.processingDelay);
    }
    
    // Handle form reset
    function handleFormReset() {
        if (!elements.form) return;
        
        elements.form.reset();
        hideLoadingState();
    }
    
    // Initialize form handlers
    function initFormHandlers() {
        if (!elements.form) return;
        
        elements.form.addEventListener('submit', handleFormSubmit);
        
        if (elements.resetBtn) {
            elements.resetBtn.addEventListener('click', handleFormReset);
        }
    }
    
    // Initialize all functionality
    function init() {
        initElements();
        initFormHandlers();
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
    // Make functions globally accessible if needed
    window.handleDynamicFormSubmit = handleFormSubmit;
    window.handleDynamicFormReset = handleFormReset;
})();
</script>
@endsection
