import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Modal,
  Alert,
  Dimensions,
  BackHandler,
  Platform,
  Animated,
  KeyboardAvoidingView,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Geolocation from '@react-native-community/geolocation';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import { Dropdown } from 'react-native-element-dropdown';
import database from '@react-native-firebase/database';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.modalOverlay}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const Hsemodule43step1 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const route = useRoute();
  const [country, setCountry] = useState(null);
  const [countryItems] = useState([{ label: 'Zambia', value: 'Zambia' }, { label: 'DRC', value: 'DRC' }]);
  const [project, setProject] = useState(null);
  const [projectItems, setProjectItems] = useState([]);
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [coordinates, setCoordinates] = useState('');
  const [gpsAddress, setGpsAddress] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [showErrors, setShowErrors] = useState(false);
  const [countryError, setCountryError] = useState(false);
  const [projectError, setProjectError] = useState(false);
  const [checklistError, setChecklistError] = useState(false);
  const [requestNumber, setRequestNumber] = useState('');
  const [countryCode, setCountryCode] = useState('');
  const [projectCode, setProjectCode] = useState('');
  const isMounted = useRef(true);

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected);
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected);
    });

    return () => unsubscribe();
  }, []);

  useEffect(() => {
    const backAction = () => {
      handleBackPress();
      return true;
    };

    const backHandler = BackHandler.addEventListener('hardwareBackPress', backAction);
    return () => backHandler.remove();
  }, []);

  useFocusEffect(
    React.useCallback(() => {
      const loadSavedData = async () => {
        try {
          const savedHse = await AsyncStorage.getItem('hseStep1');
          if (savedHse) {
            const hseData = JSON.parse(savedHse);
            
            // Always load saved data, but allow overrides from route params
            if (hseData.country) setCountry(hseData.country);
            if (hseData.project) setProject(hseData.project);
            if (hseData.selectedChecklist) setSelectedChecklist(hseData.selectedChecklist);
            if (hseData.selectedEquipment) setSelectedEquipment(hseData.selectedEquipment);
            if (hseData.coordinates) setCoordinates(hseData.coordinates);
            if (hseData.gpsAddress) setGpsAddress(hseData.gpsAddress);
            if (hseData.countryCode) setCountryCode(hseData.countryCode);
            if (hseData.projectCode) setProjectCode(hseData.projectCode);
            if (hseData.requestNumber) setRequestNumber(hseData.requestNumber);
          }
        } catch (error) {
          console.error('Error loading saved HSE data:', error);
          Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load saved data.', position: 'top' });
        }
      };
      loadSavedData();
      
      // Check current counter value when screen is focused
      checkCurrentCounter();
    }, [])
  );

  useEffect(() => {
    if (country === 'Zambia') {
      setProjectItems([
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ]);
    } else if (country === 'DRC') {
      setProjectItems([
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ]);
    } else {
      setProjectItems([]);
    }
    if (!country || (project && !projectItems.some((item) => item.value === project))) setProject(null);
  }, [country]);

  // Update request number when country or project changes
  useEffect(() => {
    const updateRequestNumber = async () => {
      if (country && project && !requestNumber) {
        const newRequestNumber = await generateRequestNumber(country, project);
        setRequestNumber(newRequestNumber);
      } else if (!country || !project) {
        setRequestNumber('');
      }
    };
    updateRequestNumber();
  }, [country, project]);

  useEffect(() => {
    if (route.params?.selectedChecklist) {
      setSelectedChecklist(route.params.selectedChecklist);
      // Save the new selection immediately
      saveSelectedChecklist(route.params.selectedChecklist);
    }
    if (route.params?.selectedEquipment) {
      const cleanedEquipment = cleanEquipmentData(route.params.selectedEquipment);
      setSelectedEquipment(cleanedEquipment);
      // Save the new equipment selection immediately
      const currentData = {
        country,
        project,
        countryCode,
        projectCode,
        requestNumber,
        selectedChecklist,
        selectedEquipment: cleanedEquipment,
        coordinates,
        gpsAddress,
      };
      saveHseData(currentData);
    }
  }, [route.params?.selectedChecklist, route.params?.selectedEquipment]);

  const cleanEquipmentData = (equipment) => {
    if (!equipment) return null;
    const cleaned = {};
    Object.keys(equipment).forEach((key) => {
      const value = equipment[key];
      cleaned[key] = typeof value === 'string' ? value : String(value || '');
    });
    return cleaned;
  };

  const deepCleanObject = (obj) => {
    if (typeof obj !== 'object' || obj === null) return obj;
    if (Array.isArray(obj)) return obj.map(deepCleanObject);
    const cleaned = {};
    Object.keys(obj).forEach((key) => {
      const value = obj[key];
      cleaned[key] = typeof value === 'object' && value !== null ? deepCleanObject(value) : String(value || '');
    });
    return cleaned;
  };

  const saveHseData = async (data) => {
    try {
      const cleanedData = deepCleanObject(data);
      await AsyncStorage.setItem('hseStep1', JSON.stringify(cleanedData));
      if (!isOnline) {
        const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
        syncQueue.push({ type: 'hseStep1', data: cleanedData });
        await AsyncStorage.setItem('syncQueue', JSON.stringify(syncQueue));
        Toast.show({ type: 'info', text1: 'Offline', text2: 'Data saved locally. Will sync when online.', position: 'top' });
      }
    } catch (error) {
      console.error('Error saving HSE data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save data locally.', position: 'top' });
    }
  };

  // Helper to save only the selected checklist in AsyncStorage
  const saveSelectedChecklist = async (selectedChecklist) => {
    try {
      const savedHse = await AsyncStorage.getItem('hseStep1');
      let hseData = savedHse ? JSON.parse(savedHse) : {};
      hseData.selectedChecklist = selectedChecklist;
      await AsyncStorage.setItem('hseStep1', JSON.stringify(hseData));
    } catch (error) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save checklist.', position: 'top' });
    }
  };

  const requestLocationPermission = async () => {
    try {
      if (Platform.OS === 'android') {
        const granted = await PermissionsAndroid.request(PermissionsAndroid.PERMISSIONS.ACCESS_FINE_LOCATION);
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } else {
        return new Promise((resolve) =>
          Geolocation.requestAuthorization(resolve, () => resolve(false))
        );
      }
    } catch (error) {
      console.error('Error requesting location permission:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to request location permission.', position: 'top' });
      return false;
    }
  };

  const fetchAddressFromCoordinates = async (latitude, longitude) => {
    const apiKey = 'YOUR_API_KEY'; // Replace with your Google Maps API key
    const url = `https://maps.googleapis.com/maps/api/geocode/json?latlng=${latitude},${longitude}&key=${apiKey}`;
    try {
      const response = await fetch(url);
      const data = await response.json();
      if (data.status === 'OK' && data.results.length > 0) return data.results[0].formatted_address;
      throw new Error(data.status === 'ZERO_RESULTS' ? 'No address found' : 'Geocoding API error');
    } catch (error) {
      console.error('Error fetching address:', error.message);
      return `Latitude: ${latitude}, Longitude: ${longitude} (Address unavailable)`;
    }
  };

  const handleGetLocation = async () => {
    try {
      const hasPermission = await requestLocationPermission();
      if (!hasPermission) {
        Toast.show({ type: 'error', text1: 'Location Error', text2: 'Location permission denied.', position: 'top' });
        return;
      }
      setSyncModalVisible(true);
      const position = await new Promise((resolve, reject) =>
        Geolocation.getCurrentPosition(resolve, reject, { enableHighAccuracy: true, timeout: 15000, maximumAge: 10000 })
      );
      const { latitude, longitude } = position.coords;
      const coords = `${latitude}, ${longitude}`;
      setCoordinates(coords);
      const address = await fetchAddressFromCoordinates(latitude, longitude);
      setGpsAddress(address);
      const savedHse = await AsyncStorage.getItem('hseStep1');
      let hseData = savedHse ? JSON.parse(savedHse) : {};
      hseData.coordinates = coords;
      hseData.gpsAddress = address;
      await saveHseData(hseData);
      Toast.show({ type: 'success', text1: 'Location Fetched', text2: `Coordinates: ${coords}\nAddress: ${address}`, position: 'top', visibilityTime: 4000 });
    } catch (error) {
      console.error('Error in handleGetLocation:', error);
      Toast.show({ type: 'error', text1: 'Location Error', text2: error.message || 'Failed to fetch location.', position: 'top' });
    } finally {
      if (isMounted.current) setSyncModalVisible(false);
    }
  };

  const handleChecklistSelect = async () => {
    // Save current form data before navigating
    const currentData = {
      country,
      project,
      countryCode,
      projectCode,
      requestNumber,
      selectedChecklist,
      selectedEquipment,
      coordinates,
      gpsAddress,
    };
    await saveHseData(currentData);
    
    navigation.navigate('HseSelectChecklsit', {
      onSelect: async (selectedChecklist) => {
        setSelectedChecklist(selectedChecklist);
        await saveSelectedChecklist(selectedChecklist);
        // Also update the main HSE data
        const updatedData = {
          ...currentData,
          selectedChecklist,
        };
        await saveHseData(updatedData);
      },
    });
  };

  const handleEquipmentSelect = async () => {
    // Save current form data before navigating
    const currentData = {
      country,
      project,
      countryCode,
      projectCode,
      requestNumber,
      selectedChecklist,
      selectedEquipment,
      coordinates,
      gpsAddress,
    };
    await saveHseData(currentData);
    
    navigation.navigate('HSEselecteq', { 
      selectedEquipment, 
      source: 'HSE_STEP1',
      onSelect: async (newEquipment) => {
        const cleanedEquipment = cleanEquipmentData(newEquipment);
        setSelectedEquipment(cleanedEquipment);
        // Update the main HSE data with new equipment
        const updatedData = {
          ...currentData,
          selectedEquipment: cleanedEquipment,
        };
        await saveHseData(updatedData);
      },
    });
  };

  const handleBackPress = () => {
    Alert.alert(
      'Exit Step',
      'Do you want to exit this step? All filled data will be lost.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'OK',
          style: 'destructive',
          onPress: async () => {
            try {
              await AsyncStorage.removeItem('hseStep1');
              navigation.goBack();
            } catch (error) {
              console.error('Error clearing HSE data:', error);
              Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to clear data.', position: 'top' });
            }
          },
        },
      ],
      { cancelable: false }
    );
  };

  const isFormValid = () => {
    return country && project && selectedChecklist;
  };

  const handleNext = async () => {
    setShowErrors(true);
    setCountryError(!country);
    setProjectError(!project);
    setChecklistError(!selectedChecklist);

    if (!isFormValid()) {
      const missingFields = [];
      if (!country) missingFields.push('Country');
      if (!project) missingFields.push('Project');
      if (!selectedChecklist) missingFields.push('Checklist');
      Toast.show({
        type: 'error',
        text1: 'Required Fields Missing',
        text2: `Please fill: ${missingFields.join(', ')}`,
        position: 'top',
      });
      return;
    }

    try {
      const hseData = {
        id: Date.now().toString(),
        country,
        project,
        countryCode,
        projectCode,
        requestNumber,
        selectedChecklist: deepCleanObject(selectedChecklist),
        selectedEquipment: deepCleanObject(selectedEquipment),
        coordinates,
        gpsAddress,
        createdAt: new Date().toISOString(),
      };
      await saveHseData(hseData);
      setShowErrors(false);
      setActiveStep(2);
    } catch (error) {
      console.error('Error saving HSE data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save HSE data.', position: 'top' });
    }
  };

  const getCurrentReportNumber = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      const currentNumber = snapshot.val() || 0;
      return (currentNumber + 1).toString().padStart(3, '0');
    } catch (error) {
      console.error('Error getting current report number:', error);
      return '001';
    }
  };

  const checkCurrentCounter = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      const currentNumber = snapshot.val() || 0;
      console.log('Current counter value in Firebase:', currentNumber);
      Toast.show({
        type: 'info',
        text1: 'Counter Info',
        text2: `Current counter value: ${currentNumber}`,
        position: 'top',
      });
      return currentNumber;
    } catch (error) {
      console.error('Error checking current counter:', error);
      return 0;
    }
  };

  const generateRequestNumber = async (selectedCountry, selectedProject) => {
    if (!selectedCountry || !selectedProject) return '';
    
    let countryCode = '';
    let projectCode = '';
    
    // Country codes
    switch (selectedCountry) {
      case 'Zambia':
        countryCode = 'ZMB';
        break;
      case 'DRC':
        countryCode = 'DRC';
        break;
      default:
        countryCode = 'UNK';
    }
    
    // Project codes
    switch (selectedProject) {
      case 'Kansanshi':
        projectCode = 'KAN';
        break;
      case 'Kalumbila':
        projectCode = 'KAL';
        break;
      case 'Kimteto':
        projectCode = 'KIM';
        break;
      case 'Kobold':
        projectCode = 'KOB';
        break;
      case 'FQM Exploration':
        projectCode = 'FQM';
        break;
      case 'Mimosa':
        projectCode = 'MIM';
        break;
      case 'Lubumbashi':
        projectCode = 'LUB';
        break;
      case 'Musompo':
        projectCode = 'MUS';
        break;
      case 'IME':
        projectCode = 'IME';
        break;
      case 'Kamoa':
        projectCode = 'KAM';
        break;
      default:
        projectCode = 'PRJ';
    }
    
    // Get the current report number (without incrementing)
    const reportNumber = await getCurrentReportNumber();
    const requestNumber = `${countryCode} - ${projectCode} - HSE - ${reportNumber}`;
    
    // Store the codes for use in step 3
    setCountryCode(countryCode);
    setProjectCode(projectCode);
    
    return requestNumber;
  };

  const getStatusColor = (status) => {
    if (typeof status !== 'string') return '#666';
    switch (status) {
      case 'Operational': return '#4CAF50';
      case 'Maintenance': return '#FFC107';
      case 'Faulty': return '#F44336';
      case 'Out of Service': return '#9E9E9E';
      default: return '#666';
    }
  };

  return (
    <KeyboardAvoidingView style={styles.container} behavior={Platform.OS === 'ios' ? 'padding' : 'height'} keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} keyboardShouldPersistTaps="handled">
        <View style={styles.headingContainer}>
          <Text style={styles.headingText}>HSE Inspection Information</Text>
        </View>
        <View style={styles.requestNumberContainer}>
          <Text style={styles.requestNumberLabel}>Request #</Text>
          <Text style={styles.requestNumberValue}>{requestNumber || 'Select Country and Project'}</Text>
        </View>
        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>Country<Text style={inputStyles.requiredError}> *</Text></Text>
          <Dropdown
            style={[inputStyles.input, countryError && showErrors ? inputStyles.inputError : null]}
            placeholderStyle={styles.placeholderStyle}
            selectedTextStyle={styles.dropdownText}
            data={countryItems}
            maxHeight={300}
            labelField="label"
            valueField="value"
            placeholder="Select Country"
            value={country}
            onChange={item => setCountry(item.value)}
          />
          {countryError && showErrors && <Text style={inputStyles.errorText}>Country is required</Text>}
        </View>
        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>Project<Text style={inputStyles.requiredError}> *</Text></Text>
          <Dropdown
            style={[inputStyles.input, !country && styles.disabledDropdown, projectError && showErrors ? inputStyles.inputError : null]}
            placeholderStyle={styles.placeholderStyle}
            selectedTextStyle={styles.dropdownText}
            data={projectItems}
            maxHeight={300}
            labelField="label"
            valueField="value"
            placeholder="Select Project"
            value={project}
            onChange={item => setProject(item.value)}
            disable={!country}
          />
          {projectError && showErrors && <Text style={inputStyles.errorText}>Project is required</Text>}
        </View>
        <View style={styles.selectRow}>
          <Text style={styles.selectLabel}>Checklist<Text style={inputStyles.requiredError}> *</Text></Text>
          <TouchableOpacity onPress={handleChecklistSelect} style={styles.selectButton}>
            <Text style={styles.selectText}>{selectedChecklist ? 'Change checklist' : 'Select'}</Text>
            <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
          </TouchableOpacity>
        </View>
        {checklistError && showErrors && <Text style={[inputStyles.errorText, { marginHorizontal: 15 }]}>Checklist is required</Text>}
        {selectedChecklist && (
          <View style={styles.selectedChecklistBox}>
            <Text style={styles.selectedChecklistText}>{selectedChecklist.document_name}</Text>
            <Text style={styles.checklistDetailsText}>ID: {selectedChecklist.document_id} | Version: {selectedChecklist.version}</Text>
          </View>
        )}
        <View style={styles.selectRow}>
          <Text style={styles.selectLabel}>Equipment</Text>
          <TouchableOpacity onPress={handleEquipmentSelect} style={styles.selectButton}>
            <Text style={styles.selectText}>{selectedEquipment ? 'Change' : 'Select'}</Text>
            <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
          </TouchableOpacity>
        </View>
        {selectedEquipment && (
          <View style={styles.selectedEquipmentBox}>
            <Text style={styles.selectedEquipmentTitle}>
              {selectedEquipment.equipmentName || selectedEquipment.equipmentNumber || 'Equipment Selected'}
            </Text>
            <View style={styles.equipmentDetailsRow}>
              <Text style={styles.equipmentDetailLabel}>Number:</Text>
              <Text style={styles.equipmentDetailValue}>
                {selectedEquipment.equipmentNumber || selectedEquipment.entryNumber || 'N/A'}
              </Text>
            </View>
            <View style={styles.equipmentDetailsRow}>
              <Text style={styles.equipmentDetailLabel}>Category:</Text>
              <Text style={styles.equipmentDetailValue}>
                {selectedEquipment.mainCategory || 'N/A'}
              </Text>
            </View>
            {selectedEquipment.model && (
              <View style={styles.equipmentDetailsRow}>
                <Text style={styles.equipmentDetailLabel}>Model:</Text>
                <Text style={styles.equipmentDetailValue}>{selectedEquipment.model}</Text>
              </View>
            )}
            {selectedEquipment.meterReading && (
              <View style={styles.equipmentDetailsRow}>
                <Text style={styles.equipmentDetailLabel}>Meter Reading:</Text>
                <Text style={styles.equipmentDetailValue}>
                  {selectedEquipment.meterReading} {selectedEquipment.meterUnit || ''}
                </Text>
              </View>
            )}
            {selectedEquipment.status && (
              <View style={styles.equipmentDetailsRow}>
                <Text style={styles.equipmentDetailLabel}>Status:</Text>
                <Text style={[styles.equipmentDetailValue, { color: getStatusColor(selectedEquipment.status) }]}>
                  {selectedEquipment.status}
                </Text>
              </View>
            )}
            {selectedEquipment.country && (
              <View style={styles.equipmentDetailsRow}>
                <Text style={styles.equipmentDetailLabel}>Country:</Text>
                <Text style={styles.equipmentDetailValue}>{selectedEquipment.country}</Text>
              </View>
            )}
            {selectedEquipment.createdAt && (
              <View style={styles.equipmentDetailsRow}>
                <Text style={styles.equipmentDetailLabel}>Created:</Text>
                <Text style={styles.equipmentDetailValue}>
                  {new Date(selectedEquipment.createdAt).toLocaleDateString()}
                </Text>
              </View>
            )}
          </View>
        )}
        <View style={styles.selectRow}>
          <Text style={styles.selectLabel}>GPS Location</Text>
          <TouchableOpacity onPress={handleGetLocation} style={styles.selectButton}>
            <MaterialIcons name="location-on" size={16} color="#0078D4" />
            <Text style={styles.selectText}>Get Location</Text>
          </TouchableOpacity>
        </View>
        <View style={styles.gpsDetailsBox}>
          {coordinates && <View style={styles.gpsInfoRow}><Text style={styles.gpsLabel}>Coordinates: </Text><Text style={styles.gpsValue}>{coordinates}</Text></View>}
          {gpsAddress && <View style={styles.gpsInfoRow}><Text style={styles.gpsLabel}>GPS Address: </Text><Text style={styles.gpsValue}>{gpsAddress}</Text></View>}
          {!coordinates && !gpsAddress && <Text style={styles.noGpsText}>Click "Get Location" to fetch your current location</Text>}
        </View>
      </ScrollView>
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Next</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message="Processing" />
      <Toast />
    </KeyboardAvoidingView>
  );
};

const inputStyles = StyleSheet.create({
  container: { marginBottom: width * 0.04 },
  label: { fontSize: width * 0.038, color: '#015185', marginBottom: 4, fontWeight: '500' },
  requiredError: { color: 'red', fontWeight: 'bold' },
  input: { borderWidth: 1, borderColor: '#015185', borderRadius: 5, paddingHorizontal: 12, paddingVertical: 14, fontSize: width * 0.04, backgroundColor: '#fff' },
  inputError: { borderColor: 'red' },
  errorText: { color: 'red', fontSize: width * 0.032, marginTop: 2, marginLeft: 2 },
});

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  mainContent: { flex: 1, paddingHorizontal: width * 0.04, paddingVertical: width * 0.025 },
  scrollContent: { paddingBottom: 130 },
  headingContainer: { justifyContent: 'center', alignItems: 'center', paddingVertical: 10 },
  headingText: { fontSize: 20, fontWeight: '500', color: '#015185' },
  requestNumberContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
    paddingHorizontal: 15,
    paddingVertical: 12,
    borderRadius: 5,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  requestNumberLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
  },
  requestNumberValue: {
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
    textAlign: 'right',
  },
  dropdownText: { fontSize: 16, color: '#333' },
  placeholderStyle: { fontSize: 16, color: '#888' },
  disabledDropdown: { backgroundColor: '#e0e0e0' },
  selectRow: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', width: '100%', height: 50, backgroundColor: '#f0f0f0', paddingHorizontal: 15, borderRadius: 5, marginTop: 5 },
  selectLabel: { fontSize: 16, color: '#015185', fontWeight: '600' },
  selectButton: { flexDirection: 'row', alignItems: 'center' },
  selectText: { fontSize: 16, color: '#0078D4', marginLeft: 5 },
  selectedChecklistBox: { width: '100%', backgroundColor: '#f0f0f0', paddingVertical: 10, paddingHorizontal: 15, marginBottom: 10, borderRadius: 5, alignSelf: 'center' },
  selectedChecklistText: { fontSize: 16, color: '#333', fontWeight: '600', marginBottom: 5 },
  checklistDetailsText: { fontSize: 14, color: '#666' },
  gpsDetailsBox: { width: '100%', backgroundColor: '#f0f0f0', paddingVertical: 10, paddingHorizontal: 15, marginBottom: 10, borderRadius: 5, alignSelf: 'center' },
  gpsInfoRow: { flexDirection: 'row', flexWrap: 'wrap', marginBottom: 5 },
  gpsLabel: { fontSize: 16, color: '#333', fontWeight: '600' },
  gpsValue: { fontSize: 16, color: '#333', flex: 1 },
  noGpsText: { fontSize: 14, color: '#888', textAlign: 'center' },
  buttonContainer: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingHorizontal: 15, paddingVertical: 15, backgroundColor: '#F2F2F2', borderTopWidth: 1, borderTopColor: '#ddd', position: 'absolute', bottom: 0, left: 0, right: 0, elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: -2 }, shadowOpacity: 0.1, shadowRadius: 2, paddingBottom: 30 },
  cancelButton: { flexDirection: 'row', alignItems: 'center', justifyContent: 'center', paddingBottom: 30, paddingHorizontal: 15 },
  cancelButtonText: { color: '#015185', fontSize: 16, fontWeight: '600' },
  nextButton: { flexDirection: 'row', alignItems: 'center', justifyContent: 'center', paddingBottom: 30, borderRadius: 5 },
  nextButtonText: { color: '#015185', fontSize: 16, fontWeight: '600' },
  modalOverlay: { flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: 'rgba(0,0,0,0.5)' },
  bottomicon: { marginTop: 2 },
  syncModalContent: { width: '80%', backgroundColor: '#fff', borderRadius: 10, padding: 20, alignItems: 'center', elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: 2 }, shadowOpacity: 0.3, shadowRadius: 4 },
  syncModalText: { fontSize: 18, fontWeight: '600', color: '#015185', marginBottom: 10 },
  syncModalTextadd: { fontSize: 14, color: '#666', textAlign: 'center' },
  syncIcondiv: { alignItems: 'center' },
  syncIcon: { width: 50, height: 50, resizeMode: 'contain' },
  selectedEquipmentBox: {
    width: '100%',
    backgroundColor: '#e8f4fd',
    paddingVertical: 15,
    paddingHorizontal: 15,
    marginBottom: 10,
    borderRadius: 8,
    alignSelf: 'center',
    borderWidth: 1,
    borderColor: '#015185',
  },
  selectedEquipmentTitle: {
    fontSize: 18,
    color: '#015185',
    fontWeight: 'bold',
    marginBottom: 10,
    textAlign: 'center',
  },
  equipmentDetailsRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 5,
    paddingVertical: 2,
  },
  equipmentDetailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
    flex: 1,
  },
  equipmentDetailValue: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
    flex: 2,
    textAlign: 'right',
  },
});

export default Hsemodule43step1;