import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  Image,
  StyleSheet,
  SafeAreaView,
  TouchableOpacity,
  Animated,
  ActivityIndicator,
  Modal,
  Platform,
  ToastAndroid,
  ScrollView,
  KeyboardAvoidingView,
  Dimensions,
  PermissionsAndroid,
  AppState,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import LinearGradient from 'react-native-linear-gradient';
import NetInfo from '@react-native-community/netinfo';
import { TextInput } from 'react-native-paper';
import Toast from 'react-native-toast-message';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import bcrypt from 'react-native-bcrypt';

// Get screen dimensions for responsive design
const { width } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;

// MongoDB API Configuration
const MONGODB_API_BASE = 'http://168.231.113.219:5003';
const EQUIPMENT_API_BASE = 'http://168.231.113.219:5001';
const DOCUMENTS_API_BASE = 'http://168.231.113.219:5002';

// Reusable Gradient Button Component
const GradientButton = ({ onPress, title, style, isLoading }) => (
  <LinearGradient colors={['#015185', '#3481BC']} style={[styles.gradientButton, style]}>
    <TouchableOpacity style={styles.buttonTouchable} onPress={onPress} activeOpacity={0.8} disabled={isLoading}>
      {isLoading ? (
        <ActivityIndicator size="small" color="#fff" />
      ) : (
        <Text style={styles.googleButtonText}>{title}</Text>
      )}
    </TouchableOpacity>
  </LinearGradient>
);

// Custom Animated Input Component
const AnimatedInput = ({
  label,
  value,
  onChangeText,
  secureTextEntry,
  keyboardType,
  autoCapitalize,
  leftIconName,
  rightIconName,
  onRightIconPress,
  hasError,
  onBlur,
  inputRef,
  onFocus,
}) => {
  const [isFocused, setIsFocused] = useState(false);
  const animatedLabel = useRef(new Animated.Value(value ? 1 : 0)).current;
  const animatedBorderWidth = useRef(new Animated.Value(1)).current;
  const animatedBorderColor = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    Animated.parallel([
      Animated.timing(animatedLabel, {
        toValue: isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderWidth, {
        toValue: isFocused ? 2 : 1,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderColor, {
        toValue: hasError ? 2 : isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
    ]).start();
  }, [isFocused, value, hasError]);

  const labelStyle = {
    position: 'absolute',
    left: scale(45),
    top: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(18), scale(-8)],
    }),
    fontSize: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(16), scale(12)],
    }),
    color: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: ['#888', '#015185'],
    }),
    backgroundColor: '#fff',
    paddingHorizontal: scale(4),
    zIndex: 2,
  };

  const borderColor = animatedBorderColor.interpolate({
    inputRange: [0, 1, 2],
    outputRange: ['#ddd', '#015185', '#ff0000'],
  });

  return (
    <View style={styles.inputContainer} ref={inputRef}>
      <Animated.View
        style={[
          styles.inputGradient,
          {
            borderWidth: animatedBorderWidth,
            borderColor: borderColor,
          },
        ]}
      >
        <Animated.Text style={labelStyle}>{label}</Animated.Text>
        <TextInput
          mode="outlined"
          value={value}
          onChangeText={onChangeText}
          secureTextEntry={secureTextEntry}
          keyboardType={keyboardType}
          autoCapitalize={autoCapitalize}
          style={styles.input}
          onFocus={() => {
            setIsFocused(true);
            onFocus && onFocus();
          }}
          onBlur={() => {
            setIsFocused(false);
            onBlur && onBlur();
          }}
          left={
            leftIconName && (
              <TextInput.Icon
                icon={() => <MaterialIcons name={leftIconName} size={scale(24)} color="#015185" />}
              />
            )
          }
          right={
            rightIconName && (
              <TextInput.Icon
                icon={() => (
                  <TouchableOpacity onPress={onRightIconPress}>
                    <MaterialIcons name={rightIconName} size={scale(24)} color="#015185" />
                  </TouchableOpacity>
                )}
              />
            )
          }
          outlineStyle={{
            borderWidth: 0,
            borderRadius: scale(8),
          }}
          theme={{
            colors: {
              primary: 'transparent',
              text: '#333',
              placeholder: '#888',
            },
          }}
        />
      </Animated.View>
    </View>
  );
};

// SyncModal Component
const SyncModal = ({ visible, onClose, message }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages based on platform
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.showWithGravity(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

export default function MongoDBLoginScreen() {
  const navigation = useNavigation();
  const [emailOrEmployeeId, setEmailOrEmployeeId] = useState('');
  const [password, setPassword] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const [emailOrEmployeeIdError, setEmailOrEmployeeIdError] = useState(false);
  const [emailOrEmployeeIdErrorMessage, setEmailOrEmployeeIdErrorMessage] = useState('');
  const [passwordError, setPasswordError] = useState(false);
  const [passwordErrorMessage, setPasswordErrorMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [isNetworkError, setIsNetworkError] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  const networkErrorAnim = useRef(new Animated.Value(-scale(50))).current;
  const scrollViewRef = useRef(null);
  const emailOrEmployeeIdInputRef = useRef(null);
  const passwordInputRef = useRef(null);

  const showNetworkError = () => {
    setIsNetworkError(true);
    Animated.timing(networkErrorAnim, {
      toValue: 0,
      duration: 300,
      useNativeDriver: true,
    }).start();

    setTimeout(() => {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }, 3000);
  };

  const hideNetworkError = () => {
    if (isNetworkError) {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }
  };

  useEffect(() => {
    const initialize = async () => {
      try {
        await checkNetworkStatus();
      } catch (error) {
        console.error('Initialization error:', error);
        showNetworkError();
        showToast('error', 'Initialization Error', 'Failed to initialize');
      }
    };

    initialize();
  }, []);

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showNetworkError();
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      } else {
        hideNetworkError();
        return true;
      }
    } catch (error) {
      console.error('Network check error:', error);
      showNetworkError();
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

  const validateEmailOrEmployeeId = (input) => {
    const inputEmpty = !input.trim();
    const isEmail = /\S+@\S+\.\S+/.test(input);
    const isEmployeeId = input.length >= 1;
    const inputTooLong = input.length > 100;

    setEmailOrEmployeeIdError(inputEmpty || (!isEmail && !isEmployeeId) || inputTooLong);
    if (inputEmpty) {
      setEmailOrEmployeeIdErrorMessage('Email/Employee ID is required');
    } else if (!isEmail && !isEmployeeId) {
      setEmailOrEmployeeIdErrorMessage('Enter a valid email address or employee ID');
    } else if (inputTooLong) {
      setEmailOrEmployeeIdErrorMessage('Input cannot exceed 100 characters');
    } else {
      setEmailOrEmployeeIdErrorMessage('');
    }

    return !(inputEmpty || (!isEmail && !isEmployeeId) || inputTooLong);
  };

  const validatePassword = (password) => {
    const passwordEmpty = !password.trim();

    setPasswordError(passwordEmpty);
    if (passwordEmpty) {
      setPasswordErrorMessage('Password is required');
    } else {
      setPasswordErrorMessage('');
    }

    return !passwordEmpty;
  };

  const findUserByEmailOrEmployeeId = async (input) => {
    try {
      console.log('🔍 Searching for user:', input);
      
      // Search by email first
      const emailResponse = await fetch(`${MONGODB_API_BASE}/users/all`);
      if (!emailResponse.ok) {
        throw new Error(`HTTP ${emailResponse.status}: ${emailResponse.statusText}`);
      }
      
      const usersData = await emailResponse.json();
      if (!usersData.success || !usersData.data) {
        throw new Error('Failed to fetch users data');
      }
      
      const users = usersData.data;
      
      // Find user by email or employee number
      const user = users.find(u => 
        u.email?.toLowerCase() === input.toLowerCase() || 
        u.employeeNumber === input
      );
      
      if (user) {
        console.log('✅ User found:', user._id);
        return user;
      }
      
      console.log('❌ User not found');
      return null;
    } catch (error) {
      console.error('❌ Error finding user:', error);
      return null;
    }
  };

  const verifyPassword = async (user, inputPassword) => {
    try {
      // Check if user has hashed password
      if (user.hashedPassword) {
        // Verify with bcrypt
        const isValid = await bcrypt.compare(inputPassword, user.hashedPassword);
        return isValid;
      } else if (user.password) {
        // Fallback to plain text comparison
        return user.password === inputPassword;
      } else {
        // Default password fallback
        return user.employeeNumber === inputPassword || inputPassword === 'password';
      }
    } catch (error) {
      console.error('❌ Password verification error:', error);
      return false;
    }
  };

  const handleLogin = async () => {
    const isInputValid = validateEmailOrEmployeeId(emailOrEmployeeId);
    const isPasswordValid = validatePassword(password);

    if (!isInputValid || !isPasswordValid) {
      return;
    }

    setIsLoading(true);
    setSyncModalVisible(true);

    try {
      const startTime = Date.now();
      console.log('⏱️ Login started at:', new Date().toISOString());
      
      if (!(await checkNetworkStatus())) {
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }
      console.log('⏱️ Network check took:', Date.now() - startTime, 'ms');

      // Find user by email or employee ID
      const searchStartTime = Date.now();
      const user = await findUserByEmailOrEmployeeId(emailOrEmployeeId);
      console.log('⏱️ User search took:', Date.now() - searchStartTime, 'ms');
      
      if (!user) {
        showToast('error', 'Login Error', 'User not found with provided email/employee ID');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // Verify password
      const passwordStartTime = Date.now();
      const isPasswordValid = await verifyPassword(user, password);
      console.log('⏱️ Password verification took:', Date.now() - passwordStartTime, 'ms');
      
      if (!isPasswordValid) {
        showToast('error', 'Login Error', 'Incorrect password');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // Check user role
      const userRole = user.role;
      if (userRole !== 'ProAdmin') {
        showToast('error', 'Access Denied', 'Please login from user dashboard');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // Store session data
      const storageStartTime = Date.now();
      await Promise.all([
        AsyncStorage.setItem('userUid', user._id),
        AsyncStorage.setItem('userEmail', user.email),
        AsyncStorage.setItem('Loginsuccess-01', 'true'),
        AsyncStorage.setItem('ProAdminLogined', 'true'),
        AsyncStorage.setItem('adminPassword', password),
        AsyncStorage.removeItem('TeamMemberLogined'),
        // Store user data for home screen
        AsyncStorage.setItem('userData', JSON.stringify(user)),
      ]);
      console.log('⏱️ Storage operations took:', Date.now() - storageStartTime, 'ms');
      console.log('⏱️ Total login time:', Date.now() - startTime, 'ms');

      // Update last login in MongoDB
      try {
        await fetch(`${MONGODB_API_BASE}/users/${user._id}`, {
          method: 'PUT',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            loginStatus: true,
            lastLogin: new Date().toISOString(),
          }),
        });
      } catch (updateError) {
        console.warn('⚠️ Failed to update last login:', updateError);
      }

      // Show success and navigate
      showToast('success', 'Success', 'Logged in successfully');
      setIsLoading(false);
      setSyncModalVisible(false);
      console.log('⏱️ Navigating to MainApp...');
      navigation.replace('MainApp');

    } catch (error) {
      console.error('Login error:', error);
      showToast('error', 'Login Error', error.message || 'An unexpected error occurred');
      setIsLoading(false);
      setSyncModalVisible(false);
    }
  };

  const toggleShowPassword = () => {
    setShowPassword(!showPassword);
  };

  const handleInputFocus = (inputRef) => {
    if (inputRef.current && scrollViewRef.current) {
      inputRef.current.measureLayout(
        scrollViewRef.current.getScrollResponder(),
        (x, y) => {
          scrollViewRef.current.scrollTo({ y: y - scale(100), animated: true });
        },
        (error) => {
          console.warn('Failed to measure layout:', error);
          scrollViewRef.current.scrollTo({ y: 0, animated: true });
        }
      );
    }
  };

  return (
    <SafeAreaView style={styles.container}>
      {isNetworkError && (
        <Animated.View
          style={[styles.networkErrorBanner, { transform: [{ translateY: networkErrorAnim }] }]}
        >
          <Text style={styles.networkErrorText}>Internet connection not available</Text>
        </Animated.View>
      )}

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        keyboardVerticalOffset={Platform.OS === 'ios' ? scale(60) : scale(20)}
        style={styles.keyboardAvoidingContainer}
      >
        <ScrollView
          ref={scrollViewRef}
          contentContainerStyle={styles.scrollContentContainer}
          showsVerticalScrollIndicator={false}
          keyboardShouldPersistTaps="handled"
        >
          <View style={styles.contentContainer}>
            <View style={styles.logoContainer}>
              <Image
                source={require('../../Images/adminlogin/logo.png')}
                style={styles.logo}
                resizeMode="contain"
              />
              <Text style={styles.loginText}>Admin Login</Text>
            </View>
            
            <View style={styles.formContainer}>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={emailOrEmployeeIdInputRef}
                  label="Email/Employee ID"
                  value={emailOrEmployeeId}
                  onChangeText={(text) => {
                    setEmailOrEmployeeId(text);
                    validateEmailOrEmployeeId(text);
                  }}
                  keyboardType="email-address"
                  autoCapitalize="none"
                  leftIconName="person"
                  hasError={emailOrEmployeeIdError}
                  onBlur={() => validateEmailOrEmployeeId(emailOrEmployeeId)}
                  onFocus={() => handleInputFocus(emailOrEmployeeIdInputRef)}
                />
                {emailOrEmployeeIdError && <Text style={styles.errorText}>{emailOrEmployeeIdErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={passwordInputRef}
                  label="Password"
                  value={password}
                  onChangeText={(text) => {
                    setPassword(text);
                    validatePassword(text);
                  }}
                  secureTextEntry={!showPassword}
                  autoCapitalize="none"
                  leftIconName="lock"
                  rightIconName={showPassword ? 'visibility' : 'visibility-off'}
                  onRightIconPress={toggleShowPassword}
                  hasError={passwordError}
                  onBlur={() => validatePassword(password)}
                  onFocus={() => handleInputFocus(passwordInputRef)}
                />
                {passwordError && <Text style={styles.errorText}>{passwordErrorMessage}</Text>}
              </View>

              <GradientButton title="Login" onPress={handleLogin} isLoading={isLoading} />

              <View style={styles.dividerContainer}>
                <View style={styles.borderLine} />
                <TouchableOpacity onPress={() => navigation.navigate('SignUp')}>
                  <Text style={styles.dividerText}>Need an account? Sign Up</Text>
                </TouchableOpacity>
                <View style={styles.borderLine} />
              </View>
            </View>
          </View>
        </ScrollView>
      </KeyboardAvoidingView>

      <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
        message="Processing Login"
      />
      <Toast />
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
    paddingHorizontal: scale(6),
    marginTop: scale(24),
  },
  keyboardAvoidingContainer: {
    flex: 1,
  },
  scrollContentContainer: {
    flexGrow: 1,
    justifyContent: 'center',
    paddingBottom: scale(20),
  },
  contentContainer: {
    flex: 1,
    justifyContent: 'center',
  },
  logoContainer: {
    alignItems: 'center',
    marginBottom: scale(20),
  },
  logo: {
    width: '80%',
    height: scale(100),
    resizeMode: 'contain',
  },
  loginText: {
    fontSize: scale(24),
    fontWeight: 'bold',
    color: '#015185',
    marginTop: scale(10),
  },
  formContainer: {
    width: '100%',
  },
  inputContainer: {
    marginBottom: scale(8),
    marginHorizontal: scale(10),
  },
  inputGradient: {
    borderRadius: scale(8),
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  input: {
    backgroundColor: '#fff',
    height: scale(56),
    borderRadius: scale(8),
    fontSize: scale(16),
  },
  gradientButton: {
    borderRadius: scale(6),
    marginVertical: scale(10),
    width: '95%',
    justifyContent: 'center',
    alignSelf: 'center',
  },
  buttonTouchable: {
    alignItems: 'center',
    justifyContent: 'center',
    height: scale(48),
    borderRadius: scale(6),
    width: '100%',
  },
  googleButtonText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  dividerContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    width: '100%',
    marginVertical: scale(20),
  },
  borderLine: {
    flex: 1,
    height: 1,
    borderStyle: 'dashed',
    borderWidth: 1,
    borderColor: '#888',
  },
  dividerText: {
    fontSize: scale(16),
    color: '#888',
    fontWeight: 'bold',
    textAlign: 'center',
    paddingHorizontal: scale(10),
  },
  networkErrorBanner: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    backgroundColor: '#ff0000',
    paddingVertical: scale(10),
    alignItems: 'center',
    zIndex: 1000,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  errorText: {
    color: 'red',
    fontSize: scale(14),
    marginLeft: scale(10),
    marginTop: scale(4),
    position: 'absolute',
    bottom: scale(-10),
  },
  mainemailpass: {
    marginBottom: scale(12),
    position: 'relative',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: scale(18),
    marginBottom: scale(10),
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: scale(24),
    height: scale(24),
    marginRight: scale(8),
  },
  syncModalTextadd: {
    fontSize: scale(14),
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});
