import React, { useState } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, StatusBar, Dimensions, ScrollView, Alert, ActivityIndicator, Modal, Platform, Linking, Share, PermissionsAndroid } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import DateTimePicker from '@react-native-community/datetimepicker';
import DropDownPicker from 'react-native-dropdown-picker';
import XLSX from 'xlsx';
import RNFS from 'react-native-fs';
import API_ENDPOINTS from '../../../../config/mongodb.endpoints';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.32;

const REPORT_MODULES = [
  { label: 'Inspection Report', value: 'inspection' },
  { label: 'Incident Report', value: 'incident' },
  { label: 'Operator Daily Reporting', value: 'dailyReport' },
  { label: 'Service Schedule', value: 'serviceSchedule' },
  { label: 'Fuel Report', value: 'fuelLog' },
  { label: 'Request Maintenance', value: 'maintenance' },
  { label: 'Job Cards', value: 'jobCards' },
  { label: 'Team Employee Data', value: 'employees' },
];

const SitesKpi = () => {
  const navigation = useNavigation();
  const [selectedModule, setSelectedModule] = useState(null);
  const [dropdownOpen, setDropdownOpen] = useState(false);
  const [startDate, setStartDate] = useState(new Date());
  const [endDate, setEndDate] = useState(new Date());
  const [showStartPicker, setShowStartPicker] = useState(false);
  const [showEndPicker, setShowEndPicker] = useState(false);
  const [exporting, setExporting] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [exportedFilePath, setExportedFilePath] = useState('');
  const [exportedFileName, setExportedFileName] = useState('');
  const [exportedRecordCount, setExportedRecordCount] = useState(0);
  const [exportFormat, setExportFormat] = useState('excel'); // 'excel' or 'csv'

  const needsDateRange = selectedModule && selectedModule !== 'employees';

  const handleExport = async () => {
    if (!selectedModule) {
      Alert.alert('Error', 'Please select a report module to export');
      return;
    }

    if (needsDateRange && startDate > endDate) {
      Alert.alert('Error', 'Start date must be before end date');
      return;
    }

    console.log('Starting export for module:', selectedModule);
    console.log('Date range:', startDate.toLocaleDateString(), 'to', endDate.toLocaleDateString());

    setExporting(true);
    try {
      let data = [];
      let fileName = '';

      switch (selectedModule) {
        case 'inspection':
          data = await fetchInspectionReports();
          fileName = `Inspection_Report_${formatDate(startDate)}_to_${formatDate(endDate)}.xlsx`;
          break;
        case 'incident':
          data = await fetchIncidentReports();
          fileName = `Incident_Report_${formatDate(startDate)}_to_${formatDate(endDate)}.xlsx`;
          break;
        case 'dailyReport':
          data = await fetchDailyReports();
          fileName = `Daily_Report_${formatDate(startDate)}_to_${formatDate(endDate)}.xlsx`;
          break;
        case 'serviceSchedule':
          data = await fetchServiceSchedules();
          fileName = `Service_Schedule_${formatDate(startDate)}_to_${formatDate(endDate)}.xlsx`;
          break;
        case 'fuelLog':
          data = await fetchFuelLogs();
          fileName = `Fuel_Log_${formatDate(startDate)}_to_${formatDate(endDate)}.xlsx`;
          break;
        case 'maintenance':
          data = await fetchMaintenanceRequests();
          fileName = `Maintenance_Report_${formatDate(startDate)}_to_${formatDate(endDate)}.xlsx`;
          break;
        case 'jobCards':
          data = await fetchJobCards();
          fileName = `Job_Cards_Report_${formatDate(startDate)}_to_${formatDate(endDate)}.xlsx`;
          break;
        case 'employees':
          data = await fetchEmployeeData();
          fileName = `Team_Employee_Data_${formatDate(new Date())}.xlsx`;
          break;
        default:
          Alert.alert('Error', 'Invalid module selected');
          setExporting(false);
          return;
      }

      if (data.length === 0) {
        Alert.alert('No Data', 'No data found for the selected criteria');
        setExporting(false);
        return;
      }

      console.log(`Exporting ${data.length} records in ${exportFormat} format...`);
      
      let filePath;
      if (exportFormat === 'csv') {
        // Change file extension to .csv
        const csvFileName = fileName.replace('.xlsx', '.csv');
        filePath = await exportToCSV(data, csvFileName);
      } else {
        filePath = await exportToExcel(data, fileName);
      }
      
      // Store export details and show success modal
      setExportedFilePath(filePath);
      setExportedFileName(exportFormat === 'csv' ? fileName.replace('.xlsx', '.csv') : fileName);
      setExportedRecordCount(data.length);
      setShowSuccessModal(true);
    } catch (error) {
      console.error('Export error:', error);
      Alert.alert('Error', `Failed to export data: ${error.message}\n\nPlease check console logs for details.`);
    }
    setExporting(false);
  };

  const requestStoragePermission = async () => {
    if (Platform.OS !== 'android') {
      return true; // iOS doesn't need explicit storage permission
    }

    try {
      // For Android 13+ (API 33+), we don't need WRITE_EXTERNAL_STORAGE
      if (Platform.Version >= 33) {
        console.log('Android 13+: No storage permission needed');
        return true;
      }

      // For Android 10-12, check if permission is already granted
      const granted = await PermissionsAndroid.check(
        PermissionsAndroid.PERMISSIONS.WRITE_EXTERNAL_STORAGE
      );

      if (granted) {
        console.log('Storage permission already granted');
        return true;
      }

      // Request permission
      const result = await PermissionsAndroid.request(
        PermissionsAndroid.PERMISSIONS.WRITE_EXTERNAL_STORAGE,
        {
          title: 'Storage Permission Required',
          message: 'This app needs access to your storage to save exported files.',
          buttonPositive: 'Grant',
          buttonNegative: 'Deny',
        }
      );

      if (result === PermissionsAndroid.RESULTS.GRANTED) {
        console.log('Storage permission granted');
        return true;
      } else {
        console.log('Storage permission denied');
        Alert.alert(
          'Permission Denied',
          'Storage permission is required to download files. Please enable it in app settings.',
          [
            { text: 'Cancel', style: 'cancel' },
            { text: 'Open Settings', onPress: () => Linking.openSettings() }
          ]
        );
        return false;
      }
    } catch (error) {
      console.error('Error requesting storage permission:', error);
      return false;
    }
  };

  const handleDownloadFile = async () => {
    try {
      console.log('Starting auto-download...');
      
      // Request storage permission first
      const hasPermission = await requestStoragePermission();
      if (!hasPermission) {
        console.log('Storage permission not granted, aborting download');
        return;
      }

      console.log('Permission granted, proceeding with download');
      console.log('File already saved at:', exportedFilePath);
      
      // Verify file exists
      const fileExists = await RNFS.exists(exportedFilePath);
      if (!fileExists) {
        throw new Error('File does not exist at path: ' + exportedFilePath);
      }

      // Get file stats
      const fileStats = await RNFS.stat(exportedFilePath);
      console.log('File verified - Size:', fileStats.size, 'bytes');
      
      // File is already downloaded to the device's Downloads folder
      // Now just show success and close modal
      closeSuccessModal();
      
      Alert.alert(
        '✅ Download Complete!',
        `Your ${exportFormat.toUpperCase()} file has been downloaded successfully!\n\n` +
        `📄 File: ${exportedFileName}\n` +
        `📁 Location: Downloads folder\n` +
        `📊 Size: ${(fileStats.size / 1024).toFixed(2)} KB\n\n` +
        `You can open it from your Downloads folder or any file manager app.`,
        [
          { 
            text: 'Open File', 
            onPress: () => openDownloadedFile()
          },
          { 
            text: 'OK',
            style: 'cancel'
          }
        ]
      );
      
      console.log('✅ Auto-download completed successfully');
    } catch (error) {
      console.error('Error in handleDownloadFile:', error);
      closeSuccessModal();
      Alert.alert(
        'Download Complete',
        `Your ${exportFormat.toUpperCase()} file has been saved to your Downloads folder.\n\n` +
        `File: ${exportedFileName}\n` +
        `Location: ${exportedFilePath}\n\n` +
        `You can access it from your file manager.`,
        [{ text: 'OK' }]
      );
    }
  };

  const openDownloadedFile = async () => {
    try {
      if (Platform.OS === 'android') {
        const FileViewer = require('react-native-file-viewer').default;
        await FileViewer.open(exportedFilePath, { 
          showOpenWithDialog: true,
          showAppsSuggestions: true,
        });
        console.log('File opened with external app');
      } else {
        // For iOS, use Share to open
        await Share.share({
          url: exportedFilePath,
          title: 'Open File',
        });
      }
    } catch (error) {
      console.log('Error opening file:', error);
      Alert.alert(
        'Info',
        `Please open the file manually from your Downloads folder.\n\nFile: ${exportedFileName}`
      );
    }
  };

  const closeSuccessModal = () => {
    setShowSuccessModal(false);
    setExportedFilePath('');
    setExportedFileName('');
    setExportedRecordCount(0);
  };

  const formatDate = (date) => {
    const d = new Date(date);
    return `${d.getDate()}-${d.getMonth() + 1}-${d.getFullYear()}`;
  };

  const fetchInspectionReports = async () => {
    console.log('Fetching inspection reports...');
    const response = await fetch(`${API_ENDPOINTS.INSPECTIONS.GET_ALL}?page=1&limit=10000`);
    const result = await response.json();
    const allData = result.success ? result.data : [];
    console.log(`Total inspections fetched: ${allData.length}`);

    // Fetch job cards for linking
    const jobCardsResponse = await fetch(`${API_ENDPOINTS.JOB_CARDS.GET_ALL}?page=1&limit=10000`);
    const jobCardsResult = await jobCardsResponse.json();
    const jobCards = jobCardsResult.success ? jobCardsResult.data : [];
    console.log(`Total job cards fetched: ${jobCards.length}`);
    
    const jobCardMap = {};
    jobCards.forEach(jc => {
      const linkId = jc.inspectionId || jc.linkedInspectionId;
      if (linkId) {
        jobCardMap[linkId] = jc;
      }
    });

    // Normalize dates for comparison
    const fromDateNormalized = new Date(startDate);
    fromDateNormalized.setHours(0, 0, 0, 0);
    const toDateNormalized = new Date(endDate);
    toDateNormalized.setHours(23, 59, 59, 999);

    const filtered = allData.filter(item => {
      const date = new Date(item.createdAt || item.step3?.createdAt);
      return date >= fromDateNormalized && date <= toDateNormalized;
    });
    console.log(`Inspections in date range: ${filtered.length}`);

    return filtered.map(item => {
      const jobCard = jobCardMap[item._id];
      return {
        'Request Number': item.requestNumber || '-',
        'Equipment': item.step1?.selectedEquipment?.equipmentName || '-',
        'Equipment Number': item.step1?.selectedEquipment?.equipmentNumber || '-',
        'Country': item.step1?.country || '-',
        'Project': item.step1?.project || '-',
        'Inspector': item.step2?.inspectorName || '-',
        'Status': item.step3?.status || item.status || '-',
        'Created Date': new Date(item.createdAt).toLocaleDateString(),
        'Job Card Number': jobCard?.title || jobCard?.requestNumber || 'No Job Card',
        'Job Card Status': jobCard?.status || '-',
        'Created By': item.userId || item.createdBy || '-',
      };
    });
  };

  const fetchIncidentReports = async () => {
    const response = await fetch(API_ENDPOINTS.INCIDENT_REPORTS.GET_ALL);
    const result = await response.json();
    const allData = result.success ? result.data : [];

    // Normalize dates for comparison
    const fromDateNormalized = new Date(startDate);
    fromDateNormalized.setHours(0, 0, 0, 0);
    const toDateNormalized = new Date(endDate);
    toDateNormalized.setHours(23, 59, 59, 999);

    const filtered = allData.filter(item => {
      const date = new Date(item.createdAt || item.incidentDate);
      return date >= fromDateNormalized && date <= toDateNormalized;
    });

    return filtered.map(item => ({
      'Incident Number': item.incidentNumber || '-',
      'Date': new Date(item.incidentDate || item.createdAt).toLocaleDateString(),
      'Area': item.incidentArea || '-',
      'Severity': item.severity || '-',
      'Category': item.incidentCategory || '-',
      'Country': item.country || '-',
      'Project': item.project || '-',
      'Equipment Involved': item.involvesEquipment ? 'Yes' : 'No',
      'Vehicle Name': item.vehicleName || '-',
      'Person Involved': item.personInvolved || '-',
      'Location': item.gpsAddress || '-',
      'Created By': item.adminUid || '-',
    }));
  };

  const fetchDailyReports = async () => {
    const response = await fetch(API_ENDPOINTS.DAILY_REPORTS.GET_ALL);
    const result = await response.json();
    const allData = result.success ? result.data : [];

    // Normalize dates for comparison
    const fromDateNormalized = new Date(startDate);
    fromDateNormalized.setHours(0, 0, 0, 0);
    const toDateNormalized = new Date(endDate);
    toDateNormalized.setHours(23, 59, 59, 999);

    const filtered = allData.filter(item => {
      const date = new Date(item.createdAt);
      return date >= fromDateNormalized && date <= toDateNormalized;
    });

    return filtered.map(item => ({
      'Request Number': item.requestNumber || '-',
      'Equipment': item.selectedEquipment?.equipmentName || '-',
      'Country': item.step1?.country || '-',
      'Project': item.step1?.project || '-',
      'Operator': item.step1?.operatorName || '-',
      'Date': new Date(item.createdAt).toLocaleDateString(),
      'Status': item.status || '-',
      'Created By': item.adminUid || '-',
    }));
  };

  const fetchServiceSchedules = async () => {
    const response = await fetch(API_ENDPOINTS.SERVICE_SCHEDULES.GET_ALL);
    const result = await response.json();
    const allData = result.success ? result.data : [];

    // Normalize dates for comparison
    const fromDateNormalized = new Date(startDate);
    fromDateNormalized.setHours(0, 0, 0, 0);
    const toDateNormalized = new Date(endDate);
    toDateNormalized.setHours(23, 59, 59, 999);

    const filtered = allData.filter(item => {
      const date = new Date(item.createdAt);
      return date >= fromDateNormalized && date <= toDateNormalized;
    });

    return filtered.map(item => ({
      'Schedule ID': item.scheduleId || '-',
      'Title': item.title || '-',
      'Equipment': item.equipment?.equipmentName || '-',
      'Country': item.country || '-',
      'Project': item.project || '-',
      'Last Service Date': item.lastServiceDate ? new Date(item.lastServiceDate).toLocaleDateString() : '-',
      'Due Date': item.dueDate ? new Date(item.dueDate).toLocaleDateString() : '-',
      'Status': item.status || '-',
      'Repeat Type': item.repeatType || '-',
      'Created Date': new Date(item.createdAt).toLocaleDateString(),
      'Created By': item.adminUid || '-',
    }));
  };

  const fetchFuelLogs = async () => {
    const response = await fetch(API_ENDPOINTS.FUEL_LOGS.GET_ALL);
    const result = await response.json();
    const allData = result.success ? result.data : [];

    // Normalize dates for comparison
    const fromDateNormalized = new Date(startDate);
    fromDateNormalized.setHours(0, 0, 0, 0);
    const toDateNormalized = new Date(endDate);
    toDateNormalized.setHours(23, 59, 59, 999);

    const filtered = allData.filter(item => {
      const date = new Date(item.createdAt);
      return date >= fromDateNormalized && date <= toDateNormalized;
    });

    return filtered.map(item => ({
      'Request Number': item.requestNumber || '-',
      'Equipment': item.equipmentName || '-',
      'Equipment Number': item.equipmentNo || '-',
      'Country': item.country || '-',
      'Project': item.project || '-',
      'Fuel Quantity': item.fuelQuantity || '-',
      'Fuel Unit': item.fuelUnit || '-',
      'Fuel Type': item.fuelType || '-',
      'Fuel Cost': item.fuelCost || '-',
      'Source': item.selectedSource || '-',
      'Receiver': item.receiverName || '-',
      'Date': new Date(item.createdAt).toLocaleDateString(),
      'Created By': item.adminUid || '-',
    }));
  };

  const fetchMaintenanceRequests = async () => {
    const response = await fetch(API_ENDPOINTS.MAINTENANCE.GET_ALL);
    const result = await response.json();
    const allData = result.success ? result.data : [];

    // Normalize dates for comparison
    const fromDateNormalized = new Date(startDate);
    fromDateNormalized.setHours(0, 0, 0, 0);
    const toDateNormalized = new Date(endDate);
    toDateNormalized.setHours(23, 59, 59, 999);

    const filtered = allData.filter(item => {
      const date = new Date(item.createdAt);
      return date >= fromDateNormalized && date <= toDateNormalized;
    });

    return filtered.map(item => ({
      'Request Number': item.requestNumber || '-',
      'Title': item.workOrderTitle || '-',
      'Type': item.requestType || '-',
      'Priority': item.priority || '-',
      'Equipment': item.equipment?.equipmentName || '-',
      'Country': item.country || '-',
      'Project': item.project || '-',
      'Status': item.status || '-',
      'Description': item.description || '-',
      'Created Date': new Date(item.createdAt).toLocaleDateString(),
      'Created By': item.adminUid || item.userId || '-',
    }));
  };

  const fetchJobCards = async () => {
    const response = await fetch(`${API_ENDPOINTS.JOB_CARDS.GET_ALL}?page=1&limit=10000`);
    const result = await response.json();
    const allData = result.success ? result.data : [];

    // Fetch inspections for linking
    const inspectionsResponse = await fetch(`${API_ENDPOINTS.INSPECTIONS.GET_ALL}?page=1&limit=10000`);
    const inspectionsResult = await inspectionsResponse.json();
    const inspections = inspectionsResult.success ? inspectionsResult.data : [];
    
    const inspectionMap = {};
    inspections.forEach(insp => {
      inspectionMap[insp._id] = insp;
    });

    // Normalize dates for comparison
    const fromDateNormalized = new Date(startDate);
    fromDateNormalized.setHours(0, 0, 0, 0);
    const toDateNormalized = new Date(endDate);
    toDateNormalized.setHours(23, 59, 59, 999);

    const filtered = allData.filter(item => {
      const date = new Date(item.createdAt);
      return date >= fromDateNormalized && date <= toDateNormalized;
    });

    return filtered.map(item => {
      const inspection = inspectionMap[item.inspectionId || item.linkedInspectionId];
      return {
        'Job Card Number': item.title || item.requestNumber || '-',
        'Status': item.status || '-',
        'Priority': item.priority || '-',
        'Equipment': item.equipment?.equipmentName || '-',
        'Assigned To': Array.isArray(item.assignTo) ? item.assignTo.join(', ') : '-',
        'Start Date': item.startDate ? new Date(item.startDate).toLocaleDateString() : '-',
        'Due Date': item.dueDate ? new Date(item.dueDate).toLocaleDateString() : '-',
        'Description': item.description || '-',
        'Location': item.location || '-',
        'Inspection Number': inspection?.requestNumber || 'No Inspection',
        'Created Date': new Date(item.createdAt).toLocaleDateString(),
        'Created By': item.adminUid || item.createdBy || '-',
      };
    });
  };

  const fetchEmployeeData = async () => {
    const response = await fetch(API_ENDPOINTS.USERS.GET_ALL);
    const result = await response.json();
    const allData = result.success ? result.data : [];

    return allData.map(user => ({
      'Employee Number': user.employeeNumber || '-',
      'Name': user.fullName || user.name || '-',
      'Email': user.email || '-',
      'Role': user.role || '-',
      'Department': user.department || '-',
      'Country': Array.isArray(user.countries) ? user.countries.join(', ') : (user.country || '-'),
      'Project': Array.isArray(user.projects) ? user.projects.join(', ') : (user.project || '-'),
      'Login Status': user.loginStatus || '-',
      'Last Login': user.lastLogin ? new Date(user.lastLogin).toLocaleDateString() : '-',
      'Created Date': user.uploadTimestamp ? new Date(user.uploadTimestamp).toLocaleDateString() : '-',
    }));
  };

  const exportToExcel = async (data, fileName) => {
    try {
      console.log('Creating Excel workbook...');
      const ws = XLSX.utils.json_to_sheet(data);
      const wb = XLSX.utils.book_new();
      XLSX.utils.book_append_sheet(wb, ws, 'Report');

      console.log('Writing Excel file...');
      const wbout = XLSX.write(wb, { type: 'binary', bookType: 'xlsx' });

      const path = `${RNFS.DownloadDirectoryPath}/${fileName}`;
      console.log('Saving to path:', path);
      
      await RNFS.writeFile(path, wbout, 'ascii');
      
      // Verify file was created
      const fileExists = await RNFS.exists(path);
      if (fileExists) {
        console.log('✅ Excel file successfully saved to:', path);
        return path; // Return the file path
      } else {
        throw new Error('File was not created');
      }
    } catch (error) {
      console.error('Error in exportToExcel:', error);
      throw error;
    }
  };

  const exportToCSV = async (data, fileName) => {
    try {
      console.log('Creating CSV file...');
      
      if (data.length === 0) {
        throw new Error('No data to export');
      }

      // Get headers from first object
      const headers = Object.keys(data[0]);
      
      // Create CSV content
      let csvContent = headers.join(',') + '\n';
      
      // Add data rows
      data.forEach(row => {
        const values = headers.map(header => {
          const value = row[header];
          // Escape quotes and wrap in quotes if contains comma, quote, or newline
          if (value === null || value === undefined) {
            return '';
          }
          const stringValue = String(value);
          if (stringValue.includes(',') || stringValue.includes('"') || stringValue.includes('\n')) {
            return `"${stringValue.replace(/"/g, '""')}"`;
          }
          return stringValue;
        });
        csvContent += values.join(',') + '\n';
      });

      const path = `${RNFS.DownloadDirectoryPath}/${fileName}`;
      console.log('Saving CSV to path:', path);
      
      await RNFS.writeFile(path, csvContent, 'utf8');
      
      // Verify file was created
      const fileExists = await RNFS.exists(path);
      if (fileExists) {
        console.log('✅ CSV file successfully saved to:', path);
        return path; // Return the file path
      } else {
        throw new Error('File was not created');
      }
    } catch (error) {
      console.error('Error in exportToCSV:', error);
      throw error;
    }
  };

  return (
    <View style={{ flex: 1, backgroundColor: '#fff' }}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={styles.headerLeftIcon}>
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Site Excel Reports</Text>
          <View style={styles.headerRightIcons} />
        </View>
      </LinearGradient>
      
      <ScrollView style={styles.container}>
        <View style={styles.card}>
          <Text style={styles.cardTitle}>Analytics Report</Text>
          
          {/* Module Dropdown */}
          <View style={styles.inputGroup}>
            <Text style={styles.label}>Select Report Module</Text>
            <DropDownPicker
              open={dropdownOpen}
              value={selectedModule}
              items={REPORT_MODULES}
              setOpen={setDropdownOpen}
              setValue={setSelectedModule}
              placeholder="Select a module..."
              style={styles.dropdown}
              dropDownContainerStyle={styles.dropdownContainer}
              textStyle={styles.dropdownText}
              zIndex={3000}
              zIndexInverse={1000}
            />
          </View>

          {/* Info message for Employee Data */}
          {selectedModule === 'employees' && (
            <View style={styles.infoBox}>
              <MaterialIcons name="info" size={20} color="#015185" />
              <Text style={styles.infoText}>
                Team Employee Data exports all employee records without date filtering.
              </Text>
            </View>
          )}

          {/* Date Range Selection (conditional) */}
          {needsDateRange && (
            <>
              <View style={styles.inputGroup}>
                <Text style={styles.label}>Start Date</Text>
                <TouchableOpacity 
                  style={styles.dateButton} 
                  onPress={() => setShowStartPicker(true)}
                >
                  <Text style={styles.dateText}>{startDate.toLocaleDateString()}</Text>
                  <MaterialIcons name="calendar-today" size={20} color="#015185" />
                </TouchableOpacity>
              </View>

              <View style={styles.inputGroup}>
                <Text style={styles.label}>End Date</Text>
                <TouchableOpacity 
                  style={styles.dateButton} 
                  onPress={() => setShowEndPicker(true)}
                >
                  <Text style={styles.dateText}>{endDate.toLocaleDateString()}</Text>
                  <MaterialIcons name="calendar-today" size={20} color="#015185" />
                </TouchableOpacity>
              </View>
            </>
          )}

          {/* Format Selection */}
          <View style={styles.formatSection}>
            <Text style={styles.label}>Export Format</Text>
            <View style={styles.formatButtons}>
              <TouchableOpacity 
                style={[styles.formatButton, exportFormat === 'excel' && styles.formatButtonActive]}
                onPress={() => setExportFormat('excel')}
              >
                <MaterialIcons 
                  name="table-chart" 
                  size={20} 
                  color={exportFormat === 'excel' ? '#fff' : '#015185'} 
                />
                <Text style={[styles.formatButtonText, exportFormat === 'excel' && styles.formatButtonTextActive]}>
                  Excel (.xlsx)
                </Text>
              </TouchableOpacity>
              
              <TouchableOpacity 
                style={[styles.formatButton, exportFormat === 'csv' && styles.formatButtonActive]}
                onPress={() => setExportFormat('csv')}
              >
                <MaterialIcons 
                  name="insert-drive-file" 
                  size={20} 
                  color={exportFormat === 'csv' ? '#fff' : '#015185'} 
                />
                <Text style={[styles.formatButtonText, exportFormat === 'csv' && styles.formatButtonTextActive]}>
                  CSV (.csv)
                </Text>
              </TouchableOpacity>
            </View>
          </View>

          {/* Export Button */}
          <TouchableOpacity 
            style={[styles.exportButton, exporting && styles.exportButtonDisabled]} 
            onPress={handleExport}
            disabled={exporting}
          >
            {exporting ? (
              <ActivityIndicator color="#fff" />
            ) : (
              <>
                <MaterialIcons name="file-download" size={24} color="#fff" style={{ marginRight: 8 }} />
                <Text style={styles.exportButtonText}>
                  Export {exportFormat === 'excel' ? 'Excel' : 'CSV'} File
                </Text>
              </>
            )}
          </TouchableOpacity>
        </View>
      </ScrollView>

      {/* Date Pickers */}
      {showStartPicker && (
        <DateTimePicker
          value={startDate}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => {
            setShowStartPicker(false);
            if (event.type === 'set' && selectedDate) {
              console.log('Start date selected:', selectedDate.toLocaleDateString());
              setStartDate(selectedDate);
            }
          }}
          maximumDate={endDate}
        />
      )}
      
      {showEndPicker && (
        <DateTimePicker
          value={endDate}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => {
            setShowEndPicker(false);
            if (event.type === 'set' && selectedDate) {
              console.log('End date selected:', selectedDate.toLocaleDateString());
              setEndDate(selectedDate);
            }
          }}
          minimumDate={startDate}
          maximumDate={new Date()}
        />
      )}

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent={true}
        animationType="fade"
        onRequestClose={closeSuccessModal}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.successIconContainer}>
              <MaterialIcons name="check-circle" size={64} color="#4CAF50" />
            </View>
            
            <Text style={styles.modalTitle}>Export Successful!</Text>
            <Text style={styles.modalSubtitle}>
              Your {exportFormat === 'excel' ? 'Excel' : 'CSV'} file has been saved to Downloads
            </Text>
            
            <View style={styles.exportDetailsBox}>
              <View style={styles.detailRow}>
                <MaterialIcons name="description" size={20} color="#666" />
                <Text style={styles.detailLabel}>File:</Text>
                <Text style={styles.detailValue} numberOfLines={1}>{exportedFileName}</Text>
              </View>
              
              <View style={styles.detailRow}>
                <MaterialIcons name="insert-chart" size={20} color="#666" />
                <Text style={styles.detailLabel}>Format:</Text>
                <Text style={styles.detailValue}>
                  {exportFormat === 'excel' ? 'Microsoft Excel (.xlsx)' : 'Comma Separated Values (.csv)'}
                </Text>
              </View>
              
              <View style={styles.detailRow}>
                <MaterialIcons name="format-list-numbered" size={20} color="#666" />
                <Text style={styles.detailLabel}>Records:</Text>
                <Text style={styles.detailValue}>{exportedRecordCount}</Text>
              </View>
              
              <View style={styles.detailRow}>
                <MaterialIcons name="folder" size={20} color="#666" />
                <Text style={styles.detailLabel}>Location:</Text>
                <Text style={styles.detailValue} numberOfLines={1}>Downloads Folder</Text>
              </View>
            </View>
            
            <TouchableOpacity 
              style={styles.downloadButton} 
              onPress={handleDownloadFile}
            >
              <MaterialIcons name="file-download" size={24} color="#fff" />
              <Text style={styles.downloadButtonText}>
                Download {exportFormat === 'excel' ? 'Excel' : 'CSV'} File
              </Text>
            </TouchableOpacity>
            
            <TouchableOpacity 
              style={styles.closeButton} 
              onPress={closeSuccessModal}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  header: {
    width: '100%',
    justifyContent: 'flex-end',
    paddingBottom: width * 0.04,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.05,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeftIcon: {
    paddingVertical: width * 0.02,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.055,
    fontWeight: '700',
    flex: 1,
    marginLeft: width * 0.03,
    textAlign: 'center',
  },
  headerRightIcons: {
    width: width * 0.06 + width * 0.02,
  },
  container: {
    flex: 1,
    padding: 16,
  },
  card: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 8,
    elevation: 3,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  cardTitle: {
    fontSize: 24,
    fontWeight: '700',
    color: '#015185',
    marginBottom: 24,
    textAlign: 'center',
  },
  inputGroup: {
    marginBottom: 20,
  },
  label: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
  },
  dropdown: {
    borderColor: '#d0d0d0',
    borderWidth: 1,
    borderRadius: 8,
    backgroundColor: '#fff',
    minHeight: 50,
  },
  dropdownContainer: {
    borderColor: '#d0d0d0',
    borderWidth: 1,
    borderRadius: 8,
  },
  dropdownText: {
    fontSize: 15,
    color: '#333',
  },
  dateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#d0d0d0',
    borderRadius: 8,
    padding: 12,
    backgroundColor: '#fff',
  },
  dateText: {
    fontSize: 15,
    color: '#333',
  },
  exportButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#015185',
    borderRadius: 8,
    padding: 16,
    marginTop: 24,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  exportButtonDisabled: {
    backgroundColor: '#8aa3b8',
  },
  exportButtonText: {
    color: '#fff',
    fontSize: 18,
    fontWeight: '700',
  },
  infoBox: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#E3F2FD',
    borderRadius: 8,
    padding: 12,
    marginTop: -8,
    marginBottom: 16,
    borderLeftWidth: 4,
    borderLeftColor: '#015185',
  },
  infoText: {
    flex: 1,
    fontSize: 14,
    color: '#015185',
    marginLeft: 8,
    lineHeight: 20,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
    elevation: 8,
  },
  successIconContainer: {
    alignItems: 'center',
    marginBottom: 16,
  },
  modalTitle: {
    fontSize: 24,
    fontWeight: '700',
    color: '#333',
    textAlign: 'center',
    marginBottom: 8,
  },
  modalSubtitle: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginBottom: 20,
  },
  exportDetailsBox: {
    backgroundColor: '#F5F5F5',
    borderRadius: 8,
    padding: 16,
    marginBottom: 20,
  },
  detailRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 12,
  },
  detailLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#666',
    marginLeft: 8,
    marginRight: 8,
    minWidth: 70,
  },
  detailValue: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  downloadButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#4CAF50',
    borderRadius: 8,
    padding: 16,
    marginBottom: 12,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  downloadButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '700',
    marginLeft: 8,
  },
  closeButton: {
    alignItems: 'center',
    padding: 12,
  },
  closeButtonText: {
    color: '#666',
    fontSize: 16,
    fontWeight: '600',
  },
  formatSection: {
    marginBottom: 20,
  },
  formatButtons: {
    flexDirection: 'row',
    gap: 12,
  },
  formatButton: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#fff',
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 8,
    padding: 12,
    gap: 8,
  },
  formatButtonActive: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  formatButtonText: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
  },
  formatButtonTextActive: {
    color: '#fff',
  },
});

export default SitesKpi;