import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  FlatList,
  Modal,
  Animated,
  ActivityIndicator,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useIsFocused } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DateTimePicker from '@react-native-community/datetimepicker';
import Toast from 'react-native-toast-message';
import InspectionAPI from '../../ConfigMongo/InspectionMongoConfig';
import JobCardAPI from '../../ConfigMongo/JobCardMongoConfig';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible, message = 'Loading Data' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const EmployeesPaticipated = () => {
  const navigation = useNavigation();
  const isFocused = useIsFocused();
  const isMounted = useRef(true);

  // Date states - Default to last 14 days
  const [fromDate, setFromDate] = useState(() => {
    const date = new Date();
    date.setDate(date.getDate() - 14);
    date.setHours(0, 0, 0, 0);
    return date;
  });
  const [toDate, setToDate] = useState(() => {
    const date = new Date();
    date.setHours(23, 59, 59, 999);
    return date;
  });
  const [showFromDatePicker, setShowFromDatePicker] = useState(false);
  const [showToDatePicker, setShowToDatePicker] = useState(false);

  // Data states
  const [allEmployees, setAllEmployees] = useState([]);
  const [activeJobsEmployees, setActiveJobsEmployees] = useState([]);
  const [completedJobsEmployees, setCompletedJobsEmployees] = useState([]);
  
  // UI states
  const [activeTab, setActiveTab] = useState('all'); // 'all', 'active', 'completed'
  const [loading, setLoading] = useState(false);

  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const formatDate = (date) => {
    if (!date) return 'N/A';
    const d = new Date(date);
    return d.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    });
  };

  const formatDateTime = (isoString) => {
    if (!isoString) return 'N/A';
    const date = new Date(isoString);
    return date.toLocaleString('en-US', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
      hour12: true,
    }).replace(',', '');
  };

  // Get status color
  const getStatusColor = (status) => {
    const statusLower = (status || '').toLowerCase();
    if (statusLower === 'complete') return '#2ecc40';
    if (statusLower === 'in_progress') return '#f39c12';
    if (statusLower === 'hold') return '#e74c3c';
    if (statusLower === 'assigned') return '#3498db';
    if (statusLower === 'cancelled') return '#95a5a6';
    return '#666';
  };

  // Load employee participation data
  const loadEmployeeParticipation = async () => {
    if (!isMounted.current) return;
    
    try {
      setLoading(true);
      
      console.log('🔄 Loading job cards and employee participation...');
      
      // Fetch all job cards from MongoDB
      const jobCardsResponse = await JobCardAPI.getAllJobCards({ page: 1, limit: 10000 });
      
      if (!jobCardsResponse.success || !jobCardsResponse.data) {
        showToast('error', 'Error', 'Failed to load job cards');
        return;
      }

      const jobCards = jobCardsResponse.data;
      console.log(`📊 Loaded ${jobCards.length} job cards`);

      // Filter by date range
      const filteredJobCards = jobCards.filter(jobCard => {
        const createdDate = new Date(jobCard.createdAt);
        return createdDate >= fromDate && createdDate <= toDate;
      });

      console.log(`📅 Filtered to ${filteredJobCards.length} job cards in date range`);

      // Build employee participation map
      const employeeMap = {};
      const userIds = new Set();

      filteredJobCards.forEach(jobCard => {
        // Get assigned users from the job card
        if (Array.isArray(jobCard.assignTo)) {
          jobCard.assignTo.forEach(assignee => {
            const userId = assignee.uid || assignee.id;
            if (!userId) return;

            userIds.add(userId);

            if (!employeeMap[userId]) {
              employeeMap[userId] = {
                userId,
                name: assignee.name || 'Unknown',
                email: assignee.email || '-',
                employeeNumber: '-',
                mobile: '-',
                jobs: [],
              };
            }

            // Add job card details
            employeeMap[userId].jobs.push({
              jobCardId: jobCard._id || jobCard.id,
              jobCardNumber: jobCard.requestNumber || jobCard.title || 'N/A',
              status: jobCard.status || 'Hold',
              priority: jobCard.priority || 'Medium',
              equipment: jobCard.equipment?.equipmentName || 'N/A',
              workOrderType: jobCard.workOrderType || '-',
              createdAt: jobCard.createdAt,
              dueDate: jobCard.dueDate,
              inspectionId: jobCard.linkInspection?.id || jobCard.serviceScheduleId || null,
              tasks: Array.isArray(jobCard.items) ? jobCard.items.filter(item => 
                Array.isArray(item.assignTo) && item.assignTo.some(a => (a.uid || a.id) === userId)
              ) : [],
            });
          });
        }

        // Also check items for assigned users
        if (Array.isArray(jobCard.items)) {
          jobCard.items.forEach(item => {
            if (Array.isArray(item.assignTo)) {
              item.assignTo.forEach(assignee => {
                const userId = assignee.uid || assignee.id;
                if (!userId) return;

                userIds.add(userId);

                if (!employeeMap[userId]) {
                  employeeMap[userId] = {
                    userId,
                    name: assignee.name || 'Unknown',
                    email: assignee.email || '-',
                    employeeNumber: '-',
                    mobile: '-',
                    jobs: [],
                  };
                }

                // Check if this job card is already added
                const existingJob = employeeMap[userId].jobs.find(
                  j => j.jobCardId === (jobCard._id || jobCard.id)
                );

                if (!existingJob) {
                  employeeMap[userId].jobs.push({
                    jobCardId: jobCard._id || jobCard.id,
                    jobCardNumber: jobCard.requestNumber || jobCard.title || 'N/A',
                    status: jobCard.status || 'Hold',
                    priority: jobCard.priority || 'Medium',
                    equipment: jobCard.equipment?.equipmentName || 'N/A',
                    workOrderType: jobCard.workOrderType || '-',
                    createdAt: jobCard.createdAt,
                    dueDate: jobCard.dueDate,
                    inspectionId: jobCard.linkInspection?.id || jobCard.serviceScheduleId || null,
                    tasks: [item],
                  });
                } else {
                  existingJob.tasks.push(item);
                }
              });
            }
          });
        }
      });

      // Fetch user details from MongoDB for all users
      console.log(`👥 Fetching details for ${userIds.size} employees...`);
      const userDetailsPromises = Array.from(userIds).map(async (userId) => {
        try {
          const response = await MongoAPI.getUserById(userId);
          if (response.success && response.data) {
            return {
              userId,
              data: response.data,
            };
          }
        } catch (error) {
          console.error(`Error fetching user ${userId}:`, error);
        }
        return null;
      });

      const userDetailsResults = await Promise.all(userDetailsPromises);
      
      // Update employee map with full user details
      userDetailsResults.forEach(result => {
        if (result && employeeMap[result.userId]) {
          employeeMap[result.userId].name = result.data.name || result.data.fullName || employeeMap[result.userId].name;
          employeeMap[result.userId].email = result.data.email || employeeMap[result.userId].email;
          employeeMap[result.userId].employeeNumber = result.data.employeeNumber || '-';
          employeeMap[result.userId].mobile = result.data.mobile || result.data.phone || '-';
        }
      });

      const employeesArray = Object.values(employeeMap);

      // Separate by job status
      const activeJobs = employeesArray.filter(emp => 
        emp.jobs.some(job => job.status.toLowerCase() !== 'complete')
      );

      const completedJobs = employeesArray.filter(emp => 
        emp.jobs.every(job => job.status.toLowerCase() === 'complete')
      );

      setAllEmployees(employeesArray);
      setActiveJobsEmployees(activeJobs);
      setCompletedJobsEmployees(completedJobs);

      console.log(`✅ Loaded ${employeesArray.length} employees (${activeJobs.length} active, ${completedJobs.length} completed)`);
    } catch (error) {
      console.error('Error loading employee participation:', error);
      showToast('error', 'Error', 'Failed to load employee participation data');
    } finally {
      if (isMounted.current) {
        setLoading(false);
      }
    }
  };

  // Initial load
  useEffect(() => {
    if (isFocused) {
      loadEmployeeParticipation();
    }

    return () => {
      isMounted.current = false;
    };
  }, [isFocused]);

  // Reload when dates change
  useEffect(() => {
    loadEmployeeParticipation();
  }, [fromDate, toDate]);

  // Render employee item
  const renderEmployeeItem = ({ item }) => {
    const totalJobs = item.jobs.length;
    const completedJobs = item.jobs.filter(j => j.status.toLowerCase() === 'complete').length;
    const activeJobs = totalJobs - completedJobs;

    return (
      <View style={styles.employeeCard}>
        <View style={styles.employeeHeader}>
          <View style={styles.employeeTitleRow}>
            <MaterialIcons name="person" size={24} color="#0055A5" />
            <Text style={styles.employeeName}>{item.name}</Text>
          </View>
          <View style={styles.jobStatsRow}>
            <View style={[styles.statBadge, { backgroundColor: '#f39c12' }]}>
              <Text style={styles.statText}>{activeJobs} Active</Text>
            </View>
            <View style={[styles.statBadge, { backgroundColor: '#2ecc40' }]}>
              <Text style={styles.statText}>{completedJobs} Done</Text>
            </View>
          </View>
        </View>

        <View style={styles.employeeInfo}>
          <View style={styles.infoRow}>
            <MaterialIcons name="badge" size={16} color="#666" />
            <Text style={styles.infoText}>Emp#: {item.employeeNumber}</Text>
          </View>
          <View style={styles.infoRow}>
            <MaterialIcons name="email" size={16} color="#666" />
            <Text style={styles.infoText}>{item.email}</Text>
          </View>
          <View style={styles.infoRow}>
            <MaterialIcons name="phone" size={16} color="#666" />
            <Text style={styles.infoText}>{item.mobile}</Text>
          </View>
        </View>

        <View style={styles.divider} />

        <Text style={styles.jobsTitle}>Job Cards Participated ({totalJobs})</Text>

        {item.jobs.map((job, index) => (
          <View key={`${job.jobCardId}-${index}`} style={styles.jobCardItem}>
            <View style={styles.jobCardHeader}>
              <Text style={styles.jobCardNumber}>{job.jobCardNumber}</Text>
              <View style={[styles.jobStatusBadge, { backgroundColor: getStatusColor(job.status) }]}>
                <Text style={styles.jobStatusText}>{job.status}</Text>
              </View>
            </View>

            <View style={styles.jobCardDetails}>
              <View style={styles.detailRow}>
                <Text style={styles.detailLabel}>Equipment:</Text>
                <Text style={styles.detailValue}>{job.equipment}</Text>
              </View>

              <View style={styles.detailRow}>
                <Text style={styles.detailLabel}>Work Order Type:</Text>
                <Text style={styles.detailValue}>{job.workOrderType}</Text>
              </View>

              <View style={styles.detailRow}>
                <Text style={styles.detailLabel}>Priority:</Text>
                <Text style={[styles.detailValue, { 
                  color: job.priority === 'Emergency' ? '#e74c3c' : 
                         job.priority === 'High' ? '#f39c12' : 
                         job.priority === 'Medium' ? '#3498db' : '#95a5a6',
                  fontWeight: '600'
                }]}>
                  {job.priority}
                </Text>
              </View>

              <View style={styles.detailRow}>
                <Text style={styles.detailLabel}>Created:</Text>
                <Text style={styles.detailValue}>{formatDate(job.createdAt)}</Text>
              </View>

              {job.dueDate && (
                <View style={styles.detailRow}>
                  <Text style={styles.detailLabel}>Due Date:</Text>
                  <Text style={styles.detailValue}>{formatDate(job.dueDate)}</Text>
                </View>
              )}

              {job.tasks.length > 0 && (
                <>
                  <Text style={styles.tasksTitle}>Assigned Tasks ({job.tasks.length}):</Text>
                  {job.tasks.map((task, taskIndex) => (
                    <View key={`${task.id}-${taskIndex}`} style={styles.taskItemCard}>
                      <View style={styles.taskHeader}>
                        <View style={styles.taskTitleRow}>
                          <MaterialIcons name="assignment" size={16} color="#0055A5" />
                          <Text style={styles.taskText}>{task.text || task.description || 'Task'}</Text>
                        </View>
                        <View style={[styles.taskStatusBadge, { backgroundColor: getStatusColor(task.status) }]}>
                          <Text style={styles.taskStatusText}>{task.status || 'Pending'}</Text>
                        </View>
                      </View>
                      
                      {task.type && (
                        <View style={styles.taskDetailRow}>
                          <Text style={styles.taskDetailLabel}>Type:</Text>
                          <Text style={styles.taskDetailValue}>{task.type}</Text>
                        </View>
                      )}
                      
                      {task.category && (
                        <View style={styles.taskDetailRow}>
                          <Text style={styles.taskDetailLabel}>Category:</Text>
                          <Text style={styles.taskDetailValue}>{task.category}</Text>
                        </View>
                      )}
                      
                      {task.note && (
                        <View style={styles.taskDetailRow}>
                          <Text style={styles.taskDetailLabel}>Note:</Text>
                          <Text style={styles.taskDetailValue}>{task.note}</Text>
                        </View>
                      )}
                      
                      {/* Show assigned workers for this specific task */}
                      {Array.isArray(task.assignTo) && task.assignTo.length > 0 && (
                        <View style={styles.taskWorkersSection}>
                          <Text style={styles.taskWorkersTitle}>Assigned Workers ({task.assignTo.length}):</Text>
                          {task.assignTo.map((worker, workerIndex) => (
                            <View key={`${worker.id}-${workerIndex}`} style={styles.workerItem}>
                              <MaterialIcons name="person" size={14} color="#00B894" />
                              <View style={styles.workerInfo}>
                                <Text style={styles.workerName}>{worker.fullName || worker.name || 'Unknown'}</Text>
                                <Text style={styles.workerDetails}>
                                  Emp#: {worker.employeeNumber || 'N/A'} | Role: {worker.role || 'N/A'}
                                </Text>
                                {worker.email && (
                                  <Text style={styles.workerEmail}>📧 {worker.email}</Text>
                                )}
                              </View>
                            </View>
                          ))}
                        </View>
                      )}
                    </View>
                  ))}
                </>
              )}
            </View>
          </View>
        ))}
      </View>
    );
  };

  const currentData = activeTab === 'all' ? allEmployees : 
                      activeTab === 'active' ? activeJobsEmployees : 
                      completedJobsEmployees;

  return (
    <View style={styles.container}>
      <LinearGradient colors={['#3481BC', '#025383']} style={styles.header}>
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()}>
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Employees Job Participation</Text>
          <View style={{ width: 24 }} />
        </View>
      </LinearGradient>

      {/* Date Range Selector */}
      <View style={styles.dateRangeContainer}>
        <Text style={styles.dateRangeTitle}>Date Range Filter</Text>
        
        <View style={styles.datePickersRow}>
          <View style={styles.datePickerContainer}>
            <Text style={styles.dateLabel}>From Date</Text>
            <TouchableOpacity 
              style={styles.dateButton}
              onPress={() => setShowFromDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#0055A5" />
              <Text style={styles.dateButtonText}>{formatDate(fromDate)}</Text>
            </TouchableOpacity>
          </View>

          <View style={styles.datePickerContainer}>
            <Text style={styles.dateLabel}>To Date</Text>
            <TouchableOpacity 
              style={styles.dateButton}
              onPress={() => setShowToDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#0055A5" />
              <Text style={styles.dateButtonText}>{formatDate(toDate)}</Text>
            </TouchableOpacity>
          </View>
        </View>

        {/* Date Pickers */}
        {showFromDatePicker && (
          <DateTimePicker
            value={fromDate}
            mode="date"
            display={Platform.OS === 'ios' ? 'spinner' : 'default'}
            onChange={(event, selectedDate) => {
              setShowFromDatePicker(false);
              if (selectedDate) {
                selectedDate.setHours(0, 0, 0, 0);
                setFromDate(selectedDate);
              }
            }}
          />
        )}

        {showToDatePicker && (
          <DateTimePicker
            value={toDate}
            mode="date"
            display={Platform.OS === 'ios' ? 'spinner' : 'default'}
            onChange={(event, selectedDate) => {
              setShowToDatePicker(false);
              if (selectedDate) {
                selectedDate.setHours(23, 59, 59, 999);
                setToDate(selectedDate);
              }
            }}
          />
        )}
      </View>

      {/* Tabs */}
      <View style={styles.tabsContainer}>
        <TouchableOpacity
          style={[styles.tab, activeTab === 'all' && styles.activeTab]}
          onPress={() => setActiveTab('all')}
        >
          <View style={styles.tabContent}>
            <MaterialIcons 
              name="people" 
              size={20} 
              color={activeTab === 'all' ? '#fff' : '#666'} 
            />
            <Text style={[styles.tabText, activeTab === 'all' && styles.activeTabText]}>
              All ({allEmployees.length})
            </Text>
          </View>
        </TouchableOpacity>

        <TouchableOpacity
          style={[styles.tab, activeTab === 'active' && styles.activeTab]}
          onPress={() => setActiveTab('active')}
        >
          <View style={styles.tabContent}>
            <MaterialIcons 
              name="work" 
              size={20} 
              color={activeTab === 'active' ? '#fff' : '#666'} 
            />
            <Text style={[styles.tabText, activeTab === 'active' && styles.activeTabText]}>
              Active Jobs ({activeJobsEmployees.length})
            </Text>
          </View>
        </TouchableOpacity>

        <TouchableOpacity
          style={[styles.tab, activeTab === 'completed' && styles.activeTab]}
          onPress={() => setActiveTab('completed')}
        >
          <View style={styles.tabContent}>
            <MaterialIcons 
              name="check-circle" 
              size={20} 
              color={activeTab === 'completed' ? '#fff' : '#666'} 
            />
            <Text style={[styles.tabText, activeTab === 'completed' && styles.activeTabText]}>
              Completed ({completedJobsEmployees.length})
            </Text>
          </View>
        </TouchableOpacity>
      </View>

      {/* Results Summary */}
      <View style={styles.summaryContainer}>
        <MaterialIcons name="info-outline" size={18} color="#0055A5" />
        <Text style={styles.summaryText}>
          Showing {currentData.length} employee{currentData.length !== 1 ? 's' : ''} who participated in job cards
        </Text>
      </View>

      {/* Content */}
      <View style={styles.content}>
        {loading ? (
          <View style={styles.loadingContainer}>
            <ActivityIndicator size="large" color="#0055A5" />
            <Text style={styles.loadingText}>Loading employee participation data...</Text>
          </View>
        ) : currentData.length > 0 ? (
          <FlatList
            data={currentData}
            keyExtractor={(item) => item.userId}
            renderItem={renderEmployeeItem}
            contentContainerStyle={styles.listContent}
            showsVerticalScrollIndicator={false}
          />
        ) : (
          <View style={styles.emptyContainer}>
            <MaterialIcons name="people-outline" size={60} color="#ccc" />
            <Text style={styles.emptyText}>
              No employees found in this category
            </Text>
            <Text style={styles.emptySubtext}>
              Try adjusting the date range or check back later
            </Text>
          </View>
        )}
      </View>

      <SyncModal visible={loading} message="Loading Employee Data..." />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  header: {
    paddingTop: StatusBar.currentHeight || 40,
    paddingBottom: 16,
    paddingHorizontal: 16,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  headerTitle: {
    color: '#fff',
    fontSize: 18,
    fontWeight: 'bold',
  },
  dateRangeContainer: {
    backgroundColor: '#fff',
    padding: 16,
    marginHorizontal: 16,
    marginTop: 16,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  dateRangeTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 12,
  },
  datePickersRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  datePickerContainer: {
    flex: 1,
    marginHorizontal: 4,
  },
  dateLabel: {
    fontSize: 12,
    color: '#666',
    marginBottom: 6,
    fontWeight: '500',
  },
  dateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingVertical: 10,
    paddingHorizontal: 12,
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#0055A5',
  },
  dateButtonText: {
    marginLeft: 8,
    fontSize: 14,
    color: '#0055A5',
    fontWeight: '500',
  },
  tabsContainer: {
    flexDirection: 'row',
    backgroundColor: '#fff',
    marginHorizontal: 16,
    marginTop: 12,
    borderRadius: 8,
    overflow: 'hidden',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  tab: {
    flex: 1,
    paddingVertical: 12,
    alignItems: 'center',
    backgroundColor: '#f5f5f5',
  },
  activeTab: {
    backgroundColor: '#0055A5',
  },
  tabContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  tabText: {
    fontSize: 12,
    color: '#666',
    fontWeight: '600',
    marginLeft: 4,
  },
  activeTabText: {
    color: '#fff',
  },
  summaryContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    marginHorizontal: 16,
    marginTop: 12,
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderRadius: 6,
    borderLeftWidth: 4,
    borderLeftColor: '#ffd205',
  },
  summaryText: {
    fontSize: 13,
    color: '#666',
    fontWeight: '500',
    marginLeft: 8,
  },
  content: {
    flex: 1,
    marginTop: 12,
  },
  listContent: {
    paddingHorizontal: 16,
    paddingBottom: 20,
  },
  employeeCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    marginBottom: 16,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    overflow: 'hidden',
  },
  employeeHeader: {
    backgroundColor: '#E9E9E9',
    paddingHorizontal: 16,
    paddingVertical: 12,
  },
  employeeTitleRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  employeeName: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginLeft: 8,
  },
  jobStatsRow: {
    flexDirection: 'row',
    gap: 8,
  },
  statBadge: {
    paddingHorizontal: 10,
    paddingVertical: 4,
    borderRadius: 4,
  },
  statText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
  },
  employeeInfo: {
    padding: 16,
    gap: 8,
  },
  infoRow: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 8,
  },
  infoText: {
    fontSize: 14,
    color: '#666',
  },
  divider: {
    height: 1,
    backgroundColor: '#ddd',
    marginHorizontal: 16,
  },
  jobsTitle: {
    fontSize: 15,
    fontWeight: '600',
    color: '#333',
    paddingHorizontal: 16,
    paddingTop: 12,
    paddingBottom: 8,
  },
  jobCardItem: {
    marginHorizontal: 16,
    marginBottom: 12,
    backgroundColor: '#f9f9f9',
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  jobCardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#f0f0f0',
    borderTopLeftRadius: 6,
    borderTopRightRadius: 6,
  },
  jobCardNumber: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
  },
  jobStatusBadge: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 4,
  },
  jobStatusText: {
    color: '#fff',
    fontSize: 11,
    fontWeight: '600',
  },
  jobCardDetails: {
    padding: 12,
  },
  detailRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 4,
  },
  detailLabel: {
    fontSize: 13,
    color: '#666',
  },
  detailValue: {
    fontSize: 13,
    color: '#015185',
    fontWeight: '500',
  },
  tasksTitle: {
    fontSize: 13,
    fontWeight: '600',
    color: '#333',
    marginTop: 8,
    marginBottom: 8,
  },
  taskItemCard: {
    backgroundColor: '#fff',
    borderRadius: 6,
    padding: 10,
    marginBottom: 8,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  taskHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  taskTitleRow: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
    gap: 6,
  },
  taskText: {
    flex: 1,
    fontSize: 13,
    color: '#015185',
    fontWeight: '500',
  },
  taskStatusBadge: {
    paddingHorizontal: 8,
    paddingVertical: 3,
    borderRadius: 4,
  },
  taskStatusText: {
    color: '#fff',
    fontSize: 10,
    fontWeight: '600',
  },
  taskDetailRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 3,
  },
  taskDetailLabel: {
    fontSize: 12,
    color: '#666',
  },
  taskDetailValue: {
    fontSize: 12,
    color: '#015185',
    fontWeight: '500',
  },
  taskWorkersSection: {
    marginTop: 8,
    paddingTop: 8,
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
  },
  taskWorkersTitle: {
    fontSize: 12,
    fontWeight: '600',
    color: '#333',
    marginBottom: 6,
  },
  workerItem: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    paddingVertical: 6,
    paddingHorizontal: 8,
    backgroundColor: '#f0f8ff',
    borderRadius: 4,
    marginBottom: 6,
    gap: 8,
  },
  workerInfo: {
    flex: 1,
  },
  workerName: {
    fontSize: 12,
    fontWeight: '600',
    color: '#015185',
  },
  workerDetails: {
    fontSize: 11,
    color: '#666',
    marginTop: 2,
  },
  workerEmail: {
    fontSize: 10,
    color: '#999',
    marginTop: 2,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    marginTop: 12,
    fontSize: 14,
    color: '#666',
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: 40,
  },
  emptyText: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
    marginTop: 16,
    fontWeight: '500',
  },
  emptySubtext: {
    fontSize: 14,
    color: '#999',
    textAlign: 'center',
    marginTop: 8,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default EmployeesPaticipated;
