import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  SafeAreaView,
  TouchableOpacity,
  ScrollView,
  Dimensions,
  Alert,
  ActivityIndicator,
  Modal,
  Platform,
  ToastAndroid,
  Animated,
  FlatList,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import LinearGradient from 'react-native-linear-gradient';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AntDesign from 'react-native-vector-icons/AntDesign';

// Get screen dimensions for responsive design
const { width, height } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;

// MongoDB API Configuration
const MONGODB_API_BASE = 'http://168.231.113.219:5003';
const EQUIPMENT_API_BASE = 'http://168.231.113.219:5001';
const DOCUMENTS_API_BASE = 'http://168.231.113.219:5002';

// Helper function to show toast messages based on platform
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.showWithGravity(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

// Module Categories Configuration
const MODULE_CATEGORIES = {
  'MainHome': {
    title: 'Main Home',
    icon: 'home',
    color: '#015185',
    gradient: ['#015185', '#3481BC'],
    modules: [
      { key: 'Dashboard', title: 'Dashboard', icon: 'dashboard', route: 'Dashboard' },
      { key: 'Profile', title: 'Profile', icon: 'person', route: 'Profile' },
      { key: 'Settings', title: 'Settings', icon: 'settings', route: 'Settings' },
    ]
  },
  'HSEmodule': {
    title: 'HSE Module',
    icon: 'security',
    color: '#28a745',
    gradient: ['#28a745', '#20c997'],
    modules: [
      { key: 'IncidentReport', title: 'Incident Report', icon: 'report', route: 'IncidentReport' },
      { key: 'SafetyChecklist', title: 'Safety Checklist', icon: 'checklist', route: 'SafetyChecklist' },
      { key: 'RiskAssessment', title: 'Risk Assessment', icon: 'assessment', route: 'RiskAssessment' },
      { key: 'Training', title: 'Training', icon: 'school', route: 'Training' },
    ]
  },
  'TeamMemberModule': {
    title: 'Team Member',
    icon: 'group',
    color: '#ffc107',
    gradient: ['#ffc107', '#fd7e14'],
    modules: [
      { key: 'TeamManagement', title: 'Team Management', icon: 'group', route: 'TeamManagement' },
      { key: 'Assignments', title: 'Assignments', icon: 'assignment', route: 'Assignments' },
      { key: 'Communication', title: 'Communication', icon: 'message', route: 'Communication' },
    ]
  },
  'WorkOrderC': {
    title: 'Work Orders',
    icon: 'work',
    color: '#6f42c1',
    gradient: ['#6f42c1', '#e83e8c'],
    modules: [
      { key: 'CreateWorkOrder', title: 'Create Work Order', icon: 'add-circle', route: 'CreateWorkOrder' },
      { key: 'WorkOrderList', title: 'Work Order List', icon: 'list', route: 'WorkOrderList' },
      { key: 'WorkOrderStatus', title: 'Work Order Status', icon: 'status', route: 'WorkOrderStatus' },
    ]
  },
  'HRmoduleKPI': {
    title: 'HR & KPI',
    icon: 'trending-up',
    color: '#dc3545',
    gradient: ['#dc3545', '#fd7e14'],
    modules: [
      { key: 'Performance', title: 'Performance', icon: 'trending-up', route: 'Performance' },
      { key: 'KPIs', title: 'KPIs', icon: 'analytics', route: 'KPIs' },
      { key: 'Reports', title: 'Reports', icon: 'description', route: 'Reports' },
    ]
  }
};

// SyncModal Component
const SyncModal = ({ visible, onClose, message }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Module Card Component
const ModuleCard = ({ module, category, onPress, isEnabled }) => (
  <TouchableOpacity
    style={[
      styles.moduleCard,
      { opacity: isEnabled ? 1 : 0.5 }
    ]}
    onPress={() => isEnabled && onPress(module, category)}
    activeOpacity={isEnabled ? 0.7 : 1}
  >
    <LinearGradient
      colors={isEnabled ? category.gradient : ['#ccc', '#999']}
      style={styles.moduleCardGradient}
    >
      <View style={styles.moduleIconContainer}>
        <MaterialIcons 
          name={module.icon} 
          size={scale(24)} 
          color="#fff" 
        />
      </View>
      <Text style={styles.moduleTitle}>{module.title}</Text>
      {!isEnabled && (
        <View style={styles.disabledOverlay}>
          <MaterialIcons name="lock" size={scale(16)} color="#fff" />
        </View>
      )}
    </LinearGradient>
  </TouchableOpacity>
);

// Category Section Component
const CategorySection = ({ category, modules, userModules, onModulePress }) => {
  const [isExpanded, setIsExpanded] = useState(true);

  // Filter modules based on user permissions
  const enabledModules = modules.filter(module => 
    userModules.includes(module.key) || userModules.includes('All')
  );

  if (enabledModules.length === 0) {
    return null; // Don't render category if no modules are enabled
  }

  return (
    <View style={styles.categorySection}>
      <TouchableOpacity
        style={styles.categoryHeader}
        onPress={() => setIsExpanded(!isExpanded)}
      >
        <LinearGradient colors={category.gradient} style={styles.categoryHeaderGradient}>
          <View style={styles.categoryHeaderContent}>
            <MaterialIcons name={category.icon} size={scale(20)} color="#fff" />
            <Text style={styles.categoryTitle}>{category.title}</Text>
            <MaterialIcons 
              name={isExpanded ? 'keyboard-arrow-up' : 'keyboard-arrow-down'} 
              size={scale(20)} 
              color="#fff" 
            />
          </View>
        </LinearGradient>
      </TouchableOpacity>

      {isExpanded && (
        <View style={styles.modulesGrid}>
          {modules.map((module) => {
            const isEnabled = userModules.includes(module.key) || userModules.includes('All');
            return (
              <ModuleCard
                key={module.key}
                module={module}
                category={category}
                onPress={onModulePress}
                isEnabled={isEnabled}
              />
            );
          })}
        </View>
      )}
    </View>
  );
};

export default function MongoDBHomeScreen() {
  const navigation = useNavigation();
  const [userData, setUserData] = useState(null);
  const [isLoading, setIsLoading] = useState(true);
  const [isNetworkError, setIsNetworkError] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [userModules, setUserModules] = useState([]);
  const [userCountries, setUserCountries] = useState([]);

  const networkErrorAnim = useRef(new Animated.Value(-scale(50))).current;

  const showNetworkError = () => {
    setIsNetworkError(true);
    Animated.timing(networkErrorAnim, {
      toValue: 0,
      duration: 300,
      useNativeDriver: true,
    }).start();

    setTimeout(() => {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }, 3000);
  };

  const hideNetworkError = () => {
    if (isNetworkError) {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }
  };

  useEffect(() => {
    const initialize = async () => {
      try {
        await checkNetworkStatus();
        await loadUserData();
      } catch (error) {
        console.error('Initialization error:', error);
        showNetworkError();
        showToast('error', 'Initialization Error', 'Failed to initialize');
      }
    };

    initialize();
  }, []);

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showNetworkError();
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      } else {
        hideNetworkError();
        return true;
      }
    } catch (error) {
      console.error('Network check error:', error);
      showNetworkError();
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

  const loadUserData = async () => {
    try {
      setIsLoading(true);
      setSyncModalVisible(true);

      // Get user data from AsyncStorage
      const storedUserData = await AsyncStorage.getItem('userData');
      if (storedUserData) {
        const user = JSON.parse(storedUserData);
        setUserData(user);
        setUserModules(user.modules || []);
        setUserCountries(user.countries || []);
        console.log('✅ User data loaded from storage:', user.name);
      } else {
        // If no stored data, fetch from MongoDB
        const userId = await AsyncStorage.getItem('userUid');
        if (userId) {
          const response = await fetch(`${MONGODB_API_BASE}/users/${userId}`);
          if (response.ok) {
            const userData = await response.json();
            if (userData.success) {
              const user = userData.data;
              setUserData(user);
              setUserModules(user.modules || []);
              setUserCountries(user.countries || []);
              // Store updated data
              await AsyncStorage.setItem('userData', JSON.stringify(user));
              console.log('✅ User data loaded from MongoDB:', user.name);
            }
          }
        }
      }

      setIsLoading(false);
      setSyncModalVisible(false);
    } catch (error) {
      console.error('❌ Error loading user data:', error);
      showToast('error', 'Error', 'Failed to load user data');
      setIsLoading(false);
      setSyncModalVisible(false);
    }
  };

  const handleModulePress = (module, category) => {
    try {
      console.log(`📱 Navigating to ${module.title} in ${category.title}`);
      
      // Check if user has permission for this specific module
      if (!userModules.includes(module.key) && !userModules.includes('All')) {
        showToast('error', 'Access Denied', 'You do not have permission to access this module');
        return;
      }

      // Navigate based on module route
      if (module.route) {
        navigation.navigate(module.route);
      } else {
        // Fallback navigation based on module key
        switch (module.key) {
          case 'Dashboard':
            navigation.navigate('Dashboard');
            break;
          case 'Profile':
            navigation.navigate('Profile');
            break;
          case 'Settings':
            navigation.navigate('Settings');
            break;
          default:
            showToast('info', 'Coming Soon', `${module.title} module is under development`);
        }
      }
    } catch (error) {
      console.error('❌ Error navigating to module:', error);
      showToast('error', 'Navigation Error', 'Failed to navigate to module');
    }
  };

  const handleLogout = () => {
    Alert.alert(
      'Logout',
      'Are you sure you want to logout?',
      [
        {
          text: 'Cancel',
          style: 'cancel',
        },
        {
          text: 'Logout',
          style: 'destructive',
          onPress: async () => {
            try {
              // Clear AsyncStorage
              await AsyncStorage.multiRemove([
                'userUid',
                'userEmail',
                'Loginsuccess-01',
                'ProAdminLogined',
                'adminPassword',
                'userData',
              ]);

              // Update login status in MongoDB
              if (userData && userData._id) {
                try {
                  await fetch(`${MONGODB_API_BASE}/users/${userData._id}`, {
                    method: 'PUT',
                    headers: {
                      'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                      loginStatus: false,
                    }),
                  });
                } catch (updateError) {
                  console.warn('⚠️ Failed to update logout status:', updateError);
                }
              }

              showToast('success', 'Logged Out', 'You have been logged out successfully');
              navigation.replace('MongoDBLoginScreen');
            } catch (error) {
              console.error('❌ Logout error:', error);
              showToast('error', 'Logout Error', 'Failed to logout properly');
            }
          },
        },
      ]
    );
  };

  const renderCategory = (categoryKey, category) => (
    <CategorySection
      key={categoryKey}
      category={category}
      modules={category.modules}
      userModules={userModules}
      onModulePress={handleModulePress}
    />
  );

  if (isLoading) {
    return (
      <SafeAreaView style={styles.container}>
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#015185" />
          <Text style={styles.loadingText}>Loading...</Text>
        </View>
        <SyncModal
          visible={syncModalVisible}
          onClose={() => setSyncModalVisible(false)}
          message="Loading User Data"
        />
      </SafeAreaView>
    );
  }

  return (
    <SafeAreaView style={styles.container}>
      {isNetworkError && (
        <Animated.View
          style={[styles.networkErrorBanner, { transform: [{ translateY: networkErrorAnim }] }]}
        >
          <Text style={styles.networkErrorText}>Internet connection not available</Text>
        </Animated.View>
      )}

      {/* Header */}
      <View style={styles.header}>
        <LinearGradient colors={['#015185', '#3481BC']} style={styles.headerGradient}>
          <View style={styles.headerContent}>
            <View style={styles.headerLeft}>
              <MaterialIcons name="home" size={scale(24)} color="#fff" />
              <Text style={styles.headerTitle}>Titan Drilling</Text>
            </View>
            <TouchableOpacity style={styles.logoutButton} onPress={handleLogout}>
              <MaterialIcons name="logout" size={scale(20)} color="#fff" />
            </TouchableOpacity>
          </View>
        </LinearGradient>
      </View>

      {/* User Info */}
      {userData && (
        <View style={styles.userInfo}>
          <Text style={styles.welcomeText}>Welcome, {userData.name}</Text>
          <Text style={styles.userRole}>{userData.role}</Text>
          {userData.department && (
            <Text style={styles.userDepartment}>{userData.department}</Text>
          )}
        </View>
      )}

      {/* Modules */}
      <ScrollView style={styles.scrollView} showsVerticalScrollIndicator={false}>
        <View style={styles.modulesContainer}>
          {Object.entries(MODULE_CATEGORIES).map(([categoryKey, category]) =>
            renderCategory(categoryKey, category)
          )}
        </View>
      </ScrollView>

      <Toast />
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    marginTop: scale(10),
    fontSize: scale(16),
    color: '#015185',
  },
  networkErrorBanner: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    backgroundColor: '#ff0000',
    paddingVertical: scale(10),
    alignItems: 'center',
    zIndex: 1000,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  header: {
    marginBottom: scale(10),
  },
  headerGradient: {
    paddingVertical: scale(15),
    paddingHorizontal: scale(20),
  },
  headerContent: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    fontSize: scale(20),
    fontWeight: 'bold',
    color: '#fff',
    marginLeft: scale(10),
  },
  logoutButton: {
    padding: scale(5),
  },
  userInfo: {
    backgroundColor: '#fff',
    marginHorizontal: scale(10),
    marginBottom: scale(10),
    padding: scale(15),
    borderRadius: scale(8),
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  welcomeText: {
    fontSize: scale(18),
    fontWeight: 'bold',
    color: '#333',
  },
  userRole: {
    fontSize: scale(14),
    color: '#015185',
    marginTop: scale(2),
  },
  userDepartment: {
    fontSize: scale(12),
    color: '#666',
    marginTop: scale(2),
  },
  scrollView: {
    flex: 1,
  },
  modulesContainer: {
    paddingHorizontal: scale(10),
    paddingBottom: scale(20),
  },
  categorySection: {
    marginBottom: scale(15),
  },
  categoryHeader: {
    borderRadius: scale(8),
    overflow: 'hidden',
    marginBottom: scale(10),
  },
  categoryHeaderGradient: {
    paddingVertical: scale(12),
    paddingHorizontal: scale(15),
  },
  categoryHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  categoryTitle: {
    fontSize: scale(16),
    fontWeight: 'bold',
    color: '#fff',
    marginLeft: scale(10),
    flex: 1,
  },
  modulesGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
  },
  moduleCard: {
    width: '48%',
    marginBottom: scale(10),
    borderRadius: scale(8),
    overflow: 'hidden',
  },
  moduleCardGradient: {
    padding: scale(15),
    alignItems: 'center',
    minHeight: scale(100),
    justifyContent: 'center',
  },
  moduleIconContainer: {
    marginBottom: scale(8),
  },
  moduleTitle: {
    fontSize: scale(14),
    fontWeight: 'bold',
    color: '#fff',
    textAlign: 'center',
  },
  disabledOverlay: {
    position: 'absolute',
    top: scale(5),
    right: scale(5),
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: scale(18),
    marginBottom: scale(10),
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: scale(24),
    height: scale(24),
    marginRight: scale(8),
  },
  syncModalTextadd: {
    fontSize: scale(14),
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});
