import React, { useState, useRef, useEffect } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  FlatList,
  StyleSheet,
  Modal,
  TextInput,
  Animated,
  Dimensions,
  Alert,
  ScrollView,
  PanResponder,
  Image,
} from 'react-native';
import Icon from 'react-native-vector-icons/Ionicons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import LinearGradient from 'react-native-linear-gradient';
import { getJobCard, updateJobCard, deleteJobCard, getAllJobCards } from '../../../../ConfigMongo/JobCardMongoConfig';

const { width, height } = Dimensions.get('window');

// Loader Modal with Animation (Standard SyncModal UI)
const SyncModal = ({ visible, message = 'Loading...' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const STATUS_OPTIONS = [
  { label: 'Hold', value: 'Hold', color: '#e0e0e0', textColor: '#333' },
  { label: 'Assigned', value: 'assigned', color: '#e3f2fd', textColor: '#1976d2' },
  { label: 'In Progress', value: 'in_progress', color: '#e0f7fa', textColor: '#3481BC' },
  { label: 'Complete', value: 'complete', color: '#e6f4ea', textColor: '#28a745' },
  { label: 'Cancelled', value: 'cancelled', color: '#ffebee', textColor: '#d32f2f' },
];

const TOP_TABS = ['Description', 'Remark'];
const TASK_TABS = ['Task', 'Parts', 'Labor', 'Additional Cost', 'Attach'];

const WorkerDashboardhome = ({ navigation, route }) => {
  // State Declarations
  const [selectedTopTab, setSelectedTopTab] = useState('Description');
  const [selectedStatus, setSelectedStatus] = useState('Initiated');
  const [selectedTaskTab, setSelectedTaskTab] = useState('Task');
  const [tasks, setTasks] = useState([]);
  const [checklistItems, setChecklistItems] = useState([]);
  const [workOrder, setWorkOrder] = useState(null);
  const [addItemModal, setAddItemModal] = useState(false);
  const [newItemTitle, setNewItemTitle] = useState('');
  const [newItemCategory, setNewItemCategory] = useState('Task');
  const [newItemType, setNewItemType] = useState('Repair'); // <-- Add this line
  const [loaderVisible, setLoaderVisible] = useState(false);
  const [detailsModalVisible, setDetailsModalVisible] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [userDataLoading, setUserDataLoading] = useState(true);
  const [userEmail, setUserEmail] = useState('');
  const [userRole, setUserRole] = useState('');
  const [currentUser, setCurrentUser] = useState(null);
  const [adminUid, setAdminUid] = useState(null);
  const [workOrderId, setWorkOrderId] = useState(null);
  
  // Loading states for different operations
  const [isLoadingWorkOrder, setIsLoadingWorkOrder] = useState(false);
  const [isUpdatingStatus, setIsUpdatingStatus] = useState(false);
  const [isAddingItem, setIsAddingItem] = useState(false);
  const [isAddingRemark, setIsAddingRemark] = useState(false);
  const [isUpdatingWorkOrderStatus, setIsUpdatingWorkOrderStatus] = useState(false);
  
  // --- TIMER STATE (Firebase-based) ---
  const [isTimerRunning, setIsTimerRunning] = useState(false);
  const [elapsedTime, setElapsedTime] = useState(0);
  const timerRef = useRef(null);
  const [timerStartTime, setTimerStartTime] = useState(null); // ms since epoch

  // For swipe tabs
  const [tabIndex, setTabIndex] = useState(0); // 0: Description, 1: Remark

  // Toast Notification Handler
  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Format Timer Display (HH:MM:SS)
  const formatTime = (seconds) => {
    const hrs = Math.floor(seconds / 3600);
    const mins = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    return `${hrs.toString().padStart(2, '0')}:${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
  };

  // --- TIMER: Simple local timer (no Firebase) ---
  useEffect(() => {
    let interval = null;
    if (isTimerRunning) {
      interval = setInterval(() => {
        setElapsedTime(prev => prev + 1);
      }, 1000);
    }
    return () => {
      if (interval) clearInterval(interval);
    };
  }, [isTimerRunning]);

  // --- TIMER: Start/Stop Handler (local only) ---
  const handleTimerToggle = async () => {
    if (isTimerRunning) {
      // Stop timer: save elapsed time to AsyncStorage
      if (workOrderId) {
        await AsyncStorage.setItem(`timerElapsed_${workOrderId}`, elapsedTime.toString());
      }
      setIsTimerRunning(false);
      setTimerStartTime(null);
    } else {
      // Start timer
      setIsTimerRunning(true);
      setTimerStartTime(Date.now());
    }
  };

  // Check Network Connectivity (simplified)
  const checkNetworkConnectivity = async () => {
    try {
      // Simple network check by trying to fetch a small resource
      const response = await fetch('http://168.231.113.219:5005/jobcards', { 
        method: 'GET',
        timeout: 5000 
      });
      return response.ok;
    } catch (error) {
      console.error('Network connectivity check failed:', error);
      return false;
    }
  };

  // Fetch User Data and Work Order from MongoDB
  useEffect(() => {
    const fetchUserAndWorkOrderData = async () => {
      setIsLoading(true);
      setUserDataLoading(true);
      setIsLoadingWorkOrder(true);
      try {
        console.log('📊 Starting data fetch...');
        
        // Get user data from AsyncStorage (no Firebase auth needed)
        const storedEmail = await AsyncStorage.getItem('userEmail');
        const storedRole = await AsyncStorage.getItem('userRole');
        
        console.log('✅ User data from storage:', { storedEmail, storedRole });
        setUserEmail(storedEmail || 'Unknown User');
        setUserRole(storedRole || 'Worker');
        
        // Create a mock user object for compatibility
        const mockUser = {
          email: storedEmail || 'Unknown User',
          uid: 'mock-uid'
        };
        setCurrentUser(mockUser);

        // Get admin UID from route params or AsyncStorage
        const routeAdminUid = route.params?.adminUid;
        const storedAdminUid = await AsyncStorage.getItem('userUid');
        const finalAdminUid = routeAdminUid || storedAdminUid;
        
        console.log('📊 Admin UID:', finalAdminUid);
        
        if (!finalAdminUid) {
          console.log('❌ No admin UID found');
          showToast('error', 'Error', 'Admin UID not found. Please try again.');
          navigation.goBack();
          return;
        }
        setAdminUid(finalAdminUid);

        // Get work order ID from route params
        const woId = route.params?.workOrderId;
        console.log('📊 Work Order ID:', woId);
        
        if (!woId) {
          console.log('❌ No work order ID provided');
          showToast('error', 'Error', 'Work order ID not provided.');
          navigation.goBack();
          return;
        }
        setWorkOrderId(woId);

        setUserDataLoading(false);

        // Fetch Work Order Details from MongoDB
        console.log('📊 Fetching work order from MongoDB:', woId);
        const response = await getJobCard(woId);
        
        console.log('📊 MongoDB Response:', JSON.stringify(response, null, 2));
        
        if (response.success && response.data) {
          const data = response.data;
          console.log('✅ Work order data loaded from MongoDB:', data);
          console.log('📊 Work order title:', data.title);
          console.log('📊 Work order status:', data.status);
          console.log('📊 Work order items count:', data.items?.length || 0);
          console.log('📊 Work order items:', data.items);
          console.log('📊 Work order tracking history:', data.tracking?.history);
          
          // Map MongoDB data structure to component expected structure
          const mappedWorkOrder = {
            ...data,
            // Map equipment data
            equipment: data.equipment ? {
              equipmentName: data.equipment.equipmentName || data.equipment.name,
              mainCategory: data.equipment.mainCategory || data.equipment.category,
              model: data.equipment.model || data.equipment.modelNumber,
              vehicleNumber: data.equipment.vehicleNumber || data.equipment.registrationNumber,
              meterReading: data.equipment.meterReading || data.equipment.currentMeterReading,
              status: data.equipment.status || 'Active'
            } : null,
            // Map location data
            country: data.location?.country || data.country || 'N/A',
            project: data.location?.project || data.project || 'N/A',
            // Map other fields
            memo: data.description || data.memo || 'N/A',
            workOrderType: data.workOrderType || data.type || 'General',
            // Map request number (use title as request number)
            requestNumber: data.title || data.requestNumber || 'N/A',
            // Map dates
            startDate: data.startDate || data.createdAt,
            dueDate: data.dueDate || data.expectedCompletionDate,
            // Map assignTo
            assignTo: data.assignTo || data.assignedTo || [],
            // Map linked inspection data
            linkInspection: data.linkInspection || data.inspection ? {
              id: data.linkInspection?.id || data.inspection?.id,
              name: data.linkInspection?.name || data.inspection?.name || data.inspection?.title,
              reportNumber: data.linkInspection?.reportNumber || data.inspection?.reportNumber,
              status: data.linkInspection?.status || data.inspection?.status
            } : null,
            // Map linked work order data
            linkedWorkOrder: data.linkedWorkOrder || data.relatedWorkOrder ? {
              id: data.linkedWorkOrder?.id || data.relatedWorkOrder?.id,
              title: data.linkedWorkOrder?.title || data.relatedWorkOrder?.title,
              status: data.linkedWorkOrder?.status || data.relatedWorkOrder?.status
            } : null,
            // Map items (tasks) - prioritize actual items over tracking history
            items: data.items && data.items.length > 0 ? data.items.map((item, index) => {
              // Determine the main text/title for the item
              let mainText = '';
              if (item.category === 'Task') {
                mainText = item.text || item.description || `Task ${index + 1}`;
              } else if (item.category === 'Parts') {
                mainText = item.itemName || item.text || `Part ${index + 1}`;
              } else if (item.category === 'Labor') {
                mainText = item.worker || item.text || `Labor ${index + 1}`;
              } else if (item.category === 'Additional Cost') {
                mainText = item.description || item.text || `Additional Cost ${index + 1}`;
              } else if (item.category === 'Attach') {
                mainText = item.fileName || item.text || `Attachment ${index + 1}`;
              } else if (item.category === 'Remarks') {
                mainText = item.description || item.text || `Remark ${index + 1}`;
              } else {
                mainText = item.text || item.itemName || item.worker || item.description || `Item ${index + 1}`;
              }

              return {
                id: item.id || `item_${Date.now()}_${index}`,
                category: item.category || 'Task',
                text: mainText,
                type: item.type || 'Repair',
                status: item.status || 'Initiated',
                createdAt: item.createdAt || new Date().getTime(),
                createdBy: item.createdBy || 'System',
                note: item.note || '',
                assignTo: item.assignTo || [],
                // Include additional fields for different item types
                itemNumber: item.itemNumber,
                partNumber: item.partNumber,
                itemName: item.itemName,
                quantity: item.quantity,
                unit: item.unit,
                itemCost: item.itemCost,
                site: item.site,
                worker: item.worker,
                laborCode: item.laborCode,
                rate: item.rate,
                hours: item.hours,
                minutes: item.minutes,
                description: item.description,
                amount: item.amount,
                fileName: item.fileName,
                fileUrl: item.fileUrl,
                userEmail: item.userEmail,
                userRole: item.userRole
              };
            }) : (data.tracking?.history ? data.tracking.history.map((item, index) => ({
              id: `tracking_${Date.now()}_${index}`,
              category: 'Task',
              text: item.notes || item.description || `Step: ${item.step}`,
              type: item.type || 'Repair',
              status: item.step === 'Initiated' ? 'Initiated' : 'Complete',
              createdAt: new Date(item.timestamp || item.createdAt).getTime(),
              createdBy: item.updatedBy || item.createdBy || 'System',
              note: item.note || '',
              assignTo: item.assignTo || []
            })) : []),
            // Map checklist items
            checklistItems: data.checklistItems || data.checklist || [],
            // Map additional fields
            priority: data.priority || 'Medium',
            estimatedCost: data.estimatedCost || data.cost || 'N/A',
            estimatedLaborTime: data.estimatedLaborTime || data.laborTime || 'N/A',
            pdfDownloadUrl: data.pdfDownloadUrl || data.pdfUrl || data.documentUrl
          };
          
          console.log('📊 Mapped work order:', mappedWorkOrder);
          console.log('📊 Mapped work order items:', mappedWorkOrder.items);
          
          setWorkOrder(mappedWorkOrder);
          
          // Process tasks more comprehensively
          const processedTasks = (mappedWorkOrder.items || []).map(task => {
            // Ensure all tasks have proper structure
            const processedTask = {
              ...task,
              // Ensure type is set for Task category
              type: task.category === 'Task' ? (task.type || 'Repair') : task.type,
              // Ensure status is valid
              status: task.status || 'Initiated',
              // Ensure category is set
              category: task.category || 'Task',
              // Ensure text/title is set
              text: task.text || task.itemName || task.worker || task.description || `Item ${Date.now()}`,
              // Ensure assignTo is array
              assignTo: Array.isArray(task.assignTo) ? task.assignTo : [],
              // Ensure note is string
              note: task.note || '',
              // Ensure createdBy is set
              createdBy: task.createdBy || 'System',
              // Ensure createdAt is set
              createdAt: task.createdAt || new Date().getTime()
            };
            
            console.log('📊 Processing task:', {
              id: processedTask.id,
              category: processedTask.category,
              text: processedTask.text,
              status: processedTask.status,
              type: processedTask.type
            });
            
            return processedTask;
          });
          
          console.log('📊 Processed tasks for display:', processedTasks);
          console.log('📊 Total tasks processed:', processedTasks.length);
          setTasks(processedTasks);
          setChecklistItems(mappedWorkOrder.checklistItems || []);
          setSelectedStatus(mappedWorkOrder.status || 'Initiated');
          
          // Load timer state for this specific work order
          AsyncStorage.getItem(`timerElapsed_${woId}`).then(savedTime => {
            if (savedTime) {
              setElapsedTime(parseInt(savedTime, 10));
            }
          });
          
          console.log('✅ Data setup complete - workOrder set:', !!data);
        } else {
          console.log('❌ Work order not found in MongoDB');
          console.log('❌ Response success:', response.success);
          console.log('❌ Response data:', response.data);
          showToast('error', 'Error', 'Work order not found.');
          navigation.goBack();
          return;
        }

        setIsLoading(false);
        setIsLoadingWorkOrder(false);
        console.log('✅ Loading complete');
      } catch (error) {
        console.error('❌ Error fetching data:', error);
        showToast('error', 'Error', 'Failed to load data.');
        setIsLoading(false);
        setUserDataLoading(false);
        setIsLoadingWorkOrder(false);
      }
    };

    fetchUserAndWorkOrderData();

    // Cleanup timer on unmount
    return () => {
      if (timerRef.current) {
        clearInterval(timerRef.current);
      }
    };
  }, [route.params?.workOrderId, route.params?.adminUid, navigation]);

  // Add Item Handler - MongoDB
  const handleAddItem = async () => {
    if (!newItemTitle.trim()) {
      showToast('error', 'Error', 'Item title is required.');
      return;
    }

    setIsAddingItem(true);
    try {
      if (!adminUid || !workOrderId) {
        showToast('error', 'Error', 'Missing required data. Please try again.');
        return;
      }

      let newItem;
      switch (newItemCategory) {
        case 'Task':
          newItem = {
            id: Date.now().toString(),
            category: 'Task',
            text: newItemTitle,
            type: newItemType, // Use selected type
            status: 'Initiated',
            note: '',
            assignTo: [],
            createdAt: Date.now(),
            createdBy: currentUser?.email || 'Unknown',
          };
          break;
        case 'Parts':
          newItem = {
            id: Date.now().toString(),
            category: 'Parts',
            itemName: newItemTitle,
            quantity: '1',
            unit: 'Piece',
            itemCost: '0',
            status: 'Initiated',
            createdAt: Date.now(),
            createdBy: currentUser?.email || 'Unknown',
          };
          break;
        case 'Labor':
          newItem = {
            id: Date.now().toString(),
            category: 'Labor',
            worker: newItemTitle,
            hours: '0',
            minutes: '0',
            rate: '0',
            status: 'Initiated',
            createdAt: Date.now(),
            createdBy: currentUser?.email || 'Unknown',
          };
          break;
        case 'Additional Cost':
          newItem = {
            id: Date.now().toString(),
            category: 'Additional Cost',
            description: newItemTitle,
            amount: '0',
            status: 'Initiated',
            createdAt: Date.now(),
            createdBy: currentUser?.email || 'Unknown',
          };
          break;
        default:
          showToast('error', 'Error', 'Invalid category selected.');
          return;
      }

      const updatedTasks = [...tasks, newItem];
      
      // Update work order in MongoDB
      console.log('📊 Updating work order in MongoDB with new item');
      const response = await updateJobCard(workOrderId, { items: updatedTasks });
      
      if (response.success) {
        setTasks(updatedTasks);
        setNewItemTitle('');
        setNewItemCategory('Task');
        setNewItemType('Repair'); // Reset type
        setAddItemModal(false);
        showToast('success', 'Success', `${newItemCategory} added successfully.`);
      } else {
        showToast('error', 'Error', `Failed to add ${newItemCategory.toLowerCase()}.`);
      }
    } catch (error) {
      console.error('Error adding item:', error);
      showToast('error', 'Error', `Failed to add ${newItemCategory.toLowerCase()}.`);
    } finally {
      setIsAddingItem(false);
    }
  };

  // Update Task Status - MongoDB
  const handleUpdateStatus = async (itemId, newStatus) => {
    try {
      if (!adminUid || !workOrderId) {
        showToast('error', 'Error', 'Missing required data.');
        return;
      }

      setIsUpdatingStatus(true);
      const updatedTasks = tasks.map((task) =>
        task.id === itemId ? { ...task, status: newStatus, updatedAt: Date.now() } : task
      );

      // Update work order in MongoDB
      console.log('📊 Updating task status in MongoDB');
      const response = await updateJobCard(workOrderId, { items: updatedTasks });

      if (response.success) {
        setTasks(updatedTasks);
        showToast('success', 'Success', 'Status updated successfully.');
      } else {
        showToast('error', 'Error', 'Failed to update status.');
      }
    } catch (error) {
      console.error('Error updating status:', error);
      showToast('error', 'Error', 'Failed to update status.');
    } finally {
      setIsUpdatingStatus(false);
    }
  };

  // Add Remark Handler - MongoDB
  const handleAddRemark = async () => {
    if (!newItemTitle.trim()) {
      showToast('error', 'Error', 'Remark cannot be empty.');
      return;
    }

    setIsAddingRemark(true);
    try {
      if (!adminUid || !workOrderId || !currentUser) {
        showToast('error', 'Error', 'Missing required data. Please try again.');
        return;
      }

      const newRemark = {
        id: Date.now().toString(),
        category: 'Remarks',
        description: newItemTitle,
        userEmail: currentUser.email,
        userRole: userRole || 'Worker',
        createdAt: Date.now(),
      };

      const updatedTasks = [...tasks, newRemark];
      
      // Update work order in MongoDB
      console.log('📊 Adding remark to work order in MongoDB');
      const response = await updateJobCard(workOrderId, { items: updatedTasks });

      if (response.success) {
        setTasks(updatedTasks);
        setNewItemTitle('');
        showToast('success', 'Success', 'Remark added successfully.');
      } else {
        showToast('error', 'Error', 'Failed to add remark.');
      }
    } catch (error) {
      console.error('Error adding remark:', error);
      showToast('error', 'Error', 'Failed to add remark.');
    } finally {
      setIsAddingRemark(false);
    }
  };

  // Update Work Order Status - MongoDB
  const handleUpdateWorkOrderStatus = async (newStatus) => {
    try {
      if (!adminUid || !workOrderId) {
        showToast('error', 'Error', 'Missing required data.');
        return;
      }

      setIsUpdatingWorkOrderStatus(true);
      
      // Update work order status in MongoDB
      console.log('📊 Updating work order status in MongoDB');
      const response = await updateJobCard(workOrderId, { 
        status: newStatus, 
        updatedAt: Date.now(),
        updatedBy: currentUser?.email || 'Unknown',
        adminUid: adminUid
      });

      if (response.success) {
        setSelectedStatus(newStatus);
        setWorkOrder(prev => ({ ...prev, status: newStatus }));
        
        // Update service schedule if this job card is linked to one
        if (workOrder?.serviceScheduleId) {
          console.log('📋 Updating service schedule job card status...');
          console.log('📋 Service schedule ID:', workOrder.serviceScheduleId);
          console.log('📋 New job card status:', newStatus);
          
          try {
            const MONGODB_SERVICE_SCHEDULE_API_URL = 'http://168.231.113.219:5010';
            const scheduleResponse = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/${workOrder.serviceScheduleId}`, {
              method: 'PUT',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify({
                jobCardStatus: newStatus,
                updatedAt: new Date().toISOString()
              }),
            });
            
            if (scheduleResponse.ok) {
              const scheduleResult = await scheduleResponse.json();
              console.log('✅ Service schedule updated with new job card status');
              console.log('✅ Updated service schedule:', scheduleResult.data);
            } else {
              console.log('⚠️ Failed to update service schedule status:', scheduleResponse.status);
            }
          } catch (scheduleError) {
            console.error('⚠️ Error updating service schedule:', scheduleError);
            // Don't fail the main status update if service schedule update fails
          }
        }
        
        showToast('success', 'Success', 'Work order status updated.');
      } else {
        showToast('error', 'Error', 'Failed to update status.');
      }
    } catch (error) {
      console.error('Error updating work order status:', error);
      showToast('error', 'Error', 'Failed to update status.');
    } finally {
      setIsUpdatingWorkOrderStatus(false);
    }
  };

  // Render Checklist Item
  const renderChecklistItem = ({ item: subItem, sectionName }) => (
    <View style={styles.checklistItem}>
      <View style={styles.checklistContent}>
        <Text style={styles.checklistText}>{subItem.text}</Text>
        <Text style={styles.checklistSection}>{sectionName}</Text>
        <Text
          style={[
            styles.checklistStatus,
            {
              color: subItem.status === 'Reject' ? '#e74c3c' : '#FBC02D',
            },
          ]}
        >
          {subItem.status}
        </Text>
      </View>
      <TouchableOpacity onPress={() => showToast('info', 'Info', 'Checklist item is view-only.')}>
        <Icon name="information-outline" size={24} color="#3481BC" />
      </TouchableOpacity>
    </View>
  );

  // --- Edit Task Modal State ---
  const [editTaskModalVisible, setEditTaskModalVisible] = useState(false);
  const [taskBeingEdited, setTaskBeingEdited] = useState(null);
  const [editTaskText, setEditTaskText] = useState('');
  const [editTaskType, setEditTaskType] = useState('Repair'); // Default to Repair
  const [editTaskNote, setEditTaskNote] = useState('');
  const [editTaskAssignTo, setEditTaskAssignTo] = useState([]);

  // --- Open Edit Modal ---
  const openEditTaskModal = (task) => {
    setTaskBeingEdited(task);
    setEditTaskText(task.text || '');
    setEditTaskType(task.type || 'Repair'); // Ensure default is Repair if not set
    setEditTaskNote(task.note || '');
    setEditTaskAssignTo(task.assignTo || []);
    setEditTaskModalVisible(true);
    setEditAssignToModalVisible(false); // Ensure only Edit Task modal is open
  };

  // --- Save Edited Task - MongoDB ---
  const handleSaveEditedTask = async () => {
    if (!editTaskText.trim()) {
      showToast('error', 'Error', 'Task description is required.');
      return;
    }
    if (!adminUid || !workOrderId) {
      showToast('error', 'Error', 'Missing required data.');
      return;
    }
    const updatedTask = {
      ...taskBeingEdited,
      text: editTaskText,
      type: editTaskType, // This will be either 'Repair' or 'Replace'
      note: editTaskNote,
      assignTo: editTaskAssignTo,
      updatedAt: Date.now(),
    };
    const updatedTasks = tasks.map(t => t.id === updatedTask.id ? updatedTask : t);
    
    // Sync to MongoDB
    try {
      console.log('📊 Updating edited task in MongoDB');
      const response = await updateJobCard(workOrderId, { items: updatedTasks });
      
      if (response.success) {
        setTasks(updatedTasks);
        setEditTaskModalVisible(false);
        setTaskBeingEdited(null);
        setEditTaskText('');
        setEditTaskType('Repair'); // Reset to default
        setEditTaskNote('');
        setEditTaskAssignTo([]);
        showToast('success', 'Success', 'Task updated successfully.');
      } else {
        showToast('error', 'Error', 'Failed to update task in database.');
      }
    } catch (error) {
      console.error('Error updating task:', error);
      showToast('error', 'Error', 'Failed to update task in database.');
    }
  };

  // --- Edit Assign To Modal for Edit Task ---
  const [editAssignToModalVisible, setEditAssignToModalVisible] = useState(false);
  const [editAssignToSearchQuery, setEditAssignToSearchQuery] = useState('');
  const [editAssignToTemp, setEditAssignToTemp] = useState([]);

  const openEditAssignToModal = () => {
    setEditAssignToTemp(editTaskAssignTo);
    fetchTeamMembers();
    setEditAssignToModalVisible(true);
  };
  const handleEditAssignToSelect = (item) => {
    setEditAssignToTemp(prev => prev.some(selected => selected.id === item.id)
      ? prev.filter(selected => selected.id !== item.id)
      : [...prev, item]);
  };
  const handleSaveEditAssignTo = () => {
    setEditTaskAssignTo(editAssignToTemp);
    setEditAssignToModalVisible(false);
    setEditAssignToSearchQuery('');
  };

  // --- Render Edit Assign To Modal ---
  const renderEditAssignToModal = () => (
    <Modal
      visible={editAssignToModalVisible}
      transparent={true}
      animationType="slide"
      onRequestClose={() => setEditAssignToModalVisible(false)}
    >
      <View style={styles.modalOverlay}>
        <View style={[styles.modalContent, { maxHeight: '80%' }]}> 
          <Text style={styles.modalTitle}>Edit Assign To</Text>
          <TextInput
            style={styles.searchInput}
            placeholder="Search team members..."
            placeholderTextColor="#888"
            value={editAssignToSearchQuery}
            onChangeText={setEditAssignToSearchQuery}
          />
          <FlatList
            data={teamMembers.filter(member =>
              member.fullName?.toLowerCase().includes(editAssignToSearchQuery.toLowerCase()) ||
              member.employeeNumber?.toLowerCase().includes(editAssignToSearchQuery.toLowerCase()) ||
              member.role?.toLowerCase().includes(editAssignToSearchQuery.toLowerCase())
            )}
            keyExtractor={item => item.id}
            renderItem={({ item }) => (
              <TouchableOpacity
                style={{
                  flexDirection: 'row',
                  alignItems: 'center',
                  paddingVertical: 10,
                  borderBottomWidth: 1,
                  borderBottomColor: '#eee',
                  backgroundColor: editAssignToTemp.some(selected => selected.id === item.id) ? '#e6f0ff' : '#fff'
                }}
                onPress={() => handleEditAssignToSelect(item)}
              >
                {item.profilePic ? (
                  <Image source={{ uri: item.profilePic.url }} style={{ width: 40, height: 40, borderRadius: 20, marginRight: 10 }} />
                ) : (
                  <View style={{ width: 40, height: 40, borderRadius: 20, backgroundColor: '#ddd', marginRight: 10, alignItems: 'center', justifyContent: 'center' }}>
                    <Icon name="person" size={24} color="#888" />
                  </View>
                )}
                <View style={{ flex: 1 }}>
                  <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 16 }}>{item.fullName}</Text>
                  <Text style={{ color: '#333', fontSize: 13 }}>Emp#: {item.employeeNumber} | Role: {item.role}</Text>
                  <Text style={{ color: '#333', fontSize: 12 }}>Projects: {item.projects?.join(', ') || 'None'}</Text>
                  <Text style={{ color: '#333', fontSize: 12 }}>Country: {item.countries?.join(', ') || 'None'}</Text>
                  <Text style={{ color: item.isBlocked ? '#d32f2f' : '#388e3c', fontSize: 12 }}>{item.isBlocked ? 'Blocked' : 'Active'}</Text>
                </View>
                {editAssignToTemp.some(selected => selected.id === item.id) && (
                  <Icon name="checkmark" size={24} color="#015185" style={{ marginLeft: 8 }} />
                )}
              </TouchableOpacity>
            )}
            style={{ marginBottom: 10 }}
          />
          <TouchableOpacity
            style={styles.modalCloseButton}
            onPress={handleSaveEditAssignTo}
          >
            <Text style={styles.modalCloseText}>Save</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  // --- Render Edit Task Modal ---
  const renderEditTaskModal = () => (
    <Modal
      visible={editTaskModalVisible}
      transparent
      animationType="slide"
      onRequestClose={() => setEditTaskModalVisible(false)}
    >
      <View style={styles.modalOverlay}>
        <View style={styles.addTaskModalContent}>
          <Text style={styles.addTaskModalTitle}>Edit Task</Text>
          {/* Task Row */}
          <View style={styles.addTaskRowAligned}>
            <Text style={styles.addTaskLabel}>Task</Text>
            <TextInput
              style={[styles.editTaskInputUnderline]}
              value={editTaskText}
              onChangeText={setEditTaskText}
              placeholderTextColor="#888"
            />
          </View>
          {/* Type Row */}
          <View style={styles.addTaskRowAligned}>
            <Text style={styles.addTaskLabel}>Type</Text>
            <View style={styles.addTaskTypeRowInline}>
              {['Repair', 'Replace'].map(type => (
                <TouchableOpacity
                  key={type}
                  style={[
                    styles.addTaskTypeBtn,
                    editTaskType === type && styles.addTaskTypeBtnActive,
                  ]}
                  onPress={() => setEditTaskType(type)}
                >
                  <Text style={[
                    styles.addTaskTypeBtnText,
                    editTaskType === type && styles.addTaskTypeBtnTextActive,
                  ]}>{type}</Text>
                </TouchableOpacity>
              ))}
            </View>
          </View>
          {/* Assign to Row */}
          <TouchableOpacity 
            style={styles.addTaskRow} 
            onPress={openEditAssignToModal}
          >
            <Text style={styles.addTaskLabel}>Assign to</Text>
            <View style={styles.addTaskRowRight}>
              <Text style={styles.addTaskRowValue}>
                {editTaskAssignTo.length > 0 ? `${editTaskAssignTo.length} Assigned` : 'Set'}
              </Text>
              <Icon name="chevron-forward" size={20} color="#015185" />
            </View>
          </TouchableOpacity>
          {/* Show selected users */}
          {editTaskAssignTo.length > 0 && (
            <View style={styles.selectedUsersContainer}>
              {editTaskAssignTo.map(user => (
                <View key={user.id} style={styles.selectedUserChip}>
                  <Text style={styles.selectedUserChipText}>{user.fullName}</Text>
                </View>
              ))}
            </View>
          )}
          {/* Note Row */}
          <View style={styles.addTaskRowAligned}>
            <Text style={styles.addTaskLabel}>Note</Text>
            <TextInput
              style={[styles.editTaskInputUnderline]}
              value={editTaskNote}
              onChangeText={setEditTaskNote}
              placeholderTextColor="#888"
            />
          </View>
          {/* Fault List Row (disabled for now) */}
          <TouchableOpacity style={styles.addTaskRow} disabled>
            <Text style={styles.addTaskLabel}>Fault List</Text>
            <View style={styles.addTaskRowRight}>
              <Text style={styles.addTaskRowValue}>Select</Text>
              <Icon name="chevron-forward" size={20} color="#015185" />
            </View>
          </TouchableOpacity>
                      <View style={styles.addTaskModalBtnRow}>
              <TouchableOpacity
                style={styles.addTaskSaveBtn}
                onPress={() => {
                  setEditTaskModalVisible(false);
                  // Reset form when canceling
                  setTaskBeingEdited(null);
                  setEditTaskText('');
                  setEditTaskType('Repair');
                  setEditTaskNote('');
                  setEditTaskAssignTo([]);
                }}
              >
                <Text style={styles.addTaskSaveBtnText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.addTaskSaveBtn}
                onPress={handleSaveEditedTask}
              >
                <Text style={styles.addTaskSaveBtnText}>Save</Text>
              </TouchableOpacity>
            </View>
        </View>
        {renderEditAssignToModal()}
      </View>
    </Modal>
  );

  // --- 1. Add state for Assign To modal and user selection ---
  const [assignToModalVisible, setAssignToModalVisible] = useState(false);
  const [teamMembers, setTeamMembers] = useState([]);
  const [assignToSearchQuery, setAssignToSearchQuery] = useState('');
  const [tempAssignTo, setTempAssignTo] = useState([]); // For current task
  const [selectedTaskId, setSelectedTaskId] = useState(null); // For editing assignment
  const [isLoadingTeamMembers, setIsLoadingTeamMembers] = useState(false);

  // --- 2. Fetch users from MongoDB (same as UsersAssignToJC.js) ---
  const fetchTeamMembers = async () => {
    setIsLoadingTeamMembers(true);
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        console.log('No admin UID found');
        setTeamMembers([]);
        return;
      }
      
      console.log('📊 Fetching team members from MongoDB for admin:', adminUid);
      
      // Import MongoAPI dynamically to avoid circular imports
      const { MongoAPI } = require('../../../../ConfigMongo/MongoDBConfig');
      
      // Fetch team members from MongoDB using fast endpoint
      const teamResponse = await MongoAPI.getUsersFast(1, 8000);
      console.log('📊 MongoDB Team Response:', {
        success: teamResponse.success,
        dataLength: teamResponse.data?.length,
        firstUser: teamResponse.data?.[0] ? {
          id: teamResponse.data[0]._id,
          name: teamResponse.data[0].name,
          employeeNumber: teamResponse.data[0].employeeNumber
        } : 'No users'
      });
      
      if (teamResponse && teamResponse.success && teamResponse.data && teamResponse.data.length > 0) {
        const membersArray = teamResponse.data.map((user) => ({
          id: user._id || user.id || `user_${Math.random()}`,
          fullName: user.name || user.fullName || 'N/A',
          employeeNumber: user.employeeNumber || 'N/A',
          email: user.email || user.employeeNumber || 'N/A',
          projects: user.projects || [],
          countries: user.countries || [],
          role: user.role || 'Operator',
          isBlocked: user.isBlocked || false,
          password: user.password || user.employeeNumber || user.email || 'N/A',
          profilePic: user.profilePic || null,
        }));
        
        // Filter out current admin
        const filteredMembers = membersArray.filter(member => member.id !== adminUid);
        
        console.log('✅ Fetched team members from MongoDB:', filteredMembers.length);
        setTeamMembers(filteredMembers);
        
      } else {
        console.log('❌ Failed to load team members from MongoDB');
        setTeamMembers([]);
      }
      
    } catch (error) {
      console.error('❌ Error fetching team members:', error);
      showToast('error', 'Error', 'Failed to fetch team members from database');
      setTeamMembers([]);
    } finally {
      setIsLoadingTeamMembers(false);
    }
  };

  // --- 3. Handle Assign To selection ---
  const handleAssignToSelect = (item) => {
    setTempAssignTo(prev => prev.some(selected => selected.id === item.id)
      ? prev.filter(selected => selected.id !== item.id)
      : [...prev, item]);
  };

  // --- Open Assign To Modal for existing task ---
  const openAssignToModalForTask = (task) => {
    setSelectedTaskId(task.id);
    setTempAssignTo(task.assignTo || []);
    fetchTeamMembers();
    setAssignToModalVisible(true);
  };

  // --- 4. Save Assign To selection - MongoDB ---
  const handleSaveAssignTo = async () => {
    // Update the tempAssignTo to the actual task being edited
    if (selectedTaskId) {
      // This is for editing an existing task
      const updatedTasks = tasks.map(task => 
        task.id === selectedTaskId 
          ? { ...task, assignTo: tempAssignTo }
          : task
      );
      
      // Save to MongoDB
      if (adminUid && workOrderId) {
        try {
          console.log('📊 Updating task assignment in MongoDB');
          const response = await updateJobCard(workOrderId, { items: updatedTasks });
          
          if (response.success) {
            setTasks(updatedTasks);
            showToast('success', 'Success', 'Assignment updated successfully.');
          } else {
            showToast('error', 'Error', 'Failed to update assignment.');
          }
        } catch (error) {
          console.error('Error updating assignment:', error);
          showToast('error', 'Error', 'Failed to update assignment.');
        }
      }
    }
    
    setAssignToModalVisible(false);
    setAssignToSearchQuery('');
    setSelectedTaskId(null);
  };

  // --- 5. Assign To Modal UI ---
  const renderAssignToModal = () => (
    <Modal
      visible={assignToModalVisible}
      transparent={true}
      animationType="slide"
      onRequestClose={() => setAssignToModalVisible(false)}
    >
      <View style={styles.modalOverlay}>
        <View style={[styles.modalContent, { maxHeight: '80%' }]}> 
          <Text style={styles.modalTitle}>Assign To</Text>
          <TextInput
            style={styles.searchInput}
            placeholder="Search team members..."
            placeholderTextColor="#888"
            value={assignToSearchQuery}
            onChangeText={setAssignToSearchQuery}
          />
          <FlatList
            data={teamMembers.filter(member =>
              member.fullName?.toLowerCase().includes(assignToSearchQuery.toLowerCase()) ||
              member.employeeNumber?.toLowerCase().includes(assignToSearchQuery.toLowerCase()) ||
              member.role?.toLowerCase().includes(assignToSearchQuery.toLowerCase())
            )}
            keyExtractor={item => item.id}
            renderItem={({ item }) => (
              <TouchableOpacity
                style={{
                  flexDirection: 'row',
                  alignItems: 'center',
                  paddingVertical: 10,
                  borderBottomWidth: 1,
                  borderBottomColor: '#eee',
                  backgroundColor: tempAssignTo.some(selected => selected.id === item.id) ? '#e6f0ff' : '#fff'
                }}
                onPress={() => handleAssignToSelect(item)}
              >
                {item.profilePic ? (
                  <Image source={{ uri: item.profilePic.url }} style={{ width: 40, height: 40, borderRadius: 20, marginRight: 10 }} />
                ) : (
                  <View style={{ width: 40, height: 40, borderRadius: 20, backgroundColor: '#ddd', marginRight: 10, alignItems: 'center', justifyContent: 'center' }}>
                    <Icon name="person" size={24} color="#888" />
                  </View>
                )}
                <View style={{ flex: 1 }}>
                  <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 16 }}>{item.fullName}</Text>
                  <Text style={{ color: '#333', fontSize: 13 }}>Emp#: {item.employeeNumber} | Role: {item.role}</Text>
                  <Text style={{ color: '#333', fontSize: 12 }}>Projects: {item.projects?.join(', ') || 'None'}</Text>
                  <Text style={{ color: '#333', fontSize: 12 }}>Country: {item.countries?.join(', ') || 'None'}</Text>
                  <Text style={{ color: item.isBlocked ? '#d32f2f' : '#388e3c', fontSize: 12 }}>{item.isBlocked ? 'Blocked' : 'Active'}</Text>
                </View>
                {tempAssignTo.some(selected => selected.id === item.id) && (
                  <Icon name="checkmark" size={24} color="#015185" style={{ marginLeft: 8 }} />
                )}
              </TouchableOpacity>
            )}
            style={{ marginBottom: 10 }}
          />
          <TouchableOpacity
            style={styles.modalCloseButton}
            onPress={handleSaveAssignTo}
          >
            <Text style={styles.modalCloseText}>Save</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  // --- 6. Update handleSaveNewTask to include assigned users - MongoDB ---
  const handleSaveNewTask = async () => {
    if (!newTaskText.trim()) {
      showToast('error', 'Error', 'Task description is required.');
      return;
    }
    if (!adminUid || !workOrderId) {
      showToast('error', 'Error', 'Missing required data.');
      return;
    }
    const newTask = {
      id: Date.now().toString(),
      category: 'Task',
      text: newTaskText,
      type: newTaskType, // This will be either 'Repair' or 'Replace'
      note: newTaskNote,
      assignTo: tempAssignTo,
      status: 'Initiated',
      createdAt: Date.now(),
      createdBy: currentUser?.email || 'Unknown',
    };
    const updatedTasks = [...tasks, newTask];
    
    try {
      console.log('📊 Adding new task to MongoDB');
      const response = await updateJobCard(workOrderId, { items: updatedTasks });
      
      if (response.success) {
        setTasks(updatedTasks);
        setAddTaskModalVisible(false);
        setNewTaskText('');
        setNewTaskType('Repair'); // Reset to default
        setNewTaskNote('');
        setTempAssignTo([]);
        showToast('success', 'Success', 'Task added successfully.');
      } else {
        showToast('error', 'Error', 'Failed to add task in database.');
      }
    } catch (error) {
      console.error('Error adding task:', error);
      showToast('error', 'Error', 'Failed to add task in database.');
    }
  };



  // Add state for each add modal
  const [addTaskModalVisible, setAddTaskModalVisible] = useState(false);
  const [addPartsModalVisible, setAddPartsModalVisible] = useState(false);
  const [addLaborModalVisible, setAddLaborModalVisible] = useState(false);
  const [addAdditionalCostModalVisible, setAddAdditionalCostModalVisible] = useState(false);
  const [addAttachModalVisible, setAddAttachModalVisible] = useState(false);

  // Add state for new task modal
  const [newTaskText, setNewTaskText] = useState('');
  const [newTaskType, setNewTaskType] = useState('Repair'); // Default to Repair
  const [newTaskNote, setNewTaskNote] = useState('');

  // New state for Parts modal
  const [partsItemNumber, setPartsItemNumber] = useState('');
  const [partsPartNumber, setPartsPartNumber] = useState('');
  const [partsItemName, setPartsItemName] = useState('');
  const [partsQuantity, setPartsQuantity] = useState('');
  const [partsUnit, setPartsUnit] = useState('');
  const [partsItemCost, setPartsItemCost] = useState('');
  const [partsSite, setPartsSite] = useState('');
  const [partsNote, setPartsNote] = useState('');

  // New state for Labor modal
  const [laborWorker, setLaborWorker] = useState('');
  const [laborCode, setLaborCode] = useState('');
  const [laborRate, setLaborRate] = useState('');
  const [laborHours, setLaborHours] = useState('');
  const [laborMinutes, setLaborMinutes] = useState('');
  const [laborNote, setLaborNote] = useState('');

  // New state for Additional Cost modal
  const [additionalCostDescription, setAdditionalCostDescription] = useState('');
  const [additionalCostAmount, setAdditionalCostAmount] = useState('');

  // New state for Attach modal
  const [attachFileName, setAttachFileName] = useState('');

  const handleSaveParts = async () => {
    if (
      !partsItemNumber.trim() ||
      !partsPartNumber.trim() ||
      !partsItemName.trim() ||
      !partsQuantity.trim() ||
      !partsUnit.trim() ||
      !partsItemCost.trim()
    ) {
      showToast('error', 'Error', 'All fields except Site and Note are required for Parts.');
      return;
    }
    if (!adminUid || !workOrderId) {
      showToast('error', 'Error', 'Missing required data.');
      return;
    }
    const newPart = {
      id: Date.now().toString(),
      category: 'Parts',
      itemNumber: partsItemNumber,
      partNumber: partsPartNumber,
      itemName: partsItemName,
      quantity: partsQuantity,
      unit: partsUnit,
      itemCost: partsItemCost,
      site: partsSite,
      note: partsNote,
      status: 'Initiated',
      createdAt: Date.now(),
      createdBy: currentUser?.email || 'Unknown',
    };
    const updatedTasks = [...tasks, newPart];
    setTasks(updatedTasks);
    setAddPartsModalVisible(false);
    setPartsItemNumber('');
    setPartsPartNumber('');
    setPartsItemName('');
    setPartsQuantity('');
    setPartsUnit('');
    setPartsItemCost('');
    setPartsSite('');
    setPartsNote('');
    try {
      console.log('📊 Adding parts to MongoDB');
      const response = await updateJobCard(workOrderId, { items: updatedTasks });
      
      if (response.success) {
        showToast('success', 'Success', 'Parts added successfully.');
      } else {
        showToast('error', 'Error', 'Failed to add parts in database.');
      }
    } catch (error) {
      console.error('Error adding parts:', error);
      showToast('error', 'Error', 'Failed to add parts in database.');
    }
  };

  const handleSaveLabor = async () => {
    if (!laborWorker.trim() || !laborRate.trim() || !laborHours.trim() || !laborMinutes.trim()) {
      showToast('error', 'Error', 'All fields are required for Labor.');
      return;
    }
    if (!adminUid || !workOrderId) {
      showToast('error', 'Error', 'Missing required data.');
      return;
    }
    const newLabor = {
      id: Date.now().toString(),
      category: 'Labor',
      worker: laborWorker,
      laborCode: laborCode,
      rate: laborRate,
      hours: laborHours,
      minutes: laborMinutes,
      note: laborNote,
      status: 'Initiated',
      createdAt: Date.now(),
      createdBy: currentUser?.email || 'Unknown',
    };
    const updatedTasks = [...tasks, newLabor];
    setTasks(updatedTasks);
    setAddLaborModalVisible(false);
    setLaborWorker('');
    setLaborCode('');
    setLaborRate('');
    setLaborHours('');
    setLaborMinutes('');
    setLaborNote('');
    try {
      console.log('📊 Adding labor to MongoDB');
      const response = await updateJobCard(workOrderId, { items: updatedTasks });
      
      if (response.success) {
        showToast('success', 'Success', 'Labor added successfully.');
      } else {
        showToast('error', 'Error', 'Failed to add labor in database.');
      }
    } catch (error) {
      console.error('Error adding labor:', error);
      showToast('error', 'Error', 'Failed to add labor in database.');
    }
  };

  const handleSaveAdditionalCost = async () => {
    if (!additionalCostDescription.trim() || !additionalCostAmount.trim()) {
      showToast('error', 'Error', 'Description and Amount are required for Additional Cost.');
      return;
    }
    if (!adminUid || !workOrderId) {
      showToast('error', 'Error', 'Missing required data.');
      return;
    }
    const newAdditionalCost = {
      id: Date.now().toString(),
      category: 'Additional Cost',
      description: additionalCostDescription,
      amount: additionalCostAmount,
      status: 'Initiated',
      createdAt: Date.now(),
      createdBy: currentUser?.email || 'Unknown',
    };
    const updatedTasks = [...tasks, newAdditionalCost];
    setTasks(updatedTasks);
    setAddAdditionalCostModalVisible(false);
    setAdditionalCostDescription('');
    setAdditionalCostAmount('');
    try {
      console.log('📊 Adding additional cost to MongoDB');
      const response = await updateJobCard(workOrderId, { items: updatedTasks });
      
      if (response.success) {
        showToast('success', 'Success', 'Additional Cost added successfully.');
      } else {
        showToast('error', 'Error', 'Failed to add additional cost in database.');
      }
    } catch (error) {
      console.error('Error adding additional cost:', error);
      showToast('error', 'Error', 'Failed to add additional cost in database.');
    }
  };

  const handlePickFile = () => {
    showToast('info', 'Info', 'File picker functionality will be implemented here.');
    // In a real app, you would use a library like react-native-document-picker
    // or a custom file picker. For now, we'll just show a toast.
  };

  const handleSaveAttach = async () => {
    if (!attachFileName) {
      showToast('error', 'Error', 'No file selected for attachment.');
      return;
    }
    const newAttach = {
      id: Date.now().toString(),
      category: 'Attach',
      fileName: attachFileName,
      fileUrl: 'placeholder_url', // Replace with actual file URL
      status: 'Initiated',
      createdAt: Date.now(),
      createdBy: currentUser?.email || 'Unknown',
    };
    setTasks([...tasks, newAttach]);
    setAddAttachModalVisible(false);
    setAttachFileName('');
    showToast('success', 'Success', 'Attachment added successfully.');
  };

  // Add state for dropdown menu
  const [dropdownTaskId, setDropdownTaskId] = useState(null);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0 });
  const [dropdownModalVisible, setDropdownModalVisible] = useState(false);
  const dropdownButtonRefs = useRef({});

  // Function to handle three dot press
  const handleThreeDotPress = (taskId) => {
    setDropdownTaskId(taskId);
    setDropdownModalVisible(true);
  };

  useEffect(() => {
    if (dropdownTaskId && dropdownButtonRefs.current[dropdownTaskId]) {
      dropdownButtonRefs.current[dropdownTaskId].measure((fx, fy, width, height, px, py) => {
        setDropdownPosition({ x: px, y: py + height });
      });
    }
  }, [dropdownTaskId]);

  // Function to handle delete - MongoDB
  const handleDeleteTask = async (taskId) => {
    const updatedTasks = tasks.filter(task => task.id !== taskId);
    setDropdownTaskId(null);
    try {
      if (adminUid && workOrderId) {
        console.log('📊 Deleting task from MongoDB');
        const response = await updateJobCard(workOrderId, { items: updatedTasks });
        
        if (response.success) {
          setTasks(updatedTasks);
          showToast('success', 'Deleted', 'Task deleted successfully.');
        } else {
          showToast('error', 'Error', 'Failed to delete task.');
        }
      }
    } catch (error) {
      console.error('Error deleting task:', error);
      showToast('error', 'Error', 'Failed to delete task.');
    }
  };

  // Add state for status dropdown
  const [statusDropdownTaskId, setStatusDropdownTaskId] = useState(null);
  const [statusDropdownPosition, setStatusDropdownPosition] = useState({ x: 0, y: 0 });
  const statusDropdownButtonRefs = useRef({});

  // Function to handle status chip press
  const handleStatusChipPress = (taskId) => {
    setStatusDropdownTaskId(taskId);
  };

  // Status options (reuse STATUS_OPTIONS)
  // ... existing code ...

  // Defensive: always use arrays for items and checklistItems
  const safeTasks = Array.isArray(tasks) ? tasks : [];
  const safeChecklistItems = Array.isArray(checklistItems) ? checklistItems : [];

  // Render Task Item
  const renderTask = ({ item }) => {
    console.log('🎨 Rendering task item:', {
      id: item.id,
      category: item.category,
      text: item.text,
      status: item.status,
      type: item.type
    });
    
    // Get the main title text based on category
    const getMainTitle = () => {
      if (item.category === 'Task') {
        return item.text || item.description || 'Task';
      } else if (item.category === 'Parts') {
        return item.itemName || item.text || 'Part';
      } else if (item.category === 'Labor') {
        return item.worker || item.text || 'Labor';
      } else if (item.category === 'Additional Cost') {
        return item.description || item.text || 'Additional Cost';
      } else if (item.category === 'Attach') {
        return item.fileName || item.text || 'Attachment';
      } else if (item.category === 'Remarks') {
        return item.description || item.text || 'Remark';
      } else {
        return item.text || item.itemName || item.worker || item.description || 'Item';
      }
    };
    
    return (
      <View style={styles.taskCard}>
        <View style={{ flex: 1 }}>
          <Text style={styles.title}>{getMainTitle()}</Text>
          <Text style={[styles.sectionText, { color: '#666', fontSize: 12 }]}>
            Category: {item.category || 'Task'} | ID: {item.id}
          </Text>
          
          {/* Task-specific details */}
          {item.category === 'Task' && (
            <>
              <Text style={styles.sectionText}>
                Type: {item.type || 'N/A'}
              </Text>
              {item.note && <Text style={styles.taskNote}>Note: {item.note}</Text>}
              {item.assignTo && item.assignTo.length > 0 && (
                <View style={styles.assignedUsersContainer}>
                  <Text style={styles.assignedUsersLabel}>Assigned to:</Text>
                  <View style={styles.assignedUsersList}>
                    {item.assignTo.map(user => (
                      <View key={user.id} style={styles.assignedUserChip}>
                        <Text style={styles.assignedUserChipText}>{user.fullName}</Text>
                      </View>
                    ))}
                  </View>
                </View>
              )}
            </>
          )}
          
          {/* Parts-specific details */}
          {item.category === 'Parts' && (
            <>
              {item.itemNumber && <Text style={styles.taskDetailText}>Item Number: {item.itemNumber}</Text>}
              {item.partNumber && <Text style={styles.taskDetailText}>Part Number: {item.partNumber}</Text>}
              {item.quantity && <Text style={styles.taskDetailText}>Quantity: {item.quantity} {item.unit || 'pcs'}</Text>}
              {item.itemCost && <Text style={styles.taskDetailText}>Cost: ${item.itemCost}</Text>}
              {item.site && <Text style={styles.taskDetailText}>Site: {item.site}</Text>}
              {item.note && <Text style={styles.taskNote}>Note: {item.note}</Text>}
            </>
          )}
          
          {/* Labor-specific details */}
          {item.category === 'Labor' && (
            <>
              {item.hours && <Text style={styles.taskDetailText}>Hours: {item.hours}h {item.minutes || 0}m</Text>}
              {item.rate && <Text style={styles.taskDetailText}>Rate: ${item.rate}</Text>}
              {item.laborCode && <Text style={styles.taskDetailText}>Labor Code: {item.laborCode}</Text>}
              {item.note && <Text style={styles.taskNote}>Note: {item.note}</Text>}
            </>
          )}
          
          {/* Additional Cost-specific details */}
          {item.category === 'Additional Cost' && (
            <>
              {item.amount && <Text style={styles.taskDetailText}>Amount: ${item.amount}</Text>}
            </>
          )}
          
          {/* Attachment-specific details */}
          {item.category === 'Attach' && (
            <>
              {item.fileUrl && <Text style={styles.taskDetailText}>URL: {item.fileUrl}</Text>}
            </>
          )}
          
          {/* Remarks-specific details */}
          {item.category === 'Remarks' && (
            <>
              {item.userEmail && <Text style={styles.taskDetailText}>By: {item.userEmail}</Text>}
              {item.userRole && <Text style={styles.taskDetailText}>Role: {item.userRole}</Text>}
            </>
          )}
          
          {/* Show created by and date for all items */}
          {item.createdBy && (
            <Text style={[styles.taskNote, { fontSize: 11, color: '#999' }]}>
              Created by: {item.createdBy} | {item.createdAt ? new Date(item.createdAt).toLocaleString() : 'Unknown date'}
            </Text>
          )}
        </View>
        <View style={styles.taskStatusRow}>
          {/* Status Chip */}
          <TouchableOpacity
            ref={ref => { if (ref) statusDropdownButtonRefs.current[item.id] = ref; }}
            style={{ marginRight: 8, paddingHorizontal: 10, paddingVertical: 4, borderRadius: 16, backgroundColor: '#e0e0e0', alignItems: 'center', justifyContent: 'center' }}
            onPress={() => handleStatusChipPress(item.id)}
          >
            <Text style={{ color: '#555', fontWeight: '600' }}>{item.status || 'Initiated'}</Text>
          </TouchableOpacity>
          {/* Three dot menu */}
          <TouchableOpacity
            ref={ref => { if (ref) dropdownButtonRefs.current[item.id] = ref; }}
            onPress={() => handleThreeDotPress(item.id)}
            style={{ padding: 6 }}
          >
            <Icon name="ellipsis-vertical" size={20} color="#3481BC" />
          </TouchableOpacity>
        </View>
      </View>
    );
  };

  // Status Dropdown Modal
  const renderStatusDropdownModal = () => {
    if (!statusDropdownTaskId) return null;
    const task = safeTasks.find(t => t.id === statusDropdownTaskId);
    if (!task) return null;
    return (
      <Modal
        visible={!!statusDropdownTaskId}
        transparent
        animationType="fade"
        onRequestClose={() => setStatusDropdownTaskId(null)}
      >
        <TouchableOpacity
          style={{ flex: 1 }}
          activeOpacity={1}
          onPress={() => setStatusDropdownTaskId(null)}
        >
          <View style={{ position: 'absolute', top: '40%', left: '30%', backgroundColor: '#fff', borderRadius: 10, elevation: 8, shadowColor: '#000', shadowOpacity: 0.2, shadowRadius: 8, zIndex: 200, minWidth: 180, paddingVertical: 8 }}>
            {STATUS_OPTIONS.map(option => (
              <TouchableOpacity
                key={option.value}
                style={{ flexDirection: 'row', alignItems: 'center', paddingVertical: 10, paddingHorizontal: 18, backgroundColor: task.status === option.value ? '#f0f8ff' : '#fff' }}
                onPress={async () => {
                  await handleUpdateStatus(task.id, option.value);
                  setStatusDropdownTaskId(null);
                }}
              >
                <Icon
                  name={task.status === option.value ? 'checkmark-circle' : 'ellipse-outline'}
                  size={20}
                  color={task.status === option.value ? option.textColor : '#bbb'}
                  style={{ marginRight: 10 }}
                />
                <Text style={{ color: option.textColor, fontWeight: task.status === option.value ? '700' : '500', fontSize: 16 }}>{option.label}</Text>
              </TouchableOpacity>
            ))}
          </View>
        </TouchableOpacity>
      </Modal>
    );
  };

  // Render Combined List for Task Tab
  const renderTaskTabContent = () => {
    console.log('🎨 Rendering task tab content for:', selectedTaskTab);
    console.log('🎨 Available tasks:', safeTasks.length);
    console.log('🎨 Task categories:', safeTasks.map(t => t.category));
    
    if (selectedTaskTab !== 'Task') {
      const filteredTasks = safeTasks.filter((task) => task.category === selectedTaskTab);
      console.log(`🎨 Filtered tasks for ${selectedTaskTab}:`, filteredTasks.length);
      
      return (
        <FlatList
          data={filteredTasks}
          renderItem={renderTask}
          keyExtractor={(item) => item.id}
          style={{ marginTop: 10 }}
          ListEmptyComponent={
            <View style={{ padding: 20, alignItems: 'center' }}>
              <Text style={styles.emptyText}>No {selectedTaskTab.toLowerCase()} items found.</Text>
              <Text style={[styles.emptyText, { fontSize: 12, marginTop: 5 }]}>
                Total tasks available: {safeTasks.length}
              </Text>
            </View>
          }
        />
      );
    }

    // Combine checklist items and tasks for Task tab
    const checklistData = safeChecklistItems.flatMap((section) =>
      Array.isArray(section.subItems)
        ? section.subItems.map((subItem) => ({ ...subItem, itemType: 'Checklist', sectionName: section.sectionName }))
        : []
    );
    const taskData = safeTasks.filter((task) => task.category === 'Task').map((item) => ({ ...item, itemType: 'Task' }));
    const combinedData = [
      ...checklistData,
      ...taskData,
    ];
    
    console.log('🎨 Combined data for Task tab:', {
      checklistItems: checklistData.length,
      tasks: taskData.length,
      total: combinedData.length
    });

    return (
      <FlatList
        data={combinedData}
        renderItem={({ item }) => 
          item.itemType === 'Checklist' 
            ? renderChecklistItem({ item, sectionName: item.sectionName })
            : renderTask({ item })
        }
        keyExtractor={(item) => item.id}
        style={{ marginTop: 10 }}
        ListEmptyComponent={
          <View style={{ padding: 20, alignItems: 'center' }}>
            <Text style={styles.emptyText}>No tasks or checklist items found.</Text>
            <Text style={[styles.emptyText, { fontSize: 12, marginTop: 5 }]}>
              Total items available: {safeTasks.length}
            </Text>
          </View>
        }
      />
    );
  };

  // Add before return statement
  const completedTasks = safeTasks.filter((task) => task.status === 'Complete').length;
  const totalTasks = safeTasks.length;

  // Dropdown Modal
  const renderDropdownModal = () => (
    <Modal
      visible={!!dropdownTaskId && dropdownModalVisible}
      transparent
      animationType="fade"
      onRequestClose={() => { setDropdownTaskId(null); setDropdownModalVisible(false); }}
    >
      <TouchableOpacity
        style={{ flex: 1 }}
        activeOpacity={1}
        onPress={() => { setDropdownTaskId(null); setDropdownModalVisible(false); }}
      >
        <View style={[
          { position: 'absolute', top: dropdownPosition.y || 200, left: dropdownPosition.x || 50, backgroundColor: '#fff', borderRadius: 8, elevation: 8, shadowColor: '#000', shadowOpacity: 0.2, shadowRadius: 8, zIndex: 100, minWidth: 120 }
        ]}>
          <TouchableOpacity
            style={{ padding: 12, minWidth: 80 }}
            onPress={() => {
              setDropdownTaskId(null);
              setDropdownModalVisible(false);
              const task = safeTasks.find(t => t.id === dropdownTaskId);
              if (task) openEditTaskModal(task);
            }}
          >
            <Text style={{ color: '#015185', fontWeight: '600' }}>Edit</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={{ padding: 12, minWidth: 80 }}
            onPress={() => {
              handleDeleteTask(dropdownTaskId);
              setDropdownTaskId(null);
              setDropdownModalVisible(false);
            }}
          >
            <Text style={{ color: '#e74c3c', fontWeight: '600' }}>Delete</Text>
          </TouchableOpacity>
        </View>
      </TouchableOpacity>
    </Modal>
  );

  if (isLoading || userDataLoading || isLoadingWorkOrder) {
    return <SyncModal visible={true} message="Loading Work Order..." />;
  }

  if (!workOrder) {
    console.log('❌ No workOrder data available');
    console.log('📊 Loading states:', { isLoading, userDataLoading, isLoadingWorkOrder });
    console.log('📊 Work Order ID from route:', route.params?.workOrderId);
    console.log('📊 Admin UID from route:', route.params?.adminUid);
    
    return (
      <View style={styles.container}>
        <View style={styles.header}>
          <View style={styles.headerRow}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <Icon name="arrow-back" size={28} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Worker Dashboard</Text>
            <View style={{ width: 28 }} />
          </View>
        </View>
        <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center' }}>
          <Text style={styles.emptyText}>Work order not found or failed to load.</Text>
          <Text style={styles.emptyText}>Work Order ID: {route.params?.workOrderId || 'Not provided'}</Text>
          <TouchableOpacity 
            style={styles.trackButton} 
            onPress={() => navigation.goBack()}
          >
            <Text style={styles.trackButtonText}>Go Back</Text>
          </TouchableOpacity>
        </View>
      </View>
    );
  }

  console.log('📊 Rendering WorkerDashboardhome with workOrder:', workOrder);
  console.log('📊 Work order title:', workOrder?.title);
  console.log('📊 Work order status:', workOrder?.status);
  console.log('📊 Tasks count:', tasks?.length || 0);

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient colors={['#3481BC', '#025383']} style={styles.header}>
        <View style={styles.headerRow}>
          <TouchableOpacity onPress={() => navigation.goBack()}>
            <Icon name="arrow-back" size={28} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Worker Dashboard</Text>
          <View style={{ width: 28 }} />
        </View>
        <View style={styles.headerInfoRow}>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <Icon
              name={
                workOrder.priority === 'Emergency'
                  ? 'alert-circle'
                  : workOrder.priority === 'High'
                  ? 'warning'
                  : 'information'
              }
              size={22}
              color={
                workOrder.priority === 'Emergency'
                  ? '#e74c3c'
                  : workOrder.priority === 'High'
                  ? '#f39c12'
                  : '#4CAF50'
              }
            />
            <View style={{ marginLeft: 8 }}>
              <Text style={styles.woNumber}>{workOrder.requestNumber || workOrder.title || 'WO1001'}</Text>
              <Text style={styles.woSub}>{workOrder._id || workOrder.id || 'N/A'}</Text>
            </View>
          </View>
          <TouchableOpacity style={styles.timerBtn} onPress={handleTimerToggle}>
            <Icon
              name={isTimerRunning ? 'pause' : 'play'}
              size={18}
              color="#fff"
            />
            <Text style={styles.timerBtnText}>
              {isTimerRunning ? formatTime(elapsedTime) : `Timer (${formatTime(elapsedTime)})`}
            </Text>
          </TouchableOpacity>
        </View>
        <View style={styles.topTabsRow}>
          {TOP_TABS.map((tab, idx) => (
            <TouchableOpacity
              key={tab}
              style={[styles.topTab, tabIndex === idx && styles.topTabActive]}
              onPress={() => setTabIndex(idx)}
            >
              <Text style={[styles.topTabText, tabIndex === idx && styles.topTabTextActive]}>
                {tab}
              </Text>
            </TouchableOpacity>
          ))}
        </View>
      </LinearGradient>

      {/* Main Content - make vertically scrollable */}
      <ScrollView style={styles.content} contentContainerStyle={{ padding: width * 0.03, paddingBottom: height * 0.12 }}>
        {tabIndex === 0 && (
          <View style={{ width: '100%' }}>
            {/* Description Tab Content */}
            <View style={styles.sectionCard}>
              <Text style={styles.label}>
                Title: <Text style={styles.value}>{workOrder.title || 'N/A'}</Text>
              </Text>
              <Text style={styles.label}>
                Memo: <Text style={styles.value}>{workOrder.memo || 'N/A'}</Text>
              </Text>
              <Text style={styles.label}>
                Country: <Text style={styles.value}>{workOrder.country || 'N/A'}</Text>
              </Text>
              <Text style={styles.label}>
                Project: <Text style={styles.value}>{workOrder.project || 'N/A'}</Text>
              </Text>
              <Text style={styles.label}>
                Equipment: <Text style={styles.value}>{workOrder.equipment?.equipmentName || workOrder.equipment?.name || 'N/A'}</Text>
              </Text>
              <Text style={styles.label}>
                Due Date:{' '}
                <Text style={styles.value}>
                  {workOrder.dueDate
                    ? new Date(workOrder.dueDate).toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'short',
                        day: 'numeric',
                      })
                    : 'N/A'}
                </Text>
              </Text>
              {/* Linked Inspection Information */}
              {workOrder.linkInspection && (
                <>
                  <Text style={styles.label}>
                    Linked Inspection: <Text style={[styles.value, { color: '#3481BC', textDecorationLine: 'underline' }]}>
                      {workOrder.linkInspection.name || workOrder.linkInspection.reportNumber || 'Inspection'}
                    </Text>
                  </Text>
                  {workOrder.linkInspection.status && (
                    <Text style={styles.label}>
                      Inspection Status: <Text style={[styles.value, {
                        color: workOrder.linkInspection.status === 'Complete' ? '#28a745' : 
                               workOrder.linkInspection.status === 'In Progress' ? '#3481BC' : '#f39c12'
                      }]}>
                        {workOrder.linkInspection.status}
                      </Text>
                    </Text>
                  )}
                </>
              )}
              {/* Linked Job Card Information */}
              {workOrder.linkedWorkOrder && (
                <>
                  <Text style={styles.label}>
                    Linked Job Card: <Text style={[styles.value, { color: '#3481BC', textDecorationLine: 'underline' }]}>
                      {workOrder.linkedWorkOrder.title || 'Related Job'}
                    </Text>
                  </Text>
                  {workOrder.linkedWorkOrder.status && (
                    <Text style={styles.label}>
                      Job Status: <Text style={[styles.value, {
                        color: workOrder.linkedWorkOrder.status === 'Complete' ? '#28a745' : 
                               workOrder.linkedWorkOrder.status === 'In Progress' ? '#3481BC' : '#f39c12'
                      }]}>
                        {workOrder.linkedWorkOrder.status}
                      </Text>
                    </Text>
                  )}
                </>
              )}
            </View>

            <View style={styles.sectionRow}>
              <Text style={styles.sectionLabel}>Additional Details</Text>
              <TouchableOpacity onPress={() => setDetailsModalVisible(true)}>
                <Text style={styles.sectionMore}>
                  More <Icon name="chevron-forward" size={16} />
                </Text>
              </TouchableOpacity>
            </View>

            {/* Horizontally scrollable status chips */}
            <ScrollView horizontal showsHorizontalScrollIndicator={false} style={styles.statusChipsRow}>
              {STATUS_OPTIONS.map((status, idx) => (
                <TouchableOpacity
                  key={status.value}
                  style={[
                    styles.statusChip,
                    idx === STATUS_OPTIONS.length - 1 && { marginRight: 0 },
                    selectedStatus === status.value && styles.statusChipActive,
                    status.value === 'cancelled' && styles.statusChipOnHold,
                    status.value === 'complete' &&
                      selectedStatus === status.value &&
                      styles.statusChipComplete,
                    status.value === 'cancelled' &&
                      selectedStatus === status.value &&
                      styles.statusChipRejected,
                  ]}
                  onPress={() => handleUpdateWorkOrderStatus(status.value)}
                >
                  <Text
                    style={[
                      styles.statusChipText,
                      selectedStatus === status.value && styles.statusChipTextActive,
                      status.value === 'cancelled' && styles.statusChipTextOnHold,
                      status.value === 'complete' &&
                        selectedStatus === status.value &&
                        styles.statusChipTextComplete,
                      status.value === 'cancelled' &&
                        selectedStatus === status.value &&
                        styles.statusChipTextRejected,
                    ]}
                  >
                    {status.label}
                  </Text>
                </TouchableOpacity>
              ))}
            </ScrollView>

            <View style={styles.progressRow}>
              <Text style={styles.progressLabel}>Progressing…</Text>
              <Text style={styles.progressValue}>
                Completed {completedTasks}/{totalTasks}
              </Text>
            </View>

            {/* Debug: Show task breakdown */}
            {__DEV__ && (
              <View style={[styles.sectionCard, { backgroundColor: '#f0f8ff', marginBottom: 10 }]}>
                <Text style={[styles.label, { color: '#015185', fontSize: 14 }]}>Debug: Task Breakdown</Text>
                <Text style={[styles.value, { fontSize: 12 }]}>Total Tasks: {safeTasks.length}</Text>
                <Text style={[styles.value, { fontSize: 12 }]}>Current Tab: {selectedTaskTab}</Text>
                <Text style={[styles.value, { fontSize: 12 }]}>Tasks by Category:</Text>
                {['Task', 'Parts', 'Labor', 'Additional Cost', 'Attach', 'Remarks'].map(category => {
                  const count = safeTasks.filter(task => task.category === category).length;
                  return count > 0 ? (
                    <Text key={category} style={[styles.value, { fontSize: 11, marginLeft: 10 }]}>
                      {category}: {count}
                    </Text>
                  ) : null;
                })}
                {safeTasks.length === 0 && (
                  <Text style={[styles.value, { fontSize: 12, color: '#e74c3c' }]}>
                    No tasks found in work order data
                  </Text>
                )}
                <Text style={[styles.value, { fontSize: 11, color: '#666', marginTop: 5 }]}>
                  Work Order Items: {workOrder?.items?.length || 0}
                </Text>
                <Text style={[styles.value, { fontSize: 11, color: '#666' }]}>
                  Checklist Items: {safeChecklistItems.length}
                </Text>
              </View>
            )}

            <View style={styles.taskTabsRow}>
              {TASK_TABS.map((tab) => (
                <TouchableOpacity
                  key={tab}
                  style={[styles.taskTab, selectedTaskTab === tab && styles.taskTabActive]}
                  onPress={() => setSelectedTaskTab(tab)}
                >
                  <Text style={[styles.taskTabText, selectedTaskTab === tab && styles.taskTabTextActive]}>
                    {tab}
                  </Text>
                </TouchableOpacity>
              ))}
              <TouchableOpacity style={styles.addTaskBtn} onPress={() => setAddItemModal(true)}>
                <Icon name="add" size={20} color="#3481BC" />
                <Text style={styles.addTaskBtnText}>Add Item</Text>
              </TouchableOpacity>
            </View>

            {selectedTaskTab === 'Attach' ? (
              <View style={styles.sectionCard}>
                <Text style={styles.label}>
                  Attachments:{' '}
                  <Text style={styles.value}>
                    {workOrder.pdfDownloadUrl ? (
                      <TouchableOpacity
                        onPress={() =>
                          navigation.navigate('PdfviewPage', { pdfUrl: workOrder.pdfDownloadUrl })
                        }
                      >
                        <Text style={styles.sectionMore}>View PDF</Text>
                      </TouchableOpacity>
                    ) : (
                      'No attachments available.'
                    )}
                  </Text>
                </Text>
              </View>
            ) : (
              renderTaskTabContent()
            )}
            {selectedTaskTab === 'Task' && (
              <View style={{ alignItems: 'flex-start', marginTop: 10, marginBottom: 20, width: '100%' }}>
                <TouchableOpacity
                  style={{
                    flexDirection: 'row',
                    alignItems: 'center',
                    backgroundColor: '#015185',
                    borderRadius: 6,
                    paddingVertical: 6,
                    paddingHorizontal: 14,
                    marginTop: 8,
                  }}
                  onPress={() => setAddTaskModalVisible(true)}
                >
                  <Icon name="add" size={16} color="#fff" />
                  <Text style={{ color: '#fff', fontWeight: '600', fontSize: 13, marginLeft: 5 }}>Add Task</Text>
                </TouchableOpacity>
              </View>
            )}
            {selectedTaskTab === 'Parts' && (
              <View style={{ alignItems: 'flex-start', marginTop: 10, marginBottom: 20, width: '100%' }}>
                <TouchableOpacity
                  style={{
                    flexDirection: 'row',
                    alignItems: 'center',
                    backgroundColor: '#015185',
                    borderRadius: 6,
                    paddingVertical: 6,
                    paddingHorizontal: 14,
                    marginTop: 8,
                  }}
                  onPress={() => setAddPartsModalVisible(true)}
                >
                  <Icon name="add" size={16} color="#fff" />
                  <Text style={{ color: '#fff', fontWeight: '600', fontSize: 13, marginLeft: 5 }}>Add Parts</Text>
                </TouchableOpacity>
              </View>
            )}
            {selectedTaskTab === 'Labor' && (
              <View style={{ alignItems: 'flex-start', marginTop: 10, marginBottom: 20, width: '100%' }}>
                <TouchableOpacity
                  style={{
                    flexDirection: 'row',
                    alignItems: 'center',
                    backgroundColor: '#015185',
                    borderRadius: 6,
                    paddingVertical: 6,
                    paddingHorizontal: 14,
                    marginTop: 8,
                  }}
                  onPress={() => setAddLaborModalVisible(true)}
                >
                  <Icon name="add" size={16} color="#fff" />
                  <Text style={{ color: '#fff', fontWeight: '600', fontSize: 13, marginLeft: 5 }}>Add Labor</Text>
                </TouchableOpacity>
              </View>
            )}
            {selectedTaskTab === 'Additional Cost' && (
              <View style={{ alignItems: 'flex-start', marginTop: 10, marginBottom: 20, width: '100%' }}>
                <TouchableOpacity
                  style={{
                    flexDirection: 'row',
                    alignItems: 'center',
                    backgroundColor: '#015185',
                    borderRadius: 6,
                    paddingVertical: 6,
                    paddingHorizontal: 14,
                    marginTop: 8,
                  }}
                  onPress={() => setAddAdditionalCostModalVisible(true)}
                >
                  <Icon name="add" size={16} color="#fff" />
                  <Text style={{ color: '#fff', fontWeight: '600', fontSize: 13, marginLeft: 5 }}>Add Additional Cost</Text>
                </TouchableOpacity>
              </View>
            )}
            {selectedTaskTab === 'Attach' && (
              <View style={{ alignItems: 'flex-start', marginTop: 10, marginBottom: 20, width: '100%' }}>
                <TouchableOpacity
                  style={{
                    flexDirection: 'row',
                    alignItems: 'center',
                    backgroundColor: '#015185',
                    borderRadius: 6,
                    paddingVertical: 6,
                    paddingHorizontal: 14,
                    marginTop: 8,
                  }}
                  onPress={() => setAddAttachModalVisible(true)}
                >
                  <Icon name="add" size={16} color="#fff" />
                  <Text style={{ color: '#fff', fontWeight: '600', fontSize: 13, marginLeft: 5 }}>Add Attach</Text>
                </TouchableOpacity>
              </View>
            )}
          </View>
        )}
        {tabIndex === 1 && (
          <View style={{ width: '100%', flex: 1 }}>
            {/* Remark Tab Content */}
            <FlatList
              data={safeTasks
                .filter((item) => item.category === 'Remarks')
                .sort((a, b) => (b.createdAt || 0) - (a.createdAt || 0))}
              keyExtractor={(item) => item.id}
              contentContainerStyle={{ padding: width * 0.03, paddingBottom: height * 0.1 }}
              renderItem={({ item }) => (
                <View style={styles.remarkCardRow}>
                  <View style={styles.remarkAvatar}>
                    <Icon name="person-circle-outline" size={40} color="#bbb" />
                  </View>
                  <View style={{ flex: 1 }}>
                    <View style={styles.remarkHeader}>
                      <Text style={styles.remarkEmail}>{item.userEmail}</Text>
                      <Text style={styles.remarkRole}>{item.userRole || 'Worker'}</Text>
                    </View>
                    <Text style={styles.remarkText}>{item.description || 'No description'}</Text>
                    <Text style={styles.remarkDate}>
                      {item.createdAt
                        ? new Date(item.createdAt).toLocaleString('en-GB', {
                            day: '2-digit',
                            month: 'short',
                            year: 'numeric',
                            hour: '2-digit',
                            minute: '2-digit',
                          })
                        : 'N/A'}
                    </Text>
                  </View>
                </View>
              )}
              ListEmptyComponent={
                <Text style={styles.emptyText}>No remarks available.</Text>
              }
            />
          </View>
        )}
        {tabIndex === 1 && (
          <View style={styles.remarkInputBar}>
            <TouchableOpacity>
              <Icon name="mic-outline" size={24} color="#888" />
            </TouchableOpacity>
            <TextInput
              style={styles.remarkInput}
              placeholder="Type a remark..."
              value={newItemTitle}
              onChangeText={setNewItemTitle}
            />
            <TouchableOpacity onPress={handleAddRemark}>
              <Icon name="send" size={24} color="#3481BC" />
            </TouchableOpacity>
          </View>
        )}
      </ScrollView>

      {/* Additional Details Modal */}
      <Modal
        visible={detailsModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setDetailsModalVisible(false)}
      >
        <View style={styles.detailsModalOverlay}>
          <View style={styles.detailsModalContent}>
            <Text style={styles.detailsModalTitle}>Work Order Details</Text>
            <View style={styles.detailsModalRow}>
              <Text style={styles.detailsModalLabel}>Job Card Type</Text>
              <Text style={styles.detailsModalValue}>{workOrder?.workOrderType || 'N/A'}</Text>
            </View>
            <View style={styles.detailsModalRow}>
              <Text style={styles.detailsModalLabel}>Estimated Cost</Text>
              <Text style={styles.detailsModalValue}>{workOrder?.estimatedCost || 'N/A'}</Text>
            </View>
            <View style={styles.detailsModalRow}>
              <Text style={styles.detailsModalLabel}>Estimated Labor Time</Text>
              <Text style={styles.detailsModalValue}>{workOrder?.estimatedLaborTime || 'N/A'}</Text>
            </View>
            <View style={styles.detailsModalRow}>
              <Text style={styles.detailsModalLabel}>Linked Inspection</Text>
              <TouchableOpacity 
                onPress={() => {
                  if (workOrder?.linkInspection?.id) {
                    // Navigate to inspection details
                    navigation.navigate('InspectionReportHome', { 
                      inspectionId: workOrder.linkInspection.id 
                    });
                  }
                }}
              >
                <Text style={[
                  styles.detailsModalValue,
                  workOrder?.linkInspection?.name && styles.detailsModalLink
                ]}>
                  {workOrder?.linkInspection?.name || workOrder?.linkInspection?.reportNumber || 'None'}
                </Text>
              </TouchableOpacity>
            </View>
            {workOrder?.linkInspection?.status && (
              <View style={styles.detailsModalRow}>
                <Text style={styles.detailsModalLabel}>Inspection Status</Text>
                <Text style={[styles.detailsModalValue, {
                  color: workOrder.linkInspection.status === 'Complete' ? '#28a745' : 
                         workOrder.linkInspection.status === 'In Progress' ? '#3481BC' : '#f39c12'
                }]}>
                  {workOrder.linkInspection.status}
                </Text>
              </View>
            )}
            <View style={styles.detailsModalRow}>
              <Text style={styles.detailsModalLabel}>Linked Job Card</Text>
              <TouchableOpacity 
                onPress={() => {
                  if (workOrder?.linkedWorkOrder?.id) {
                    // Navigate to linked work order
                    navigation.navigate('WorkerDashboardhome', { 
                      workOrderId: workOrder.linkedWorkOrder.id,
                      adminUid: adminUid
                    });
                  }
                }}
              >
                <Text style={[
                  styles.detailsModalValue,
                  workOrder?.linkedWorkOrder?.title && styles.detailsModalLink
                ]}>
                  {workOrder?.linkedWorkOrder?.title || 'None'}
                </Text>
              </TouchableOpacity>
            </View>
            {workOrder?.linkedWorkOrder?.status && (
              <View style={styles.detailsModalRow}>
                <Text style={styles.detailsModalLabel}>Linked Job Status</Text>
                <Text style={[styles.detailsModalValue, {
                  color: workOrder.linkedWorkOrder.status === 'Complete' ? '#28a745' : 
                         workOrder.linkedWorkOrder.status === 'In Progress' ? '#3481BC' : '#f39c12'
                }]}>
                  {workOrder.linkedWorkOrder.status}
                </Text>
              </View>
            )}
            <View style={styles.detailsModalRow}>
              <Text style={styles.detailsModalLabel}>Start Date</Text>
              <Text style={styles.detailsModalValue}>
                {workOrder?.startDate
                  ? new Date(workOrder.startDate).toLocaleDateString('en-US', {
                      year: 'numeric',
                      month: 'short',
                      day: 'numeric',
                    })
                  : 'N/A'}
              </Text>
            </View>
            <TouchableOpacity
              style={styles.detailsModalCloseBtn}
              onPress={() => setDetailsModalVisible(false)}
            >
              <Text style={styles.detailsModalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {/* Add Item Modal */}
      <Modal
        visible={addItemModal}
        transparent
        animationType="slide"
        onRequestClose={() => setAddItemModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Item</Text>
            <TextInput
              placeholder="Item Title"
              value={newItemTitle}
              onChangeText={setNewItemTitle}
              style={styles.input}
            />
            <View style={styles.categoryRow}>
              <Text style={styles.label}>Category</Text>
              <View style={styles.categoryButtons}>
                {TASK_TABS.slice(0, -1).map((category) => (
                  <TouchableOpacity
                    key={category}
                    style={[
                      styles.categoryButton,
                      newItemCategory === category && styles.categoryButtonActive,
                    ]}
                    onPress={() => setNewItemCategory(category)}
                  >
                    <Text
                      style={[
                        styles.categoryButtonText,
                        newItemCategory === category && styles.categoryButtonTextActive,
                      ]}
                    >
                      {category}
                    </Text>
                  </TouchableOpacity>
                ))}
              </View>
            </View>
            {/* Type selection for Task only */}
            {newItemCategory === 'Task' && (
              <View style={styles.addTaskRowAligned}>
                <Text style={styles.addTaskLabel}>Type</Text>
                <View style={styles.addTaskTypeRowInline}>
                  {['Repair', 'Replace'].map(type => (
                    <TouchableOpacity
                      key={type}
                      style={[
                        styles.addTaskTypeBtn,
                        newItemType === type && styles.addTaskTypeBtnActive,
                      ]}
                      onPress={() => setNewItemType(type)}
                    >
                      <Text style={[
                        styles.addTaskTypeBtnText,
                        newItemType === type && styles.addTaskTypeBtnTextActive,
                      ]}>{type}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
              </View>
            )}
            <View style={styles.modalBtnRow}>
              <TouchableOpacity
                style={styles.modalBtn}
                onPress={() => {
                  setAddItemModal(false);
                  setNewItemTitle('');
                  setNewItemCategory('Task');
                  setNewItemType('Repair'); // Reset type
                }}
              >
                <Text style={styles.modalBtnText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalBtn, styles.modalBtnAdd]}
                onPress={handleAddItem}
              >
                <Text style={styles.modalBtnTextAdd}>Add</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Add Task Modal */}
      <Modal
        visible={addTaskModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAddTaskModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.addTaskModalContent}>
            <Text style={styles.addTaskModalTitle}>Add Task</Text>
            {/* Task Row */}
            <View style={styles.addTaskRowAligned}>
              <Text style={styles.addTaskLabel}>Task</Text>
              <TextInput
                style={[styles.editTaskInputUnderline]}
                value={newTaskText}
                onChangeText={setNewTaskText}
                placeholderTextColor="#888"
              />
            </View>
            {/* Type Row */}
            <View style={styles.addTaskRowAligned}>
              <Text style={styles.addTaskLabel}>Type</Text>
              <View style={styles.addTaskTypeRowInline}>
                {['Repair', 'Replace'].map(type => (
                  <TouchableOpacity
                    key={type}
                    style={[
                      styles.addTaskTypeBtn,
                      newTaskType === type && styles.addTaskTypeBtnActive,
                    ]}
                    onPress={() => setNewTaskType(type)}
                  >
                    <Text style={[
                      styles.addTaskTypeBtnText,
                      newTaskType === type && styles.addTaskTypeBtnTextActive,
                    ]}>{type}</Text>
                  </TouchableOpacity>
                ))}
              </View>
            </View>
            {/* Assign to Row */}
            <TouchableOpacity 
              style={styles.addTaskRow} 
              onPress={() => { 
                fetchTeamMembers(); 
                setAssignToModalVisible(true); 
              }}
            >
              <Text style={styles.addTaskLabel}>Assign to</Text>
              <View style={styles.addTaskRowRight}>
                <Text style={styles.addTaskRowValue}>
                  {tempAssignTo.length > 0 ? `${tempAssignTo.length} Assigned` : 'Set'}
                </Text>
                <Icon name="chevron-forward" size={20} color="#015185" />
              </View>
            </TouchableOpacity>
            {/* Show selected users */}
            {tempAssignTo.length > 0 && (
              <View style={styles.selectedUsersContainer}>
                {tempAssignTo.map(user => (
                  <View key={user.id} style={styles.selectedUserChip}>
                    <Text style={styles.selectedUserChipText}>{user.fullName}</Text>
                  </View>
                ))}
              </View>
            )}
            {/* Note Row */}
            <View style={styles.addTaskRowAligned}>
              <Text style={styles.addTaskLabel}>Note</Text>
              <TextInput
                style={[styles.editTaskInputUnderline]}
                value={newTaskNote}
                onChangeText={setNewTaskNote}
                placeholderTextColor="#888"
              />
            </View>
            {/* Fault List Row */}
            <TouchableOpacity style={styles.addTaskRow}>
              <Text style={styles.addTaskLabel}>Fault List</Text>
              <View style={styles.addTaskRowRight}>
                <Text style={styles.addTaskRowValue}>Select</Text>
                <Icon name="chevron-forward" size={20} color="#015185" />
              </View>
            </TouchableOpacity>
            <View style={styles.addTaskModalBtnRow}>
              <TouchableOpacity
                style={styles.addTaskSaveBtn}
                onPress={() => {
                  setAddTaskModalVisible(false);
                  // Reset form when canceling
                  setNewTaskText('');
                  setNewTaskType('Repair');
                  setNewTaskNote('');
                  setTempAssignTo([]);
                }}
              >
                <Text style={styles.addTaskSaveBtnText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.addTaskSaveBtn}
                onPress={handleSaveNewTask}
              >
                <Text style={styles.addTaskSaveBtnText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
          {renderAssignToModal()}
        </View>
      </Modal>

      {/* PARTS MODAL (EXACT COPY) */}
      <Modal
        visible={addPartsModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAddPartsModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.addPartsModalContent}>
            <Text style={styles.addPartsModalTitle}>Add Parts</Text>
            <TextInput style={styles.addPartsInput} placeholder="Item Number" placeholderTextColor="#888" value={partsItemNumber} onChangeText={setPartsItemNumber} />
            <TextInput style={styles.addPartsInput} placeholder="Part Number" placeholderTextColor="#888" value={partsPartNumber} onChangeText={setPartsPartNumber} />
            <TextInput style={styles.addPartsInput} placeholder="Item Name" placeholderTextColor="#888" value={partsItemName} onChangeText={setPartsItemName} />
            <TextInput style={styles.addPartsInput} placeholder="Quantity" placeholderTextColor="#888" value={partsQuantity} onChangeText={setPartsQuantity} keyboardType="numeric" />
            <TextInput style={styles.addPartsInput} placeholder="Unit" placeholderTextColor="#888" value={partsUnit} onChangeText={setPartsUnit} />
            <TextInput style={styles.addPartsInput} placeholder="Item Cost" placeholderTextColor="#888" value={partsItemCost} onChangeText={setPartsItemCost} keyboardType="numeric" />
            <TextInput style={styles.addPartsInput} placeholder="Site" placeholderTextColor="#888" value={partsSite} onChangeText={setPartsSite} />
            <TextInput style={[styles.addPartsInput, styles.addPartsInputNote]} placeholder="Note" placeholderTextColor="#888" value={partsNote} onChangeText={setPartsNote} multiline numberOfLines={3} />
            <View style={styles.addPartsModalBtnRow}>
              <TouchableOpacity style={styles.addPartsSaveBtn} onPress={()=>setAddPartsModalVisible(false)}>
                <Text style={styles.addPartsSaveBtnText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.addPartsSaveBtn} onPress={handleSaveParts}>
                <Text style={styles.addPartsSaveBtnText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      {/* LABOR MODAL (EXACT COPY) */}
      <Modal
        visible={addLaborModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAddLaborModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.addLaborModalContent}>
            <Text style={styles.addLaborModalTitle}>Add Labor Details</Text>
            <TextInput style={styles.addLaborInput} placeholder="Worker Name" placeholderTextColor="#888" value={laborWorker} onChangeText={setLaborWorker} />
            <TextInput style={styles.addLaborInput} placeholder="Labor Code" placeholderTextColor="#888" value={laborCode} onChangeText={setLaborCode} />
            <TextInput style={styles.addLaborInput} placeholder="Rate" placeholderTextColor="#888" value={laborRate} onChangeText={setLaborRate} keyboardType="numeric" />
            <View style={styles.addLaborWorkHoursRow}>
              <Text style={styles.addLaborLabel}>Work Hours</Text>
              <TextInput style={styles.addLaborInputSmall} placeholder="Hours" placeholderTextColor="#888" value={laborHours} onChangeText={setLaborHours} keyboardType="numeric" />
              <TextInput style={styles.addLaborInputSmall} placeholder="Minut...." placeholderTextColor="#888" value={laborMinutes} onChangeText={setLaborMinutes} keyboardType="numeric" />
            </View>
            <TextInput style={[styles.addLaborInput, styles.addLaborInputNote]} placeholder="Note" placeholderTextColor="#888" value={laborNote} onChangeText={setLaborNote} multiline numberOfLines={3} />
            <View style={styles.addLaborModalBtnRow}>
              <TouchableOpacity style={styles.addLaborSaveBtn} onPress={()=>setAddLaborModalVisible(false)}>
                <Text style={styles.addLaborSaveBtnText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.addLaborSaveBtn} onPress={handleSaveLabor}>
                <Text style={styles.addLaborSaveBtnText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      {/* ADDITIONAL COST MODAL (EXACT COPY) */}
      <Modal
        visible={addAdditionalCostModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAddAdditionalCostModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.addAdditionalCostModalContent}>
            <Text style={styles.addAdditionalCostModalTitle}>Add Additional Cost</Text>
            <TextInput style={[styles.addAdditionalCostInput, styles.addAdditionalCostInputNote]} placeholder="Description" placeholderTextColor="#888" value={additionalCostDescription} onChangeText={setAdditionalCostDescription} multiline numberOfLines={3} />
            <TextInput style={styles.addAdditionalCostInput} placeholder="Select Labor Code" placeholderTextColor="#888" value={additionalCostAmount} onChangeText={setAdditionalCostAmount} />
            <View style={styles.addAdditionalCostModalBtnRow}>
              <TouchableOpacity style={styles.addAdditionalCostSaveBtn} onPress={()=>setAddAdditionalCostModalVisible(false)}>
                <Text style={styles.addAdditionalCostSaveBtnText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.addAdditionalCostSaveBtn} onPress={handleSaveAdditionalCost}>
                <Text style={styles.addAdditionalCostSaveBtnText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      {/* ATTACH MODAL (SIMPLE PLACEHOLDER) */}
      <Modal
        visible={addAttachModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAddAttachModalVisible(false)}
      >
        <View style={{flex:1,backgroundColor:'rgba(0,0,0,0.5)',justifyContent:'center',alignItems:'center'}}>
          <View style={{backgroundColor:'#fff',borderRadius:12,padding:20,width:'90%',maxWidth:400,alignItems:'center'}}>
            <Text style={{fontSize:22,color:'#015185',fontWeight:'700',textAlign:'center',marginBottom:18}}>Add Attachment</Text>
            <TouchableOpacity style={{backgroundColor:'#e0e0e0',borderRadius:6,padding:12,marginBottom:16}} onPress={handlePickFile}>
              <Icon name="attach" size={22} color="#015185" />
              <Text style={{color:'#015185',fontWeight:'600',marginTop:4}}>Pick File</Text>
            </TouchableOpacity>
            <Text style={{color:'#888',marginBottom:10}}>{attachFileName || 'No file selected.'}</Text>
            <View style={{flexDirection:'row',justifyContent:'flex-end',marginTop:10}}>
              <TouchableOpacity style={{borderRadius:5,paddingVertical:8,paddingHorizontal:16,backgroundColor:'#e0e0e0',marginRight:10}} onPress={()=>setAddAttachModalVisible(false)}>
                <Text style={{color:'#015185',fontSize:14,fontWeight:'600'}}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={{borderRadius:5,paddingVertical:8,paddingHorizontal:16,backgroundColor:'#015185'}} onPress={handleSaveAttach}>
                <Text style={{color:'#fff',fontSize:14,fontWeight:'600'}}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Show appropriate loader based on operation */}
      <SyncModal 
        visible={isLoadingWorkOrder} 
        message="Loading Work Order..." 
      />
      <SyncModal 
        visible={isUpdatingStatus} 
        message="Updating Status..." 
      />
      <SyncModal 
        visible={isAddingItem} 
        message="Adding Item..." 
      />
      <SyncModal 
        visible={isAddingRemark} 
        message="Adding Remark..." 
      />
      <SyncModal 
        visible={isUpdatingWorkOrderStatus} 
        message="Updating Work Order..." 
      />
      <SyncModal visible={loaderVisible} message="Processing..." />
      {renderEditTaskModal()}
      {renderDropdownModal()}
      {renderStatusDropdownModal()}
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f7f9fa',
  },
  header: {
    backgroundColor: '#3481BC',
    paddingTop: height * 0.05,
    paddingBottom: 0, // Remove extra space below header
    elevation: 4,
  },
  headerRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: width * 0.04,
    marginBottom: height * 0.01,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '700',
    flex: 1,
    textAlign: 'center',
  },
  headerInfoRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    marginBottom: height * 0.01,
  },
  woNumber: {
    color: '#fff',
    fontWeight: '600',
    fontSize: width * 0.04,
  },
  woSub: {
    color: '#cce6ff',
    fontSize: width * 0.035,
  },
  timerBtn: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#ff7e7e',
    borderRadius: 18,
    paddingHorizontal: width * 0.03,
    paddingVertical: height * 0.005,
  },
  timerBtnText: {
    color: '#fff',
    marginLeft: width * 0.01,
    fontWeight: '600',
    fontSize: width * 0.035,
  },
  topTabsRow: {
    flexDirection: 'row',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  topTab: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: height * 0.015,
    borderBottomWidth: 2,
    borderBottomColor: 'transparent',
  },
  topTabActive: {
    borderBottomColor: '#fff',
  },
  topTabText: {
    color: '#e0e0e0',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  topTabTextActive: {
    color: '#fff',
    fontWeight: '700',
  },
  content: {
    flex: 1,
    padding: width * 0.04,
  },
  sectionCard: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: width * 0.035,
    marginBottom: height * 0.015,
    elevation: 1,
  },
  label: {
    color: '#333',
    fontWeight: '600',
    fontSize: width * 0.04,
    marginBottom: height * 0.005,
  },
  value: {
    color: '#015185',
    fontWeight: '500',
    fontSize: width * 0.035,
  },
  sectionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#f7f7f7',
    padding: width * 0.03,
    borderRadius: 8,
    marginBottom: height * 0.015,
  },
  sectionLabel: {
    color: '#888',
    fontWeight: '600',
    fontSize: width * 0.035,
  },
  sectionMore: {
    color: '#3481BC',
    fontWeight: '600',
    fontSize: width * 0.035,
  },
  statusChipsRow: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: width * 0.015,
    marginBottom: height * 0.015,
  },
  statusChip: {
    paddingHorizontal: width * 0.035,
    paddingVertical: height * 0.01,
    borderRadius: 18,
    backgroundColor: '#f0f0f0',
    borderWidth: 1,
    borderColor: '#e0e0e0',
    marginRight: width * 0.015, // Add spacing between badges
  },
  statusChipActive: {
    backgroundColor: '#e0f7fa',
    borderColor: '#3481BC',
  },
  statusChipOnHold: {
    backgroundColor: '#fffbe6',
    borderColor: '#ffe066',
  },
  statusChipComplete: {
    backgroundColor: '#e6f4ea',
    borderColor: '#28a745',
  },
  statusChipRejected: {
    backgroundColor: '#ffe6e6',
    borderColor: '#e74c3c',
  },
  statusChipText: {
    color: '#555',
    fontWeight: '500',
    fontSize: width * 0.035,
  },
  statusChipTextActive: {
    color: '#3481BC',
    fontWeight: '600',
  },
  statusChipTextOnHold: {
    color: '#bfa100',
  },
  statusChipTextComplete: {
    color: '#28a745',
  },
  statusChipTextRejected: {
    color: '#e74c3c',
  },
  progressRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: height * 0.01,
  },
  progressLabel: {
    color: '#888',
    fontSize: width * 0.035,
  },
  progressValue: {
    color: '#015185',
    fontWeight: '600',
    fontSize: width * 0.035,
  },
  taskTabsRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginTop: height * 0.01,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    paddingBottom: height * 0.005,
  },
  taskTab: {
    paddingHorizontal: width * 0.02,
    paddingVertical: height * 0.01,
    marginRight: width * 0.02,
    borderBottomWidth: 2,
    borderBottomColor: 'transparent',
  },
  taskTabActive: {
    borderBottomColor: '#3481BC',
  },
  taskTabText: {
    color: '#888',
    fontSize: width * 0.035,
  },
  taskTabTextActive: {
    color: '#3481BC',
    fontWeight: '600',
  },
  addTaskBtn: {
    flexDirection: 'row',
    alignItems: 'center',
    marginLeft: 'auto',
    paddingHorizontal: width * 0.02,
    paddingVertical: height * 0.008,
    borderRadius: 16,
    backgroundColor: '#eaf4fb',
  },
  addTaskBtnText: {
    color: '#3481BC',
    fontWeight: '600',
    marginLeft: width * 0.01,
    fontSize: width * 0.035,
  },
  taskCard: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: width * 0.035,
    marginBottom: height * 0.015,
    flexDirection: 'row',
    alignItems: 'center',
    elevation: 1,
  },
  title: {
    color: '#222',
    fontWeight: '600',
    fontSize: width * 0.04,
  },
  sectionText: {
    color: '#e74c3c',
    fontSize: width * 0.035,
    marginTop: height * 0.005,
  },
  taskNote: {
    color: '#666',
    fontSize: width * 0.035,
    marginTop: height * 0.005,
  },
  taskDetailText: {
    color: '#666',
    fontSize: width * 0.035,
    marginTop: height * 0.005,
  },
  taskStatusRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginLeft: width * 0.025,
  },
  taskStatusChip: {
    borderRadius: 12,
    paddingHorizontal: width * 0.025,
    paddingVertical: height * 0.005,
    marginRight: width * 0.015,
  },
  taskStatusText: {
    fontSize: width * 0.035,
    fontWeight: '600',
  },
  checklistItem: {
    backgroundColor: '#f9f9f9',
    borderRadius: 10,
    padding: width * 0.035,
    marginBottom: height * 0.015,
    flexDirection: 'row',
    alignItems: 'center',
    elevation: 1,
  },
  checklistContent: {
    flex: 1,
  },
  checklistText: {
    color: '#222',
    fontWeight: '600',
    fontSize: width * 0.04,
  },
  checklistSection: {
    color: '#888',
    fontSize: width * 0.035,
    marginTop: height * 0.005,
  },
  checklistStatus: {
    fontSize: width * 0.035,
    fontWeight: '600',
    marginTop: height * 0.005,
  },
  emptyText: {
    color: '#888',
    fontSize: width * 0.04,
    marginTop: height * 0.03,
    textAlign: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: width * 0.05,
    width: width * 0.85,
    elevation: 4,
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: '700',
    color: '#3481BC',
    marginBottom: height * 0.015,
  },
  input: {
    borderWidth: 1,
    borderColor: '#e0e0e0',
    borderRadius: 8,
    padding: width * 0.025,
    marginBottom: height * 0.015,
    fontSize: width * 0.04,
    color: '#333',
  },
  categoryRow: {
    marginBottom: height * 0.015,
  },
  categoryButtons: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: width * 0.02,
    marginTop: height * 0.01,
  },
  categoryButton: {
    paddingHorizontal: width * 0.03,
    paddingVertical: height * 0.01,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e0e0f0',
    backgroundColor: '#f0f0f0',
  },
  categoryButtonActive: {
    backgroundColor: '#e0f7fa',
    borderColor: '#3481BC',
  },
  categoryButtonText: {
    fontSize: width * 0.035,
    color: '#555',
  },
  categoryButtonTextActive: {
    color: '#3481BC',
    fontWeight: '600',
  },
  modalBtnRow: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: height * 0.01,
  },
  modalBtn: {
    paddingHorizontal: width * 0.04,
    paddingVertical: height * 0.01,
    borderRadius: 8,
    backgroundColor: '#e0e0e0',
    marginLeft: width * 0.025,
  },
  modalBtnAdd: {
    backgroundColor: '#3481BC',
  },
  modalBtnText: {
    color: '#3481BC',
    fontWeight: '600',
    fontSize: width * 0.04,
  },
  modalBtnTextAdd: {
    color: '#fff',
    fontWeight: '600',
    fontSize: width * 0.04,
  },
  loaderModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loaderModalContent: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: width * 0.075,
    alignItems: 'center',
    elevation: 6,
  },
  loaderModalText: {
    marginTop: height * 0.015,
    fontSize: width * 0.04,
    color: '#3481BC',
    fontWeight: '600',
  },
  detailsModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  detailsModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: width * 0.05,
    width: width * 0.85,
    elevation: 6,
  },
  detailsModalTitle: {
    fontSize: width * 0.045,
    fontWeight: '700',
    color: '#222',
    textAlign: 'center',
    marginBottom: height * 0.02,
  },
  detailsModalRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    paddingVertical: height * 0.01,
  },
  detailsModalLabel: {
    color: '#888',
    fontSize: width * 0.04,
    flex: 1,
  },
  detailsModalValue: {
    color: '#222',
    fontSize: width * 0.04,
    flex: 1,
    textAlign: 'right',
  },
  detailsModalLink: {
    color: '#3481BC',
    fontSize: width * 0.04,
    flex: '1',
    textAlign: 'right',
    textDecorationLine: 'underline',
  },
  detailsModalCloseBtn: {
    marginTop: height * 0.02,
    alignSelf: 'center',
    paddingVertical: height * 0.01,
    paddingHorizontal: width * 0.075,
    borderRadius: 8,
    backgroundColor: '#eaf4fb',
  },
  detailsModalCloseText: {
    color: '#3481BC',
    fontWeight: '600',
    fontSize: width * 0.04,
  },
  remarkCardRow: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    padding: width * 0.03,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  remarkAvatar: {
    marginRight: width * 0.03,
  },
  remarkHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: height * 0.005,
  },
  remarkEmail: {
    color: '#333',
    fontWeight: '600',
    fontSize: width * 0.04,
    marginRight: width * 0.02,
  },
  remarkRole: {
    color: '#888',
    fontSize: width * 0.035,
  },
  remarkText: {
    color: '#333',
    fontSize: width * 0.037,
    fontWeight: '400',
  },
  remarkDate: {
    color: '#888',
    fontSize: width * 0.035,
    marginTop: height * 0.005,
  },
  remarkInputBar: {
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
    backgroundColor: '#fff',
    borderTopWidth: 1,
    borderTopColor: '#eee',
  },
  remarkInput: {
    flex: 1,
    padding: width * 0.02,
    fontSize: width * 0.04,
    color: '#333',
    backgroundColor: '#f5f5f5',
    borderRadius: 8,
    marginHorizontal: width * 0.02,
  },
  // Loader styles (same as Inspection_Report_Home)
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
    marginLeft: 10,
  },
  syncIcon: {
    width: 26,
    height: 26,
    resizeMode: 'contain',
    marginRight: 0,
    marginBottom: 10,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    elevation: 6,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
    marginTop: 0,
    textAlign: 'center',
  },
  addTaskModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignSelf: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  addTaskModalTitle: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  addTaskFieldGroup: {
    marginBottom: 12,
  },
  addTaskLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
    marginBottom: 4,
  },
  addTaskInput: {
    borderWidth: 1,
    borderColor: '#e0e0e0',
    borderRadius: 8,
    padding: 10,
    fontSize: 16,
    color: '#333',
    backgroundColor: '#f7f9fa',
  },
  addTaskTypeRow: {
    flexDirection: 'row',
    gap: 8,
  },
  addTaskTypeBtn: {
    backgroundColor: '#e0e0e0',
    borderRadius: 5,
    paddingVertical: 6,
    paddingHorizontal: 18,
    marginRight: 8,
  },
  addTaskTypeBtnActive: {
    backgroundColor: '#015185',
  },
  addTaskTypeBtnText: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '500',
  },
  addTaskTypeBtnTextActive: {
    color: '#fff',
  },
  addTaskRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  addTaskRowRight: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 4,
  },
  addTaskRowValue: {
    color: '#015185',
    fontWeight: '600',
    fontSize: 15,
    marginRight: 2,
  },
  addTaskModalBtnRow: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: 18,
    gap: 10,
  },
  addTaskCancelBtn: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 24,
    backgroundColor: '#e0e0e0',
    marginRight: 10,
  },
  addTaskCancelBtnText: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '600',
  },
  addTaskSaveBtn: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 24,
    backgroundColor: '#015185',
    marginRight: 0,
  },
  addTaskSaveBtnText: {
    color: '#fff',
    fontSize: 15,
    fontWeight: '600',
  },
  addTaskRowAligned: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: 12,
  },
  addTaskInputInline: {
    borderWidth: 1,
    borderColor: '#e0e0e0',
    borderRadius: 8,
    padding: 10,
    fontSize: 16,
    color: '#333',
    backgroundColor: '#f7f9fa',
    flex: 1,
    marginLeft: 16,
  },
  addTaskTypeRowInline: {
    flexDirection: 'row',
    gap: 8,
    marginLeft: 16,
  },
  addTaskInputInlineNoBorder: {
    fontSize: 16,
    color: '#333',
    backgroundColor: 'transparent',
    flex: 1,
    marginLeft: 16,
    padding: 10,
  },
  addPartsModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignSelf: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  addPartsModalTitle: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  addPartsInput: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    padding: 10,
    fontSize: 16,
    color: '#015185',
    backgroundColor: 'transparent',
    marginBottom: 12,
  },
  addPartsInputNote: {
    height: 70,
    textAlignVertical: 'top',
  },
  addPartsLabel: {
    color: '#015185',
    fontWeight: '600',
    fontSize: 15,
    marginBottom: 6,
    marginTop: 0,
  },
  addPartsModalBtnRow: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
    gap: 10,
  },
  addPartsSaveBtn: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 24,
    backgroundColor: '#015185',
    marginRight: 0,
  },
  addPartsSaveBtnText: {
    color: '#fff',
    fontSize: 15,
    fontWeight: '600',
  },
  addLaborModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignSelf: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  addLaborModalTitle: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  addLaborInput: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    padding: 10,
    fontSize: 16,
    color: '#015185',
    backgroundColor: 'transparent',
    marginBottom: 12,
  },
  addLaborInputSmall: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    padding: 10,
    fontSize: 16,
    color: '#015185',
    backgroundColor: 'transparent',
    marginLeft: 8,
    width: 100,
    marginBottom: 0,
  },
  addLaborInputNote: {
    height: 70,
    textAlignVertical: 'top',
  },
  addLaborLabel: {
    color: '#015185',
    fontWeight: '600',
    fontSize: 15,
    marginBottom: 6,
    marginTop: 0,
    marginRight: 8,
    alignSelf: 'center',
  },
  addLaborWorkHoursRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 12,
  },
  addLaborModalBtnRow: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
    gap: 10,
  },
  addLaborSaveBtn: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 24,
    backgroundColor: '#015185',
    marginRight: 0,
  },
  addLaborSaveBtnText: {
    color: '#fff',
    fontSize: 15,
    fontWeight: '600',
  },
  addAdditionalCostModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignSelf: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  addAdditionalCostModalTitle: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  addAdditionalCostInput: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    padding: 10,
    fontSize: 16,
    color: '#015185',
    backgroundColor: 'transparent',
    marginBottom: 12,
  },
  addAdditionalCostInputNote: {
    height: 70,
    textAlignVertical: 'top',
  },
  addAdditionalCostModalBtnRow: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
    gap: 10,
  },
  addAdditionalCostSaveBtn: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 24,
    backgroundColor: '#015185',
    marginRight: 0,
  },
  addAdditionalCostSaveBtnText: {
    color: '#fff',
    fontSize: 15,
    fontWeight: '600',
  },
  assignToModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  assignToModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 18,
    width: '90%',
    maxWidth: 400,
    alignItems: 'center',
    elevation: 6,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  assignToModalTitle: {
    fontSize: 20,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 10,
  },
  assignToSearchInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 5,
    padding: 10,
    marginBottom: 10,
    fontSize: 16,
    color: '#888',
    backgroundColor: '#f5f5f5',
  },
  assignToUserItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    backgroundColor: '#fff',
  },
  assignToUserItemSelected: {
    backgroundColor: '#e6f0ff',
  },
  assignToUserAvatar: {
    width: 40,
    height: 40,
    borderRadius: 20,
    backgroundColor: '#ddd',
    marginRight: 10,
    alignItems: 'center',
    justifyContent: 'center',
  },
  assignToUserAvatarImage: {
    width: 40,
    height: 40,
    borderRadius: 20,
    backgroundColor: '#ddd',
    marginRight: 10,
    alignItems: 'center',
    justifyContent: 'center',
  },
  assignToUserInfo: {
    flex: 1,
  },
  assignToUserName: {
    fontWeight: 'bold',
    color: '#015185',
    fontSize: 16,
  },
  assignToUserDetails: {
    color: '#333',
    fontSize: 13,
  },
  assignToUserProjects: {
    color: '#333',
    fontSize: 12,
  },
  assignToUserCountries: {
    color: '#333',
    fontSize: 12,
  },
  assignToUserStatus: {
    fontSize: 12,
  },
  assignToCheckmark: {
    marginLeft: 8,
  },
  assignToUserList: {
    marginBottom: 10,
    maxHeight: 300,
  },
  assignToSaveButton: {
    borderRadius: 5,
    paddingVertical: 10,
    backgroundColor: '#015185',
    alignItems: 'center',
    marginTop: 8,
  },
  assignToSaveButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  selectedUsersContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginBottom: 8,
  },
  selectedUserChip: {
    backgroundColor: '#e6f0ff',
    borderRadius: 12,
    paddingHorizontal: 10,
    paddingVertical: 4,
    margin: 2,
  },
  selectedUserChipText: {
    color: '#015185',
    fontWeight: '600',
  },
  assignedUsersContainer: {
    marginTop: 8,
  },
  assignedUsersLabel: {
    color: '#015185',
    fontSize: 13,
    fontWeight: '600',
    marginBottom: 4,
  },
  assignedUsersList: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  assignedUserChip: {
    backgroundColor: '#e6f0ff',
    borderRadius: 12,
    paddingHorizontal: 8,
    paddingVertical: 4,
    margin: 2,
  },
  assignedUserChipText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '500',
  },
  noUsersContainer: {
    alignItems: 'center',
    marginBottom: 10,
  },
  noUsersText: {
    color: '#666',
    fontSize: 16,
    marginBottom: 4,
  },
  noUsersSubtext: {
    color: '#999',
    fontSize: 14,
    textAlign: 'center',
  },
  noSearchResultsContainer: {
    alignItems: 'center',
    marginTop: 10,
  },
  noSearchResultsText: {
    color: '#999',
    fontSize: 14,
    textAlign: 'center',
  },
  loadingContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    marginTop: 10,
  },
  loadingText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 5,
  },
  assignToModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  refreshButton: {
    padding: 5,
    borderRadius: 5,
    backgroundColor: '#e0e0e0',
  },
  searchInput: {
    borderWidth: 1,
    borderColor: '#e0e0e0',
    borderRadius: 8,
    padding: 10,
    marginBottom: 10,
    fontSize: 16,
    color: '#333',
    backgroundColor: '#f5f5f5',
  },
  modalCloseButton: {
    backgroundColor: '#3481BC',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 8,
    alignSelf: 'center',
    marginTop: 10,
  },
  modalCloseText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: 16,
  },
  editTaskInputUnderline: {
    borderBottomWidth: 2,
    borderBottomColor: '#000',
    borderRadius: 0,
    paddingHorizontal: 0,
    paddingVertical: 4,
    fontSize: 16,
    color: '#333',
    backgroundColor: 'transparent',
    marginLeft: 16,
    width: '70%',
  },
});

export default WorkerDashboardhome;