import React, { useState, useRef } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, Dimensions, TextInput, ScrollView, KeyboardAvoidingView, Platform, Modal, Keyboard } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import RiskManagementHeader from './RiskManagementHeader';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';

const MINI_RISK_ITEMS = [
  'Steps to complete the task',
  'Hazards Associated with Task',
  'Assess the Risk',
  'Steps to Eliminate / Control the Hazards',
];

const { width } = Dimensions.get('window');
const MAX_ROWS = 10;
const INITIAL_ROWS = 1;
const GUIDANCE_POINTS = [
  'Think through the task – Plan the steps to be taken to complete the task',
  'Look for the exposure – Identify the hazards associated with the job',
  'Assess the risk – Is this High Risk, Medium Risk or Low Risk',
  'Take Precautions – What steps have been taken to remove / control the hazard',
  'Do the Job Safely – You may carry on and work safely',
];
const createEmptyRow = () => ({
  step: '',
  hazard: '',
  assess: '',
  control: '',
});

const Risk_step2 = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const linkedReport = route.params?.report || null;
  const [currentStep, setCurrentStep] = useState(2);
  // Dynamic rows, each row has four simple inputs
  const [rows, setRows] = useState(Array.from({ length: INITIAL_ROWS }, () => createEmptyRow()));
  const [keyboardSpace, setKeyboardSpace] = useState(0);
  const [successModalVisible, setSuccessModalVisible] = useState(false);
  const [isGuidanceOpen, setIsGuidanceOpen] = useState(false);
  const scrollRef = useRef(null);
  const inputRefs = useRef({});
  const scrollYRef = useRef(0);

  const scrollIntoView = (key) => {
    setTimeout(() => {
      const ref = inputRefs.current[key];
      if (!ref || !ref.measureInWindow) return;
      ref.measureInWindow((x, y, w, h) => {
        const screenH = Dimensions.get('window').height;
        const padding = 100; // header + spacing
        const visibleBottom = screenH - keyboardSpace - padding;
        const overlap = y + h - visibleBottom;
        if (overlap > 0 && scrollRef.current && typeof scrollRef.current.scrollTo === 'function') {
          scrollRef.current.scrollTo({ y: scrollYRef.current + overlap + 12, animated: true });
        }
      });
    }, 250);
  };

  // Load persisted Step 1 and Step 2
  React.useEffect(() => {
    (async () => {
      try {
        const savedRows = await AsyncStorage.getItem('riskMgmt_step2_rows');
        if (savedRows) {
          const parsed = JSON.parse(savedRows);
          if (Array.isArray(parsed) && parsed.length > 0) setRows(parsed);
        }
        // If linked to a report, prefer Firebase data when available
        if (linkedReport && linkedReport.adminId && linkedReport.id) {
          try {
            const ref = database().ref(`/GlobalRiskAssessments/admins/${linkedReport.adminId}/${linkedReport.id}`);
            const snap = await ref.once('value');
            const data = snap.val();
            if (data?.step2?.rows && Array.isArray(data.step2.rows)) {
              setRows(data.step2.rows);
              await AsyncStorage.setItem('riskMgmt_step2_rows', JSON.stringify(data.step2.rows));
            }
          } catch (e) {
            // ignore fetch error
          }
        }
      } catch (e) {}
    })();
  }, []);

  const handleRowInputChange = (rowIndex, fieldKey, value) => {
    setRows((prev) => {
      const next = [...prev];
      next[rowIndex] = { ...next[rowIndex], [fieldKey]: value };
      AsyncStorage.setItem('riskMgmt_step2_rows', JSON.stringify(next)).catch(() => {});
      return next;
    });
  };

  const handleAddRow = () => {
    setRows((prev) => {
      if (prev.length >= MAX_ROWS) return prev;
      const next = [...prev, createEmptyRow()];
      AsyncStorage.setItem('riskMgmt_step2_rows', JSON.stringify(next)).catch(() => {});
      return next;
    });
  };

  const handleRemoveRow = (rowIndex) => {
    setRows((prev) => {
      const next = prev.filter((_, idx) => idx !== rowIndex);
      AsyncStorage.setItem('riskMgmt_step2_rows', JSON.stringify(next)).catch(() => {});
      return next;
    });
  };

  const handleSubmit = async () => {
    try {
      await AsyncStorage.setItem('riskMgmt_step2_rows', JSON.stringify(rows));
      // Ensure meta for Firebase path /GlobalRiskAssessments/admins/{uid}/{inspectionId}
      if (!linkedReport) {
        try {
          const userUid = await AsyncStorage.getItem('userUid');
          const step1Raw = await AsyncStorage.getItem('inspectionStep1');
          const step1 = step1Raw ? JSON.parse(step1Raw) : null;
          if (userUid && step1?.id) {
            await AsyncStorage.setItem('riskMgmt_meta', JSON.stringify({ reportId: step1.id, adminId: userUid }));
          }
        } catch (e) {
          // ignore
        }
      } else {
        try {
          await AsyncStorage.setItem('riskMgmt_meta', JSON.stringify({ reportId: linkedReport.id, adminId: linkedReport.adminId }));
        } catch (e) {}
      }
      navigation.navigate('MiniRiskAssessment_step3', { rows, report: linkedReport || null });
    } catch (e) {
      navigation.navigate('MiniRiskAssessment_step3', { rows, report: linkedReport || null });
    }
  };

  const handleOk = () => {
    setSuccessModalVisible(false);
    navigation.navigate('Inspection_Report_Home');
  };

  React.useEffect(() => {
    const showEvt = Platform.OS === 'ios' ? 'keyboardWillShow' : 'keyboardDidShow';
    const hideEvt = Platform.OS === 'ios' ? 'keyboardWillHide' : 'keyboardDidHide';
    const onShow = (e) => setKeyboardSpace(e?.endCoordinates?.height || 0);
    const onHide = () => setKeyboardSpace(0);
    const s = Keyboard.addListener(showEvt, onShow);
    const h = Keyboard.addListener(hideEvt, onHide);
    return () => {
      s.remove();
      h.remove();
    };
  }, []);

  return (
    <View style={styles.container}>
      {/* Risk Management Header */}
      <RiskManagementHeader 
        currentStep={currentStep} 
        setCurrentStep={setCurrentStep} 
      />
      {/* Heading above Mini Risk Assessment */}
      <Text style={styles.pageHeading}>MINI RISK ASSESSMENT</Text>
      <KeyboardAvoidingView
        style={{ flex: 1 }}
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        keyboardVerticalOffset={Platform.OS === 'ios' ? 64 : 0}
        enabled
      >
        <ScrollView
          ref={scrollRef}
          style={styles.scrollContent}
          contentContainerStyle={{ paddingBottom: 140 + keyboardSpace }}
          keyboardShouldPersistTaps="handled"
          keyboardDismissMode="on-drag"
          nestedScrollEnabled
          showsVerticalScrollIndicator={true}
          onScroll={(e) => { scrollYRef.current = e.nativeEvent.contentOffset.y; }}
          scrollEventThrottle={16}
        >
          <View style={styles.content}>
            {/* Guidance collapsible */}
            <View style={styles.infoCard}>
              <TouchableOpacity style={styles.infoHeader} onPress={() => setIsGuidanceOpen((p) => !p)}>
                <Text style={styles.infoTitle}>Guidance</Text>
                <MaterialIcons name={isGuidanceOpen ? 'expand-less' : 'expand-more'} size={22} color="#015185" />
              </TouchableOpacity>
              {isGuidanceOpen && (
                <View style={styles.infoBody}>
                  {GUIDANCE_POINTS.map((txt, i) => (
                    <View key={`g-${i}`} style={styles.bulletRow}>
                      <Text style={styles.bulletNum}>{i + 1}.</Text>
                      <Text style={styles.bulletText}>{txt}</Text>
                    </View>
                  ))}
                </View>
              )}
            </View>

            {rows.map((row, rowIndex) => (
              <View style={styles.rowCard} key={`row-${rowIndex}`}>
                <View style={styles.rowHeader}>
                  <Text style={styles.rowTitle}>Row {rowIndex + 1}</Text>
                  <TouchableOpacity onPress={() => handleRemoveRow(rowIndex)} disabled={rows.length === 1}>
                    <MaterialIcons name="delete-outline" size={20} color={rows.length === 1 ? '#b0b0b0' : '#b00020'} />
                  </TouchableOpacity>
                </View>
                <View style={styles.rowBody}>
                  <Text style={styles.fieldLabel}>{MINI_RISK_ITEMS[0]}</Text>
                  <TextInput
                    ref={(r) => { inputRefs.current[`row-${rowIndex}-step`] = r; }}
                    style={styles.singleInput}
                    value={row.step}
                    onChangeText={(text) => handleRowInputChange(rowIndex, 'step', text)}
                    placeholder="Enter step"
                    placeholderTextColor="#8fb7d8"
                    underlineColorAndroid="transparent"
                    returnKeyType="next"
                    onFocus={() => scrollIntoView(`row-${rowIndex}-step`)}
                  />

                  <Text style={styles.fieldLabel}>{MINI_RISK_ITEMS[1]}</Text>
                  <TextInput
                    ref={(r) => { inputRefs.current[`row-${rowIndex}-hazard`] = r; }}
                    style={styles.singleInput}
                    value={row.hazard}
                    onChangeText={(text) => handleRowInputChange(rowIndex, 'hazard', text)}
                    placeholder="Enter hazard"
                    placeholderTextColor="#8fb7d8"
                    underlineColorAndroid="transparent"
                    returnKeyType="next"
                    onFocus={() => scrollIntoView(`row-${rowIndex}-hazard`)}
                  />

                  <Text style={styles.fieldLabel}>{MINI_RISK_ITEMS[2]}</Text>
                  <TextInput
                    ref={(r) => { inputRefs.current[`row-${rowIndex}-assess`] = r; }}
                    style={styles.singleInput}
                    value={row.assess}
                    onChangeText={(text) => handleRowInputChange(rowIndex, 'assess', text)}
                    placeholder="Enter risk assessment"
                    placeholderTextColor="#8fb7d8"
                    underlineColorAndroid="transparent"
                    returnKeyType="next"
                    onFocus={() => scrollIntoView(`row-${rowIndex}-assess`)}
                  />

                  <Text style={styles.fieldLabel}>{MINI_RISK_ITEMS[3]}</Text>
                  <TextInput
                    ref={(r) => { inputRefs.current[`row-${rowIndex}-control`] = r; }}
                    style={styles.singleInput}
                    value={row.control}
                    onChangeText={(text) => handleRowInputChange(rowIndex, 'control', text)}
                    placeholder="Enter control steps"
                    placeholderTextColor="#8fb7d8"
                    underlineColorAndroid="transparent"
                    returnKeyType="done"
                    onFocus={() => scrollIntoView(`row-${rowIndex}-control`)}
                  />
                </View>
              </View>
            ))}

            <TouchableOpacity
              style={[styles.addRowButton, rows.length >= MAX_ROWS && styles.addRowButtonDisabled]}
              onPress={handleAddRow}
              disabled={rows.length >= MAX_ROWS}
            >
              <MaterialIcons name="add-circle-outline" size={20} color="#015185" />
              <Text style={styles.addRowText}>Add Row ({rows.length}/{MAX_ROWS})</Text>
            </TouchableOpacity>
          </View>
        </ScrollView>
      </KeyboardAvoidingView>
      {/* Bottom Buttons */}
      <View style={[styles.buttonContainer, { bottom: keyboardSpace }]} pointerEvents="box-none">
        <TouchableOpacity style={styles.cancelButton} onPress={() => navigation.goBack()}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleSubmit}>
          <Text style={styles.nextButtonText}>Next</Text>
          <MaterialIcons style={styles.bottomicon} name="check" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      {/* Success Modal */}
      <Modal visible={successModalVisible} animationType="fade" transparent={true}>
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Success</Text>
            <Text style={styles.modalText}>Mini Risk Assessment submitted successfully!</Text>
            <TouchableOpacity style={styles.okButton} onPress={handleOk}>
              <Text style={styles.okButtonText}>OK</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  pageHeading: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'left',
    marginTop: 10,
    marginBottom: 0,
    marginLeft: 20,
  },
  scrollContent: {
    flex: 1,
    backgroundColor: '#fff',
  },
  content: {
    flex: 1,
    paddingTop: 10,
  },
  card: {
    backgroundColor: '#e0e0e0',
    borderRadius: 16,
    flexDirection: 'column',
    alignItems: 'flex-start',
    padding: 16,
    marginHorizontal: 16,
    marginBottom: 18,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 2,
    width: 'auto',
  },
  rowCard: {
    backgroundColor: '#e0e0e0',
    borderRadius: 16,
    padding: 16,
    marginHorizontal: 16,
    marginBottom: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 2,
  },
  rowHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  rowTitle: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '700',
  },
  rowBody: {
    backgroundColor: '#ffffff',
    borderRadius: 12,
    borderWidth: 1,
    borderColor: '#cfe2f3',
    padding: 12,
  },
  fieldLabel: {
    color: '#3481BC',
    fontSize: 13,
    fontWeight: '600',
    marginTop: 6,
    marginBottom: 6,
  },
  cardTitle: {
    color: '#3481BC',
    fontSize: 16,
    fontWeight: '600',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#3481BC',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 10,
    color: '#3481BC',
    fontWeight: '600',
    fontSize: 15,
    backgroundColor: '#fff',
    width: width - 64,
    alignSelf: 'center',
  },
  singleInput: {
    borderWidth: 1,
    borderColor: '#3481BC',
    borderRadius: 8,
    paddingHorizontal: 10,
    paddingVertical: 8,
    color: '#015185',
    fontSize: 14,
    backgroundColor: '#fff',
  },
  addRowButton: {
    alignSelf: 'center',
    flexDirection: 'row',
    alignItems: 'center',
    gap: 6,
    paddingVertical: 10,
    paddingHorizontal: 14,
    borderRadius: 10,
    borderWidth: 1,
    borderColor: '#015185',
    marginTop: 4,
  },
  addRowButtonDisabled: {
    opacity: 0.5,
  },
  addRowText: {
    color: '#015185',
    fontWeight: '700',
  },
  infoCard: {
    backgroundColor: '#e0e0e0',
    borderRadius: 16,
    marginHorizontal: 16,
    marginBottom: 12,
    overflow: 'hidden',
  },
  infoHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 16,
    paddingVertical: 12,
    backgroundColor: '#ffffff',
    borderBottomWidth: 1,
    borderBottomColor: '#cfe2f3',
  },
  infoTitle: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '700',
  },
  infoBody: {
    backgroundColor: '#ffffff',
    paddingHorizontal: 16,
    paddingVertical: 12,
  },
  bulletRow: {
    flexDirection: 'row',
    marginBottom: 8,
  },
  bulletNum: {
    color: '#015185',
    fontWeight: 'bold',
    marginRight: 6,
  },
  bulletText: {
    color: '#015185',
    flex: 1,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
    marginRight: 6,
  },
  bottomicon: {
    marginTop: 2,
  },
  // Success Modal styles
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    paddingHorizontal: 30,
    paddingVertical: 18,
    width: '80%',
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
  },
  modalText: {
    fontSize: 16,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  okButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 30,
    borderRadius: 5,
    marginTop: 10,
  },
  okButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  successIcon: {
    alignSelf: 'center',
    marginBottom: 10,
  },
});

export default Risk_step2;      