/**
 * OfflineDataHelper.js
 * Helper module for managing offline data for Inspection Reports
 * Loads equipment and checklists from local JSON files when offline
 */

import AsyncStorage from '@react-native-async-storage/async-storage';
import equipmentData from './Offlinedata/titan_drilling.equipment.json';
import checklistsData from './Offlinedata/titan_drilling.inspectionchecklists.json';

class OfflineDataHelper {
  /**
   * Get equipment data - either from cache or local JSON
   * @returns {Promise<Array>} Array of equipment objects
   */
  static async getEquipmentData() {
    try {
      // Try to load from AsyncStorage cache first
      const cachedEquipment = await AsyncStorage.getItem('equipment');
      
      if (cachedEquipment) {
        const parsed = JSON.parse(cachedEquipment);
        if (parsed && parsed.length > 0) {
          console.log('📦 Loaded equipment from AsyncStorage cache:', parsed.length);
          return parsed;
        }
      }
      
      // If no cache, load from local JSON file
      console.log('📂 Loading equipment from local JSON file...');
      const processedEquipment = equipmentData.map(item => ({
        ...item,
        id: item.id || item._id?.$oid || item._id,
        equipmentName: item.equipmentName || item.vehicleNumber || 'Unknown Equipment',
        model: item.model || 'N/A',
        status: item.status || 'Available',
        country: item.country || 'N/A',
        meterReading: item.meterReading || 0,
        meterUnit: item.meterUnit || 'KM',
        mainCategory: item.mainCategory || item.category || 'N/A',
        equipmentCategory: item.equipmentCategory || item.category || 'N/A',
        vehicleNumber: item.vehicleNumber || item.equipmentNumber || 'N/A',
        registrationNumber: item.registrationNumber || 'N/A',
        createdAt: item.createdAt?.$date || new Date().toISOString(),
        updatedAt: item.updatedAt?.$date || new Date().toISOString(),
      }));
      
      // Cache to AsyncStorage for future use
      await AsyncStorage.setItem('equipment', JSON.stringify(processedEquipment));
      
      console.log('✅ Loaded and cached equipment from JSON:', processedEquipment.length);
      return processedEquipment;
    } catch (error) {
      console.error('❌ Error loading equipment data:', error);
      return [];
    }
  }

  /**
   * Get checklists data - either from cache or local JSON
   * @returns {Promise<Array>} Array of checklist objects
   */
  static async getChecklistsData() {
    try {
      // Try to load from AsyncStorage cache first
      const cachedChecklists = await AsyncStorage.getItem('checklists');
      
      if (cachedChecklists) {
        const parsed = JSON.parse(cachedChecklists);
        if (parsed && parsed.length > 0) {
          console.log('📦 Loaded checklists from AsyncStorage cache:', parsed.length);
          return parsed;
        }
      }
      
      // If no cache, load from local JSON file
      console.log('📂 Loading checklists from local JSON file...');
      const processedChecklists = checklistsData.map(item => {
        // Convert checklist structure to app format
        const items = Object.entries(item.checklist || {}).map(([sectionKey, sectionItems]) => ({
          text: sectionKey,
          subItems: Array.isArray(sectionItems)
            ? sectionItems.map((subItem, index) => ({
                itemName: typeof subItem === 'string' ? subItem : subItem.itemName || 'Unnamed Item',
                inputType: 'Ok/Reject/N/A',
                description: '',
                instruction: '',
                sortOrder: index + 1,
                isInspectionRequired: false,
              }))
            : [],
        }));

        return {
          id: item._id || item.documentId,
          title: item.documentName || item.headerTitle || 'Untitled Checklist',
          description: '',
          items,
          status: 'Available',
          isUsed: false,
          isUserChecklist: false,
          version: item.version || '',
          version_date: item.versionDate || '',
          approved_by: item.approvedBy || '',
          document_id: item.documentId || item._id,
          document_name: item.documentName || '',
          Header_title: item.headerTitle || '',
          header_inputs: item.header_inputs || {},
          footer_inputs: item.footer_inputs || {},
          checklist: item.checklist || {},
          createdAt: item.uploadedAt?.$date || new Date().toISOString(),
        };
      });
      
      // Cache to AsyncStorage for future use
      await AsyncStorage.setItem('checklists', JSON.stringify(processedChecklists));
      
      console.log('✅ Loaded and cached checklists from JSON:', processedChecklists.length);
      return processedChecklists;
    } catch (error) {
      console.error('❌ Error loading checklists data:', error);
      return [];
    }
  }

  /**
   * Save inspection report locally for offline sync
   * @param {Object} report - Complete inspection report data
   * @returns {Promise<boolean>} Success status
   */
  static async saveOfflineReport(report) {
    try {
      // Get existing offline reports
      const offlineReportsRaw = await AsyncStorage.getItem('offlineReports');
      const offlineReports = offlineReportsRaw ? JSON.parse(offlineReportsRaw) : [];
      
      // Add new report with offline flag
      const offlineReport = {
        ...report,
        offlineCreated: true,
        offlineSyncPending: true,
        offlineCreatedAt: new Date().toISOString(),
      };
      
      offlineReports.push(offlineReport);
      
      // Save to AsyncStorage
      await AsyncStorage.setItem('offlineReports', JSON.stringify(offlineReports));
      
      console.log('✅ Saved offline report:', report._id || report.id);
      console.log('📊 Total offline reports:', offlineReports.length);
      
      return true;
    } catch (error) {
      console.error('❌ Error saving offline report:', error);
      return false;
    }
  }

  /**
   * Get all offline reports waiting to be synced
   * @returns {Promise<Array>} Array of offline reports
   */
  static async getOfflineReports() {
    try {
      const offlineReportsRaw = await AsyncStorage.getItem('offlineReports');
      const offlineReports = offlineReportsRaw ? JSON.parse(offlineReportsRaw) : [];
      return offlineReports;
    } catch (error) {
      console.error('❌ Error getting offline reports:', error.message);
      return [];
    }
  }

  /**
   * Get all reports (both online cached and offline)
   * @returns {Promise<Array>} Array of all reports
   */
  static async getAllReports() {
    try {
      // Get cached online reports
      const cachedReportsRaw = await AsyncStorage.getItem('inspectionReports');
      const cachedReports = cachedReportsRaw ? JSON.parse(cachedReportsRaw) : [];
      
      // Get offline reports
      const offlineReports = await this.getOfflineReports();
      
      // Combine both arrays
      const allReports = [...cachedReports, ...offlineReports];
      
      return allReports;
    } catch (error) {
      console.error('❌ Error getting all reports:', error.message);
      return [];
    }
  }

  /**
   * Sync offline reports to MongoDB
   * @param {Function} uploadCallback - Function to upload report to MongoDB
   * @returns {Promise<Object>} Sync results
   */
  static async syncOfflineReports(uploadCallback) {
    try {
      const offlineReports = await this.getOfflineReports();
      
      if (offlineReports.length === 0) {
        console.log('✅ No offline reports to sync');
        return { success: true, synced: 0, failed: 0 };
      }
      
      console.log('🔄 Starting sync of', offlineReports.length, 'offline reports...');
      
      let synced = 0;
      let failed = 0;
      const failedReports = [];
      
      for (const report of offlineReports) {
        try {
          // Call the upload callback function
          const result = await uploadCallback(report);
          
          if (result && result.success) {
            synced++;
            console.log('✅ Synced report:', report._id || report.id);
          } else {
            failed++;
            failedReports.push(report);
            console.error('❌ Failed to sync report:', report._id || report.id);
          }
        } catch (error) {
          failed++;
          failedReports.push(report);
          console.error('❌ Error syncing report:', report._id || report.id, error);
        }
      }
      
      // Update offline reports (keep only failed ones)
      await AsyncStorage.setItem('offlineReports', JSON.stringify(failedReports));
      
      console.log('✅ Sync complete - Synced:', synced, 'Failed:', failed);
      
      return {
        success: true,
        synced,
        failed,
        total: offlineReports.length,
      };
    } catch (error) {
      console.error('❌ Error syncing offline reports:', error);
      return {
        success: false,
        synced: 0,
        failed: 0,
        error: error.message,
      };
    }
  }

  /**
   * Clear all offline data (use with caution)
   */
  static async clearOfflineData() {
    try {
      await AsyncStorage.removeItem('offlineReports');
      console.log('✅ Cleared offline reports');
      return true;
    } catch (error) {
      console.error('❌ Error clearing offline data:', error);
      return false;
    }
  }

  /**
   * Save PDF locally for offline access
   * @param {string} reportId - Report ID
   * @param {string} pdfPath - Local PDF file path
   * @returns {Promise<boolean>} Success status
   */
  static async saveOfflinePDF(reportId, pdfPath) {
    try {
      const offlinePDFsRaw = await AsyncStorage.getItem('offlinePDFs');
      const offlinePDFs = offlinePDFsRaw ? JSON.parse(offlinePDFsRaw) : {};
      
      offlinePDFs[reportId] = {
        pdfPath,
        savedAt: new Date().toISOString(),
      };
      
      await AsyncStorage.setItem('offlinePDFs', JSON.stringify(offlinePDFs));
      
      console.log('✅ Saved offline PDF for report:', reportId);
      return true;
    } catch (error) {
      console.error('❌ Error saving offline PDF:', error);
      return false;
    }
  }

  /**
   * Get offline PDF path for a report
   * @param {string} reportId - Report ID
   * @returns {Promise<string|null>} PDF path or null
   */
  static async getOfflinePDF(reportId) {
    try {
      const offlinePDFsRaw = await AsyncStorage.getItem('offlinePDFs');
      const offlinePDFs = offlinePDFsRaw ? JSON.parse(offlinePDFsRaw) : {};
      
      const pdfData = offlinePDFs[reportId];
      return pdfData ? pdfData.pdfPath : null;
    } catch (error) {
      console.error('❌ Error getting offline PDF:', error);
      return null;
    }
  }
}

export default OfflineDataHelper;

