import AsyncStorage from '@react-native-async-storage/async-storage';

/**
 * OfflineIncidentHelper - Manages offline incident report storage
 * Similar to OfflineDataHelper but for incident reports
 */
class OfflineIncidentHelper {
  /**
   * Save an incident report for offline access
   * @param {object} report - The incident report to save
   * @returns {Promise<boolean>} Success status
   */
  static async saveOfflineReport(report) {
    try {
      const offlineReportsRaw = await AsyncStorage.getItem('offlineIncidentReports');
      const offlineReports = offlineReportsRaw ? JSON.parse(offlineReportsRaw) : [];
      
      // Add timestamp and offline flag
      const offlineReport = {
        ...report,
        offlineCreated: true,
        offlineSyncPending: true,
        offlineCreatedAt: new Date().toISOString(),
      };
      
      // Check if report already exists (by ID)
      const existingIndex = offlineReports.findIndex(r => r.id === report.id || r.incidentNumber === report.incidentNumber);
      
      if (existingIndex >= 0) {
        // Update existing report
        offlineReports[existingIndex] = offlineReport;
        console.log('✅ Updated offline incident report:', report.incidentNumber);
      } else {
        // Add new report
        offlineReports.push(offlineReport);
        console.log('✅ Saved new offline incident report:', report.incidentNumber);
      }
      
      await AsyncStorage.setItem('offlineIncidentReports', JSON.stringify(offlineReports));
      
      return true;
    } catch (error) {
      console.error('❌ Error saving offline incident report:', error);
      return false;
    }
  }

  /**
   * Get all offline incident reports
   * @returns {Promise<Array>} Array of offline reports
   */
  static async getOfflineReports() {
    try {
      const offlineReportsRaw = await AsyncStorage.getItem('offlineIncidentReports');
      const offlineReports = offlineReportsRaw ? JSON.parse(offlineReportsRaw) : [];
      
      console.log(`📦 Retrieved ${offlineReports.length} offline incident reports`);
      
      return offlineReports;
    } catch (error) {
      console.error('❌ Error getting offline incident reports:', error);
      return [];
    }
  }

  /**
   * Sync offline incident reports to MongoDB
   * @param {Function} uploadCallback - Function to upload each report
   * @returns {Promise<object>} Sync results
   */
  static async syncOfflineReports(uploadCallback) {
    try {
      const offlineReports = await this.getOfflineReports();
      
      if (offlineReports.length === 0) {
        console.log('✅ No offline incident reports to sync');
        return { success: true, synced: 0, failed: 0 };
      }
      
      console.log('🔄 Starting sync of', offlineReports.length, 'offline incident reports...');
      
      let synced = 0;
      let failed = 0;
      const failedReports = [];
      
      for (const report of offlineReports) {
        try {
          // Call the upload callback function
          const result = await uploadCallback(report);
          
          if (result && result.success) {
            synced++;
            console.log('✅ Synced incident report:', report.incidentNumber);
          } else {
            failed++;
            failedReports.push(report);
            console.error('❌ Failed to sync incident report:', report.incidentNumber);
          }
        } catch (error) {
          failed++;
          failedReports.push(report);
          console.error('❌ Error syncing incident report:', report.incidentNumber, error);
        }
      }
      
      // Update offline reports (keep only failed ones)
      await AsyncStorage.setItem('offlineIncidentReports', JSON.stringify(failedReports));
      
      console.log('✅ Sync complete - Synced:', synced, 'Failed:', failed);
      
      return {
        success: true,
        synced,
        failed,
        total: offlineReports.length,
      };
    } catch (error) {
      console.error('❌ Error syncing offline incident reports:', error);
      return {
        success: false,
        synced: 0,
        failed: 0,
        error: error.message,
      };
    }
  }

  /**
   * Clear all offline incident data (use with caution)
   */
  static async clearOfflineData() {
    try {
      await AsyncStorage.removeItem('offlineIncidentReports');
      console.log('✅ Cleared offline incident reports');
      return true;
    } catch (error) {
      console.error('❌ Error clearing offline incident data:', error);
      return false;
    }
  }

  /**
   * Save PDF locally for offline access
   * @param {string} reportId - Report ID
   * @param {string} pdfPath - Local PDF file path
   * @returns {Promise<boolean>} Success status
   */
  static async saveOfflinePDF(reportId, pdfPath) {
    try {
      const offlinePDFsRaw = await AsyncStorage.getItem('offlineIncidentPDFs');
      const offlinePDFs = offlinePDFsRaw ? JSON.parse(offlinePDFsRaw) : {};
      
      offlinePDFs[reportId] = {
        pdfPath,
        savedAt: new Date().toISOString(),
      };
      
      await AsyncStorage.setItem('offlineIncidentPDFs', JSON.stringify(offlinePDFs));
      
      console.log('✅ Saved offline PDF for incident report:', reportId);
      return true;
    } catch (error) {
      console.error('❌ Error saving offline PDF:', error);
      return false;
    }
  }

  /**
   * Get offline PDF path for a report
   * @param {string} reportId - Report ID
   * @returns {Promise<string|null>} PDF path or null
   */
  static async getOfflinePDF(reportId) {
    try {
      const offlinePDFsRaw = await AsyncStorage.getItem('offlineIncidentPDFs');
      const offlinePDFs = offlinePDFsRaw ? JSON.parse(offlinePDFsRaw) : {};
      
      return offlinePDFs[reportId]?.pdfPath || null;
    } catch (error) {
      console.error('❌ Error getting offline PDF:', error);
      return null;
    }
  }

  /**
   * Delete offline PDF
   * @param {string} reportId - Report ID
   * @returns {Promise<boolean>} Success status
   */
  static async deleteOfflinePDF(reportId) {
    try {
      const offlinePDFsRaw = await AsyncStorage.getItem('offlineIncidentPDFs');
      const offlinePDFs = offlinePDFsRaw ? JSON.parse(offlinePDFsRaw) : {};
      
      delete offlinePDFs[reportId];
      
      await AsyncStorage.setItem('offlineIncidentPDFs', JSON.stringify(offlinePDFs));
      
      console.log('✅ Deleted offline PDF for incident report:', reportId);
      return true;
    } catch (error) {
      console.error('❌ Error deleting offline PDF:', error);
      return false;
    }
  }
}

export default OfflineIncidentHelper;

