import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, StatusBar, Dimensions, ScrollView, Modal, ActivityIndicator } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

const { width } = Dimensions.get('window');
const HEADER_HEIGHT = width * 0.32;

const SupervisorReporting = () => {
  const navigation = useNavigation();
  const [teamMembers, setTeamMembers] = useState([]);
  const [totalInspections, setTotalInspections] = useState(0);
  const [supervisorCount, setSupervisorCount] = useState(0);
  const [operatorCount, setOperatorCount] = useState(0);
  const [filterRole, setFilterRole] = useState('All'); // Filter state for roles
  const [modalVisible, setModalVisible] = useState(false);
  const [modalLoading, setModalLoading] = useState(false);
  const [modalError, setModalError] = useState(null);
  const [userDetails, setUserDetails] = useState(null);
  const [allowedCountries, setAllowedCountries] = useState([]);
  const [allowedProjects, setAllowedProjects] = useState([]);
  const [teamMemberPermissions, setTeamMemberPermissions] = useState([]);
  const [activeTable, setActiveTable] = useState('all'); // 'all', 'supervisor', 'operator'

  // Helper function to show toast messages
  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Fetch user permissions (country, project, permissions) on mount
  useEffect(() => {
    const fetchUserPermissions = async () => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) return;
        const userRef = database().ref(`/Globalusers/admins/${userUid}`);
        const snapshot = await userRef.once('value');
        const data = snapshot.val();
        if (data) {
          setAllowedCountries(Array.isArray(data.countries) ? data.countries : (data.countries ? [data.countries] : []));
          setAllowedProjects(Array.isArray(data.projects) ? data.projects : (data.projects ? [data.projects] : []));
          let perms = [];
          if (Array.isArray(data.modules)) {
            const teamModule = data.modules.find(m => m.module === 'teamMember');
            if (teamModule && teamModule.permissions) {
              perms = teamModule.permissions;
            }
          }
          setTeamMemberPermissions(perms);
        }
      } catch (e) {}
    };
    fetchUserPermissions();
  }, []);

  // Fetch team members from Firebase or AsyncStorage
  useEffect(() => {
    const fetchTeamMembers = async () => {
      try {
        const teamMemberId = await AsyncStorage.getItem('teamMemberId');
        if (!teamMemberId) {
          showToast('error', 'Error', 'Team Member ID not found in storage.');
          return;
        }

        const state = await NetInfo.fetch();
        if (!state.isConnected) {
          // Offline mode: Fetch from AsyncStorage
          const storedTeam = await AsyncStorage.getItem('teamMembers');
          if (storedTeam) {
            const members = JSON.parse(storedTeam);
            processTeamMembers(members);
            showToast('info', 'Offline Mode', 'Displaying local data.');
          } else {
            showToast('error', 'Offline Error', 'No local data available.');
          }
          return;
        }

        // Online mode: Fetch from Firebase
        const teamMembersRef = database().ref(`/GlobalTeamMemberInformation`);
        teamMembersRef.on('value', (snapshot) => {
          const data = snapshot.val();
          if (data) {
            const membersArray = Object.keys(data).map((memberId) => ({
              id: memberId,
              ...data[memberId],
            }));
            processTeamMembers(membersArray);
            // Update AsyncStorage for offline use
            AsyncStorage.setItem('teamMembers', JSON.stringify(membersArray)).catch((err) =>
              console.error('Error saving to AsyncStorage:', err)
            );
          } else {
            setTeamMembers([]);
            setTotalInspections(0);
            setSupervisorCount(0);
            setOperatorCount(0);
            showToast('info', 'Info', 'No team members found.');
          }
        });

        // Cleanup listener on unmount
        return () => teamMembersRef.off('value');
      } catch (error) {
        console.error('Error fetching team members:', error.message, error.stack);
        showToast('error', 'Error', 'Failed to load team members.');
      }
    };

    fetchTeamMembers();
  }, []);

  // Filter by allowed country/project
  const filterByCountryProject = (member) => {
    const countryMatch = allowedCountries.length === 0 || (member.countries && member.countries.some(c => allowedCountries.includes(c)));
    const projectMatch = allowedProjects.length === 0 || (member.projects && member.projects.some(p => allowedProjects.includes(p)));
    return countryMatch && projectMatch;
  };

  // Process team members data
  const processTeamMembers = (members) => {
    const filtered = members.filter(filterByCountryProject);
    let total = 0;
    let supervisors = 0;
    let operators = 0;
    const membersArray = [];
    filtered.forEach((memberData) => {
      const inspectionCount = memberData.inspections || 0;
      const role = memberData.role || 'Employee';
      const name = memberData.fullName || memberData.name || 'Unknown';
      membersArray.push({
        id: memberData.id,
        name,
        role,
        inspections: inspectionCount,
      });
      total += inspectionCount;
      if (role === 'Supervisor') supervisors++;
      // Operator: not Supervisor, not Admin, not Manager
      if (!['Supervisor', 'Admin', 'Manager'].includes(role)) operators++;
    });
    membersArray.sort((a, b) => a.name.localeCompare(b.name));
    setTeamMembers(membersArray);
    setTotalInspections(total);
    setSupervisorCount(supervisors);
    setOperatorCount(operators);
  };

  // Modal for permissions
  const handleLockPress = async () => {
    setModalVisible(true);
    setModalLoading(true);
    setModalError(null);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        setModalError('User ID not found. Please log in again.');
        setModalLoading(false);
        return;
      }
      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const snapshot = await userRef.once('value');
      const data = snapshot.val();
      if (!data) {
        setUserDetails(null);
        setModalError('User details not found.');
      } else {
        const country = Array.isArray(data.countries) && data.countries.length > 0
          ? data.countries.join(', ')
          : (data.selectedCountry || data.country || 'N/A');
        const project = Array.isArray(data.projects) && data.projects.length > 0
          ? data.projects.join(', ')
          : (data.selectedProject || data.project || 'N/A');
        let perms = [];
        if (Array.isArray(data.modules)) {
          const teamModule = data.modules.find(m => m.module === 'teamMember');
          if (teamModule && teamModule.permissions) {
            perms = teamModule.permissions;
          }
        }
        setUserDetails({ country, project, teamMemberPermissions: perms });
      }
    } catch (err) {
      setModalError('Failed to fetch user details.');
      setUserDetails(null);
    }
    setModalLoading(false);
  };

  // Clickable summary logic
  const displayedMembers = activeTable === 'supervisor'
    ? teamMembers.filter(m => m.role === 'Supervisor')
    : activeTable === 'operator'
      ? teamMembers.filter(m => !['Supervisor', 'Admin', 'Manager'].includes(m.role))
      : teamMembers;

  // Handle role filter change
  const handleFilterChange = (role) => {
    setFilterRole(role);
  };

  // Filter team members based on selected role
  const filteredTeamMembers = filterRole === 'All'
    ? teamMembers
    : teamMembers.filter((member) => member.role === filterRole);

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={styles.headerLeftIcon}>
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Inspections Completed by Team Members</Text>
          <View style={styles.headerRightIcons}>
            <TouchableOpacity onPress={() => alert('Cloud sync pressed')} style={styles.headerIconButton}>
              <MaterialIcons name="cloud" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity onPress={handleLockPress} style={styles.headerIconButton} accessibilityLabel="Show user details" accessibilityRole="button">
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>
      {/* Modal for user details */}
      <Modal
        visible={modalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setModalVisible(false)}
      >
        <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' }}>
          <View style={{
            width: '80%',
            backgroundColor: '#fff',
            borderRadius: 16,
            padding: 24,
            alignItems: 'center',
            shadowColor: '#000',
            shadowOffset: { width: 0, height: 2 },
            shadowOpacity: 0.2,
            shadowRadius: 8,
            elevation: 5,
          }}>
            <Text style={{ fontSize: 20, fontWeight: 'bold', marginBottom: 16, color: '#015185' }}>User Details</Text>
            {modalLoading ? (
              <ActivityIndicator size="large" color="#015185" />
            ) : modalError ? (
              <Text style={{ color: '#FF6B6B', fontSize: 16, marginBottom: 12, textAlign: 'center' }}>{modalError}</Text>
            ) : userDetails ? (
              <View style={{ marginBottom: 16, alignSelf: 'stretch' }}>
                <Text style={{ fontSize: 16, marginBottom: 8, color: '#222' }}><Text style={{ fontWeight: 'bold', color: '#015185' }}>Country:</Text> {userDetails.country}</Text>
                <Text style={{ fontSize: 16, marginBottom: 8, color: '#222' }}><Text style={{ fontWeight: 'bold', color: '#015185' }}>Project:</Text> {userDetails.project}</Text>
                <Text style={{ fontSize: 16, marginBottom: 8, color: '#222' }}><Text style={{ fontWeight: 'bold', color: '#015185' }}>Team Member Permissions:</Text> {userDetails.teamMemberPermissions?.join(', ') || 'None'}</Text>
              </View>
            ) : null}
            <TouchableOpacity
              style={{
                marginTop: 8,
                backgroundColor: '#015185',
                borderRadius: 8,
                paddingVertical: 8,
                paddingHorizontal: 24,
              }}
              onPress={() => setModalVisible(false)}
              accessibilityLabel="Close user details"
              accessibilityRole="button"
            >
              <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
      <ScrollView contentContainerStyle={styles.body}>
        {/* Role Filter */}
        <View style={styles.filterWrapper}>
          <Text style={styles.filterTitle}>Filter by Role:</Text>
          <View style={styles.filterButtons}>
            {['All', 'Supervisor', 'Operator', 'Employee'].map((role) => (
              <TouchableOpacity
                key={role}
                style={[styles.filterButton, filterRole === role && styles.activeFilterButton]}
                onPress={() => setFilterRole(role)}
              >
                <Text style={[styles.filterButtonText, filterRole === role && styles.activeFilterButtonText]}>
                  {role}
                </Text>
              </TouchableOpacity>
            ))}
          </View>
        </View>
        {/* Overall Summary */}
        <View style={styles.summaryWrapper}>
          <Text style={styles.summaryTitle}>Overall Summary</Text>
          <TouchableOpacity onPress={() => setActiveTable('all')}>
            <Text style={styles.summaryText}>
              Total Members: <Text style={styles.summaryValue}>{teamMembers.length}</Text>
            </Text>
          </TouchableOpacity>
          <TouchableOpacity onPress={() => setActiveTable('supervisor')}>
            <Text style={styles.summaryText}>
              Total Supervisors: <Text style={styles.summaryValue}>{teamMembers.filter(m => m.role === 'Supervisor').length}</Text>
            </Text>
          </TouchableOpacity>
          <TouchableOpacity onPress={() => setActiveTable('operator')}>
            <Text style={styles.summaryText}>
              Total Operators: <Text style={styles.summaryValue}>{teamMembers.filter(m => !['Supervisor', 'Admin', 'Manager'].includes(m.role)).length}</Text>
            </Text>
          </TouchableOpacity>
          <Text style={styles.summaryText}>
            Total Inspections: <Text style={styles.summaryValue}>{totalInspections}</Text>
          </Text>
        </View>
        {/* Table of Team Members */}
        <View style={styles.tableWrapper}>
          <View style={styles.tableHeaderRow}>
            <Text style={[styles.tableHeaderCell, { flex: 1 }]}>S.No</Text>
            <Text style={[styles.tableHeaderCell, { flex: 2 }]}>Role</Text>
            <Text style={[styles.tableHeaderCell, { flex: 3 }]}>Name</Text>
            <Text style={[styles.tableHeaderCell, { flex: 3 }]}>Total Inspections</Text>
          </View>
          {displayedMembers.length > 0 ? (
            displayedMembers.map((member, idx) => (
              <View
                key={member.id}
                style={[styles.tableRow, idx % 2 === 0 ? styles.tableRowEven : styles.tableRowOdd]}
              >
                <Text style={[styles.tableCell, { flex: 1 }]}>{idx + 1}</Text>
                <Text
                  style={[
                    styles.tableCell,
                    { flex: 2 },
                    member.role === 'Supervisor'
                      ? styles.supervisorRole
                      : member.role === 'Operator'
                      ? styles.operatorRole
                      : styles.defaultRole,
                  ]}
                >
                  {member.role}
                </Text>
                <Text style={[styles.tableCell, { flex: 3, color: '#2267A2', fontWeight: '500' }]}> {member.name} </Text>
                <Text style={[styles.tableCell, { flex: 3, color: '#025383', fontWeight: '700' }]}> {member.inspections} </Text>
              </View>
            ))
          ) : (
            <View style={styles.emptyRow}>
              <Text style={styles.emptyText}>No team members found for this role.</Text>
            </View>
          )}
        </View>
      </ScrollView>
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'flex-end',
    paddingBottom: width * 0.04,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.05,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeftIcon: {
    paddingVertical: width * 0.02,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '700',
    flex: 1,
    marginLeft: width * 0.03,
  },
  headerRightIcons: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerIconButton: {
    marginLeft: width * 0.03,
    padding: width * 0.02,
  },
  body: {
    flexGrow: 1,
    alignItems: 'center',
    justifyContent: 'flex-start',
    paddingTop: width * 0.08,
    paddingBottom: width * 0.08,
    backgroundColor: '#fff',
  },
  filterWrapper: {
    width: '92%',
    marginBottom: width * 0.04,
    alignSelf: 'center',
  },
  filterTitle: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#025383',
    marginBottom: width * 0.02,
  },
  filterButtons: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: width * 0.02,
  },
  filterButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: width * 0.015,
    borderWidth: 1,
    borderColor: '#015185',
    backgroundColor: '#fff',
  },
  activeFilterButton: {
    backgroundColor: '#015185',
  },
  filterButtonText: {
    fontSize: width * 0.035,
    color: '#015185',
  },
  activeFilterButtonText: {
    color: '#fff',
  },
  summaryWrapper: {
    width: '92%',
    backgroundColor: '#f8f8f8',
    borderRadius: 6,
    padding: width * 0.04,
    marginBottom: width * 0.04,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    alignSelf: 'center',
  },
  summaryTitle: {
    fontSize: width * 0.045,
    fontWeight: '700',
    color: '#025383',
    marginBottom: width * 0.02,
  },
  summaryText: {
    fontSize: width * 0.04,
    color: '#444',
    marginBottom: width * 0.01,
  },
  summaryValue: {
    fontWeight: '700',
    color: '#2267A2',
  },
  tableWrapper: {
    width: '92%',
    backgroundColor: '#f8f8f8',
    borderRadius: 6,
    overflow: 'hidden',
    borderWidth: 1,
    borderColor: '#e0e0e0',
    alignSelf: 'center',
    marginTop: width * 0.02,
  },
  tableHeaderRow: {
    flexDirection: 'row',
    backgroundColor: '#e5e5e5',
    paddingVertical: width * 0.03,
    borderTopLeftRadius: 6,
    borderTopRightRadius: 6,
  },
  tableHeaderCell: {
    fontWeight: '700',
    color: '#444',
    fontSize: width * 0.04,
    textAlign: 'center',
  },
  tableRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.028,
    borderBottomWidth: 1,
    borderBottomColor: '#ececec',
  },
  tableRowEven: {
    backgroundColor: '#f8f8f8',
  },
  tableRowOdd: {
    backgroundColor: '#f1f1f1',
  },
  tableCell: {
    textAlign: 'center',
    fontSize: width * 0.038,
    color: '#333',
  },
  supervisorRole: {
    color: '#2267A2',
    fontWeight: '600',
  },
  operatorRole: {
    color: '#D81B60',
    fontWeight: '600',
  },
  defaultRole: {
    color: '#444',
    fontWeight: '500',
  },
  emptyRow: {
    padding: width * 0.04,
    alignItems: 'center',
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
  },
});

export default SupervisorReporting;