# HomeScreen MongoDB Sync Improvements

## Overview
This document outlines the comprehensive online/offline sync improvements implemented in `HomeScreen.js` for robust data synchronization with MongoDB and local storage fallback.

---

## Key Features Implemented

### 1. **Smart Online/Offline Sync**
- ✅ Automatic detection of network status
- ✅ Seamless switching between online and offline modes
- ✅ Auto-sync when connection is restored
- ✅ Local storage fallback when offline

### 2. **User Data Management**
- ✅ Fetch user data from MongoDB when online
- ✅ Store user data locally with timestamp
- ✅ Load from local storage when offline
- ✅ Modules loaded based on user permissions
- ✅ Countries loaded from user profile

### 3. **Visual Sync Status**
- ✅ Real-time online/offline indicator
- ✅ Last sync timestamp display
- ✅ Manual sync button
- ✅ Sync progress modal
- ✅ Color-coded status bar (Green = Online, Orange = Offline)

### 4. **Toast Notifications**
- ✅ Success: "Data synced successfully"
- ✅ Info: "Using locally stored data" (offline mode)
- ✅ Warning: "Connection lost - Switching to offline mode"
- ✅ Success: "Connection restored" (auto-sync triggered)
- ✅ Error: "Failed to load user data" / "Sync failed"

---

## How It Works

### Initial Load
```javascript
1. Check network status
2. Get user ID from AsyncStorage
3. If ONLINE:
   - Fetch from MongoDB
   - Store locally with timestamp
   - Show modules based on permissions
4. If OFFLINE:
   - Load from AsyncStorage
   - Show "Offline Mode" toast
   - Display cached modules
```

### When Connection Restored
```javascript
1. Network listener detects online status
2. Show "Connection restored" toast
3. Auto-trigger sync from MongoDB
4. Update local storage
5. Refresh modules and permissions
6. Show "Data synced successfully" toast
```

### Manual Sync
```javascript
1. User clicks "Sync" button
2. Check if online (disabled if offline)
3. Fetch latest data from MongoDB
4. Update local storage
5. Refresh UI
6. Show sync status toast
```

---

## Data Storage Structure

### AsyncStorage Keys
- `userUid` - User ID
- `userData` - Complete user object with timestamp
- `modules` - Array of module names
- `selectedCountries` - Array of countries
- `selectedCountry` - Single selected country
- `lastSyncTimestamp` - ISO timestamp of last sync

### User Data Object
```json
{
  "_id": "user123",
  "name": "John Doe",
  "email": "john@example.com",
  "role": "ProAdmin",
  "modules": [
    { "module": "Equipments" },
    { "module": "Documents" },
    { "module": "inspectionReport" }
  ],
  "countries": ["Zambia", "DRC"],
  "lastSyncTime": "2025-10-24T10:30:00.000Z"
}
```

---

## UI Components

### Sync Status Bar
```
┌─────────────────────────────────────────────┐
│ 🌐 Online • Last sync: 2m ago   [Sync] │
└─────────────────────────────────────────────┘
```

**Features:**
- Green background when online
- Orange background when offline
- Pulsing dot indicator
- Last sync time (Just now, 5m ago, 2h ago, 3d ago)
- Manual sync button (disabled when offline)
- Loading state during sync

### Sync Modal
```
┌──────────────────────┐
│    Please Wait       │
│  ⟳ Syncing with...  │
│     Server...        │
└──────────────────────┘
```

---

## Module Permission Logic

### Default Modules (Fallback)
```javascript
['Equipments', 'Documents', 'teamEmploy', 'hseInspection']
```

### Permission-Based Modules
- Modules are loaded from user's MongoDB profile
- Only valid modules (defined in `moduleDefinitions`) are shown
- Modules are filtered based on user role and permissions
- Cached locally for offline access

### Module Categories
1. **Master Data** - Equipments, Documents, Team/Employee
2. **Inspection and Compliances** - Inspection Reports, HSE, DVIR, Checklist, Incident Reports, Daily Reports
3. **Maintenance & Premium Services** - Request Maintenance, Job Card, Service Schedule
4. **Logbook** - Fuel Log, Logbook
5. **HR** - HR Form, HR Applications, Team KPI

---

## Error Handling

### Network Errors
- Automatically falls back to local storage
- Shows "Offline Mode" toast
- Continues operation with cached data

### MongoDB Errors
- Catches and logs errors
- Falls back to AsyncStorage
- Shows user-friendly error messages
- Doesn't break app functionality

### Data Validation
- Validates modules against `moduleDefinitions`
- Filters out invalid/undefined modules
- Provides default modules if none found
- Validates country options

---

## Code Improvements

### Before
```javascript
// Simple fetch with no offline handling
await MongoAPI.getUserById(uid);
```

### After
```javascript
// Smart sync with offline fallback
const fetchUserDataFromMongoDB = async (uid, showSyncModal = true) => {
  try {
    // Fetch from MongoDB
    const result = await MongoAPI.getUserById(uid);
    
    // Store locally with timestamp
    await AsyncStorage.setItem('userData', JSON.stringify(data));
    await AsyncStorage.setItem('lastSyncTimestamp', new Date().toISOString());
    
    // Show success toast
    showToast('success', 'Synced', 'Data synced successfully');
  } catch (error) {
    // Fallback to local storage
    const storedData = await AsyncStorage.getItem('userData');
    if (storedData) {
      setUserData(JSON.parse(storedData));
      showToast('info', 'Offline Mode', 'Using locally stored data');
    }
  }
};
```

---

## Testing Scenarios

### ✅ Scenario 1: Fresh Login (Online)
1. User logs in
2. Data fetched from MongoDB
3. Modules loaded based on permissions
4. Data cached locally
5. Shows "Data synced successfully"

### ✅ Scenario 2: App Opened (Offline)
1. User opens app without internet
2. Shows "Offline Mode" toast
3. Loads from AsyncStorage
4. Displays cached modules
5. All features work with cached data

### ✅ Scenario 3: Connection Lost During Use
1. User is using app
2. Internet disconnects
3. Shows "Offline" toast
4. Switches to offline mode
5. Continues using cached data

### ✅ Scenario 4: Connection Restored
1. Internet reconnects
2. Shows "Connection restored" toast
3. Auto-syncs from MongoDB
4. Updates local cache
5. Refreshes modules
6. Shows "Data synced successfully"

### ✅ Scenario 5: Manual Sync
1. User clicks "Sync" button
2. Shows sync modal
3. Fetches latest data
4. Updates UI
5. Shows success/error toast

---

## Performance Optimizations

1. **Lazy Loading**
   - Only fetch when needed
   - Use cached data when available

2. **Efficient Storage**
   - Store only essential data
   - Use timestamps for cache validation

3. **Smart Updates**
   - Only sync when necessary
   - Batch multiple updates

4. **Memory Management**
   - Clean up listeners on unmount
   - Use `isMounted` ref to prevent state updates

---

## Future Enhancements

### Potential Improvements
- [ ] Background sync with service workers
- [ ] Conflict resolution for offline edits
- [ ] Sync queue for pending operations
- [ ] Delta sync (only changed data)
- [ ] Compression for large datasets
- [ ] Sync history/logs
- [ ] Auto-sync interval configuration

---

## Dependencies

### Required Packages
```json
{
  "@react-native-async-storage/async-storage": "^1.x.x",
  "@react-native-community/netinfo": "^9.x.x",
  "react-native-toast-message": "^2.x.x",
  "react-native-vector-icons": "^9.x.x",
  "react-native-linear-gradient": "^2.x.x"
}
```

---

## Configuration

### MongoDB API Config
Located in: `components/ConfigMongo/MongoDBConfig.js`

```javascript
MongoAPI.getUserById(userId)      // Get user data
MongoAPI.updateUser(userId, data) // Update user
```

---

## Troubleshooting

### Issue: Modules not showing
**Solution:** Check user's modules array in MongoDB profile

### Issue: Sync not working
**Solution:** 
1. Check network connection
2. Verify MongoDB API endpoint
3. Check console logs for errors

### Issue: Old data showing
**Solution:** 
1. Click manual sync button
2. Clear app cache
3. Re-login

---

## Summary

The improved HomeScreen now provides:
- ✅ Robust online/offline functionality
- ✅ Automatic sync when online
- ✅ Local storage fallback when offline
- ✅ Visual sync status indicators
- ✅ User-friendly toast notifications
- ✅ Permission-based module display
- ✅ Manual sync capability
- ✅ Real-time network status monitoring
- ✅ Proper error handling
- ✅ Optimized performance

**Result:** Users can now work seamlessly both online and offline, with automatic synchronization and clear visual feedback.

---

**Last Updated:** October 24, 2025
**Version:** 2.0
**Author:** AI Assistant

