import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Dimensions,
  Modal,
  Animated,
  TextInput,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
// Removed Firebase database import - now using MongoDB
// Removed Firebase auth import - now using local storage for authentication
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import AntDesign from 'react-native-vector-icons/AntDesign';

const { width } = Dimensions.get('window');

const STATUS_OPTIONS = ['Okay', 'Not Okay'];

const Daily_Do_Ins2 = ({ setActiveStep }) => {
  const [sections, setSections] = useState([]);
  const [expanded, setExpanded] = useState({});
  const [status, setStatus] = useState({});
  const [inspectionId, setInspectionId] = useState(null);
  const [isOnline, setIsOnline] = useState(true);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [checklistDetails, setChecklistDetails] = useState(null); // Added to store checklist details for PDF
  
  // Header input fields state
  const [headerInputs, setHeaderInputs] = useState({});
  const [step1Data, setStep1Data] = useState(null);

  // Function to refresh equipment data from MongoDB
  const refreshEquipmentData = async () => {
    try {
      if (!isOnline || !step1Data?.selectedEquipment?.id) return;
      
      // Fetch latest equipment data from MongoDB
      const response = await fetch(`http://168.231.113.219:5001/equipment/${step1Data.selectedEquipment.id}`);
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.success && result.data) {
        const latestEquipmentData = result.data;
        
        if (latestEquipmentData && step1Data.selectedEquipment) {
          const needsUpdate = (
            latestEquipmentData.equipmentName !== step1Data.selectedEquipment.equipmentName ||
            latestEquipmentData.model !== step1Data.selectedEquipment.model ||
            latestEquipmentData.serialNumber !== step1Data.selectedEquipment.serialNumber
          );
          
          if (needsUpdate) {
            console.log('Refreshing equipment data from MongoDB in Step 2');
            
            // Update header inputs with latest data
            setHeaderInputs(prev => ({
              ...prev,
              machineName: prev.machineName ? { ...prev.machineName, value: latestEquipmentData.equipmentName } : prev.machineName,
              model: prev.model ? { ...prev.model, value: latestEquipmentData.model } : prev.model,
              serialNumber: prev.serialNumber ? { ...prev.serialNumber, value: latestEquipmentData.serialNumber } : prev.serialNumber,
            }));
            
            // Update stored data
            const updatedEquipment = {
              ...step1Data.selectedEquipment,
              equipmentName: latestEquipmentData.equipmentName,
              model: latestEquipmentData.model,
              serialNumber: latestEquipmentData.serialNumber,
              updatedAt: new Date().toISOString()
            };
            
            const updatedStep1 = { ...step1Data, selectedEquipment: updatedEquipment };
            await AsyncStorage.setItem('inspectionStep1', JSON.stringify(updatedStep1));
            setStep1Data(updatedStep1);
            
            Toast.show({
              type: 'success',
              text1: 'Equipment Updated',
              text2: 'Equipment data refreshed from database',
              position: 'top',
              visibilityTime: 2000,
            });
          } else {
            Toast.show({
              type: 'info',
              text1: 'Equipment Data',
              text2: 'Equipment data is already up to date',
              position: 'top',
              visibilityTime: 2000,
            });
          }
        }
      } else {
        throw new Error(result.error || 'Failed to fetch equipment data');
      }
    } catch (error) {
      console.warn('Error refreshing equipment data:', error);
      Toast.show({
        type: 'error',
        text1: 'Refresh Failed',
        text2: 'Could not refresh equipment data. Please check your connection.',
        position: 'top',
        visibilityTime: 3000,
      });
    }
  };

  // Load checklist data and inspection results from MongoDB
  useEffect(() => {
    let isMounted = true;

    const loadData = async () => {
      try {
        setSyncModalVisible(true);
        setIsLoading(true);

        // Load selected checklist ID
        const savedChecklistIds = await AsyncStorage.getItem('selectedChecklistIds');
        const checklistIds = savedChecklistIds ? JSON.parse(savedChecklistIds) : [];
        if (!checklistIds || checklistIds.length === 0) {
          throw new Error('No checklist selected');
        }
        const selectedChecklistId = checklistIds[0];

        // Fetch checklist from MongoDB
        const checklistResponse = await fetch(`http://168.231.113.219:5008/checklists/${selectedChecklistId}`);
        
        if (!checklistResponse.ok) {
          throw new Error(`HTTP error! status: ${checklistResponse.status}`);
        }

        const checklistResponseData = await checklistResponse.json();
        const checklistData = checklistResponseData.success ? checklistResponseData.data : null;

        if (!checklistData) {
          throw new Error('Selected checklist not found in MongoDB');
        }

        if (!checklistData.inspectionItems || !Array.isArray(checklistData.inspectionItems)) {
          throw new Error('Invalid checklist data: inspectionItems missing or not an array');
        }

        // Save checklist details for PDF
        const checklistInfo = {
          id: selectedChecklistId,
          formTitle: checklistData.formTitle || 'Inspection Items',
          inspectionItems: checklistData.inspectionItems,
          header_inputs: checklistData.header_inputs || {},
          machineDetails: checklistData.machineDetails || {},
          document: checklistData.document || {},
        };
        setChecklistDetails(checklistInfo);
        await AsyncStorage.setItem('checklistDetailsStep2', JSON.stringify(checklistInfo));

        // Load Step 1 data for header inputs
        const savedInspection = await AsyncStorage.getItem('inspectionStep1');
        if (savedInspection) {
          const step1Data = JSON.parse(savedInspection);
          setStep1Data(step1Data);
          
          // Initialize header inputs with specific layout and data
          const initialHeaderInputs = {};
          
          // Fetch latest equipment data from MongoDB for dynamic updates
          let latestEquipmentData = null;
          try {
            if (isOnline && step1Data.selectedEquipment?.id) {
              const equipmentResponse = await fetch(`http://168.231.113.219:5001/equipment/${step1Data.selectedEquipment.id}`);
              if (equipmentResponse.ok) {
                const equipmentResult = await equipmentResponse.json();
                if (equipmentResult.success && equipmentResult.data) {
                  latestEquipmentData = equipmentResult.data;
                }
              }
            }
          } catch (error) {
            console.warn('Error fetching latest equipment data from MongoDB in Step 2:', error);
          }
          
          // Get equipment details from latest MongoDB data (priority), then Step 1 data, then checklist data (fallback)
          const equipmentName = latestEquipmentData?.equipmentName || 
                               step1Data.selectedEquipment?.equipmentName || 
                               checklistData.machineDetails?.machineName || '';
          const equipmentModel = latestEquipmentData?.model || 
                                step1Data.selectedEquipment?.model || 
                                checklistData.machineDetails?.model || '';
          const vehicleSerialNumber = latestEquipmentData?.serialNumber || 
                                     step1Data.selectedEquipment?.serialNumber || 
                                     checklistData.machineDetails?.serialNumber || '';
          
          // Set up header inputs in specific order
          if (checklistData.header_inputs) {
            // Machine Name field - should show Equipment Name
            if (checklistData.header_inputs.machineName) {
              initialHeaderInputs.machineName = {
                label: 'MACHINE NAME:',
                value: equipmentName,
                editable: true,
                required: true,
                order: 1
              };
            }
            
            // Model field - should show Equipment Model
            if (checklistData.header_inputs.model) {
              initialHeaderInputs.model = {
                label: 'MODEL:',
                value: equipmentModel,
                editable: true,
                required: true,
                order: 2
              };
            }
            
            // Serial Number field - should show Vehicle Serial Number
            if (checklistData.header_inputs.serialNumber) {
              initialHeaderInputs.serialNumber = {
                label: 'SERIAL #:',
                value: vehicleSerialNumber,
                editable: true,
                required: true,
                order: 3
              };
            }
            
            // Month field - always last
            if (checklistData.header_inputs.month) {
              const today = new Date();
              const monthNames = [
                'January', 'February', 'March', 'April', 'May', 'June',
                'July', 'August', 'September', 'October', 'November', 'December'
              ];
              initialHeaderInputs.month = {
                label: 'MONTH:',
                value: monthNames[today.getMonth()],
                editable: false,
                required: true,
                order: 4
              };
            }
          }
          
          setHeaderInputs(initialHeaderInputs);
          
          // Update equipment data if it's different from what's stored
          if (latestEquipmentData && step1Data.selectedEquipment) {
            const needsUpdate = (
              latestEquipmentData.equipmentName !== step1Data.selectedEquipment.equipmentName ||
              latestEquipmentData.model !== step1Data.selectedEquipment.model ||
              latestEquipmentData.serialNumber !== step1Data.selectedEquipment.serialNumber
            );
            
            if (needsUpdate) {
              console.log('Equipment data updated in Step 2 from database:', {
                old: {
                  equipmentName: step1Data.selectedEquipment.equipmentName,
                  model: step1Data.selectedEquipment.model,
                  serialNumber: step1Data.selectedEquipment.serialNumber
                },
                new: {
                  equipmentName: latestEquipmentData.equipmentName,
                  model: latestEquipmentData.model,
                  serialNumber: latestEquipmentData.serialNumber
                }
              });
              
              // Update the stored equipment data
              const updatedEquipment = {
                ...step1Data.selectedEquipment,
                equipmentName: latestEquipmentData.equipmentName,
                model: latestEquipmentData.model,
                serialNumber: latestEquipmentData.serialNumber,
                updatedAt: new Date().toISOString()
              };
              
              // Update AsyncStorage
              try {
                const updatedStep1 = { ...step1Data, selectedEquipment: updatedEquipment };
                await AsyncStorage.setItem('inspectionStep1', JSON.stringify(updatedStep1));
                
                // Update global equipment list
                const storedEquipment = await AsyncStorage.getItem('equipment');
                if (storedEquipment) {
                  let equipmentList = JSON.parse(storedEquipment);
                  const index = equipmentList.findIndex(eq => eq.id === updatedEquipment.id);
                  if (index !== -1) {
                    equipmentList[index] = updatedEquipment;
                    await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
                  }
                }
                
                // Update local state
                setStep1Data(updatedStep1);
              } catch (error) {
                console.warn('Error updating stored equipment data in Step 2:', error);
              }
            }
          }
        }

        // Format sections
        const formattedSections = [
          {
            id: 'section_0',
            title: checklistData.formTitle || 'Inspection Items',
            items: checklistData.inspectionItems.map((item, itemIdx) => ({
              id: `item_0_${itemIdx}`,
              name: item,
            })),
          },
        ];

        if (formattedSections[0].items.length === 0) {
          throw new Error('No valid items found in checklist');
        }

        if (isMounted) {
          setSections(formattedSections);

          // Initialize expanded state
          const initialExpanded = formattedSections.reduce((acc, section) => ({
            ...acc,
            [section.id]: true,
          }), {});
          setExpanded(initialExpanded);

          // Initialize status state
          const initialStatus = formattedSections.reduce((acc, section) => {
            section.items.forEach((item) => {
              acc[item.id] = '';
            });
            return acc;
          }, {});
          setStatus(initialStatus);

          // Load inspection data
          const savedInspection = await AsyncStorage.getItem('inspectionStep1');
          if (savedInspection) {
            const inspectionData = JSON.parse(savedInspection);
            if (inspectionData.id) {
              setInspectionId(inspectionData.id);
            } else {
              throw new Error('Inspection ID not found in saved data');
            }

            // Skip fetching inspection results since we removed authentication requirement
            console.log('📋 Skipping inspection results fetch - authentication removed');
            
            try {
              const inspectionResponse = await fetch(`http://168.231.113.219:5008/inspections/${userId}/${inspectionData.id}`);
              if (inspectionResponse.ok) {
                const inspectionDataResponse = await inspectionResponse.json();
                if (inspectionDataResponse.success && inspectionDataResponse.data && inspectionDataResponse.data.step2) {
                  const step2Data = inspectionDataResponse.data.step2;
                  if (step2Data.results) {
                    setStatus((prevStatus) => ({
                      ...prevStatus,
                      ...step2Data.results,
                    }));
                  }
                }
              }
            } catch (error) {
              console.warn('Error fetching inspection results from MongoDB:', error);
              // Don't show error toast for this as it's not critical
            }
          } else {
            throw new Error('Inspection data not found');
          }

          // Check network status
          const netInfo = await NetInfo.fetch();
          setIsOnline(netInfo.isConnected);
        }
      } catch (error) {
        console.error('Error loading checklist data:', error);
        if (isMounted) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: error.message || 'Failed to load checklist data. Please go back and try again.',
            position: 'top',
          });
        }
      } finally {
        if (isMounted) {
          setSyncModalVisible(false);
          setIsLoading(false);
        }
      }
    };

    loadData();

    // Removed network monitoring - data will be synced from step 3 only
    return () => {
      isMounted = false;
    };
  }, []);

  // Removed syncDataToFirebase function - data will be synced from step 3 only

  // Save inspection data to AsyncStorage only
  const saveInspectionData = async (data) => {
    try {
      await AsyncStorage.setItem('inspectionStep2', JSON.stringify(data));
      // Data will be submitted to database from step 3
    } catch (error) {
      console.error('Error saving inspection data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save inspection data locally.',
        position: 'top',
      });
    }
  };

  const handleExpand = (sectionId) => {
    setExpanded((prev) => ({ ...prev, [sectionId]: !prev[sectionId] }));
  };

  const handleStatusChange = (itemId, statusValue) => {
    setStatus((prev) => {
      const updatedStatus = { ...prev, [itemId]: statusValue };
      // Load inspectionDate from inspectionStep1
      AsyncStorage.getItem('inspectionStep1').then((savedInspection) => {
        let inspectionDate = null;
        if (savedInspection) {
          const inspectionData = JSON.parse(savedInspection);
          inspectionDate = inspectionData.inspectionDate;
        }
        const inspectionDataToSave = {
        inspectionId,
        results: updatedStatus,
        headerInputs, // Include header input values
        checklistDetails, // Include checklist details for PDF
          inspectionDate, // Always save the selected inspection date
        updatedAt: new Date().toISOString(),
      };
        saveInspectionData(inspectionDataToSave);
      });
      return updatedStatus;
    });
  };

  const handleHeaderInputChange = (key, value) => {
    setHeaderInputs(prev => ({
      ...prev,
      [key]: {
        ...prev[key],
        value: value
      }
    }));
  };

  const getStatusBtnStyle = (selected, status) => {
    if (!selected) return styles.statusBtn;
    switch (status) {
      case 'Okay':
        return [styles.statusBtn, styles.statusOk];
      case 'Not Okay':
        return [styles.statusBtn, styles.statusNotOkay];
      default:
        return styles.statusBtn;
    }
  };

  const getStatusTextStyle = (selected, status) => {
    if (!selected) return styles.statusBtnText;
    switch (status) {
      case 'Okay':
        return [styles.statusBtnText, { color: '#222' }];
      case 'Not Okay':
        return [styles.statusBtnText, { color: '#fff' }];
      default:
        return styles.statusBtnText;
    }
  };

  const handleNextPress = async () => {
    if (isLoading || sections.length === 0) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Checklist data is still loading or unavailable. Please try again.',
        position: 'top',
      });
      return;
    }

    // Validate that all items have a status
    const allAnswered = sections.every((section) =>
      section.items.every((item) => status[item.id] !== '')
    );

    if (!allAnswered) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Please provide a status for all inspection items.',
        position: 'top',
      });
      return;
    }

    // Validate required header fields
    const requiredHeaderFields = Object.keys(headerInputs).filter(key => 
      headerInputs[key].required && (!headerInputs[key].value || headerInputs[key].value.trim() === '')
    );

    if (requiredHeaderFields.length > 0) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Please fill all required header fields.',
        position: 'top',
      });
      return;
    }

    if (!inspectionId) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Inspection ID is missing. Please go back and try again.',
        position: 'top',
      });
      return;
    }

    try {
      setSyncModalVisible(true);
      // Remove authentication requirement - use local storage instead
      console.log('📋 Proceeding to Step 3 without authentication requirement');
      // Load inspectionDate from inspectionStep1
      let inspectionDate = null;
      const savedInspection = await AsyncStorage.getItem('inspectionStep1');
      if (savedInspection) {
        const inspectionData = JSON.parse(savedInspection);
        inspectionDate = inspectionData.inspectionDate;
      }
      const inspectionDataToSave = {
        inspectionId,
        results: status,
        headerInputs, // Include header input values
        checklistDetails, // Include checklist details for PDF
        inspectionDate, // Always save the selected inspection date
        updatedAt: new Date().toISOString(),
      };
      await saveInspectionData(inspectionDataToSave);
      setActiveStep(3);
    } catch (error) {
      console.error('Error saving inspection step 2:', error);
      let errorMessage = 'Failed to save inspection data.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Check Firebase security rules.';
      } else if (error.code === 'NETWORK_ERROR') {
        errorMessage = 'Network error. Data saved locally and will sync when online.';
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
    }
  };

  const renderItem = (item, index) => (
    <View key={item.id} style={styles.centeredItem}>
      <View style={styles.itemContainer}>
        <Text style={styles.itemText}>
          {index + 1}. {item.name}
        </Text>
        <View style={styles.itemRow}>
          {STATUS_OPTIONS.map((statusOption) => {
            const isSelected = status[item.id] === statusOption;
            return (
              <TouchableOpacity
                key={statusOption}
                style={[
                  styles.optionButton,
                  isSelected && styles.selectedOptionButton,
                  isSelected && statusOption === 'Okay' && styles.okButton,
                  isSelected && statusOption === 'Not Okay' && styles.rejectButton,
                ]}
                onPress={() => handleStatusChange(item.id, statusOption)}
              >
                <Text
                  style={[
                    styles.optionText,
                    isSelected && styles.optionTextSelected,
                  ]}
                >
                  {statusOption}
                </Text>
              </TouchableOpacity>
            );
          })}
        </View>
      </View>
      <View style={styles.itemBorder} />
    </View>
  );

  if (isLoading) {
    return (
      <View style={styles.container}>
        <View style={styles.topBar}>
          <Text style={styles.topBarText}>Loading checklist...</Text>
        </View>
        <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message="Loading Checklist" />
      </View>
    );
  }

  if (sections.length === 0) {
    return (
      <View style={styles.container}>
        <View style={styles.topBar}>
          <Text style={styles.topBarText}>Error: No checklist data available</Text>
        </View>
        <View style={styles.emptyContainer}>
          <Text style={styles.emptyText}>No checklist data available. Please go back and select a checklist.</Text>
          <TouchableOpacity style={styles.backBtn} onPress={() => setActiveStep && setActiveStep(1)}>
            <Text style={styles.backText}>BACK</Text>
          </TouchableOpacity>
        </View>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        {/* Header Input Fields */}
        {Object.keys(headerInputs).length > 0 && (
          <View style={styles.headerInputsContainer}>
            <View style={styles.headerInputsHeader}>
              <Text style={styles.headerInputsTitle}>Equipment Details</Text>
              <TouchableOpacity 
                style={styles.refreshButton} 
                onPress={refreshEquipmentData}
                disabled={!isOnline}
              >
                <MaterialIcons 
                  name="refresh" 
                  size={20} 
                  color={isOnline ? "#015185" : "#ccc"} 
                />
              </TouchableOpacity>
            </View>
            
            {/* Machine Name Field */}
            {headerInputs.machineName && (
              <View style={styles.headerInputField}>
                <Text style={styles.headerInputLabel}>
                  {headerInputs.machineName.label}
                  {headerInputs.machineName.required && <Text style={styles.requiredStar}> *</Text>}
                </Text>
                <TextInput
                  style={[
                    styles.headerInputTextInput,
                    !headerInputs.machineName.editable && styles.headerInputDisabled
                  ]}
                  value={headerInputs.machineName.value}
                  onChangeText={(text) => handleHeaderInputChange('machineName', text)}
                  editable={headerInputs.machineName.editable}
                  placeholder="_________________________"
                  placeholderTextColor="#888"
                />
              </View>
            )}
            
            {/* Model Field */}
            {headerInputs.model && (
              <View style={styles.headerInputField}>
                <Text style={styles.headerInputLabel}>
                  {headerInputs.model.label}
                  {headerInputs.model.required && <Text style={styles.requiredStar}> *</Text>}
                </Text>
                <TextInput
                  style={[
                    styles.headerInputTextInput,
                    !headerInputs.model.editable && styles.headerInputDisabled
                  ]}
                  value={headerInputs.model.value}
                  onChangeText={(text) => handleHeaderInputChange('model', text)}
                  editable={headerInputs.model.editable}
                  placeholder="__________________________"
                  placeholderTextColor="#888"
                />
              </View>
            )}
            
            {/* Serial Number Field */}
            {headerInputs.serialNumber && (
              <View style={styles.headerInputField}>
                <Text style={styles.headerInputLabel}>
                  {headerInputs.serialNumber.label}
                  {headerInputs.serialNumber.required && <Text style={styles.requiredStar}> *</Text>}
                </Text>
                <TextInput
                  style={[
                    styles.headerInputTextInput,
                    !headerInputs.serialNumber.editable && styles.headerInputDisabled
                  ]}
                  value={headerInputs.serialNumber.value}
                  onChangeText={(text) => handleHeaderInputChange('serialNumber', text)}
                  editable={headerInputs.serialNumber.editable}
                  placeholder="_______________________"
                  placeholderTextColor="#888"
                />
              </View>
            )}
            
            {/* Month Field */}
            {headerInputs.month && (
              <View style={styles.headerInputField}>
                <Text style={styles.headerInputLabel}>
                  {headerInputs.month.label}
                  {headerInputs.month.required && <Text style={styles.requiredStar}> *</Text>}
                </Text>
                <TextInput
                  style={[
                    styles.headerInputTextInput,
                    !headerInputs.month.editable && styles.headerInputDisabled
                  ]}
                  value={headerInputs.month.value}
                  onChangeText={(text) => handleHeaderInputChange('month', text)}
                  editable={headerInputs.month.editable}
                  placeholder="_______________"
                  placeholderTextColor="#888"
                />
              </View>
            )}
          </View>
        )}
        
        <View style={styles.headingContainer}>
          <Text style={styles.headingText}>Tap on section to expand</Text>
        </View>
        {sections.map((section, idx) => (
          <View key={section.id}>
            <TouchableOpacity
              style={styles.sectionHeader}
              onPress={() => handleExpand(section.id)}
            >
              <View style={styles.sectionHeaderContent}>
                <AntDesign
                  name={expanded[section.id] ? 'caretup' : 'caretdown'}
                  size={20}
                  color={expanded[section.id] ? 'red' : 'black'}
                />
                <Text style={styles.sectionHeaderText}>{section.title}</Text>
              </View>
            </TouchableOpacity>
            {expanded[section.id] && (
              <View style={styles.sectionContent}>
                {section.items.length > 0 ? (
                  section.items.map((item, itemIdx) => renderItem(item, itemIdx))
                ) : (
                  <Text style={styles.emptySectionText}>No inspection items</Text>
                )}
              </View>
            )}
          </View>
        ))}
      </ScrollView>
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep && setActiveStep(1)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={handleNextPress}
        >
          <Text style={styles.nextButtonText}>Inspection Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
        message="Saving Inspection Data"
      />
      <Toast />
    </View>
  );
};

// SyncModal Component
const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = React.useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  React.useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.modalOverlay}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[{ width: 30, height: 30, marginRight: 10 }, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  headerInputsContainer: {
    backgroundColor: '#f8f9fa',
    padding: 15,
    marginHorizontal: 15,
    marginVertical: 10,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  headerInputsHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
  },
  headerInputsTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    flex: 1,
    textAlign: 'center',
  },
  refreshButton: {
    padding: 8,
    borderRadius: 20,
    backgroundColor: '#f0f0f0',
  },
  headerInputField: {
    width: '100%',
    marginBottom: 15,
  },
  headerInputLabel: {
    fontSize: 14,
    fontWeight: '500',
    color: '#333',
    marginBottom: 5,
  },
  requiredStar: {
    color: 'red',
    fontWeight: 'bold',
  },
  headerInputTextInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
  },

  headerInputDisabled: {
    backgroundColor: '#f5f5f5',
    borderColor: '#ccc',
    color: '#666',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 80,
  },
  headingContainer: {
    paddingVertical: 10,
    marginHorizontal: '5%',
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  sectionHeader: {
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  sectionHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 5,
  },
  sectionHeaderText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
  },
  sectionContent: {
    backgroundColor: '#fff',
  },
  centeredItem: {
    width: '90%',
    alignSelf: 'center',
    marginBottom: 15,
  },
  itemContainer: {
    paddingBottom: 15,
  },
  itemText: {
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-start',
    marginTop: 8,
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  rejectButton: {
    backgroundColor: '#FF4B2B',
  },
  optionText: {
    fontSize: 14,
    color: '#333',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  itemBorder: {
    width: '80%',
    alignSelf: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    marginTop: 10,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  emptySectionText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    padding: 20,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIconPlaceholder: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  topBar: {
    backgroundColor: '#4A6B8A',
    paddingVertical: 4,
    alignItems: 'flex-start',
    paddingHorizontal: 8,
  },
  topBarText: {
    color: '#fff',
    fontSize: 13,
    fontWeight: '500',
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
  },
  emptyText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginBottom: 20,
  },
  backBtn: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: 18,
    backgroundColor: '#fff',
  },
  backText: {
    color: '#B71C1C',
    fontWeight: 'bold',
    fontSize: 16,
    letterSpacing: 1,
  },
});

export default Daily_Do_Ins2;