import React, { useState, useRef } from 'react';
import {
  View,
  Text,
  TextInput,
  TouchableOpacity,
  StyleSheet,
  Modal,
  ScrollView,
  Animated,
  Image,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Toast from 'react-native-toast-message';

const { width, height } = Dimensions.get('window');

const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  inputRef,
  onFocus,
  ...props
}) => (
  <View style={inputStyles.container}>
    <Text style={inputStyles.label}>
      {label}
      {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
    </Text>
    <TextInput
      ref={inputRef}
      style={[
        inputStyles.input,
        showError ? inputStyles.inputError : null,
        props.multiline ? styles.descriptionInput : null,
      ]}
      value={value}
      onChangeText={onChangeText}
      onFocus={onFocus}
      {...props}
      placeholderTextColor="#888"
    />
    {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
  </View>
);

const Add_inspection_item = ({ parentIndex, onSave, onCancel, initialData, subItemIndex }) => {
  const isEditing = !!initialData;

  const [itemName, setItemName] = useState(initialData?.itemName || '');
  const [description, setDescription] = useState(initialData?.description || '');
  const [inputType, setInputType] = useState(initialData?.inputType || 'Ok/Reject/N/A');
  const [openFieldPlaceholder, setOpenFieldPlaceholder] = useState(initialData?.openFieldPlaceholder || '');
  const [instruction, setInstruction] = useState(initialData?.instruction || '');
  const [sortOrder, setSortOrder] = useState(initialData?.sortOrder?.toString() || '');
  const [isInspectionRequired, setIsInspectionRequired] = useState(initialData?.isInspectionRequired || false);
  const [isDetailsOpen, setIsDetailsOpen] = useState(!!initialData);
  const [isModalVisible, setIsModalVisible] = useState(false);
  const [isSavePopupVisible, setIsSavePopupVisible] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [showErrors, setShowErrors] = useState(false);

  const scrollViewRef = useRef(null);
  const itemNameRef = useRef(null);
  const descriptionRef = useRef(null);
  const openFieldPlaceholderRef = useRef(null);
  const instructionRef = useRef(null);
  const sortOrderRef = useRef(null);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const handleFocus = (ref) => () => {
    if (ref.current) {
      ref.current.measure((x, y, width, height, pageX, pageY) => {
        scrollViewRef.current.scrollTo({ y: pageY - 100, animated: true });
      });
    }
  };

  const handleSave = async () => {
    setShowErrors(true);
    if (!itemName.trim()) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Inspection item name is required.',
      });
      return;
    }

    if (inputType === 'Open Field' && !openFieldPlaceholder.trim()) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Open field placeholder is required when input type is "Open Field".',
      });
      return;
    }

    setIsSaving(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    const inspectionItem = {
      itemName: itemName.trim(),
      description: description.trim(),
      inputType,
      openFieldPlaceholder: inputType === 'Open Field' ? openFieldPlaceholder.trim() : undefined,
      instruction: instruction.trim(),
      sortOrder: parseInt(sortOrder) || 0,
      isInspectionRequired,
    };

    try {
      await onSave(inspectionItem);
      setIsSavePopupVisible(true);
    } catch (error) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: `Failed to ${isEditing ? 'update' : 'save'} inspection item.`,
      });
    } finally {
      setIsSaving(false);
      rotateAnim.setValue(0);
    }
  };

  const handleInputTypeSelect = (type) => {
    setInputType(type);
    if (type !== 'Open Field') {
      setOpenFieldPlaceholder('');
    }
    setIsModalVisible(false);
  };

  const handlePopupYes = () => {
    setItemName('');
    setDescription('');
    setInputType('Ok/Reject/N/A');
    setOpenFieldPlaceholder('');
    setInstruction('');
    setSortOrder('');
    setIsInspectionRequired(false);
    setIsDetailsOpen(false);
    setIsSavePopupVisible(false);
    setShowErrors(false);
  };

  const handlePopupNo = () => {
    setIsSavePopupVisible(false);
    setShowErrors(false);
    onCancel();
  };

  return (
    <KeyboardAvoidingView
      style={styles.container}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.select({ ios: 60, android: 80 })}
    >
      <ScrollView
        ref={scrollViewRef}
        style={styles.mainContent}
        contentContainerStyle={styles.scrollContent}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={true}
      >
        <Text style={styles.headerTitle}>
          {isEditing ? 'Edit Inspection Item' : 'Add Inspection Item'}
        </Text>
        <CustomInputField
          label="Inspection Item Name"
          value={itemName}
          onChangeText={setItemName}
          placeholder="Enter inspection item name"
          isRequired
          showError={showErrors && !itemName.trim()}
          errorMessage="Inspection item name is required"
          inputRef={itemNameRef}
          onFocus={handleFocus(itemNameRef)}
        />
        <CustomInputField
          label="Description"
          value={description}
          onChangeText={setDescription}
          placeholder="Enter description"
          multiline
          numberOfLines={4}
          inputRef={descriptionRef}
          onFocus={handleFocus(descriptionRef)}
        />
        <View style={styles.inputTypeHeader}>
          <Text style={inputStyles.label}>Input Type</Text>
          <TouchableOpacity onPress={() => setIsModalVisible(true)}>
            <MaterialIcons name="add-circle" size={24} color="#015185" />
          </TouchableOpacity>
        </View>
        {inputType && (
          <View style={styles.selectedTypeContainer}>
            <Text style={styles.optionText}>{inputType}</Text>
          </View>
        )}
        {inputType === 'Open Field' && (
          <CustomInputField
            label="Open Field Placeholder"
            value={openFieldPlaceholder}
            onChangeText={setOpenFieldPlaceholder}
            placeholder="Enter placeholder (e.g., Enter comments)"
            isRequired
            showError={showErrors && !openFieldPlaceholder.trim()}
            errorMessage="Open field placeholder is required"
            inputRef={openFieldPlaceholderRef}
            onFocus={handleFocus(openFieldPlaceholderRef)}
          />
        )}
        <TouchableOpacity style={styles.detailsRow} onPress={() => setIsDetailsOpen(!isDetailsOpen)}>
          <Text style={inputStyles.label}>Additional Details</Text>
          <MaterialIcons
            name={isDetailsOpen ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
            size={24}
            color="#015185"
          />
        </TouchableOpacity>
        {isDetailsOpen && (
          <>
            <CustomInputField
              label="Instruction"
              value={instruction}
              onChangeText={setInstruction}
              placeholder="Enter instruction"
              multiline
              numberOfLines={4}
              inputRef={instructionRef}
              onFocus={handleFocus(instructionRef)}
            />
            <CustomInputField
              label="Sort Order"
              value={sortOrder}
              onChangeText={setSortOrder}
              placeholder="Enter sort order"
              keyboardType="numeric"
              inputRef={sortOrderRef}
              onFocus={handleFocus(sortOrderRef)}
            />
            <View style={styles.checkboxRow}>
              <TouchableOpacity
                style={styles.checkbox}
                onPress={() => setIsInspectionRequired(!isInspectionRequired)}
              >
                {isInspectionRequired && (
                  <MaterialIcons name="check" size={18} color="#015185" />
                )}
              </TouchableOpacity>
              <Text style={styles.checkboxText}>Inspection required for this item</Text>
            </View>
          </>
        )}
      </ScrollView>
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={onCancel}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.saveButton} onPress={handleSave} disabled={isSaving}>
          <Text style={styles.saveButtonText}>{isEditing ? 'Update' : 'Save'}</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <View style={styles.footerContainer}>
        <LinearGradient
          colors={['#fff', '#025383']}
          style={styles.gradientFooter}
          start={{ x: 0, y: 0 }}
          end={{ x: 1, y: 0 }}
        />
      </View>
      <Modal
        visible={isModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setIsModalVisible(false)}
      >
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Input Type</Text>
            <ScrollView style={styles.modalScroll}>
              {['Ok/Reject/N/A', 'Ok/Not Ok', 'Numeric', 'Open Field'].map((type) => (
                <TouchableOpacity
                  key={type}
                  style={styles.modalOption}
                  onPress={() => handleInputTypeSelect(type)}
                >
                  <View style={styles.optionContent}>
                    <View style={styles.checkbox}>
                      {inputType === type && (
                        <MaterialIcons name="check" size={18} color="#015185" />
                      )}
                    </View>
                    <Text style={styles.optionText}>{type}</Text>
                  </View>
                </TouchableOpacity>
              ))}
            </ScrollView>
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setIsModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
      <Modal
        visible={isSavePopupVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setIsSavePopupVisible(false)}
      >
        <View style={styles.popupContainer}>
          <View style={styles.popupContent}>
            <Text style={styles.popupTitle}>Information</Text>
            <MaterialIcons name="check-circle" size={40} color="#28a745" style={styles.popupIcon} />
            <View style={styles.popupDivider} />
            <Text style={styles.popupText}>
              Inspection item {isEditing ? 'updated' : 'saved'}. Do you want to add another inspection
              item?
            </Text>
            <View style={styles.popupButtonContainer}>
              <TouchableOpacity style={styles.popupButton} onPress={handlePopupNo}>
                <Text style={styles.popupButtonText}>No</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.popupButton} onPress={handlePopupYes}>
                <Text style={styles.popupButtonText}>Yes</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      <Modal transparent={true} animationType="fade" visible={isSaving} onRequestClose={() => {}}>
        <View style={styles.modalContainer}>
          <View style={styles.loaderContent}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.loaderText}>
              {isEditing ? 'Updating' : 'Saving'} Inspection Item...
            </Text>
          </View>
        </View>
      </Modal>
      <Toast />
    </KeyboardAvoidingView>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    flexGrow: 1,
    paddingBottom: height * 0.18,
  },
  headerTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  descriptionInput: {
    height: 100,
    textAlignVertical: 'top',
  },
  inputTypeHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: width * 0.04,
  },
  selectedTypeContainer: {
    padding: width * 0.03,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    backgroundColor: '#fff',
    marginBottom: width * 0.04,
  },
  detailsRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: width * 0.04,
    paddingVertical: width * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  checkboxRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: width * 0.04,
  },
  checkbox: {
    width: 24,
    height: 24,
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 4,
    justifyContent: 'center',
    alignItems: 'center',
    marginRight: width * 0.02,
  },
  checkboxText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 16,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    paddingHorizontal: 15,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  saveButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  saveButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  footerContainer: {
    height: 8,
    width: '100%',
    backgroundColor: 'transparent',
    position: 'absolute',
    bottom: 0,
  },
  gradientFooter: {
    height: 8,
    width: '80%',
    alignSelf: 'center',
    borderRadius: 4,
  },
  modalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    padding: width * 0.05,
    borderRadius: 12,
    width: '85%',
    maxHeight: '60%',
  },
  modalScroll: {
    maxHeight: 200,
  },
  modalTitle: {
    fontSize: width * 0.05,
    fontWeight: '600',
    color: '#333',
    textAlign: 'center',
    marginBottom: width * 0.05,
  },
  modalOption: {
    marginBottom: width * 0.03,
  },
  optionContent: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    backgroundColor: '#f8f9fa',
  },
  optionText: {
    marginLeft: width * 0.02,
    fontSize: width * 0.04,
    color: '#333',
  },
  modalCloseButton: {
    padding: width * 0.03,
    alignItems: 'center',
    borderRadius: 5,
    backgroundColor: '#015185',
    marginTop: width * 0.03,
  },
  modalCloseText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  popupContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  popupContent: {
    backgroundColor: '#fff',
    padding: width * 0.05,
    borderRadius: 12,
    width: '85%',
    alignItems: 'center',
  },
  popupTitle: {
    fontSize: width * 0.05,
    fontWeight: '600',
    color: '#333',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  popupIcon: {
    marginBottom: width * 0.04,
  },
  popupDivider: {
    width: '80%',
    height: 1,
    backgroundColor: '#ddd',
    marginBottom: width * 0.04,
  },
  popupText: {
    fontSize: width * 0.04,
    textAlign: 'center',
    color: '#555',
    marginBottom: width * 0.05,
  },
  popupButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
  },
  popupButton: {
    padding: width * 0.03,
    width: width * 0.3,
    alignItems: 'center',
    borderRadius: 5,
    backgroundColor: '#015185',
    marginHorizontal: width * 0.02,
  },
  popupButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  loaderContent: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    alignItems: 'center',
  },
  loaderText: {
    marginTop: 10,
    fontSize: 16,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginBottom: 10,
  },
});

export default Add_inspection_item;