
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  TextInput,
  Dimensions,
  Alert,
  Modal,
  ActivityIndicator,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';
import NetInfo from '@react-native-community/netinfo';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import { Buffer } from 'buffer';
import HSEInspectionAPI from '../../../ConfigMongo/HSEInspectionMongoConfig';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible, message = 'Generating Report', progress = 0 }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Generating PDF Report</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
            {progress > 0 && (
              <View style={styles.progressContainer}>
                <View style={styles.progressBar}>
                  <View style={[styles.progressFill, { width: `${progress}%` }]} />
                </View>
                <Text style={styles.progressText}>{progress}%</Text>
              </View>
            )}
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_131Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [formData, setFormData] = useState({
    checkedByInitials: '',
    dateOfInspectionInitials: '',
    dateOfInspectionReviewDate: new Date(),
    managementOverInspectionInitials: '',
    managementOverInspectionReviewDate: new Date(),
  });
  // Keep minimal placeholders to satisfy existing logic but not used in UI
  const [deviations, setDeviations] = useState([]);
  const [inspectionRegister, setInspectionRegister] = useState([]);
  const [selectedMonth, setSelectedMonth] = useState('');
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [showTargetDatePicker, setShowTargetDatePicker] = useState(false);
  const [showSpotCheckDatePicker, setShowSpotCheckDatePicker] = useState(false);
  const [selectedDateField, setSelectedDateField] = useState('');
  const [selectedDeviationId, setSelectedDeviationId] = useState(null);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [reportId, setReportId] = useState(null);
  const [isLoadingData, setIsLoadingData] = useState(false);
  const isMounted = useRef(true);

  useEffect(() => {
    loadSavedData();
    loadStep2Data();
    loadExistingDataFromMongoDB();
  }, []);

  // Auto-select months with data and latest month
  useEffect(() => {
    if (inspectionRegister.length > 0) {
      const register = inspectionRegister[0];
      
      // Always check database for latest data
      loadExistingDataFromMongoDB();
      
      // Find months with existing data
      const monthsWithData = [];
      Object.keys(register.checklistStatus || {}).forEach(month => {
        const hasData = Object.values(register.checklistStatus[month] || {}).some(value => 
          value === 'Yes' || value === 'No' || value === 'N/A'
        );
        if (hasData) {
          monthsWithData.push(month);
        }
      });
      
      // Get current month
      const currentDate = new Date();
      const currentMonth = currentDate.getMonth();
      const monthMap = {
        0: 'january', 1: 'february', 2: 'march', 3: 'april', 4: 'may', 5: 'june',
        6: 'july', 7: 'august', 8: 'september', 9: 'october', 10: 'november', 11: 'december'
      };
      const latestMonth = monthMap[currentMonth];
      
      // Check if any month is already selected
      const hasSelectedMonth = Object.values(register.monthlyStatus || {}).some(status => status === 'OK');
      
      if (!hasSelectedMonth) {
        // Auto-select latest month (current month)
        console.log(`Auto-selecting latest month: ${latestMonth}`);
        const updatedInspectionRegister = inspectionRegister.map(r => 
          r.id === register.id 
            ? {
                ...r,
                monthlyStatus: {
                  ...r.monthlyStatus,
                  [latestMonth]: 'OK'
                }
              }
            : r
        );
        setInspectionRegister(updatedInspectionRegister);
      }
    }
  }, [inspectionRegister.length]); // Only run when inspectionRegister length changes

  const loadSavedData = async () => {
    try {
      const savedStep3 = await AsyncStorage.getItem('hseStep3');
      if (savedStep3) {
        const step3Data = JSON.parse(savedStep3);
        setFormData(step3Data.formData || formData);
      }
    } catch (error) {
      console.error('Error loading Step 3 data:', error);
    }
  };

  const loadStep2Data = async () => {
    try {
      console.log('🔄 Loading Step 2 data...');
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      if (savedStep2) {
        const step2Data = JSON.parse(savedStep2);
        console.log('✅ Step 2 data loaded:', step2Data);
        console.log('📋 Step 2 checklist data:', step2Data.checklistData);
        
        // Log detailed Step 2 data for debugging
        if (step2Data.checklistData) {
          const step2Checklist = step2Data.checklistData;
          console.log('🔍 Webbing data:', step2Checklist.webbing);
          console.log('🔍 Metal Fitting data:', step2Checklist.metalFitting);
          console.log('🔍 Lanyard data:', step2Checklist.lanyard);
        }
        
        // Auto-fill checklist data from Step 2
        if (step2Data.checklistData) {
          const step2Checklist = step2Data.checklistData;
          
          // Update inspection register with Step 2 data
          setInspectionRegister(prev => 
            prev.map(register => {
              // Find which month is currently selected
              const selectedMonth = Object.keys(register.monthlyStatus).find(
                month => register.monthlyStatus[month] === 'OK'
              );
              
              if (selectedMonth) {
                // Map Step 2 data to Step 3 format
                const mappedChecklistData = {};
                
                // Webbing items
                if (step2Checklist.webbing) {
                  mappedChecklistData.webbing = step2Checklist.webbing.webbing || '';
                  mappedChecklistData.dirty = step2Checklist.webbing.dirty || '';
                  mappedChecklistData.heatDamage = step2Checklist.webbing.heatDamage || '';
                  mappedChecklistData.cutsOrChafing = step2Checklist.webbing.cutsOrChafing || '';
                  mappedChecklistData.seamDamaged = step2Checklist.webbing.seamDamaged || '';
                  mappedChecklistData.anyOtherPhysicalDamage = step2Checklist.webbing.anyOtherPhysicalDamage || '';
                }
                
                // Metal Fitting items
                if (step2Checklist.metalFitting) {
                  mappedChecklistData.metalFitting = step2Checklist.metalFitting.metalFitting || '';
                  mappedChecklistData.couplingsDamaged = step2Checklist.metalFitting.couplingsDamaged || '';
                  mappedChecklistData.shacklesDamaged = step2Checklist.metalFitting.shacklesDamaged || '';
                  mappedChecklistData.clipsDamaged = step2Checklist.metalFitting.clipsDamaged || '';
                  mappedChecklistData.lifelineConnectionsDamaged = step2Checklist.metalFitting.lifelineConnectionsDamaged || '';
                  mappedChecklistData.lanyardConnectionsDamaged = step2Checklist.metalFitting.lanyardConnectionsDamaged || '';
                  mappedChecklistData.deformationOfHookOrFittings = step2Checklist.metalFitting.deformationOfHookOrFittings || '';
                }
                
                // Lanyard items
                if (step2Checklist.lanyard) {
                  mappedChecklistData.lanyard = step2Checklist.lanyard.lanyard || '';
                  mappedChecklistData.lanyardMarkedShockAbsorber = step2Checklist.lanyard.lanyardMarkedShockAbsorber || '';
                  mappedChecklistData.shockAbsorberHousedProperly = step2Checklist.lanyard.shockAbsorberHousedProperly || '';
                  mappedChecklistData.lanyardShowingSignsOfWear = step2Checklist.lanyard.lanyardShowingSignsOfWear || '';
                }
                
                console.log(`📋 Auto-filling ${selectedMonth} with Step 2 data:`, mappedChecklistData);
                
                return {
                  ...register,
                  checklistStatus: {
                    ...register.checklistStatus,
                    [selectedMonth]: mappedChecklistData
                  }
                };
              }
              return register;
            })
          );
          
          Toast.show({
            type: 'success',
            text1: 'Step 2 Data Loaded',
            text2: 'Checklist auto-filled from Step 2',
            position: 'bottom',
          });
        }
      }
    } catch (error) {
      console.error('❌ Error loading Step 2 data:', error);
    }
  };

  const loadExistingDataFromMongoDB = async () => {
    try {
      console.log('🔄 Checking MongoDB for existing data...');
      const userId = await getAdminUserId();
      const response = await HSEInspectionAPI.getHSEInspections(userId);
      
      if (response.success && response.data && response.data.length > 0) {
        const reports = response.data;
        console.log('📊 Found reports:', reports.length);
        
        // Collect all historical data
        let allHistoricalData = {};
        
        reports.forEach(reportData => {
          if (reportData.step3 && reportData.step3.inspectionRegister) {
            reportData.step3.inspectionRegister.forEach(register => {
              if (register.checklistStatus) {
                Object.keys(register.checklistStatus).forEach(month => {
                  const monthData = register.checklistStatus[month];
                  const hasData = Object.values(monthData).some(value => 
                    value === 'Yes' || value === 'No' || value === 'N/A'
                  );
                  
                  if (hasData) {
                    if (!allHistoricalData[month]) {
                      allHistoricalData[month] = {};
                    }
                    allHistoricalData[month] = { ...allHistoricalData[month], ...monthData };
                    console.log(`📋 Added historical data for ${month}:`, monthData);
                  }
                });
              }
            });
          }
        });
        
        // Update inspection register with all historical data
        if (Object.keys(allHistoricalData).length > 0) {
          const updatedInspectionRegister = inspectionRegister.map(register => ({
            ...register,
            checklistStatus: {
              ...register.checklistStatus,
              ...allHistoricalData
            }
          }));
          
          setInspectionRegister(updatedInspectionRegister);
          console.log('✅ Updated inspection register with historical data');
        }
      } else {
        console.log('❌ No reports found in database');
      }
    } catch (error) {
      console.error('❌ Error loading existing data from MongoDB:', error);
    }
  };

  const getMonthStatus = (monthKey) => {
    if (!inspectionRegister[0]) return { hasData: false, isSelected: false, isLatestMonth: false, isPreviousMonth: false };
    
    const register = inspectionRegister[0];
    const hasData = Object.values(register.checklistStatus[monthKey] || {}).some(value => 
      value === 'Yes' || value === 'No' || value === 'N/A'
    );
    const isSelected = register.monthlyStatus[monthKey] === 'OK';
    
    // Check if this is the latest present month (current month)
    const currentDate = new Date();
    const currentMonth = currentDate.getMonth(); // 0-11
    const monthMap = {
      'january': 0, 'february': 1, 'march': 2, 'april': 3, 'may': 4, 'june': 5,
      'july': 6, 'august': 7, 'september': 8, 'october': 9, 'november': 10, 'december': 11
    };
    const isLatestMonth = monthMap[monthKey] === currentMonth;
    
    // Check if this is a previous month (before current month)
    const isPreviousMonth = monthMap[monthKey] < currentMonth;
    
    return { hasData, isSelected, isLatestMonth, isPreviousMonth };
  };

  const renderMonthButton = (month) => {
    const { hasData, isSelected, isLatestMonth, isPreviousMonth } = getMonthStatus(month.key);
    
    let buttonStyle = [styles.monthItem];
    let labelStyle = [styles.monthLabel];
    let statusStyle = [styles.monthStatus];
    
    if (isPreviousMonth && hasData) {
      // Red background for previous months with data
      buttonStyle.push(styles.monthItemWithPreviousData);
      labelStyle.push(styles.monthLabelWithPreviousData);
      statusStyle.push(styles.monthStatusWithPreviousData);
    } else if (isLatestMonth) {
      // Blue background for latest month (auto-selected)
      buttonStyle.push(styles.monthItemLatest);
      labelStyle.push(styles.monthLabelLatest);
      statusStyle.push(styles.monthStatusLatest);
    } else if (hasData) {
      // Green background for current/future months with data
      buttonStyle.push(styles.monthItemWithData);
      labelStyle.push(styles.monthLabelWithData);
      statusStyle.push(styles.monthStatusWithData);
    } else if (isSelected) {
      // Blue background for manually selected month
      buttonStyle.push(styles.monthItemSelected);
      labelStyle.push(styles.monthLabelSelected);
      statusStyle.push(styles.monthStatusSelected);
    }
    
    return (
      <TouchableOpacity
        key={month.key}
        style={buttonStyle}
        onPress={() => {
          // All months are non-selectable
          console.log('Month selection disabled - all months are read-only');
        }}
        disabled={true} // All months are disabled
      >
        <Text style={labelStyle}>
          {month.label}
        </Text>
        <Text style={statusStyle}>
          {isPreviousMonth && hasData ? '✓' : (hasData ? '✓' : (isSelected ? 'OK' : (isLatestMonth ? 'Current' : '')))}
        </Text>
      </TouchableOpacity>
    );
  };

  const saveStep3Data = async () => {
    try {
      const step3Data = {
        formData,
        timestamp: new Date().toISOString(),
      };
      await AsyncStorage.setItem('hseStep3', JSON.stringify(step3Data));
    } catch (error) {
      console.error('Error saving Step 3 data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data.',
        position: 'bottom',
      });
    }
  };

  const handleInputChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
    saveStep3Data();
  };

  const handleDeviationChange = (id, field, value) => {
    setDeviations(prev => 
      prev.map(deviation => 
        deviation.id === id 
          ? { ...deviation, [field]: value }
          : deviation
      )
    );
    saveStep3Data();
  };

  const handleInspectionRegisterChange = (id, field, value) => {
    setInspectionRegister(prev => 
      prev.map(register => 
        register.id === id 
          ? { ...register, [field]: value }
          : register
      )
    );
    saveStep3Data();
  };

  const handleMonthlyStatusChange = (registerId, month, value) => {
    setInspectionRegister(prev => 
      prev.map(register => 
        register.id === registerId 
          ? { 
              ...register, 
              monthlyStatus: { 
                ...register.monthlyStatus, 
                [month]: value 
              } 
            }
          : register
      )
    );
    saveStep3Data();
  };

  const loadPreviousMonthData = async (registerId, selectedMonth) => {
    try {
      setIsLoadingData(true);
      console.log(`Loading previous data for month: ${selectedMonth}`);
      
      const userId = await getAdminUserId();
      const response = await HSEInspectionAPI.getHSEInspections(userId);
      
      if (response.success && response.data && response.data.length > 0) {
        const reports = response.data;
        let foundData = false;
        
        // Look through all reports for data for this specific month
        reports.forEach(reportData => {
          if (reportData.step3 && reportData.step3.inspectionRegister) {
            reportData.step3.inspectionRegister.forEach(register => {
              if (register.checklistStatus && register.checklistStatus[selectedMonth]) {
                const monthData = register.checklistStatus[selectedMonth];
                const hasData = Object.values(monthData).some(value => 
                  value === 'Yes' || value === 'No' || value === 'N/A'
                );
                
                if (hasData) {
                  console.log(`Found previous data for ${selectedMonth}:`, monthData);
                  
                  // Update the current inspection register
                  setInspectionRegister(prev => prev.map(r => 
                    r.id === registerId 
                      ? {
                          ...r,
                          checklistStatus: {
                            ...r.checklistStatus,
                            [selectedMonth]: { ...r.checklistStatus[selectedMonth], ...monthData }
                          }
                        }
                      : r
                  ));
                  
                  foundData = true;
                }
              }
            });
          }
        });
        
        if (foundData) {
          Toast.show({
            type: 'success',
            text1: 'Data Loaded',
            text2: `Loaded previous data for ${selectedMonth}`,
            position: 'bottom',
          });
        } else {
          Toast.show({
            type: 'info',
            text1: 'No Previous Data',
            text2: `No previous data found for ${selectedMonth}`,
            position: 'bottom',
          });
        }
      } else {
        Toast.show({
          type: 'info',
          text1: 'No Reports Found',
          text2: 'No previous reports found',
          position: 'bottom',
        });
      }
    } catch (error) {
      console.error('Error loading previous month data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load previous data',
        position: 'bottom',
      });
    } finally {
      setIsLoadingData(false);
    }
  };

  const handleMonthSelection = async (registerId, month) => {
    console.log(`Month selection: ${month} for register: ${registerId}`);
    
    // Clear all months first
    const updatedRegister = inspectionRegister.find(r => r.id === registerId);
    if (updatedRegister) {
      const newMonthlyStatus = { ...updatedRegister.monthlyStatus };
      
      // Clear all months
      Object.keys(newMonthlyStatus).forEach(m => {
        newMonthlyStatus[m] = '';
      });
      
      // Set selected month to 'OK'
      newMonthlyStatus[month] = 'OK';
      
      // Update the register
      const updatedInspectionRegister = inspectionRegister.map(r => 
        r.id === registerId 
          ? { ...r, monthlyStatus: newMonthlyStatus }
          : r
      );
      
      setInspectionRegister(updatedInspectionRegister);
      
      console.log(`Updated monthly status for ${month}:`, newMonthlyStatus);
      
      // Load Step 2 data for this month
      await loadStep2DataForMonth(registerId, month);
      
      // Load previous month data if available
      await loadPreviousMonthData(registerId, month);
    }
  };

  const loadStep2DataForMonth = async (registerId, month) => {
    try {
      console.log(`🔄 Loading Step 2 data for month: ${month}`);
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      if (savedStep2) {
        const step2Data = JSON.parse(savedStep2);
        
        if (step2Data.checklistData) {
          const step2Checklist = step2Data.checklistData;
          
          // Map Step 2 data to Step 3 format
          const mappedChecklistData = {};
          
          // Webbing items
          if (step2Checklist.webbing) {
            mappedChecklistData.webbing = step2Checklist.webbing.webbing || '';
            mappedChecklistData.dirty = step2Checklist.webbing.dirty || '';
            mappedChecklistData.heatDamage = step2Checklist.webbing.heatDamage || '';
            mappedChecklistData.cutsOrChafing = step2Checklist.webbing.cutsOrChafing || '';
            mappedChecklistData.seamDamaged = step2Checklist.webbing.seamDamaged || '';
            mappedChecklistData.anyOtherPhysicalDamage = step2Checklist.webbing.anyOtherPhysicalDamage || '';
          }
          
          // Metal Fitting items
          if (step2Checklist.metalFitting) {
            mappedChecklistData.metalFitting = step2Checklist.metalFitting.metalFitting || '';
            mappedChecklistData.couplingsDamaged = step2Checklist.metalFitting.couplingsDamaged || '';
            mappedChecklistData.shacklesDamaged = step2Checklist.metalFitting.shacklesDamaged || '';
            mappedChecklistData.clipsDamaged = step2Checklist.metalFitting.clipsDamaged || '';
            mappedChecklistData.lifelineConnectionsDamaged = step2Checklist.metalFitting.lifelineConnectionsDamaged || '';
            mappedChecklistData.lanyardConnectionsDamaged = step2Checklist.metalFitting.lanyardConnectionsDamaged || '';
            mappedChecklistData.deformationOfHookOrFittings = step2Checklist.metalFitting.deformationOfHookOrFittings || '';
          }
          
          // Lanyard items
          if (step2Checklist.lanyard) {
            mappedChecklistData.lanyard = step2Checklist.lanyard.lanyard || '';
            mappedChecklistData.lanyardMarkedShockAbsorber = step2Checklist.lanyard.lanyardMarkedShockAbsorber || '';
            mappedChecklistData.shockAbsorberHousedProperly = step2Checklist.lanyard.shockAbsorberHousedProperly || '';
            mappedChecklistData.lanyardShowingSignsOfWear = step2Checklist.lanyard.lanyardShowingSignsOfWear || '';
          }
          
          console.log(`📋 Auto-filling ${month} with Step 2 data:`, mappedChecklistData);
          
          // Update the specific register with Step 2 data
          setInspectionRegister(prev => 
            prev.map(register => 
              register.id === registerId 
                ? {
                    ...register,
                    checklistStatus: {
                      ...register.checklistStatus,
                      [month]: mappedChecklistData
                    }
                  }
                : register
            )
          );
          
          Toast.show({
            type: 'success',
            text1: 'Step 2 Data Applied',
            text2: `Checklist auto-filled for ${month}`,
            position: 'bottom',
          });
        }
      }
    } catch (error) {
      console.error('❌ Error loading Step 2 data for month:', error);
    }
  };

  const handleChecklistStatusChange = (registerId, month, itemKey, value) => {
    console.log(`📝 Updating checklist: Register ${registerId}, Month ${month}, Item ${itemKey}, Value ${value}`);
    
    setInspectionRegister(prev => {
      const updated = prev.map(register => 
        register.id === registerId 
          ? { 
              ...register, 
              checklistStatus: { 
                ...register.checklistStatus, 
                [month]: {
                  ...register.checklistStatus[month],
                  [itemKey]: value
                }
              } 
            }
          : register
      );
      
      // Log the updated data
      const updatedRegister = updated.find(r => r.id === registerId);
      console.log(`✅ Updated Register ${registerId} - ${month} data:`, updatedRegister?.checklistStatus[month]);
      
      return updated;
    });
    
    saveStep3Data();
  };

  const addInspectionRegister = () => {
    if (inspectionRegister.length >= 20) {
      Toast.show({
        type: 'info',
        text1: 'Maximum Limit',
        text2: 'You can add up to 20 inspection entries.',
        position: 'bottom',
      });
      return;
    }

    const newId = Math.max(...inspectionRegister.map(r => r.id)) + 1;
    const newRegister = {
      id: newId,
      idNo: '',
      locationOfHarness: '',
      typeRating: '',
      site: '',
      year: new Date().getFullYear(),
      monthlyStatus: {
        january: '',
        february: '',
        march: '',
        april: '',
        may: '',
        june: '',
        july: '',
        august: '',
        september: '',
        october: '',
        november: '',
        december: ''
      },
      checklistStatus: {
        january: {},
        february: {},
        march: {},
        april: {},
        may: {},
        june: {},
        july: {},
        august: {},
        september: {},
        october: {},
        november: {},
        december: {}
      }
    };
    setInspectionRegister(prev => [...prev, newRegister]);
    saveStep3Data();
  };

  const removeInspectionRegister = (id) => {
    if (inspectionRegister.length <= 1) {
      Toast.show({
        type: 'error',
        text1: 'Cannot Remove',
        text2: 'At least one inspection entry is required.',
        position: 'bottom',
      });
      return;
    }

    Alert.alert(
      'Remove Entry',
      'Are you sure you want to remove this inspection entry?',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Remove',
          style: 'destructive',
          onPress: () => {
            setInspectionRegister(prev => prev.filter(r => r.id !== id));
            saveStep3Data();
          }
        }
      ]
    );
  };

  const addDeviation = () => {
    if (deviations.length >= 16) {
      Toast.show({
        type: 'info',
        text1: 'Maximum Limit',
        text2: 'You can add up to 16 deviations.',
        position: 'bottom',
      });
      return;
    }

    const newId = Math.max(...deviations.map(d => d.id)) + 1;
    const newDeviation = {
      id: newId,
      date: new Date(),
      descriptionOfDeviationNoted: '',
      actionByWhom: '',
      targetDate: new Date(),
      complete: '',
    };
    setDeviations(prev => [...prev, newDeviation]);
    saveStep3Data();
  };

  const removeDeviation = (id) => {
    if (deviations.length <= 1) {
      Toast.show({
        type: 'error',
        text1: 'Cannot Remove',
        text2: 'At least one deviation entry is required.',
        position: 'bottom',
      });
      return;
    }

    Alert.alert(
      'Remove Deviation',
      'Are you sure you want to remove this deviation entry?',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Remove',
          style: 'destructive',
          onPress: () => {
            setDeviations(prev => prev.filter(d => d.id !== id));
            saveStep3Data();
          }
        }
      ]
    );
  };

  const showDatePickerModal = (field, deviationId = null) => {
    setSelectedDateField(field);
    setSelectedDeviationId(deviationId);
    setShowDatePicker(true);
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      if (selectedDeviationId) {
        // Update deviation date
        handleDeviationChange(selectedDeviationId, selectedDateField, selectedDate);
      } else {
        // Update form data date
        handleInputChange(selectedDateField, selectedDate);
      }
    }
  };

  const formatDate = (date) => {
    if (!date) return 'Select Date';
    if (typeof date === 'string') date = new Date(date);
    return date.toLocaleDateString('en-GB');
  };

  const getAdminUserId = async () => {
    const userId = await AsyncStorage.getItem('userUid');
    if (!userId) throw new Error('User ID not found. Please log in again.');
    return userId;
  };

  const validateForm = () => {
    if (!formData.checkedByInitials.trim()) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Please enter CHECKED BY initials.',
        position: 'bottom',
      });
      return false;
    }
    if (!formData.dateOfInspectionInitials.trim()) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Please enter Date of inspection initials.',
        position: 'bottom',
      });
      return false;
    }
    if (!formData.managementOverInspectionInitials.trim()) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Please enter Management over inspection initials.',
        position: 'bottom',
      });
      return false;
    }
    return true;
  };

  const generatePDFWithHtml = async (report) => {
    try {
      console.log('=== Starting PDF Generation ===');
      
      const step2 = report.step2;
      const step3 = report.step3;
      console.log('Step 2 data for PDF:', step2);
      console.log('Step 3 data for PDF:', step3);
      
      // Get Step 2 checklist data
      const step2Checklist = step2?.checklistData || {};
      console.log('Step 2 checklist data:', step2Checklist);
      
      // Get Step 3 form data
      const step3FormData = step3?.formData || {};
      console.log('Step 3 form data:', step3FormData);
      
      // Get request number from Step 1
      const requestNumber = report?.step1?.requestNumber || 'N/A';
      console.log('Request number:', requestNumber);
      
      // Generate inspection register rows using Step 2 data
      let inspectionRegisterRows = '';
      
      // Define 131 Fire Extinguisher checklist items based on Step 2 structure
      const checklistItems = [
        { 
          key: 'corrosionOrDamage', 
          label: 'Corrosion or Damage',
          category: 'webbing',
          step2Key: 'corrosionOrDamage'
        },
        { 
          key: 'safetyPinInPlace', 
          label: 'Safety Pin in Place',
          category: 'webbing',
          step2Key: 'safetyPinInPlace'
        },
        { 
          key: 'antiTamperSeal', 
          label: 'Anti-Tamper Seal',
          category: 'webbing',
          step2Key: 'antiTamperSeal'
        },
        { 
          key: 'perishedOrBlockedHoseExt', 
          label: 'Perished/Blocked Hose (Extinguishers)',
          category: 'webbing',
          step2Key: 'perishedOrBlockedHose'
        },
        { 
          key: 'leakingGlandsOrSeals', 
          label: 'Leaking Glands or Seals',
          category: 'metalFitting',
          step2Key: 'leakingGlandsOrSeals'
        },
        { 
          key: 'perishedOrBlockedHoseReel', 
          label: 'Perished/Blocked Hose (Reels)',
          category: 'metalFitting',
          step2Key: 'perishedOrBlockedHose'
        },
        { 
          key: 'nozzleMissingOrBlocked', 
          label: 'Nozzle Missing/Blocked',
          category: 'metalFitting',
          step2Key: 'nozzleMissingOrBlocked'
        },
      ];

      console.log('Processing checklist items:', checklistItems);
      
      // Generate one row per checklist item
      checklistItems.forEach((item, itemIndex) => {
        // Get the value from Step 2 data
        const step2Value = step2Checklist[item.category]?.[item.step2Key] || '';
        console.log(`Item ${item.key}: Step 2 value = ${step2Value}`);
        
        // Convert Step 2 values to display format
        let displayValue = '–'; // Default dash for empty/NA
        if (step2Value === 'Yes') displayValue = '✓';
        else if (step2Value === 'No') displayValue = '✗';
        else if (step2Value === 'N/A') displayValue = 'N/A';
        
        inspectionRegisterRows += `
          <tr>
            <td>${itemIndex + 1}</td>
            <td>${item.label}</td>
            <td colspan="12" style="text-align: center; font-weight: bold; color: #015185;">${displayValue}</td>
          </tr>
        `;
      });
      
      console.log('Generated inspection register rows:', inspectionRegisterRows);

      console.log('PDF Generation - Final HTML Rows:', inspectionRegisterRows);

      // Generate deviations rows
      let deviationsRows = '';
      if (step3.deviations && step3.deviations.length > 0) {
        step3.deviations.forEach((deviation) => {
          deviationsRows += `
            <tr>
              <td>${deviation.date ? new Date(deviation.date).toLocaleDateString() : ''}</td>
              <td>${deviation.descriptionOfDeviationNoted || ''}</td>
              <td>${deviation.actionByWhom || ''}</td>
              <td>${deviation.targetDate ? new Date(deviation.targetDate).toLocaleDateString() : ''}</td>
              <td>${deviation.complete || ''}</td>
            </tr>
          `;
        });
      }

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8" />
          <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
          <title>Fire Extinguishers Monthly Checklist</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 20px;
            }
            .header-row {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              margin-bottom: 10px;
            }
            .logo-left {
              display: flex;
              align-items: flex-start;
            }
            .logo-left img {
              width: 140px;
              height: 50px;
              object-fit: contain;
            }
            .doc-info-box {
              width: 340px;
            }
            .doc-info-box table {
              width: 100%;
              border: 1px solid #222;
              border-collapse: collapse;
              font-size: 12px;
            }
            .doc-info-box td {
              border: 1px solid #222;
              padding: 4px 8px;
              text-align: left;
            }
            .title-section {
              text-align: center;
              margin: 10px 0 0 0;
            }
            .title-section h2 {
              margin: 0;
              font-size: 18px;
              font-weight: bold;
              text-decoration: underline;
            }
            .title-section h3 {
              margin: 4px 0;
              font-size: 14px;
              font-weight: normal;
            }
            .main-table {
              width: 100%;
              border-collapse: collapse;
              margin-bottom: 20px;
            }
            .main-table th,
            .main-table td {
              border: 1px solid #000;
              padding: 5px;
              text-align: center;
            }
            .main-table th {
              background-color: #f2f2f2;
            }
            .main-table .year-header {
              width: 50%;
            }
            .footer-table {
              width: 100%;
              border-collapse: collapse;
            }
            .footer-table th,
            .footer-table td {
              border: 1px solid #000;
              padding: 5px;
              text-align: center;
            }
            .footer-table .date-inspected {
              width: 50%;
            }
            .note {
              font-size: 10px;
              text-align: right;
              margin-top: 5px;
            }
            .deviations-table {
              width: 100%;
              border-collapse: collapse;
              margin-bottom: 20px;
            }
            .deviations-table th,
            .deviations-table td {
              border: 1px solid #000;
              padding: 5px;
              text-align: center;
            }
            .deviations-table th {
              background-color: #f2f2f2;
            }
          </style>
        </head>
        <body>
          <div class="header-row">
            <div class="logo-left">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="TITAN DRILLING Logo"/>
            </div>
            <div class="doc-info-box">
              <table>
                <tr><td>Document Name</td><td>HSE-FOR-8.1 Fire Extinguisher Checklist</td></tr>
                <tr><td>Document ID</td><td>HSE-131</td></tr>
                <tr><td>Version Date</td><td>10/03/2024</td></tr>
                <tr><td>Version</td><td>1</td></tr>
                <tr><td>Approved</td><td>HSM</td></tr>
                <tr><td>Request #</td><td>${requestNumber}</td></tr>
              </table>
            </div>
          </div>

          <div class="title-section">
            <h2>Fire Extinguishers Monthly Checklist</h2>
            <h3>Inspection Register and Deviations</h3>
          </div>

          <table class="main-table">
            <tr>
              <th>S. No.</th>
              <th>Inspection Item</th>
              <th colspan="12" class="year-header">Year: ${new Date().getFullYear()}</th>
            </tr>
            <tr>
              <td>1</td>
              <td></td>
              <td>Jan</td>
              <td>Feb</td>
              <td>Mar</td>
              <td>Apr</td>
              <td>May</td>
              <td>Jun</td>
              <td>Jul</td>
              <td>Aug</td>
              <td>Sep</td>
              <td>Oct</td>
              <td>Nov</td>
              <td>Dec</td>
            </tr>
            ${inspectionRegisterRows}
          </table>

          <table class="footer-table">
            <tr>
              <td colspan="6">CHECKED BY (initials)</td>
              <td colspan="8">${step3FormData?.checkedByInitials || ''}</td>
            </tr>
            <tr>
              <td colspan="6">Date of inspection (initials)</td>
              <td colspan="8">${step3FormData?.dateOfInspectionInitials || ''}</td>
            </tr>
            <tr>
              <td colspan="6">Date of inspection (review date)</td>
              <td colspan="8">${step3FormData?.dateOfInspectionReviewDate ? new Date(step3FormData.dateOfInspectionReviewDate).toLocaleDateString() : ''}</td>
            </tr>
            <tr>
              <td colspan="6">Management over inspection (initials)</td>
              <td colspan="8">${step3FormData?.managementOverInspectionInitials || ''}</td>
            </tr>
            <tr>
              <td colspan="6">Management over inspection (review date)</td>
              <td colspan="8">${step3FormData?.managementOverInspectionReviewDate ? new Date(step3FormData.managementOverInspectionReviewDate).toLocaleDateString() : ''}</td>
            </tr>
          </table>

          <table class="deviations-table">
            <tr>
              <th>Date</th>
              <th>Description of deviation noted</th>
              <th>Action by whom</th>
              <th>Target date</th>
              <th>Complete Y/N</th>
            </tr>
            ${deviationsRows}
          </table>

          <div class="note">
            Note: ✓ = Yes, ✗ = No, – = N/A or Empty. Data from Step 2 inspection checklist.
          </div>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `Fire_Extinguisher_Inspection_Report_${report.id}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const uploadPDFToMongoDB = async (filePath, userId) => {
    try {
      console.log('📤 Uploading PDF to MongoDB backend...');
      const pdfFile = {
        uri: `file://${filePath}`,
        type: 'application/pdf',
        name: `hse_131_${userId}_${Date.now()}.pdf`
      };

      const uploadResponse = await HSEInspectionAPI.uploadPDF(userId, pdfFile);
      console.log('📥 Upload response:', uploadResponse);
      
      if (!uploadResponse || !uploadResponse.success) {
        const errorMsg = uploadResponse?.error || 'Unknown upload error';
        console.error('❌ Upload failed with response:', uploadResponse);
        throw new Error(`PDF upload failed: ${errorMsg}`);
      }

      const pdfDownloadUrl = uploadResponse.downloadUrl;
      console.log('✅ PDF uploaded, download URL:', pdfDownloadUrl);
      return pdfDownloadUrl;
    } catch (error) {
      console.error('❌ PDF upload failed:', error);
      throw new Error(`PDF upload failed: ${error.message}`);
    }
  };

  const flushStepData = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hseStep2');
      await AsyncStorage.removeItem('hseStep3');
      await AsyncStorage.removeItem(`hseNotes_${hseId}`);
      await AsyncStorage.removeItem(`hseResults_${hseId}`);
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const verifyDataForPDF = (step3, step2) => {
    console.log('=== PDF Data Verification ===');
    
    // Check Step 2 checklist data
    if (!step2?.checklistData) {
      console.log('❌ No Step 2 checklist data found');
      return false;
    }

    const step2Data = step2.checklistData;
    console.log('✅ Step 2 data found:', step2Data);
    
    // Check if any category has data
    const hasWebbingData = step2Data.webbing && Object.values(step2Data.webbing).some(value => 
      value === 'Yes' || value === 'No' || value === 'N/A'
    );
    
    const hasMetalFittingData = step2Data.metalFitting && Object.values(step2Data.metalFitting).some(value => 
      value === 'Yes' || value === 'No' || value === 'N/A'
    );
    
    const hasLanyardData = step2Data.lanyard && Object.values(step2Data.lanyard).some(value => 
      value === 'Yes' || value === 'No' || value === 'N/A'
    );
    
    console.log(`📋 Data summary: Webbing: ${hasWebbingData}, Metal Fitting: ${hasMetalFittingData}, Lanyard: ${hasLanyardData}`);
    
    if (!hasWebbingData && !hasMetalFittingData && !hasLanyardData) {
      console.log('❌ No checklist data found in any category');
      return false;
    }
    
    // Check Step 3 form data
    if (!step3?.formData) {
      console.log('❌ No Step 3 form data found');
      return false;
    }
    
    const formData = step3.formData;
    const requiredFields = ['checkedByInitials', 'dateOfInspectionInitials', 'managementOverInspectionInitials'];
    
    const missingFields = requiredFields.filter(field => !formData[field] || formData[field].trim() === '');
    
    if (missingFields.length > 0) {
      console.log(`❌ Missing required fields: ${missingFields.join(', ')}`);
      return false;
    }
    
    console.log('✅ All required data is present');
    return true;
  };

  const handleSubmit = async () => {
    if (!validateForm()) {
      return;
    }

    // Verify data before PDF generation
    const step3Data = {
      formData,
      deviations,
      inspectionRegister,
    };
    
    console.log('=== Submitting Report ===');
    // Data verification will be done in the PDF generation function

    setSyncModalVisible(true);
    setUploadProgress(5);
    
    try {
      const userId = await getAdminUserId();
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      
      if (!savedStep1 || !savedStep2) {
        throw new Error('Previous step data is missing');
      }
      
      const step1 = JSON.parse(savedStep1);
      const step2 = JSON.parse(savedStep2);

      const newReportId = step1?.requestNumber || Date.now().toString();
      setReportId(newReportId);

      const reportData = {
        request: newReportId,
        requestNumber: newReportId,
        id: newReportId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        step1: step1,
        step2: step2,
        step3: step3Data,
      };

      console.log('=== Final Report Data ===');
      verifyDataForPDF(reportData.step3, reportData.step2);

      setUploadProgress(20);

      // Generate PDF
      console.log('📄 Generating PDF for HSE-131...');
      let pdfPath;
      try {
        pdfPath = await generatePDFWithHtml(reportData);
        console.log('✅ PDF generated at:', pdfPath);
      } catch (pdfError) {
        console.error('❌ PDF generation failed:', pdfError);
        throw new Error(`PDF generation failed: ${pdfError.message}`);
      }

      setUploadProgress(50);

      // Upload PDF to MongoDB backend
      let pdfDownloadUrl;
      try {
        pdfDownloadUrl = await uploadPDFToMongoDB(pdfPath, userId);
      } catch (uploadError) {
        console.error('❌ PDF upload failed:', uploadError);
        throw new Error(`PDF upload failed: ${uploadError.message}`);
      }

      setUploadProgress(70);

      // Save complete report to MongoDB
      console.log('💾 Saving report to MongoDB...');
      
      const completeReportData = {
        _id: newReportId,
        userId,
        requestNumber: newReportId,
        documentType: 'HSE-131',
        documentName: 'Fire Extinguisher Checklist',
        step1: step1,
        step2: step2,
        step3: step3Data,
        pdfDownloadUrl,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        status: 'completed',
        createdBy: userId,
        lastModifiedBy: userId
      };

      console.log('📋 Report data to save:', JSON.stringify(completeReportData, null, 2));
      
      let saveResponse;
      try {
        saveResponse = await HSEInspectionAPI.createHSEInspection(userId, completeReportData);
        console.log('📥 Save response:', saveResponse);
      } catch (saveError) {
        console.error('❌ MongoDB save failed:', saveError);
        throw new Error(`MongoDB save failed: ${saveError.message}`);
      }
      
      if (!saveResponse || !saveResponse.success) {
        const errorMsg = saveResponse?.error || 'Unknown save error';
        console.error('❌ Save failed with response:', saveResponse);
        throw new Error(`MongoDB save failed: ${errorMsg}`);
      }

      setUploadProgress(90);

      // Send HSE inspection notifications
      console.log('📧 Sending HSE inspection notifications...');
      try {
        const userData = await AsyncStorage.getItem('userData');
        const user = userData ? JSON.parse(userData) : null;
        const createdByName = user?.name || formData.checkedByInitials || 'Unknown Inspector';
        const inspectorEmail = user?.email || '';
        
        const notificationPayload = {
          hseInspectionData: {
            _id: newReportId,
            id: newReportId,
            requestNumber: newReportId,
            checklistName: 'HSE-131 Fire Extinguisher Checklist',
            country: step1.country || '',
            project: step1.project || '',
            status: 'completed',
          },
          createdByName,
          inspectorEmail,
          pdfUrl: pdfDownloadUrl
        };
        
        const notifResponse = await fetch('http://168.231.113.219:5028/api/hse-inspection-notifications/new-inspection', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(notificationPayload),
        });
        
        if (notifResponse.ok) {
          console.log('✅ HSE notifications sent successfully');
        }
      } catch (notifError) {
        console.error('❌ Error sending HSE notifications:', notifError);
      }

      // Delete temp PDF file
      await RNFS.unlink(pdfPath).catch((err) => console.warn('⚠️ Error deleting temp PDF:', err));

      // Clear local storage
      await flushStepData(newReportId);
      setUploadProgress(100);

      console.log('✅ HSE-131 Fire Extinguisher Report saved to MongoDB:', newReportId);

      if (isMounted.current) {
        setSyncModalVisible(false);
        setShowSuccessModal(true);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'HSE-131 report saved to MongoDB successfully.',
          position: 'bottom',
        });
      }
    } catch (error) {
      console.error('❌ Error in handleSubmit:', error);
      console.error('❌ Error details:', error.message);
      if (isMounted.current) {
        setSyncModalVisible(false);
        setUploadProgress(0);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to save report to MongoDB.',
          position: 'bottom',
          visibilityTime: 4000,
        });
      }
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
      }
    }
  };

  const handleBackPress = () => {
    Alert.alert(
      'Go Back',
      'Do you want to return to the previous step? Unsaved changes will be saved locally.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'OK',
          onPress: async () => {
            await saveStep3Data();
            setActiveStep(2);
          },
        },
      ],
      { cancelable: false }
    );
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'Report ID not available.' });
      return;
    }

    try {
      const response = await HSEInspectionAPI.getHSEInspectionById(reportId);

      if (!response.success || !response.data || !response.data.pdfDownloadUrl) {
        Toast.show({ type: 'error', text1: 'Error', text2: 'PDF not available in MongoDB.' });
        return;
      }

      // Close modal first
      setShowSuccessModal(false);
      
      // Navigate to HSEViewPDF screen
      navigation.navigate('HSEViewPDF', {
        pdfUrl: response.data.pdfDownloadUrl,
        onGoBack: () => {
          navigation.navigate('HSEInspectionHome');
        }
      });
    } catch (error) {
      console.error('❌ Error fetching PDF URL from MongoDB:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Could not fetch PDF from MongoDB.' });
    }
  };

  const renderInputField = (label, value, onChangeText, placeholder = '', keyboardType = 'default') => (
    <View style={styles.inputContainer}>
      <Text style={styles.inputLabel}>{label}</Text>
      <TextInput
        style={styles.textInput}
        value={value}
        onChangeText={onChangeText}
        placeholder={placeholder}
        keyboardType={keyboardType}
        placeholderTextColor="#888"
      />
    </View>
  );

  const renderDateField = (label, value, onPress) => (
    <View style={styles.inputContainer}>
      <Text style={styles.inputLabel}>{label}</Text>
      <TouchableOpacity style={styles.dateButton} onPress={onPress}>
        <Text style={styles.dateButtonText}>{formatDate(value)}</Text>
        <MaterialIcons name="calendar-today" size={20} color="#015185" />
      </TouchableOpacity>
    </View>
  );

  const renderDeviationRow = (deviation, index) => (
    <View key={deviation.id} style={styles.deviationRow}>
      <View style={styles.deviationHeader}>
        <Text style={styles.deviationNumber}>Deviation {index + 1}</Text>
        {deviations.length > 1 && (
          <TouchableOpacity
            style={styles.removeButton}
            onPress={() => removeDeviation(deviation.id)}
          >
            <MaterialIcons name="delete" size={20} color="#f44336" />
          </TouchableOpacity>
        )}
      </View>
      
      <View style={styles.deviationContent}>
        <View style={styles.deviationField}>
          <Text style={styles.deviationLabel}>Date</Text>
          <TouchableOpacity 
            style={styles.deviationDateButton}
            onPress={() => showDatePickerModal('date', deviation.id)}
          >
            <Text style={styles.deviationDateText}>{formatDate(deviation.date)}</Text>
            <MaterialIcons name="calendar-today" size={16} color="#015185" />
          </TouchableOpacity>
        </View>

        <View style={styles.deviationField}>
          <Text style={styles.deviationLabel}>Description of deviation noted</Text>
          <TextInput
            style={styles.deviationTextInput}
            value={deviation.descriptionOfDeviationNoted}
            onChangeText={(text) => handleDeviationChange(deviation.id, 'descriptionOfDeviationNoted', text)}
            placeholder="Enter description"
            multiline
            numberOfLines={3}
            placeholderTextColor="#888"
          />
        </View>

        <View style={styles.deviationField}>
          <Text style={styles.deviationLabel}>Action by whom</Text>
          <TextInput
            style={styles.deviationTextInput}
            value={deviation.actionByWhom}
            onChangeText={(text) => handleDeviationChange(deviation.id, 'actionByWhom', text)}
            placeholder="Enter responsible person"
            placeholderTextColor="#888"
          />
        </View>

        <View style={styles.deviationField}>
          <Text style={styles.deviationLabel}>Target date</Text>
          <TouchableOpacity 
            style={styles.deviationDateButton}
            onPress={() => showDatePickerModal('targetDate', deviation.id)}
          >
            <Text style={styles.deviationDateText}>{formatDate(deviation.targetDate)}</Text>
            <MaterialIcons name="calendar-today" size={16} color="#015185" />
          </TouchableOpacity>
        </View>

        <View style={styles.deviationField}>
          <Text style={styles.deviationLabel}>Complete Y/N</Text>
          <View style={styles.completeButtons}>
            <TouchableOpacity
              style={[
                styles.completeButton,
                deviation.complete === 'Y' && styles.completeButtonSelected
              ]}
              onPress={() => handleDeviationChange(deviation.id, 'complete', 'Y')}
            >
              <Text style={[
                styles.completeButtonText,
                deviation.complete === 'Y' && styles.completeButtonTextSelected
              ]}>
                Y
              </Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[
                styles.completeButton,
                deviation.complete === 'N' && styles.completeButtonSelected
              ]}
              onPress={() => handleDeviationChange(deviation.id, 'complete', 'N')}
            >
              <Text style={[
                styles.completeButtonText,
                deviation.complete === 'N' && styles.completeButtonTextSelected
              ]}>
                N
              </Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </View>
  );

  const renderInspectionRegisterRow = (register, index) => (
    <View key={register.id} style={styles.registerRow}>
      <View style={styles.registerHeader}>
        <Text style={styles.registerNumber}>Entry {index + 1}</Text>
        {inspectionRegister.length > 1 && (
          <TouchableOpacity
            style={styles.removeButton}
            onPress={() => removeInspectionRegister(register.id)}
          >
            <MaterialIcons name="delete" size={20} color="#f44336" />
          </TouchableOpacity>
        )}
      </View>
      
      <View style={styles.registerContent}>
        <View style={styles.registerField}>
          <Text style={styles.registerLabel}>Site</Text>
          <TextInput
            style={styles.registerTextInput}
            value={register.site}
            onChangeText={(text) => handleInspectionRegisterChange(register.id, 'site', text)}
            placeholder="Enter site"
            placeholderTextColor="#888"
          />
        </View>

        <View style={styles.registerField}>
          <Text style={styles.registerLabel}>Year</Text>
          <TextInput
            style={styles.registerTextInput}
            value={register.year.toString()}
            onChangeText={(text) => handleInspectionRegisterChange(register.id, 'year', parseInt(text) || new Date().getFullYear())}
            placeholder="Enter year"
            keyboardType="numeric"
            placeholderTextColor="#888"
          />
        </View>
      </View>
    </View>
  );

  const getMonthDataStatus = (register) => {
    const monthsWithData = [];
    Object.keys(register.checklistStatus).forEach(month => {
      const hasData = Object.values(register.checklistStatus[month]).some(value => 
        value === 'Yes' || value === 'No' || value === 'N/A'
      );
      if (hasData) {
        monthsWithData.push(month);
      }
    });
    return monthsWithData;
  };

  const renderMonthStatusIndicator = (register) => {
    const monthsWithData = getMonthDataStatus(register);
    if (monthsWithData.length === 0) return null;

    return (
      <View style={styles.dataStatusContainer}>
        <Text style={styles.dataStatusText}>
          📊 Data loaded for: {monthsWithData.map(month => 
            month.charAt(0).toUpperCase() + month.slice(1)
          ).join(', ')}
        </Text>
      </View>
    );
  };

  const renderCurrentChecklistData = (register, selectedMonth) => {
    if (!selectedMonth) return null;
    
    const checklistStatus = register.checklistStatus[selectedMonth] || {};
    const hasData = Object.values(checklistStatus).some(value => 
      value === 'Yes' || value === 'No' || value === 'N/A'
    );
    
    if (!hasData) return null;

    return (
      <View style={styles.currentDataContainer}>
        <Text style={styles.currentDataTitle}>
          📋 Current {selectedMonth.charAt(0).toUpperCase() + selectedMonth.slice(1)} Data:
        </Text>
        {Object.entries(checklistStatus).map(([key, value]) => {
          if (value && (value === 'Yes' || value === 'No' || value === 'N/A')) {
            return (
              <Text key={key} style={styles.currentDataItem}>
                • {key.replace(/([A-Z])/g, ' $1').trim()}: {value}
              </Text>
            );
          }
          return null;
        })}
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
        <View style={styles.section}>
          {renderInputField(
            'CHECKED BY (initials)',
            formData.checkedByInitials,
            (text) => handleInputChange('checkedByInitials', text),
            'Enter initials'
          )}

          {renderInputField(
            'Date of inspection (initials)',
            formData.dateOfInspectionInitials,
            (text) => handleInputChange('dateOfInspectionInitials', text),
            'Enter initials'
          )}

          {renderDateField(
            'Date of inspection (review date)',
            formData.dateOfInspectionReviewDate,
            () => showDatePickerModal('dateOfInspectionReviewDate')
          )}

          {renderInputField(
            'Management over inspection (initials)',
            formData.managementOverInspectionInitials,
            (text) => handleInputChange('managementOverInspectionInitials', text),
            'Enter initials'
          )}

          {renderDateField(
            'Management over inspection (review date)',
            formData.managementOverInspectionReviewDate,
            () => showDatePickerModal('managementOverInspectionReviewDate')
          )}
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.nextButton, syncModalVisible && styles.disabledButton]}
          onPress={handleSubmit}
          disabled={syncModalVisible}
        >
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {showDatePicker && (
        <DateTimePicker
          value={selectedDeviationId 
            ? deviations.find(d => d.id === selectedDeviationId)?.[selectedDateField] || new Date()
            : formData[selectedDateField] || new Date()
          }
          mode="date"
          display="default"
          onChange={handleDateChange}
        />
      )}

      <Modal visible={showSuccessModal} transparent animationType="fade">
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <MaterialIcons name="check-circle" size={64} color="#4CAF50" />
            <Text style={styles.successModalTitle}>Report Submitted Successfully!</Text>
            <Text style={styles.successModalText}>
              Your Fire Extinguisher Inspection Report has been submitted successfully.
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity 
                style={styles.modalButton} 
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.modalButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={[styles.modalButton, { marginLeft: 10 }]} 
                onPress={handleViewPdf}
              >
                <Text style={styles.modalButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <SyncModal 
        visible={syncModalVisible} 
        message={`Processing... ${uploadProgress}%`}
        progress={uploadProgress}
      />

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  headerContainer: {
    alignItems: 'center',
    paddingVertical: 20,
  },
  headerTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 10,
    textAlign: 'center',
  },
  headerSubtitle: {
    fontSize: 16,
    color: '#666',
    marginTop: 5,
    textAlign: 'center',
  },
  section: {
    marginBottom: 25,
  },
  sectionHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
  },
  sectionTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
  },
  inputContainer: {
    marginBottom: 15,
  },
  inputLabel: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 5,
    fontWeight: '500',
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  dateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 12,
    backgroundColor: '#fff',
  },
  dateButtonText: {
    fontSize: 16,
    color: '#333',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#015185',
    paddingHorizontal: 12,
    paddingVertical: 8,
    borderRadius: 5,
    alignSelf: 'flex-start',
    marginTop: 10,
  },
  addButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
    marginLeft: 5,
  },
  deviationRow: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 15,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  deviationHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  deviationNumber: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
  },
  removeButton: {
    padding: 5,
  },
  deviationContent: {
    gap: 10,
  },
  deviationField: {
    marginBottom: 10,
  },
  deviationLabel: {
    fontSize: 14,
    color: '#333',
    marginBottom: 5,
    fontWeight: '500',
  },
  deviationTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 8,
    fontSize: 14,
    backgroundColor: '#fff',
  },
  deviationDateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 8,
    backgroundColor: '#fff',
  },
  deviationDateText: {
    fontSize: 14,
    color: '#333',
  },
  completeButtons: {
    flexDirection: 'row',
    gap: 10,
  },
  completeButton: {
    paddingHorizontal: 20,
    paddingVertical: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#ddd',
    backgroundColor: '#f8f9fa',
    minWidth: 50,
    alignItems: 'center',
  },
  completeButtonSelected: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  completeButtonText: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
  },
  completeButtonTextSelected: {
    color: '#fff',
  },
  registerRow: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 15,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  registerHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  registerNumber: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
  },
  registerContent: {
    gap: 10,
  },
  registerField: {
    marginBottom: 10,
  },
  registerLabel: {
    fontSize: 14,
    color: '#333',
    marginBottom: 5,
    fontWeight: '500',
  },
  registerTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 8,
    fontSize: 14,
    backgroundColor: '#fff',
  },
  monthlySection: {
    marginTop: 10,
  },
  monthlyHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  monthlyLabel: {
    fontSize: 14,
    color: '#333',
    marginBottom: 10,
    fontWeight: '600',
  },
  monthlyScrollView: {
    maxHeight: 80,
  },
  monthlyScrollContent: {
    paddingHorizontal: 10,
  },
  monthItem: {
    width: 60,
    height: 60,
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    marginHorizontal: 4,
    borderWidth: 2,
    borderColor: '#ddd',
  },
  monthItemSelected: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  monthLabel: {
    fontSize: 12,
    color: '#666',
    fontWeight: '500',
    marginBottom: 2,
  },
  monthLabelSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  monthStatus: {
    fontSize: 10,
    color: '#999',
    fontWeight: '600',
  },
  monthStatusSelected: {
    color: '#fff',
  },
  checklistSection: {
    marginTop: 15,
    backgroundColor: '#f0f8ff',
    borderRadius: 8,
    padding: 15,
    borderWidth: 1,
    borderColor: '#015185',
  },
  checklistSectionTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  checklistCategory: {
    marginBottom: 15,
  },
  checklistCategoryTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
    backgroundColor: '#e8f4fd',
    padding: 5,
    borderRadius: 4,
  },
  checklistItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 5,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  checklistItemLabel: {
    fontSize: 12,
    color: '#333',
    flex: 1,
    marginRight: 10,
  },
  checklistOptions: {
    flexDirection: 'row',
    gap: 5,
  },
  checklistOption: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 3,
    backgroundColor: '#f8f9fa',
    borderWidth: 1,
    borderColor: '#ddd',
    minWidth: 35,
    alignItems: 'center',
  },
  checklistOptionSelected: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  checklistOptionText: {
    fontSize: 10,
    color: '#333',
    fontWeight: '500',
  },
  checklistOptionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  disabledButton: {
    opacity: 0.5,
  },
  bottomicon: {
    marginRight: 5,
  },
  bottomicon1: {
    marginLeft: 5,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  modalButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
  },
  progressContainer: {
    marginTop: 15,
    width: '100%',
    alignItems: 'center',
  },
  progressBar: {
    width: '100%',
    height: 8,
    backgroundColor: '#e0e0e0',
    borderRadius: 4,
    overflow: 'hidden',
    marginBottom: 8,
  },
  progressFill: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 4,
  },
  progressText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
  loadDataButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e0e0e0',
    paddingHorizontal: 10,
    paddingVertical: 6,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  loadDataButtonText: {
    fontSize: 12,
    color: '#333',
    marginLeft: 5,
  },
  dataStatusContainer: {
    backgroundColor: '#e0f7fa',
    borderRadius: 5,
    padding: 8,
    marginTop: 10,
    alignSelf: 'flex-start',
    borderWidth: 1,
    borderColor: '#00bcd4',
  },
  dataStatusText: {
    fontSize: 12,
    color: '#00796b',
    fontWeight: '500',
  },
  currentDataContainer: {
    backgroundColor: '#fff',
    borderRadius: 5,
    padding: 10,
    marginTop: 10,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  currentDataTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 5,
  },
  currentDataItem: {
    fontSize: 12,
    color: '#333',
    marginBottom: 2,
  },
  checklistHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  step2Indicator: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 5,
  },
  step2IndicatorText: {
    fontSize: 12,
    color: '#333',
    fontWeight: '500',
  },
  checklistValueDisplay: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 5,
  },
  checklistValueText: {
    fontSize: 12,
    color: '#333',
    fontWeight: '500',
  },
  checklistValueYes: {
    color: '#4CAF50',
  },
  checklistValueNo: {
    color: '#f44336',
  },
  checklistValueNA: {
    color: '#999',
  },
  monthItemWithDataSelected: {
    backgroundColor: '#4CAF50',
    borderColor: '#4CAF50',
  },
  monthLabelSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  monthStatusSelected: {
    color: '#fff',
  },
  monthItemWithData: {
    backgroundColor: '#4CAF50',
    borderColor: '#4CAF50',
  },
  monthLabelWithData: {
    color: '#fff',
    fontWeight: '600',
  },
  monthStatusWithData: {
    color: '#fff',
  },
  monthItemLatest: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  monthLabelLatest: {
    color: '#fff',
    fontWeight: '600',
  },
  monthStatusLatest: {
    color: '#fff',
  },
  monthItemWithPreviousData: {
    backgroundColor: '#ffcccc',
    borderColor: '#ffcccc',
  },
  monthLabelWithPreviousData: {
    color: '#333',
    fontWeight: '600',
  },
  monthStatusWithPreviousData: {
    color: '#333',
  },
});

export default HSE_131Step3;