
import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  TextInput,
  Modal,
  Dimensions,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';

const { width } = Dimensions.get('window');
const { height } = Dimensions.get('window');

const HSE_123Step2 = ({ setActiveStep }) => {
  const [sections, setSections] = useState([]);
  const [expandedSections, setExpandedSections] = useState({});
  const [selections, setSelections] = useState({});
  const [notes, setNotes] = useState({});
  const [inspectionId, setInspectionId] = useState('hse123');
  const [noteModal, setNoteModal] = useState({
    visible: false,
    itemKey: '',
    currentNote: '',
  });

  useEffect(() => {
    loadSavedData();
    setupChecklistData();
  }, []);

  const setupChecklistData = () => {
    const checklistData = [
      {
        key: '150000km',
        name: '150,000 KM',
        subItems: [
          { key: 'engineOilChanged', itemName: 'Engine oil changed', inputType: 'Yes/No' },
          { key: 'engineOilFilterChanged', itemName: 'Engine oil filter changed', inputType: 'Yes/No' },
          { key: 'engineFuelFilterChanged', itemName: 'Engine fuel filter changed', inputType: 'Yes/No' },
          { key: 'airFilter', itemName: 'Air filter', inputType: 'Yes/No' },
          { key: 'radiatorCoolantChecked', itemName: 'Radiator coolant checked', inputType: 'Yes/No' },
        ]
      },
      {
        key: '300000km_or_12_month',
        name: '300,000 KM OR 12 MONTH',
        subItems: [
          { key: 'radiatorCoolantChanged', itemName: 'Radiator coolant changed', inputType: 'Yes/No' },
          { key: 'powerSteeringFluidChecked', itemName: 'Power steering fluid checked', inputType: 'Yes/No' },
        ]
      },
      {
        key: '500000km',
        name: '500,000 KM',
        subItems: [
          { key: 'powerSteeringFluidChanged', itemName: 'Power steering fluid changed', inputType: 'Yes/No' },
          { key: 'windscreenWasherFluidChecked', itemName: 'Windscreen washer fluid checked', inputType: 'Yes/No' },
          { key: 'battery', itemName: 'Battery', inputType: 'Yes/No' },
          { key: 'batteryClampSecured', itemName: 'Battery clamp secured', inputType: 'Yes/No' },
          { key: 'fanAirconBeltsChecked', itemName: 'Fan/aircon belts checked', inputType: 'Yes/No' },
          { key: 'airCompressor', itemName: 'Air compressor', inputType: 'Yes/No' },
          { key: 'idlerBearingsChecked', itemName: 'Idler bearings checked', inputType: 'Yes/No' },
          { key: 'viscousFanFanBladesChecked', itemName: 'Viscous fan/fan blades checked', inputType: 'Yes/No' },
          { key: 'engineBrakeRetarderCheck', itemName: 'Engine brake/retarder check/trailer brake test', inputType: 'Yes/No' },
          { key: 'brakeFluidPedalChecked', itemName: 'Brake fluid/pedal checked (if applicable)', inputType: 'Yes/No' },
          { key: 'clutchFluidPedalChecked', itemName: 'Clutch fluid/pedal checked (if applicable)', inputType: 'Yes/No' },
          { key: 'airconFilterChecked', itemName: 'Aircon filter checked', inputType: 'Yes/No' },
          { key: 'airconFilterChanged', itemName: 'Aircon filter changed', inputType: 'Yes/No' },
          { key: 'airconHeaterWorking', itemName: 'Aircon/heater working', inputType: 'Yes/No' },
          { key: 'engineBayLeaksLooseItemsCheck', itemName: 'Engine bay leaks/loose items check', inputType: 'Yes/No' },
          { key: 'windscreenChecked', itemName: 'Windscreen checked', inputType: 'Yes/No' },
          { key: 'windscreenWipersCheckedChanged', itemName: 'Windscreen wipers checked/changed', inputType: 'Yes/No' },
          { key: 'gearboxOilLevelCheck', itemName: 'Gearbox oil level check', inputType: 'Yes/No' },
        ]
      },
      {
        key: '50000km_or_18_months',
        name: '50,000 KM OR 18 MONTHS',
        subItems: [
          { key: 'gearboxOilChange', itemName: 'Gearbox oil change', inputType: 'Yes/No' },
          { key: 'splitterBoxCheck', itemName: 'Splitter box check (if applicable)', inputType: 'Yes/No' },
          { key: 'transferBoxOilLevelCheck', itemName: 'Transfer box oil level check (if applicable)', inputType: 'Yes/No' },
          { key: 'diffOilLevelCheck', itemName: 'Diff oil level check', inputType: 'Yes/No' },
          { key: 'diffOilChange', itemName: 'Diff oil change', inputType: 'Yes/No' },
          { 
            key: 'kingPinPlayCheck', 
            itemName: 'King pin play check', 
            inputType: 'AxlePositions',
            positions: ['1ST AXLE', 'MID AXLE', 'REAR AXLE']
          },
          { 
            key: 'wheelBearingCheck', 
            itemName: 'Wheel bearing check', 
            inputType: 'AxlePositions',
            positions: ['1ST AXLE', 'MID AXLE', 'REAR AXLE']
          },
          { 
            key: 'brakePadsShoesCheck', 
            itemName: 'Brake pads/shoes check', 
            inputType: 'AxlePercentages',
            positions: ['1ST AXLE %', 'MID AXLE %', 'REAR AXLE %']
          },
        ]
      },
      {
        key: 'air_leaks',
        name: 'AIR LEAKS & GENERAL CHECKS',
        subItems: [
          { key: 'handbrakeShoesCheckAdjustment', itemName: 'Handbrake shoes check/adjustment', inputType: 'Yes/No' },
          { key: 'shockAbsorbersDampersCheck', itemName: 'Shock absorbers/dampers check', inputType: 'Yes/No' },
          { key: 'suspensionSpringsSplayedChecked', itemName: 'Suspension springs splayed/checked', inputType: 'Yes/No' },
          { key: 'hangersChecked', itemName: 'Hangers checked', inputType: 'Yes/No' },
          { key: 'shacklePinsBushes', itemName: 'Shackle pins/bushes', inputType: 'Yes/No' },
          { key: 'tieRodEndBallJointChangeCheck', itemName: 'Tie rod end/ball joint change/check', inputType: 'Yes/No' },
          { 
            key: 'wheelNutsSecure', 
            itemName: 'Wheel nuts secure', 
            inputType: 'TirePositions',
            positions: ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE']
          },
          { 
            key: 'tyreWearPressure', 
            itemName: 'Tyre wear/pressure', 
            inputType: 'TirePositions',
            positions: ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE']
          },
          { 
            key: 'headTailSignalBrakeLightsHazards', 
            itemName: 'Head/tail/signal/brake lights/hazards', 
            inputType: 'TirePositions',
            positions: ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE']
          },
          { key: 'beacon', itemName: 'Beacon', inputType: 'Yes/No' },
          { key: 'horn', itemName: 'Horn', inputType: 'Yes/No' },
          { 
            key: 'doorHandles', 
            itemName: 'Door handles', 
            inputType: 'DoorPositions',
            positions: ['LEFT FRONT', 'RIGHT FRONT', 'RIGHT REAR', 'LEFT REAR']
          },
          { 
            key: 'windowWindingMechanism', 
            itemName: 'Window winding mechanism', 
            inputType: 'WindowPositions',
            positions: ['LEFT FRONT', 'RIGHT FRONT', 'RIGHT REAR', 'LEFT REAR']
          },
          { key: 'exteriorDentsPaintwork', itemName: 'Exterior (dents/paintwork)', inputType: 'Yes/No' },
          { 
            key: 'greased', 
            itemName: 'Greased', 
            inputType: 'GreasePositions',
            positions: ['1ST PROPSHAFT', '2ND PROPSHAFT', 'AXLES', 'SHACKLES']
          },
          { key: 'buggyWhip', itemName: 'Buggy whip', inputType: 'Yes/No' },
          { key: 'fifthWheelTested', itemName: '5th wheel tested (if applicable)', inputType: 'Yes/No' },
          { key: 'chevron', itemName: 'Chevron', inputType: 'Yes/No' },
          { key: 'fireExtinguisherCheck', itemName: 'Fire extinguisher check/pressure check', inputType: 'Yes/No' },
        ]
      }
    ];

    setSections(checklistData);
    
    // Initialize expanded sections
    const initialExpanded = {};
    checklistData.forEach((section) => {
      initialExpanded[section.key] = false;
    });
    setExpandedSections(initialExpanded);
  };

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse123Step2');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setSelections(parsedData.selections || {});
        setNotes(parsedData.notes || {});
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        selections,
        notes,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse123Step2', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const getOptionsForInputType = (inputType) => {
    switch (inputType) {
      case 'Yes/No':
        return ['YES/OK', 'NO/DEF'];
      case 'AxlePositions':
      case 'TirePositions':
      case 'DoorPositions':
      case 'WindowPositions':
      case 'GreasePositions':
      case 'AxlePercentages':
        return ['YES/OK', 'NO/DEF'];
      default:
        return ['YES/OK', 'NO/DEF'];
    }
  };

  const handleSelectOption = async (itemKey, option) => {
    setSelections((prev) => {
      const updatedSelections = {
        ...prev,
        [itemKey]: option,
      };
      saveData();
      return updatedSelections;
    });
  };

  const openNoteModal = (itemKey) => {
    setNoteModal({
      visible: true,
      itemKey,
      currentNote: notes[itemKey] || '',
    });
  };

  const saveNote = () => {
    const { itemKey, currentNote } = noteModal;
    if (currentNote.trim()) {
      setNotes(prev => ({
        ...prev,
        [itemKey]: currentNote.trim()
      }));
      saveData();
    }
    setNoteModal({ visible: false, itemKey: '', currentNote: '' });
  };

  const toggleSection = (sectionKey) => {
    setExpandedSections((prev) => ({
      ...prev,
      [sectionKey]: !prev[sectionKey],
    }));
  };

  const renderItem = (item, index) => {
    const selectedOption = selections[item.key];
    const itemNote = notes[item.key] || '';
    const options = getOptionsForInputType(item.inputType);

    return (
      <View key={item.key} style={styles.centeredItem}>
        <View style={styles.itemContainer}>
          <Text style={styles.itemText}>
            {index + 1}. {item.itemName}
          </Text>
          <View style={styles.itemRow}>
            <View style={styles.optionsContainer}>
              {options.map((option) => {
                const isSelected = selectedOption === option;
                return (
                  <TouchableOpacity
                    key={option}
                    style={[
                      styles.optionButton,
                      isSelected && styles.selectedOptionButton,
                      isSelected && option === 'YES/OK' && styles.okButton,
                      isSelected && option === 'NO/DEF' && styles.rejectButton,
                    ]}
                    onPress={() => handleSelectOption(item.key, option)}
                  >
                    <Text
                      style={[
                        styles.optionText,
                        isSelected && styles.optionTextSelected,
                      ]}
                    >
                      {option}
                    </Text>
                  </TouchableOpacity>
                );
              })}
            </View>
            <View style={styles.iconsContainer}>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => openNoteModal(item.key)}
              >
                <MaterialIcons name="note-add" size={24} color="#333" />
              </TouchableOpacity>
            </View>
          </View>
          
          {/* Position-specific input fields */}
          {item.positions && (
            <View style={styles.positionsContainer}>
              <Text style={styles.positionsLabel}>Check positions:</Text>
              <View style={styles.positionsGrid}>
                {item.positions.map((position) => {
                  const positionKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
                  const positionValue = selections[positionKey] || '';
                  
                  return (
                    <View key={position} style={styles.positionItem}>
                      <Text style={styles.positionLabel}>{position}:</Text>
                      <View style={styles.positionOptions}>
                        {['OK', 'CHECK', 'REPLACE'].map((option) => {
                          const isSelected = positionValue === option;
                          return (
                            <TouchableOpacity
                              key={option}
                              style={[
                                styles.positionOptionButton,
                                isSelected && styles.positionOptionSelected,
                                isSelected && option === 'OK' && styles.okButton,
                                isSelected && option === 'CHECK' && styles.checkButton,
                                isSelected && option === 'REPLACE' && styles.replaceButton,
                              ]}
                              onPress={() => handleSelectOption(positionKey, option)}
                            >
                              <Text
                                style={[
                                  styles.positionOptionText,
                                  isSelected && styles.positionOptionTextSelected,
                                ]}
                              >
                                {option}
                              </Text>
                            </TouchableOpacity>
                          );
                        })}
                      </View>
                    </View>
                  );
                })}
              </View>
            </View>
          )}

          {itemNote && (
            <Text style={styles.noteText}>
              Note: {itemNote}
            </Text>
          )}
        </View>
        <View style={styles.itemBorder} />
      </View>
    );
  };

  const validateSelections = () => {
    const allItems = sections.flatMap((section) => section.subItems);
    if (allItems.length === 0) {
      Alert.alert('No Items', 'The selected checklist has no valid items.');
      return false;
    }

    // Check if at least one item has been selected
    const hasAtLeastOneSelection = allItems.some((item) => selections[item.key] !== null && selections[item.key] !== '');
    if (!hasAtLeastOneSelection) {
      Alert.alert('Incomplete Assessment', 'Please make at least one selection before proceeding.');
      return false;
    }

    return true;
  };

  const handleNext = () => {
    if (validateSelections()) {
      setActiveStep(3);
    }
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.checklistTitle}>Workshop Truck Check Sheet</Text>

        {sections.length > 0 ? (
          sections.map((section) => {
            // Calculate selected items count for this section
            const sectionItems = section.subItems || [];
            const selectedItemsCount = sectionItems.filter(item => selections[item.key]).length;
            const totalItemsCount = sectionItems.length;
            
  return (
              <View key={section.key}>
                <TouchableOpacity
                  style={styles.sectionHeader}
                  onPress={() => toggleSection(section.key)}
                >
                  <View style={styles.sectionHeaderContent}>
                    <MaterialIcons
                      name={expandedSections[section.key] ? 'expand-less' : 'expand-more'}
                      size={20}
                      color={expandedSections[section.key] ? '#015185' : '#333'}
                    />
                    <View style={styles.sectionHeaderTextContainer}>
                      <Text style={styles.sectionHeaderText}>{section.name}</Text>
                      <Text style={styles.sectionItemCount}>
                        ({selectedItemsCount}/{totalItemsCount} selected)
                      </Text>
                    </View>
                  </View>
                </TouchableOpacity>
                {expandedSections[section.key] && (
                  <View style={styles.sectionContent}>
                    {section.subItems.length > 0 ? (
                      section.subItems.map((item, index) => renderItem(item, index))
                    ) : (
                      <Text style={styles.emptySectionText}>No inspection items</Text>
                    )}
                  </View>
                )}
              </View>
            );
          })
        ) : (
          <Text style={styles.emptySectionText}>No sections available</Text>
        )}
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(1)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Note Modal */}
      <Modal
        visible={noteModal.visible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setNoteModal({ visible: false, itemKey: '', currentNote: '' })}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Note</Text>
            <TextInput
              style={styles.modalTextInput}
              value={noteModal.currentNote}
              onChangeText={(value) => setNoteModal(prev => ({ ...prev, currentNote: value }))}
              placeholder="Enter your note here..."
              multiline
              numberOfLines={4}
            />
            <View style={styles.modalButtons}>
              <TouchableOpacity
                style={styles.modalButton}
                onPress={() => setNoteModal({ visible: false, itemKey: '', currentNote: '' })}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.modalButton, styles.modalButtonPrimary]} onPress={saveNote}>
                <Text style={[styles.modalButtonText, styles.modalButtonTextPrimary]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  checklistTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 80,
  },
  headingContainer: {
    paddingVertical: 10,
    marginHorizontal: '5%',
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  sectionHeader: {
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  sectionHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 5,
  },
  sectionHeaderText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
  },
  sectionHeaderTextContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  sectionItemCount: {
    fontSize: 14,
    color: '#666',
    marginLeft: 5,
  },
  sectionContent: {
    backgroundColor: '#fff',
  },
  centeredItem: {
    width: '90%',
    alignSelf: 'center',
    marginBottom: 15,
  },
  itemContainer: {
    paddingBottom: 15,
  },
  itemText: {
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  optionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    marginVertical: 2,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  rejectButton: {
    backgroundColor: '#FF4B2B',
  },
  optionText: {
    fontSize: 14,
    color: '#333',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  iconsContainer: {
    flexDirection: 'row',
  },
  iconButton: {
    padding: 10,
  },
  noteText: {
    fontSize: 14,
    color: '#333',
    marginTop: 8,
    marginBottom: 8,
    fontStyle: 'italic',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  emptySectionText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    padding: 20,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 20,
    width: width * 0.9,
    maxHeight: height * 0.6,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  modalTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 100,
    textAlignVertical: 'top',
    marginBottom: 20,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 6,
    marginHorizontal: 5,
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#ddd',
    backgroundColor: '#fff',
  },
  modalButtonPrimary: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  modalButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  modalButtonTextPrimary: {
    color: '#fff',
  },
  positionsContainer: {
    marginTop: 15,
    padding: 15,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  positionsLabel: {
    fontSize: 15,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 12,
    textAlign: 'center',
  },
  positionsGrid: {
    flexDirection: 'column',
    gap: 12,
  },
  positionItem: {
    width: '100%',
    marginBottom: 8,
  },
  positionLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
    marginBottom: 8,
    textAlign: 'center',
  },
  positionOptions: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    backgroundColor: '#fff',
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#ddd',
    padding: 8,
  },
  positionOptionButton: {
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 6,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#ECF0F3',
    borderWidth: 1,
    borderColor: '#ddd',
  },
  positionOptionSelected: {
    backgroundColor: '#333',
    borderColor: '#333',
  },
  positionOptionText: {
    fontSize: 13,
    color: '#333',
    fontWeight: '500',
  },
  positionOptionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  checkButton: {
    backgroundColor: '#FFD700', // Gold color for CHECK
  },
  replaceButton: {
    backgroundColor: '#FF4B2B', // Red color for REPLACE
  },
  itemBorder: {
    width: '80%',
    alignSelf: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
});

export default HSE_123Step2;