import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  TextInput,
  Dimensions,
  Alert,
  Image,
  Modal,
  Platform,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import Toast from 'react-native-toast-message';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import { useNavigation } from '@react-navigation/native';
import HSEInspectionAPI from '../../../ConfigMongo/HSEInspectionMongoConfig';

const { width, height } = Dimensions.get('window');

const SyncModal = ({ visible, message = 'Processing...' }) => {
  const rotateAnim = React.useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Generating PDF Report</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[
                styles.syncIcon,
                { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] },
              ]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_034Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [anticipatedBenefits, setAnticipatedBenefits] = useState('');
  const [photos, setPhotos] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [generatedPdfPath, setGeneratedPdfPath] = useState(null);
  const [reportId, setReportId] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  useEffect(() => {
    (async () => {
      try {
        const saved2 = JSON.parse(await AsyncStorage.getItem('hse034Step2') || '{}');
        const saved3 = JSON.parse(await AsyncStorage.getItem('hse034Step3') || '{}');
        if (saved3.anticipatedBenefits !== undefined) setAnticipatedBenefits(saved3.anticipatedBenefits);
        if (Array.isArray(saved3.photos)) setPhotos(saved3.photos);
        // migrate from step2 if previously stored
        if (!saved3.photos && Array.isArray(saved2?.photos)) setPhotos(saved2.photos);
        if (!saved3.anticipatedBenefits && saved2?.formData?.anticipatedBenefits) setAnticipatedBenefits(saved2.formData.anticipatedBenefits);
      } catch {}
    })();
  }, []);

  const persist = async (next) => {
    await AsyncStorage.setItem('hse034Step3', JSON.stringify({
      anticipatedBenefits,
      photos,
      ...next,
      timestamp: new Date().toISOString(),
    }));
  };

  const handleBenefitsChange = (t) => {
    setAnticipatedBenefits(t);
    persist({ anticipatedBenefits: t });
  };

  const addPhotoFromCamera = async () => {
    try {
      if (photos.length >= 4) {
        Toast.show({ type: 'info', text1: 'Maximum 4 photos allowed', position: 'bottom' });
        return;
      }
      const response = await launchCamera({ mediaType: 'photo', quality: 0.8 });
      if (response?.assets?.length) {
        const a = response.assets[0];
        const next = [...photos, { uri: a.uri, fileName: a.fileName || `photo_${Date.now()}.jpg` }].slice(0, 4);
        setPhotos(next);
        persist({ photos: next });
      }
    } catch {
      Toast.show({ type: 'error', text1: 'Could not open camera', position: 'bottom' });
    }
  };

  const addPhotoFromLibrary = async () => {
    try {
      const remaining = Math.max(0, 4 - photos.length);
      if (remaining === 0) {
        Toast.show({ type: 'info', text1: 'Maximum 4 photos allowed', position: 'bottom' });
        return;
      }
      const response = await launchImageLibrary({ mediaType: 'photo', quality: 0.8, selectionLimit: remaining });
      if (response?.assets?.length) {
        const newOnes = response.assets.map(a => ({ uri: a.uri, fileName: a.fileName || `photo_${Date.now()}.jpg` }));
        const next = [...photos, ...newOnes].slice(0, 4);
        setPhotos(next);
        persist({ photos: next });
      }
    } catch {
      Toast.show({ type: 'error', text1: 'Could not open gallery', position: 'bottom' });
    }
  };

  const removePhoto = (index) => {
    Alert.alert('Remove Photo', 'Delete this photo?', [
      { text: 'Cancel', style: 'cancel' },
      { text: 'Remove', style: 'destructive', onPress: async () => {
        const next = photos.filter((_, i) => i !== index);
        setPhotos(next);
        persist({ photos: next });
      }},
    ]);
  };

  const openPhotoPicker = () => {
    if (photos.length >= 4) {
      Toast.show({ type: 'info', text1: 'Maximum 4 photos allowed', position: 'bottom' });
      return;
    }
    Alert.alert('Add Photo', 'Choose a source', [
      { text: 'Camera', onPress: addPhotoFromCamera },
      { text: 'Gallery', onPress: addPhotoFromLibrary },
      { text: 'Cancel', style: 'cancel' },
    ]);
  };

  const handleSubmit = async () => {
    try {
      await persist({});
      await handleGenerateReport();
    } catch (error) {
      console.error('Error submitting form:', error);
      Alert.alert('Error', 'Failed to submit form. Please try again.');
    }
  };

  const handleGoToHSEHome = () => {
    setShowSuccessModal(false);
    navigation.navigate('HSEInspectionHome');
  };

  const handleViewPDF = async () => {
    if (!reportId) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'Report ID not available.', position: 'bottom' });
      return;
    }
    try {
      const response = await HSEInspectionAPI.getHSEInspectionById(reportId);
      
      if (!response.success || !response.data) {
        Alert.alert('Error', 'Report not found in MongoDB.');
        return;
      }

      const pdfUrl = response.data.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      setShowSuccessModal(false);
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => navigation.navigate('HSEInspectionHome')
      });
    } catch (error) {
      console.error('❌ Error opening PDF from MongoDB:', error);
      Alert.alert('Error', 'Could not open the PDF from MongoDB. Please try again.');
    }
  };

  const generatePDFWithHtml = async (report) => {
    const html = generatePDFHTML(report.step1, report.step2, report.step3, report.id);
    const options = {
      html,
      fileName: report.id,
      directory: 'Documents',
    };
    const file = await RNHTMLtoPDF.convert(options);
    return file.filePath;
  };

  const deepCleanObject = (obj) => {
    if (typeof obj !== 'object' || obj === null) return obj;
    if (Array.isArray(obj)) return obj.map(deepCleanObject);
    const cleaned = {};
    Object.keys(obj).forEach((key) => {
      const value = obj[key];
      cleaned[key] = typeof value === 'object' && value !== null ? deepCleanObject(value) : String(value ?? '');
    });
    return cleaned;
  };

  const getAdminUserId = async () => {
    const userId = await AsyncStorage.getItem('userUid');
    if (!userId) throw new Error('User ID not found. Please log in again.');
    return userId;
  };

  const uploadPDFToMongoDB = async (filePath, userId) => {
    try {
      const pdfFile = {
        uri: `file://${filePath}`,
        type: 'application/pdf',
        name: `hse_034_${userId}_${Date.now()}.pdf`
      };
      const response = await HSEInspectionAPI.uploadPDF(userId, pdfFile);
      if (!response.success) throw new Error('Failed to upload PDF to MongoDB');
      return response.downloadUrl;
    } catch (error) {
      console.error('❌ Error uploading PDF to MongoDB:', error);
      throw error;
    }
  };

  const uploadImageToMongoDB = async (imageUri, userId) => {
    try {
      const imageFile = {
        uri: imageUri,
        type: 'image/jpeg',
        name: `hse_034_photo_${userId}_${Date.now()}.jpg`
      };
      const response = await HSEInspectionAPI.uploadAttachment(userId, imageFile);
      if (!response.success) throw new Error('Failed to upload image to MongoDB');
      return response.downloadUrl;
    } catch (error) {
      console.error('❌ Error uploading image to MongoDB:', error);
      throw error;
    }
  };

  const getCurrentReportNumber034 = async () => {
    try {
      const response = await HSEInspectionAPI.getCurrentCounter();
      if (response.success && response.data) {
        return response.data.formattedNumber || response.data.nextNumber || '001';
      }
      return '001';
    } catch (error) {
      console.error('❌ Error getting current report number from MongoDB:', error);
      return '001';
    }
  };

  const incrementReportCounter034 = async () => {
    try {
      const response = await HSEInspectionAPI.incrementCounter();
      if (response.success && response.data) {
        return response.data.formattedNumber || '001';
      }
      return '001';
    } catch (error) {
      console.error('❌ Error incrementing report counter in MongoDB:', error);
      return '001';
    }
  };

  const flushStepData034 = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hse034Step2');
      await AsyncStorage.removeItem('hse034Step3');
    } catch (error) {

      console.error('Error flushing local data:', error);
      

    }
  };

  const handleGenerateReport = async () => {
    try {
      console.log('📄 Starting HSE-034 report generation...');
      setIsLoading(true);
      setSyncModalVisible(true);

      const userId = await getAdminUserId();
      const savedStep1Raw = await AsyncStorage.getItem('hseStep1');
      const savedStep2Raw = await AsyncStorage.getItem('hse034Step2');
      if (!savedStep1Raw || !savedStep2Raw) {
        throw new Error('Previous step data is missing.');
      }
      const step1 = JSON.parse(savedStep1Raw);
      const step2 = JSON.parse(savedStep2Raw);

      const newReportId = step1.requestNumber || `${step1.countryCode || 'ZMB'}-${step1.projectCode || 'PRJ'}-HSE-034`;
      setReportId(newReportId);
      console.log('📋 Report ID:', newReportId);

      // Upload photos to MongoDB first
      console.log('📸 Uploading photos to MongoDB...');
      const uploadedPhotos = [];
      for (const photo of photos) {
        try {
          const photoUrl = await uploadImageToMongoDB(photo.uri, userId);
          uploadedPhotos.push({ ...photo, uri: photoUrl, uploadedUrl: photoUrl });
          console.log('✅ Photo uploaded:', photoUrl);
        } catch (error) {
          console.error('⚠️ Failed to upload photo:', error);
        }
      }

      const step1Data = deepCleanObject(step1);
      const step2Data = deepCleanObject(step2);
      const step3Data = { anticipatedBenefits, photos: uploadedPhotos };

      // Generate PDF
      console.log('📄 Generating PDF...');
      const report = { id: newReportId, step1: step1Data, step2: step2Data, step3: step3Data };
      const pdfPath = await generatePDFWithHtml(report);
      setGeneratedPdfPath(pdfPath);
      console.log('✅ PDF generated at:', pdfPath);

      // Upload PDF to MongoDB
      console.log('📤 Uploading PDF to MongoDB...');
      const pdfDownloadUrl = await uploadPDFToMongoDB(pdfPath, userId);
      console.log('✅ PDF uploaded:', pdfDownloadUrl);

      // Save complete report to MongoDB
      console.log('💾 Saving report to MongoDB...');
      const completeReportData = {
        _id: newReportId,
        userId,
        requestNumber: newReportId,
        status: 'completed',
        step1: step1Data,
        step2: step2Data,
        step3: step3Data,
        pdfDownloadUrl,
        createdAt: new Date().toISOString(),
        createdBy: userId,
        lastModifiedBy: userId
      };

      const saveResponse = await HSEInspectionAPI.createHSEInspection(userId, completeReportData);
      if (!saveResponse.success) {
        throw new Error('Failed to save report to MongoDB');
      }
      console.log('✅ Report saved to MongoDB');

      // Send HSE inspection notifications
      console.log('📧 Sending HSE inspection notifications...');
      try {
        const userData = await AsyncStorage.getItem('userData');
        const user = userData ? JSON.parse(userData) : null;
        const createdByName = user?.name || 'Unknown Inspector';
        const inspectorEmail = user?.email || '';
        
        const notificationPayload = {
          hseInspectionData: {
            _id: newReportId,
            id: newReportId,
            requestNumber: newReportId,
            checklistName: 'HSE-034 Management of Change',
            country: step1.country || '',
            project: step1.project || '',
            status: 'completed',
          },
          createdByName,
          inspectorEmail,
          pdfUrl: pdfDownloadUrl
        };
        
        const notifResponse = await fetch('http://168.231.113.219:5028/api/hse-inspection-notifications/new-inspection', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(notificationPayload),
        });
        
        if (notifResponse.ok) {
          console.log('✅ HSE notifications sent successfully');
        }
      } catch (notifError) {
        console.error('❌ Error sending HSE notifications:', notifError);
      }

      // Increment counter
      const reportSerial = await incrementReportCounter034();
      console.log('✅ Counter incremented:', reportSerial);

      // Delete temp PDF
      await RNFS.unlink(pdfPath).catch(() => {});
      await flushStepData034(newReportId);

      setShowSuccessModal(true);
      Toast.show({ type: 'success', text1: 'Success', text2: 'Report saved to MongoDB successfully.', position: 'bottom' });
    } catch (error) {
      console.error('❌ Error generating report:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: error.message || 'Failed to save report to MongoDB.', position: 'bottom' });
      Alert.alert('Save Error', `Failed to save report to MongoDB.\n\nError: ${error.message}`);
    } finally {
      setIsLoading(false);
      setSyncModalVisible(false);
    }
  };

  const loadStep1Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hseStep1');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 1 data:', error);
      return {};
    }
  };

  const loadStep2Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse034Step2');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 2 data:', error);
      return {};
    }
  };

  const generatePDFHTML = (step1, step2, step3, pdfReportId) => {
    const { formData = {}, hazardsRisks = [] } = step2;
    const asBool = (v) => v === true || v === 'true' || v === 'Yes' || v === 'yes' || v === 1 || v === '1';
    const isPlant = asBool(formData.plantOrEquipment);
    const isDesign = asBool(formData.designOrLayoutOfArea);
    const isOperating = asBool(
      formData.operatingProcess !== undefined
        ? formData.operatingProcess
        : (formData.standardSopOrWorkProcedure || formData.standardSopWorkProcedure)
    );
    const isCompetencyPeopleRole = asBool(
      formData.competencyPeopleRole !== undefined
        ? formData.competencyPeopleRole
        : (formData.competency || formData.peopleRole)
    );
    
    // Generate hazards and risks rows
    let hazardsRisksRows = '';
    hazardsRisks.forEach((row, index) => {
      hazardsRisksRows += `
        <tr>
          <td>${row.hazard || ''}</td>
          <td>${row.risk || ''}</td>
        </tr>
      `;
    });
    
    // Add empty rows if less than 3
    for (let i = hazardsRisks.length; i < 3; i++) {
      hazardsRisksRows += `
        <tr>
          <td></td>
          <td></td>
        </tr>
      `;
    }

    // Generate photos HTML
    let photosHTML = '';
    if (step3.photos && step3.photos.length > 0) {
      photosHTML = `
        <div style="margin-top: 20px;">
          <h4 style="color: #015185; margin-bottom: 10px;">PHOTOS:</h4>
          <div style="display: flex; flex-wrap: wrap; gap: 10px;">
            ${step3.photos.map((photo, index) => `
              <div style="width: 150px; height: 150px; border: 1px solid #ddd; border-radius: 8px; overflow: hidden;">
                <img src="${photo.uri}" alt="Photo ${index + 1}" style="width: 100%; height: 100%; object-fit: cover;" />
              </div>
            `).join('')}
          </div>
        </div>
      `;
    }

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Titan Drilling - Management of Change Form</title>
          <style>
              body {
                  font-family: Arial, sans-serif;
                  margin: 0;
                  padding: 10px;
                  background-color: #f0f8ff;
                  font-size: 14px;
                  line-height: 1.4;
              }
              .container {
                  max-width: 100%;
                  margin: 0 auto;
                  background-color: white;
                  padding: 20px;
                  box-shadow: 0 0 10px rgba(0,0,0,0.1);
              }
              .header {
                  display: flex;
                  align-items: center;
                  justify-content: space-between;
                  margin-bottom: 20px;
              }
              .logo {
                  font-size: 24px;
                  color: blue;
                  margin-right: 10px;
              }
              .title {
                  font-size: 18px;
                  font-weight: bold;
                  flex-grow: 1;
              }
              .doc-info {
                  text-align: right;
                  font-size: 12px;
              }
              .doc-info table {
                  border-collapse: collapse;
              }
              .doc-info td {
                  padding: 2px 5px;
                  border: 1px solid black;
              }
              .form-table {
                  width: 100%;
                  border-collapse: collapse;
                  margin-bottom: 20px;
              }
              .form-table th, .form-table td {
                  border: 1px solid black;
                  padding: 8px;
                  text-align: left;
              }
              .form-table th {
                  background-color: #e0f7e0;
                  font-weight: bold;
              }
              .empty-cell {
                  height: 50px;
              }
              .section-header {
                  background-color: #90ee90;
                  color: black;
                  font-weight: bold;
              }
              .sub-table {
                  width: 100%;
                  border-collapse: collapse;
              }
              .sub-table td {
                  border: none;
                  padding: 5px;
              }
              .checklist {
                  width: 100%;
                  border-collapse: collapse;
              }
              .checklist th {
                  background-color: #90ee90;
                  text-align: left;
                  padding: 8px;
                  border: 1px solid black;
              }
              .checklist td {
                  border: 1px solid black;
                  padding: 8px;
              }
              .photos {
                  font-weight: bold;
                  margin-top: 20px;
              }
              .checkbox {
                  width: 20px;
                  height: 20px;
                  border: 2px solid #000;
                  display: inline-block;
                  margin-right: 10px;
                  text-align: center;
                  line-height: 18px;
                  font-weight: bold;
              }
              .checkbox.checked::after {
                  content: "✓";
                  color: #000;
              }
              @media (max-width: 600px) {
                  body {
                      padding: 5px;
                      font-size: 12px;
                  }
                  .header {
                      flex-direction: column;
                      align-items: flex-start;
                  }
                  .doc-info {
                      text-align: left;
                      margin-top: 10px;
                  }
                  .title {
                      font-size: 16px;
                  }
              }
          </style>
      </head>
      <body>
          <div class="container">
              <div class="header">
                  <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" style="width: 200px; height: auto;">
              <div class="doc-info">
                      <table>
                          <tr><td>Document Name</td><td>HSE-FOR-8.1.3-Management of Change</td></tr>
                          <tr><td>Document ID</td><td>HSE-034</td></tr>
                          <tr><td>Version Date</td><td>Date: 16/05/2023</td></tr>
                          <tr><td>Version</td><td>4</td></tr>
                          <tr><td>Approved</td><td>HSM</td></tr>
                      <tr><td>Date</td><td></td></tr>
                      <tr><td>Request #</td><td>${pdfReportId || step1?.requestNumber || ''}</td></tr>
                      </table>
                  </div>
              </div>
              <h3>8.1.3 Management of Change Form</h3>
              <table class="form-table">
                  <tr><th>INITIATOR</th><td>${formData.initiator || ''}</td></tr>
                  <tr><th>TEAM</th><td>${formData.team || ''}</td></tr>
                  <tr><th>DATE</th><td>${formData.date ? new Date(formData.date).toLocaleDateString('en-GB') : ''}</td></tr>
              </table>
              <table class="form-table">
                  <tr><th class="section-header" colspan="2">A. Indicate TYPE OF CHANGE (✓) and provide details below</th></tr>
                  <tr>
                      <td>Plant or Equipment: ${isPlant ? '<span class="checkbox checked"></span>' : ''}</td>
                      <td>Design or Layout of Area: ${isDesign ? '<span class="checkbox checked"></span>' : ''}</td>
                  </tr>
                  <tr>
                      <td>Operating Process (Standard, SOP, or Work Procedure): ${isOperating ? '<span class="checkbox checked"></span>' : ''}</td>
                      <td>Competency (People / Role): ${isCompetencyPeopleRole ? '<span class="checkbox checked"></span>' : ''}</td>
                  </tr>
              </table>
              <table class="form-table">
                  <tr><th class="section-header">B. Describe the PROPOSED CHANGE</th></tr>
                  <tr><td>${formData.describeProposedChange || ''}</td></tr>
              </table>
              <table class="form-table">
                  <tr><th class="section-header">C. REASON FOR THE PROPOSED CHANGE</th></tr>
                  <tr><td>${formData.reasonForProposedChange || ''}</td></tr>
              </table>
              <table class="form-table">
                  <tr><th class="section-header" colspan="2">D. What HAZARDS / RISKS are presented by the proposed change?</th></tr>
                  <tr>
                      <th>HAZARDS</th>
                      <th>RISKS</th>
                  </tr>
                  ${hazardsRisksRows}
                  <tr><td colspan="2">Can the Risk be managed to an acceptable level?</td></tr>
                  <tr><td colspan="2">${formData.canRiskBeManaged || ''}</td></tr>
              </table>
              <table class="form-table">
                  <tr><th class="section-header">E. What are the ANTICIPATED BENEFITS (OPPORTUNITIES)</th></tr>
                  <tr><td>${step3.anticipatedBenefits || ''}</td></tr>
              </table>
              <table class="form-table">
                  <tr><th class="section-header">F. IMPACT / ACTION PLAN Checklist</th></tr>
                  <tr><td>NOTE: Supporting documents required</td></tr>
                  <tr><td>Does the change require engineering/design or approval?</td></tr>
                  <tr><td>Does the change require a process / operating review?</td></tr>
                  <tr><td>Does the change require new, or changes to existing standards or procedures?</td></tr>
                  <tr><td>Does the change require consultation with workforce, or others?</td></tr>
                  <tr><td>Does the change affect warranties or certifications?</td></tr>
                  <tr><td>Does the change require drawing updates?</td></tr>
                  <tr><td>Does the change require stocking of new parts/materials?</td></tr>
                  <tr><td>Does the change affect competencies?</td></tr>
                  <tr><td>Does the change require communication or notification? / Does the change require record to be maintained? / eg. modification to plant, risk assessment / review, record of training, etc.</td></tr>
                  <tr><td>Does the change require any legal notification?</td></tr>
                  <tr><td>Does the change require cliental notification?</td></tr>
              </table>
              ${photosHTML}
          </div>
      </body>
      </html>
    `;
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
        {isLoading && (
          <View style={styles.loadingContainer}>
            <MaterialIcons name="hourglass-empty" size={40} color="#015185" />
            <Text style={styles.loadingText}>Generating PDF Report...</Text>
            <Text style={styles.loadingSubtext}>Please wait while we compile your data</Text>
          </View>
        )}

        <View style={styles.section}>
          <Text style={styles.subSectionTitle}>What are the ANTICIPATED BENEFITS (OPPORTUNITIES)</Text>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Benefits (Opportunities)</Text>
            <TextInput
              style={[styles.textInput, { minHeight: 80, textAlignVertical: 'top' }]}
              value={anticipatedBenefits}
              onChangeText={handleBenefitsChange}
              placeholder="List the anticipated benefits"
              placeholderTextColor="#888"
              multiline
            />
          </View>

          <Text style={styles.subSectionTitle}>IMPACT – ACTION PLAN checklist</Text>
          <Text style={styles.impactNote}>NOTE: Supporting documents required</Text>
          <View style={styles.impactBox}>
            <Text style={styles.impactLine}>• Does the change require engineering design and/or approval?</Text>
            <Text style={styles.impactLine}>• Does the change require a process / operating review?</Text>
            <Text style={styles.impactLine}>• Does the change require new, or changes to existing standards or procedures?</Text>
            <Text style={styles.impactLine}>• Does the change require consultation with workforce, or others?</Text>
            <Text style={styles.impactLine}>• Does the change affect warranties or certifications?</Text>
            <Text style={styles.impactLine}>• Does the change require drawing updates?</Text>
            <Text style={styles.impactLine}>• Does the change require stocking of new parts/materials?</Text>
            <Text style={styles.impactLine}>• Does the change affect require competencies?</Text>
            <Text style={styles.impactLine}>• Does the change require communication or notification?</Text>
            <Text style={styles.impactLine}>• Does the change require records to be maintained? (eg: modification to plant, risk assessment / review, record of training, etc.)</Text>
            <Text style={styles.impactLine}>• Does the change require any legal documents to be drawn?</Text>
            <Text style={styles.impactLine}>• Does the change require cliental notification?</Text>
          </View>

          <Text style={styles.photosLabel}>PHOTOS</Text>
          <TouchableOpacity style={[styles.photoUploadBox, photos.length >= 4 && styles.photoUploadDisabled]} onPress={openPhotoPicker}>
            <MaterialIcons name="camera-alt" size={36} color="#015185" />
          </TouchableOpacity>
          <View style={styles.photoGrid}>
            {photos.map((p, idx) => (
              <View key={`${p.uri}_${idx}`} style={styles.photoItem}>
                <Image source={{ uri: p.uri }} style={styles.photoImage} />
                <TouchableOpacity style={styles.photoRemove} onPress={() => removePhoto(idx)}>
                  <MaterialIcons name="close" size={16} color="#fff" />
                </TouchableOpacity>
              </View>
            ))}
          </View>
        </View>
      </ScrollView>

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <View style={styles.successIconContainer}>
              <MaterialIcons name="check-circle" size={60} color="#015185" />
            </View>
            <Text style={styles.successModalTitle}>Form Submitted Successfully!</Text>
            <Text style={styles.successModalSubtitle}>PDF report has been generated.</Text>
            
            <View style={styles.successModalButtons}>
              <TouchableOpacity style={styles.successModalButton} onPress={handleGoToHSEHome}>
                <Text style={styles.successModalButtonText}>Back</Text>
              </TouchableOpacity>
              
              <TouchableOpacity style={[styles.successModalButton, styles.viewPdfButton]} onPress={handleViewPDF}>
                <Text style={styles.successModalButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep?.(2)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleSubmit} disabled={isLoading}>
          <Text style={styles.nextButtonText}>
            {isLoading ? 'Generating PDF...' : 'Generate Report'}
          </Text>
          <MaterialIcons style={styles.bottomicon1} name={isLoading ? "hourglass-empty" : "check"} size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <Toast />
      <SyncModal visible={syncModalVisible} message={isLoading ? 'Processing...' : ''} />
    </View>
  );
};

const styles = StyleSheet.create({
  container: { 
    flex: 1, 
    backgroundColor: '#fff' 
  },
  mainContent: { 
    flex: 1, 
    paddingHorizontal: width * 0.04, 
    paddingVertical: width * 0.025 
  },
  scrollContent: { 
    paddingBottom: 110 
  },
  section: { 
    marginBottom: 25 
  },
  inputContainer: { 
    marginBottom: 15 
  },
  inputLabel: { 
    fontSize: 16, 
    color: '#015185', 
    marginBottom: 5, 
    fontWeight: '500' 
  },
  textInput: { 
    borderWidth: 1, 
    borderColor: '#015185', 
    borderRadius: 5, 
    paddingHorizontal: 12, 
    paddingVertical: 12, 
    fontSize: 16, 
    backgroundColor: '#fff' 
  },
  subSectionTitle: { 
    fontSize: 16, 
    fontWeight: '700', 
    color: '#015185', 
    marginTop: 8, 
    marginBottom: 6 
  },
  impactNote: { 
    fontSize: 13, 
    color: '#015185', 
    fontWeight: '600', 
    marginTop: 2, 
    marginBottom: 6 
  },
  impactBox: { 
    marginTop: 4, 
    padding: 12, 
    backgroundColor: '#F9FBFD', 
    borderRadius: 6, 
    borderWidth: 1, 
    borderColor: '#E2ECF5' 
  },
  impactLine: { 
    fontSize: 14, 
    color: '#333', 
    lineHeight: 20, 
    marginBottom: 4 
  },
  photosLabel: { 
    marginTop: 10, 
    fontSize: 14, 
    color: '#333', 
    fontWeight: '600' 
  },
  photoUploadBox: { 
    height: 140, 
    borderWidth: 1, 
    borderColor: '#015185', 
    borderRadius: 6, 
    alignItems: 'center', 
    justifyContent: 'center', 
    backgroundColor: '#F9FBFD', 
    marginTop: 8, 
    marginBottom: 10 
  },
  photoUploadDisabled: { 
    opacity: 0.6 
  },
  photoGrid: { 
    flexDirection: 'row', 
    flexWrap: 'wrap', 
    gap: 10 
  },
  photoItem: { 
    width: (width * 0.92 - 30) / 3, 
    height: (width * 0.92 - 30) / 3, 
    borderRadius: 6, 
    overflow: 'hidden', 
    position: 'relative' 
  },
  photoImage: { 
    width: '100%', 
    height: '100%' 
  },
  photoRemove: { 
    position: 'absolute', 
    top: 6, 
    right: 6, 
    backgroundColor: 'rgba(0,0,0,0.6)', 
    borderRadius: 12, 
    padding: 2 
  },
  buttonContainer: { 
    flexDirection: 'row', 
    justifyContent: 'space-between', 
    alignItems: 'center', 
    paddingHorizontal: 15, 
    paddingVertical: 15, 
    backgroundColor: '#F2F2F2', 
    borderTopWidth: 1, 
    borderTopColor: '#ddd', 
    position: 'absolute', 
    bottom: 0, 
    left: 0, 
    right: 0, 
    elevation: 5, 
    shadowColor: '#000', 
    shadowOffset: { width: 0, height: -2 }, 
    shadowOpacity: 0.1, 
    shadowRadius: 2, 
    paddingBottom: 30 
  },
  cancelButton: { 
    flexDirection: 'row', 
    alignItems: 'center', 
    justifyContent: 'center', 
    paddingHorizontal: 15, 
    paddingBottom: 30 
  },
  cancelButtonText: { 
    color: '#015185', 
    fontSize: 16, 
    fontWeight: '600' 
  },
  nextButton: { 
    flexDirection: 'row', 
    alignItems: 'center', 
    justifyContent: 'center', 
    paddingHorizontal: 15, 
    paddingBottom: 18, 
    borderRadius: 5 
  },
  nextButtonText: { 
    color: '#015185', 
    fontSize: 16, 
    fontWeight: '600' 
  },
  bottomicon: { 
    marginRight: 5 
  },
  bottomicon1: { 
    marginLeft: 5 
  },
  loadingContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    padding: 30,
    backgroundColor: '#f8f9fa',
    margin: 20,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  loadingText: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginTop: 15,
    textAlign: 'center',
  },
  loadingSubtext: {
    fontSize: 14,
    color: '#666',
    marginTop: 8,
    textAlign: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 30,
    width: width * 0.85,
    alignItems: 'center',
  },
  successIconContainer: {
    marginBottom: 20,
  },
  successModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  successModalSubtitle: {
    fontSize: 16,
    color: '#666',
    marginBottom: 25,
    textAlign: 'center',
  },
  successModalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  successModalButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    flex: 1,
    marginHorizontal: 5,
  },
  viewPdfButton: {
    backgroundColor: '#28a745',
  },
  successModalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 8,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default HSE_034Step3;