const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const cors = require('cors');
const fetch = require('node-fetch');

const app = express();
const PORT = process.env.HRFORM_PORT || 5013;

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB connection - reuse existing connection if available
// Check if mongoose is already connected
if (mongoose.connection.readyState === 0) {
  // Using shared MongoDB connection from dbConnection.js
console.log('✅ Using shared MongoDB connection');
} else {
  console.log('✅ HR Form Handler: Using existing MongoDB connection');
}

// Counter Schema for generating sequential application numbers
const CounterSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  sequence_value: { type: Number, default: 0 }
});

// Use existing Counter model if already compiled, otherwise create new one
const Counter = mongoose.models.Counter || mongoose.model('Counter', CounterSchema);

// GlobalUser Schema for getting user data
const GlobalUserSchema = new mongoose.Schema({
  _id: String,
  name: String,
  email: String,
  role: String,
  countries: [String],
  projects: [String],
}, { strict: false, collection: 'GlobalUsers' });

// Check if GlobalUser model exists to prevent OverwriteModelError
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [HR Form Handler] Using existing GlobalUser model');
} catch (error) {
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [HR Form Handler] Created new GlobalUser model');
}

// HR Application Schema
const HRApplicationSchema = new mongoose.Schema({
  id: { type: String, required: true, unique: true },
  applicationType: { type: String, required: true }, // leave_application, application_for_appeal, etc.
  applicationNumber: { type: String, required: true },
  
  // User Information
  adminUid: { type: String, required: true },
  userUid: { type: String }, // Applicant UID (for tracking)
  applicantName: { type: String },
  employeeNumber: { type: String },
  department: { type: String },
  position: { type: String },
  
  // Location
  country: { type: String },
  project: { type: String },
  
  // Application Status
  status: { type: String, default: 'Pending' }, // Pending, Approved, Rejected
  approvedBy: { type: String },
  approvedAt: { type: Date },
  rejectedBy: { type: String },
  rejectedAt: { type: Date },
  approvalComments: { type: String },
  
  // Common Fields
  dateFrom: { type: Date },
  dateTo: { type: Date },
  reason: { type: String },
  description: { type: String },
  
  // Leave Application Specific
  leaveType: { type: String },
  numberOfDays: { type: Number },
  contactDuringLeave: { type: String },
  
  // Salary Advance Specific
  advanceAmount: { type: Number },
  repaymentPeriod: { type: String },
  purpose: { type: String },
  
  // Hospital Attendance Specific
  patientName: { type: String },
  relationship: { type: String },
  hospitalName: { type: String },
  diagnosis: { type: String },
  
  // Funeral Grant Specific
  deceasedName: { type: String },
  relationshipToDeceased: { type: String },
  dateOfDeath: { type: Date },
  funeralDate: { type: Date },
  grantAmount: { type: Number },
  
  // Grievance/Appeal Specific
  grievanceType: { type: String },
  incidentDate: { type: Date },
  witnessNames: [{ type: String }],
  
  // Additional flexible fields
  additionalData: { type: mongoose.Schema.Types.Mixed },
  
  // Metadata
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
}, { strict: false, timestamps: true });

// Create indexes for faster queries
HRApplicationSchema.index({ adminUid: 1, applicationType: 1 });
HRApplicationSchema.index({ status: 1 });
HRApplicationSchema.index({ country: 1, project: 1 });
HRApplicationSchema.index({ createdAt: -1 });

const HRApplication = mongoose.model('HRApplication', HRApplicationSchema, 'GlobalHRApplications');

// Helper function to get next sequence number
async function getNextSequenceValue(sequenceName) {
  const counter = await Counter.findByIdAndUpdate(
    sequenceName,
    { $inc: { sequence_value: 1 } },
    { new: true, upsert: true }
  );
  return counter.sequence_value;
}

// Generate application number based on type, country, and project
function generateApplicationNumber(type, country, project, sequenceNumber) {
  // Get country code
  let countryCode = 'XXX';
  if (country === 'Zambia') {
    countryCode = 'ZMB';
  } else if (country === 'DRC') {
    countryCode = 'DRC';
  } else if (country) {
    countryCode = country.substring(0, 3).toUpperCase();
  }

  // Get project code (first 3 letters)
  const projectCode = project ? project.substring(0, 3).toUpperCase() : 'XXX';

  // Get type code
  const typeCodes = {
    'leave_application': 'LA',
    'application_for_appeal': 'AA',
    'funeral_grant_form': 'FG',
    'grievance_record': 'GR',
    'hospital_attendance_form': 'HA',
    'salary_advance_application': 'SA',
  };
  const typeCode = typeCodes[type] || 'HR';

  // Format: COUNTRY-PROJECT-TYPE-###
  return `${countryCode}-${projectCode}-${typeCode}-${String(sequenceNumber).padStart(3, '0')}`;
}

// Helper function to send HR application notifications
const sendHRApplicationNotifications = async (applicationData, createdByName, creatorEmail) => {
  try {
    console.log('🔔 ========== STARTING HR NOTIFICATION PROCESS ==========');
    console.log('📋 Application Data:', JSON.stringify(applicationData, null, 2));
    console.log('👤 Created By:', createdByName);
    console.log('📧 Creator Email:', creatorEmail);
    
    const notificationPayload = {
      applicationData: {
        _id: applicationData._id,
        id: applicationData.id,
        applicationNumber: applicationData.applicationNumber,
        applicationType: applicationData.applicationType,
        fullName: applicationData.fullName || createdByName,
        employeeNo: applicationData.employeeNo || applicationData.employeeNumber || '',
        department: applicationData.department || '',
        country: applicationData.country,
        project: applicationData.project,
        status: applicationData.status || 'Pending',
      },
      createdByName,
      creatorEmail,
    };
    
    console.log('📤 Sending payload to notification service:', JSON.stringify(notificationPayload, null, 2));
    
    const response = await fetch('http://api.titandrillingzm.com:5027/api/hr-notifications/new-application', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(notificationPayload),
    });
    
    console.log('📥 Response status:', response.status, response.statusText);
    
    if (!response.ok) {
      const errorText = await response.text();
      console.error('❌ HTTP Error Response:', errorText);
      throw new Error(`HTTP ${response.status}: ${errorText}`);
    }
    
    const result = await response.json();
    console.log('📊 Notification service response:', JSON.stringify(result, null, 2));
    
    if (result.success) {
      console.log('✅ HR application notifications sent successfully!');
      console.log('📊 Recipients:', result.recipients);
    } else {
      console.error('❌ Failed to send notifications:', result.error);
    }
    
    console.log('🔔 ========== NOTIFICATION PROCESS COMPLETE ==========');
    return result;
  } catch (error) {
    console.error('❌ ========== NOTIFICATION PROCESS FAILED ==========');
    console.error('❌ Error:', error.message);
    console.error('❌ Stack:', error.stack);
    return { success: false, error: error.message };
  }
};

// ===== API ENDPOINTS =====

// Health check
app.get('/health', (req, res) => {
  res.json({ 
    status: 'OK', 
    message: 'HR Form Applications API is running',
    timestamp: new Date().toISOString()
  });
});

// Get counts by application type and admin UID
app.get('/hr-applications/counts/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    const { countries, projects, onlyMineView } = req.query;

    console.log('📊 Fetching counts for admin:', adminUid);
    console.log('🔍 Filters:', { countries, projects, onlyMineView });

    const countriesArray = countries ? countries.split(',') : [];
    const projectsArray = projects ? projects.split(',') : [];
    const isOnlyMineView = onlyMineView === 'true';

    const applicationTypes = [
      'leave_application',
      'application_for_appeal',
      'funeral_grant_form',
      'grievance_record',
      'hospital_attendance_form',
      'salary_advance_application'
    ];

    const counts = {};
    const unapprovedCounts = {};

    for (const type of applicationTypes) {
      let query = { applicationType: type };

      if (isOnlyMineView) {
        // Only show applications created by this admin
        query.adminUid = adminUid;
      } else {
        // Filter by country and project
        if (countriesArray.length > 0) {
          query.country = { $in: countriesArray };
        }
        if (projectsArray.length > 0) {
          query.project = { $in: projectsArray };
        }
      }

      const total = await HRApplication.countDocuments(query);
      const unapproved = await HRApplication.countDocuments({ ...query, status: { $ne: 'Approved' } });

      counts[type] = total;
      unapprovedCounts[type] = unapproved;
    }

    res.json({
      success: true,
      data: {
        totalCounts: counts,
        unapprovedCounts: unapprovedCounts
      }
    });
  } catch (error) {
    console.error('❌ Error fetching counts:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch application counts',
      error: error.message
    });
  }
});

// Get all applications by type for an admin
app.get('/hr-applications/:applicationType/:adminUid', async (req, res) => {
  try {
    const { applicationType, adminUid } = req.params;
    const { countries, projects, onlyMineView, status } = req.query;

    console.log('📋 Fetching applications:', { applicationType, adminUid });

    const countriesArray = countries ? countries.split(',') : [];
    const projectsArray = projects ? projects.split(',') : [];
    const isOnlyMineView = onlyMineView === 'true';

    let query = { applicationType };

    if (isOnlyMineView) {
      query.adminUid = adminUid;
    } else {
      if (countriesArray.length > 0) {
        query.country = { $in: countriesArray };
      }
      if (projectsArray.length > 0) {
        query.project = { $in: projectsArray };
      }
    }

    if (status) {
      query.status = status;
    }

    const applications = await HRApplication.find(query).sort({ createdAt: -1 });

    res.json({
      success: true,
      data: applications,
      count: applications.length
    });
  } catch (error) {
    console.error('❌ Error fetching applications:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch applications',
      error: error.message
    });
  }
});

// Create new application
app.post('/hr-applications', async (req, res) => {
  try {
    const applicationData = req.body;

    console.log('📝 Creating new HR application:', {
      type: applicationData.applicationType,
      country: applicationData.country,
      project: applicationData.project
    });

    // Generate unique ID if not provided
    if (!applicationData.id) {
      applicationData.id = `hr_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    }

    // Generate application number if not provided
    if (!applicationData.applicationNumber) {
      const sequenceNumber = await getNextSequenceValue(`hrApplicationNo_${applicationData.applicationType}`);
      applicationData.applicationNumber = generateApplicationNumber(
        applicationData.applicationType,
        applicationData.country,
        applicationData.project,
        sequenceNumber
      );
    }

    // Set default status if not provided
    if (!applicationData.status) {
      applicationData.status = 'Pending';
    }

    const newApplication = new HRApplication(applicationData);
    await newApplication.save();

    console.log('✅ HR application created:', newApplication.applicationNumber);

    // Send notifications in background (fire-and-forget)
    setImmediate(async () => {
      try {
        console.log('🔔 Starting notification process...');
        console.log('📊 Application data:', {
          id: newApplication.id,
          applicationNumber: newApplication.applicationNumber,
          adminUid: newApplication.adminUid,
          fullName: newApplication.fullName,
          employeeNo: newApplication.employeeNo,
          country: newApplication.country,
          project: newApplication.project
        });
        
        // Get creator name and email from adminUid
        const creator = await GlobalUser.findOne({ _id: newApplication.adminUid }).select('name email');
        console.log('👤 Creator found:', creator ? `${creator.name} (${creator.email})` : 'NOT FOUND');
        
        const createdByName = creator?.name || newApplication.fullName || 'Unknown User';
        const creatorEmail = creator?.email || '';
        
        console.log('📧 Sending HR application notifications for creator:', createdByName, 'Email:', creatorEmail);
        const notifResult = await sendHRApplicationNotifications(newApplication.toObject(), createdByName, creatorEmail);
        console.log('✅ Notification result:', notifResult);
      } catch (notifError) {
        console.error('❌ Error sending HR application notifications:', notifError);
        console.error('❌ Error stack:', notifError.stack);
      }
    });

    res.status(201).json({
      success: true,
      message: 'Application created successfully',
      data: newApplication
    });
  } catch (error) {
    console.error('❌ Error creating application:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create application',
      error: error.message
    });
  }
});

// Get single application by ID
app.get('/hr-applications/entry/:id', async (req, res) => {
  try {
    console.log('🔍 Searching for application with ID:', req.params.id);
    
    // Try multiple ID fields
    let application = await HRApplication.findOne({ id: req.params.id });
    if (!application) {
      console.log('🔍 Not found by id field, trying applicationId...');
      application = await HRApplication.findOne({ applicationId: req.params.id });
    }
    if (!application) {
      console.log('🔍 Not found by applicationId field, trying _id...');
      application = await HRApplication.findById(req.params.id);
    }
    
    console.log('🔍 Found application:', application ? 'Yes' : 'No');
    if (application) {
      console.log('🔍 Application details:', {
        id: application.id,
        applicationId: application.applicationId,
        _id: application._id,
        applicationType: application.applicationType
      });
    }
    
    if (!application) {
      console.log('❌ Application not found for ID:', req.params.id);
      return res.status(404).json({
        success: false,
        message: 'Application not found'
      });
    }

    console.log('✅ Returning application data for ID:', req.params.id);
    res.json({
      success: true,
      data: application
    });
  } catch (error) {
    console.error('❌ Error fetching application:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch application',
      error: error.message
    });
  }
});

// Update application
app.put('/hr-applications/:id', async (req, res) => {
  try {
    const updatedApplication = await HRApplication.findOneAndUpdate(
      { id: req.params.id },
      { ...req.body, updatedAt: new Date() },
      { new: true }
    );

    if (!updatedApplication) {
      return res.status(404).json({
        success: false,
        message: 'Application not found'
      });
    }

    console.log('✅ Application updated:', updatedApplication.applicationNumber);

    res.json({
      success: true,
      message: 'Application updated successfully',
      data: updatedApplication
    });
  } catch (error) {
    console.error('❌ Error updating application:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update application',
      error: error.message
    });
  }
});

// Approve/Reject application
app.put('/hr-applications/:id/status', async (req, res) => {
  try {
    const { status, comments, approverUid, approverName, approverRole, reason, timestamp } = req.body;

    console.log('🔄 Updating application status:', {
      id: req.params.id,
      status,
      approverUid,
      approverName,
      approverRole,
      comments,
      reason,
      timestamp
    });

    // First, get the current application to check existing approvals
    let currentApplication = await HRApplication.findOne({ id: req.params.id });
    if (!currentApplication) {
      console.log('🔍 Not found by id field, trying applicationId...');
      currentApplication = await HRApplication.findOne({ applicationId: req.params.id });
    }
    if (!currentApplication) {
      console.log('🔍 Not found by applicationId field, trying _id...');
      currentApplication = await HRApplication.findById(req.params.id);
    }
    
    if (!currentApplication) {
      console.log('❌ Application not found for ID:', req.params.id);
      return res.status(404).json({
        success: false,
        message: 'Application not found'
      });
    }
    
    console.log('✅ Found application for status update:', {
      id: currentApplication.id,
      applicationId: currentApplication.applicationId,
      _id: currentApplication._id
    });

    // Build update data with role-specific approval tracking
    const updateData = {
      status,
      approvalComments: comments,
      updatedAt: new Date()
    };

    // Add role-specific approval data
    if (approverRole === 'Supervisor') {
      updateData.Supervisor = {
        status: status,
        name: approverName,
        approverUid: approverUid,
        timestamp: timestamp || new Date().toISOString(),
        reason: reason || comments || (status === 'Rejected' ? 'No reason provided' : `Approved by ${approverRole}`),
        comments: comments || reason
      };
      console.log('✅ Adding Supervisor approval data:', updateData.Supervisor);
    } else if (approverRole === 'HrManager') {
      updateData.HrManager = {
        status: status,
        name: approverName,
        approverUid: approverUid,
        timestamp: timestamp || new Date().toISOString(),
        reason: reason || comments || (status === 'Rejected' ? 'No reason provided' : `Approved by ${approverRole}`),
        comments: comments || reason
      };
      console.log('✅ Adding HR Manager approval data:', updateData.HrManager);
    } else if (approverRole === 'FinanceManager') {
      updateData.FinanceManager = {
        status: status,
        name: approverName,
        approverUid: approverUid,
        timestamp: timestamp || new Date().toISOString(),
        reason: reason || comments || (status === 'Rejected' ? 'No reason provided' : `Approved by ${approverRole}`),
        comments: comments || reason
      };
      console.log('✅ Adding Finance Manager approval data:', updateData.FinanceManager);
    }

    // Set general approval fields
    if (status === 'Approved') {
      updateData.approvedBy = approverUid;
      updateData.approvedAt = new Date();
    } else if (status === 'Rejected') {
      updateData.rejectedBy = approverUid;
      updateData.rejectedAt = new Date();
    }

    // Use the same ID field that we found the application with
    const updateQuery = currentApplication.id ? { id: currentApplication.id } : 
                       currentApplication.applicationId ? { applicationId: currentApplication.applicationId } :
                       { _id: currentApplication._id };
    
    console.log('🔄 Updating with query:', updateQuery);
    console.log('🔄 Update data being sent to MongoDB:', JSON.stringify(updateData, null, 2));
    
    const updatedApplication = await HRApplication.findOneAndUpdate(
      updateQuery,
      updateData,
      { new: true }
    );
    
    console.log('✅ Updated application from MongoDB:', {
      id: updatedApplication.id,
      applicationId: updatedApplication.applicationId,
      status: updatedApplication.status,
      Supervisor: updatedApplication.Supervisor,
      HrManager: updatedApplication.HrManager,
      FinanceManager: updatedApplication.FinanceManager
    });

    console.log(`✅ Application ${status} by ${approverRole}:`, updatedApplication.applicationNumber);

    res.json({
      success: true,
      message: `Application ${status.toLowerCase()} successfully by ${approverRole}`,
      data: updatedApplication
    });
  } catch (error) {
    console.error('❌ Error updating application status:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update application status',
      error: error.message
    });
  }
});

// Delete application
app.delete('/hr-applications/:id', async (req, res) => {
  try {
    const deletedApplication = await HRApplication.findOneAndDelete({ id: req.params.id });

    if (!deletedApplication) {
      return res.status(404).json({
        success: false,
        message: 'Application not found'
      });
    }

    console.log('🗑️ Application deleted:', deletedApplication.applicationNumber);

    res.json({
      success: true,
      message: 'Application deleted successfully'
    });
  } catch (error) {
    console.error('❌ Error deleting application:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete application',
      error: error.message
    });
  }
});

// Get statistics for admin
app.get('/hr-applications/stats/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    const { countries, projects, onlyMineView } = req.query;

    const countriesArray = countries ? countries.split(',') : [];
    const projectsArray = projects ? projects.split(',') : [];
    const isOnlyMineView = onlyMineView === 'true';

    let baseQuery = {};
    if (isOnlyMineView) {
      baseQuery.adminUid = adminUid;
    } else {
      if (countriesArray.length > 0) {
        baseQuery.country = { $in: countriesArray };
      }
      if (projectsArray.length > 0) {
        baseQuery.project = { $in: projectsArray };
      }
    }

    const stats = await HRApplication.aggregate([
      { $match: baseQuery },
      {
        $group: {
          _id: '$applicationType',
          total: { $sum: 1 },
          pending: {
            $sum: {
              $cond: [{ $eq: ['$status', 'Pending'] }, 1, 0]
            }
          },
          approved: {
            $sum: {
              $cond: [{ $eq: ['$status', 'Approved'] }, 1, 0]
            }
          },
          rejected: {
            $sum: {
              $cond: [{ $eq: ['$status', 'Rejected'] }, 1, 0]
            }
          }
        }
      }
    ]);

    res.json({
      success: true,
      data: stats
    });
  } catch (error) {
    console.error('❌ Error fetching stats:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch statistics',
      error: error.message
    });
  }
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`🚀 HR Form Applications API running on port ${PORT}`);
  console.log(`📡 API URL: http://api.titandrillingzm.com:${PORT}`);
  console.log(`🔗 Health check: http://api.titandrillingzm.com:${PORT}/health`);
});

module.exports = app;

