const express = require('express');
const cors = require('cors');
const { mongoose } = require('./dbConnection'); // Use shared connection
const fetch = require('node-fetch');

const app = express();
const PORT = process.env.FUEL_LOG_PORT || 5011;

// Middleware
app.use(cors());
app.use(express.json());

// Using shared MongoDB connection from dbConnection.js
console.log('✅ Using shared MongoDB connection');

// Fuel Log Schema
const fuelLogSchema = new mongoose.Schema({
  requestNumber: { type: String, required: true, unique: true },
  adminUid: { type: String, required: true, index: true },
  country: { type: String, required: true },
  project: { type: String, required: true },
  equipmentNo: { type: String, required: true },
  equipmentName: { type: String, required: true },
  meterReading: { type: Number, required: true },
  meterUnit: { type: String, required: true },
  fuelQuantity: { type: Number, required: true },
  fuelUnit: { type: String, required: true },
  selectedSource: { type: String, required: true },
  selectedBunker: { type: String, required: true },
  fuelType: { type: String },
  fuelCost: { type: Number },
  note: { type: String },
  receiverName: { type: String },
  receiptImageUri: { type: String },
  fuelBrowserImageUri: { type: String },
  createdAt: { type: String, required: true },
  updatedAt: { type: String, default: () => new Date().toISOString() },
}, { 
  collection: 'GlobalFuelLog',
  timestamps: false 
});

// Add indexes for better query performance
fuelLogSchema.index({ adminUid: 1, createdAt: -1 });
fuelLogSchema.index({ country: 1, project: 1 });
fuelLogSchema.index({ equipmentNo: 1 });

const FuelLog = mongoose.model('FuelLog', fuelLogSchema);

// Counter Schema for request numbers
const counterSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  sequence_value: { type: Number, default: 0 }
}, { collection: 'counters' });

const Counter = mongoose.model('Counter', counterSchema);

// Function to get next sequence number
async function getNextSequence(sequenceName) {
  const counter = await Counter.findByIdAndUpdate(
    sequenceName,
    { $inc: { sequence_value: 1 } },
    { new: true, upsert: true }
  );
  return counter.sequence_value;
}

// Helper function to send fuel log notifications
const sendFuelLogNotifications = async (fuelLogData, createdByName, creatorEmail) => {
  try {
    console.log('🔔 Sending fuel log notifications...');
    
    const notificationPayload = {
      fuelLogData: {
        _id: fuelLogData._id?.toString(),
        id: fuelLogData._id?.toString(),
        requestNumber: fuelLogData.requestNumber,
        equipmentName: fuelLogData.equipmentName || 'N/A',
        equipmentNo: fuelLogData.equipmentNo || 'N/A',
        fuelQuantity: fuelLogData.fuelQuantity,
        fuelUnit: fuelLogData.fuelUnit,
        fuelType: fuelLogData.fuelType || '',
        selectedSource: fuelLogData.selectedSource || '',
        selectedBunker: fuelLogData.selectedBunker || '',
        fuelCost: fuelLogData.fuelCost || 0,
        meterReading: fuelLogData.meterReading,
        meterUnit: fuelLogData.meterUnit,
        country: fuelLogData.country,
        project: fuelLogData.project,
        note: fuelLogData.note || '',
        receiverName: fuelLogData.receiverName || '',
      },
      createdByName,
      creatorEmail,
    };
    
    const response = await fetch('http://api.titandrillingzm.com:5025/api/fuel-log-notifications/new-fuel-log', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(notificationPayload),
    });
    
    const result = await response.json();
    
    if (result.success) {
      console.log('✅ Fuel log notifications sent successfully!');
      console.log('📊 Recipients:', result.recipients);
    } else {
      console.error('❌ Failed to send notifications:', result.error);
    }
    
    return result;
  } catch (error) {
    console.error('❌ Error sending fuel log notifications:', error);
    return { success: false, error: error.message };
  }
};

// ==================== API ENDPOINTS ====================

// GET - Fetch all fuel logs for a specific admin
app.get('/fuel-logs', async (req, res) => {
  try {
    const { adminUid, country, project, equipmentNo } = req.query;
    
    console.log(`📋 [GET /fuel-logs] Fetching fuel logs for adminUid: ${adminUid}`);
    
    if (!adminUid) {
      return res.status(400).json({
        success: false,
        message: 'adminUid is required'
      });
    }

    // Build query
    let query = { adminUid };
    
    if (country) {
      query.country = country;
    }
    
    if (project) {
      query.project = project;
    }
    
    if (equipmentNo) {
      query.equipmentNo = equipmentNo;
    }

    const fuelLogs = await FuelLog.find(query).sort({ createdAt: -1 });
    
    console.log(`✅ [GET /fuel-logs] Found ${fuelLogs.length} fuel logs`);
    
    res.json({
      success: true,
      data: fuelLogs,
      count: fuelLogs.length
    });
  } catch (error) {
    console.error('❌ [GET /fuel-logs] Error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch fuel logs',
      error: error.message
    });
  }
});

// GET - Fetch a single fuel log by ID
app.get('/fuel-logs/:id', async (req, res) => {
  try {
    const { id } = req.params;
    
    console.log(`📋 [GET /fuel-logs/${id}] Fetching fuel log`);
    
    const fuelLog = await FuelLog.findById(id);
    
    if (!fuelLog) {
      return res.status(404).json({
        success: false,
        message: 'Fuel log not found'
      });
    }
    
    console.log(`✅ [GET /fuel-logs/${id}] Fuel log found`);
    
    res.json({
      success: true,
      data: fuelLog
    });
  } catch (error) {
    console.error(`❌ [GET /fuel-logs/${id}] Error:`, error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch fuel log',
      error: error.message
    });
  }
});

// POST - Create a new fuel log
app.post('/fuel-logs', async (req, res) => {
  try {
    const fuelLogData = req.body;
    
    console.log(`📋 [POST /fuel-logs] Creating new fuel log`);
    console.log('Request data:', JSON.stringify(fuelLogData, null, 2));
    
    if (!fuelLogData.adminUid) {
      return res.status(400).json({
        success: false,
        message: 'adminUid is required'
      });
    }

    // Generate request number if not provided
    if (!fuelLogData.requestNumber) {
      const nextNumber = await getNextSequence('fuellog_request_number');
      const countryCode = fuelLogData.country === 'Zambia' ? 'ZMB' : 
                          fuelLogData.country === 'DRC' ? 'DRC' : 
                          fuelLogData.country.substring(0, 3).toUpperCase();
      const projectCode = fuelLogData.project.substring(0, 3).toUpperCase();
      fuelLogData.requestNumber = `${countryCode}-${projectCode}-FL-${nextNumber.toString().padStart(3, '0')}`;
    }

    // Set timestamps
    if (!fuelLogData.createdAt) {
      fuelLogData.createdAt = new Date().toISOString();
    }
    fuelLogData.updatedAt = new Date().toISOString();

    const newFuelLog = new FuelLog(fuelLogData);
    const savedFuelLog = await newFuelLog.save();
    
    console.log(`✅ [POST /fuel-logs] Fuel log created successfully with ID: ${savedFuelLog._id}`);
    console.log(`✅ Request Number: ${savedFuelLog.requestNumber}`);
    
    // Send notifications in background (fire-and-forget)
    setImmediate(async () => {
      try {
        // Get creator name and email from adminUid - using different database
        const GlobalUserSchema = new mongoose.Schema({}, { strict: false, collection: 'GlobalUsers' });
        let GlobalUser;
        try {
          GlobalUser = mongoose.models.GlobalUser || mongoose.model('GlobalUser', GlobalUserSchema);
        } catch (error) {
          GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
        }
        
        // Use titan_drilling database for users
        const titanDrillingDB = mongoose.connection.useDb('titan_drilling');
        const GlobalUserModel = titanDrillingDB.model('GlobalUser', GlobalUserSchema);
        
        const creator = await GlobalUserModel.findOne({ _id: fuelLogData.adminUid }).select('name email');
        const createdByName = creator?.name || 'Unknown User';
        const creatorEmail = creator?.email || '';
        
        console.log('📧 Sending fuel log notifications for creator:', createdByName);
        await sendFuelLogNotifications(savedFuelLog.toObject(), createdByName, creatorEmail);
      } catch (notifError) {
        console.error('❌ Error sending fuel log notifications:', notifError);
      }
    });
    
    res.status(201).json({
      success: true,
      message: 'Fuel log created successfully',
      data: savedFuelLog
    });
  } catch (error) {
    console.error('❌ [POST /fuel-logs] Error:', error);
    
    if (error.code === 11000) {
      return res.status(400).json({
        success: false,
        message: 'Fuel log with this request number already exists',
        error: error.message
      });
    }
    
    res.status(500).json({
      success: false,
      message: 'Failed to create fuel log',
      error: error.message
    });
  }
});

// PUT - Update a fuel log
app.put('/fuel-logs/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const updateData = req.body;
    
    console.log(`📋 [PUT /fuel-logs/${id}] Updating fuel log`);
    console.log('Update data:', JSON.stringify(updateData, null, 2));
    
    // Update timestamp
    updateData.updatedAt = new Date().toISOString();
    
    const updatedFuelLog = await FuelLog.findByIdAndUpdate(
      id,
      updateData,
      { new: true, runValidators: true }
    );
    
    if (!updatedFuelLog) {
      return res.status(404).json({
        success: false,
        message: 'Fuel log not found'
      });
    }
    
    console.log(`✅ [PUT /fuel-logs/${id}] Fuel log updated successfully`);
    
    res.json({
      success: true,
      message: 'Fuel log updated successfully',
      data: updatedFuelLog
    });
  } catch (error) {
    console.error(`❌ [PUT /fuel-logs/${id}] Error:`, error);
    res.status(500).json({
      success: false,
      message: 'Failed to update fuel log',
      error: error.message
    });
  }
});

// DELETE - Delete a fuel log
app.delete('/fuel-logs/:id', async (req, res) => {
  try {
    const { id } = req.params;
    
    console.log(`📋 [DELETE /fuel-logs/${id}] Deleting fuel log`);
    
    const deletedFuelLog = await FuelLog.findByIdAndDelete(id);
    
    if (!deletedFuelLog) {
      return res.status(404).json({
        success: false,
        message: 'Fuel log not found'
      });
    }
    
    console.log(`✅ [DELETE /fuel-logs/${id}] Fuel log deleted successfully`);
    
    res.json({
      success: true,
      message: 'Fuel log deleted successfully',
      data: deletedFuelLog
    });
  } catch (error) {
    console.error(`❌ [DELETE /fuel-logs/${id}] Error:`, error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete fuel log',
      error: error.message
    });
  }
});

// GET - Get statistics for admin
app.get('/fuel-logs/stats/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    
    console.log(`📋 [GET /fuel-logs/stats/${adminUid}] Fetching statistics`);
    
    const stats = await FuelLog.aggregate([
      { $match: { adminUid } },
      {
        $group: {
          _id: null,
          totalLogs: { $sum: 1 },
          totalFuelQuantity: { $sum: '$fuelQuantity' },
          totalFuelCost: { $sum: '$fuelCost' },
          avgFuelQuantity: { $avg: '$fuelQuantity' },
          avgFuelCost: { $avg: '$fuelCost' }
        }
      }
    ]);
    
    console.log(`✅ [GET /fuel-logs/stats/${adminUid}] Statistics fetched`);
    
    res.json({
      success: true,
      data: stats.length > 0 ? stats[0] : {
        totalLogs: 0,
        totalFuelQuantity: 0,
        totalFuelCost: 0,
        avgFuelQuantity: 0,
        avgFuelCost: 0
      }
    });
  } catch (error) {
    console.error(`❌ [GET /fuel-logs/stats/${adminUid}] Error:`, error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch statistics',
      error: error.message
    });
  }
});

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({
    success: true,
    message: 'Fuel Log Handler API is running',
    port: PORT,
    timestamp: new Date().toISOString()
  });
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`🚀 [Fuel Log Handler] Server running on port ${PORT}`);
  console.log(`📡 API URL: http://api.titandrillingzm.com:${PORT}`);
});

module.exports = app;

