@extends('welcome')

@section('title', 'Parts Inventory')

@section('content')
<div class="parts-inventory-container">
    <div class="parts-inventory-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Parts Inventory</h1>
                <p>Manage and track parts inventory efficiently</p>
            </div>
        </div>
    </div>
    
    <div class="parts-inventory-content">
        <div class="tab-navigation">
            <div class="tab-list">
                <button class="tab-btn active" data-tab="parts">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M12 2L2 7l10 5 10-5-10-5z"/>
                        <path d="M2 17l10 5 10-5"/>
                        <path d="M2 12l10 5 10-5"/>
                    </svg>
                    <span>Parts</span>
                </button>
                <button class="tab-btn" data-tab="transfer-history">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                        <circle cx="9" cy="7" r="4"/>
                        <path d="m22 21-3-3m0 0a2 2 0 1 0-4 0 2 2 0 0 0 4 0Z"/>
                    </svg>
                    <span>Transfer History</span>
                </button>
            </div>
        </div>
        <div class="tab-content">
            <div id="parts" class="tab-pane active">
                <div class="action-bar">
                    <div class="action-content">
                        <div class="search-section">
                            <div class="search-container">
                                <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" class="search-input" placeholder="Search parts..." id="partsSearch">
                            </div>
                        </div>
                        
                        <div class="action-buttons">
                            <button class="btn-category" data-bs-toggle="modal" data-bs-target="#add-category">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M20 4H4a1 1 0 0 0-1 1v14a1 1 0 0 0 1 1h16a1 1 0 0 0 1-1V5a1 1 0 0 0-1-1z"/>
                                    <path d="M8 8h8M8 12h8M8 16h5"/>
                                </svg>
                                <span>Parts Category</span>
                            </button>
                            
                            <button class="btn-add" data-bs-toggle="modal" data-bs-target="#add-parts-item">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M12 5v14m-7-7h14"/>
                                </svg>
                                <span>Add Parts Item</span>
                            </button>
                            
                            <button class="btn-upload" data-bs-toggle="modal" data-bs-target="#add-checklist">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                    <polyline points="17,8 12,3 7,8"/>
                                    <line x1="12" y1="3" x2="12" y2="15"/>
                                </svg>
                                <span>Bulk Upload</span>
                            </button>
                            
                            <button class="btn-sort" data-bs-toggle="modal" data-bs-target="#sorting">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M3 6h18M7 12h10M10 18h4"/>
                                </svg>
                                <span>Sorting</span>
                            </button>
                            
                            <button class="btn-export">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                    <polyline points="7,10 12,15 17,10"/>
                                    <line x1="12" y1="15" x2="12" y2="3"/>
                                </svg>
                                <span>Excel Export</span>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="table-section">
                    <div class="table-container">
                        <table class="parts-inventory-table">
                            <thead class="table-header">
                                <tr>
                                    <th>Action</th>
                                    <th>Parts</th>
                                    <th>Quantity</th>
                                    <th>Item Cost</th>
                                    <th>Item Type</th>
                                    <th>Manufacturer</th>
                                    <th>Supplier</th>
                                    <th>Description</th>
                                    <th>Site</th>
                                </tr>
                            </thead>
                            <tbody class="table-body" id="partsTableBody">
                                <tr class="empty-row">
                                    <td colspan="9" class="empty-state">
                                        <div class="empty-content">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                                <path d="M12 2L2 7l10 5 10-5-10-5z"/>
                                                <path d="M2 17l10 5 10-5"/>
                                                <path d="M2 12l10 5 10-5"/>
                                            </svg>
                                            <h3>No Parts Data</h3>
                                            <p>There are currently no parts in inventory to display.</p>
                                        </div>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <div id="transfer-history" class="tab-pane">
                <div class="action-bar">
                    <div class="action-content">
                        <div class="search-section">
                            <div class="search-container">
                                <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" class="search-input" placeholder="Search transfer history..." id="transferSearch">
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="table-section">
                    <div class="table-container">
                        <table class="parts-inventory-table">
                            <thead class="table-header">
                                <tr>
                                    <th>Transfer Date</th>
                                    <th>Parts</th>
                                    <th>Quantity</th>
                                    <th>Manufacturer</th>
                                    <th>Site From</th>
                                    <th>Site To</th>
                                    <th>Transfer By</th>
                                    <th>Transfer To</th>
                                    <th>Note</th>
                                </tr>
                            </thead>
                            <tbody class="table-body" id="transferTableBody">
                                <tr class="empty-row">
                                    <td colspan="9" class="empty-state">
                                        <div class="empty-content">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                                <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                                                <circle cx="9" cy="7" r="4"/>
                                                <path d="m22 21-3-3m0 0a2 2 0 1 0-4 0 2 2 0 0 0 4 0Z"/>
                                            </svg>
                                            <h3>No Transfer History</h3>
                                            <p>There are currently no transfer records to display.</p>
                                        </div>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
    <script>
        // Tab functionality
        document.addEventListener('DOMContentLoaded', function() {
            const tabButtons = document.querySelectorAll('.tab-btn');
            const tabPanes = document.querySelectorAll('.tab-pane');
            
            tabButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const targetTab = this.getAttribute('data-tab');
                    
                    // Remove active class from all buttons and panes
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    tabPanes.forEach(pane => pane.classList.remove('active'));
                    
                    // Add active class to clicked button and corresponding pane
                    this.classList.add('active');
                    document.getElementById(targetTab).classList.add('active');
                });
            });
        });

        // Search functionality
        document.addEventListener('DOMContentLoaded', function() {
            const partsSearch = document.getElementById('partsSearch');
            const transferSearch = document.getElementById('transferSearch');
            
            if (partsSearch) {
                partsSearch.addEventListener('input', function() {
                    filterTable('partsTableBody', this.value);
                });
            }
            
            if (transferSearch) {
                transferSearch.addEventListener('input', function() {
                    filterTable('transferTableBody', this.value);
                });
            }
        });

        function filterTable(tableBodyId, searchTerm) {
            const tableBody = document.getElementById(tableBodyId);
            if (!tableBody) return;
            
            const rows = tableBody.getElementsByTagName('tr');
            
            for (let row of rows) {
                const cells = row.getElementsByTagName('td');
                let found = false;
                
                for (let cell of cells) {
                    if (cell.textContent.toLowerCase().includes(searchTerm.toLowerCase())) {
                        found = true;
                        break;
                    }
                }
                
                row.style.display = found ? '' : 'none';
            }
        }

        // Notification system
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `parts-notification parts-notification-${type}`;
            notification.textContent = message;
            
            notification.style.cssText = `
                position: fixed;
                top: 80px;
                right: 20px;
                padding: 12px 20px;
                border-radius: 0;
                color: white;
                font-weight: 500;
                font-size: 14px;
                z-index: 10001;
                opacity: 0;
                transform: translateX(100%);
                transition: all 0.3s ease;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                font-family: 'Inter', sans-serif;
            `;
            
            if (type === 'success') {
                notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
            } else if (type === 'error') {
                notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
            } else if (type === 'warning') {
                notification.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
            } else {
                notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
            }
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.opacity = '1';
                notification.style.transform = 'translateX(0)';
            }, 100);
            
            setTimeout(() => {
                notification.style.opacity = '0';
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        document.body.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }
    </script>
</div>
<!-- Add New Category Popup modals -->
<div class="modal" id="add-category">
    <div class="modal-dialog" style="">
        <div class="modal-content">
            <!-- Modal Header -->
            <div class="modal-header">
                <h4 class="modal-title">Add New Category</h4>
            </div>
            <!-- Modal body -->
            <div class="modal-body">
                <form class="form-inline" action="">
                    <div class="input-group">
                        <label for="text">New Category :</label>
                        <input type="text" class="form-control" placeholder="" id="title">
                    </div>
                    <h1>Category</h1>
                    <div class="text-center"><span class="empty-data text-center">No Data</span></div>
                    <hr>
                    <div class="row my-3 align-items-center justify-content-center">
                        <button type="submit" class="save btn btn-danger col-md-4 mx-2"><svg xmlns="http://www.w3.org/2000/svg" width="34" height="34" viewBox="0 0 24 24">
                                <path fill="currentColor" d="M18 12.998h-5v5a1 1 0 0 1-2 0v-5H6a1 1 0 0 1 0-2h5v-5a1 1 0 0 1 2 0v5h5a1 1 0 0 1 0 2" />
                            </svg> Add</button>
                        <!-- Modal footer -->
                        <div class="modal-footer col-md-4 p-0 w-auto">
                            <button type="button" class="btn btn-danger" data-bs-dismiss="modal"><svg xmlns="http://www.w3.org/2000/svg" width="35" height="35" viewBox="0 0 24 24">
                                    <path fill="currentColor" fill-rule="evenodd" d="M7.345 4.017a42.3 42.3 0 0 1 9.31 0c1.713.192 3.095 1.541 3.296 3.26a40.7 40.7 0 0 1 0 9.446c-.201 1.719-1.583 3.068-3.296 3.26a42.3 42.3 0 0 1-9.31 0c-1.713-.192-3.095-1.541-3.296-3.26a40.7 40.7 0 0 1 0-9.445a3.734 3.734 0 0 1 3.295-3.26M8.47 8.47a.75.75 0 0 1 1.06 0L12 10.94l2.47-2.47a.75.75 0 1 1 1.06 1.06L13.06 12l2.47 2.47a.75.75 0 0 1-1.06 1.06L12 13.06l-2.47 2.47a.75.75 0 1 1-1.06-1.06L10.94 12L8.47 9.53a.75.75 0 0 1 0-1.06" clip-rule="evenodd" />
                                </svg>Cancel</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<!-- Add Parts Item Popup modals -->
<div class="modal" id="add-parts-item">
    <div class="modal-dialog" style="">
        <div class="modal-content">
            <!-- Modal Header -->
            <div class="modal-header">
                <h4 class="modal-title">Add Parts Item</h4>
            </div>
            <!-- Modal body -->
            <div class="modal-body">
                <div class="form-inline" action="">
                    <div class="row">
                        <div class="input-group">
                            <label for="text">Category * :</label>
                            <select class="form-select" id="sel1" name="sellist1">
                                <option></option>
                                <option></option>
                                <option></option>
                                <option></option>
                            </select>
                            <button type="submit" class="plus-btn mx-2" data-bs-toggle="modal" data-bs-target="#plus-add-category"><svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24">
                                    <path fill="currentColor" d="M18 12.998h-5v5a1 1 0 0 1-2 0v-5H6a1 1 0 0 1 0-2h5v-5a1 1 0 0 1 2 0v5h5a1 1 0 0 1 0 2" />
                                </svg></button>
                        </div>
                        <div class="input-group">
                            <label for="text">Serial Number :</label>
                            <input type="text" class="form-control" placeholder="Input serial number/barcode and pres......" id="vnum"> <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" id="list-select">
                                <path fill="currentColor" d="M7.024 3.75c0-.966.784-1.75 1.75-1.75H20.25c.966 0 1.75.784 1.75 1.75v11.498a1.75 1.75 0 0 1-1.75 1.75H8.774a1.75 1.75 0 0 1-1.75-1.75Zm1.75-.25a.25.25 0 0 0-.25.25v11.498c0 .139.112.25.25.25H20.25a.25.25 0 0 0 .25-.25V3.75a.25.25 0 0 0-.25-.25Z" />
                                <path fill="currentColor" d="M1.995 10.749a1.75 1.75 0 0 1 1.75-1.751H5.25a.75.75 0 1 1 0 1.5H3.745a.25.25 0 0 0-.25.25L3.5 20.25c0 .138.111.25.25.25h9.5a.25.25 0 0 0 .25-.25v-1.51a.75.75 0 1 1 1.5 0v1.51A1.75 1.75 0 0 1 13.25 22h-9.5A1.75 1.75 0 0 1 2 20.25z" />
                            </svg>
                        </div>
                    </div>
                    <div class="row">
                        <div class="input-group">
                            <label for="text">SKU ID * :</label>
                            <input type="text" class="form-control" placeholder="" id="title">
                        </div>
                        <div class="input-group">
                            <label for="text">Item Type :</label>
                            <input type="text" class="form-control" placeholder="" id="title">
                        </div>
                    </div>
                    <div class="row">
                        <div class="input-group">
                            <label for="text">Part Number * :</label>
                            <input type="text" class="form-control" placeholder="" id="title">
                        </div>
                        <div class="input-group">
                            <label for="text">Stored Location :</label>
                            <input type="text" class="form-control" placeholder="" id="title">
                        </div>
                    </div>
                    <div class="row">
                        <div class="input-group">
                            <label for="text">Item Name * :</label>
                            <input type="text" class="form-control" placeholder="" id="title">
                        </div>
                        <div class="input-group">
                            <label for="text">Lookup Tag :</label>
                            <input type="text" class="form-control" placeholder="Input any tag and press enter key" id="vnum"> <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" id="list-select">
                                <path fill="currentColor" d="M7.024 3.75c0-.966.784-1.75 1.75-1.75H20.25c.966 0 1.75.784 1.75 1.75v11.498a1.75 1.75 0 0 1-1.75 1.75H8.774a1.75 1.75 0 0 1-1.75-1.75Zm1.75-.25a.25.25 0 0 0-.25.25v11.498c0 .139.112.25.25.25H20.25a.25.25 0 0 0 .25-.25V3.75a.25.25 0 0 0-.25-.25Z" />
                                <path fill="currentColor" d="M1.995 10.749a1.75 1.75 0 0 1 1.75-1.751H5.25a.75.75 0 1 1 0 1.5H3.745a.25.25 0 0 0-.25.25L3.5 20.25c0 .138.111.25.25.25h9.5a.25.25 0 0 0 .25-.25v-1.51a.75.75 0 1 1 1.5 0v1.51A1.75 1.75 0 0 1 13.25 22h-9.5A1.75 1.75 0 0 1 2 20.25z" />
                            </svg>
                        </div>
                    </div>
                    <div class="row">
                        <div class="input-group">
                            <label for="text">Quantity * :</label>
                            <input type="text" class="form-control" placeholder="" id="title">
                            <select class="form-select" id="sel1" name="sellist1">
                                <option>Piece</option>
                                <option>KG</option>
                                <option>Gram</option>
                                <option>Gallon</option>
                                <option>Ltr</option>
                                <option>ML</option>
                                <option>Meter</option>
                            </select>
                        </div>
                        <div class="input-group">
                            <label for="text">Critical Part :</label>
                            <input class="form-check-input" type="checkbox" id="check1" name="option1" value="something">
                        </div>
                    </div>
                    <div class="row">
                        <div class="input-group">
                            <label for="text">Item Cost * :</label>
                            <input type="text" class="form-control" placeholder="" id="title">
                        </div>
                        <div class="input-group">
                            <label for="text">Description :</label>
                            <input type="text" class="form-control" placeholder="" id="title">
                        </div>
                    </div>
                    <div class="row">
                        <div class="input-group">
                            <label for="">Manufacturer * :</label>
                            <input type="text" class="form-control" placeholder="" id="">
                        </div>
                        <div class="input-group">
                            <label for="">Supplier :</label>
                            <input type="text" class="form-control" placeholder="" id="">
                        </div>
                    </div>
                    <div class="row">
                        <div class="input-group">
                            <label for="">Linked Vehicle Model :</label>
                            <input type="text" class="form-control" placeholder="" id=""> <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" id="list-select">
                                <path fill="currentColor" d="M7.024 3.75c0-.966.784-1.75 1.75-1.75H20.25c.966 0 1.75.784 1.75 1.75v11.498a1.75 1.75 0 0 1-1.75 1.75H8.774a1.75 1.75 0 0 1-1.75-1.75Zm1.75-.25a.25.25 0 0 0-.25.25v11.498c0 .139.112.25.25.25H20.25a.25.25 0 0 0 .25-.25V3.75a.25.25 0 0 0-.25-.25Z" />
                                <path fill="currentColor" d="M1.995 10.749a1.75 1.75 0 0 1 1.75-1.751H5.25a.75.75 0 1 1 0 1.5H3.745a.25.25 0 0 0-.25.25L3.5 20.25c0 .138.111.25.25.25h9.5a.25.25 0 0 0 .25-.25v-1.51a.75.75 0 1 1 1.5 0v1.51A1.75 1.75 0 0 1 13.25 22h-9.5A1.75 1.75 0 0 1 2 20.25z" />
                            </svg>
                        </div>
                        <div class="input-group">
                            <label for="">Low Part Threshold :</label>
                            <input type="text" class="form-control" placeholder="" id="">
                        </div>
                    </div>
                    <div class="row">

                        <div class="input-group">
                            <label for="">Warranty Term(In Months) :</label>
                            <input type="text" class="form-control" placeholder="" id="">
                        </div>
                        <div class="input-group">
                            <label for="">Create automatic<br>demand :</label>
                            <input class="form-check-input" type="checkbox" id="check1" name="option1" value="something"><br>
                            System will send email alert when the part is lower than threshold
                        </div>
                    </div>
                    <div class="row">
                        <div class="input-group">
                            <label for="">Reference Part :</label>
                            <input type="text" class="form-control" placeholder="" id=""> <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" id="list-select">
                                <path fill="currentColor" d="M7.024 3.75c0-.966.784-1.75 1.75-1.75H20.25c.966 0 1.75.784 1.75 1.75v11.498a1.75 1.75 0 0 1-1.75 1.75H8.774a1.75 1.75 0 0 1-1.75-1.75Zm1.75-.25a.25.25 0 0 0-.25.25v11.498c0 .139.112.25.25.25H20.25a.25.25 0 0 0 .25-.25V3.75a.25.25 0 0 0-.25-.25Z" />
                                <path fill="currentColor" d="M1.995 10.749a1.75 1.75 0 0 1 1.75-1.751H5.25a.75.75 0 1 1 0 1.5H3.745a.25.25 0 0 0-.25.25L3.5 20.25c0 .138.111.25.25.25h9.5a.25.25 0 0 0 .25-.25v-1.51a.75.75 0 1 1 1.5 0v1.51A1.75 1.75 0 0 1 13.25 22h-9.5A1.75 1.75 0 0 1 2 20.25z" />
                            </svg>
                        </div>

                    </div>
                    <!-- <div class="input-group">
                            <div class="input-group-text br-0">
                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24">
                                    <path fill="currentColor" d="M15.5 14h-.79l-.28-.27A6.47 6.47 0 0 0 16 9.5A6.5 6.5 0 1 0 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5S14 7.01 14 9.5S11.99 14 9.5 14" /></svg>
                            </div>
                            <input type="text" class="form-control" placeholder="Search">
                        </div> -->
                    <div class="row my-3 align-items-center justify-content-center">
                        <button type="submit" class="save btn btn-danger col-md-4 mx-2"><svg xmlns="http://www.w3.org/2000/svg" width="34" height="34" viewBox="0 0 24 24">
                                <path fill="currentColor" d="M18 12.998h-5v5a1 1 0 0 1-2 0v-5H6a1 1 0 0 1 0-2h5v-5a1 1 0 0 1 2 0v5h5a1 1 0 0 1 0 2" />
                            </svg> Add</button>
                        <!-- Modal footer -->
                        <div class="modal-footer col-md-4 p-0 w-auto">
                            <button type="button" class="save btn btn-danger" data-bs-dismiss="modal"><svg xmlns="http://www.w3.org/2000/svg" width="35" height="35" viewBox="0 0 24 24">
                                    <path fill="currentColor" fill-rule="evenodd" d="M7.345 4.017a42.3 42.3 0 0 1 9.31 0c1.713.192 3.095 1.541 3.296 3.26a40.7 40.7 0 0 1 0 9.446c-.201 1.719-1.583 3.068-3.296 3.26a42.3 42.3 0 0 1-9.31 0c-1.713-.192-3.095-1.541-3.296-3.26a40.7 40.7 0 0 1 0-9.445a3.734 3.734 0 0 1 3.295-3.26M8.47 8.47a.75.75 0 0 1 1.06 0L12 10.94l2.47-2.47a.75.75 0 1 1 1.06 1.06L13.06 12l2.47 2.47a.75.75 0 0 1-1.06 1.06L12 13.06l-2.47 2.47a.75.75 0 1 1-1.06-1.06L10.94 12L8.47 9.53a.75.75 0 0 1 0-1.06" clip-rule="evenodd" />
                                </svg>Cancel</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<!-- inner Add New Category Popup modals -->
<div class="modal" id="plus-add-category">
    <div class="modal-dialog" style="">
        <div class="modal-content">
            <!-- Modal Header -->
            <div class="modal-header">
                <h4 class="modal-title">Add New Category</h4>
            </div>
            <!-- Modal body -->
            <div class="modal-body">
                <form class="form-inline" action="">
                    <div class="input-group">
                        <label for="text">New Category :</label>
                        <input type="text" class="form-control" placeholder="" id="title">
                    </div>
                    <div class="row my-3 align-items-center justify-content-center">
                        <button type="submit" class="save btn btn-danger col-md-4 mx-2"><svg xmlns="http://www.w3.org/2000/svg" width="34" height="34" viewBox="0 0 24 24">
                                <path fill="currentColor" d="M18 12.998h-5v5a1 1 0 0 1-2 0v-5H6a1 1 0 0 1 0-2h5v-5a1 1 0 0 1 2 0v5h5a1 1 0 0 1 0 2" />
                            </svg> Add</button>
                        <!-- Modal footer -->
                        <div class="modal-footer col-md-4 p-0 w-auto">
                            <button type="button" class="btn btn-danger" data-bs-dismiss="modal"><svg xmlns="http://www.w3.org/2000/svg" width="35" height="35" viewBox="0 0 24 24">
                                    <path fill="currentColor" fill-rule="evenodd" d="M7.345 4.017a42.3 42.3 0 0 1 9.31 0c1.713.192 3.095 1.541 3.296 3.26a40.7 40.7 0 0 1 0 9.446c-.201 1.719-1.583 3.068-3.296 3.26a42.3 42.3 0 0 1-9.31 0c-1.713-.192-3.095-1.541-3.296-3.26a40.7 40.7 0 0 1 0-9.445a3.734 3.734 0 0 1 3.295-3.26M8.47 8.47a.75.75 0 0 1 1.06 0L12 10.94l2.47-2.47a.75.75 0 1 1 1.06 1.06L13.06 12l2.47 2.47a.75.75 0 0 1-1.06 1.06L12 13.06l-2.47 2.47a.75.75 0 1 1-1.06-1.06L10.94 12L8.47 9.53a.75.75 0 0 1 0-1.06" clip-rule="evenodd" />
                                </svg>Cancel</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<!-- Sorting Popup modals -->
<div class="modal" id="sorting">
    <div class="modal-dialog">
        <div class="modal-content">
            <!-- Modal Header -->
            <div class="modal-header">
                <h4 class="modal-title">Sorting</h4>
            </div>
            <!-- Modal body -->
            <div class="modal-body">
                <form action="">
                    <form action="">
                        <h1>Sort Order</h1>
                        <div class="form-check">
                            <input type="radio" class="form-check-input" id="radio1" name="optradio" value="Ascending" checked>
                            <label class="form-check-label" for="radio1">Ascending</label>
                        </div>
                        <div class="form-check">
                            <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Descending">
                            <label class="form-check-label" for="radio2">Descending</label>
                        </div>
                    </form>
                    <h1>Sort By</h1>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio1" name="optradio" value="Report Number" checked>
                        <label class="form-check-label" for="radio1">Item Number</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Start Date">
                        <label class="form-check-label" for="radio2">Item Name</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Number">
                        <label class="form-check-label" for="radio2">Description</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Name">
                        <label class="form-check-label" for="radio2">Quantity</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Name">
                        <label class="form-check-label" for="radio2">Item Cost</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Name">
                        <label class="form-check-label" for="radio2">Item Type</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Name">
                        <label class="form-check-label" for="radio2">Supplier</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Name">
                        <label class="form-check-label" for="radio2">Created Date</label>
                    </div>
                    <div class="row my-3 align-items-center justify-content-center">
                        <button type="submit" class="save btn btn-danger col-md-4 mx-2">Ok</button>
                        <button type="reset" class="save btn btn-danger col-md-4 mx-2">Reset</button>
                        <!-- Modal footer -->
                        <div class="modal-footer col-md-4 p-0">
                            <button type="button" class="btn btn-danger" data-bs-dismiss="modal">Cancel</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<script>
(function() {
    'use strict';

    // Configuration object for centralized settings
    const CONFIG = {
        selectors: {
            dropdownBtn: '.dropdown-btn',
            partsSearch: '#partsSearch',
            partsTableBody: '#partsTableBody',
            transferHistoryTableBody: '#transferHistoryTableBody'
        },
        messages: {
            loading: 'Loading...',
            error: 'An error occurred. Please try again.',
            success: 'Operation completed successfully!'
        },
        timing: {
            debounceDelay: 300
        }
    };

    // DOM element caching
    let elements = {};

    // Initialize DOM elements
    function initElements() {
        elements = {
            dropdownBtns: document.querySelectorAll(CONFIG.selectors.dropdownBtn),
            partsSearch: document.querySelector(CONFIG.selectors.partsSearch),
            partsTableBody: document.querySelector(CONFIG.selectors.partsTableBody),
            transferHistoryTableBody: document.querySelector(CONFIG.selectors.transferHistoryTableBody)
        };
    }

    // Initialize the application
    function init() {
        initElements();
        initEventListeners();
        initDropdowns();
    }

    // Initialize event listeners
    function initEventListeners() {
        // Search functionality
        if (elements.partsSearch) {
            elements.partsSearch.addEventListener('input', function() {
                filterTable(CONFIG.selectors.partsTableBody, this.value);
            });
        }
    }

    // Initialize dropdowns
    function initDropdowns() {
        elements.dropdownBtns.forEach(dropdown => {
            dropdown.addEventListener('click', function() {
                this.classList.toggle('active');
                const dropdownContent = this.nextElementSibling;
                if (dropdownContent.style.display === 'block') {
                    dropdownContent.style.display = 'none';
                } else {
                    dropdownContent.style.display = 'block';
                }
            });
        });
    }

    // Filter table function
    function filterTable(tableBodyId, searchTerm) {
        const tableBody = document.querySelector(tableBodyId);
        if (!tableBody) return;
        
        const rows = tableBody.getElementsByTagName('tr');
        
        for (let row of rows) {
            const cells = row.getElementsByTagName('td');
            let found = false;
            
            for (let cell of cells) {
                if (cell.textContent.toLowerCase().includes(searchTerm.toLowerCase())) {
                    found = true;
                    break;
                }
            }
            
            row.style.display = found ? '' : 'none';
        }
    }

    // Initialize the application
    init();
})();
</script>

<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --secondary-hover: #5a6268;
    --success-color: #28a745;
    --success-hover: #218838;
    --warning-color: #ffc107;
    --warning-hover: #e0a800;
    --danger-color: #dc3545;
    --danger-hover: #c82333;
    --info-color: #17a2b8;
    --info-hover: #138496;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #6c757d;
    --text-light: #ffffff;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-hover: #e9ecef;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --transition-slow: all 0.3s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-modal: 1050;
    --z-index-dropdown: 1000;
}

/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== PARTS INVENTORY CONTAINER ===== */
.parts-inventory-container {
    background: var(--bg-light);
}

/* ===== HEADER SECTION ===== */
.parts-inventory-header {
    background: var(--primary-color);
    border-radius: 0;
    padding: 32px;
    margin-bottom: 24px;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.header-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.page-title h1 {
    color: var(--text-light);
    font-size: 32px;
    font-weight: 700;
    margin: 0 0 8px 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.page-title p {
    color: rgba(255, 255, 255, 0.9);
    font-size: 16px;
    font-weight: 400;
    margin: 0;
}

/* ===== CONTENT SECTION ===== */
.parts-inventory-content {
    background: var(--bg-primary);
    border-radius: 0;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

/* ===== TAB NAVIGATION ===== */
.tab-navigation {
    background: var(--bg-light);
    border-bottom: 1px solid var(--border-light);
    padding: 0;
}

.tab-list {
    display: flex;
    gap: 0;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 16px 24px;
    background: transparent;
    border: none;
    color: var(--text-muted);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    border-radius: 0;
    position: relative;
}

.tab-btn:hover {
    background: rgba(0, 85, 165, 0.05);
    color: var(--primary-color);
}

.tab-btn.active {
    background: var(--bg-primary);
    color: var(--primary-color);
    border-bottom: 2px solid var(--primary-color);
}

.tab-btn svg {
    flex-shrink: 0;
}

/* ===== TAB CONTENT ===== */
.tab-content {
    position: relative;
}

.tab-pane {
    display: none;
    padding: 0;
}

.tab-pane.active {
    display: block;
}

/* ===== ACTION BAR ===== */
.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.action-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 20px;
}

.search-section {
    flex: 1;
    max-width: 400px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #0055a5;
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.action-buttons {
    display: flex;
    align-items: center;
    gap: 12px;
}

.action-buttons button {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-buttons button:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    transform: translateY(-1px);
}

.btn-category {
    background: #ffd205 !important;
    color: #000 !important;
    border-color: #ffd205 !important;
}

.btn-category:hover {
    background: #e6bd04 !important;
    border-color: #e6bd04 !important;
    color: #000 !important;
}

.btn-add {
    background: #00B894 !important;
    color: white !important;
    border-color: #00B894 !important;
}

.btn-add:hover {
    background: #00a085 !important;
    border-color: #00a085 !important;
}

.btn-upload {
    background: #0055a5 !important;
    color: white !important;
    border-color: #0055a5 !important;
}

.btn-upload:hover {
    background: #003f7d !important;
    border-color: #003f7d !important;
}

.btn-sort {
    background: #545d6b !important;
    color: white !important;
    border-color: #545d6b !important;
}

.btn-sort:hover {
    background: #3d4449 !important;
    border-color: #3d4449 !important;
}

.btn-export {
    background: #0055a5 !important;
    color: white !important;
    border-color: #0055a5 !important;
}

.btn-export:hover {
    background: #003f7d !important;
    border-color: #003f7d !important;
}

/* ===== TABLE SECTION ===== */
.table-section {
    padding: 0;
}

.table-container {
    overflow-x: auto;
}

.parts-inventory-table {
    width: 100%;
    border-collapse: collapse;
    background: white;
}

.table-header {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.table-header th {
    padding: 16px 12px;
    text-align: left;
    font-size: 14px;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e5e7eb;
    border-radius: 0;
}

.table-body {
    background: white;
}

.table-row {
    border-bottom: 1px solid #f3f4f6;
    transition: all 0.2s ease;
}

.table-row:hover {
    background: #f8fafc;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.table-row td {
    padding: 16px 12px;
    font-size: 14px;
    color: #374151;
    vertical-align: middle;
}

/* ===== EMPTY STATE ===== */
.empty-row {
    background: white;
}

.empty-state {
    text-align: center;
    padding: 48px 24px;
    background: white;
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.empty-content svg {
    color: #9ca3af;
}

.empty-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: #374151;
    margin: 0;
}

.empty-content p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .parts-inventory-container {
        padding: 20px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 16px;
    }
    
    .search-section {
        max-width: none;
    }
    
    .action-buttons {
        justify-content: center;
        flex-wrap: wrap;
    }
    
    .modal-dialog {
        width: 90%;
        max-width: 800px;
    }
}

@media (max-width: 992px) {
    .parts-inventory-container {
        padding: 16px;
    }
    
    .parts-inventory-header {
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .page-title h1 {
        font-size: 26px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .tab-btn {
        padding: 12px 16px;
        font-size: 13px;
    }
    
    .action-bar {
        padding: 16px 20px;
    }
    
    .action-buttons {
        gap: 8px;
        flex-wrap: wrap;
    }
    
    .action-buttons button {
        padding: 8px 12px;
        font-size: 13px;
        flex: 1;
        min-width: 120px;
    }
    
    .modal-dialog {
        width: 95%;
        max-width: 700px;
    }
    
    .modal-body .row {
        margin-bottom: 12px;
    }
    
    .input-group {
        margin-bottom: 12px;
    }
}

@media (max-width: 768px) {
    .parts-inventory-container {
        padding: 12px;
    }
    
    .parts-inventory-header {
        padding: 16px;
        margin-bottom: 16px;
    }
    
    .page-title h1 {
        font-size: 22px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .tab-list {
        flex-direction: column;
    }
    
    .tab-btn {
        justify-content: center;
        border-bottom: 1px solid #e2e8f0;
        padding: 14px 16px;
    }
    
    .tab-btn.active {
        border-bottom: 2px solid #0055a5;
    }
    
    .action-bar {
        padding: 12px 16px;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 8px;
    }
    
    .action-buttons button {
        width: 100%;
        justify-content: center;
        padding: 12px 16px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .parts-inventory-table {
        min-width: 800px;
    }
    
    .table-header th,
    .table-row td {
        padding: 10px 8px;
        font-size: 13px;
    }
    
    .modal-dialog {
        width: 95%;
        margin: 10px auto;
    }
    
    .modal-content {
        border-radius: 0;
    }
    
    .modal-body .row {
        flex-direction: column;
        margin-bottom: 8px;
    }
    
    .input-group {
        margin-bottom: 8px;
    }
    
    .input-group label {
        font-size: 13px;
        margin-bottom: 4px;
    }
    
    .form-control, .form-select {
        font-size: 14px;
        padding: 8px 12px;
    }
    
    .btn {
        padding: 10px 16px;
        font-size: 14px;
    }
}

@media (max-width: 576px) {
    .parts-inventory-container {
        padding: 8px;
    }
    
    .parts-inventory-header {
        padding: 12px;
        margin-bottom: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .search-input {
        padding: 10px 10px 10px 36px;
        font-size: 13px;
    }
    
    .search-icon {
        left: 10px;
        width: 16px;
        height: 16px;
    }
    
    .table-header th,
    .table-row td {
        padding: 8px 6px;
        font-size: 12px;
    }
    
    .empty-state {
        padding: 24px 12px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
    
    .modal-dialog {
        width: 98%;
        margin: 5px auto;
    }
    
    .modal-header {
        padding: 12px 16px;
    }
    
    .modal-title {
        font-size: 18px;
    }
    
    .modal-body {
        padding: 16px;
    }
    
    .modal-footer {
        padding: 12px 16px;
    }
    
    .btn {
        padding: 8px 12px;
        font-size: 13px;
    }
    
    .form-control, .form-select {
        font-size: 13px;
        padding: 6px 10px;
    }
    
    .input-group label {
        font-size: 12px;
    }
}

/* ===== TABLET LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 1024px) and (orientation: landscape) {
    .parts-inventory-container {
        padding: 16px;
    }
    
    .action-content {
        flex-direction: row;
        align-items: center;
    }
    
    .search-section {
        max-width: 300px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 6px;
    }
    
    .action-buttons button {
        padding: 8px 12px;
        font-size: 12px;
    }
}

/* ===== MOBILE LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 896px) and (orientation: landscape) {
    .parts-inventory-header {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .tab-btn {
        padding: 10px 12px;
        font-size: 12px;
    }
}

/* ===== MODAL RESPONSIVE IMPROVEMENTS ===== */
@media (max-width: 768px) {
    .modal-dialog {
        max-height: 90vh;
        overflow-y: auto;
    }
    
    .modal-body {
        max-height: 70vh;
        overflow-y: auto;
    }
    
    .form-inline .row {
        margin: 0;
    }
    
    .form-inline .input-group {
        margin-bottom: 12px;
    }
    
    .form-inline .btn {
        margin: 4px;
    }
}

/* ===== TABLE RESPONSIVE IMPROVEMENTS ===== */
@media (max-width: 768px) {
    .parts-inventory-table {
        font-size: 12px;
    }
    
    .table-header th:first-child,
    .table-row td:first-child {
        position: sticky;
        left: 0;
        background: #f8fafc;
        z-index: 10;
        border-right: 1px solid #e2e8f0;
    }
    
    .table-header th:first-child {
        background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    }
}

/* ===== ACCESSIBILITY IMPROVEMENTS ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        transition: none !important;
        animation: none !important;
    }
}

/* ===== HIGH DPI DISPLAYS ===== */
@media (-webkit-min-device-pixel-ratio: 2), (min-resolution: 192dpi) {
    .search-icon,
    .tab-btn svg,
    .action-buttons button svg {
        image-rendering: -webkit-optimize-contrast;
        image-rendering: crisp-edges;
    }
}

/* ===== FOCUS STATES ===== */
.search-input:focus,
.action-buttons button:focus,
.tab-btn:focus {
    outline: 2px solid rgba(0, 85, 165, 0.5);
    outline-offset: 2px;
}

/* ===== SMOOTH ANIMATIONS ===== */
* {
    transition: all 0.2s ease;
}

/* ===== ADDITIONAL REQUEST_MAINTENANCE-STYLE IMPROVEMENTS ===== */
.parts-inventory-container {
    background: #f8fafc;
}

.parts-inventory-content {
    border-radius: 0;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
}

.table-header th {
    font-weight: 600;
    color: #374151;
    border-bottom: 2px solid #e5e7eb;
}

.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.action-buttons button {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-buttons button:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    transform: translateY(-1px);
}

/* ===== HOVER EFFECTS ===== */
.action-buttons button:active,
.tab-btn:active {
    transform: translateY(0);
}

/* ===== LOADING STATES ===== */
.action-buttons button:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
}

.action-buttons button:disabled:hover {
    background: white;
    border-color: #d1d5db;
    transform: none;
}
</style>
@endsection