/**
 * Incident Report Form JavaScript
 * Handles form validation, step navigation, GPS location, and form submission
 */

class IncidentForm {
    constructor() {
        this.currentStep = 1;
        this.totalSteps = 3;
        this.formData = {};
        this.init();
    }

    init() {
        this.bindEvents();
        this.initializeForm();
        this.setupCountryProjectMapping();
    }

    bindEvents() {
        // Step navigation
        document.getElementById('nextBtn').addEventListener('click', () => this.nextStep());
        document.getElementById('prevBtn').addEventListener('click', () => this.prevStep());
        document.getElementById('submitBtn').addEventListener('click', () => this.submitForm());

        // Country change
        document.getElementById('country').addEventListener('change', (e) => this.handleCountryChange(e.target.value));
        
        // Incident category change
        document.getElementById('incidentCategory').addEventListener('change', (e) => this.handleIncidentCategoryChange(e.target.value));
        
        // Equipment involvement
        document.querySelectorAll('input[name="involvesEquipment"]').forEach(radio => {
            radio.addEventListener('change', (e) => this.handleEquipmentToggle(e.target.value === '1'));
        });

        // Incident types
        document.querySelectorAll('input[name="incidentTypes[]"]').forEach(checkbox => {
            checkbox.addEventListener('change', () => this.handleIncidentTypeChange());
        });

        // GPS location
        document.getElementById('getLocationBtn').addEventListener('click', () => this.getCurrentLocation());

        // Image uploads
        for (let i = 1; i <= 8; i++) {
            const input = document.getElementById(`incidentImage${i}`);
            if (input) {
                input.addEventListener('change', (e) => this.handleImageUpload(e, i));
            }
        }

        // Form validation on input
        document.querySelectorAll('input, select, textarea').forEach(field => {
            field.addEventListener('blur', () => this.validateField(field));
        });
    }

    initializeForm() {
        // Set current date as default
        const now = new Date();
        const dateInput = document.getElementById('incidentDate');
        if (dateInput) {
            // Format date for datetime-local input
            const year = now.getFullYear();
            const month = String(now.getMonth() + 1).padStart(2, '0');
            const day = String(now.getDate()).padStart(2, '0');
            const hours = String(now.getHours()).padStart(2, '0');
            const minutes = String(now.getMinutes()).padStart(2, '0');
            
            dateInput.value = `${year}-${month}-${day}T${hours}:${minutes}`;
        }
    }

    setupCountryProjectMapping() {
        this.countryProjectMap = {
            'Zambia': [
                { label: 'Kansanshi', value: 'Kansanshi' },
                { label: 'Kalumbila', value: 'Kalumbila' },
                { label: 'Kimteto', value: 'Kimteto' },
                { label: 'Kobold', value: 'Kobold' },
                { label: 'FQM Exploration', value: 'FQM Exploration' },
                { label: 'Mimosa', value: 'Mimosa' }
            ],
            'DRC': [
                { label: 'Lubumbashi', value: 'Lubumbashi' },
                { label: 'Musompo', value: 'Musompo' },
                { label: 'IME', value: 'IME' },
                { label: 'Kamoa', value: 'Kamoa' }
            ]
        };
    }

    handleCountryChange(country) {
        const projectSelect = document.getElementById('project');
        projectSelect.innerHTML = '<option value="">Select Project</option>';
        
        if (country && this.countryProjectMap[country]) {
            projectSelect.disabled = false;
            this.countryProjectMap[country].forEach(project => {
                const option = document.createElement('option');
                option.value = project.value;
                option.textContent = project.label;
                projectSelect.appendChild(option);
            });
        } else {
            projectSelect.disabled = true;
        }

        // Generate incident number when both country and project are selected
        this.generateIncidentNumber();
    }

    handleIncidentCategoryChange(category) {
        const otherGroup = document.getElementById('otherIncidentCategoryGroup');
        const otherInput = document.getElementById('otherIncidentCategory');
        
        if (category === 'Other') {
            otherGroup.style.display = 'block';
            otherInput.required = true;
        } else {
            otherGroup.style.display = 'none';
            otherInput.required = false;
            otherInput.value = '';
        }
    }

    handleEquipmentToggle(involvesEquipment) {
        const equipmentDetails = document.getElementById('equipmentDetails');
        if (involvesEquipment) {
            equipmentDetails.style.display = 'block';
        } else {
            equipmentDetails.style.display = 'none';
            // Clear equipment fields
            document.getElementById('vehicleId').value = '';
            document.getElementById('vehicleName').value = '';
        }
    }

    handleIncidentTypeChange() {
        const otherGroup = document.getElementById('otherIncidentNoteGroup');
        const otherCheckbox = document.getElementById('other');
        
        if (otherCheckbox && otherCheckbox.checked) {
            otherGroup.style.display = 'block';
        } else {
            otherGroup.style.display = 'none';
            document.getElementById('otherIncidentNote').value = '';
        }
    }

    async generateIncidentNumber() {
        const country = document.getElementById('country').value;
        const project = document.getElementById('project').value;
        
        if (!country || !project) {
            document.getElementById('incidentNumber').value = '';
            return;
        }

        try {
            this.showLoading('Generating incident number...');
            
            const response = await fetch('/inspection/generate-incident-number', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({ country, project })
            });

            const data = await response.json();
            
            if (data.success) {
                document.getElementById('incidentNumber').value = data.incidentNumber;
            } else {
                this.showNotification('Failed to generate incident number', 'error');
            }
        } catch (error) {
            console.error('Error generating incident number:', error);
            this.showNotification('Error generating incident number', 'error');
        } finally {
            this.hideLoading();
        }
    }

    async getCurrentLocation() {
        if (!navigator.geolocation) {
            this.showNotification('Geolocation is not supported by this browser', 'error');
            return;
        }

        try {
            this.showLoading('Getting your location...');
            
            navigator.geolocation.getCurrentPosition(
                async (position) => {
                    const { latitude, longitude } = position.coords;
                    const coordinates = `${latitude}, ${longitude}`;
                    
                    // Get address from coordinates
                    const address = await this.getAddressFromCoordinates(latitude, longitude);
                    
                    // Update form fields
                    document.getElementById('coordinates').value = coordinates;
                    document.getElementById('gpsAddress').value = address;
                    
                    // Show GPS details
                    document.getElementById('gpsDetails').style.display = 'block';
                    
                    this.showNotification('Location fetched successfully', 'success');
                },
                (error) => {
                    let message = 'Error getting location: ';
                    switch (error.code) {
                        case error.PERMISSION_DENIED:
                            message += 'Please allow location access';
                            break;
                        case error.POSITION_UNAVAILABLE:
                            message += 'Location information unavailable';
                            break;
                        case error.TIMEOUT:
                            message += 'Location request timed out';
                            break;
                        default:
                            message += 'Unknown error occurred';
                            break;
                    }
                    this.showNotification(message, 'error');
                },
                {
                    enableHighAccuracy: true,
                    timeout: 10000,
                    maximumAge: 60000
                }
            );
        } catch (error) {
            console.error('Error getting location:', error);
            this.showNotification('Error getting location', 'error');
        } finally {
            this.hideLoading();
        }
    }

    async getAddressFromCoordinates(lat, lng) {
        try {
            // Using a free geocoding service (you can replace with Google Maps API)
            const response = await fetch(`https://api.bigdatacloud.net/data/reverse-geocode-client?latitude=${lat}&longitude=${lng}&localityLanguage=en`);
            const data = await response.json();
            
            if (data.locality && data.principalSubdivision) {
                return `${data.locality}, ${data.principalSubdivision}, ${data.countryName}`;
            } else {
                return `Latitude: ${lat}, Longitude: ${lng}`;
            }
        } catch (error) {
            console.error('Error getting address:', error);
            return `Latitude: ${lat}, Longitude: ${lng}`;
        }
    }

    handleImageUpload(event, imageNumber) {
        const file = event.target.files[0];
        if (!file) return;

        // Validate file type
        if (!file.type.startsWith('image/')) {
            this.showNotification('Please select an image file', 'error');
            event.target.value = '';
            return;
        }

        // Validate file size (5MB max)
        if (file.size > 5 * 1024 * 1024) {
            this.showNotification('Image size must be less than 5MB', 'error');
            event.target.value = '';
            return;
        }

        // Create preview
        const reader = new FileReader();
        reader.onload = (e) => {
            const preview = document.getElementById(`preview${imageNumber}`);
            preview.innerHTML = `<img src="${e.target.result}" alt="Preview">`;
            preview.classList.add('show');
        };
        reader.readAsDataURL(file);
    }

    validateField(field) {
        const value = field.value.trim();
        const isRequired = field.hasAttribute('required');
        
        // Remove existing validation classes
        field.classList.remove('is-invalid');
        const feedback = field.parentNode.querySelector('.invalid-feedback');
        if (feedback) {
            feedback.remove();
        }

        // Check if required field is empty
        if (isRequired && !value) {
            this.showFieldError(field, 'This field is required');
            return false;
        }

        // Additional validations
        if (field.type === 'email' && value && !this.isValidEmail(value)) {
            this.showFieldError(field, 'Please enter a valid email address');
            return false;
        }

        return true;
    }

    showFieldError(field, message) {
        field.classList.add('is-invalid');
        const feedback = document.createElement('div');
        feedback.className = 'invalid-feedback';
        feedback.textContent = message;
        field.parentNode.appendChild(feedback);
    }

    isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }

    validateStep(step) {
        let isValid = true;
        const stepElement = document.getElementById(`step${step}`);
        const requiredFields = stepElement.querySelectorAll('[required]');
        
        requiredFields.forEach(field => {
            if (!this.validateField(field)) {
                isValid = false;
            }
        });

        // Special validation for step 1
        if (step === 1) {
            // Check if at least one incident type is selected
            const incidentTypes = document.querySelectorAll('input[name="incidentTypes[]"]:checked');
            if (incidentTypes.length === 0) {
                this.showNotification('Please select at least one incident type', 'error');
                isValid = false;
            }

            // Check equipment involvement
            const equipmentRadio = document.querySelector('input[name="involvesEquipment"]:checked');
            if (!equipmentRadio) {
                this.showNotification('Please specify if the incident involves equipment', 'error');
                isValid = false;
            }
        }

        return isValid;
    }

    nextStep() {
        if (this.validateStep(this.currentStep)) {
            if (this.currentStep < this.totalSteps) {
                this.currentStep++;
                this.updateStepDisplay();
                this.updateSummary();
            }
        }
    }

    prevStep() {
        if (this.currentStep > 1) {
            this.currentStep--;
            this.updateStepDisplay();
        }
    }

    updateStepDisplay() {
        // Update step indicators
        document.querySelectorAll('.step').forEach((step, index) => {
            step.classList.remove('active', 'completed');
            if (index + 1 < this.currentStep) {
                step.classList.add('completed');
            } else if (index + 1 === this.currentStep) {
                step.classList.add('active');
            }
        });

        // Update form steps
        document.querySelectorAll('.form-step').forEach((step, index) => {
            step.classList.remove('active');
            if (index + 1 === this.currentStep) {
                step.classList.add('active');
            }
        });

        // Update navigation buttons
        const prevBtn = document.getElementById('prevBtn');
        const nextBtn = document.getElementById('nextBtn');
        const submitBtn = document.getElementById('submitBtn');

        prevBtn.style.display = this.currentStep > 1 ? 'inline-flex' : 'none';
        
        if (this.currentStep < this.totalSteps) {
            nextBtn.style.display = 'inline-flex';
            submitBtn.style.display = 'none';
        } else {
            nextBtn.style.display = 'none';
            submitBtn.style.display = 'inline-flex';
        }
    }

    updateSummary() {
        if (this.currentStep === 3) {
            document.getElementById('summaryIncidentNumber').textContent = 
                document.getElementById('incidentNumber').value || '-';
            
            const incidentDate = document.getElementById('incidentDate').value;
            if (incidentDate) {
                const date = new Date(incidentDate);
                document.getElementById('summaryDate').textContent = 
                    date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
            } else {
                document.getElementById('summaryDate').textContent = '-';
            }

            const country = document.getElementById('country').value;
            const project = document.getElementById('project').value;
            document.getElementById('summaryLocation').textContent = 
                country && project ? `${project}, ${country}` : '-';

            document.getElementById('summaryCategory').textContent = 
                document.getElementById('incidentCategory').value || '-';

            document.getElementById('summarySeverity').textContent = 
                document.getElementById('severity').value || '-';
        }
    }

    async submitForm() {
        if (!this.validateStep(this.currentStep)) {
            return;
        }

        try {
            this.showLoading('Creating incident report...');
            
            const formData = new FormData(document.getElementById('incidentForm'));
            
            const response = await fetch('/inspection/store-incident', {
                method: 'POST',
                body: formData,
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            });

            const data = await response.json();
            
            if (data.success) {
                this.showNotification('Incident report created successfully!', 'success');
                // Redirect to incident report list after a short delay
                setTimeout(() => {
                    window.location.href = '/inspection/incident_report';
                }, 2000);
            } else {
                this.showNotification(data.message || 'Failed to create incident report', 'error');
            }
        } catch (error) {
            console.error('Error submitting form:', error);
            this.showNotification('Error submitting form', 'error');
        } finally {
            this.hideLoading();
        }
    }

    showLoading(message = 'Processing...') {
        document.getElementById('loadingMessage').textContent = message;
        const modal = new bootstrap.Modal(document.getElementById('loadingModal'));
        modal.show();
    }

    hideLoading() {
        const modal = bootstrap.Modal.getInstance(document.getElementById('loadingModal'));
        if (modal) {
            modal.hide();
        }
    }

    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `alert alert-${type === 'error' ? 'danger' : type} alert-dismissible fade show position-fixed`;
        notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
        notification.innerHTML = `
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        document.body.appendChild(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 5000);
    }
}

// Initialize the form when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new IncidentForm();
});