<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MaintenanceController extends Controller
{
    private $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json';
    private $requestMaintenanceUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalRequestMaintenancereport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json';
    private $serviceScheduleUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalServiceScheduled/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/serverschedule.json';

    public function workorder()
    {
        $workorderData = $this->fetchWorkorderData();
        return view('maintenance.Workorder', compact('workorderData'));
    }

    private function fetchWorkorderData()
    {
        try {
            Log::info('Fetching workorder data from Firebase', ['url' => $this->firebaseUrl]);
            
            $workorderResponse = Http::timeout(30)->get($this->firebaseUrl);
            
            if (!$workorderResponse->successful()) {
                Log::error('Failed to fetch workorder data from Firebase', [
                    'status' => $workorderResponse->status(),
                    'body' => $workorderResponse->body()
                ]);
                return [];
            }

            $workorderData = $workorderResponse->json() ?? [];
            
            if (empty($workorderData)) {
                Log::info('No workorder data found at the specified path');
                return [];
            }

            Log::info('Successfully fetched workorder data', ['count' => count($workorderData)]);

            // Process the data to extract required fields for multiple workorders
            $processedData = [];
            
            foreach ($workorderData as $workorderId => $workorder) {
                if (is_array($workorder)) {
                    $processedWorkorder = [
                        'id' => $workorderId,
                        'woNumber' => $workorder['requestNumber'] ?? $workorder['woNumber'] ?? 'N/A',
                        'createdDate' => $workorder['createdAt'] ?? $workorder['createdDate'] ?? 'N/A',
                        'vehicle' => isset($workorder['equipment']['equipmentName']) ? $workorder['equipment']['equipmentName'] : 
                                   (isset($workorder['vehicle']) ? $workorder['vehicle'] : 'N/A'),
                        'dueDate' => $workorder['dueDate'] ?? 'N/A',
                        'completedDate' => $workorder['completedDate'] ?? 'N/A',
                        'status' => $workorder['status'] ?? 'N/A',
                        'priority' => $workorder['priority'] ?? 'N/A',
                        'assigned' => isset($workorder['assignTo'][0]['fullName']) ? $workorder['assignTo'][0]['fullName'] : 
                                    (isset($workorder['assignedTo']) ? $workorder['assignedTo'] : 'N/A'),
                        'memo' => $workorder['memo'] ?? $workorder['description'] ?? 'N/A'
                    ];
                    
                    $processedData[] = $processedWorkorder;
                }
            }

            return $processedData;

        } catch (\Exception $e) {
            Log::error('Error fetching workorder data: ' . $e->getMessage());
            return [];
        }
    }

    public function requestMaintenance()
    {
        $requestMaintenanceData = $this->fetchRequestMaintenanceData();
        return view('maintenance.Request_Maintenance', compact('requestMaintenanceData'));
    }

    private function fetchRequestMaintenanceData()
    {
        try {
            Log::info('Fetching request maintenance data from Firebase', ['url' => $this->requestMaintenanceUrl]);
            
            $requestMaintenanceResponse = Http::timeout(30)->get($this->requestMaintenanceUrl);
            
            if (!$requestMaintenanceResponse->successful()) {
                Log::error('Failed to fetch request maintenance data from Firebase', [
                    'status' => $requestMaintenanceResponse->status(),
                    'body' => $requestMaintenanceResponse->body()
                ]);
                return [];
            }

            $requestMaintenanceData = $requestMaintenanceResponse->json() ?? [];
            
            if (empty($requestMaintenanceData)) {
                Log::info('No request maintenance data found at the specified path');
                return [];
            }

            Log::info('Successfully fetched request maintenance data', ['count' => count($requestMaintenanceData)]);

            // Process the data to extract required fields for multiple requests
            $processedData = [];
            
            foreach ($requestMaintenanceData as $requestId => $request) {
                if (is_array($request)) {
                    $processedRequest = [
                        'id' => $requestId,
                        'woNumber' => $request['requestNumber'] ?? $request['woNumber'] ?? 'N/A',
                        'createdDate' => $request['createdAt'] ?? $request['createdDate'] ?? 'N/A',
                        'vehicle' => isset($request['equipment']['equipmentName']) ? $request['equipment']['equipmentName'] : 
                                   (isset($request['vehicle']) ? $request['vehicle'] : 'N/A'),
                        'status' => $request['status'] ?? 'N/A',
                        'priority' => $request['priority'] ?? 'N/A',
                        'assigned' => isset($request['assignTo'][0]['fullName']) ? $request['assignTo'][0]['fullName'] : 
                                    (isset($request['assignedTo']) ? $request['assignedTo'] : 'N/A'),
                        'memo' => $request['memo'] ?? $request['description'] ?? 'N/A',
                        'requestedBy' => isset($request['requestedBy']['fullName']) ? $request['requestedBy']['fullName'] : 
                                       (isset($request['requestedBy']) ? $request['requestedBy'] : 'N/A')
                    ];
                    
                    $processedData[] = $processedRequest;
                }
            }

            return $processedData;

        } catch (\Exception $e) {
            Log::error('Error fetching request maintenance data: ' . $e->getMessage());
            return [];
        }
    }

    public function serviceSchedule()
    {
        $serviceScheduleData = $this->fetchServiceScheduleData();
        return view('maintenance.Service_Schedule', compact('serviceScheduleData'));
    }

    private function fetchServiceScheduleData()
    {
        try {
            Log::info('Fetching service schedule data from Firebase', ['url' => $this->serviceScheduleUrl]);
            
            $serviceScheduleResponse = Http::timeout(30)->get($this->serviceScheduleUrl);
            
            if (!$serviceScheduleResponse->successful()) {
                Log::error('Failed to fetch service schedule data from Firebase', [
                    'status' => $serviceScheduleResponse->status(),
                    'body' => $serviceScheduleResponse->body()
                ]);
                return [];
            }

            $serviceScheduleData = $serviceScheduleResponse->json() ?? [];
            
            if (empty($serviceScheduleData)) {
                Log::info('No service schedule data found at the specified path');
                return [];
            }

            Log::info('Successfully fetched service schedule data', ['count' => count($serviceScheduleData)]);

            // Process the data to extract required fields for multiple schedules
            $processedData = [];
            
            foreach ($serviceScheduleData as $scheduleId => $schedule) {
                if (is_array($schedule)) {
                    $processedSchedule = [
                        'id' => $scheduleId,
                        'scheduleId' => $schedule['scheduleId'] ?? $schedule['id'] ?? 'N/A',
                        'vehicleNumber' => isset($schedule['equipment']['equipmentName']) ? $schedule['equipment']['equipmentName'] : 
                                         (isset($schedule['vehicleNumber']) ? $schedule['vehicleNumber'] : 'N/A'),
                        'dueDate' => $schedule['dueDate'] ?? $schedule['nextServiceDate'] ?? 'N/A',
                        'workorder' => $schedule['workorder'] ?? $schedule['workOrderNumber'] ?? 'N/A',
                        'status' => $schedule['status'] ?? 'N/A',
                        'scheduleTitle' => $schedule['scheduleTitle'] ?? $schedule['title'] ?? 'N/A',
                        'lastServiceDate' => $schedule['lastServiceDate'] ?? 'N/A',
                        'lastServiceMeter' => $schedule['lastServiceMeter'] ?? 'N/A',
                        'nextServiceMeter' => $schedule['nextServiceMeter'] ?? 'N/A',
                        'completedDate' => $schedule['completedDate'] ?? 'N/A'
                    ];
                    
                    $processedData[] = $processedSchedule;
                }
            }

            return $processedData;

        } catch (\Exception $e) {
            Log::error('Error fetching service schedule data: ' . $e->getMessage());
            return [];
        }
    }

    public function partsInventory()
    {
        return view('maintenance.Parts_Inventory');
    }

    public function laborCodes()
    {
        return view('maintenance.Labor_Codes');
    }

    public function taskMaster()
    {
        return view('maintenance.Task_Master');
    }

    public function createSchedule()
    {
        return view('maintenance.create-schedule');
    }

    public function createWorkorder()
    {
        return view('maintenance.Create_New_Workorder');
    }

    public function storeWorkorder(Request $request)
    {
        try {
            Log::info('Workorder creation request received', [
                'request_data' => $request->all(),
                'headers' => $request->headers->all()
            ]);

            // Validate the request data
            $request->validate([
                'title' => 'required|string|max:255',
                'vehicleNumber' => 'required|string|max:255',
                'vehicleName' => 'required|string|max:255',
                'priority' => 'nullable|string|max:50',
                'assignedTo' => 'nullable|string|max:255',
                'startDate' => 'nullable|date',
                'dueDate' => 'nullable|date',
                'workorderType' => 'nullable|string|max:100',
                'estimatedCost' => 'nullable|numeric',
                'estimatedLaborTime' => 'nullable|numeric',
                'memo' => 'nullable|string',
                'model' => 'nullable|string|max:255',
                'meterReading' => 'nullable|string|max:255',
                'meterUnit' => 'nullable|string|max:50',
                'linkInspection' => 'nullable|string|max:255',
                'linkWorkorder' => 'nullable|string|max:255'
            ]);

            // Prepare the workorder data
            $workorderData = [
                'title' => $request->input('title'),
                'vehicleNumber' => $request->input('vehicleNumber'),
                'vehicleName' => $request->input('vehicleName'),
                'priority' => $request->input('priority'),
                'assignedTo' => $request->input('assignedTo'),
                'startDate' => $request->input('startDate'),
                'dueDate' => $request->input('dueDate'),
                'workorderType' => $request->input('workorderType'),
                'estimatedCost' => $request->input('estimatedCost'),
                'estimatedLaborTime' => $request->input('estimatedLaborTime'),
                'memo' => $request->input('memo'),
                'model' => $request->input('model'),
                'meterReading' => $request->input('meterReading'),
                'meterUnit' => $request->input('meterUnit'),
                'linkInspection' => $request->input('linkInspection'),
                'linkWorkorder' => $request->input('linkWorkorder'),
                'status' => 'Open',
                'createdAt' => now()->toISOString(),
                'updatedAt' => now()->toISOString()
            ];

            Log::info('Processed workorder data', ['workorderData' => $workorderData]);

            // Generate a unique key for the new workorder
            $uniqueKey = 'WO-' . time() . '-' . uniqid();
            
            Log::info('Attempting to store workorder in Firebase', [
                'uniqueKey' => $uniqueKey,
                'firebaseUrl' => 'https://titan-drilling-1881a-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/' . $uniqueKey . '.json'
            ]);
            
            // Store the data in Firebase with auto-incrementing index
            $response = Http::put('https://titan-drilling-1881a-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/' . $uniqueKey . '.json', $workorderData);
            
            Log::info('Firebase response received', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            
            if ($response->successful()) {
                Log::info('Workorder created successfully', ['workorderId' => $uniqueKey]);
                return response()->json([
                    'success' => true,
                    'message' => 'Workorder created successfully',
                    'workorderId' => $uniqueKey,
                    'data' => $workorderData
                ]);
            } else {
                Log::error('Failed to create workorder in Firebase', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create workorder',
                    'status' => $response->status()
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error creating workorder: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error creating workorder: ' . $e->getMessage()
            ], 500);
        }
    }

    public function testWorkorderData()
    {
        try {
            $response = Http::get('https://titan-drilling-1881a-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json');
            
            if ($response->successful()) {
                $workorderData = $response->json() ?? [];
                return response()->json([
                    'success' => true,
                    'status' => $response->status(),
                    'total_records' => is_array($workorderData) ? count($workorderData) : 'Not array',
                    'sample_data' => is_array($workorderData) && !empty($workorderData) ? array_slice($workorderData, 0, 3, true) : [],
                    'message' => 'Workorder data retrieved successfully'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'status' => $response->status(),
                    'message' => 'Failed to fetch workorder data'
                ], 500);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'Error fetching workorder data'
            ], 500);
        }
    }

    public function storeRequest(Request $request)
    {
        try {
            Log::info('Maintenance request creation received', [
                'request_data' => $request->all(),
                'headers' => $request->headers->all()
            ]);

            // Validate the request data
            $request->validate([
                'title' => 'required|string|max:255',
                'vehicleNumber' => 'required|string|max:255',
                'vehicleName' => 'required|string|max:255',
                'priority' => 'nullable|string|max:50',
                'assignedTo' => 'nullable|string|max:255',
                'startDate' => 'nullable|string',
                'dueDate' => 'nullable|string',
                'workorderType' => 'nullable|string|max:100',
                'estimatedCost' => 'nullable|numeric',
                'estimatedLaborTime' => 'nullable|numeric',
                'memo' => 'nullable|string',
                'model' => 'nullable|string|max:255',
                'meterReading' => 'nullable|string|max:255',
                'meterUnit' => 'nullable|string|max:50',
                'linkInspection' => 'nullable|string|max:255',
                'linkWorkorder' => 'nullable|string|max:255'
            ]);

            // Get the next available index
            $nextIndex = $this->getNextRequestIndex();
            
            // Prepare the maintenance request data
            $requestData = [
                'title' => $request->input('title'),
                'vehicleNumber' => $request->input('vehicleNumber'),
                'vehicleName' => $request->input('vehicleName'),
                'priority' => $request->input('priority'),
                'assignedTo' => $request->input('assignedTo'),
                'startDate' => $request->input('startDate'),
                'dueDate' => $request->input('dueDate'),
                'workorderType' => $request->input('workorderType'),
                'estimatedCost' => $request->input('estimatedCost'),
                'estimatedLaborTime' => $request->input('estimatedLaborTime'),
                'memo' => $request->input('memo'),
                'model' => $request->input('model'),
                'meterReading' => $request->input('meterReading'),
                'meterUnit' => $request->input('meterUnit'),
                'linkInspection' => $request->input('linkInspection'),
                'linkWorkorder' => $request->input('linkWorkorder'),
                'status' => 'Open',
                'createdAt' => now()->toISOString(),
                'updatedAt' => now()->toISOString()
            ];

            Log::info('Processed maintenance request data', [
                'nextIndex' => $nextIndex,
                'requestData' => $requestData
            ]);

            // Store the data in Firebase with auto-incrementing index
            $response = Http::put('https://titan-drilling-1881a-default-rtdb.firebaseio.com/GlobalRequestMaintenancereport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/' . $nextIndex . '.json', $requestData);
            
            Log::info('Firebase response received', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            
            if ($response->successful()) {
                Log::info('Maintenance request created successfully', ['index' => $nextIndex]);
                return response()->json([
                    'success' => true,
                    'message' => 'Maintenance request created successfully',
                    'requestId' => $nextIndex,
                    'data' => $requestData
                ]);
            } else {
                Log::error('Failed to create maintenance request in Firebase', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create maintenance request',
                    'status' => $response->status()
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error creating maintenance request: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error creating maintenance request: ' . $e->getMessage()
            ], 500);
        }
    }

    private function getNextRequestIndex()
    {
        try {
            $response = Http::get('https://titan-drilling-1881a-default-rtdb.firebaseio.com/GlobalRequestMaintenancereport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json');
            
            if ($response->successful()) {
                $data = $response->json();
                if (is_array($data)) {
                    $maxIndex = 0;
                    foreach ($data as $key => $value) {
                        if (is_numeric($key) && $key > $maxIndex) {
                            $maxIndex = $key;
                        }
                    }
                    return $maxIndex + 1;
                }
            }
            
            // Fallback to timestamp-based index
            return (int)(microtime(true) * 1000);
        } catch (\Exception $e) {
            Log::error('Error getting next request index: ' . $e->getMessage());
            return (int)(microtime(true) * 1000);
        }
    }

    public function viewWorkorderPdf($id)
    {
        try {
            Log::info('Fetching workorder PDF', ['id' => $id]);
            
            // Fetch workorder data from Firebase using the specified path
            $response = Http::get("https://titan-drilling-1881a-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/{$id}/pdfDownloadUrl.json");
            
            if ($response->successful()) {
                $pdfUrl = $response->json();
                
                if ($pdfUrl) {
                    Log::info('Workorder PDF found', ['id' => $id, 'pdfUrl' => $pdfUrl]);
                    
                    // Redirect to the PDF URL in a new tab
                    return redirect()->away($pdfUrl);
                } else {
                    Log::warning('No PDF found for workorder', ['id' => $id]);
                    return response()->json([
                        'success' => false,
                        'message' => 'No PDF report available for this workorder'
                    ], 404);
                }
            } else {
                Log::error('Failed to fetch workorder from Firebase', ['id' => $id, 'status' => $response->status()]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch workorder data'
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error viewing workorder PDF: ' . $e->getMessage(), ['id' => $id]);
            return response()->json([
                'success' => false,
                'message' => 'Error viewing workorder PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    public function storeSchedule(Request $request)
    {
        try {
            Log::info('Service schedule creation request received', [
                'request_data' => $request->all(),
                'headers' => $request->headers->all()
            ]);

            $data = $request->input('data');
            
            // Validate the request data
            $request->validate([
                'data' => 'required|array',
                'data.scheduleId' => 'required|string|max:255',
                'data.country' => 'required|string|max:100',
                'data.project' => 'required|string|max:100',
                'data.title' => 'required|string|max:255',
                'data.note' => 'required|string',
                'data.equipment' => 'required|array',
                'data.tasks' => 'required|array|min:1',
                'data.repeatType' => 'required|string|in:Repeat,One time',
                'data.meterReading' => 'required|numeric|min:0',
                'data.meterUnit' => 'required|string|max:50',
                'data.lastServiceDate' => 'required|string',
            ]);
            
            // Get the next schedule number by counting existing schedules
            $nextScheduleNumber = $this->getNextScheduleNumber();
            
            // Generate a unique key for the new schedule
            $uniqueKey = time() . rand(1000, 9999);
            
            // Format the last service date to match the expected format (DD-MMM-YYYY)
            $lastServiceDate = $data['lastServiceDate'];
            if (strpos($lastServiceDate, '-') !== false && strlen($lastServiceDate) === 10) {
                // Convert from YYYY-MM-DD to DD-MMM-YYYY format
                $date = \Carbon\Carbon::createFromFormat('Y-m-d', $lastServiceDate);
                $lastServiceDate = $date->format('d-M-Y');
            }

            // Process meter reading - ensure we get the correct values
            $equipmentMeterReading = $data['equipment']['meterReading'] ?? 0;
            $formMeterReading = $data['meterReading'] ?? 0;
            $finalMeterReading = $formMeterReading > 0 ? $formMeterReading : $equipmentMeterReading;
            
            $equipmentMeterUnit = $data['equipment']['meterUnit'] ?? 'Miles';
            $formMeterUnit = $data['meterUnit'] ?? 'Miles';
            $finalMeterUnit = !empty($formMeterUnit) ? $formMeterUnit : $equipmentMeterUnit;

            Log::info('Meter reading processing', [
                'equipment_meter_reading' => $equipmentMeterReading,
                'form_meter_reading' => $formMeterReading,
                'final_meter_reading' => $finalMeterReading,
                'equipment_meter_unit' => $equipmentMeterUnit,
                'form_meter_unit' => $formMeterUnit,
                'final_meter_unit' => $finalMeterUnit
            ]);

            // Generate the proper schedule ID with counter
            $countryCode = $this->getCountryCode($data['country']);
            $projectCode = $this->getProjectCode($data['project']);
            $formattedCounter = str_pad($nextScheduleNumber, 3, '0', STR_PAD_LEFT);
            $generatedScheduleId = $countryCode . '-' . $projectCode . '-SS-' . $formattedCounter;

            // Prepare the schedule data with proper structure matching your format
            $scheduleData = [
                'id' => (string)$uniqueKey,
                'scheduleId' => $generatedScheduleId,
                'country' => $data['country'],
                'project' => $data['project'],
                'title' => $data['title'],
                'note' => $data['note'],
                'equipment' => [
                    'entryNumber' => $data['equipment']['entryNumber'] ?? '',
                    'equipmentName' => $data['equipment']['equipmentName'] ?? 'TYRE - TYRE',
                    'equipmentNumber' => $data['equipment']['equipmentNumber'] ?? '',
                    'meterReading' => (int)$finalMeterReading,
                    'meterUnit' => $finalMeterUnit
                ],
                'meterReading' => (int)$finalMeterReading,
                'meterUnit' => $finalMeterUnit,
                'lastServiceDate' => $lastServiceDate,
                'tasks' => $data['tasks'],
                'repeatType' => $data['repeatType'],
                'intervals' => $data['intervals'] ?? [],
                'intervalUnit' => $data['intervalUnit'] ?? 'Day',
                'notifyBefore' => [
                    'value' => $data['notifyBefore']['value'] ?? '',
                    'unit' => $data['notifyBefore']['unit'] ?? $finalMeterUnit,
                    'time' => $data['notifyBefore']['time'] ?? 'Day'
                ],
                'status' => 'upcoming',
                'createdAt' => now()->toISOString(),
                'dueHours' => '12',
                'hoursToGo' => '10',
                'daysToGo' => '69'
            ];

            Log::info('Processed schedule data', ['scheduleData' => $scheduleData]);

            // Store the data with the unique key at the specified Firebase path
            $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalServiceScheduled/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/serverschedule/' . $uniqueKey . '.json';
            
            Log::info('Attempting to store schedule in Firebase', [
                'uniqueKey' => $uniqueKey,
                'firebaseUrl' => $firebaseUrl
            ]);
            
            $response = Http::timeout(30)->put($firebaseUrl, $scheduleData);
            
            Log::info('Firebase response received', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            
            if ($response->successful()) {
                Log::info('Service schedule created successfully', ['scheduleId' => $uniqueKey]);
                return response()->json([
                    'success' => true,
                    'message' => 'Service schedule created successfully',
                    'scheduleId' => $uniqueKey,
                    'data' => $scheduleData
                ]);
            } else {
                Log::error('Failed to create service schedule in Firebase', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create service schedule',
                    'status' => $response->status(),
                    'error' => $response->body()
                ], 500);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation error in storeSchedule', [
                'errors' => $e->errors(),
                'request_data' => $request->all()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Error creating service schedule: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error creating service schedule: ' . $e->getMessage()
            ], 500);
        }
    }

    public function testServiceScheduleFirebase()
    {
        try {
            $testData = [
                'id' => 'test_' . time(),
                'scheduleId' => 'TEST-SS-001',
                'country' => 'DRC',
                'project' => 'Musompo',
                'title' => 'Test Service Schedule',
                'note' => 'This is a test schedule',
                'equipment' => [
                    'entryNumber' => '',
                    'equipmentName' => 'TYRE - TYRE',
                    'equipmentNumber' => '',
                    'meterReading' => 0,
                    'meterUnit' => 'Miles'
                ],
                'meterReading' => 0,
                'meterUnit' => 'Miles',
                'lastServiceDate' => '22-Aug-2025',
                'tasks' => ['Test Task 1', 'Test Task 2'],
                'repeatType' => 'One time',
                'intervals' => [],
                'intervalUnit' => 'Day',
                'notifyBefore' => [
                    'value' => '',
                    'unit' => 'Miles',
                    'time' => 'Day'
                ],
                'status' => 'upcoming',
                'createdAt' => now()->toISOString(),
                'dueHours' => '12',
                'hoursToGo' => '10',
                'daysToGo' => '69'
            ];

            $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalServiceScheduled/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/serverschedule/test_' . time() . '.json';
            
            $response = Http::timeout(30)->put($firebaseUrl, $testData);
            
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Test service schedule created successfully',
                    'firebase_url' => $firebaseUrl,
                    'data' => $testData
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create test service schedule',
                    'status' => $response->status(),
                    'error' => $response->body()
                ], 500);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error testing service schedule Firebase connection: ' . $e->getMessage()
            ], 500);
        }
    }

    private function getNextScheduleNumber()
    {
        try {
            // Fetch all existing service schedules from Firebase
            $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalServiceScheduled/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/serverschedule.json';
            
            Log::info('Fetching existing schedules to get next number', ['firebase_url' => $firebaseUrl]);
            
            $response = Http::timeout(30)->get($firebaseUrl);
            
            if ($response->successful()) {
                $schedules = $response->json();
                
                if (is_array($schedules)) {
                    $count = count($schedules);
                    $nextNumber = $count + 1;
                    
                    Log::info('Schedule count calculation', [
                        'existing_schedules_count' => $count,
                        'next_schedule_number' => $nextNumber
                    ]);
                    
                    return $nextNumber;
                } else {
                    Log::info('No existing schedules found, starting with 1');
                    return 1;
                }
            } else {
                Log::error('Failed to fetch existing schedules', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                // Fallback to timestamp-based number
                return (int)(microtime(true) * 1000) % 10000;
            }
        } catch (\Exception $e) {
            Log::error('Error getting next schedule number: ' . $e->getMessage());
            // Fallback to timestamp-based number
            return (int)(microtime(true) * 1000) % 10000;
        }
    }

    private function getCountryCode($country)
    {
        $countryCodes = [
            'Zambia' => 'ZMB',
            'DRC' => 'DRC',
        ];
        return $countryCodes[$country] ?? strtoupper(substr($country, 0, 3));
    }

    private function getProjectCode($project)
    {
        if (!$project) return 'XXX';
        return strtoupper(substr($project, 0, 3));
    }

    public function testScheduleCount()
    {
        try {
            // Get the current count
            $currentCount = $this->getNextScheduleNumber() - 1;
            $nextNumber = $this->getNextScheduleNumber();
            
            // Test schedule ID generation
            $testCountry = 'DRC';
            $testProject = 'Musompo';
            $countryCode = $this->getCountryCode($testCountry);
            $projectCode = $this->getProjectCode($testProject);
            $formattedCounter = str_pad($nextNumber, 3, '0', STR_PAD_LEFT);
            $testScheduleId = $countryCode . '-' . $projectCode . '-SS-' . $formattedCounter;
            
            return response()->json([
                'success' => true,
                'message' => 'Schedule count test successful',
                'data' => [
                    'current_schedule_count' => $currentCount,
                    'next_schedule_number' => $nextNumber,
                    'test_schedule_id' => $testScheduleId,
                    'country_code' => $countryCode,
                    'project_code' => $projectCode,
                    'formatted_counter' => $formattedCounter,
                    'firebase_url' => 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalServiceScheduled/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/serverschedule.json'
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error testing schedule count: ' . $e->getMessage()
            ], 500);
            }
    }
}
