import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  FlatList,
  ActivityIndicator,
  Alert,
} from 'react-native';
import { useSafeAreaInsets } from 'react-native-safe-area-context';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { MongoAPI } from '../ConfigMongo/MongoDBConfig';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.30;

const UsersAssignToJC = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const insets = useSafeAreaInsets();
  const [searchQuery, setSearchQuery] = useState('');
  const [teamMembers, setTeamMembers] = useState([]);
  const [adminProfile, setAdminProfile] = useState(null);
  const [userPermissions, setUserPermissions] = useState([]);
  const [userCountry, setUserCountry] = useState([]);
  const [userProject, setUserProject] = useState([]);
  const [hasOnlyMineView, setHasOnlyMineView] = useState(false);
  const [selectedUsers, setSelectedUsers] = useState([]);
  const [isLoading, setIsLoading] = useState(true);
  const [connectionStatus, setConnectionStatus] = useState('Connecting...');

  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      const isConnected = netInfo.isConnected && netInfo.isInternetReachable !== false;
      console.log('🌐 Network status check:', {
        isConnected: netInfo.isConnected,
        isInternetReachable: netInfo.isInternetReachable,
        type: netInfo.type,
        finalResult: isConnected
      });
      return isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      // Assume connected if we can't check
      return true;
    }
  };

  // Get previously selected users from route params
  useEffect(() => {
    if (route.params?.selectedUsers) {
      setSelectedUsers(route.params.selectedUsers);
    }
  }, [route.params]);

  const filterTeamMembersByPermissions = (members, userCountry, userProject, userPermissions, adminUid) => {
    let filteredMembers = [...members];
    
    // Debug logging
    console.log('🔍 Filtering debug:', {
      totalMembers: members.length,
      userCountry: userCountry,
      userProject: userProject,
      userPermissions: userPermissions,
      adminUid: adminUid
    });
    
    if (userPermissions.includes('OnlyMineView')) {
      filteredMembers = filteredMembers.filter(member => member.id === adminUid);
      console.log('🔍 OnlyMineView filter applied:', filteredMembers.length);
    } else {
      // Show ALL members - no restrictions
      // This ensures all users are loaded and displayed
      filteredMembers = filteredMembers;
      console.log('🔍 No restrictions - showing all members:', filteredMembers.length);
    }
    
    // Sort to put admin profile at the top
    filteredMembers.sort((a, b) => {
      if (a.id === adminUid) return -1; // Admin profile first
      if (b.id === adminUid) return 1;
      return 0; // Keep original order for others
    });
    
    console.log('🔍 After sorting - admin profile at top:', filteredMembers.length);
    
    return filteredMembers;
  };

  const fetchFilteredTeamMembers = async () => {
    try {
      setIsLoading(true);
      const adminUid = await AsyncStorage.getItem('userUid');
      console.log('Admin UID:', adminUid);
      
      if (!adminUid) {
        console.log('No admin UID found');
        showToast('error', 'Error', 'User not authenticated');
        setConnectionStatus('Failed - No user ID');
        setIsLoading(false);
        return;
      }

      // Check network status first
      const online = await checkNetworkStatus();
      if (!online) {
        console.log('Network offline');
        setConnectionStatus('Failed - No internet connection');
        showToast('error', 'Error', 'No internet connection');
        setIsLoading(false);
        return;
      }

      // Get admin profile from MongoDB - same as Team_employment_Home.js
      let adminProfile = null;
      if (adminUid) {
        const adminResponse = await MongoAPI.getUserById(adminUid);
        if (adminResponse.success && adminResponse.data) {
          adminProfile = adminResponse.data;
        }
      }

      // Set admin profile if available
      if (adminProfile) {
        setAdminProfile({
          ...adminProfile,
          id: adminUid,
          fullName: adminProfile.name || 'N/A',
          employeeNumber: adminProfile.employeeNumber || 'N/A',
          projects: adminProfile.projects || [],
          countries: adminProfile.countries || [],
          role: adminProfile.role || 'Operator',
          isBlocked: adminProfile.isBlocked || false,
          password: adminProfile.password || 'N/A',
          profilePic: adminProfile.profilePic || null,
        });
        console.log('Admin profile set:', adminProfile);
      } else {
        console.log('Admin profile not found - continuing without it');
      }

      // Get permissions - same as Team_employment_Home.js
      if (adminProfile) {
        setUserCountry(adminProfile.countries || []);
        setUserProject(adminProfile.projects || []);
        const teamEmployModule = adminProfile.modules?.find(m => m.module === 'teamEmploy');
        const permissions = teamEmployModule?.permissions || [];
        setUserPermissions(permissions);
        
        console.log('Admin countries:', adminProfile.countries);
        console.log('Admin projects:', adminProfile.projects);
        console.log('Admin permissions:', permissions);

        // Check for OnlyMineView permission
        const hasOnlyMineView = permissions.some(p => p.toLowerCase() === 'onlymineview');
        setHasOnlyMineView(hasOnlyMineView);
        
        console.log('Has OnlyMineView permission:', hasOnlyMineView);
      } else {
        // Set default values if no admin profile
        setUserCountry([]);
        setUserProject([]);
        setUserPermissions([]);
        setHasOnlyMineView(false);
      }

      const hasOnlyMineView = adminProfile ? 
        (adminProfile.modules?.find(m => m.module === 'teamEmploy')?.permissions || [])
          .some(p => p.toLowerCase() === 'onlymineview') : false;
      
      if (hasOnlyMineView) {
        // If user has OnlyMineView permission, show only their profile
        console.log('User has OnlyMineView permission, showing only admin profile');
        setTeamMembers([]); // No other team members
        setIsLoading(false);
        return;
      }

      // Load team members from MongoDB using fast endpoint - same as Team_employment_Home.js
      console.log('📊 Loading team members from MongoDB...');
      setConnectionStatus('Connecting to MongoDB...');
      
      let teamResponse;
      try {
        console.log('📊 Calling MongoAPI.getUsersFast(1, 8000)...');
        teamResponse = await MongoAPI.getUsersFast(1, 8000);
        console.log('📊 MongoDB Response:', {
          success: teamResponse.success,
          dataLength: teamResponse.data?.length,
          hasData: !!teamResponse.data,
          responseKeys: Object.keys(teamResponse || {}),
          firstUser: teamResponse.data?.[0] ? {
            id: teamResponse.data[0]._id,
            name: teamResponse.data[0].name,
            employeeNumber: teamResponse.data[0].employeeNumber
          } : 'No users'
        });
      } catch (apiError) {
        console.error('❌ MongoDB API Error:', apiError);
        console.error('❌ Error details:', {
          message: apiError.message,
          stack: apiError.stack,
          name: apiError.name
        });
        
        // Try fallback API call
        console.log('🔄 Trying fallback API call...');
        try {
          const fallbackResponse = await fetch('https://api.titandrillingzm.com:6003/users/all');
          if (fallbackResponse.ok) {
            const fallbackData = await fallbackResponse.json();
            console.log('🔄 Fallback API successful:', fallbackData);
            teamResponse = { success: true, data: fallbackData.data || fallbackData };
          } else {
            throw new Error('Fallback API also failed');
          }
        } catch (fallbackError) {
          console.error('❌ Fallback API also failed:', fallbackError);
          setConnectionStatus('Failed - API Error');
          showToast('error', 'Error', `Failed to connect to MongoDB: ${apiError.message}`);
          setTeamMembers([]);
          setIsLoading(false);
          return;
        }
      }
      
      if (teamResponse.success && teamResponse.data) {
        const membersArray = teamResponse.data.map((user) => ({
          // Essential fields only for better performance - exact same as Team_employment_Home.js
          id: user._id,
          fullName: user.name || 'N/A',
          employeeNumber: user.employeeNumber || 'N/A',
          email: user.email || user.employeeNumber || 'N/A',
          projects: user.projects || [],
          countries: user.countries || [],
          role: user.role || 'Operator',
          isBlocked: user.isBlocked || false,
          password: user.password || user.employeeNumber || user.email || 'N/A',
          profilePic: user.profilePic || null,
          // Additional fields only if needed
          driverLicenseNumber: user.driverLicenseNumber || '',
          driverLicenseCode: user.driverLicenseCode || '',
          driverLicenseExpiry: user.driverLicenseExpiry || '',
          dateCompletedFirstAid: user.dateCompletedFirstAid || '',
          expiryDateFirstAid: user.expiryDateFirstAid || '',
          fireFightingDateAttended: user.fireFightingDateAttended || '',
          blastingLicenseDateIssued: user.blastingLicenseDateIssued || '',
        }));
        
        console.log('📊 Total members found from MongoDB:', membersArray.length);

        // Apply admin permissions filter
        const filteredMembers = filterTeamMembersByPermissions(
          membersArray, 
          adminProfile?.countries || [], 
          adminProfile?.projects || [], 
          adminProfile?.modules?.find(m => m.module === 'teamEmploy')?.permissions || [], 
          adminUid
        );
        console.log('🔍 Filtered result:', {
          filteredCount: filteredMembers.length,
          originalCount: membersArray.length
        });
        
        setTeamMembers(filteredMembers);
        setConnectionStatus(`Connected - ${filteredMembers.length} users loaded`);
        showToast('success', 'Success', `Loaded ${filteredMembers.length} team members`);
        
      } else {
        console.log('❌ Failed to load team members from MongoDB');
        setConnectionStatus('Connected - No users found');
        setTeamMembers([]);
        showToast('error', 'Error', 'No users found in database');
      }
      
      setIsLoading(false);
    } catch (error) {
      console.error('Error fetching team members:', error);
      setConnectionStatus('Failed - Connection error');
      setTeamMembers([]);
      setIsLoading(false);
      showToast('error', 'Error', 'Failed to load team members');
    }
  };

  useEffect(() => {
    fetchFilteredTeamMembers();
  }, []);

  const toggleUserSelection = (user) => {
    setSelectedUsers(prev => {
      const isSelected = prev.some(selected => selected.id === user.id);
      if (isSelected) {
        const newSelection = prev.filter(selected => selected.id !== user.id);
        console.log(`Removed user: ${user.fullName || user.name} from selection`);
        return newSelection;
        } else {
        const newSelection = [...prev, user];
        console.log(`Added user: ${user.fullName || user.name} to selection`);
        return newSelection;
      }
    });
  };

  const handleDone = () => {
    // Navigate back to CreateWorkOrder with selected users
    console.log('Sending selected users back to CreateWorkOrder:', selectedUsers);
    console.log('Selected users details:', selectedUsers.map(user => ({
      id: user.id,
      name: user.name,
      fullName: user.fullName,
      employeeNumber: user.employeeNumber,
      role: user.role
    })));
    
    // Show confirmation dialog
    Alert.alert(
      'Confirm Selection',
      `You have selected ${selectedUsers.length} user(s). Do you want to proceed?`,
      [
        {
          text: 'Cancel',
          style: 'cancel',
        },
        {
          text: 'Proceed',
          onPress: () => {
            // Pass data back through route params
            if (route.params?.onUserSelection) {
              // Small delay to ensure navigation completes
              setTimeout(() => {
                route.params.onUserSelection(selectedUsers);
              }, 200);
            }
            
            navigation.goBack();
          },
        },
      ]
    );
  };

  const handleSelectAll = () => {
    const combinedMembers = hasOnlyMineView
      ? [adminProfile ? { ...adminProfile, isSelf: true } : null].filter(Boolean)
      : [
          adminProfile ? { ...adminProfile, isSelf: true } : null,
          ...teamMembers.map(m => ({ ...m, isSelf: false })),
        ].filter(Boolean);

    const filteredMembers = combinedMembers.filter(member =>
      member.fullName?.toLowerCase().includes(searchQuery.toLowerCase()) ||
      member.employeeNumber?.toLowerCase().includes(searchQuery.toLowerCase()) ||
      member.role?.toLowerCase().includes(searchQuery.toLowerCase())
    );

    setSelectedUsers(filteredMembers);
  };

  const handleClearAll = () => {
    setSelectedUsers([]);
  };

  // Combine admin profile and team members for FlatList
  const combinedMembers = hasOnlyMineView
      ? [adminProfile ? { ...adminProfile, isSelf: true } : null].filter(Boolean)
      : [
          adminProfile ? { ...adminProfile, isSelf: true } : null,
          ...teamMembers.filter(member => member.id !== adminProfile?.id).map(m => ({ ...m, isSelf: false })),
        ].filter(Boolean);

  // Apply search filter
  const filteredMembers = combinedMembers.filter(member =>
    member.fullName?.toLowerCase().includes(searchQuery.toLowerCase()) ||
    member.employeeNumber?.toLowerCase().includes(searchQuery.toLowerCase()) ||
    member.role?.toLowerCase().includes(searchQuery.toLowerCase())
  );

  // Debug logging for user display
  console.log('🔍 User Display Debug:', {
    hasOnlyMineView,
    adminProfile: adminProfile ? { id: adminProfile.id, name: adminProfile.fullName } : null,
    teamMembersCount: teamMembers.length,
    combinedMembersCount: combinedMembers.length,
    filteredMembersCount: filteredMembers.length,
    searchQuery,
    teamMembers: teamMembers.slice(0, 3).map(m => ({ id: m.id, name: m.fullName })) // Show first 3 for debugging
  });

  const renderUserItem = ({ item }) => {
    const isSelf = item.isSelf;
    const isSelected = selectedUsers.some(selected => selected.id === item.id);
    
    return (
      <View style={[
        styles.userItem,
        isSelf ? { borderWidth: 1, borderColor: '#4CAF50', marginBottom: 8 } : {},
        isSelected && styles.userItemSelected
      ]}>
        <View style={styles.userItemLeft}>
          {item.profilePic && (
            <Image
              source={{ uri: item.profilePic.url }}
              style={styles.profileImage}
            />
          )}
          <View style={styles.userInfo}>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Employee Number: </Text>
            {item.employeeNumber || 'N/A'}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Name: </Text>
            {item.fullName || 'N/A'}
            {isSelf && <Text style={{ color: '#4CAF50', fontWeight: 'bold' }}> (My Profile)</Text>}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Role: </Text>
            {item.role || 'Operator'}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Projects: </Text>
            {item.projects?.length > 0 ? item.projects.join(', ') : 'None'}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Country: </Text>
            {item.countries?.length > 0 ? item.countries.join(', ') : 'None'}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Status: </Text>
            {item.isBlocked ? 'Blocked' : 'Active'}
          </Text>
        </View>
        </View>
        <View style={styles.userItemRight}>
          <View style={styles.roleBadge}>
            <Text style={styles.roleText}>{item.role || 'Operator'}</Text>
          </View>
          <TouchableOpacity 
          style={[
              styles.selectButton,
              isSelected ? styles.selectButtonSelected : styles.selectButtonUnselected
            ]}
            onPress={() => toggleUserSelection(item)}
          >
            <Text style={[
              styles.selectButtonText,
              isSelected ? styles.selectButtonTextSelected : styles.selectButtonTextUnselected
            ]}>
              {isSelected ? 'Selected' : 'Select'}
                  </Text>
                </TouchableOpacity>
          </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Select Users to Assign</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity 
              style={styles.refreshButton} 
              onPress={() => {
                console.log('🔄 Manual refresh triggered');
                fetchFilteredTeamMembers();
              }}
            >
              <MaterialIcons name="refresh" size={width * 0.05} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.selectedCount}>
              {selectedUsers.length} selected
            </Text>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by name, employee number, or role..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={setSearchQuery}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.searchIcon} />
          </View>
        </View>
      </LinearGradient>

      <View style={styles.statusContainer}>
        <Text style={styles.statusText}>Status: {connectionStatus}</Text>
      </View>

      <View style={styles.actionButtons}>
        <TouchableOpacity style={styles.actionButton} onPress={handleSelectAll}>
          <MaterialIcons name="select-all" size={width * 0.05} color="#3481BC" />
          <Text style={styles.actionButtonText}>Select All</Text>
            </TouchableOpacity>
        <TouchableOpacity style={styles.actionButton} onPress={handleClearAll}>
          <MaterialIcons name="clear-all" size={width * 0.05} color="#f44336" />
          <Text style={[styles.actionButtonText, { color: '#f44336' }]}>Clear All</Text>
        </TouchableOpacity>
        <TouchableOpacity 
          style={[styles.actionButton, { backgroundColor: '#FF9800' }]} 
          onPress={async () => {
            console.log('🧪 Testing MongoDB connection...');
            setConnectionStatus('Testing connection...');
            
            try {
              // Test network first
              const isOnline = await checkNetworkStatus();
              console.log('🧪 Network status:', isOnline);
              
              if (!isOnline) {
                setConnectionStatus('API Test - No internet connection');
                showToast('error', 'API Test', 'No internet connection');
                return;
              }

              // Test direct API call
              console.log('🧪 Making direct API call to: https://api.titandrillingzm.com:6003/users/all');
              const testResponse = await fetch('https://api.titandrillingzm.com:6003/users/all');
              console.log('🧪 Direct API test response:', testResponse.status, testResponse.ok);
              
              if (testResponse.ok) {
                const testData = await testResponse.json();
                console.log('🧪 Test data received:', testData);
                setConnectionStatus(`API Test - ${testData?.data?.length || 0} users found`);
                showToast('success', 'API Test', 'Direct API call successful');
                
                // If direct API works, try the main function again
                setTimeout(() => {
                  console.log('🧪 Retrying main fetch after successful API test');
                  fetchFilteredTeamMembers();
                }, 1000);
              } else {
                console.log('🧪 Server error details:', testResponse.status, testResponse.statusText);
                setConnectionStatus('API Test - Server error');
                showToast('error', 'API Test', `Server error: ${testResponse.status}`);
              }
            } catch (error) {
              console.log('🧪 API test failed:', error);
              setConnectionStatus('API Test - Connection failed');
              showToast('error', 'API Test', `Connection failed: ${error.message}`);
            }
          }}
        >
          <MaterialIcons name="bug-report" size={width * 0.05} color="#fff" />
          <Text style={[styles.actionButtonText, { color: '#fff' }]}>Test API</Text>
        </TouchableOpacity>
      </View>

      {isLoading ? (
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#3481BC" />
          <Text style={styles.loadingText}>Loading users...</Text>
            </View>
      ) : (
        <FlatList
          data={filteredMembers}
          renderItem={renderUserItem}
          keyExtractor={(item) => item.id + (item.isSelf ? '_self' : '')}
          style={styles.userList}
          ListEmptyComponent={
            <View style={styles.emptyContainer}>
              <MaterialIcons name="people" size={48} color="#ccc" />
              <Text style={styles.emptyText}>No users found</Text>
              <Text style={styles.emptySubText}>
                {hasOnlyMineView ? 
                  'Only your profile is visible due to OnlyMineView permission' : 
                  'No users found for your country/project permissions'
                }
              </Text>
            </View>
          }
        />
      )}

      <View
        style={[
          styles.bottomContainer,
          { paddingBottom: Math.max(insets.bottom, width * 0.02) },
        ]}
      >
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.doneButton}>
            <TouchableOpacity
            style={styles.doneButtonTouchable}
            onPress={handleDone}
            >
            <MaterialIcons name="check" size={width * 0.06} color="#fff" />
            <Text style={styles.doneButtonText}>
              Done ({selectedUsers.length} selected)
            </Text>
            </TouchableOpacity>
        </LinearGradient>
          </View>
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  refreshButton: {
    padding: width * 0.015,
    marginRight: width * 0.02,
  },
  selectedCount: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: 'bold',
  },
  searchContainer: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  searchWrapper: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  searchIcon: {
    marginLeft: width * 0.012,
  },
  statusContainer: {
    backgroundColor: '#f0f0f0',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.015,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  statusText: {
    fontSize: width * 0.035,
    color: '#666',
    fontWeight: '500',
  },
  actionButtons: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.02,
    backgroundColor: '#f5f5f5',
  },
  actionButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    marginRight: width * 0.02,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  actionButtonText: {
    fontSize: width * 0.035,
    color: '#3481BC',
    marginLeft: width * 0.01,
  },
  userList: {
    flex: 1,
  },
  userItem: {
    flexDirection: 'row',
    padding: width * 0.04,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
    backgroundColor: '#fff',
    marginHorizontal: width * 0.02,
    marginVertical: width * 0.01,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  userItemSelected: {
    backgroundColor: '#e8f5e8',
    borderColor: '#4CAF50',
  },
  userItemLeft: {
    flex: 2,
    flexDirection: 'row',
    paddingRight: width * 0.02,
  },
  userItemRight: {
    flex: 1,
    alignItems: 'flex-end',
    justifyContent: 'space-between',
  },
  profileImage: {
    width: width * 0.1,
    height: width * 0.1,
    borderRadius: width * 0.05,
    marginRight: width * 0.02,
  },
  userInfo: {
    flex: 1,
  },
  userName: {
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.015,
    lineHeight: width * 0.05,
  },
  labelText: {
    fontWeight: 'bold',
    color: '#333',
  },
  roleBadge: {
    backgroundColor: '#4CAF50',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: 12,
    alignItems: 'center',
    marginBottom: width * 0.02,
  },
  roleText: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    color: '#fff',
  },
  selectButton: {
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    borderWidth: 1,
    alignItems: 'center',
    minWidth: width * 0.2,
  },
  selectButtonUnselected: {
    backgroundColor: '#fff',
    borderColor: '#3481BC',
  },
  selectButtonSelected: {
    backgroundColor: '#3481BC',
    borderColor: '#3481BC',
  },
  selectButtonText: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
  },
  selectButtonTextUnselected: {
    color: '#3481BC',
  },
  selectButtonTextSelected: {
    color: '#fff',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    marginTop: width * 0.02,
    fontSize: width * 0.04,
    color: '#666',
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: width * 0.1,
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#666',
    marginTop: width * 0.02,
  },
  emptySubText: {
    fontSize: width * 0.035,
    color: '#999',
    textAlign: 'center',
    marginTop: width * 0.01,
    paddingHorizontal: width * 0.04,
  },
  bottomContainer: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.02,
    backgroundColor: '#fff',
    borderTopWidth: 1,
    borderTopColor: '#E0E0E0',
  },
  doneButton: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  doneButtonTouchable: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  doneButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
});

export default UsersAssignToJC;