import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, StatusBar, Dimensions, ScrollView, Modal, ActivityIndicator, Animated, Image } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import DropDownPicker from 'react-native-dropdown-picker';
import API_ENDPOINTS from '../../../../config/mongodb.endpoints';

const { width } = Dimensions.get('window');
const HEADER_HEIGHT = width * 0.32;

// Loader Modal (SyncModal) from Inspection_Report_Home.js
const SyncModal = ({ visible, message = 'Loading Roles' }) => {
  const rotateAnim = React.useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  React.useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);
  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Full list of roles from Basic_Details_TeamTab1.js
const ALL_ROLES = [
  'Operations Manager',
  'Health & Safety Manager',
  'Project Manager',
  'Site Manager',
  'Supervisor',
  'HrManager',
  'FinanceManager',
  'Driller',
  'Trainee Driller',
  'Junior Driller',
  'Coremarker',
  'Assistant',
  'Workshop Manager',
  'Workshop Foreman',
  'Maintenance Manager',
  'Mechanical Engineer',
  'Boilermaker',
  'Welder',
  'Mechanic',
  'Machinist',
  'Electrician',
  'Housekeeper',
  'Cook',
  'Stores Officer',
  'General Worker',
];

const EmployeesKpi = () => {
  const navigation = useNavigation();
  const [allUsers, setAllUsers] = useState([]); // All admins from Firebase
  const [filteredUsers, setFilteredUsers] = useState([]); // Filtered by country/project
  const [totalMembers, setTotalMembers] = useState(0);
  const [supervisorCount, setSupervisorCount] = useState(0);
  const [operatorCount, setOperatorCount] = useState(0);
  const [filterRole, setFilterRole] = useState('All');
  const [modalVisible, setModalVisible] = useState(false);
  const [modalLoading, setModalLoading] = useState(false);
  const [modalError, setModalError] = useState(null);
  const [userDetails, setUserDetails] = useState(null);
  const [activeTable, setActiveTable] = useState('all');
  const [myCountry, setMyCountry] = useState('');
  const [myProject, setMyProject] = useState('');
  const [roleCounts, setRoleCounts] = useState({});
  const [managerCount, setManagerCount] = useState(0);
  const [dropdownOpen, setDropdownOpen] = useState(false);
  const [selectedRole, setSelectedRole] = useState('');
  const [dropdownItems, setDropdownItems] = useState([]);
  const [userStats, setUserStats] = useState({}); // { userId: { inspections: n, jobCards: n } }
  const [loadingStats, setLoadingStats] = useState(false);

  const showToast = (type, title, message) => {
    // Only show non-error toasts
    if (type !== 'error') {
      Toast.show({
        type: type,
        text1: title,
        text2: message,
        position: 'bottom',
      });
    }
  };

  // Fetch all admins and current user's country/project from MongoDB
  useEffect(() => {
    const fetchAllUsers = async () => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          showToast('error', 'Error', 'User ID not found. Please log in again.');
          return;
        }

        // Fetch current user's details from MongoDB
        const userResponse = await fetch(API_ENDPOINTS.USERS.GET_BY_ID(userUid));
        const userData = await userResponse.json();
        
        if (!userData.success || !userData.data) {
          showToast('error', 'Error', 'User details not found.');
          return;
        }

        const myData = userData.data;
        let myCountries = Array.isArray(myData?.countries) ? myData.countries : (myData?.countries ? [myData.countries] : []);
        let myProjects = Array.isArray(myData?.projects) ? myData.projects : (myData?.projects ? [myData.projects] : []);
        setMyCountry(myCountries[0] || '');
        setMyProject(myProjects[0] || '');

        // Fetch all users from MongoDB with pagination
        const allUsersResponse = await fetch(`${API_ENDPOINTS.USERS.GET_TEAM_MANAGEMENT}?page=1&limit=500`);
        const allUsersData = await allUsersResponse.json();

        if (!allUsersData.success || !allUsersData.data) {
          setAllUsers([]);
          setFilteredUsers([]);
          setTotalMembers(0);
          setSupervisorCount(0);
          setOperatorCount(0);
          return;
        }

        const usersArr = allUsersData.data.map((user) => ({
          id: user._id || user.uid,
          ...user,
        }));

        setAllUsers(usersArr);

        // Filter users by country/project
        const filtered = usersArr.filter((user) => {
          const userCountries = Array.isArray(user.countries) ? user.countries : (user.countries ? [user.countries] : []);
          const userProjects = Array.isArray(user.projects) ? user.projects : (user.projects ? [user.projects] : []);
          const countryMatch = myCountries.some((c) => userCountries.includes(c));
          const projectMatch = myProjects.some((p) => userProjects.includes(p));
          return countryMatch && projectMatch;
        });

        setFilteredUsers(filtered);
        setTotalMembers(filtered.length);

        // Count by role
        const counts = {};
        let managerTotal = 0;
        ALL_ROLES.forEach(role => { counts[role] = 0; });
        filtered.forEach(u => {
          const role = u.role || '';
          if (ALL_ROLES.includes(role)) {
            counts[role] = (counts[role] || 0) + 1;
          }
          if (role.endsWith('Manager')) {
            managerTotal++;
          }
        });

        setRoleCounts(counts);
        setManagerCount(managerTotal);
        setSupervisorCount(filtered.filter(u => (u.role || '').toLowerCase() === 'supervisor').length);
        setOperatorCount(filtered.filter(u => !['supervisor', 'admin', 'manager'].includes((u.role || '').toLowerCase())).length);

        // Build dropdown items
        const items = ALL_ROLES
          .filter(role => (counts[role] || 0) > 0)
          .map(role => ({ label: `${role} (${counts[role] || 0})`, value: role }));
        setDropdownItems(items);
        setSelectedRole(items.length > 0 ? items[0].value : '');
      } catch (error) {
        console.error('Error fetching all users:', error);
        showToast('error', 'Error', 'Failed to fetch users.');
      }
    };
    fetchAllUsers();
  }, []);

  // Removed fetchTeamMembers - using allUsers from MongoDB instead

  // Fetch inspection/job card counts for each user from MongoDB
  useEffect(() => {
    const fetchStats = async () => {
      setLoadingStats(true);
      try {
        if (!filteredUsers.length) {
          setUserStats({});
          setLoadingStats(false);
          return;
        }

        const stats = {};
        
        // Fetch all inspections from MongoDB
        const inspectionsResponse = await fetch(`${API_ENDPOINTS.INSPECTIONS.GET_ALL}?page=1&limit=10000`);
        const inspectionsData = await inspectionsResponse.json();
        const allInspections = inspectionsData.success ? inspectionsData.data : [];

        // Fetch all job cards from MongoDB
        const jobCardsResponse = await fetch(`${API_ENDPOINTS.JOB_CARDS.GET_ALL}?page=1&limit=10000`);
        const jobCardsData = await jobCardsResponse.json();
        const allJobCards = jobCardsData.success ? jobCardsData.data : [];

        // Count inspections and job cards per user
        filteredUsers.forEach(user => {
          const userId = user.id;
          const inspections = allInspections.filter(insp => insp.userId === userId || insp.createdBy === userId).length;
          const jobCards = allJobCards.filter(jc => jc.adminUid === userId || jc.createdBy === userId).length;
          
          stats[userId] = { inspections, jobCards };
          console.log('Stats for', userId, '- Inspections:', inspections, 'Job Cards:', jobCards);
        });

        setUserStats(stats);
      } catch (e) {
        setUserStats({});
        console.log('Error in fetchStats:', e);
      }
      setLoadingStats(false);
    };
    fetchStats();
  }, [filteredUsers]);

  const handleLockPress = async () => {
    setModalVisible(true);
    setModalLoading(true);
    setModalError(null);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        setModalError('User ID not found. Please log in again.');
        setModalLoading(false);
        return;
      }

      // Fetch user details from MongoDB
      const userResponse = await fetch(API_ENDPOINTS.USERS.GET_BY_ID(userUid));
      const userData = await userResponse.json();
      
      if (!userData.success || !userData.data) {
        setModalError('User details not found.');
      } else {
        const data = userData.data;
        const country = Array.isArray(data.countries) && data.countries.length > 0
          ? data.countries.join(', ')
          : (data.selectedCountry || data.country || 'N/A');
        const project = Array.isArray(data.projects) && data.projects.length > 0
          ? data.projects.join(', ')
          : (data.selectedProject || data.project || 'N/A');
        let teamEmployPerms = [];
        if (Array.isArray(data.modules)) {
          const teamEmployModule = data.modules.find(m => m.module === 'teamEmploy');
          if (teamEmployModule && teamEmployModule.permissions) {
            teamEmployPerms = teamEmployModule.permissions;
          }
        }
        setUserDetails({ country, project, teamEmployPermissions: teamEmployPerms });
      }
    } catch (err) {
      console.error('Error fetching user details:', err);
      setModalError('Failed to fetch user details.');
    }
    setModalLoading(false);
  };

  // Table data based on selectedRole
  const displayedUsers = selectedRole
    ? filteredUsers.filter((user) => (user.role || '') === selectedRole)
    : filteredUsers;

  const renderPermissionModal = () => (
    <Modal visible={modalVisible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={[styles.syncModalContent, { maxWidth: '90%', minWidth: '70%' }]}>
          <Text style={styles.syncModalText}>User Permissions</Text>
          {modalLoading ? (
            <ActivityIndicator size="large" color="#015185" />
          ) : modalError ? (
            <Text style={{ color: '#FF6B6B', fontSize: 16, marginBottom: 12, textAlign: 'center' }}>{modalError}</Text>
          ) : userDetails ? (
            <View style={styles.syncIcondiv}>
              <View style={{ flex: 1 }}>
                <Text style={styles.permissionLabel}>Country:</Text>
                <Text style={styles.permissionValue}>
                  {userDetails.country}
                </Text>
                <Text style={styles.permissionLabel}>Project:</Text>
                <Text style={styles.permissionValue}>
                  {userDetails.project}
                </Text>
                <Text style={styles.permissionLabel}>Team Employ Permissions:</Text>
                <Text style={styles.permissionValue}>
                  {userDetails.teamEmployPermissions && userDetails.teamEmployPermissions.length > 0
                    ? userDetails.teamEmployPermissions.join(', ')
                    : 'None'}
                </Text>
              </View>
            </View>
          ) : null}
          <TouchableOpacity
            style={styles.closeButton}
            onPress={() => setModalVisible(false)}
            accessibilityLabel="Close user permissions"
            accessibilityRole="button"
          >
            <Text style={styles.closeButtonText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  return (
    <View style={styles.container}>
      <SyncModal visible={loadingStats || (filteredUsers.length === 0 && allUsers.length === 0)} message="Loading Roles" />
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={styles.headerLeftIcon}>
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Employees KPI</Text>
          <View style={styles.headerRightIcons}>
            <TouchableOpacity onPress={() => alert('Cloud sync pressed')} style={styles.headerIconButton}>
              <MaterialIcons name="cloud" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity onPress={handleLockPress} style={styles.headerIconButton} accessibilityLabel="Show user permissions" accessibilityRole="button">
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>
      <ScrollView contentContainerStyle={[styles.body, {paddingTop: 0, paddingBottom: 0}]}>
        {/* Filter by Role Dropdown */}
        <View style={{ width: '92%', alignSelf: 'center', marginTop: 0, marginBottom: 0, paddingBottom: 0 }}>
          <Text style={{ fontSize: width * 0.045, fontWeight: '600', color: '#025383', marginBottom: width * 0.01 }}>Filter by Role:</Text>
          <DropDownPicker
            open={dropdownOpen}
            value={selectedRole}
            items={dropdownItems}
            setOpen={setDropdownOpen}
            setValue={setSelectedRole}
            setItems={setDropdownItems}
            placeholder="Select Role"
            style={{ borderColor: '#015185', borderRadius: 8, minHeight: 48, marginBottom: 0, paddingBottom: 0 }}
            containerStyle={{ marginBottom: 0, paddingBottom: 0 }}
            dropDownContainerStyle={{ borderColor: '#015185', borderRadius: 8, marginBottom: 0, paddingBottom: 0 }}
            textStyle={{ fontSize: width * 0.04, color: '#015185' }}
            placeholderStyle={{ color: '#888', fontSize: width * 0.04 }}
            listMode="SCROLLVIEW"
            zIndex={10000}
          />
        </View>
        {/* Table */}
        <ScrollView horizontal showsHorizontalScrollIndicator={false} style={{ width: '100%', marginTop: 0, marginBottom: 0, paddingTop: 0, paddingBottom: 0 }}>
          <View style={[styles.tableWrapperEnhanced, { minWidth: 800, marginTop: 0, marginBottom: 0 }]}> {/* minWidth for horizontal scroll */}
            <LinearGradient
              colors={['#3481BC', '#025383']}
              start={{ x: 0, y: 0 }}
              end={{ x: 1, y: 0 }}
              style={styles.tableHeaderRowEnhanced}
            >
              <Text style={[styles.tableHeaderCellEnhanced, { width: 50, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>S.No</Text>
              <Text style={[styles.tableHeaderCellEnhanced, { width: 120, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>Employee Number</Text>
              <Text style={[styles.tableHeaderCellEnhanced, { width: 100, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>Role</Text>
              <Text style={[styles.tableHeaderCellEnhanced, { width: 160, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>Name</Text>
              <Text style={[styles.tableHeaderCellEnhanced, { width: 180, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>Email</Text>
              <Text style={[styles.tableHeaderCellEnhanced, { width: 90, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>Inspections</Text>
              <Text style={[styles.tableHeaderCellEnhanced, { width: 110 }]}>Job Cards Created</Text>
            </LinearGradient>
            {loadingStats ? (
              <View style={styles.emptyRow}><Text style={styles.emptyText}>Loading stats...</Text></View>
            ) : displayedUsers.length > 0 ? (
              displayedUsers.map((user, idx) => (
                <TouchableOpacity
                  key={user.id}
                  activeOpacity={0.85}
                  style={[
                    styles.tableRowEnhanced,
                    idx % 2 === 0 ? styles.tableRowEvenEnhanced : styles.tableRowOddEnhanced,
                  ]}
                >
                  <Text style={[styles.tableCellEnhanced, { width: 50, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>{idx + 1}</Text>
                  <Text style={[styles.tableCellEnhanced, { width: 120, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>{user.employeeNumber || '-'}</Text>
                  <Text
                    style={[
                      styles.tableCellEnhanced,
                      { width: 100, borderRightWidth: 1, borderColor: '#d0d7e2' },
                      (user.role || '').toLowerCase() === 'supervisor'
                        ? styles.supervisorRole
                        : (user.role || '').toLowerCase() === 'operator'
                        ? styles.operatorRole
                        : styles.defaultRole,
                    ]}
                  >
                    {user.role || 'Employee'}
                  </Text>
                  <Text style={[styles.tableCellEnhanced, { width: 160, color: '#2267A2', fontWeight: '700', borderRightWidth: 1, borderColor: '#d0d7e2' }]}>{user.fullName || user.name || 'Unknown'}</Text>
                  <Text style={[styles.tableCellEnhanced, { width: 180, color: '#025383', fontWeight: '500', borderRightWidth: 1, borderColor: '#d0d7e2' }]}>{user.email || '-'}</Text>
                  <Text style={[styles.tableCellEnhanced, { width: 90, color: '#025383', fontWeight: '700', borderRightWidth: 1, borderColor: '#d0d7e2' }]}>{userStats[user.id]?.inspections ?? 0}</Text>
                  <Text style={[styles.tableCellEnhanced, { width: 110 }]}>{userStats[user.id]?.jobCards ?? 0}</Text>
                </TouchableOpacity>
              ))
            ) : (
              <View style={styles.emptyRow}>
                <Text style={styles.emptyText}>{Object.keys(userStats).length === 0 ? 'No data found or stats fetch timed out.' : 'No users found for this role.'}</Text>
              </View>
            )}
          </View>
        </ScrollView>
      </ScrollView>
      {/* Summary below table */}
      <View style={{
        width: '92%',
        alignSelf: 'center',
        marginTop: 16,
        marginBottom: 24,
        backgroundColor: '#f8f8f8',
        borderRadius: 8,
        padding: 16,
        flexDirection: 'column',
        justifyContent: 'flex-start',
        alignItems: 'flex-start',
        borderWidth: 1,
        borderColor: '#e0e0e0',
      }}>
        <Text style={{ fontSize: width * 0.045, color: '#015185', fontWeight: '600', marginBottom: 6, textAlign: 'left' }}>
          Total Inspections: <Text style={{ color: '#2267A2', fontWeight: '700' }}>{displayedUsers.reduce((sum, u) => sum + (userStats[u.id]?.inspections ?? 0), 0)}</Text>
        </Text>
        <Text style={{ fontSize: width * 0.045, color: '#015185', fontWeight: '600', textAlign: 'left' }}>
          Total Job Cards Created: <Text style={{ color: '#015185', fontWeight: '700' }}>{displayedUsers.reduce((sum, u) => sum + (userStats[u.id]?.jobCards ?? 0), 0)}</Text>
        </Text>
      </View>
      {renderPermissionModal()}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'flex-end',
    paddingBottom: width * 0.04,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.05,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeftIcon: {
    paddingVertical: width * 0.02,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '700',
    flex: 1,
    marginLeft: width * 0.03,
  },
  headerRightIcons: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerIconButton: {
    marginLeft: width * 0.03,
    padding: width * 0.02,
  },
  body: {
    flexGrow: 1,
    alignItems: 'center',
    justifyContent: 'flex-start',
    paddingTop: 0,
    paddingBottom: 0,
    backgroundColor: '#fff',
  },
  filterWrapper: {
    width: '92%',
    marginBottom: width * 0.04,
    alignSelf: 'center',
  },
  filterTitle: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#025383',
    marginBottom: width * 0.02,
  },
  filterButtons: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: width * 0.02,
  },
  filterButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: width * 0.015,
    borderWidth: 1,
    borderColor: '#015185',
    backgroundColor: '#fff',
  },
  activeFilterButton: {
    backgroundColor: '#015185',
  },
  filterButtonText: {
    fontSize: width * 0.035,
    color: '#015185',
  },
  activeFilterButtonText: {
    color: '#fff',
  },
  summaryWrapper: {
    width: '92%',
    backgroundColor: '#f8f8f8',
    borderRadius: 6,
    padding: width * 0.04,
    marginBottom: width * 0.04,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    alignSelf: 'center',
  },
  summaryTitle: {
    fontSize: width * 0.045,
    fontWeight: '700',
    color: '#025383',
    marginBottom: width * 0.02,
  },
  summaryText: {
    fontSize: width * 0.04,
    color: '#444',
    marginBottom: width * 0.01,
  },
  summaryValue: {
    fontWeight: '700',
    color: '#2267A2',
  },
  tableWrapper: {
    width: '92%',
    backgroundColor: '#f8f8f8',
    borderRadius: 6,
    overflow: 'hidden',
    borderWidth: 1,
    borderColor: '#e0e0e0',
    alignSelf: 'center',
    marginTop: 0,
    marginBottom: 0,
  },
  tableHeaderRow: {
    flexDirection: 'row',
    backgroundColor: '#e5e5e5',
    paddingVertical: width * 0.03,
    borderTopLeftRadius: 6,
    borderTopRightRadius: 6,
  },
  tableHeaderCell: {
    fontWeight: '700',
    color: '#444',
    fontSize: width * 0.04,
    textAlign: 'center',
  },
  tableRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.028,
    borderBottomWidth: 1,
    borderBottomColor: '#ececec',
  },
  tableRowEven: {
    backgroundColor: '#f8f8f8',
  },
  tableRowOdd: {
    backgroundColor: '#f1f1f1',
  },
  tableCell: {
    textAlign: 'center',
    fontSize: width * 0.038,
    color: '#333',
  },
  supervisorRole: {
    color: '#2267A2',
    fontWeight: '600',
  },
  operatorRole: {
    color: '#D81B60',
    fontWeight: '600',
  },
  defaultRole: {
    color: '#444',
    fontWeight: '500',
  },
  emptyRow: {
    padding: width * 0.04,
    alignItems: 'center',
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    maxWidth: 400,
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  permissionLabel: {
    fontWeight: 'bold',
    color: '#333',
    fontSize: width * 0.037,
    marginTop: 6,
    marginBottom: 2,
  },
  permissionValue: {
    color: '#333',
    fontSize: width * 0.037,
    marginBottom: 4,
    flexWrap: 'wrap',
    width: '100%',
    lineHeight: width * 0.045,
  },
  closeButton: {
    backgroundColor: '#3481BC',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.012,
    marginTop: 10,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  tableWrapperEnhanced: {
    width: '92%',
    backgroundColor: '#f8f8f8',
    alignSelf: 'flex-start',
    marginTop: 16,
    marginBottom: 16,
 
    overflow: 'hidden',
    borderWidth: 1,
    borderColor: '#d0d7e2',
    shadowColor: '#3481BC',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.12,
    shadowRadius: 8,
    elevation: 6,
  },
  tableHeaderRowEnhanced: {
    flexDirection: 'row',
    paddingVertical: width * 0.035,
    alignItems: 'center',
    justifyContent: 'center',
    borderBottomWidth: 1,
    borderColor: '#d0d7e2',
  },
  tableHeaderCellEnhanced: {
    fontWeight: 'bold',
    color: '#fff',
    fontSize: width * 0.043,
    textAlign: 'center',
    letterSpacing: 0.5,
    paddingVertical: 6,
    paddingHorizontal: 4,
  },
  tableRowEnhanced: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.032,
    borderBottomWidth: 1,
    borderBottomColor: '#d0d7e2',
    backgroundColor: '#fafdff',
  },
  tableRowEvenEnhanced: {
    backgroundColor: '#fafdff',
  },
  tableRowOddEnhanced: {
    backgroundColor: '#eaf3fa',
  },
  tableCellEnhanced: {
    textAlign: 'center',
    fontSize: width * 0.041,
    color: '#333',
    paddingVertical: 8,
    paddingHorizontal: 6,
  },
});

export default EmployeesKpi;