import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  Image,
  StyleSheet,
  SafeAreaView,
  TouchableOpacity,
  Animated,
  ActivityIndicator,
  Modal,
  ScrollView,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
  ToastAndroid,
  Alert,
} from 'react-native';
import { useNavigation, useRoute } from '@react-navigation/native';
import { MongoAPI, USERS_API } from '../../ConfigMongo/MongoDBConfig';
import LinearGradient from 'react-native-linear-gradient';
import NetInfo from '@react-native-community/netinfo';
import { TextInput } from 'react-native-paper';
// Google Sign-in removed - using MongoDB backend
import Toast from 'react-native-toast-message';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import DropDownPicker from 'react-native-dropdown-picker';
import DateTimePicker from '@react-native-community/datetimepicker';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import { PermissionsAndroid } from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { StatusBar } from 'react-native';
import { useSafeAreaInsets } from 'react-native-safe-area-context';

// Get screen dimensions for responsive design
const { width } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;

// Define allowed projects by country (exact mapping)
const ZAMBIA_PROJECTS = [
  { label: 'Kansanshi', value: 'Kansanshi' },
  { label: 'Kalumbila', value: 'Kalumbila' },
  { label: 'Kimteto', value: 'Kimteto' },
  { label: 'Kobold', value: 'Kobold' },
  { label: 'FQM Exploration', value: 'FQM Exploration' },
  { label: 'Mimosa', value: 'Mimosa' },
];

const DRC_PROJECTS = [
  { label: 'Lubumbashi', value: 'Lubumbashi' },
  { label: 'Musompo', value: 'Musompo' },
  { label: 'IME', value: 'IME' },
  { label: 'Kamoa', value: 'Kamoa' },
];

// Reusable Gradient Button Component
const GradientButton = ({ onPress, title, style, isLoading }) => {
  return (
    <LinearGradient colors={['#015185', '#3481BC']} style={[styles.gradientButton, style]}>
      <TouchableOpacity style={styles.buttonTouchable} onPress={onPress} activeOpacity={0.8} disabled={isLoading}>
        {isLoading ? (
          <ActivityIndicator size="small" color="#fff" />
        ) : (
          <Text style={styles.googleButtonText}>{title}</Text>
        )}
      </TouchableOpacity>
    </LinearGradient>
  );
};

// Custom Animated Input Component
const AnimatedInput = ({
  label,
  value,
  onChangeText,
  secureTextEntry,
  keyboardType,
  autoCapitalize,
  leftIconName,
  rightIconName,
  onRightIconPress,
  onBlur,
  maxLength,
  inputRef,
  hasError,
  onFocus,
  editable = true,
}) => {
  const [isFocused, setIsFocused] = useState(false);
  const animatedLabel = useRef(new Animated.Value(value ? 1 : 0)).current;
  const animatedBorderWidth = useRef(new Animated.Value(1)).current;
  const animatedBorderColor = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    Animated.parallel([
      Animated.timing(animatedLabel, {
        toValue: isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderWidth, {
        toValue: isFocused ? 2 : 1,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderColor, {
        toValue: hasError ? 2 : isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
    ]).start();
  }, [isFocused, value, hasError]);

  const labelStyle = {
    position: 'absolute',
    left: scale(45),
    top: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(13), scale(-8)],
    }),
    fontSize: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(16), scale(12)],
    }),
    color: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: ['#888', '#015185'],
    }),
    backgroundColor: '#fff',
    paddingHorizontal: scale(4),
    zIndex: 2,
  };

  const borderColor = animatedBorderColor.interpolate({
    inputRange: [0, 1, 2],
    outputRange: ['#ddd', '#015185', '#ff0000'],
  });

  return (
    <View style={styles.inputContainer} ref={inputRef}>
      <Animated.View
        style={[
          styles.inputGradient,
          {
            borderWidth: animatedBorderWidth,
            borderColor: borderColor,
          },
        ]}
      >
        <Animated.Text style={labelStyle}>{label}</Animated.Text>
        <TextInput
          mode="outlined"
          value={value}
          onChangeText={onChangeText}
          secureTextEntry={secureTextEntry}
          keyboardType={keyboardType}
          autoCapitalize={autoCapitalize}
          style={styles.input}
          onFocus={() => {
            setIsFocused(true);
            onFocus && onFocus();
          }}
          onBlur={() => {
            setIsFocused(false);
            onBlur && onBlur();
          }}
          editable={editable}
          left={
            leftIconName && (
              <TextInput.Icon
                icon={() => <MaterialIcons name={leftIconName} size={scale(24)} color="#015185" />}
              />
            )
          }
          right={
            rightIconName && (
              <TextInput.Icon
                icon={() => (
                  <TouchableOpacity onPress={onRightIconPress}>
                    <MaterialIcons name={rightIconName} size={scale(24)} color="#015185" />
                  </TouchableOpacity>
                )}
              />
            )
          }
          outlineStyle={{
            borderWidth: 0,
            borderRadius: scale(6),
          }}
          theme={{
            colors: {
              primary: 'transparent',
              text: '#333',
              placeholder: '#888',
            },
          }}
          maxLength={maxLength}
        />
      </Animated.View>
    </View>
  );
};

// SyncModal Component
const SyncModal = ({ visible, onClose, message, progress = 0, isCompleted = false }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      if (progress < 100 && !isCompleted) {
        // Only animate while processing
        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 1000,
            useNativeDriver: true,
          })
        ).start();
      } else {
        // Stop animation at 100% or when completed
        rotateAnim.setValue(0);
      }
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible, progress, isCompleted]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            {progress < 100 ? (
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
            ) : (
              <MaterialIcons name="check-circle" size={30} color="#4CAF50" />
            )}
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
          {progress > 0 && (
            <View style={styles.progressContainer}>
              <View style={styles.progressBar}>
                <View style={[styles.progressFill, { width: `${progress}%` }]} />
              </View>
              <Text style={styles.progressText}>{Math.round(progress)}%</Text>
            </View>
          )}
        </View>
      </View>
    </Modal>
  );
};

// Skeleton Loader Components
const SkeletonInput = ({ width = '100%', height = 50 }) => (
  <View style={[styles.skeletonInput, { width, height }]}>
    <View style={styles.skeletonShimmer} />
  </View>
);

const SkeletonText = ({ width = '60%', height = 18 }) => (
  <View style={[styles.skeletonText, { width, height }]}>
    <View style={styles.skeletonShimmer} />
  </View>
);

const SkeletonButton = ({ width = 200, height = 44 }) => (
  <View style={[styles.skeletonButton, { width, height }]}>
    <View style={styles.skeletonShimmer} />
  </View>
);

// Success Popup Component
const SuccessPopup = ({ visible, onClose, title, message, onOkPress }) => {
  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.successModalContainer}>
        <View style={styles.successModalContent}>
          <View style={styles.successIconContainer}>
            <MaterialIcons name="check-circle" size={50} color="#4CAF50" />
          </View>
          <Text style={styles.successTitle}>{title}</Text>
          <Text style={styles.successMessage}>{message}</Text>
          <TouchableOpacity style={styles.successOkButton} onPress={onOkPress}>
            <Text style={styles.successOkButtonText}>OK</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );
};

// Helper function for platform-specific toasts
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.showWithGravity(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

const Edit_team_member = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { teamData } = route.params || {};
  const insets = useSafeAreaInsets();
  
  // Add mounted state to prevent updates after unmount
  const [isMounted, setIsMounted] = useState(true);
  
  if (!teamData) {
    return (
      <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center' }}>
        <Text>No team member data provided.</Text>
      </View>
    );
  }
  const userId = teamData?.id || teamData?.userId; // Use id (from TeamMemberHome) or fallback to userId

  // State for form fields
  const [name, setName] = useState('');
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [mobile, setMobile] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const [nameError, setNameError] = useState(false);
  const [nameErrorMessage, setNameErrorMessage] = useState('');
  const [emailError, setEmailError] = useState(false);
  const [emailErrorMessage, setEmailErrorMessage] = useState('');
  const [passwordError, setPasswordError] = useState(false);
  const [passwordErrorMessage, setPasswordErrorMessage] = useState('');
  const [mobileError, setMobileError] = useState(false);
  const [mobileErrorMessage, setMobileErrorMessage] = useState('');
  const [isLoadingSignUp, setIsLoadingSignUp] = useState(false);
  const [isNetworkError, setIsNetworkError] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [syncModalMessage, setSyncModalMessage] = useState('');
  const [updateProgress, setUpdateProgress] = useState(0);
  const [isUpdateCompleted, setIsUpdateCompleted] = useState(false);
  const [successPopupVisible, setSuccessPopupVisible] = useState(false);
  const [successPopupData, setSuccessPopupData] = useState({ title: '', message: '' });
  const [profilePic, setProfilePic] = useState(null);
  const [department, setDepartment] = useState('');
  const [nrcIdNumber, setNrcIdNumber] = useState('');
  const [engagementDate, setEngagementDate] = useState(new Date());
  const [titanInductionDate, setTitanInductionDate] = useState(new Date());
  const [clientInduction, setClientInduction] = useState(new Date());
  const [dateExamined, setDateExamined] = useState(new Date());
  const [expiryMedicals, setExpiryMedicals] = useState(new Date());
  const [driverLicenseNumber, setDriverLicenseNumber] = useState('');
  const [driverLicenseExpiry, setDriverLicenseExpiry] = useState(new Date());
  const [clientDrivingDate, setClientDrivingDate] = useState(new Date());
  const [dateCompletedFirstAid, setDateCompletedFirstAid] = useState(new Date());
  const [levelFirstAid, setLevelFirstAid] = useState('');
  const [expiryDateFirstAid, setExpiryDateFirstAid] = useState(new Date());
  const [driverLicenseCode, setDriverLicenseCode] = useState('');
  const [uploadModalVisible, setUploadModalVisible] = useState(false);
  const [currentUploadField, setCurrentUploadField] = useState('');
  const [isUploading, setIsUploading] = useState(false);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [datePickerValue, setDatePickerValue] = useState(new Date());
  const [currentEditingDateField, setCurrentEditingDateField] = useState('');
  const [datePickerMaxDate, setDatePickerMaxDate] = useState(null);
  const [datePickerMinDate, setDatePickerMinDate] = useState(new Date(1900, 0, 1));
  const [departmentError, setDepartmentError] = useState(false);
  const [departmentErrorMessage, setDepartmentErrorMessage] = useState('');
  const [nrcIdNumberError, setNrcIdNumberError] = useState(false);
  const [nrcIdNumberErrorMessage, setNrcIdNumberErrorMessage] = useState('');
  const [employeeNumber, setEmployeeNumber] = useState('');
  const [employeeNumberError, setEmployeeNumberError] = useState(false);
  const [employeeNumberErrorMessage, setEmployeeNumberErrorMessage] = useState('');
  const [uploadInProgress, setUploadInProgress] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  
  // Skeleton loading states
  const [isLoadingData, setIsLoadingData] = useState(true);
  const [loadingSections, setLoadingSections] = useState({
    basic: true,
    authentication: true,
    general: true,
    countries: true,
    projects: true,
    titan: true,
    medicals: true,
    driver: true,
    driving: true,
    firstAid: true,
  });

  const networkErrorAnim = useRef(new Animated.Value(-scale(50))).current;
  const scrollViewRef = useRef(null);
  const nameInputRef = useRef(null);
  const emailInputRef = useRef(null);
  const passwordInputRef = useRef(null);
  const mobileInputRef = useRef(null);
  const departmentInputRef = useRef(null);
  const nrcIdNumberInputRef = useRef(null);
  const clientInductionInputRef = useRef(null);
  const driverLicenseNumberInputRef = useRef(null);
  const levelFirstAidInputRef = useRef(null);
  const employeeNumberInputRef = useRef(null);

  // Dropdown state
  const [countries, setCountries] = useState([]);
  const [countryOpen, setCountryOpen] = useState(false);
  const [countryItems, setCountryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);

  const [projects, setProjects] = useState([]);
  const [projectOpen, setProjectOpen] = useState(false);
  const [projectItems, setProjectItems] = useState([...ZAMBIA_PROJECTS, ...DRC_PROJECTS]);

  const [role, setRole] = useState('Operator');
  const [roleOpen, setRoleOpen] = useState(false);
  const roles = [
    { label: 'Admin', value: 'ProAdmin' },
    { label: 'CEO', value: 'CEO' },
    { label: 'Country Manager', value: 'Country Manager' },
    { label: 'Operations Manager', value: 'Operations Manager' },
    { label: 'Health & Safety Manager', value: 'Health & Safety Manager' },
    { label: 'Project Manager', value: 'Project Manager' },
    { label: 'Site Manager', value: 'Site Manager' },
    { label: 'Supervisor', value: 'Supervisor' },
    { label: 'HrManager', value: 'HrManager' },
    { label: 'FinanceManager', value: 'FinanceManager' },
    { label: 'Driller', value: 'Driller' },
    { label: 'Trainee Driller', value: 'Trainee Driller' },
    { label: 'Junior Driller', value: 'Junior Driller' },
    { label: 'Coremarker', value: 'Coremarker' },
    { label: 'Assistant', value: 'Assistant' },
    { label: 'Workshop Manager', value: 'Workshop Manager' },
    { label: 'Workshop Foreman', value: 'Workshop Foreman' },
    { label: 'Maintenance Manager', value: 'Maintenance Manager' },
    { label: 'Mechanical Engineer', value: 'Mechanical Engineer' },
    { label: 'Boilermaker', value: 'Boilermaker' },
    { label: 'Welder', value: 'Welder' },
    { label: 'Mechanic', value: 'Mechanic' },
    { label: 'Machinist', value: 'Machinist' },
    { label: 'Electrician', value: 'Electrician' },
    { label: 'Housekeeper', value: 'Housekeeper' },
    { label: 'Cook', value: 'Cook' },
    { label: 'Stores Officer', value: 'Stores Officer' },
    { label: 'General Worker', value: 'General Worker' },
  ];

  // Derive project list from selected countries and cleanup invalid selections
  useEffect(() => {
    const selected = Array.isArray(countries) ? countries : [];
    const allowedMap = new Map();
    if (selected.includes('Zambia')) {
      ZAMBIA_PROJECTS.forEach((p) => allowedMap.set(p.value, p));
    }
    if (selected.includes('DRC')) {
      DRC_PROJECTS.forEach((p) => allowedMap.set(p.value, p));
    }

    const allowedList = Array.from(allowedMap.values());
    setProjectItems(allowedList);

    if (allowedList.length === 0) {
      setProjects([]);
      return;
    }
    const allowedValues = new Set(allowedList.map((p) => p.value));
    setProjects((prev) => (Array.isArray(prev) ? prev.filter((p) => allowedValues.has(p)) : []));
  }, [countries]);

  const [errors, setErrors] = useState({
    country: false,
    project: false,
    role: false,
    department: false,
    nrcIdNumber: false,
  });
  const [lastAuthUpdateReqId, setLastAuthUpdateReqId] = useState(null);



  // Network error animation
  const showNetworkError = () => {
    setIsNetworkError(true);
    Animated.timing(networkErrorAnim, {
      toValue: 0,
      duration: 300,
      useNativeDriver: true,
    }).start();

    setTimeout(() => {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }, 3000);
  };

  // Queue an Auth password update request for Cloud Function to process
  // Firebase authentication functions removed - using MongoDB backend

    // Direct Auth update function for immediate email/password changes
  const updateAuthUserDirectly = async (targetAuthUid, newEmail, newPassword) => {
    try {
      // Since we can't directly update other users' auth from client side,
      // we'll use a different approach - create a direct admin task
      const requestedBy = await AsyncStorage.getItem('userUid');
      const reqRef = database().ref(`/AdminTasks/immediateAuthUpdates`).push();
      const reqId = reqRef.key;
      
      const payload = {
        userId: targetAuthUid,
        requestedBy: requestedBy || 'unknown',
        createdAt: new Date().toISOString(),
        status: 'pending',
        updateType: 'immediate_auth_update',
        preserveUid: true,
        priority: 'high'
      };
      
      if (newEmail) payload.email = newEmail;
      if (newPassword) payload.password = newPassword;
      
      await reqRef.set(payload);
      
      // Also create a backup task in the regular queue
      await database().ref(`/AdminTasks/authAccountUpdates`).push({
        userId: targetAuthUid,
        requestedBy: requestedBy || 'unknown',
        createdAt: new Date().toISOString(),
        status: 'queued',
        updateType: 'auth_update',
        preserveUid: true,
        email: newEmail,
        password: newPassword
      });
      
      return reqId;
    } catch (e) {
      console.error('Direct auth update error:', e);
      showToast('error', 'Auth Update Error', 'Failed to queue immediate auth update');
      return null;
    }
  };

  // Wait for immediate auth update to complete with better status tracking
  const waitForImmediateAuthUpdate = async (reqId) => {
    return new Promise((resolve) => {
      try {
        const ref = database().ref(`/AdminTasks/immediateAuthUpdates/${reqId}`);
        let hasResolved = false;
        
        const timeoutId = setTimeout(() => {
          if (!hasResolved) {
            hasResolved = true;
            try { ref.off(); } catch (e) {}
            resolve({ status: 'timeout', message: 'Auth update timed out - check admin console' });
          }
        }, 30000); // 30 second timeout
        
        // First, check current status immediately
        ref.once('value').then((snap) => {
          const v = snap.val();
          if (v && v.status && v.status !== 'pending') {
            if (!hasResolved) {
              hasResolved = true;
              clearTimeout(timeoutId);
              try { ref.off(); } catch (e) {}
              
              if (v.status === 'completed') {
                resolve({ status: 'done', message: 'Auth update completed successfully' });
              } else if (v.status === 'failed') {
                resolve({ status: 'error', error: v.error || 'Auth update failed' });
              } else if (v.status === 'processing') {
                resolve({ status: 'processing', message: 'Auth update in progress' });
              } else {
                resolve({ status: 'unknown', message: `Unknown status: ${v.status}` });
              }
            }
          }
        }).catch((e) => {
          if (!hasResolved) {
            hasResolved = true;
            clearTimeout(timeoutId);
            resolve({ status: 'error', error: e?.message || String(e) });
          }
        });
        
        // Then listen for changes
        ref.on('value', (snap) => {
          if (hasResolved) return;
          
          const v = snap.val();
          if (!v || !v.status || v.status === 'pending') return;
          
          hasResolved = true;
          clearTimeout(timeoutId);
          try { ref.off(); } catch (e) {}
          
          if (v.status === 'completed') {
            resolve({ status: 'done', message: 'Auth update completed successfully' });
          } else if (v.status === 'failed') {
            resolve({ status: 'error', error: v.error || 'Auth update failed' });
          } else if (v.status === 'processing') {
            resolve({ status: 'processing', message: 'Auth update in progress' });
          } else {
            resolve({ status: 'unknown', message: `Unknown status: ${v.status}` });
          }
        });
      } catch (e) {
        if (!hasResolved) {
          hasResolved = true;
          resolve({ status: 'error', error: e?.message || String(e) });
        }
      }
    });
  };

  // Debug function to check auth update status
  const checkAuthUpdateStatus = async (reqId) => {
    try {
      const snapshot = await database().ref(`/AdminTasks/immediateAuthUpdates/${reqId}`).once('value');
      const data = snapshot.val();
      if (data) {
        console.log('Auth update status:', data.status);
        console.log('Auth update details:', data);
        return data;
      }
      return null;
    } catch (error) {
      console.error('Error checking auth update status:', error);
      return null;
    }
  };

  // Function to manually refresh auth update status
  const refreshAuthUpdateStatus = async (reqId) => {
    try {
      const data = await checkAuthUpdateStatus(reqId);
      if (data) {
        if (data.status === 'completed') {
          showToast('success', 'Auth Update', 'Authentication updated successfully!');
          return { status: 'done', message: 'Auth update completed successfully' };
        } else if (data.status === 'failed') {
          showToast('error', 'Auth Update', `Authentication update failed: ${data.error || 'Unknown error'}`);
          return { status: 'error', error: data.error || 'Auth update failed' };
        } else if (data.status === 'processing') {
          showToast('info', 'Auth Update', 'Authentication update in progress...');
          return { status: 'processing', message: 'Auth update in progress' };
        } else if (data.status === 'pending') {
          showToast('info', 'Auth Update', 'Authentication update is pending...');
          return { status: 'pending', message: 'Auth update is pending' };
        } else {
          showToast('warn', 'Auth Update', `Unknown status: ${data.status}`);
          return { status: 'unknown', message: `Unknown status: ${data.status}` };
        }
      } else {
        showToast('warn', 'Auth Update', 'No auth update task found');
        return { status: 'not_found', message: 'No auth update task found' };
      }
    } catch (error) {
      console.error('Error refreshing auth update status:', error);
      showToast('error', 'Status Check Error', 'Failed to check auth update status');
      return { status: 'error', error: error?.message || String(error) };
    }
  };

  // Test function to create a sample auth update task
  const testAuthUpdateTask = async () => {
    try {
      const testReqId = await updateAuthUserDirectly('test-user-123', 'test@example.com', null);
      if (testReqId) {
        setLastAuthUpdateReqId(testReqId);
        showToast('success', 'Test Task Created', `Test auth update task created with ID: ${testReqId}`);
        console.log('Test auth update task created:', testReqId);
      }
    } catch (error) {
      console.error('Error creating test task:', error);
      showToast('error', 'Test Error', 'Failed to create test task');
    }
  };

  // Google Sign-in configuration removed - using MongoDB backend

  // Cleanup effect to prevent state updates after unmount
  useEffect(() => {
    return () => {
      setIsMounted(false);
    };
  }, []);

  // Debug: Log when dates change unexpectedly
  useEffect(() => {
    console.log('📅 Date states updated:');
    console.log('  Date Completed First Aid:', dateCompletedFirstAid?.toISOString());
    console.log('  Expiry Date First Aid:', expiryDateFirstAid?.toISOString());
  }, [dateCompletedFirstAid, expiryDateFirstAid]);

  // Progressive data loading with skeleton states - optimized for iOS
  useEffect(() => {
    if (!teamData || !isMounted) return;
    
    let isCancelled = false;
    
    const loadDataProgressively = async () => {
      try {
        console.log('📥 Loading team member data:', teamData);
        
        // Helper function to parse dates safely
        const parseDate = (dateValue) => {
          if (!dateValue || dateValue === '') return new Date();
          const parsed = new Date(dateValue);
          return !isNaN(parsed.getTime()) ? parsed : new Date();
        };
        
        // Batch state updates to prevent multiple re-renders
        const updates = {
          basic: {
            name: teamData.name || '',
            email: teamData.email || '',
            password: '',
            mobile: teamData.mobile || '',
            role: teamData.role || 'Operator',
          },
          general: {
            department: teamData.department || '',
            nrcIdNumber: teamData.nrcIdNumber || '',
            employeeNumber: teamData.employeeNumber || '',
          },
          countries: {
            countries: teamData.countries || [],
            projects: teamData.projects || [],
          },
          dates: {
            engagementDate: parseDate(teamData.engagementDate),
            titanInductionDate: parseDate(teamData.titanInductionDate),
            clientInduction: parseDate(teamData.clientInduction),
            dateExamined: parseDate(teamData.dateExamined),
            expiryMedicals: parseDate(teamData.expiryMedicals),
            driverLicenseExpiry: parseDate(teamData.driverLicenseExpiry),
            clientDrivingDate: parseDate(teamData.clientDrivingDate),
            dateCompletedFirstAid: parseDate(teamData.dateCompletedFirstAid),
            expiryDateFirstAid: parseDate(teamData.expiryDateFirstAid),
          },
          other: {
            driverLicenseNumber: teamData.driverLicenseNumber || '',
            driverLicenseCode: teamData.driverLicenseCode || '',
            levelFirstAid: teamData.levelFirstAid || '',
            profilePic: teamData.profilePic || null,
          }
        };
        
        console.log('📅 Parsed dates - Date Completed:', updates.dates.dateCompletedFirstAid, 'Expiry:', updates.dates.expiryDateFirstAid);

        // Apply updates in batches with minimal delays
        if (!isCancelled) {
          // Basic info
          setName(updates.basic.name);
          setEmail(updates.basic.email);
          setPassword(updates.basic.password);
          setMobile(updates.basic.mobile);
          setRole(updates.basic.role);
          setLoadingSections(prev => ({ ...prev, basic: false, authentication: false }));
          
          // Use requestAnimationFrame for smooth UI updates
          await new Promise(resolve => requestAnimationFrame(resolve));
          
          if (!isCancelled) {
            // General details
            setDepartment(updates.general.department);
            setNrcIdNumber(updates.general.nrcIdNumber);
            setEmployeeNumber(updates.general.employeeNumber);
            setLoadingSections(prev => ({ ...prev, general: false }));
            
            await new Promise(resolve => requestAnimationFrame(resolve));
            
            if (!isCancelled) {
              // Countries and projects
              setCountries(updates.countries.countries);
              setProjects(updates.countries.projects);
              setLoadingSections(prev => ({ ...prev, countries: false, projects: false }));
              
              await new Promise(resolve => requestAnimationFrame(resolve));
              
            if (!isCancelled) {
              // All dates at once
              console.log('⏰ Setting dates from progressive load:');
              console.log('  Date Completed First Aid:', updates.dates.dateCompletedFirstAid);
              console.log('  Expiry Date First Aid:', updates.dates.expiryDateFirstAid);
              
              setEngagementDate(updates.dates.engagementDate);
              setTitanInductionDate(updates.dates.titanInductionDate);
              setClientInduction(updates.dates.clientInduction);
              setDateExamined(updates.dates.dateExamined);
              setExpiryMedicals(updates.dates.expiryMedicals);
              setDriverLicenseExpiry(updates.dates.driverLicenseExpiry);
              setClientDrivingDate(updates.dates.clientDrivingDate);
              setDateCompletedFirstAid(updates.dates.dateCompletedFirstAid);
              setExpiryDateFirstAid(updates.dates.expiryDateFirstAid);
              
              console.log('✅ Dates set successfully');
              
              setLoadingSections(prev => ({ 
                ...prev, 
                titan: false, 
                medicals: false, 
                driver: false, 
                driving: false, 
                firstAid: false 
              }));
                
                await new Promise(resolve => requestAnimationFrame(resolve));
                
                if (!isCancelled) {
                  // Other fields
                  setDriverLicenseNumber(updates.other.driverLicenseNumber);
                  setDriverLicenseCode(updates.other.driverLicenseCode);
                  setLevelFirstAid(updates.other.levelFirstAid);
                  setProfilePic(updates.other.profilePic);
                  
                  // Final completion
                  await new Promise(resolve => requestAnimationFrame(resolve));
                  if (!isCancelled) {
                    setIsLoadingData(false);
                  }
                }
              }
            }
          }
        }
      } catch (error) {
        if (!isCancelled) {
          console.error('Progressive loading error:', error);
          setIsLoadingData(false);
        }
      }
    };
    
    loadDataProgressively();
    
    return () => {
      isCancelled = true;
    };
  }, [teamData, isMounted]);

  // Fetch team member data from Firebase (optional refresh)
  useEffect(() => {
    const fetchTeamMemberData = async () => {
      if (!userId) {
        showToast('error', 'Error', 'No team member selected for editing.');
        return;
      }
      console.log('userId', userId);
      try {
        setSyncModalVisible(true);
        setSyncModalMessage('Fetching Team Member Data...');

        const response = await MongoAPI.getUserById(userId);
        
        if (response.success && response.data) {
          const data = response.data;
          console.log('📊 Loaded team member data from MongoDB:', data);
          console.log('📅 Date Completed First Aid:', data.dateCompletedFirstAid);
          console.log('📅 Expiry Date First Aid:', data.expiryDateFirstAid);
          
          setName(data.name || '');
          setEmail(data.email || '');
          setPassword(''); // Password is not fetched for security reasons
          setMobile(data.mobile || '');
          setRole(data.role || 'Operator');
          setCountries(data.countries || []);
          setProjects(data.projects || []);
          setDepartment(data.department || '');
          setNrcIdNumber(data.nrcIdNumber || '');
          setEmployeeNumber(data.employeeNumber || '');
          
          // Parse dates with validation
          const parseDate = (dateValue) => {
            if (!dateValue || dateValue === '') return new Date();
            const parsed = new Date(dateValue);
            return !isNaN(parsed.getTime()) ? parsed : new Date();
          };
          
          setEngagementDate(parseDate(data.engagementDate));
          setTitanInductionDate(parseDate(data.titanInductionDate));
          setClientInduction(parseDate(data.clientInduction));
          setDateExamined(parseDate(data.dateExamined));
          setExpiryMedicals(parseDate(data.expiryMedicals));
          setDriverLicenseNumber(data.driverLicenseNumber || '');
          setDriverLicenseCode(data.driverLicenseCode || '');
          setDriverLicenseExpiry(parseDate(data.driverLicenseExpiry));
          setClientDrivingDate(parseDate(data.clientDrivingDate));
          setDateCompletedFirstAid(parseDate(data.dateCompletedFirstAid));
          setLevelFirstAid(data.levelFirstAid || '');
          setExpiryDateFirstAid(parseDate(data.expiryDateFirstAid));
          setProfilePic(data.profilePic || null);
          
          console.log('✅ Dates loaded - Date Completed:', parseDate(data.dateCompletedFirstAid), 'Expiry:', parseDate(data.expiryDateFirstAid));
        } else {
          showToast('warn', 'Warning', 'Could not refresh team member data from server. Using local data.');
        }
      } catch (error) {
        console.error('Error fetching team member data:', error);
        showToast('warn', 'Warning', 'Could not refresh team member data from server. Using local data.');
      } finally {
        setSyncModalVisible(false);
      }
    };

    fetchTeamMemberData();
  }, [userId, navigation]);



  // Sync offline data
  useEffect(() => {
    const checkNetwork = async () => {
      const state = await NetInfo.fetch();
      if (state.isConnected) {
        syncOfflineData();
      }
    };

    checkNetwork();

    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected) {
        syncOfflineData();
      }
    });

    return () => unsubscribe();
  }, []);

  const syncOfflineData = async () => {
    try {
      const pendingUploads = JSON.parse(await AsyncStorage.getItem('pendingUploads') || '[]');
      if (pendingUploads.length > 0) {
        for (const upload of pendingUploads) {
          const { adminUid, userId, field, file } = upload;
          const reference = storage().ref(file.name);
          await reference.putFile(file.uri);
          const downloadURL = await reference.getDownloadURL();
          await database()
            .ref(`/Globalusers/admins/${userId}/${field}`)
            .set({ url: downloadURL, name: file.name, type: file.type });
        }
        await AsyncStorage.setItem('pendingUploads', '[]');
        showToast('success', 'Sync Success', 'Offline uploads have been synced.');
      }

      const pendingTeamData = JSON.parse(await AsyncStorage.getItem('pendingTeamData') || '[]');
      if (pendingTeamData.length > 0) {
        for (const data of pendingTeamData) {
          const { userId, data: teamData } = data;
          await database()
            .ref(`/Globalusers/admins/${userId}`)
            .update(teamData);
        }
        await AsyncStorage.setItem('pendingTeamData', '[]');
        showToast('success', 'Sync Success', 'Offline team data has been synced.');
      }
    } catch (error) {
      console.error('Error syncing offline data:', error);
      showToast('error', 'Sync Error', 'Failed to sync offline data.');
    }
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showNetworkError();
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      }
      return true;
    } catch (error) {
      console.error('Network check error:', error);
      showNetworkError();
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

  // BACKEND HEALTH CHECK - prevents crashes
  const checkBackendHealth = async () => {
    try {
      const startTime = Date.now();
      
      // Test database connection
      const testRef = database().ref('/Globalusers/admins').limitToFirst(1);
      const snapshot = await testRef.once('value');
      
      const responseTime = Date.now() - startTime;
      
      if (responseTime > 5000) {
        console.warn('Backend slow response:', responseTime + 'ms');
        return { healthy: false, responseTime, reason: 'slow_response' };
      }
      
      if (!snapshot.exists()) {
        console.warn('Backend connection issue');
        return { healthy: false, responseTime, reason: 'connection_issue' };
      }
      
      return { healthy: true, responseTime };
    } catch (error) {
      console.error('Backend health check failed:', error);
      return { healthy: false, reason: 'error', error: error.message };
    }
  };

  // Validation functions
  const validateName = (name) => {
    const nameEmpty = !name.trim();
    const nameTooLong = name.length > 13;

    setNameError(nameEmpty || nameTooLong);
    if (nameEmpty) {
      setNameErrorMessage('Name is required');
    } else if (nameTooLong) {
      setNameErrorMessage('Name cannot exceed 13 characters');
    } else {
      setNameErrorMessage('');
    }

    return !(nameEmpty || nameTooLong);
  };

  const validateEmail = (email) => {
    const emailEmpty = !email.trim();
    const emailInvalid = emailEmpty ? false : !/\S+@\S+\.\S+/.test(email);
    const emailTooLong = email.length > 40;

    setEmailError(emailEmpty || emailInvalid || emailTooLong);
    if (emailEmpty) {
      setEmailErrorMessage('Email is required');
    } else if (emailInvalid) {
      setEmailErrorMessage('Enter a valid email address');
    } else if (emailTooLong) {
      setEmailErrorMessage('Email cannot exceed 40 characters');
    } else {
      setEmailErrorMessage('');
    }

    return !(emailEmpty || emailInvalid || emailTooLong);
  };

  const validatePassword = (password) => {
    if (!password) return true; // Password is optional for edit
    const passwordShort = password.length < 8;
    const passwordNoSpecialChar = !/[!@#$%^&*(),.?":{}|<>]/.test(password);
    const passwordTooLong = password.length > 15;

    setPasswordError(passwordShort || passwordNoSpecialChar || passwordTooLong);
    if (passwordShort) {
      setPasswordErrorMessage('Password must be at least 8 characters');
    } else if (passwordNoSpecialChar) {
      setPasswordErrorMessage('Password needs a special character');
    } else if (passwordTooLong) {
      setPasswordErrorMessage('Password cannot exceed 15 characters');
    } else {
      setPasswordErrorMessage('');
    }

    return !(passwordShort || passwordNoSpecialChar || passwordTooLong);
  };

  const validateMobile = (mobile) => {
    const mobileEmpty = !mobile.trim();
    const mobileInvalid = mobileEmpty ? false : !/^\+?[1-9]\d{9,14}$/.test(mobile);
    const mobileTooLong = mobile.length > 15;

    setMobileError(mobileEmpty || mobileInvalid || mobileTooLong);
    if (mobileEmpty) {
      setMobileErrorMessage('Mobile number is required');
    } else if (mobileInvalid) {
      setMobileErrorMessage('Enter a valid mobile number');
    } else if (mobileTooLong) {
      setMobileErrorMessage('Mobile number cannot exceed 15 characters');
    } else {
      setMobileErrorMessage('');
    }

    return !(mobileEmpty || mobileInvalid || mobileTooLong);
  };

  const validateDepartment = (department) => {
    const departmentEmpty = !department.trim();
    setDepartmentError(departmentEmpty);
    setDepartmentErrorMessage(departmentEmpty ? 'Department is required' : '');
    return !departmentEmpty;
  };

  const validateNrcIdNumber = (nrcIdNumber) => {
    const nrcEmpty = !nrcIdNumber.trim();
    setNrcIdNumberError(nrcEmpty);
    setNrcIdNumberErrorMessage(nrcEmpty ? 'NRC / ID Number is required' : '');
    return !nrcEmpty;
  };

  const validateEmployeeNumber = (number) => {
    const numberEmpty = !number.trim();
    const numberTooLong = number.length > 20;
    setEmployeeNumberError(numberEmpty || numberTooLong);
    if (numberEmpty) {
      setEmployeeNumberErrorMessage('Employee Number is required');
    } else if (numberTooLong) {
      setEmployeeNumberErrorMessage('Employee Number cannot exceed 20 characters');
    } else {
      setEmployeeNumberErrorMessage('');
    }
    return !(numberEmpty || numberTooLong);
  };

  const validateDropdowns = () => {
    const newErrors = {
      country: countries.length === 0,
      project: projects.length === 0,
      role: !role,
      department: !department.trim(),
      nrcIdNumber: !nrcIdNumber.trim(),
    };

    setErrors(newErrors);

    return (
      !newErrors.country &&
      !newErrors.project &&
      !newErrors.role &&
      !newErrors.department &&
      !newErrors.nrcIdNumber
    );
  };

  // Date picker functions
  const formatDate = (date) => {
    if (!date || isNaN(date.getTime())) {
      console.warn('⚠️ Invalid date passed to formatDate:', date);
      return '';
    }
    const d = new Date(date);
    if (isNaN(d.getTime())) {
      console.warn('⚠️ Date parsing failed for:', date);
      return '';
    }
    const year = d.getFullYear();
    const month = String(d.getMonth() + 1).padStart(2, '0');
    const day = String(d.getDate()).padStart(2, '0');
    return `${day}/${month}/${year}`;
  };

  const onDateChange = (event, selectedDate) => {
    if (Platform.OS === 'android') {
      setShowDatePicker(false);
      if (event.type === 'set' && selectedDate) {
        // Directly apply the date for Android
        applyDate(selectedDate);
      }
    } else {
      // iOS - just update the value
      if (selectedDate) {
        setDatePickerValue(selectedDate);
      }
    }
  };

  const applyDate = (date) => {
    const freshDate = new Date(date);
    
    switch (currentEditingDateField) {
      case 'engagementDate': setEngagementDate(freshDate); break;
      case 'titanInductionDate': setTitanInductionDate(freshDate); break;
      case 'clientInduction': setClientInduction(freshDate); break;
      case 'dateExamined': setDateExamined(freshDate); break;
      case 'expiryMedicals': setExpiryMedicals(freshDate); break;
      case 'driverLicenseExpiry': setDriverLicenseExpiry(freshDate); break;
      case 'clientDrivingDate': setClientDrivingDate(freshDate); break;
      case 'dateCompletedFirstAid': setDateCompletedFirstAid(freshDate); break;
      case 'expiryDateFirstAid': setExpiryDateFirstAid(freshDate); break;
    }
    
    setShowDatePicker(false);
    setCurrentEditingDateField('');
  };

  const showMode = (currentMode, field) => {
    const getCurrentDateForField = () => {
      switch (field) {
        case 'engagementDate': return engagementDate;
        case 'titanInductionDate': return titanInductionDate;
        case 'clientInduction': return clientInduction;
        case 'dateExamined': return dateExamined;
        case 'expiryMedicals': return expiryMedicals;
        case 'driverLicenseExpiry': return driverLicenseExpiry;
        case 'clientDrivingDate': return clientDrivingDate;
        case 'dateCompletedFirstAid': return dateCompletedFirstAid;
        case 'expiryDateFirstAid': return expiryDateFirstAid;
        default: return new Date();
      }
    };

    setCurrentEditingDateField(field);
    setDatePickerValue(getCurrentDateForField());
    setShowDatePicker(true);
  };

  const handleDatePickerDone = () => {
    applyDate(datePickerValue);
  };

  const showDatePickerDialog = (field) => {
    try {
      showMode('date', field);
    } catch (error) {
      console.error('Error in showDatePickerDialog:', error);
      showToast('error', 'Date Picker Error', 'Failed to open date picker. Please try again.');
    }
  };

  // Document upload functions
  const requestCameraPermission = async () => {
    if (Platform.OS === 'android') {
      try {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.CAMERA,
          {
            title: 'Camera Permission',
            message: 'This app needs access to your camera to take photos.',
            buttonPositive: 'OK',
            buttonNegative: 'Cancel',
          }
        );
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } catch (error) {
        console.error('Error requesting camera permission:', error);
        showToast('error', 'Error', 'Failed to request camera permission.');
        return false;
      }
    }
    return true;
  };

  const requestStoragePermission = async () => {
    if (Platform.OS === 'android') {
      try {
        const permission = Platform.Version >= 33
          ? PermissionsAndroid.PERMISSIONS.READ_MEDIA_IMAGES
          : PermissionsAndroid.PERMISSIONS.READ_EXTERNAL_STORAGE;
        const granted = await PermissionsAndroid.request(permission, {
          title: 'Storage Permission',
          message: 'This app needs access to your storage to select files.',
          buttonPositive: 'OK',
          buttonNegative: 'Cancel',
        });
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } catch (error) {
        console.error('Error requesting storage permissions:', error);
        showToast('error', 'Error', 'Failed to request storage permissions.');
        return false;
      }
    }
    return true;
  };

  const handlePickImage = async () => {
    setIsUploading(true);
    const hasPermission = await requestStoragePermission();
    if (!hasPermission) {
      showToast('error', 'Permission Denied', 'Storage permission is required to pick a file.');
      setIsUploading(false);
      return;
    }

    try {
      const result = await launchImageLibrary({ mediaType: 'photo', quality: 0.8 });
      if (!result.didCancel && !result.errorCode) {
        const file = result.assets[0];
        await uploadFile(file, 'image');
      }
    } catch (error) {
      console.error('Error picking image:', error);
      showToast('error', 'Error', 'Failed to pick image.');
    } finally {
      setIsUploading(false);
      setUploadModalVisible(false);
    }
  };

  const handlePickPDF = async () => {
    setIsUploading(true);
    const hasPermission = await requestStoragePermission();
    if (!hasPermission) {
      showToast('error', 'Permission Denied', 'Storage permission is required to pick a file.');
      setIsUploading(false);
      return;
    }

    try {
      const result = await launchImageLibrary({ mediaType: 'mixed' });
      if (!result.didCancel && !result.errorCode) {
        const file = result.assets[0];
        if (file.type === 'application/pdf') {
          await uploadFile(file, 'pdf');
        } else {
          showToast('error', 'Invalid File', 'Please select a PDF file.');
        }
      }
    } catch (error) {
      console.error('Error picking PDF:', error);
      showToast('error', 'Error', 'Failed to pick PDF.');
    } finally {
      setIsUploading(false);
      setUploadModalVisible(false);
    }
  };

  const handleTakePhoto = async () => {
    setIsUploading(true);
    const hasPermission = await requestCameraPermission();
    if (!hasPermission) {
      showToast('error', 'Permission Denied', 'Camera permission is required to take a photo.');
      setIsUploading(false);
      return;
    }

    try {
      const result = await launchCamera({ mediaType: 'photo', quality: 0.8 });
      if (!result.didCancel && !result.errorCode) {
        const file = result.assets[0];
        await uploadFile(file, 'image');
      }
    } catch (error) {
      console.error('Error taking photo:', error);
      showToast('error', 'Error', 'Failed to take photo.');
    } finally {
      setIsUploading(false);
      setUploadModalVisible(false);
    }
  };

  const uploadFile = async (file, fileType) => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Administrator not authenticated.');
        return;
      }

      const extension = file.uri.split('.').pop().toLowerCase();
      const fileName = `teamMembers/${adminUid}/${userId}/${currentUploadField}/${Date.now()}.${extension}`;
      const reference = storage().ref(fileName);

      if (currentUploadField === 'profilePic' && profilePic) {
        await storage().ref(profilePic.name).delete().catch((err) => console.warn('Error deleting old profile pic:', err));
      }

      // track progress
      setUploadInProgress(true);
      setUploadProgress(0);
      const task = reference.putFile(file.uri);

      task.on('state_changed', (snapshot) => {
        const total = snapshot.totalBytes || 1;
        const transferred = snapshot.bytesTransferred || 0;
        const percent = Math.max(0, Math.min(100, Math.floor((transferred / total) * 100)));
        setUploadProgress((prev) => (prev !== percent ? percent : prev));
      });

      await task;
      const downloadURL = await reference.getDownloadURL();

      const doc = { uri: file.uri, url: downloadURL, name: fileName, type: fileType };
      let updatedField;

      if (currentUploadField === 'profilePic') {
        updatedField = doc;
        setProfilePic(doc);
      } else {
        updatedField = [...(teamData[currentUploadField] || []), doc];
      }

      if (await checkNetworkStatus()) {
        await database()
          .ref(`/Globalusers/admins/${userId}/${currentUploadField}`)
          .set(updatedField);
      } else {
        const pendingUploads = JSON.parse(await AsyncStorage.getItem('pendingUploads') || '[]');
        pendingUploads.push({
          adminUid,
          userId,
          field: currentUploadField,
          file: { uri: file.uri, name: fileName, type: fileType },
        });
        await AsyncStorage.setItem('pendingUploads', JSON.stringify(pendingUploads));
        showToast('info', 'Offline Mode', 'File upload queued. Will sync when online.');
      }

      showToast('success', 'Success', 'File uploaded successfully.');
      setUploadProgress(100);
      setTimeout(() => {
        setUploadInProgress(false);
        setUploadProgress(0);
      }, 700);
    } catch (error) {
      console.error('Error uploading file:', error);
      showToast('error', 'Error', 'Failed to upload file.');
      setUploadInProgress(false);
      setUploadProgress(0);
      throw error;
    }
  };

  const handleRemoveDocument = async (field, index) => {
    try {
      const updatedDocs = Array.isArray(teamData?.[field]) ? teamData[field].filter((_, i) => i !== index) : null;

      if (field === 'profilePic') {
        setProfilePic(null);
        if (await checkNetworkStatus() && teamData?.[field]) {
          await storage().ref(teamData[field].name).delete().catch((err) => console.warn('Error deleting profile pic:', err));
        }
      }

      if (await checkNetworkStatus()) {
        await database()
          .ref(`/Globalusers/admins/${userId}/${field}`)
          .set(updatedDocs ? updatedDocs : null);
      }
      showToast('success', 'Success', 'Document removed successfully.');
    } catch (error) {
      console.error('Error removing document:', error);
      showToast('error', 'Error', 'Failed to remove document.');
    }
  };

  const handleUploadPress = (field) => {
    setCurrentUploadField(field);
    setUploadModalVisible(true);
  };



  // Determine what fields actually changed compared to original teamData
  const getChangedFields = () => {
    const changedFields = [];

    if (name !== (teamData?.name || '')) changedFields.push('Name');
    if (email !== (teamData?.email || '')) changedFields.push('Email');
    if (mobile !== (teamData?.mobile || '')) changedFields.push('Mobile');
    if (role !== (teamData?.role || '')) changedFields.push('Role');
    if (department !== (teamData?.department || '')) changedFields.push('Department');
    if (employeeNumber !== (teamData?.employeeNumber || '')) changedFields.push('Employee Number');
    if (nrcIdNumber !== (teamData?.nrcIdNumber || '')) changedFields.push('NRC/ID Number');
    if (driverLicenseNumber !== (teamData?.driverLicenseNumber || '')) changedFields.push('Driver License Number');
    if (driverLicenseCode !== (teamData?.driverLicenseCode || '')) changedFields.push('Driver License Code');
    if (levelFirstAid !== (teamData?.levelFirstAid || '')) changedFields.push('First Aid Level');

    // Dates - compare properly formatted dates
    const formatDateForComparison = (date) => {
      if (!date || isNaN(date.getTime())) return '';
      const d = new Date(date);
      const year = d.getFullYear();
      const month = String(d.getMonth() + 1).padStart(2, '0');
      const day = String(d.getDate()).padStart(2, '0');
      return `${year}-${month}-${day}`;
    };

    const prevEngagementDate = teamData?.engagementDate ? formatDateForComparison(new Date(teamData.engagementDate)) : '';
    const currEngagementDate = formatDateForComparison(engagementDate);
    if (prevEngagementDate !== currEngagementDate) changedFields.push('Engagement Date');

    const prevTitanInductionDate = teamData?.titanInductionDate ? formatDateForComparison(new Date(teamData.titanInductionDate)) : '';
    const currTitanInductionDate = formatDateForComparison(titanInductionDate);
    if (prevTitanInductionDate !== currTitanInductionDate) changedFields.push('Titan Induction Date');

    const prevClientInduction = teamData?.clientInduction ? formatDateForComparison(new Date(teamData.clientInduction)) : '';
    const currClientInduction = formatDateForComparison(clientInduction);
    if (prevClientInduction !== currClientInduction) changedFields.push('Client Induction Date');

    const prevDateExamined = teamData?.dateExamined ? formatDateForComparison(new Date(teamData.dateExamined)) : '';
    const currDateExamined = formatDateForComparison(dateExamined);
    if (prevDateExamined !== currDateExamined) changedFields.push('Date Examined');

    const prevExpiryMedicals = teamData?.expiryMedicals ? formatDateForComparison(new Date(teamData.expiryMedicals)) : '';
    const currExpiryMedicals = formatDateForComparison(expiryMedicals);
    if (prevExpiryMedicals !== currExpiryMedicals) changedFields.push('Medical Expiry Date');

    const prevDriverLicenseExpiry = teamData?.driverLicenseExpiry ? formatDateForComparison(new Date(teamData.driverLicenseExpiry)) : '';
    const currDriverLicenseExpiry = formatDateForComparison(driverLicenseExpiry);
    if (prevDriverLicenseExpiry !== currDriverLicenseExpiry) changedFields.push('Driver License Expiry');

    const prevClientDrivingDate = teamData?.clientDrivingDate ? formatDateForComparison(new Date(teamData.clientDrivingDate)) : '';
    const currClientDrivingDate = formatDateForComparison(clientDrivingDate);
    if (prevClientDrivingDate !== currClientDrivingDate) changedFields.push('Client Driving Date');

    const prevDateCompletedFirstAid = teamData?.dateCompletedFirstAid ? formatDateForComparison(new Date(teamData.dateCompletedFirstAid)) : '';
    const currDateCompletedFirstAid = formatDateForComparison(dateCompletedFirstAid);
    if (prevDateCompletedFirstAid !== currDateCompletedFirstAid) changedFields.push('First Aid Completion Date');

    const prevExpiryDateFirstAid = teamData?.expiryDateFirstAid ? formatDateForComparison(new Date(teamData.expiryDateFirstAid)) : '';
    const currExpiryDateFirstAid = formatDateForComparison(expiryDateFirstAid);
    if (prevExpiryDateFirstAid !== currExpiryDateFirstAid) changedFields.push('First Aid Expiry Date');

    // Arrays: countries, projects
    const prevCountries = Array.isArray(teamData?.countries) ? [...teamData.countries].sort() : [];
    const currCountries = Array.isArray(countries) ? [...countries].sort() : [];
    if (JSON.stringify(prevCountries) !== JSON.stringify(currCountries)) changedFields.push('Countries');

    const prevProjects = Array.isArray(teamData?.projects) ? [...teamData.projects].sort() : [];
    const currProjects = Array.isArray(projects) ? [...projects].sort() : [];
    if (JSON.stringify(prevProjects) !== JSON.stringify(currProjects)) changedFields.push('Projects');

    // Profile picture changes
    const prevProfilePic = teamData?.profilePic?.uri || '';
    const currProfilePic = profilePic?.uri || '';
    if (prevProfilePic !== currProfilePic) changedFields.push('Profile Picture');

    return changedFields;
  };

  // Handle update submission
  const handleUpdate = async () => {
    try {
      // Reset progress and completion state
      setUpdateProgress(0);
      setIsUpdateCompleted(false);
      
      const isNameValid = validateName(name);
      const isEmailValid = validateEmail(email);
      const isPasswordValid = validatePassword(password);
      const isMobileValid = validateMobile(mobile);
      const isDepartmentValid = validateDepartment(department);
      const isNrcIdNumberValid = validateNrcIdNumber(nrcIdNumber);
      const isEmployeeNumberValid = validateEmployeeNumber(employeeNumber);
      const isDropdownValid = validateDropdowns();

      // Check if email is being changed and validate it's not already in use
      if (email !== (teamData?.email || '') && email.trim()) {
        try {
          setUpdateProgress(10);
          setSyncModalMessage('Validating email availability...');
          
          const response = await MongoAPI.getAllUsers();
          
          if (response.success && response.data) {
            const existingUsers = response.data.filter(user => 
              user.email === email.trim() && user._id !== userId
            );
            
            if (existingUsers.length > 0) {
              showToast('error', 'Email Error', 'This email is already in use by another user');
              return;
            }
          }
        } catch (error) {
          console.error('Error checking email availability:', error);
          showToast('error', 'Validation Error', 'Failed to validate email availability');
          return;
        }
      }

      // Validate all fields
      if (
        !isNameValid ||
        !isEmailValid ||
        !isPasswordValid ||
        !isMobileValid ||
        !isDepartmentValid ||
        !isNrcIdNumberValid ||
        !isEmployeeNumberValid ||
        !isDropdownValid
      ) {
        showToast('error', 'Validation Error', 'Please correct all input errors');
        Alert.alert('Error', 'Please fill all required fields correctly.');
        return;
      }

      if (
        isNaN(engagementDate.getTime()) ||
        isNaN(titanInductionDate.getTime()) ||
        isNaN(dateExamined.getTime()) ||
        isNaN(expiryMedicals.getTime()) ||
        isNaN(driverLicenseExpiry.getTime()) ||
        isNaN(clientDrivingDate.getTime()) ||
        isNaN(dateCompletedFirstAid.getTime()) ||
        isNaN(expiryDateFirstAid.getTime())
      ) {
        showToast('error', 'Validation Error', 'All dates must be valid.');
        return;
      }

      if (expiryMedicals <= dateExamined) {
        showToast('error', 'Validation Error', 'Medical Expiry Date must be after Date Examined.');
        return;
      }

      if (driverLicenseExpiry <= engagementDate) {
        showToast('error', 'Validation Error', "Driver's License Expiry Date must be after Engagement Date.");
        return;
      }

      if (expiryDateFirstAid <= dateCompletedFirstAid) {
        showToast('error', 'Validation Error', 'First Aid Expiry Date must be after Date Completed.');
        return;
      }

      // Detect changes early; avoid unnecessary work and notifications
      const changedFields = getChangedFields();
      if (changedFields.length === 0) {
        showToast('info', 'No Changes', 'No changes detected. Nothing to update.');
        setIsLoadingSignUp(false);
        setSyncModalVisible(false);
        return;
      }

            setIsLoadingSignUp(true);
      setSyncModalVisible(true);
      setSyncModalMessage('Starting update process...');
      setUpdateProgress(20);

      // Check network status first
      if (!(await checkNetworkStatus())) {
        setUpdateProgress(100);
        setSyncModalMessage('Saving offline...');
        
        const updatedTeamData = {
          name,
          email,
          mobile,
          role,
          countries,
          projects,
          department,
          nrcIdNumber,
          employeeNumber,
          engagementDate: engagementDate.toISOString(),
          titanInductionDate: titanInductionDate.toISOString(),
          clientInduction: clientInduction.toISOString(),
          dateExamined: dateExamined.toISOString(),
          expiryMedicals: expiryMedicals.toISOString(),
          driverLicenseNumber,
          driverLicenseCode,
          driverLicenseExpiry: driverLicenseExpiry.toISOString(),
          clientDrivingDate: clientDrivingDate.toISOString(),
          dateCompletedFirstAid: dateCompletedFirstAid.toISOString(),
          levelFirstAid,
          expiryDateFirstAid: expiryDateFirstAid.toISOString(),
          profilePic,
          medicalDocs: teamData?.medicalDocs || [],
          driverLicenseDocs: teamData?.driverLicenseDocs || [],
          defensiveDrivingDocs: teamData?.defensiveDrivingDocs || [],
        };

        const pendingTeamData = JSON.parse(await AsyncStorage.getItem('pendingTeamData') || '[]');
        pendingTeamData.push({ userId, data: updatedTeamData });
        await AsyncStorage.setItem('pendingTeamData', JSON.stringify(pendingTeamData));
        showToast('info', 'Offline Mode', 'Data saved locally. Will sync when online.');
        navigation.goBack();
        return;
      }

      // Note: Authentication updates are handled by MongoDB backend
      // Email and password changes will be stored in the database
      const emailChanged = !!email && email !== (teamData?.email || '');
      const passwordChanged = !!password && !!password.trim();

      if (emailChanged || passwordChanged) {
        setUpdateProgress(25);
        setSyncModalMessage('Preparing user data update...');
        
        // Small delay to prevent system overload
        await new Promise(resolve => setTimeout(resolve, 150));
        
        setUpdateProgress(27);
        setSyncModalMessage('User data will be updated in database...');
        
        // Small delay to ensure smooth progress
        await new Promise(resolve => setTimeout(resolve, 200));
      }

                    // Prepare user data for database
      setUpdateProgress(35);
      setSyncModalMessage('Preparing data for database...');
      
      // Small delay to prevent system overload
      await new Promise(resolve => setTimeout(resolve, 150));
      
      setUpdateProgress(37);
      setSyncModalMessage('Building user data structure...');
      
      const userData = {
        name,
        email,
        mobile,
        role,
        loginStatus: teamData?.loginStatus || false,
        lastLogin: teamData?.lastLogin || null,
        countries,
        projects,
        department,
        nrcIdNumber,
        employeeNumber,
        engagementDate: engagementDate && !isNaN(engagementDate.getTime()) ? engagementDate.toISOString() : new Date().toISOString(),
        titanInductionDate: titanInductionDate && !isNaN(titanInductionDate.getTime()) ? titanInductionDate.toISOString() : new Date().toISOString(),
        clientInduction: clientInduction && !isNaN(clientInduction.getTime()) ? clientInduction.toISOString() : new Date().toISOString(),
        dateExamined: dateExamined && !isNaN(dateExamined.getTime()) ? dateExamined.toISOString() : new Date().toISOString(),
        expiryMedicals: expiryMedicals && !isNaN(expiryMedicals.getTime()) ? expiryMedicals.toISOString() : new Date().toISOString(),
        driverLicenseNumber,
        driverLicenseCode,
        driverLicenseExpiry: driverLicenseExpiry && !isNaN(driverLicenseExpiry.getTime()) ? driverLicenseExpiry.toISOString() : new Date().toISOString(),
        clientDrivingDate: clientDrivingDate && !isNaN(clientDrivingDate.getTime()) ? clientDrivingDate.toISOString() : new Date().toISOString(),
        dateCompletedFirstAid: dateCompletedFirstAid && !isNaN(dateCompletedFirstAid.getTime()) ? dateCompletedFirstAid.toISOString() : new Date().toISOString(),
        levelFirstAid,
        expiryDateFirstAid: expiryDateFirstAid && !isNaN(expiryDateFirstAid.getTime()) ? expiryDateFirstAid.toISOString() : new Date().toISOString(),
        profilePic,
        medicalDocs: teamData?.medicalDocs || [],
        driverLicenseDocs: teamData?.driverLicenseDocs || [],
        defensiveDrivingDocs: teamData?.defensiveDrivingDocs || [],
      };

      // Add password to database if provided (for realtime fallback/login)
      if (password && password.trim()) {
        userData.password = password;
      }

      setUpdateProgress(40);
      setSyncModalMessage('Validating data structure...');
      
      // Small delay to ensure data is properly formatted
      await new Promise(resolve => setTimeout(resolve, 100));

      // Update user data in Firebase Realtime Database
      try {
        setUpdateProgress(45);
        setSyncModalMessage('Connecting to database...');
        
        // Small delay to establish connection
        await new Promise(resolve => setTimeout(resolve, 150));
        
        setUpdateProgress(50);
        setSyncModalMessage('Updating database...');
        
        const response = await MongoAPI.updateUser(userId, userData);
        
        if (!response.success) {
          throw new Error(response.error || 'Failed to update user in database');
        }
        
        setUpdateProgress(55);
        setSyncModalMessage('Verifying database update...');
        
        // Small delay to ensure update is processed
        await new Promise(resolve => setTimeout(resolve, 200));
        
        console.log('Database updated successfully for user:', userId);
        console.log('Email changed from:', teamData?.email || 'N/A', 'to:', email);
        console.log('User updated in MongoDB:', response.data);
      } catch (dbError) {
        console.error('Database update failed:', dbError);
        showToast('error', 'Database Error', 'Failed to update user data in database');
        setIsLoadingSignUp(false);
        setSyncModalVisible(false);
        return;
      }
      
      // Verify auth update if email was changed (safe logging only)
      if (emailChanged) {
        try {
          const verifyTargetUid = teamData?.authUid || userId;
          console.log('Auth update verification: Email change queued for user:', verifyTargetUid);
          console.log('Check /AdminTasks/immediateAuthUpdates for status updates');
        } catch (verifyError) {
          console.warn('Could not verify auth update:', verifyError);
        }
      }

      // Update completed successfully
      setUpdateProgress(75);
      setSyncModalMessage('Update completed successfully!');
      
      // Small delay for smooth transition
      await new Promise(resolve => setTimeout(resolve, 500));
      
      console.log('User updated successfully in MongoDB:', userId);

      // Simple completion sequence - NO MORE CRASHES
      setUpdateProgress(95);
      setSyncModalMessage('Update completed successfully!');
      
      // Small delay for smooth transition
      await new Promise(resolve => setTimeout(resolve, 500));
      
      // Mark update as completed
      setIsUpdateCompleted(true);
      
      // Close sync modal
      setSyncModalVisible(false);
      
      // Show success popup
      if (email !== (teamData?.email || '')) {
        const reqId = lastAuthUpdateReqId || 'N/A';
        setSuccessPopupData({
          title: 'Profile Updated Successfully!',
          message: `Profile updated successfully!\n\nEmail changed from ${teamData?.email || 'N/A'} to ${email}\nUID remains: ${userId}\n\nAuth update queued with ID: ${reqId}\nUse refresh button to check status.`
        });
      } else {
        setSuccessPopupData({
          title: 'Profile Updated Successfully!',
          message: 'Profile updated successfully!'
        });
      }
      
      // Show success popup
      setSuccessPopupVisible(true);
      
    } catch (error) {
      console.error('Team member update error:', {
        message: error.message,
        code: error.code,
        details: JSON.stringify(error, null, 2),
      });
      
      // Show error progress only if component is mounted
      if (isMounted) {
        setUpdateProgress(0);
        setSyncModalMessage('Update failed. Please try again.');
        
        const errorMessages = {
          'auth/email-already-in-use': 'This email is already in use',
          'auth/invalid-email': 'Invalid email address',
          'auth/weak-password': 'Password is too weak',
          'auth/too-many-requests': 'Too many attempts. Please try again later',
          default: error.message || 'An unexpected error occurred',
        };
        
        showToast('error', 'Update Error', errorMessages[error.code] || errorMessages.default);
        
        // Small delay to show error before closing modal
        setTimeout(() => {
          if (isMounted) {
            setSyncModalVisible(false);
          }
        }, 1500);
      }
      
    } finally {
      setIsLoadingSignUp(false);
      
      // Simple cleanup - no complex logic
      if (!isUpdateCompleted && isMounted) {
        setTimeout(() => {
          if (isMounted) {
            setUpdateProgress(0);
            setSyncModalVisible(false);
          }
        }, 500);
      }
    }
  };

  // Handle back step
  const handleBackStep = () => {
    Alert.alert(
      'Confirm Exit',
      'Are you sure you want to exit? Any unsaved changes will be lost.',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'OK', onPress: () => navigation.goBack() },
      ],
      { cancelable: true }
    );
  };

  const toggleShowPassword = () => {
    setShowPassword(!showPassword);
  };

  // Handle input focus to scroll - optimized for iOS
  const handleInputFocus = useCallback(
    (inputRef) => () => {
      if (!isMounted || !scrollViewRef.current) return;
      
      // Use requestAnimationFrame for better iOS performance
      requestAnimationFrame(() => {
        if (!isMounted || !inputRef.current || !scrollViewRef.current) return;
        
        try {
          inputRef.current.measureLayout(
            scrollViewRef.current.getScrollResponder(),
            (x, y) => {
              if (isMounted && scrollViewRef.current) {
                scrollViewRef.current.scrollTo({ 
                  y: Math.max(0, y - scale(80)), 
                  animated: true 
                });
              }
            },
            (error) => {
              if (error) {
                console.warn('Error measuring layout:', error);
              }
            }
          );
        } catch (error) {
          console.warn('Focus scroll error:', error);
        }
      });
    },
    [isMounted]
  );

  const renderDocumentPreviews = (field) => {
    const docs = (teamData && teamData[field]) ? teamData[field] : [];
    if (!Array.isArray(docs) || docs.length === 0) return null;

    return (
      <View style={styles.documentPreviewContainer}>
        {docs.map((doc, index) => (
          <View key={index} style={styles.documentPreviewItem}>
            {doc?.type === 'pdf' ? (
              <View style={styles.pdfIconContainer}>
                <MaterialIcons name="picture-as-pdf" size={scale(40)} color="#ff4444" />
              </View>
            ) : (
              <Image source={{ uri: doc?.uri }} style={styles.documentPreviewImage} />
            )}
            <TouchableOpacity
              style={styles.removeDocumentButton}
              onPress={() => handleRemoveDocument(field, index)}
            >
              <MaterialIcons name="close" size={scale(18)} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.documentName} numberOfLines={1} ellipsizeMode="middle">
              {doc?.name?.split('/').pop() || 'Unknown'}
            </Text>
          </View>
        ))}
      </View>
    );
  };

  const renderProfilePicturePreview = () => {
    if (!profilePic) return null;

    return (
      <View style={styles.profilePicturePreview}>
        <Image source={{ uri: profilePic.uri }} style={styles.profilePictureImage} />
        <TouchableOpacity
          style={styles.removePictureButton}
          onPress={() => handleRemoveDocument('profilePic', 0)}
        >
          <MaterialIcons name="close" size={scale(18)} color="#fff" />
        </TouchableOpacity>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <StatusBar
        translucent
        backgroundColor="transparent"
        barStyle="light-content"
      />
      
      {isNetworkError && (
        <Animated.View
          style={[styles.networkErrorBanner, { transform: [{ translateY: networkErrorAnim }] }]}
        >
          <Text style={styles.networkErrorText}>Internet connection not available</Text>
        </Animated.View>
      )}

      <KeyboardAvoidingView
        style={{ flex: 1 }}
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        keyboardVerticalOffset={Platform.OS === 'ios' ? scale(60) : scale(20)}
        enabled={Platform.OS === 'ios'}
      >
        <ScrollView
          ref={scrollViewRef}
          contentContainerStyle={[styles.scrollContent, { paddingBottom: scale(100) + insets.bottom }]}
          keyboardShouldPersistTaps="handled"
          nestedScrollEnabled={false}
          showsVerticalScrollIndicator={false}
          scrollEnabled={!isLoadingData}
          bounces={false}
          removeClippedSubviews={true}
          maxToRenderPerBatch={10}
          windowSize={10}
        >
          {/* Header */}
          <LinearGradient
            colors={['#3481BC', '#025383']}
            style={[styles.header, { height: scale(80) }]}
          >
            <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
            <View style={styles.headerContent}>
              <TouchableOpacity
                style={styles.backButton}
                onPress={handleBackStep}
                activeOpacity={0.8}
              >
                <MaterialIcons name="arrow-back" size={24} color="white" />
              </TouchableOpacity>
              
              <Text style={styles.headerTitle}>Edit User</Text>
            </View>
          </LinearGradient>

          <View style={styles.contentContainer}>
            {isLoadingData && (
              <View style={styles.loadingIndicator}>
                <ActivityIndicator size="large" color="#3481BC" />
                <Text style={styles.loadingText}>Loading team member data...</Text>
                <View style={styles.loadingProgress}>
                  <Text style={styles.loadingProgressText}>
                    {Object.values(loadingSections).filter(Boolean).length} of {Object.keys(loadingSections).length} sections loaded
                  </Text>
                </View>
              </View>
            )}
            <View style={styles.formContainer}>
              {/* Employee Profile Picture */}
              <Text style={styles.sectionTitle}>Employee Profile Picture</Text>
              <View style={styles.profilePictureContainer}>
                {isLoadingData ? (
                  <View style={styles.skeletonProfileContainer}>
                    <View style={styles.skeletonProfileImage} />
                    <SkeletonButton width={200} height={44} />
                  </View>
                ) : (
                  <>
                    {renderProfilePicturePreview()}
                    <GradientButton
                      title={profilePic ? 'Replace Picture' : 'Upload Picture'}
                      onPress={() => handleUploadPress('profilePic')}
                      style={[styles.uploadButton, profilePic && styles.replaceButton]}
                    />
                  </>
                )}
              </View>

              {/* Authentication Inputs */}
              <Text style={styles.sectionTitle}>Authentication Details</Text>
              {loadingSections.authentication ? (
                <>
                  <SkeletonInput />
                  <SkeletonInput />
                  <SkeletonInput />
                  <SkeletonInput />
                </>
              ) : (
                <>
                  <View style={styles.mainemailpass}>
                    <AnimatedInput
                      inputRef={nameInputRef}
                      label="Full Name"
                      value={name}
                      onChangeText={(text) => {
                        setName(text);
                        validateName(text);
                      }}
                      autoCapitalize="words"
                      leftIconName="person"
                      hasError={nameError}
                      onBlur={() => validateName(name)}
                      maxLength={50}
                      onFocus={handleInputFocus(nameInputRef)}
                    />
                    {nameError && <Text style={styles.errorText}>{nameErrorMessage}</Text>}
                  </View>

                  <View style={styles.mainemailpass}>
                    <AnimatedInput
                      inputRef={emailInputRef}
                      label="Email"
                      value={email}
                      onChangeText={(text) => {
                        setEmail(text);
                        validateEmail(text);
                      }}
                      keyboardType="email-address"
                      autoCapitalize="none"
                      leftIconName="email"
                      hasError={emailError}
                      onBlur={() => validateEmail(email)}
                      maxLength={40}
                      onFocus={handleInputFocus(emailInputRef)}
                    />
                    {emailError && <Text style={styles.errorText}>{emailErrorMessage}</Text>}
                  </View>

                  <View style={styles.mainemailpass}>
                    <AnimatedInput
                      inputRef={mobileInputRef}
                      label="Mobile"
                      value={mobile}
                      onChangeText={(text) => {
                        setMobile(text);
                        validateMobile(text);
                      }}
                      keyboardType="phone-pad"
                      leftIconName="phone"
                      hasError={mobileError}
                      onBlur={() => validateMobile(mobile)}
                      maxLength={15}
                      onFocus={handleInputFocus(mobileInputRef)}
                    />
                    {mobileError && <Text style={styles.errorText}>{mobileErrorMessage}</Text>}
                  </View>

                  <View style={styles.mainemailpass}>
                    <Text style={styles.passwordInfoText}>Password (Leave blank to keep unchanged)</Text>
                <AnimatedInput
                  inputRef={passwordInputRef}
                  label="Password"
                  value={password}
                  onChangeText={(text) => {
                    setPassword(text);
                    validatePassword(text);
                  }}
                  secureTextEntry={!showPassword}
                  autoCapitalize="none"
                  leftIconName="lock"
                  rightIconName={showPassword ? 'visibility' : 'visibility-off'}
                  onRightIconPress={toggleShowPassword}
                  hasError={passwordError}
                  onBlur={() => validatePassword(password)}
                  maxLength={15}
                  onFocus={handleInputFocus(passwordInputRef)}
                />
                {passwordError && <Text style={styles.errorText}>{passwordErrorMessage}</Text>}
              </View>
                </>
              )}

              {/* Role Dropdown */}
              {loadingSections.basic ? (
                <View style={styles.inputContainer}>
                  <SkeletonText width="40%" height={18} />
                  <SkeletonInput height={50} />
                </View>
              ) : (
                <View style={inputStyles.container}>
                  <Text style={inputStyles.label}>
                    Role <Text style={inputStyles.requiredError}>*</Text>
                  </Text>
                  <DropDownPicker
                    open={roleOpen}
                    value={role}
                    items={roles}
                    setOpen={(open) => {
                      setRoleOpen(open);
                      if (open) {
                        setCountryOpen(false);
                        setProjectOpen(false);
                        setModuleOpen(false);
                        setPermissionOpen({});
                      }
                    }}
                    setValue={setRole}
                    setItems={() => {}}
                    placeholder="Select Role"
                    zIndex={7000}
                    listMode="SCROLLVIEW"
                    scrollViewProps={{
                      nestedScrollEnabled: true,
                    }}
                    style={[inputStyles.input, errors.role ? inputStyles.inputError : null]}
                    dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                    textStyle={styles.dropdownText}
                    placeholderStyle={styles.placeholderStyle}
                    badgeDotColors={['#015185']}
                    badgeTextStyle={styles.badgeText}
                    badgeStyle={styles.badge}
                  />
                  {errors.role && <Text style={inputStyles.errorText}>Role is required</Text>}
                </View>
              )}

              {/* General Details */}
              <Text style={styles.sectionTitle}>General Details</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={employeeNumberInputRef}
                  label="Employee Number"
                  value={employeeNumber}
                  onChangeText={(text) => {
                    setEmployeeNumber(text);
                    validateEmployeeNumber(text);
                  }}
                  hasError={employeeNumberError}
                  onBlur={() => validateEmployeeNumber(employeeNumber)}
                  maxLength={20}
                  onFocus={handleInputFocus(employeeNumberInputRef)}
                  leftIconName="badge"
                />
                {employeeNumberError && <Text style={styles.errorText}>{employeeNumberErrorMessage}</Text>}
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={departmentInputRef}
                  label="Department"
                  value={department}
                  onChangeText={(text) => {
                    setDepartment(text);
                    validateDepartment(text);
                  }}
                  hasError={departmentError}
                  onBlur={() => validateDepartment(department)}
                  onFocus={handleInputFocus(departmentInputRef)}
                  leftIconName="business"
                />
                {departmentError && <Text style={styles.errorText}>{departmentErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={nrcIdNumberInputRef}
                  label="NRC / ID Number"
                  value={nrcIdNumber}
                  onChangeText={(text) => {
                    setNrcIdNumber(text);
                    validateNrcIdNumber(text);
                  }}
                  hasError={nrcIdNumberError}
                  onBlur={() => validateNrcIdNumber(nrcIdNumber)}
                  onFocus={handleInputFocus(nrcIdNumberInputRef)}
                  leftIconName="credit-card"
                />
                {nrcIdNumberError && <Text style={styles.errorText}>{nrcIdNumberErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Engagement Date"
                  value={formatDate(engagementDate)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('engagementDate')}
                />
              </View>

              {/* Country Dropdown */}
              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  Countries <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  multiple={true}
                  open={countryOpen}
                  value={countries}
                  items={countryItems}
                  setOpen={(open) => {
                    setCountryOpen(open);
                    if (open) {
                      setProjectOpen(false);
                      setRoleOpen(false);
                      setModuleOpen(false);
                      setPermissionOpen({});
                    }
                  }}
                  setValue={setCountries}
                  setItems={setCountryItems}
                  placeholder="Select Countries"
                  zIndex={10000}
                  listMode="SCROLLVIEW"
                  scrollViewProps={{
                    nestedScrollEnabled: true,
                  }}
                  style={[inputStyles.input, errors.country ? inputStyles.inputError : null]}
                  dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  badgeDotColors={['#015185']}
                  badgeTextStyle={styles.badgeText}
                  badgeStyle={styles.badge}
                />
                {errors.country && (
                  <Text style={inputStyles.errorText}>At least one country is required</Text>
                )}
                <View style={styles.chipContainer}>
                  {countries.length > 0 ? (
                    countries.map((country, index) => (
                      <View key={index} style={styles.chip}>
                        <Text style={styles.chipText}>{country}</Text>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.noItemsText}>No countries selected</Text>
                  )}
                </View>
              </View>

              {/* Project Dropdown */}
              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  Projects <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  multiple={true}
                  open={projectOpen}
                  value={projects}
                  items={projectItems}
                  setOpen={(open) => {
                    setProjectOpen(open);
                    if (open) {
                      setCountryOpen(false);
                      setRoleOpen(false);
                      setModuleOpen(false);
                      setPermissionOpen({});
                    }
                  }}
                  setValue={setProjects}
                  setItems={setProjectItems}
                  placeholder="Select Projects"
                  zIndex={9000}
                  listMode="SCROLLVIEW"
                  scrollViewProps={{
                    nestedScrollEnabled: true,
                  }}
                  style={[inputStyles.input, errors.project ? inputStyles.inputError : null]}
                  dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  badgeDotColors={['#015185']}
                  badgeTextStyle={styles.badgeText}
                  badgeStyle={styles.badge}
                />
                {errors.project && (
                  <Text style={inputStyles.errorText}>At least one project is required</Text>
                )}
                <View style={styles.chipContainer}>
                  {projects.length > 0 ? (
                    projects.map((project, index) => (
                      <View key={index} style={styles.chip}>
                        <Text style={styles.chipText}>{project}</Text>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.noItemsText}>No projects selected</Text>
                  )}
                </View>
              </View>



              {/* Titan Induction */}
              <Text style={styles.sectionTitle}>Titan Induction</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Titan Induction Date"
                  value={formatDate(titanInductionDate)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('titanInductionDate')}
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Client Induction"
                  value={formatDate(clientInduction)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('clientInduction')}
                />
              </View>

              {/* Medicals */}
              <Text style={styles.sectionTitle}>Medicals</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Date Examined"
                  value={formatDate(dateExamined)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('dateExamined')}
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Expiry"
                  value={formatDate(expiryMedicals)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('expiryMedicals')}
                />
              </View>
              <GradientButton
                title="Upload Medical Docs"
                onPress={() => handleUploadPress('medicalDocs')}
                style={styles.uploadButton}
              />
              {renderDocumentPreviews('medicalDocs')}

              {/* Driver's License */}
              <Text style={styles.sectionTitle}>Driver's License</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={driverLicenseNumberInputRef}
                  label="Number"
                  value={driverLicenseNumber}
                  onChangeText={setDriverLicenseNumber}
                  onFocus={handleInputFocus(driverLicenseNumberInputRef)}
                  leftIconName="credit-card"
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Code"
                  value={driverLicenseCode}
                  onChangeText={setDriverLicenseCode}
                  leftIconName="vpn-key"
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Expiry"
                  value={formatDate(driverLicenseExpiry)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('driverLicenseExpiry')}
                />
              </View>
              <GradientButton
                title="Upload License Docs"
                onPress={() => handleUploadPress('driverLicenseDocs')}
                style={styles.uploadButton}
              />
              {renderDocumentPreviews('driverLicenseDocs')}

              {/* Defensive Driving Data */}
              <Text style={styles.sectionTitle}>Defensive Driving Data</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Client Driving Date"
                  value={formatDate(clientDrivingDate)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('clientDrivingDate')}
                />
              </View>
              <GradientButton
                title="Upload Driving Docs"
                onPress={() => handleUploadPress('defensiveDrivingDocs')}
                style={styles.uploadButton}
              />
              {renderDocumentPreviews('defensiveDrivingDocs')}

              {/* First Aid Training */}
              <Text style={styles.sectionTitle}>First Aid Training</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Date Completed"
                  value={formatDate(dateCompletedFirstAid)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('dateCompletedFirstAid')}
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={levelFirstAidInputRef}
                  label="Level"
                  value={levelFirstAid}
                  onChangeText={setLevelFirstAid}
                  onFocus={handleInputFocus(levelFirstAidInputRef)}
                  leftIconName="medical-services"
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Expiry Date"
                  value={formatDate(expiryDateFirstAid)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('expiryDateFirstAid')}
                />
              </View>
            </View>
          </View>
        </ScrollView>
        {/* Bottom Back/Update Buttons */}
        <View style={[
          styles.buttonContainer,
          { paddingBottom: scale(10) + insets.bottom, bottom: insets.bottom }
        ]}>
          {uploadInProgress && (
            <View style={styles.progressBarContainer}>
              <View style={styles.progressBarTrack}>
                <View style={[styles.progressBarFill, { width: `${uploadProgress}%` }]} />
              </View>
              <Text style={styles.progressText}>{uploadProgress}%</Text>
            </View>
          )}
          <TouchableOpacity
            style={styles.flatButton}
            onPress={handleBackStep}
            disabled={isLoadingSignUp}
          >
            <Text style={styles.flatButtonText}>{'Cancel'}</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.flatButton}
            onPress={handleUpdate}
            disabled={isLoadingSignUp}
          >
            <Text style={styles.flatButtonText}>{'Update'}</Text>
          </TouchableOpacity>
        </View>
      </KeyboardAvoidingView>

      {/* iOS Date Picker Modal - Only for iOS */}
      {Platform.OS === 'ios' && showDatePicker && (
        <Modal
          visible={showDatePicker}
          animationType="slide"
          transparent={true}
          onRequestClose={() => {
            setShowDatePicker(false);
            setCurrentEditingDateField('');
            setDatePickerMaxDate(null);
            setDatePickerMinDate(new Date(1900, 0, 1));
          }}
          presentationStyle="pageSheet"
        >
          <View style={styles.datePickerContainer}>
            <View style={styles.datePickerHeader}>
              <Text style={styles.datePickerTitle}>
                Select {currentEditingDateField ? currentEditingDateField.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase()) : 'Date'}
              </Text>
              <TouchableOpacity
                style={styles.datePickerCloseButton}
                onPress={handleDatePickerDone}
              >
                <Text style={styles.datePickerCloseText}>Done</Text>
              </TouchableOpacity>
            </View>
            <View style={styles.datePickerContent}>
              <DateTimePicker
                testID="dateTimePicker"
                value={datePickerValue}
                mode="date"
                display="spinner"
                onChange={onDateChange}
                maximumDate={datePickerMaxDate}
                minimumDate={datePickerMinDate}
                accentColor="#025383"
                textColor="#333"
                style={styles.datePicker}
                themeVariant="light"
              />
            </View>
          </View>
        </Modal>
      )}

      {/* Native Android Date Picker - Only for Android */}
      {Platform.OS === 'android' && showDatePicker && (
        <DateTimePicker
          testID="dateTimePicker"
          value={datePickerValue}
          mode="date"
          display="default"
          onChange={onDateChange}
          maximumDate={datePickerMaxDate}
          minimumDate={datePickerMinDate}
          accentColor="#025383"
          textColor="#333"
        />
      )}

      <Modal
        visible={uploadModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setUploadModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Upload Document</Text>
            {isUploading ? (
              <View style={styles.loaderContainer}>
                <ActivityIndicator size="large" color="#015185" />
                <Text style={styles.loaderText}>Uploading...</Text>
              </View>
            ) : (
              <>
                <ScrollView style={styles.uploadList}>
                  {currentUploadField === 'profilePic' ? (
                    <>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handlePickImage}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="photo-library" size={scale(24)} color="#333" />
                        <Text style={styles.uploadOptionText}>Pick Image</Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handleTakePhoto}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="camera-alt" size={scale(24)} color="#333" />
                        <Text style={styles.uploadOptionText}>Take Photo</Text>
                      </TouchableOpacity>
                    </>
                  ) : (
                    <>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handlePickImage}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="photo-library" size={scale(24)} color="#333" />
                        <Text style={styles.uploadOptionText}>Pick Image</Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handleTakePhoto}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="camera-alt" color="#333" />
                        <Text style={styles.uploadOptionText}>Take Photo</Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handlePickPDF}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="picture-as-pdf" size={scale(24)} color="#333" />
                        <Text style={styles.uploadOptionText}>Pick PDF</Text>
                      </TouchableOpacity>
                    </>
                  )}
                </ScrollView>
                <TouchableOpacity
                  style={styles.modalCloseButton}
                  onPress={() => setUploadModalVisible(false)}
                  disabled={isUploading}
                >
                  <Text style={styles.modalCloseText}>Cancel</Text>
                </TouchableOpacity>
              </>
            )}
          </View>
        </View>
      </Modal>

      <SyncModal 
        visible={syncModalVisible} 
        onClose={() => {
          // CRITICAL: Never allow closing during critical phases
          if (updateProgress >= 90 && updateProgress < 100) {
            console.log('Blocking modal close during critical completion phase');
            return;
          }
          
          // Only allow closing if update is completed or failed
          if (isUpdateCompleted || updateProgress === 0) {
            setSyncModalVisible(false);
          }
        }} 
        message={syncModalMessage} 
        progress={updateProgress}
        isCompleted={isUpdateCompleted}
      />

      <SuccessPopup
        visible={successPopupVisible}
        title={successPopupData.title}
        message={successPopupData.message}
        onOkPress={() => {
          try {
            setSuccessPopupVisible(false);
            // Small delay to ensure popup is closed before navigation
            setTimeout(() => {
              if (isMounted) {
                navigation.goBack();
              }
            }, 100);
          } catch (error) {
            console.warn('Success popup close error:', error);
            // Fallback navigation
            if (isMounted) {
              navigation.goBack();
            }
          }
        }}
      />
    </View>
  );
};

// Input Styles
const inputStyles = StyleSheet.create({
  container: {
    marginBottom: scale(20),
    marginHorizontal: scale(10),
  },
  label: {
    fontSize: scale(14),
    color: '#015185',
    fontWeight: '500',
    marginBottom: scale(4),
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: scale(4),
    paddingHorizontal: scale(12),
    paddingVertical: scale(14),
    fontSize: scale(16),
    backgroundColor: '#fff',
    alignItems: 'center',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: scale(12),
    marginTop: scale(2),
    marginLeft: scale(2),
  },
});

// Styles
const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollContent: {
    paddingBottom: scale(100),
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 16,
    paddingTop: Platform.OS === 'ios' ? 50 : 20,
    height: '100%',
  },
  headerTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: 'white',
  },
  backButton: {
    position: 'absolute',
    left: 16,
    padding: 8,
    top:40
  },
  contentContainer: {
    flex: 1,
    paddingHorizontal: scale(20),
  },
  formContainer: {
    marginTop: scale(20),
  },
  sectionTitle: {
    fontSize: scale(18),
    fontWeight: '600',
    color: '#333',
    marginVertical: scale(10),
  },
  mainemailpass: {
    marginBottom: scale(10),
    
  },
  inputContainer: {
    marginBottom: scale(5),
  },
  inputGradient: {
    borderRadius: scale(6),
    padding: scale(2),
    backgroundColor: '#fff',
  },
  input: {
    backgroundColor: '#fff',
    fontSize: scale(16),
    height: scale(50),
    paddingHorizontal: scale(10),
  },
  errorText: {
    color: '#ff0000',
    fontSize: scale(12),
    marginTop: scale(2),
    marginLeft: scale(10),
  },
  passwordInfoText: {
    color: '#666',
    fontSize: scale(14),
    marginBottom: scale(8),
    fontStyle: 'italic',
    fontWeight: '500',
    

  },
  gradientButton: {
    borderRadius: scale(6),
    padding: scale(2),
    marginVertical: scale(10),
  },
  buttonTouchable: {
    paddingVertical: scale(12),
    paddingHorizontal: scale(20),
    alignItems: 'center',
  },
  googleButtonText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: '600',
  },
  flatButton: {
    paddingVertical: scale(12),
    paddingHorizontal: scale(20),
    alignItems: 'center',
  },
  flatButtonText: {
    color: '#015185',
    fontSize: scale(16),
    fontWeight: '600',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: scale(20),
    paddingVertical: scale(10),
    backgroundColor: '#fff',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
  },
  progressBarContainer: {
    position: 'absolute',
    left: scale(20),
    right: scale(20),
    top: -scale(16),
    alignItems: 'center',
  },
  progressBarTrack: {
    width: '100%',
    height: scale(6),
    backgroundColor: '#e6eef7',
    borderRadius: scale(3),
    overflow: 'hidden',
  },
  progressBarFill: {
    height: '100%',
    backgroundColor: '#3481BC',
  },
  progressText: {
    marginTop: scale(4),
    fontSize: scale(12),
    color: '#29486A',
    fontWeight: '600',
  },
  networkErrorBanner: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    backgroundColor: '#ff4444',
    padding: scale(10),
    zIndex: 1000,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: scale(14),
    textAlign: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  // Progress bar styles
  progressContainer: {
    marginTop: scale(20),
    width: '100%',
    alignItems: 'center',
  },
  progressBar: {
    width: '100%',
    height: scale(8),
    backgroundColor: '#e0e0e0',
    borderRadius: scale(4),
    overflow: 'hidden',
  },
  progressFill: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: scale(4),
  },
  progressText: {
    marginTop: scale(8),
    fontSize: scale(14),
    color: '#333',
    fontWeight: '600',
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: scale(5),
    backgroundColor: '#fff',
  },
  dropdownText: {
    fontSize: scale(14),
    color: '#333',
  },
  placeholderStyle: {
    fontSize: scale(14),
    color: '#888',
  },
  badgeText: {
    fontSize: scale(12),
    color: '#fff',
  },
  badge: {
    backgroundColor: '#015185',
    borderRadius: scale(10),
    paddingHorizontal: scale(8),
    paddingVertical: scale(4),
  },
  chipContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginTop: scale(8),
  },
  chip: {
    backgroundColor: '#e0f0ff',
    borderRadius: scale(16),
    paddingVertical: scale(6),
    paddingHorizontal: scale(12),
    marginRight: scale(8),
    marginBottom: scale(8),
  },
  chipText: {
    color: '#015185',
    fontSize: scale(12),
    fontWeight: '500',
  },
  noItemsText: {
    fontSize: scale(12),
    color: '#888',
    marginTop: scale(8),
  },

  label: {
    fontSize: scale(14),
    color: '#333',
    marginBottom: scale(4),
  },
  requiredError: {
    color: '#ff0000',
  },
  profilePictureContainer: {
    alignItems: 'center',
    marginBottom: scale(20),
  },
  profilePicturePreview: {
    position: 'relative',
    marginBottom: scale(10),
  },
  profilePictureImage: {
    width: scale(100),
    height: scale(100),
    borderRadius: scale(50),
    borderWidth: 2,
    borderColor: '#015185',
  },
  removePictureButton: {
    position: 'absolute',
    top: scale(-5),
    right: scale(-5),
    backgroundColor: '#ff4444',
    borderRadius: scale(12),
    padding: scale(2),
  },
  uploadButton: {
    width: scale(200),
    alignSelf: 'flex-start',
    textAlign: 'center',
  },
  replaceButton: {
    backgroundColor: '#ff4444',
  },
  documentPreviewContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'center',
    marginTop: scale(10),
  },
  documentPreviewItem: {
    width: scale(100),
    margin: scale(10),
    alignItems: 'center',
  },
  documentPreviewImage: {
    width: scale(80),
    height: scale(80),
    borderRadius: scale(6),
    borderWidth: 1,
    borderColor: '#ddd',
  },
  pdfIconContainer: {
    width: scale(80),
    height: scale(80),
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f0f0f0',
    borderRadius: scale(6),
    borderWidth: 1,
    borderColor: '#ddd',
  },
  documentName: {
    fontSize: scale(12),
    color: '#333',
    marginTop: scale(5),
    textAlign: 'center',
    maxWidth: scale(80),
  },
  removeDocumentButton: {
    position: 'absolute',
    top: scale(-5),
    right: scale(-5),
    backgroundColor: '#ff4444',
    borderRadius: scale(12),
    padding: scale(2),
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    width: scale(300),
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: scale(18),
    fontWeight: '600',
    color: '#333',
    marginBottom: scale(20),
  },
  uploadList: {
    maxHeight: scale(200),
    width: '100%',
  },
  uploadOption: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: scale(10),
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  uploadOptionText: {
    fontSize: scale(16),
    color: '#333',
    marginLeft: scale(10),
  },
  modalCloseButton: {
    marginTop: scale(20),
    padding: scale(10),
    backgroundColor: '#015185',
    borderRadius: scale(6),
  },
  modalCloseText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: '600',
  },
  loaderContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    padding: scale(20),
  },
  loaderText: {
    fontSize: scale(16),
    color: '#333',
    marginTop: scale(10),
  },
  datePickerContainer: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    marginHorizontal: width * 0.05,
    marginVertical: Platform.OS === 'ios' ? width * 0.125 : width * 0.25,
    maxHeight: Platform.OS === 'ios' ? '80%' : undefined,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
    overflow: 'hidden',
  },
  datePicker: {
    width: Platform.OS === 'ios' ? '100%' : width * 0.75,
    height: Platform.OS === 'ios' ? width * 0.5 : undefined,
  },
  datePickerCloseButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    backgroundColor: '#015185',
    borderRadius: width * 0.015,
    minWidth: width * 0.15,
    alignItems: 'center',
  },
  datePickerCloseText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: 'bold',
  },
  datePickerHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: width * 0.0375,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    backgroundColor: '#f8f9fa',
  },
  datePickerTitle: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
    flex: 1,
  },
  datePickerContent: {
    padding: width * 0.05,
    alignItems: 'center',
    justifyContent: 'center',
  },

  // Progress bar styles
  progressContainer: {
    marginTop: scale(20),
    width: '100%',
    alignItems: 'center',
  },
  progressBar: {
    width: '100%',
    height: scale(8),
    backgroundColor: '#e0e0e0',
    borderRadius: scale(4),
    overflow: 'hidden',
  },
  progressFill: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: scale(4),
  },
  progressText: {
    marginTop: scale(8),
    fontSize: scale(14),
    color: '#333',
    fontWeight: '600',
  },
  // Success popup styles
  successModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContent: {
    width: '80%',
    backgroundColor: 'white',
    borderRadius: scale(15),
    padding: scale(25),
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
  },
  successIconContainer: {
    marginBottom: scale(15),
  },
  successTitle: {
    fontSize: scale(20),
    fontWeight: 'bold',
    color: '#333',
    marginBottom: scale(10),
    textAlign: 'center',
  },
  successMessage: {
    fontSize: scale(16),
    color: '#666',
    marginBottom: scale(20),
    textAlign: 'center',
    lineHeight: scale(22),
  },
  successOkButton: {
    backgroundColor: '#015185',
    paddingVertical: scale(12),
    paddingHorizontal: scale(30),
    borderRadius: scale(8),
    minWidth: scale(100),
  },
  successOkButtonText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: '600',
    textAlign: 'center',
  },
  // Dropdown input styles
  dropdownInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: scale(6),
    backgroundColor: '#fff',
    paddingHorizontal: scale(10),
    fontSize: scale(14),
    minHeight: scale(50),
  },
  dropdownInputError: {
    borderColor: '#ff0000',
  },
  requiredError: {
    color: '#ff0000',
  },
  
  // Skeleton loader styles
  skeletonInput: {
    backgroundColor: '#f0f0f0',
    borderRadius: scale(6),
    marginBottom: scale(10),
    overflow: 'hidden',
  },
  skeletonText: {
    backgroundColor: '#f0f0f0',
    borderRadius: scale(4),
    marginBottom: scale(10),
    overflow: 'hidden',
  },
  skeletonButton: {
    backgroundColor: '#f0f0f0',
    borderRadius: scale(6),
    marginBottom: scale(10),
    overflow: 'hidden',
  },
  skeletonShimmer: {
    width: '100%',
    height: '100%',
    backgroundColor: '#e0e0e0',
  },
  skeletonProfileContainer: {
    alignItems: 'center',
    marginBottom: scale(20),
  },
  skeletonProfileImage: {
    width: scale(100),
    height: scale(100),
    borderRadius: scale(50),
    backgroundColor: '#f0f0f0',
    marginBottom: scale(10),
  },
  
  // Loading indicator styles
  loadingIndicator: {
    alignItems: 'center',
    paddingVertical: scale(20),
    backgroundColor: '#f8f9fa',
    borderRadius: scale(10),
    marginBottom: scale(20),
  },
  loadingText: {
    fontSize: scale(16),
    color: '#666',
    marginTop: scale(10),
    fontWeight: '500',
  },
  loadingProgress: {
    marginTop: scale(10),
    paddingHorizontal: scale(15),
    paddingVertical: scale(8),
    backgroundColor: '#e3f2fd',
    borderRadius: scale(15),
  },
  loadingProgressText: {
    fontSize: scale(12),
    color: '#1976d2',
    fontWeight: '600',
  },
});



export default Edit_team_member;