import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;
const MARGIN = 8;

const dropdownData = [
  { label: 'Approve', icon: 'check-circle', iconColor: '#3481BC', action: 'handleApprove' },
  { label: 'Edit', icon: 'edit', iconColor: '#3481BC', action: 'handleEdit' },
  { label: 'View', icon: 'visibility', iconColor: '#4ECB71', action: 'handleView' },
  { label: 'Delete', icon: 'delete', iconColor: '#C2185B', action: 'handleDelete' },
];
const DROPDOWN_HEIGHT = dropdownData.length * 44;

// Define priority icons and colors
const priorityIcons = {
  Low: { icon: 'arrow-downward', color: '#4ECB71' }, // Green
  Medium: { icon: 'priority-high', color: '#FFC107' }, // Yellow
  High: { icon: 'warning', color: '#FF5722' }, // Orange
  Emergency: { icon: 'error', color: '#C2185B' }, // Red
};

// Define status badge colors
const statusColors = {
  Requested: '#3481BC', // Blue
  Approved: '#4ECB71',  // Green
  Rejected: '#C2185B',  // Red
};

const RequestmaintenanceHome = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [maintenanceRequests, setMaintenanceRequests] = useState([]);
  const [filteredRequests, setFilteredRequests] = useState([]);
  const [userId, setUserId] = useState(null);
  const [selectedRequest, setSelectedRequest] = useState(null);
  const [dropdownVisible, setDropdownVisible] = useState(false);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0 });
  const [deleteConfirmVisible, setDeleteConfirmVisible] = useState(false);

  // Animation setup for sync icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  // Show toast notification
  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  // Fetch user UID from AsyncStorage
  useEffect(() => {
    const loadUserId = async () => {
      try {
        const uid = await AsyncStorage.getItem('userUid');
        if (uid) {
          setUserId(uid);
        } else {
          showToast('error', 'Error', 'User not logged in. Please log in again.');
          navigation.replace('AdminLogin');
        }
      } catch (error) {
        console.error('Error fetching user UID:', error);
        showToast('error', 'Error', 'Failed to load user data.');
      }
    };

    loadUserId();
  }, [navigation]);

  // Fetch maintenance requests
  const fetchMaintenanceRequests = useCallback(async () => {
    if (!userId) return;
    setSyncModalVisible(true);
    try {
      const isOnline = await checkNetworkStatus();
      let requests = [];

      if (isOnline) {
        const snapshot = await database()
          .ref(`/maintenanceRequests/${userId}`)
          .once('value');
        const data = snapshot.val();
        requests = data ? Object.values(data) : [];
        await AsyncStorage.setItem('maintenanceRequests', JSON.stringify(requests));
        showToast('success', 'Success', 'Maintenance requests synced successfully.');
      } else {
        const storedRequests = await AsyncStorage.getItem('maintenanceRequests');
        requests = storedRequests ? JSON.parse(storedRequests) : [];
        showToast('info', 'Offline', 'Using offline data. Will sync when online.');
      }

      setMaintenanceRequests(requests);
      setFilteredRequests(requests);
    } catch (error) {
      console.error('Error fetching maintenance requests:', error);
      let errorMessage = 'Failed to fetch maintenance requests.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (error.code === 'NETWORK_ERROR' || !(await checkNetworkStatus())) {
        errorMessage = 'Offline mode. Showing locally stored data.';
        const storedRequests = await AsyncStorage.getItem('maintenanceRequests');
        requests = storedRequests ? JSON.parse(storedRequests) : [];
        setMaintenanceRequests(requests);
        setFilteredRequests(requests);
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      setSyncModalVisible(false);
    }
  }, [userId]);

  // Auto-refresh on screen focus
  useFocusEffect(
    useCallback(() => {
      if (userId) {
        fetchMaintenanceRequests();
      }
    }, [fetchMaintenanceRequests, userId])
  );

  // Handle search
  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text.trim()) {
      const filtered = maintenanceRequests.filter(
        (request) =>
          (request.workOrderTitle?.toLowerCase() || '').includes(text.toLowerCase()) ||
          (request.requestType?.toLowerCase() || '').includes(text.toLowerCase()) ||
          (request.equipmentName?.toLowerCase() || '').includes(text.toLowerCase()) ||
          (request.requestNumber?.toLowerCase() || '').includes(text.toLowerCase())
      );
      setFilteredRequests(filtered);
    } else {
      setFilteredRequests(maintenanceRequests);
    }
  };

  // Handle refresh/sync
  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    fetchMaintenanceRequests().finally(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    });
  };

  // Dropdown handlers
  const handleApprove = () => {
    setDropdownVisible(false);
    if (selectedRequest) {
      navigation.navigate('RequestMApproveWO', { requestId: selectedRequest.id });
    }
  };

  const handleEdit = () => {
    setDropdownVisible(false);
    if (selectedRequest) {
      navigation.navigate('UpdateRequestMaintenance', { requestId: selectedRequest.id });
    }
  };

  const handleView = () => {
    setDropdownVisible(false);
    if (selectedRequest) {
      navigation.navigate('ViewMRequest', { requestId: selectedRequest.id });
    }
  };

  const handleDelete = () => {
    setDropdownVisible(false);
    setDeleteConfirmVisible(true);
  };

  const confirmDelete = async () => {
    if (!selectedRequest || !userId) return;

    setDeleteConfirmVisible(false);
    setSyncModalVisible(true);

    try {
      const isOnline = await checkNetworkStatus();

      const updatedRequests = maintenanceRequests.filter(
        (request) => request.id !== selectedRequest.id
      );
      setMaintenanceRequests(updatedRequests);
      setFilteredRequests(updatedRequests);

      await AsyncStorage.setItem('maintenanceRequests', JSON.stringify(updatedRequests));

      if (isOnline) {
        await database()
          .ref(`/maintenanceRequests/${userId}/${selectedRequest.id}`)
          .remove();
        showToast('success', 'Deleted', 'Maintenance request deleted successfully.');
      } else {
        showToast('info', 'Offline', 'Request deleted locally. Will sync when online.');
      }
    } catch (error) {
      console.error('Error deleting maintenance request:', error);
      let errorMessage = 'Failed to delete maintenance request.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (error.code === 'NETWORK_ERROR' || !(await checkNetworkStatus())) {
        errorMessage = 'Offline mode. Deletion saved locally and will sync when online.';
      }
      showToast('error', 'Error', errorMessage);
      await fetchMaintenanceRequests();
    } finally {
      setSyncModalVisible(false);
      setSelectedRequest(null);
    }
  };

  const cancelDelete = () => {
    setDeleteConfirmVisible(false);
    setSelectedRequest(null);
  };

  const dropdownActions = {
    handleApprove,
    handleEdit,
    handleView,
    handleDelete,
  };

  // Render each maintenance request
  const renderMaintenanceRequest = ({ item }) => {
    const priority = item.priority || 'Low';
    const priorityConfig = priorityIcons[priority] || priorityIcons['Low'];
    const status = item.status || 'Requested';
    const statusColor = statusColors[status] || statusColors['Requested'];

    return (
      <View style={styles.requestCard}>
        <View style={styles.cardHeader}>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <MaterialIcons name="build" size={22} color="#4ECB71" />
            <Text style={styles.woNumber}>{item.requestNumber || 'WO-NA'}</Text>
            <MaterialIcons
              name={priorityConfig.icon}
              size={18}
              color={priorityConfig.color}
              style={{ marginLeft: 6 }}
            />
            <View style={[styles.statusBadge, { backgroundColor: statusColor }]}>
              <Text style={styles.statusText}>{status}</Text>
            </View>
          </View>
          <TouchableOpacity
            onPress={(event) => {
              setSelectedRequest(item);
              const { pageX, pageY } = event.nativeEvent;
              const screenWidth = Dimensions.get('window').width;
              const screenHeight = Dimensions.get('window').height;
              const dropdownWidth = 220;
              let top = pageY + MARGIN;
              let left = pageX - dropdownWidth + 30; // Adjust based on button width
              if (pageY + DROPDOWN_HEIGHT + MARGIN > screenHeight) {
                top = pageY - DROPDOWN_HEIGHT - MARGIN;
              }
              if (left < MARGIN) {
                left = MARGIN;
              } else if (left + dropdownWidth > screenWidth - MARGIN) {
                left = screenWidth - dropdownWidth - MARGIN;
              }
              setDropdownPosition({ x: left, y: top });
              setDropdownVisible(true);
            }}
            style={styles.menuButton}
          >
            <MaterialIcons name="more-vert" size={24} color="#333" />
          </TouchableOpacity>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Title</Text>
          <Text style={styles.cardValue}>{item.workOrderTitle || 'N/A'}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Vehicle #</Text>
          <Text style={styles.cardValue}>{item.equipmentName || 'N/A'}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Assign</Text>
          <Text style={styles.cardValue}>{item.assignTo?.map(a => a.name).join(', ') || 'Unassigned'}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Create Date</Text>
          <Text style={styles.cardValue}>
            {item.createdAt
              ? new Date(item.createdAt).toLocaleString('en-GB', {
                  day: '2-digit',
                  month: 'short',
                  year: '2-digit',
                  hour: '2-digit',
                  minute: '2-digit',
                  hour12: true,
                }).replace(',', '')
              : 'N/A'}
          </Text>
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Request Maintenance</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleRefresh}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
          </View>
        </View>

        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by title, type, vehicle..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity style={styles.filterButton}>
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Main Content */}
      <FlatList
        data={filteredRequests}
        renderItem={renderMaintenanceRequest}
        keyExtractor={(item) => item.id || Math.random().toString()}
        contentContainerStyle={styles.mainContent}
        ListEmptyComponent={
          <Text style={styles.defaultText}>No maintenance requests found.</Text>
        }
      />

      {/* Bottom Button */}
      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity
            style={styles.startButton}
            onPress={() => navigation.navigate('CreateRequestMaintenance')}
          >
            <Ionicons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}>New Request</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>
                {deleteConfirmVisible ? 'Deleting Request...' : 'Syncing Data...'}
              </Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Delete Confirmation Modal */}
      <Modal
        visible={deleteConfirmVisible}
        transparent
        animationType="fade"
        onRequestClose={cancelDelete}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.confirmModalContent}>
            <Text style={styles.confirmModalText}>Are you sure?</Text>
            <Text style={styles.confirmModalSubText}>
              Do you really want to delete this maintenance request?
            </Text>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={cancelDelete}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.deleteButton]}
                onPress={confirmDelete}
              >
                <Text style={styles.deleteButtonText}>Delete</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Dropdown Modal */}
      <Modal
        visible={dropdownVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setDropdownVisible(false)}
      >
        <TouchableOpacity
          style={styles.dropdownOverlay}
          onPress={() => setDropdownVisible(false)}
          activeOpacity={1}
        >
          <View
            style={[
              styles.dropdownContainer,
              {
                position: 'absolute',
                top: dropdownPosition.y,
                left: dropdownPosition.x,
              },
            ]}
          >
            {dropdownData.map((option, index) => (
              <TouchableOpacity
                key={option.label}
                style={[
                  styles.dropdownItem,
                  index === dropdownData.length - 1 && { borderBottomWidth: 0 },
                ]}
                onPress={() => dropdownActions[option.action]()}
              >
                <MaterialIcons name={option.icon} size={22} color={option.iconColor} />
                <Text style={styles.dropdownText}>{option.label}</Text>
              </TouchableOpacity>
            ))}
          </View>
        </TouchableOpacity>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  mainContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.2,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  requestCard: {
    backgroundColor: '#f9f9f9',
    borderRadius: width * 0.02,
    padding: width * 0.04,
    marginBottom: width * 0.03,
    borderWidth: 1,
    borderColor: '#ddd',
    elevation: 2,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  woNumber: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#3481BC',
    marginLeft: 6,
  },
  statusBadge: {
    borderRadius: 4,
    paddingHorizontal: 8,
    paddingVertical: 2,
    marginLeft: 10,
  },
  statusText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: 'bold',
  },
  menuButton: {
    padding: 4,
    hitSlop: { top: 10, bottom: 10, left: 10, right: 10 },
  },
  cardRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 4,
  },
  cardLabel: {
    color: '#333',
    fontWeight: '500',
    fontSize: width * 0.035,
    width: '30%',
  },
  cardValue: {
    color: '#222',
    fontSize: width * 0.035,
    fontWeight: '400',
    flex: 1,
    textAlign: 'right',
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.2)',
    top: '-4%',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#eee',
    width: 220,
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.12,
    shadowRadius: 8,
    paddingVertical: 4,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  dropdownText: {
    fontSize: width * 0.04,
    color: '#222',
    marginLeft: 16,
    fontWeight: '500',
  },
  buttonContainer: {
    position: 'absolute',
    bottom: width * 0.05,
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: 10,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  confirmModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
    alignItems: 'center',
  },
  confirmModalText: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
  },
  confirmModalSubText: {
    fontSize: width * 0.04,
    color: '#333',
    marginBottom: width * 0.05,
    textAlign: 'center',
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.02,
    alignItems: 'center',
    flex: 1,
    marginHorizontal: width * 0.01,
  },
  cancelButton: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  deleteButton: {
    backgroundColor: '#C2185B',
  },
  modalButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  deleteButtonText: {
    color: '#FFFFFF',
    fontSize: width * 0.04,
    fontWeight: '600',
  },

  






  
});

export default RequestmaintenanceHome;