import React, { useState, useEffect, useRef, useCallback, useMemo } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  ScrollView,
  KeyboardAvoidingView,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
// Firebase removed - using MongoDB only
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { getFilteredTeamMembers } from '../Utils/teamMemberUtils';
import { Dropdown } from 'react-native-element-dropdown';
import DropDownPicker from 'react-native-dropdown-picker';
import OfflineRequestMaintenanceHelper from './OfflineRequestMaintenanceHelper';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;
const DROPDOWN_HEIGHT = 2 * 44;
const MARGIN = 8;

// MongoDB API URLs
const MONGODB_REQUEST_MAINTENANCE_API_URL = 'https://api.titandrillingzm.com:6009';
const MONGODB_ADMIN_USERS_API_URL = 'https://api.titandrillingzm.com:6003';

// Reusing CustomInputField from CreateRequestMaintenance
const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  icon,
  onIconPress,
  editable = true,
  showSelectButton = false,
  onSelectPress,
  noBackground = false,
  multiline = false,
  tags = [],
  onRemoveTag,
  noBorder = false,
  ...props
}) => (
  <View style={inputStyles.container}>
    <View style={inputStyles.labelContainer}>
      <Text style={inputStyles.label}>
        {label}
        {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
      </Text>
      {showSelectButton && (
        <TouchableOpacity style={inputStyles.labelSelectButton} onPress={onSelectPress}>
          <Text style={inputStyles.selectText}>Select</Text>
          <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
        </TouchableOpacity>
      )}
    </View>
    <View style={inputStyles.inputContainer}>
      {tags.length > 0 && (
        <View style={inputStyles.tagsContainer}>
          {tags.map((item, index) => (
            <View key={index} style={inputStyles.tag}>
              <Text style={inputStyles.tagText}>{item.name}</Text>
              {onRemoveTag && (
                <TouchableOpacity
                  style={inputStyles.tagCloseButton}
                  onPress={() => onRemoveTag(item)}
                >
                  <View style={inputStyles.tagCloseCircle}>
                    <MaterialIcons name="close" size={width * 0.035} color="#fff" />
                  </View>
                </TouchableOpacity>
              )}
            </View>
          ))}
        </View>
      )}
      <TextInput
        style={[
          inputStyles.input,
          showError ? inputStyles.inputError : null,
          !editable && inputStyles.disabledInput,
          noBackground && inputStyles.noBackground,
          multiline && inputStyles.multilineInput,
          tags.length > 0 && inputStyles.inputWithTags,
          noBorder && inputStyles.noBorder,
        ]}
        value={value}
        onChangeText={onChangeText}
        editable={editable}
        placeholderTextColor="#888"
        multiline={multiline}
        {...props}
      />
      {icon && (
        <TouchableOpacity style={inputStyles.inputIcon} onPress={onIconPress}>
          {icon}
        </TouchableOpacity>
      )}
    </View>
    {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
  </View>
);

const RequestMApproveWO = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { requestId } = route.params || {}; // Get requestId from navigation params
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [workOrderTitle, setWorkOrderTitle] = useState('');
  const [memo, setMemo] = useState('');
  const [priority, setPriority] = useState(null);
  const [assignTo, setAssignTo] = useState([]);
  const [assignToModalVisible, setAssignToModalVisible] = useState(false);
  const [selectedAssignTo, setSelectedAssignTo] = useState([]);
  const [requestTypeModalVisible, setRequestTypeModalVisible] = useState(false);
  const [selectedRequestType, setSelectedRequestType] = useState(null);
  const [selectedEquipmentName, setSelectedEquipmentName] = useState('');
  const [searchQuery, setSearchQuery] = useState('');
  const [taskModalVisible, setTaskModalVisible] = useState(false);
  const [taskTitle, setTaskTitle] = useState('');
  const [taskDescription, setTaskDescription] = useState('');
  const [tasks, setTasks] = useState([]);
  const [requestNumber, setRequestNumber] = useState('');
  const [errors, setErrors] = useState({ workOrderTitle: '', taskTitle: '' });
  const [userId, setUserId] = useState(null);
  const [taskDropdownVisible, setTaskDropdownVisible] = useState(false);
  const [taskDropdownPosition, setTaskDropdownPosition] = useState({ x: 0, y: 0 });
  const [selectedTask, setSelectedTask] = useState(null);
  const [approveModalVisible, setApproveModalVisible] = useState(false);
  const [approveSuccessModalVisible, setApproveSuccessModalVisible] = useState(false);
  const [rejectSuccessModalVisible, setRejectSuccessModalVisible] = useState(false);
  const [note, setNote] = useState('');
  const [requestStatus, setRequestStatus] = useState(null); // To store the current status of the request
  const [country, setCountry] = useState(null);
  const [project, setProject] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [teamMembers, setTeamMembers] = useState([]);
  const [allUsers, setAllUsers] = useState([]);
  const [loadingTeamMembers, setLoadingTeamMembers] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);
  const [hasMoreUsers, setHasMoreUsers] = useState(true);
  const [loadingMoreUsers, setLoadingMoreUsers] = useState(false);
  const [userPermissions, setUserPermissions] = useState([]);
  const [adminProfile, setAdminProfile] = useState(null);
  const [requestNotFound, setRequestNotFound] = useState(false);

  // Add per-task assign state
  const [taskAssignees, setTaskAssignees] = useState([]);
  const [taskAssignToModalVisible, setTaskAssignToModalVisible] = useState(false);
  const [taskAssignSearchQuery, setTaskAssignSearchQuery] = useState('');
  // Add per-task note state
  const [taskNote, setTaskNote] = useState('');

  // Country and Project state
  const [countryItems, setCountryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);
  const [projectItems, setProjectItems] = useState([]);

  const requestTypes = [
    { id: '1', name: 'General' },
    { id: '2', name: 'Preventive' },
    { id: '3', name: 'Breakdown' },
    { id: '4', name: 'Corrective' },
    { id: '5', name: 'Safety' },
    { id: '6', name: 'Upgrade' },
    { id: '7', name: 'Cleaning' },
    { id: '8', name: 'Testing' },
  ];

  // Removed hardcoded assignToOptions - now using filtered team members from MongoDB

  const priorityOptions = [
    { label: 'Low', image: require('../../../Images/adminhome/inspreport/Priority1.png') },
    { label: 'Medium', image: require('../../../Images/adminhome/inspreport/Priority2.png') },
    { label: 'High', image: require('../../../Images/adminhome/inspreport/Priority3.png') },
    { label: 'Emergency', image: require('../../../Images/adminhome/inspreport/Priority4.png') },
  ];

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const retryLoadRequest = async () => {
    console.log('🔄 Retrying to load request...');
    setRequestNotFound(false);
    setSyncModalVisible(true);
    
    try {
      const isOnline = await checkNetworkStatus();
      let request = null;
      
      if (isOnline) {
        try {
          // Try the direct endpoint first
          const response = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance/${requestId}`);
          if (response.ok) {
            const result = await response.json();
            if (result.success && result.data) {
              request = result.data;
              console.log('✅ Request found via direct endpoint on retry');
            }
          }
          
          // If not found, try the list endpoint
          if (!request) {
            console.log('🔄 Trying list endpoint on retry...');
            const listResponse = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance`);
            if (listResponse.ok) {
              const listResult = await listResponse.json();
              if (listResult.success && listResult.data && Array.isArray(listResult.data)) {
                request = listResult.data.find(req => 
                  req.id === requestId || 
                  req._id === requestId ||
                  req.requestNumber === requestId
                );
                if (request) {
                  console.log('✅ Request found via list endpoint on retry');
                } else {
                  console.log('❌ Request still not found in list endpoint on retry');
                  console.log('🔍 Available request IDs:', listResult.data.map(req => ({ id: req.id, _id: req._id, requestNumber: req.requestNumber })));
                }
              }
            }
          }
        } catch (error) {
          console.error('❌ Retry fetch error:', error);
        }
      }
      
      if (request) {
        console.log('✅ Request found on retry:', request.requestNumber);
        setWorkOrderTitle(request.workOrderTitle || '');
        setMemo(request.description || request.memo || '');
        setPriority(request.priority || null);
        setAssignTo(request.assignTo || []);
        setSelectedAssignTo(request.assignTo || []);
        setSelectedRequestType(request.requestType || null);
        setSelectedEquipmentName(request.equipment?.equipmentName || '');
        setTasks(request.tasks || []);
        setRequestNumber(request.requestNumber || '');
        setNote(request.note || '');
        setRequestStatus(request.status || null);
        setCountry(request.country || null);
        setProject(request.project || null);
        setSelectedEquipment(request.equipment || null);
        setRequestNotFound(false);
        showToast('success', 'Success', 'Request loaded successfully!');
      } else {
        console.log('❌ Request still not found on retry');
        showToast('error', 'Error', 'Request still not found. Please check the request ID.');
        setRequestNotFound(true);
      }
    } catch (error) {
      console.error('❌ Retry error:', error);
      showToast('error', 'Error', 'Failed to retry loading request.');
      setRequestNotFound(true);
    } finally {
      setSyncModalVisible(false);
    }
  };

  // Fetch user ID and request data from MongoDB
  useEffect(() => {
    const loadUserIdAndRequest = async () => {
      try {
        console.log('🔄 Loading user ID and request data...');
        console.log('📋 Request ID:', requestId);
        
        const uid = await AsyncStorage.getItem('userUid');
        setUserId(uid);
        
        if (!requestId) {
          console.error('❌ No request ID provided');
          showToast('error', 'Error', 'No request ID provided.');
          navigation.goBack();
          return;
        }

        setSyncModalVisible(true);
        const isOnline = await checkNetworkStatus();
        let request = null;
        
        console.log('🌐 Online status:', isOnline);
        
        if (isOnline) {
          try {
            console.log('🔄 Fetching from MongoDB...');
            // Try the new direct endpoint first
            const response = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance/${requestId}`);
            console.log('📡 MongoDB response status:', response.status);
            
            if (response.ok) {
              const result = await response.json();
              console.log('📊 MongoDB result:', {
                success: result.success,
                hasData: !!result.data,
                dataKeys: result.data ? Object.keys(result.data) : 'No data'
              });
              
              if (result.success && result.data) {
                request = result.data;
                console.log('✅ Request loaded from MongoDB direct endpoint:', request.requestNumber);
                
                // Save to local storage for offline
                const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
                let requests = storedRequests ? JSON.parse(storedRequests) : [];
                requests = requests.filter(req => req.id !== requestId);
                requests.push(request);
                await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(requests));
              } else {
                console.log('❌ Direct endpoint failed, trying list endpoint...');
                
                // Try the list endpoint and filter by ID
                const listResponse = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance`);
                if (listResponse.ok) {
                  const listResult = await listResponse.json();
                  console.log('📋 List API response:', {
                    success: listResult.success,
                    count: listResult.count,
                    dataLength: listResult.data?.length
                  });
                  
                  if (listResult.success && listResult.data && Array.isArray(listResult.data)) {
                    const foundRequest = listResult.data.find(req => 
                      req.id === requestId || 
                      req._id === requestId ||
                      req.requestNumber === requestId
                    );
                    if (foundRequest) {
                      request = foundRequest;
                      console.log('✅ Request found in list API:', request.requestNumber);
                      
                      // Save to local storage
                      const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
                      let requests = storedRequests ? JSON.parse(storedRequests) : [];
                      requests = requests.filter(req => req.id !== requestId);
                      requests.push(request);
                      await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(requests));
                    } else {
                      console.log('❌ Request not found in list API either');
                      console.log('🔍 Available request IDs:', listResult.data.map(req => ({ id: req.id, _id: req._id, requestNumber: req.requestNumber })));
                    }
                  }
                } else {
                  console.log('❌ List API also failed:', listResponse.status);
                }
              }
            } else {
              console.log('❌ Direct endpoint failed:', response.status);
              const errorText = await response.text();
              console.log('❌ Error response:', errorText);
              
              // Try list endpoint as fallback
              console.log('🔄 Trying list endpoint as fallback...');
              const listResponse = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance`);
              if (listResponse.ok) {
                const listResult = await listResponse.json();
                if (listResult.success && listResult.data && Array.isArray(listResult.data)) {
                  const foundRequest = listResult.data.find(req => 
                    req.id === requestId || 
                    req._id === requestId ||
                    req.requestNumber === requestId
                  );
                  if (foundRequest) {
                    request = foundRequest;
                    console.log('✅ Request found in list API fallback:', request.requestNumber);
                  }
                }
              }
            }
          } catch (mongoError) {
            console.error('❌ MongoDB fetch error:', mongoError);
            // Fallback to local storage
            const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
            const requests = storedRequests ? JSON.parse(storedRequests) : [];
            request = requests.find(req => req.id === requestId);
            console.log('📱 Fallback to local storage, found:', !!request);
          }
        } else {
          console.log('📱 Offline mode - using local storage');
          const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
          const requests = storedRequests ? JSON.parse(storedRequests) : [];
          request = requests.find(req => req.id === requestId);
          console.log('📱 Local storage result:', !!request);
          showToast('info', 'Offline', 'Using offline data. Will sync when online.');
        }
        
        if (request) {
          console.log('✅ Setting request data:', {
            workOrderTitle: request.workOrderTitle,
            requestNumber: request.requestNumber,
            status: request.status
          });
          
          setWorkOrderTitle(request.workOrderTitle || '');
          setMemo(request.description || request.memo || '');
          setPriority(request.priority || null);
          setAssignTo(request.assignTo || []);
          setSelectedAssignTo(request.assignTo || []);
          setSelectedRequestType(request.requestType || null);
          setSelectedEquipmentName(request.equipment?.equipmentName || '');
          setTasks(request.tasks || []);
          setRequestNumber(request.requestNumber || '');
          setNote(request.note || '');
          setRequestStatus(request.status || null);
          setCountry(request.country || null);
          setProject(request.project || null);
          setSelectedEquipment(request.equipment || null);
          
          console.log('✅ Request data loaded successfully');
        } else {
          console.error('❌ Request not found in any source');
          console.log('🔍 Debug info:', {
            requestId,
            isOnline,
            mongoResponse: 'Check logs above',
            localStorageCount: 'Check logs above'
          });
          
          // Don't navigate back immediately, show error and let user retry
          showToast('error', 'Error', 'Maintenance request not found. Please check the request ID and try again.');
          
          // Set some default values to prevent crashes
          setWorkOrderTitle('Request Not Found');
          setRequestNumber(requestId || 'Unknown');
          setRequestStatus('Not Found');
          setRequestNotFound(true);
          
          // Only navigate back after a delay to let user see the error
          setTimeout(() => {
            navigation.goBack();
          }, 5000);
          return;
        }
        
        setSyncModalVisible(false);
      } catch (error) {
        console.error('❌ Error in loadUserIdAndRequest:', error);
        showToast('error', 'Error', 'Failed to load request data.');
        setSyncModalVisible(false);
        navigation.goBack();
      }
    };
    
    if (requestId) {
      loadUserIdAndRequest();
    } else {
      console.error('❌ No requestId in useEffect');
      showToast('error', 'Error', 'No request ID provided.');
      navigation.goBack();
    }
  }, [navigation, requestId]);

  // Start rotation animation for sync modal
  useEffect(() => {
    if (syncModalVisible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [syncModalVisible, rotateAnim]);

  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    setTimeout(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }, 2000);
  };

  const handlePrioritySelect = (selectedPriority) => {
    setPriority(selectedPriority);
  };

  const handleRequestTypeSelect = (requestType) => {
    setSelectedRequestType(requestType.name);
    setRequestTypeModalVisible(false);
  };

  const handleEquipmentSelect = () => {
    navigation.navigate('SelectAllInOneEquipment', {
      selectedEquipment,
      onSelect: (equipment) => {
        setSelectedEquipment(equipment);
      },
    });
  };

  const handleAssignToSelect = (item) => {
    const isSelected = selectedAssignTo.some(selected => selected.id === item.id);
    if (isSelected) {
      setSelectedAssignTo(selectedAssignTo.filter(selected => selected.id !== item.id));
    } else {
      setSelectedAssignTo([...selectedAssignTo, item]);
    }
  };

  const handleSaveAssignTo = () => {
    setAssignTo(selectedAssignTo);
    setAssignToModalVisible(false);
  };

  const handleRemoveAssignTo = (itemToRemove) => {
    setAssignTo(assignTo.filter(item => item.id !== itemToRemove.id));
    setSelectedAssignTo(selectedAssignTo.filter(item => item.id !== itemToRemove.id));
  };

  const handleAddTask = () => {
    if (!taskTitle.trim()) {
      setErrors({ ...errors, taskTitle: 'Task Title is required' });
      return;
    }
    setTasks([
      ...tasks,
      {
        id: `${tasks.length + 1}`,
        title: taskTitle,
        description: taskDescription,
        assignees: taskAssignees,
        note: taskNote,
      },
    ]);
    setTaskTitle('');
    setTaskDescription('');
    setTaskAssignees([]);
    setTaskNote('');
    setErrors({ ...errors, taskTitle: '' });
    setTaskModalVisible(false);
  };

  const handleEditTask = (task) => {
    setTaskTitle(task.title);
    setTaskDescription(task.description);
    setTaskAssignees(task.assignees || []);
    setTaskNote(task.note || '');
    setSelectedTask(task);
    setTaskModalVisible(true);
  };

  const handleSaveEditedTask = () => {
    if (!taskTitle.trim()) {
      setErrors({ ...errors, taskTitle: 'Task Title is required' });
      return;
    }
    setTasks(tasks.map(task =>
      task.id === selectedTask.id
        ? { ...task, title: taskTitle, description: taskDescription, assignees: taskAssignees, note: taskNote }
        : task
    ));
    setTaskTitle('');
    setTaskDescription('');
    setTaskAssignees([]);
    setTaskNote('');
    setErrors({ ...errors, taskTitle: '' });
    setTaskModalVisible(false);
    setSelectedTask(null);
  };

  const handleDeleteTask = (taskId) => {
    setTasks(tasks.filter(task => task.id !== taskId));
    setTaskDropdownVisible(false);
  };

  const handleApproveRequest = async () => {
    if (!userId) {
      showToast('error', 'Error', 'User not logged in. Please log in again.');
      navigation.replace('AdminLogin');
      return;
    }

    if (!workOrderTitle.trim()) {
      setErrors({ ...errors, workOrderTitle: 'Work Order Title is required' });
      showToast('error', 'Error', 'Work Order Title is required');
      return;
    }

    setApproveModalVisible(false);
    setSyncModalVisible(true);

    try {
      const isOnline = await checkNetworkStatus();
      const updatedRequest = {
        id: requestId,
        adminUid: userId,
        requestNumber,
        workOrderTitle,
        description: memo,
        priority,
        country,
        project,
        assignTo,
        requestType: selectedRequestType,
        equipment: selectedEquipment || (selectedEquipmentName ? { equipmentName: selectedEquipmentName } : null),
        tasks,
        status: 'Approved',
        note: note.trim() || 'No note provided.',
        updatedAt: new Date().toISOString(),
        userId,
        isArchived: false,
      };

      // Update AsyncStorage (local cache) - Update existing or add new
      const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
      let requests = storedRequests ? JSON.parse(storedRequests) : [];
      requests = requests.map(req => req.id === requestId ? updatedRequest : req);
      if (!requests.find(req => req.id === requestId)) {
        requests.push(updatedRequest);
      }
      await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(requests));

      if (isOnline) {
        try {
          // Update in MongoDB
          const response = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance/${requestId}`, {
            method: 'PUT',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(updatedRequest),
          });

          if (!response.ok) {
            throw new Error(`MongoDB update failed: ${response.status}`);
          }

          const result = await response.json();
          
          if (result.success) {
            console.log('✅ Maintenance request approved successfully in MongoDB!');
            showToast('success', 'Success', 'Maintenance request approved and updated successfully!');
          } else {
            throw new Error(result.error || 'Failed to update in MongoDB');
          }
        } catch (mongoError) {
          console.error('❌ MongoDB update failed:', mongoError);
          // Queue offline update for later sync
          await OfflineRequestMaintenanceHelper.saveOfflineUpdate({
            id: requestId,
            action: 'approve',
            updatedRequest,
          });
          showToast('warning', 'Saved Locally', 'Request approved locally. Will sync when connection is stable.');
        }
      } else {
        // Queue offline update for later sync
        await OfflineRequestMaintenanceHelper.saveOfflineUpdate({
          id: requestId,
          action: 'approve',
          updatedRequest,
        });
        showToast('info', 'Offline', 'Request approved locally. Will sync when online.');
      }
    } catch (error) {
      console.error('Error approving maintenance request:', error);
      let errorMessage = 'Failed to approve maintenance request.';
      if (!(await checkNetworkStatus())) {
        errorMessage = 'Offline mode. Request approved locally and will sync when online.';
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      setSyncModalVisible(false);
      setTimeout(() => {
        setApproveSuccessModalVisible(true);
      }, 100); // Delay to ensure modal transition is smooth
    }
  };

  const handleRejectRequest = async () => {
    if (!userId) {
      showToast('error', 'Error', 'User not logged in. Please log in again.');
      navigation.replace('AdminLogin');
      return;
    }

    if (!note.trim()) {
      showToast('error', 'Error', 'Please provide a note explaining the rejection.');
      return;
    }

    if (!workOrderTitle.trim()) {
      setErrors({ ...errors, workOrderTitle: 'Work Order Title is required' });
      showToast('error', 'Error', 'Work Order Title is required');
      return;
    }

    setApproveModalVisible(false);
    setSyncModalVisible(true);

    try {
      const isOnline = await checkNetworkStatus();
      const updatedRequest = {
        id: requestId,
        adminUid: userId,
        requestNumber,
        workOrderTitle,
        description: memo,
        priority,
        country,
        project,
        assignTo,
        requestType: selectedRequestType,
        equipment: selectedEquipment || (selectedEquipmentName ? { equipmentName: selectedEquipmentName } : null),
        tasks,
        status: 'Rejected',
        note: note.trim(),
        updatedAt: new Date().toISOString(),
        userId,
        isArchived: false,
      };

      // Update AsyncStorage (local cache) - Update existing or add new
      const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
      let requests = storedRequests ? JSON.parse(storedRequests) : [];
      requests = requests.map(req => req.id === requestId ? updatedRequest : req);
      if (!requests.find(req => req.id === requestId)) {
        requests.push(updatedRequest);
      }
      await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(requests));

      if (isOnline) {
        try {
          // Update in MongoDB
          const response = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance/${requestId}`, {
            method: 'PUT',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(updatedRequest),
          });

          if (!response.ok) {
            throw new Error(`MongoDB update failed: ${response.status}`);
          }

          const result = await response.json();
          
          if (result.success) {
            console.log('✅ Maintenance request rejected successfully in MongoDB!');
            showToast('success', 'Success', 'Maintenance request rejected and updated successfully!');
          } else {
            throw new Error(result.error || 'Failed to update in MongoDB');
          }
        } catch (mongoError) {
          console.error('❌ MongoDB update failed:', mongoError);
          // Queue offline update for later sync
          await OfflineRequestMaintenanceHelper.saveOfflineUpdate({
            id: requestId,
            action: 'reject',
            updatedRequest,
          });
          showToast('warning', 'Saved Locally', 'Request rejected locally. Will sync when connection is stable.');
        }
      } else {
        // Queue offline update for later sync
        await OfflineRequestMaintenanceHelper.saveOfflineUpdate({
          id: requestId,
          action: 'reject',
          updatedRequest,
        });
        showToast('info', 'Offline', 'Request rejected locally. Will sync when online.');
      }
    } catch (error) {
      console.error('Error rejecting maintenance request:', error);
      let errorMessage = 'Failed to reject maintenance request.';
      if (!(await checkNetworkStatus())) {
        errorMessage = 'Offline mode. Request rejected locally and will sync when online.';
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      setSyncModalVisible(false);
      setTimeout(() => {
        setRejectSuccessModalVisible(true);
      }, 100); // Delay to ensure modal transition is smooth
    }
  };

  const handleApproveSuccessModalClose = () => {
    setApproveSuccessModalVisible(false);
    navigation.reset({
          index: 1,
          routes: [
            { name: 'MainApp' },
            { name: 'RequestmaintenanceHome' },
          ],
        });
  };

  const handleRejectSuccessModalClose = () => {
    setRejectSuccessModalVisible(false);
    navigation.reset({
          index: 1,
          routes: [
            { name: 'MainApp' },
            { name: 'RequestmaintenanceHome' },
          ],
        });
  };

  // Fetch all team members using the EXACT same approach as CreateRequestMaintenance.js
  const fetchAllTeamMembers = async () => {
    try {
      setLoadingTeamMembers(true);
      console.log('🔍 Fetching team members using CreateRequestMaintenance method...');
      
      try {
        // Use the EXACT same API call as CreateRequestMaintenance.js
        const teamResponse = await fetch(`${MONGODB_ADMIN_USERS_API_URL}/users/team-management?page=1&limit=8000`);
        
        if (!teamResponse.ok) {
          throw new Error(`Failed to fetch team members: ${teamResponse.status}`);
        }
        
        const result = await teamResponse.json();
        console.log('🔍 Team Management API Response:', {
          success: result.success,
          count: result.count,
          totalUsers: result.totalUsers,
          dataLength: result.data?.length,
          firstUser: result.data?.[0] ? {
            id: result.data[0]._id,
            name: result.data[0].name,
            employeeNumber: result.data[0].employeeNumber
          } : 'No users'
        });
        
        if (result.success && result.data && Array.isArray(result.data)) {
          console.log('✅ Processing team management data:', result.data.length);
          
          // Use the EXACT same mapping as CreateRequestMaintenance.js
          const allMembersArray = result.data.map((user) => ({
            id: user._id,
            fullName: user.name || 'N/A',
            employeeNumber: user.employeeNumber || 'N/A',
            email: user.email || user.employeeNumber || 'N/A',
            projects: user.projects || [],
            countries: user.countries || [],
            role: user.role || 'Operator',
            isBlocked: user.isBlocked || false,
            password: user.password || user.employeeNumber || user.email || 'N/A',
            profilePic: user.profilePic || null,
            // Additional fields for consistency
            driverLicenseNumber: user.driverLicenseNumber || '',
            driverLicenseCode: user.driverLicenseCode || '',
            driverLicenseExpiry: user.driverLicenseExpiry || '',
            dateCompletedFirstAid: user.dateCompletedFirstAid || '',
            expiryDateFirstAid: user.expiryDateFirstAid || '',
            fireFightingDateAttended: user.fireFightingDateAttended || '',
            blastingLicenseDateIssued: user.blastingLicenseDateIssued || '',
          }));
          
          console.log(`✅ Successfully loaded ${allMembersArray.length} team members from team-management API`);
          setAllUsers(allMembersArray);
          setLoadingTeamMembers(false);
          
          // Cache the data
          try {
            await AsyncStorage.setItem('allTeamMembers', JSON.stringify(allMembersArray));
            console.log('💾 Cached team members data');
          } catch (cacheError) {
            console.error('Error caching team members:', cacheError);
          }
          
          // Load users into teamMembers
          loadPaginatedTeamMembers(1, false);
          
        } else {
          console.log('❌ No valid user data found in team-management response');
          setAllUsers([]);
          setLoadingTeamMembers(false);
          showToast('error', 'Error', 'No users found in team-management API response');
        }
      } catch (fetchError) {
        console.error('❌ Error fetching team members from team-management:', fetchError);
        setAllUsers([]);
        setLoadingTeamMembers(false);
        showToast('error', 'Error', 'Failed to load team members from team-management API');
      }
    } catch (error) {
      console.error('❌ Error in fetchAllTeamMembers:', error);
      setAllUsers([]);
      setLoadingTeamMembers(false);
      showToast('error', 'Error', 'An unexpected error occurred');
    }
  };

  // Load all team members (no pagination for now)
  const loadPaginatedTeamMembers = (page = 1, appendData = false) => {
    console.log(`📄 Loading users, allUsers length: ${allUsers.length}`);
    
    if (allUsers.length === 0) {
      console.log('No users available in allUsers');
      setTeamMembers([]);
      return;
    }

    // Show all users at once for now (we can add pagination later if needed)
    console.log(`📄 Showing all ${allUsers.length} users`);
    setTeamMembers([...allUsers]); // Create a new array to trigger re-render
    setHasMoreUsers(false);
    setCurrentPage(1);
    console.log('📄 Users loaded successfully into teamMembers');
  };

  // Memoized filtered data for better performance
  const filteredTeamMembers = useMemo(() => {
    const filtered = (Array.isArray(teamMembers) ? teamMembers : []).filter(member => {
      if (!member) return false;
      
      // Safe search query handling
      const searchLower = (searchQuery || '').toLowerCase();
      const fullName = member.fullName || '';
      const employeeNumber = member.employeeNumber || '';
      const role = member.role || '';
      
      // If no search query, show all members
      if (!searchQuery || searchQuery.trim() === '') {
        return true;
      }
      
      return (
        fullName.toLowerCase().includes(searchLower) ||
        employeeNumber.toLowerCase().includes(searchLower) ||
        role.toLowerCase().includes(searchLower)
      );
    });
    
    console.log('🔍 Filtered team members:', {
      totalTeamMembers: teamMembers.length,
      searchQuery: searchQuery || 'empty',
      filteredCount: filtered.length,
      firstFiltered: filtered[0]
    });
    return filtered;
  }, [teamMembers, searchQuery]);

  // Memoized render item for better performance
  const renderTeamMemberItem = useCallback(({ item }) => {
    if (!item) return null;
    
    // Safe property access with fallbacks
    const fullName = item.fullName || item.name || 'Unknown User';
    const employeeNumber = item.employeeNumber || 'N/A';
    const role = item.role || 'N/A';
    const profilePic = item.profilePic || null;
    
    const isSelected = assignTo.some(selected => selected && selected.id === item.id);
    
    return (
      <TouchableOpacity
        style={{ 
          flexDirection: 'row', 
          alignItems: 'center', 
          paddingVertical: 12, 
          paddingHorizontal: 16,
          borderBottomWidth: 1, 
          borderBottomColor: '#eee', 
          backgroundColor: isSelected ? '#e6f0ff' : '#fff' 
        }}
        onPress={() => handleAssignToSelect(item)}
        activeOpacity={0.7}
      >
        {profilePic && profilePic.url ? (
          <Image 
            source={{ uri: profilePic.url }} 
            style={{ width: 40, height: 40, borderRadius: 20, marginRight: 12 }} 
            resizeMode="cover"
          />
        ) : (
          <View style={{ 
            width: 40, 
            height: 40, 
            borderRadius: 20, 
            backgroundColor: '#ddd', 
            marginRight: 12, 
            alignItems: 'center', 
            justifyContent: 'center' 
          }}>
            <MaterialIcons name="person" size={24} color="#888" />
          </View>
        )}
        <View style={{ flex: 1 }}>
          <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 16, marginBottom: 2 }}>
            {fullName}
          </Text>
          <Text style={{ color: '#333', fontSize: 13 }}>
            Emp#: {employeeNumber} | Role: {role}
          </Text>
        </View>
        {isSelected && (
          <MaterialIcons name="check" size={24} color="#015185" style={{ marginLeft: 8 }} />
        )}
      </TouchableOpacity>
    );
  }, [assignTo, handleAssignToSelect]);

  // Fetch team members when modal opens
  useEffect(() => {
    if (assignToModalVisible) {
      console.log('🔄 Assign To modal opened');
      console.log('📊 Current allUsers length:', allUsers.length);
      console.log('📊 Current teamMembers length:', teamMembers.length);
      
      // Reset state
      setSearchQuery('');
      
      // Always fetch fresh data when modal opens
      console.log('🔄 Modal opened - fetching fresh users...');
      fetchAllTeamMembers();
    }
  }, [assignToModalVisible]);

  // Auto-load users when allUsers changes
  useEffect(() => {
    if (allUsers.length > 0 && (assignToModalVisible || taskAssignToModalVisible)) {
      console.log('🔄 allUsers updated, loading into teamMembers...');
      loadPaginatedTeamMembers(1, false);
    }
  }, [allUsers, assignToModalVisible, taskAssignToModalVisible]);

  // Fetch team members when task assign modal opens
  useEffect(() => {
    if (taskAssignToModalVisible) {
      console.log('🔄 Task Assign To modal opened');
      console.log('📊 Current allUsers length:', allUsers.length);
      console.log('📊 Current teamMembers length:', teamMembers.length);
      
      // Reset state
      setTaskAssignSearchQuery('');
      
      // Always fetch fresh data when modal opens
      console.log('🔄 Task modal opened - fetching fresh users...');
      fetchAllTeamMembers();
    }
  }, [taskAssignToModalVisible]);

  // Handle project items based on country selection
  useEffect(() => {
    if (country === 'Zambia') {
      setProjectItems([
        { label: 'Kobold', value: 'Kobold' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kimiteto', value: 'Kimiteto' },
        { label: 'Mimosa', value: 'Mimosa' },
      ]);
    } else if (country === 'DRC') {
      setProjectItems([
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
      ]);
    } else {
      setProjectItems([]);
    }
  }, [country]);

  return (
    <KeyboardAvoidingView
      style={styles.container}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Update Request</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity
              style={[styles.iconButton, { marginLeft: width * 0.02 }]}
              onPress={() => setApproveModalVisible(true)}
            >
              <Text style={styles.createButtonText}>Approve</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <ScrollView
        style={styles.mainContent}
        keyboardShouldPersistTaps="handled"
        contentContainerStyle={{ paddingBottom: width * 0.1 }}
      >
        <View style={styles.formGroup}>
          <View style={styles.requestNumberRow}>
            <Text style={styles.requestNumberLabel}>Request #</Text>
            <Text style={styles.requestNumberValue}>{requestNumber}</Text>
          </View>
          {requestNotFound && (
            <View style={styles.retryContainer}>
              <Text style={styles.retryText}>Request not found. Please try again.</Text>
              <TouchableOpacity style={styles.retryButton} onPress={retryLoadRequest}>
                <MaterialIcons name="refresh" size={20} color="#fff" />
                <Text style={styles.retryButtonText}>Retry Load</Text>
              </TouchableOpacity>
            </View>
          )}
        </View>

        <View style={styles.formGroup}>
          <CustomInputField
            label="Work Order Title"
            value={workOrderTitle}
            onChangeText={(text) => {
              setWorkOrderTitle(text);
              setErrors({ ...errors, workOrderTitle: text.trim() ? '' : 'Work Order Title is required' });
            }}
            isRequired
            showError={!!errors.workOrderTitle}
            errorMessage={errors.workOrderTitle}
            placeholder="Write Request Title"
          />
        </View>

        <View style={styles.formGroup}>
          <CustomInputField
            label="Memo"
            value={memo}
            onChangeText={setMemo}
            multiline
            placeholder="Write Memo"
          />
        </View>

        <View style={styles.formGroup}>
          <Text style={styles.label}>Priority</Text>
          <View style={styles.priorityContainer}>
            {priorityOptions.map((option) => {
              const isSelected = priority === option.label;
              return (
                <View key={option.label} style={styles.priorityItem}>
                  <TouchableOpacity
                    style={[
                      styles.priorityButton,
                      isSelected ? styles.priorityButtonSelected : styles.priorityButtonUnselected,
                    ]}
                    onPress={() => handlePrioritySelect(option.label)}
                  >
                    <Image source={option.image} style={styles.priorityImage} />
                    <Text style={isSelected ? styles.priorityTextSelected : styles.priorityText}>
                      {option.label}
                    </Text>
                  </TouchableOpacity>
                </View>
              );
            })}
          </View>
        </View>

        <View style={styles.formGroup}>
          <TouchableOpacity style={styles.selectRow} onPress={handleEquipmentSelect}>
            <Text style={styles.selectLabel}>Equipment</Text>
            <View style={{ flexDirection: 'row', alignItems: 'center' }}>
              <Text style={styles.selectText}>Select</Text>
              <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
            </View>
          </TouchableOpacity>
          {selectedEquipment ? (
            <View style={{backgroundColor:'#f0f0f0', borderRadius:8, padding:10, marginTop:6}}>
              <Text style={{fontWeight:'bold', color:'#015185', fontSize:16}}>{selectedEquipment.equipmentName || 'N/A'}</Text>
              <Text style={{color:'#333'}}>Number: {selectedEquipment.equipmentNumber || 'N/A'}</Text>
              <Text style={{color:'#333'}}>Model: {selectedEquipment.model || 'N/A'}</Text>
              <Text style={{color:'#333'}}>Meter: {selectedEquipment.meterReading} {selectedEquipment.meterUnit}</Text>
              <Text style={{color:'#333'}}>Status: {selectedEquipment.status || 'N/A'}</Text>
            </View>
          ) : (
            <Text style={styles.selectedText}>No Equipment Selected</Text>
          )}
        </View>

        <View style={styles.formGroup}>
          <TouchableOpacity style={styles.selectRow} onPress={() => setAssignToModalVisible(true)}>
            <Text style={styles.selectLabel}>Assign to</Text>
            <View style={styles.selectRight}>
              <Text style={styles.selectText}>Select</Text>
              <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
            </View>
          </TouchableOpacity>
          {assignTo.length > 0 ? (
            <Text style={styles.selectedText}>
              Assigned to: {assignTo.map(item => item.name).join(', ')}
            </Text>
          ) : (
            <Text style={styles.selectedText}>No Assignees Selected</Text>
          )}
        </View>

        <View style={styles.formGroup}>
          <TouchableOpacity style={styles.selectRow} onPress={() => setRequestTypeModalVisible(true)}>
            <Text style={styles.selectLabel}>Request Type</Text>
            <View style={styles.selectRight}>
              <Text style={styles.selectText}>Select</Text>
              <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
            </View>
          </TouchableOpacity>
          {selectedRequestType ? (
            <Text style={styles.selectedText}>
              Selected Request Type: {selectedRequestType}
            </Text>
          ) : (
            <Text style={styles.selectedText}>No Request Type Selected</Text>
          )}
        </View>

        <View style={styles.formGroup}>
          <View style={styles.row}>
            <Text style={styles.label}>Task</Text>
            <TouchableOpacity
              style={styles.addTaskButton}
              onPress={() => setTaskModalVisible(true)}
            >
              <Text style={styles.addTaskButtonText}>Add Task</Text>
            </TouchableOpacity>
          </View>
          {tasks.length > 0 && (
            <View style={styles.taskListContainer}>
              {tasks.map((task) => (
                <View key={task.id} style={styles.taskCard}>
                  <View style={styles.taskCardHeader}>
                    <Text style={styles.taskCardTitle}>{task.title}</Text>
                    <View style={styles.taskCardStatusPill}>
                      <Text style={styles.taskCardStatusText}>Initiated</Text>
                    </View>
                    <TouchableOpacity
                      style={styles.taskCardMenuButton}
                      onPress={(event) => {
                        setSelectedTask(task);
                        const { pageX, pageY } = event.nativeEvent;
                        const screenHeight = Dimensions.get('window').height;
                        let top = pageY;
                        if (pageY + DROPDOWN_HEIGHT + MARGIN > screenHeight) {
                          top = pageY - DROPDOWN_HEIGHT - MARGIN;
                        }
                        setTaskDropdownPosition({ x: pageX - 120, y: top });
                        setTaskDropdownVisible(true);
                      }}
                    >
                      <MaterialIcons name="more-vert" size={22} color="#222" />
                    </TouchableOpacity>
                  </View>
                  <Text style={styles.taskCardType}>{task.description || 'Repair'}</Text>
                  {/* Show assigned users for this task */}
                  {task.assignees && task.assignees.length > 0 && (
                    <ScrollView
                      horizontal={true}
                      showsHorizontalScrollIndicator={false}
                      contentContainerStyle={{ flexDirection: 'row', alignItems: 'center', marginTop: 4 }}
                    >
                      {task.assignees.map((item, index) => (
                        <View key={index} style={{
                          flexDirection: 'row',
                          alignItems: 'center',
                          backgroundColor: '#e6f0ff',
                          borderRadius: 8,
                          paddingVertical: 4,
                          paddingHorizontal: 8,
                          marginRight: 6,
                          marginBottom: 2,
                        }}>
                          {item.profilePic ? (
                            <Image
                              source={{ uri: item.profilePic.url }}
                              style={{ width: 24, height: 24, borderRadius: 12, marginRight: 6 }}
                            />
                          ) : (
                            <View style={{
                              width: 24,
                              height: 24,
                              borderRadius: 12,
                              backgroundColor: '#ddd',
                              marginRight: 6,
                              alignItems: 'center',
                              justifyContent: 'center',
                            }}>
                              <MaterialIcons name="person" size={14} color="#888" />
                            </View>
                          )}
                          <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 12 }}>{item.fullName}</Text>
                        </View>
                      ))}
                    </ScrollView>
                  )}
                  <MaterialIcons name="person" size={22} color="#888" style={{ marginTop: 2 }} />
                </View>
              ))}
              <Modal
                visible={taskDropdownVisible}
                transparent
                animationType="fade"
                onRequestClose={() => setTaskDropdownVisible(false)}
              >
                <TouchableOpacity
                  style={styles.taskDropdownOverlay}
                  onPress={() => setTaskDropdownVisible(false)}
                  activeOpacity={1}
                >
                  <View
                    style={[
                      styles.taskDropdownContainer,
                      {
                        position: 'absolute',
                        top: taskDropdownPosition.y,
                        left: taskDropdownPosition.x,
                      },
                    ]}
                  >
                    <TouchableOpacity
                      style={styles.taskDropdownItem}
                      onPress={() => {
                        handleEditTask(selectedTask);
                        setTaskDropdownVisible(false);
                      }}
                    >
                      <MaterialIcons name="edit" size={20} color="#3481BC" style={{ marginRight: 8 }} />
                      <Text style={styles.taskDropdownText}>Edit</Text>
                    </TouchableOpacity>
                    <TouchableOpacity
                      style={styles.taskDropdownItem}
                      onPress={() => handleDeleteTask(selectedTask.id)}
                    >
                      <MaterialIcons name="delete" size={20} color="#C2185B" style={{ marginRight: 8 }} />
                      <Text style={styles.taskDropdownText}>Delete</Text>
                    </TouchableOpacity>
                  </View>
                </TouchableOpacity>
              </Modal>
            </View>
          )}
        </View>
      </ScrollView>

      {/* Sync Modal */}
      {/* <SyncModal
        visible={
          syncModalVisible &&
          !approveSuccessModalVisible &&
          !rejectSuccessModalVisible
        }
        message={
          approveSuccessModalVisible
            ? 'Approving Maintenance Request...'
            : rejectSuccessModalVisible
            ? 'Rejecting Maintenance Request...'
            : 'Updating Maintenance Request...'
        }
      /> */}

      {/* Approve/Reject Confirmation Modal */}
      <Modal
        visible={approveModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setApproveModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.noteModalContent}>
            <Text style={styles.noteModalTitle}>Add Note</Text>
            <View style={styles.noteInputWrapper}>
              <TextInput
                style={styles.noteInput}
                value={note}
                onChangeText={setNote}
                placeholder="Enter your note here..."
                placeholderTextColor="#888"
                multiline
                numberOfLines={5}
                textAlignVertical="top"
              />
              {[...Array(5)].map((_, i) => (
                <View
                  key={i}
                  style={[
                    styles.noteLine,
                    {
                      top: 24 + i * 24,
                    },
                  ]}
                />
              ))}
            </View>
            <View style={styles.noteModalButtonRow}>
              <TouchableOpacity
                onPress={handleRejectRequest}
                style={[styles.noteModalButton, styles.rejectButton]}
              >
                <Text style={styles.noteModalButtonText}>REJECT</Text>
              </TouchableOpacity>
              <TouchableOpacity
                onPress={() => setApproveModalVisible(false)}
                style={[styles.noteModalButton, styles.closeButton]}
              >
                <Text style={styles.noteModalButtonText}>CLOSE</Text>
              </TouchableOpacity>
              <TouchableOpacity
                onPress={handleApproveRequest}
                style={[styles.noteModalButton, styles.approveButton]}
              >
                <Text style={styles.noteModalButtonText}>APPROVE</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Approve Success Modal */}
      <Modal
        visible={approveSuccessModalVisible}
        transparent
        animationType="fade"
        onRequestClose={handleApproveSuccessModalClose}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalText}>Success</Text>
            <Text style={styles.successModalSubText}>
              Maintenance request approved successfully!
            </Text>
            <TouchableOpacity
              style={styles.successModalButton}
              onPress={handleApproveSuccessModalClose}
            >
              <Text style={styles.successModalButtonText}>OK</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {/* Reject Success Modal */}
      <Modal
        visible={rejectSuccessModalVisible}
        transparent
        animationType="fade"
        onRequestClose={handleRejectSuccessModalClose}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalText}>Success</Text>
            <Text style={styles.successModalSubText}>
              Maintenance request rejected successfully!
            </Text>
            <TouchableOpacity
              style={styles.successModalButton}
              onPress={handleRejectSuccessModalClose}
            >
              <Text style={styles.successModalButtonText}>OK</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={requestTypeModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setRequestTypeModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.requestTypeModalContent}>
            {requestTypes.map((item, idx) => {
              const isSelected = selectedRequestType === item.name;
              return (
                <React.Fragment key={item.id}>
                  <TouchableOpacity
                    style={styles.requestTypeRow}
                    onPress={() => setSelectedRequestType(item.name)}
                  >
                    <Text style={[styles.requestTypeLabel, isSelected && styles.requestTypeLabelSelected]}>{item.name}</Text>
                    <View style={styles.requestTypeRadioOuter}>
                      {isSelected ? <View style={styles.requestTypeRadioInner} /> : null}
                    </View>
                  </TouchableOpacity>
                  {idx !== requestTypes.length - 1 && <View style={styles.requestTypeDivider} />}
                </React.Fragment>
              );
            })}
            <View style={styles.requestTypeButtonRow}>
              <TouchableOpacity
                style={styles.requestTypeCloseButton}
                onPress={() => setRequestTypeModalVisible(false)}
              >
                <Text style={styles.requestTypeCloseButtonText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={assignToModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAssignToModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.assignModalContentImproved, { maxHeight: '80%' }]}>
            <Text style={styles.assignModalTitleImproved}>Assign To</Text>
            <Text style={{ fontSize: 12, color: '#666', textAlign: 'center', marginBottom: 10 }}>
              Displayed: {teamMembers.length} | Total: {allUsers.length} | Loading: {loadingTeamMembers ? 'Yes' : 'No'}
            </Text>
            <Text style={{ fontSize: 10, color: '#999', textAlign: 'center', marginBottom: 10 }}>
              Using Team Management API (same as Create Request)
            </Text>
            <View style={{ flexDirection: 'row', alignItems: 'center', marginBottom: 10 }}>
              <TextInput
                style={[styles.searchInput, { flex: 1, marginRight: 10 }]}
                placeholder="Search team members..."
                placeholderTextColor="#888"
                value={searchQuery || ''}
                onChangeText={(text) => setSearchQuery(text || '')}
              />
              <TouchableOpacity
                style={{ padding: 8, backgroundColor: '#4CAF50', borderRadius: 5 }}
                onPress={() => {
                  console.log('🔍 Search triggered...');
                  // The search is already handled by the TextInput onChangeText
                }}
              >
                <MaterialIcons name="search" size={16} color="#fff" />
              </TouchableOpacity>
            </View>
            <FlatList
              data={filteredTeamMembers}
              keyExtractor={(item, index) => item?.id || `team-member-${index}`}
              renderItem={renderTeamMemberItem}
              style={{ marginBottom: 10, maxHeight: 400 }}
              ListEmptyComponent={
                loadingTeamMembers ? (
                  <View style={{ padding: 20, alignItems: 'center' }}>
                    <Text style={styles.defaultText}>Loading team members...</Text>
                  </View>
                ) : allUsers.length === 0 ? (
                  <View style={{ padding: 20, alignItems: 'center' }}>
                    <Text style={styles.defaultText}>No team members available</Text>
                    <Text style={[styles.defaultText, { fontSize: 12, marginTop: 5 }]}>
                      Total users loaded: {allUsers.length}
                    </Text>
                    <TouchableOpacity
                      style={{ marginTop: 10, padding: 10, backgroundColor: '#015185', borderRadius: 5 }}
                      onPress={() => {
                        console.log('🔄 Manual retry triggered');
                        fetchAllTeamMembers();
                      }}
                    >
                      <Text style={{ color: '#fff' }}>Retry Load Users</Text>
                    </TouchableOpacity>
                  </View>
                ) : teamMembers.length === 0 ? (
                  <View style={{ padding: 20, alignItems: 'center' }}>
                    <Text style={styles.defaultText}>Data loaded but not displayed</Text>
                    <Text style={[styles.defaultText, { fontSize: 12, marginTop: 5 }]}>
                      All Users: {allUsers.length} | Team Members: {teamMembers.length}
                    </Text>
                    <TouchableOpacity
                      style={{ marginTop: 10, padding: 10, backgroundColor: '#4CAF50', borderRadius: 5 }}
                      onPress={() => {
                        console.log('🔄 Manual load triggered');
                        loadPaginatedTeamMembers(1, false);
                      }}
                    >
                      <Text style={{ color: '#fff' }}>Load Users Now</Text>
                    </TouchableOpacity>
                  </View>
                ) : (
                  <View style={{ padding: 20, alignItems: 'center' }}>
                    <Text style={styles.defaultText}>No team members found matching your search</Text>
                    <Text style={[styles.defaultText, { fontSize: 12, marginTop: 5 }]}>
                      Try clearing your search or check the console for errors
                    </Text>
                  </View>
                )
              }
              removeClippedSubviews={false}
              maxToRenderPerBatch={20}
              updateCellsBatchingPeriod={100}
              initialNumToRender={20}
              windowSize={10}
            />
            <View style={{ alignItems: 'center', marginTop: 10 }}>
              <TouchableOpacity
                style={styles.saveAssignButton}
                onPress={handleSaveAssignTo}
              >
                <Text style={styles.saveAssignButtonText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={taskModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setTaskModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.addTaskModalContent}>
            <Text style={styles.addTaskTitle}>{selectedTask ? 'Edit Task' : 'Add Task'}</Text>
            <View style={styles.addTaskRow}>
              <Text style={styles.addTaskLabel}>Task</Text>
              <TextInput
                style={styles.addTaskInput}
                placeholder="Enter task"
                value={taskTitle}
                onChangeText={(text) => {
                  setTaskTitle(text);
                  setErrors({ ...errors, taskTitle: text.trim() ? '' : 'Task Title is required' });
                }}
                placeholderTextColor="#888"
              />
            </View>
            <View style={styles.addTaskRow}>
              <Text style={styles.addTaskLabel}>Type</Text>
              <View style={styles.addTaskTypeButtons}>
                <TouchableOpacity
                  style={[
                    styles.addTaskTypeButton,
                    taskDescription === 'Repair' ? styles.addTaskTypeButtonSelected : styles.addTaskTypeButtonUnselected,
                  ]}
                  onPress={() => setTaskDescription('Repair')}
                >
                  <Text style={taskDescription === 'Repair' ? styles.addTaskTypeTextSelected : styles.addTaskTypeText}>Repair</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[
                    styles.addTaskTypeButton,
                    taskDescription === 'Replace' ? styles.addTaskTypeButtonSelected : styles.addTaskTypeButtonUnselected,
                  ]}
                  onPress={() => setTaskDescription('Replace')}
                >
                  <Text style={taskDescription === 'Replace' ? styles.addTaskTypeTextSelected : styles.addTaskTypeText}>Replace</Text>
                </TouchableOpacity>
              </View>
            </View>
            {/* Assign to Row for Task */}
            <View style={styles.addTaskRow}>
              <Text style={styles.addTaskLabel}>Assign to</Text>
              <TouchableOpacity
                style={styles.addTaskSetRight}
                onPress={() => {
                  setTaskModalVisible(false);
                  setTimeout(() => setTaskAssignToModalVisible(true), 300);
                }}
              >
                <Text style={styles.addTaskSetText}>Set</Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View>
            {/* Show selected assignees for task */}
            {taskAssignees.length > 0 && (
              <View style={{ marginTop: width * 0.01 }}>
                <ScrollView
                  horizontal={true}
                  showsHorizontalScrollIndicator={false}
                  contentContainerStyle={{ flexDirection: 'row', alignItems: 'center' }}
                >
                  {taskAssignees.map((item, index) => (
                    <View key={index} style={{
                      flexDirection: 'row',
                      alignItems: 'center',
                      backgroundColor: '#e6f0ff',
                      borderRadius: 8,
                      paddingVertical: 6,
                      paddingHorizontal: 10,
                      marginRight: 8,
                      marginBottom: 4,
                    }}>
                      {item.profilePic ? (
                        <Image
                          source={{ uri: item.profilePic.url }}
                          style={{ width: 32, height: 32, borderRadius: 16, marginRight: 8 }}
                        />
                      ) : (
                        <View style={{
                          width: 32,
                          height: 32,
                          borderRadius: 16,
                          backgroundColor: '#ddd',
                          marginRight: 8,
                          alignItems: 'center',
                          justifyContent: 'center',
                        }}>
                          <MaterialIcons name="person" size={18} color="#888" />
                        </View>
                      )}
                      <View style={{ marginRight: 6 }}>
                        <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 14 }}>{item.fullName}</Text>
                        <Text style={{ color: '#333', fontSize: 11 }}>Emp#: {item.employeeNumber} | Role: {item.role}</Text>
                      </View>
                      <TouchableOpacity
                        style={{ marginLeft: 2, padding: 2 }}
                        onPress={() => setTaskAssignees(taskAssignees.filter(u => u.id !== item.id))}
                      >
                        <View style={{
                          width: 18,
                          height: 18,
                          borderRadius: 9,
                          backgroundColor: '#d32f2f',
                          justifyContent: 'center',
                          alignItems: 'center',
                        }}>
                          <MaterialIcons name="close" size={13} color="#fff" />
                        </View>
                      </TouchableOpacity>
                    </View>
                  ))}
                </ScrollView>
              </View>
            )}
            {/* Note Row (per-task) */}
            <View style={styles.addTaskRow}>
              <Text style={styles.addTaskLabel}>Note</Text>
              <TextInput
                style={styles.addTaskInput}
                placeholder="Add note"
                value={taskNote}
                onChangeText={setTaskNote}
                placeholderTextColor="#888"
              />
            </View>
            <View style={styles.addTaskRow}>
              <Text style={styles.addTaskLabel}>Fault List</Text>
              <Text style={styles.addTaskValue}>-</Text>
            </View>
            <View style={styles.addTaskButtonRow}>
              <TouchableOpacity
                style={[styles.addTaskActionButton, styles.addTaskCloseButton]}
                onPress={() => {
                  setTaskModalVisible(false);
                  setTaskTitle('');
                  setTaskDescription('');
                  setTaskAssignees([]);
                  setTaskNote('');
                  setSelectedTask(null);
                }}
              >
                <Text style={styles.addTaskCloseButtonText}>Close</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.addTaskActionButton, styles.addTaskSaveButton]}
                onPress={selectedTask ? handleSaveEditedTask : handleAddTask}
              >
                <Text style={styles.addTaskSaveButtonText}>{selectedTask ? 'Update' : 'Save'}</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Add Task Assign To Modal */}
      <Modal
        visible={taskAssignToModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setTaskAssignToModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.assignModalContentImproved, { maxHeight: '80%' }]}>
            <Text style={styles.assignModalTitleImproved}>Assign To (Task)</Text>
            <Text style={{ fontSize: 12, color: '#666', textAlign: 'center', marginBottom: 10 }}>
              Displayed: {teamMembers.length} | Total: {allUsers.length} | Loading: {loadingTeamMembers ? 'Yes' : 'No'}
            </Text>
            <Text style={{ fontSize: 10, color: '#999', textAlign: 'center', marginBottom: 10 }}>
              Using Team Management API (same as Create Request)
            </Text>
            <View style={{ flexDirection: 'row', alignItems: 'center', marginBottom: 10 }}>
              <TextInput
                style={[styles.searchInput, { flex: 1, marginRight: 10 }]}
                placeholder="Search team members..."
                placeholderTextColor="#888"
                value={taskAssignSearchQuery || ''}
                onChangeText={(text) => setTaskAssignSearchQuery(text || '')}
              />
              <TouchableOpacity
                style={{ padding: 8, backgroundColor: '#4CAF50', borderRadius: 5 }}
                onPress={() => {
                  console.log('🔍 Search triggered...');
                  // The search is already handled by the TextInput onChangeText
                }}
              >
                <MaterialIcons name="search" size={16} color="#fff" />
              </TouchableOpacity>
            </View>
            <FlatList
              data={teamMembers.filter(member => {
                if (!member) return false;
                const searchLower = (taskAssignSearchQuery || '').toLowerCase();
                const fullName = member.fullName || '';
                const employeeNumber = member.employeeNumber || '';
                const role = member.role || '';
                
                if (!taskAssignSearchQuery || taskAssignSearchQuery.trim() === '') {
                  return true;
                }
                
                return (
                  fullName.toLowerCase().includes(searchLower) ||
                  employeeNumber.toLowerCase().includes(searchLower) ||
                  role.toLowerCase().includes(searchLower)
                );
              })}
              keyExtractor={(item, index) => item?.id || `team-member-${index}`}
              renderItem={({ item }) => {
                if (!item) return null;
                
                const fullName = item.fullName || item.name || 'Unknown User';
                const employeeNumber = item.employeeNumber || 'N/A';
                const role = item.role || 'N/A';
                const profilePic = item.profilePic || null;
                
                const isSelected = taskAssignees.some(selected => selected && selected.id === item.id);
                
                return (
                  <TouchableOpacity
                    style={{
                      flexDirection: 'row',
                      alignItems: 'center',
                      paddingVertical: 12,
                      paddingHorizontal: 16,
                      borderBottomWidth: 1,
                      borderBottomColor: '#eee',
                      backgroundColor: isSelected ? '#e6f0ff' : '#fff',
                    }}
                    onPress={() => {
                      if (isSelected) {
                        setTaskAssignees(taskAssignees.filter(u => u.id !== item.id));
                      } else {
                        setTaskAssignees([...taskAssignees, item]);
                      }
                    }}
                    activeOpacity={0.7}
                  >
                    {profilePic && profilePic.url ? (
                      <Image
                        source={{ uri: profilePic.url }}
                        style={{ width: 40, height: 40, borderRadius: 20, marginRight: 12 }}
                        resizeMode="cover"
                      />
                    ) : (
                      <View style={{
                        width: 40,
                        height: 40,
                        borderRadius: 20,
                        backgroundColor: '#ddd',
                        marginRight: 12,
                        alignItems: 'center',
                        justifyContent: 'center',
                      }}>
                        <MaterialIcons name="person" size={24} color="#888" />
                      </View>
                    )}
                    <View style={{ flex: 1 }}>
                      <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 16, marginBottom: 2 }}>
                        {fullName}
                      </Text>
                      <Text style={{ color: '#333', fontSize: 13 }}>
                        Emp#: {employeeNumber} | Role: {role}
                      </Text>
                    </View>
                    {isSelected && (
                      <MaterialIcons name="check" size={24} color="#015185" style={{ marginLeft: 8 }} />
                    )}
                  </TouchableOpacity>
                );
              }}
              style={{ marginBottom: 10, maxHeight: 400 }}
              ListEmptyComponent={
                loadingTeamMembers ? (
                  <View style={{ padding: 20, alignItems: 'center' }}>
                    <Text style={styles.defaultText}>Loading team members...</Text>
                  </View>
                ) : allUsers.length === 0 ? (
                  <View style={{ padding: 20, alignItems: 'center' }}>
                    <Text style={styles.defaultText}>No team members available</Text>
                    <Text style={[styles.defaultText, { fontSize: 12, marginTop: 5 }]}>
                      Total users loaded: {allUsers.length}
                    </Text>
                    <TouchableOpacity
                      style={{ marginTop: 10, padding: 10, backgroundColor: '#015185', borderRadius: 5 }}
                      onPress={() => {
                        console.log('🔄 Manual retry triggered');
                        fetchAllTeamMembers();
                      }}
                    >
                      <Text style={{ color: '#fff' }}>Retry Load Users</Text>
                    </TouchableOpacity>
                  </View>
                ) : teamMembers.length === 0 ? (
                  <View style={{ padding: 20, alignItems: 'center' }}>
                    <Text style={styles.defaultText}>Data loaded but not displayed</Text>
                    <Text style={[styles.defaultText, { fontSize: 12, marginTop: 5 }]}>
                      All Users: {allUsers.length} | Team Members: {teamMembers.length}
                    </Text>
                    <TouchableOpacity
                      style={{ marginTop: 10, padding: 10, backgroundColor: '#4CAF50', borderRadius: 5 }}
                      onPress={() => {
                        console.log('🔄 Manual load triggered');
                        loadPaginatedTeamMembers(1, false);
                      }}
                    >
                      <Text style={{ color: '#fff' }}>Load Users Now</Text>
                    </TouchableOpacity>
                  </View>
                ) : (
                  <View style={{ padding: 20, alignItems: 'center' }}>
                    <Text style={styles.defaultText}>No team members found matching your search</Text>
                    <Text style={[styles.defaultText, { fontSize: 12, marginTop: 5 }]}>
                      Try clearing your search or check the console for errors
                    </Text>
                  </View>
                )
              }
              removeClippedSubviews={false}
              maxToRenderPerBatch={20}
              updateCellsBatchingPeriod={100}
              initialNumToRender={20}
              windowSize={10}
            />
            <View style={{ alignItems: 'center', marginTop: 10 }}>
              <TouchableOpacity
                style={styles.saveAssignButton}
                onPress={() => {
                  setTaskAssignToModalVisible(false);
                  setTimeout(() => setTaskModalVisible(true), 300);
                }}
              >
                <Text style={styles.saveAssignButtonText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </KeyboardAvoidingView>
  );
};

// Add SyncModal component for robust loader animation
const SyncModal = ({ visible, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    let animation;
    if (visible) {
      animation = Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      );
      animation.start();
    } else {
      rotateAnim.setValue(0);
      if (animation) animation.stop && animation.stop();
    }
    return () => {
      if (animation) animation.stop && animation.stop();
    };
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[
                styles.syncIcon,
                { transform: [{ rotate: rotateInterpolate }] },
              ]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Reusing styles from CreateRequestMaintenance and adding note modal styles
const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.02,
  },
  labelContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 4,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  inputContainer: {
    position: 'relative',
    flexDirection: 'row',
    alignItems: 'center',
    flexWrap: 'wrap',
  },
  tagsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginBottom: width * 0.02,
    paddingHorizontal: 12,
  },
  tag: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e0e0e0',
    borderRadius: width * 0.05,
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.02,
    marginRight: width * 0.02,
    marginBottom: width * 0.01,
  },
  tagText: {
    fontSize: width * 0.035,
    color: '#015185',
    marginRight: width * 0.01,
  },
  tagCloseButton: {
    padding: width * 0.005,
  },
  tagCloseCircle: {
    width: width * 0.035,
    height: width * 0.035,
    borderRadius: (width * 0.035) / 2,
    backgroundColor: '#d32f2f',
    justifyContent: 'center',
    alignItems: 'center',
  },
  input: {
    flex: 1,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  multilineInput: {
    height: width * 0.2,
    textAlignVertical: 'top',
  },
  inputError: {
    borderColor: 'red',
  },
  disabledInput: {
    backgroundColor: '#f5f5f5',
  },
  noBackground: {
    backgroundColor: 'transparent',
  },
  noBorder: {
    borderWidth: 0,
  },
  inputWithTags: {
    paddingTop: 10,
    paddingBottom: 10,
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
  inputIcon: {
    position: 'absolute',
    right: 12,
    top: '50%',
    transform: [{ translateY: -width * 0.03 }],
  },
  labelSelectButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.01,
  },
  selectText: {
    fontSize: width * 0.038,
    color: '#015185',
    marginRight: width * 0.01,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
    backgroundColor: '#fff',
    borderRadius: 5,
  },
  createButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  mainContent: {
    flex: 1,
  },
  formGroup: {
    marginBottom: width * 0.03,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.015,
  },
  label: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#333',
    marginBottom: width * 0.02,
  },
  requestNumberRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: 6,
    paddingHorizontal: 16,
    paddingVertical: 14,
    marginBottom: width * 0.02,
  },
  requestNumberLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  requestNumberValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
  priorityContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: width * 0.02,
    backgroundColor: '#fff',
    borderRadius: 5,
    width: '100%',
  },
  priorityItem: {
    alignItems: 'center',
  },
  priorityButton: {
    padding: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    width: 85,
    height: 60,
    justifyContent: 'center',
    alignItems: 'center',
  },
  priorityButtonSelected: {
    backgroundColor: '#015185',
  },
  priorityButtonUnselected: {
    backgroundColor: '#FFFFFF',
  },
  priorityImage: {
    width: 24,
    height: 24,
  },
  priorityText: {
    fontSize: 10,
    color: '#333',
    marginTop: 3,
    textAlign: 'center',
  },
  priorityTextSelected: {
    fontSize: 10,
    color: '#fff',
    fontWeight: '600',
    marginTop: 3,
    textAlign: 'center',
  },
  row: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '70%',
    alignItems: 'center',
  },
  successModalText: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
  },
  successModalSubText: {
    fontSize: width * 0.04,
    color: '#333',
    marginBottom: width * 0.05,
    textAlign: 'center',
  },
  successModalButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.02,
  },
  successModalButtonText: {
    color: '#FFFFFF',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  taskListContainer: {
    marginTop: width * 0.025,
    maxHeight: width * 0.3,
  },
  taskCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: width * 0.035,
    marginBottom: width * 0.025,
    borderWidth: 1,
    borderColor: '#eee',
    position: 'relative',
  },
  taskCardHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  taskCardTitle: {
    fontSize: width * 0.04,
    color: '#222',
    fontWeight: '500',
    flex: 1,
  },
  taskCardStatusPill: {
    backgroundColor: '#bfc2c5',
    borderRadius: 16,
    paddingHorizontal: 14,
    paddingVertical: 4,
    marginLeft: 8,
  },
  taskCardStatusText: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.035,
  },
  taskCardMenuButton: {
    padding: 4,
    marginLeft: 8,
  },
  taskCardType: {
    color: '#e6b800',
    fontWeight: '500',
    fontSize: width * 0.037,
    marginTop: 6,
    marginBottom: 2,
  },
  taskDropdownOverlay: {
    flex: 1,
    top: '-4%',
    backgroundColor: 'rgba(0,0,0,0.1)',
  },
  taskDropdownContainer: {
    backgroundColor: '#fff',
    borderRadius: 8,
    elevation: 8,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 6,
    width: 120,
    zIndex: 999,
  },
  taskDropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 14,
    paddingHorizontal: 18,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
    backgroundColor: '#fff',
  },
  taskDropdownText: {
    fontSize: width * 0.04,
    color: '#222',
  },
  addTaskButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#015185',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
  },
  addTaskButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
  },
  selectRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F5F5F5',
    borderRadius: 4,
    paddingVertical: width * 0.035,
    paddingHorizontal: width * 0.03,
    marginBottom: width * 0.01,
  },
  selectLabel: {
    fontSize: width * 0.038,
    color: '#707173',
    fontWeight: '500',
  },
  selectRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectText: {
    color: '#015185',
    fontSize: width * 0.038,
    fontWeight: '500',
    marginRight: 2,
  },
  selectedText: {
    fontSize: width * 0.035,
    color: '#333',
    marginTop: width * 0.01,
    textAlign: 'left',
  },
  addTaskModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.04,
    padding: width * 0.05,
    width: '90%',
    alignItems: 'stretch',
    alignSelf: 'center',
  },
  addTaskTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  addTaskRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F5F5F5',
    borderRadius: 4,
    paddingVertical: width * 0.035,
    paddingHorizontal: width * 0.03,
    marginBottom: width * 0.01,
  },
  addTaskLabel: {
    fontSize: width * 0.038,
    color: '#707173',
    fontWeight: '500',
    flex: 1,
  },
  addTaskInput: {
    flex: 2,
    fontSize: width * 0.038,
    color: '#222',
    backgroundColor: 'transparent',
    borderWidth: 0,
    textAlign: 'right',
  },
  addTaskTypeButtons: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: width * 0.02,
  },
  addTaskTypeButton: {
    borderWidth: 1,
    borderRadius: 4,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.04,
    marginHorizontal: width * 0.01,
  },
  addTaskTypeButtonSelected: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  addTaskTypeButtonUnselected: {
    backgroundColor: '#fff',
    borderColor: '#015185',
  },
  addTaskTypeText: {
    color: '#015185',
    fontWeight: '500',
    fontSize: width * 0.038,
  },
  addTaskTypeTextSelected: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.038,
  },
  addTaskSetRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  addTaskSetText: {
    color: '#015185',
    fontSize: width * 0.038,
    fontWeight: '500',
    marginRight: 2,
  },
  addTaskValue: {
    color: '#222',
    fontSize: width * 0.038,
    fontWeight: '400',
  },
  addTaskButtonRow: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: width * 0.04,
    gap: width * 0.03,
  },
  addTaskActionButton: {
    borderRadius: 4,
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.07,
    marginLeft: width * 0.02,
  },
  addTaskCloseButton: {
    borderWidth: 1,
    borderColor: '#015185',
    backgroundColor: '#fff',
  },
  addTaskCloseButtonText: {
    color: '#015185',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
  addTaskSaveButton: {
    backgroundColor: '#015185',
  },
  addTaskSaveButtonText: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
  assignModalContentImproved: {
    backgroundColor: '#fff',
    borderRadius: 14,
    padding: width * 0.06,
    width: '92%',
    maxWidth: 400,
    alignItems: 'stretch',
    alignSelf: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 8,
    elevation: 8,
  },
  assignModalTitleImproved: {
    fontSize: width * 0.052,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  assignSearchRowImproved: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F5F5F5',
    borderRadius: 8,
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.03,
    marginBottom: width * 0.03,
  },
  assignSearchInputImproved: {
    flex: 1,
    fontSize: width * 0.038,
    color: '#222',
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  assignUserCardImproved: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: width * 0.035,
    marginBottom: width * 0.02,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.07,
    shadowRadius: 3,
    elevation: 2,
  },
  assignUserCardSelectedImproved: {
    borderWidth: 2,
    borderColor: '#015185',
    backgroundColor: '#e6f0fa',
  },
  assignUserRowImproved: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: width * 0.01,
  },
  assignUserLabelImproved: {
    fontSize: width * 0.037,
    color: '#707173',
    fontWeight: '500',
  },
  assignUserValueImproved: {
    fontSize: width * 0.037,
    color: '#222',
    marginTop: 2,
  },
  assignRolePillImproved: {
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingVertical: 4,
    paddingHorizontal: 12,
    alignItems: 'center',
    justifyContent: 'center',
    marginLeft: 8,
  },
  assignRoleTextImproved: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: '500',
  },
  assignSelectButtonImproved: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 18,
    alignSelf: 'flex-end',
  },
  assignSelectButtonSelectedImproved: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  assignSelectButtonTextImproved: {
    color: '#015185',
    fontWeight: '600',
    fontSize: width * 0.038,
  },
  assignSelectButtonTextSelectedImproved: {
    color: '#fff',
    fontWeight: '600',
    fontSize: width * 0.038,
  },
  assignUserDividerImproved: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginVertical: width * 0.01,
  },
  assignButtonRowImproved: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: width * 0.04,
    gap: width * 0.03,
  },
  assignActionButtonImproved: {
    borderRadius: 6,
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.07,
    marginLeft: width * 0.02,
  },
  assignCancelButtonImproved: {
    borderWidth: 1,
    borderColor: '#015185',
    backgroundColor: '#fff',
  },
  assignCancelButtonTextImproved: {
    color: '#333',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
  assignSaveButtonImproved: {
    backgroundColor: '#015185',
  },
  assignSaveButtonTextImproved: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
  requestTypeModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.04,
    padding: width * 0.03,
    width: '90%',
    alignSelf: 'center',
    marginTop: width * 0.15,
  },
  requestTypeRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: width * 0.045,
    paddingHorizontal: width * 0.01,
    backgroundColor: 'transparent',
  },
  requestTypeLabel: {
    color: '#015185',
    fontSize: width * 0.045,
    fontWeight: '400',
  },
  requestTypeLabelSelected: {
    fontWeight: 'bold',
  },
  requestTypeRadioOuter: {
    width: width * 0.055,
    height: width * 0.055,
    borderRadius: width * 0.0275,
    borderWidth: 2,
    borderColor: '#015185',
    alignItems: 'center',
    justifyContent: 'center',
  },
  requestTypeRadioInner: {
    width: width * 0.03,
    height: width * 0.03,
    borderRadius: width * 0.015,
    backgroundColor: '#015185',
  },
  requestTypeDivider: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginLeft: 0,
    marginRight: 0,
  },
  requestTypeButtonRow: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: width * 0.04,
  },
  requestTypeCloseButton: {
    backgroundColor: '#015185',
    borderRadius: 4,
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.07,
  },
  requestTypeCloseButtonText: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
  noteModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    width: '85%',
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 5,
  },
  noteModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    fontStyle: 'italic',
    color: '#3a4a5a',
    marginBottom: 15,
    alignSelf: 'center',
  },
  noteInputWrapper: {
    width: '100%',
    height: 120,
    borderWidth: 1,
    borderColor: '#bfc2c5',
    borderRadius: 8,
    marginBottom: 20,
    backgroundColor: '#f9f9f9',
    position: 'relative',
    overflow: 'hidden',
    justifyContent: 'flex-start',
  },
  noteInput: {
    width: '100%',
    height: '100%',
    fontSize: 16,
    color: '#222',
    padding: 10,
    backgroundColor: 'transparent',
    zIndex: 2,
    textAlignVertical: 'top',
  },
  noteLine: {
    position: 'absolute',
    left: 0,
    right: 0,
    height: 1,
    backgroundColor: '#e0e0e0',
    zIndex: 1,
  },
  noteModalButtonRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    marginTop: 10,
  },
  noteModalButton: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: 12,
    borderRadius: 5,
    marginHorizontal: 5,
  },
  rejectButton: {
    backgroundColor: '#C2185B',
  },
  closeButton: {
    backgroundColor: '#888',
  },
  approveButton: {
    backgroundColor: '#015185',
  },
  noteModalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    letterSpacing: 1,
  },
  saveAssignButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.07,
    borderRadius: width * 0.02,
  },
  saveAssignButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  searchInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.025,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  retryContainer: {
    backgroundColor: '#fff3cd',
    borderColor: '#ffeaa7',
    borderWidth: 1,
    borderRadius: 8,
    padding: 16,
    marginTop: 10,
    alignItems: 'center',
  },
  retryText: {
    fontSize: width * 0.04,
    color: '#856404',
    marginBottom: 10,
    textAlign: 'center',
  },
  retryButton: {
    backgroundColor: '#015185',
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 6,
  },
  retryButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: '600',
    marginLeft: 8,
  },
});

export default RequestMApproveWO;