import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  TextInput,
  Dimensions,
  Image,
  Modal,
  Alert,
  Share,
  Platform,
  Linking,
  Animated,
  KeyboardAvoidingView,
  PermissionsAndroid,
  BackHandler,
  ActivityIndicator,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import { Buffer } from 'buffer';
import { check, request, PERMISSIONS, RESULTS } from 'react-native-permissions';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import SignatureCanvas from 'react-native-signature-canvas';
import Toast from 'react-native-toast-message';
import InspectionAPI from '../../ConfigMongo/InspectionMongoConfig.js';
import NetInfo from '@react-native-community/netinfo';
import OfflineDataHelper from './OfflineDataHelper';
import VoiceToText, { VoiceToTextEvents } from '@appcitor/react-native-voice-to-text';
import LottieView from 'lottie-react-native';


const { height, width } = Dimensions.get('window');

const canvasStyle = `
  .m-signature-pad { height: ${height * 0.4}px; width: 100%; margin: 0; padding: 0; }
  .m-signature-pad--body { border: 1px solid #ddd; height: 100%; width: 100%; }
  canvas { height: 100% !important; width: 100% !important; }
`;

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[
                styles.syncIcon,
                { transform: [{ rotate: rotateInterpolate }] },
              ]}
            />
            <Text style={styles.syncModalTextadd}>Generating Report</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const Summerystep3 = ({
  setActiveStep = () => {},
  onSignaturePress = () => {},
}) => {
  const navigation = useNavigation();
  const [equipmentStatus, setEquipmentStatus] = useState(null);
  const [priority, setPriority] = useState(null);
  const [safeToUse, setSafeToUse] = useState(null);
  const [additionalNote, setAdditionalNote] = useState('');
  const [overallCondition, setOverallCondition] = useState(null);
  const [modalVisible, setModalVisible] = useState(false);
  const [faultyItems, setFaultyItems] = useState(0);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [savedPdfPath, setSavedPdfPath] = useState(null);
  const [reportId, setReportId] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [errors, setErrors] = useState({
    overallCondition: '',
    equipmentStatus: '',
    safeToUse: '',
    signature: '',
  });
  const [isListening, setIsListening] = useState(false);
  const [voiceError, setVoiceError] = useState(null);
  const [micPermissionGranted, setMicPermissionGranted] = useState(false);
  const [isVoiceModuleAvailable, setIsVoiceModuleAvailable] = useState(false);
  const [isRecognitionAvailable, setIsRecognitionAvailable] = useState(true);
  const [voiceModalVisible, setVoiceModalVisible] = useState(false);
  const [partialText, setPartialText] = useState('');
  const [volume, setVolume] = useState(0);
  const [ignoreVoiceEvents, setIgnoreVoiceEvents] = useState(false);
  const [isStopping, setIsStopping] = useState(false);
  const lottieRef = useRef(null);
  const ignoreVoiceEventsRef = useRef(false);
  const isMountedRef = useRef(true);
  const listenersRef = useRef([]);
  const isCleaningUpRef = useRef(false);
  const isListeningRef = useRef(false);
  const cleanupTimeoutRef = useRef(null);
  const stopFallbackTimeoutRef = useRef(null);
  const pendingStopShouldAddRef = useRef(false);
  const hasStoppedRecordingRef = useRef(false);
  const hasReceivedFinalResultsRef = useRef(false);
  const recordingStartTimeRef = useRef(null);
  const isRestartingRef = useRef(false);
  const lastFinalTextRef = useRef('');
  const [checklistMeta, setChecklistMeta] = useState(null);

  const [signatureFields, setSignatureFields] = useState([]);
  const [checklistSignatureFields, setChecklistSignatureFields] = useState([]);
  const [signatureValues, setSignatureValues] = useState({});
  const [signatureErrors, setSignatureErrors] = useState({});
  const [signatureCount, setSignatureCount] = useState(1);
  const [showSignatureCountModal, setShowSignatureCountModal] = useState(false);
  const [selectedSignatureType, setSelectedSignatureType] = useState('');
  const [showSignatureTypeModal, setShowSignatureTypeModal] = useState(false);
  const [selectedSignatureTypes, setSelectedSignatureTypes] = useState([]);
  const [showMultiSignatureModal, setShowMultiSignatureModal] = useState(false);
  const [individualSignatures, setIndividualSignatures] = useState({});
  const [individualSignatureStatus, setIndividualSignatureStatus] = useState({});
  const [currentSignatureField, setCurrentSignatureField] = useState(null);
  const [signatureModalVisible, setSignatureModalVisible] = useState(false);
  const [currentSignatureType, setCurrentSignatureType] = useState(null);
  const [signatureStatus, setSignatureStatus] = useState(false);
  const [signatureData, setSignatureData] = useState(null);
  const signatureRef = useRef(null);
  const [footerGroupLabels, setFooterGroupLabels] = useState([]);

  useEffect(() => {
    const fetchChecklistMeta = async () => {
      try {
        const inspectionStep1Raw = await AsyncStorage.getItem('inspectionStep1');
        const inspectionStep1 = inspectionStep1Raw ? JSON.parse(inspectionStep1Raw) : {};
        const checklistId = inspectionStep1.selectedChecklistId;

        let checklistMetaObj = null;
        if (checklistId) {
          // Only load the full checklist object (with metadata) from AsyncStorage
          const storedChecklist = await AsyncStorage.getItem(`checklist_${checklistId}`);
          if (storedChecklist) {
            checklistMetaObj = JSON.parse(storedChecklist);
            console.log('Loaded checklist meta:', checklistMetaObj);
            console.log('Footer inputs from checklist:', checklistMetaObj?.footer_inputs);
          }
          // Load cached footer inputs saved in Step 1 for instant summary display
          try {
            const cachedFooter = await AsyncStorage.getItem('selectedChecklistFooterInputs');
            if (cachedFooter) {
              const parsed = JSON.parse(cachedFooter);
              await loadFooterAndSignatureDataFromStored(parsed);
            }
          } catch (e) {
            console.warn('Unable to read cached footer inputs:', e?.message);
          }
        }
        setChecklistMeta(checklistMetaObj || null);
        
        // Load footer inputs and signature fields from Firebase
        await loadFooterAndSignatureData(checklistId);
        
        // Also try to load from stored checklist data
        if (checklistMetaObj && checklistMetaObj.footer_inputs) {
          console.log('Loading footer inputs from stored checklist:', checklistMetaObj.footer_inputs);
          await loadFooterAndSignatureDataFromStored(checklistMetaObj.footer_inputs);
        }
      } catch (e) {
        console.error('Error in fetchChecklistMeta:', e);
        setChecklistMeta(null);
      }
    };
    fetchChecklistMeta();
  }, []);

  useEffect(() => {
    ignoreVoiceEventsRef.current = ignoreVoiceEvents;
  }, [ignoreVoiceEvents]);

  useEffect(() => {
    if (!VoiceToText || typeof VoiceToText.addEventListener !== 'function') {
      const msg =
        'VoiceToText module is unavailable. Ensure @appcitor/react-native-voice-to-text is installed and linked correctly.';
      console.error(msg);
      setVoiceError(msg);
      return;
    }

    setIsVoiceModuleAvailable(true);
    isMountedRef.current = true;
    isCleaningUpRef.current = false;

    const resultsListener = VoiceToText.addEventListener(VoiceToTextEvents.RESULTS, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current) {
        onSpeechResults(event);
      }
    });
    const partialListener = VoiceToText.addEventListener(VoiceToTextEvents.PARTIAL_RESULTS, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current) {
        if (isListeningRef.current) {
          setPartialText(event?.value || '');
        }
      }
    });
    const volumeListener = VoiceToText.addEventListener(VoiceToTextEvents.VOLUME_CHANGED, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current) {
        setVolume(event?.value || 0);
      }
    });
    const errorListener = VoiceToText.addEventListener(VoiceToTextEvents.ERROR, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current) {
        onSpeechError(event);
      }
    });
    const startListener = VoiceToText.addEventListener(VoiceToTextEvents.START, () => {
      if (isMountedRef.current && !isCleaningUpRef.current) {
        isListeningRef.current = true;
        hasStoppedRecordingRef.current = false;
        hasReceivedFinalResultsRef.current = false;
        isRestartingRef.current = false;
        recordingStartTimeRef.current = Date.now();
        setIsListening(true);
        setVoiceModalVisible(true);
        setPartialText('');
        setIgnoreVoiceEvents(false);
        setIsStopping(false);
        if (stopFallbackTimeoutRef.current) {
          clearTimeout(stopFallbackTimeoutRef.current);
          stopFallbackTimeoutRef.current = null;
        }
        if (lottieRef.current) {
          lottieRef.current.play();
        }
      }
    });
    const endListener = VoiceToText.addEventListener(VoiceToTextEvents.END, () => {
      if (
        isMountedRef.current &&
        !isCleaningUpRef.current &&
        !hasStoppedRecordingRef.current &&
        isListeningRef.current &&
        !isRestartingRef.current
      ) {
        isRestartingRef.current = true;
        setTimeout(async () => {
          if (
            isMountedRef.current &&
            !isCleaningUpRef.current &&
            !hasStoppedRecordingRef.current &&
            isListeningRef.current
          ) {
            await restartListening();
          } else {
            isRestartingRef.current = false;
          }
        }, 200);
      }
    });

    listenersRef.current = [
      resultsListener,
      partialListener,
      volumeListener,
      errorListener,
      startListener,
      endListener,
    ];

    (async () => {
      try {
        if (typeof VoiceToText.isRecognitionAvailable === 'function') {
          const avail = await VoiceToText.isRecognitionAvailable();
          setIsRecognitionAvailable(avail === true);
          if (!avail && Platform.OS === 'ios') {
            setVoiceError(
              'Speech recognition not available on this device. Test on a real device and verify Info.plist permissions.'
            );
          }
        }
      } catch (err) {
        console.warn('isRecognitionAvailable check failed', err);
      }
    })();

    const checkPermissions = async () => {
      if (Platform.OS === 'android') {
        try {
          const granted = await checkMicrophonePermission();
          if (!granted) {
            const requested = await requestMicrophonePermission();
            if (!requested) {
              setVoiceError('Microphone permission denied. Please enable it in app settings.');
            }
          }
        } catch (err) {
          console.warn('Permission check error:', err);
          setVoiceError('Failed to check microphone permissions.');
        }
      }
    };

    checkPermissions();

    return () => {
      isMountedRef.current = false;
      isCleaningUpRef.current = true;

      if (cleanupTimeoutRef.current) {
        clearTimeout(cleanupTimeoutRef.current);
      }
      if (stopFallbackTimeoutRef.current) {
        clearTimeout(stopFallbackTimeoutRef.current);
        stopFallbackTimeoutRef.current = null;
      }

      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }

      const stopRecordingImmediately = async () => {
        try {
          if (isListeningRef.current && VoiceToText && !hasStoppedRecordingRef.current) {
            hasStoppedRecordingRef.current = true;

            const stopMethods = [
              () => VoiceToText.stopListening && VoiceToText.stopListening(),
              () => VoiceToText.stop && VoiceToText.stop(),
              () => VoiceToText.cancelListening && VoiceToText.cancelListening(),
              () => VoiceToText.cancel && VoiceToText.cancel(),
              () => VoiceToText.destroy && VoiceToText.destroy(),
            ];

            for (const stopMethod of stopMethods) {
              try {
                if (typeof stopMethod === 'function') {
                  await stopMethod();
                  break;
                }
              } catch (e) {
                // continue
              }
            }
          }
        } catch (e) {
          console.warn('Error during cleanup (non-fatal):', e.message);
        }
      };

      stopRecordingImmediately();
      listenersRef.current = [];
    };
  }, [appendVoiceTextToAdditionalNote, restartListening, onSpeechResults, onSpeechError]);

  useFocusEffect(
    useCallback(() => {
      hasStoppedRecordingRef.current = false;
      isCleaningUpRef.current = false;
      isMountedRef.current = true;

      return () => {
        safeStopRecording();
      };
    }, [safeStopRecording])
  );

  useEffect(() => {
    const backHandler = BackHandler.addEventListener('hardwareBackPress', () => {
      if (isListeningRef.current) {
        safeStopRecording();
        return true;
      }
      return false;
    });

    return () => backHandler.remove();
  }, [safeStopRecording]);

  useEffect(() => {
    if (voiceModalVisible && isListening && lottieRef.current) {
      const timer = setTimeout(() => {
        if (lottieRef.current) {
          lottieRef.current.play();
        }
      }, 100);
      return () => clearTimeout(timer);
    } else if (!isListening && lottieRef.current) {
      lottieRef.current.pause();
      lottieRef.current.reset();
    }
  }, [voiceModalVisible, isListening]);

  // Debug useEffect to track signatureFields changes
  useEffect(() => {
    console.log('signatureFields changed:', signatureFields);
    console.log('signatureFields length:', signatureFields.length);
    signatureFields.forEach((field, index) => {
      console.log(`Field ${index}:`, field);
    });
  }, [signatureFields]);

  // Monitor signatureFields changes for debugging
  useEffect(() => {
    console.log('=== signatureFields State Change ===');
    console.log('signatureFields updated:', signatureFields);
    console.log('signatureFields.length:', signatureFields.length);
    console.log('selectedSignatureTypes:', selectedSignatureTypes);
    console.log('signatureValues:', signatureValues);
  }, [signatureFields, selectedSignatureTypes, signatureValues]);

  // Monitor individual signature states
  useEffect(() => {
    console.log('=== Individual Signature States ===');
    console.log('individualSignatureStatus:', individualSignatureStatus);
    console.log('individualSignatures keys:', Object.keys(individualSignatures));
  }, [individualSignatureStatus, individualSignatures]);

  const loadFooterAndSignatureData = async (checklistId) => {
    try {
      if (!checklistId) return;

      console.log('Loading footer data for checklistId:', checklistId);

      // First try to get footer inputs from stored checklist
      const storedChecklist = await AsyncStorage.getItem(`checklist_${checklistId}`);
      if (storedChecklist) {
        const checklist = JSON.parse(storedChecklist);
        if (checklist.footer_inputs) {
          console.log('Loading footer inputs from stored checklist:', checklist.footer_inputs);
          await loadFooterAndSignatureDataFromStored(checklist.footer_inputs);
          return;
        }
      }

      // Try to fetch footer inputs from MongoDB
      try {
        console.log('Fetching checklist from MongoDB:', checklistId);
        const checklistResponse = await InspectionAPI.getChecklist(checklistId);
        
        if (checklistResponse.success && checklistResponse.data) {
          const checklistData = checklistResponse.data;
          console.log('Loaded checklist from MongoDB:', checklistData);
          
          if (checklistData.footer_inputs) {
            console.log('Setting footer inputs from MongoDB:', checklistData.footer_inputs);
            await loadFooterAndSignatureDataFromStored(checklistData.footer_inputs);
            return;
          }
        }
      } catch (mongoError) {
        console.warn('MongoDB footer_inputs fetch failed:', mongoError?.message);
      }

      // Fallback to default footer inputs
      console.log('Using default footer inputs');
      const defaultFooterInputs = {
        driller_name: '',
        driller_signature: '',
        mechanic_name: '',
        mechanic_signature: '',
      };
      await loadFooterAndSignatureDataFromStored(defaultFooterInputs);
    } catch (error) {
      console.error('Error loading footer data:', error);
      // Set default footer inputs on error
      const defaultFooterInputs = {
        'driller_name': 'Driller Name',
        'driller_signature': 'Driller Signature',
        'mechanic_name': 'Mechanic Name',
        'mechanic_signature': 'Mechanic Signature'
      };
      await loadFooterAndSignatureDataFromStored(defaultFooterInputs);
    }
  };

  const loadFooterAndSignatureDataFromStored = async (footerInputs) => {
    try {
      console.log('Processing footer inputs:', footerInputs);
      
      if (!footerInputs || typeof footerInputs !== 'object') {
        console.log('Invalid footer inputs, using defaults');
        footerInputs = {
          driller_name: '',
          driller_signature: '',
          mechanic_name: '',
          mechanic_signature: '',
        };
      }

      // Convert footer inputs object to signature fields format (robust grouping)
      const footerSignatureFields = [];

      if (typeof footerInputs === 'object' && !Array.isArray(footerInputs)) {
        const groupMap = {};

        const setGroup = (groupKey, part, keyName) => {
          if (!groupMap[groupKey]) groupMap[groupKey] = {};
          groupMap[groupKey][part] = keyName;
        };

        Object.keys(footerInputs).forEach((keyName) => {
          if (keyName === 'signature') {
            setGroup('signature', 'signature', keyName);
            return;
          }
          if (keyName === 'comments') {
            setGroup('comments', 'comments', keyName);
            return;
          }
          if (keyName === 'next_service_due_km') {
            setGroup('service', 'service', keyName);
            return;
          }

          // Use full prefix before trailing _name/_signature as group key
          if (/_name$/.test(keyName)) {
            const groupKey = keyName.replace(/_name$/, '');
            setGroup(groupKey, 'nameKey', keyName);
            return;
          }
          if (/_signature$/.test(keyName)) {
            const groupKey = keyName.replace(/_signature$/, '');
            setGroup(groupKey, 'signatureKey', keyName);
            return;
          }

          // Fallback: single field without suffix, treat as input-only
          setGroup(keyName, 'nameKey', keyName);
        });

        const toLabel = (groupKey) => groupKey
          .split('_')
          .map((w) => (w ? w.charAt(0).toUpperCase() + w.slice(1) : w))
          .join(' ');

        // If there is a bare `signature` without a role, try to infer the role from any available *_name
        if (groupMap.signature && !groupMap.signature.nameKey) {
          const priority = ['operator', 'driller', 'driver', 'mechanic', 'supervisor', 'artisan', 'inspector', 'manager', 'technician'];
          let candidate = undefined;
          for (const p of priority) {
            if (groupMap[p]?.nameKey) { candidate = groupMap[p].nameKey; break; }
          }
          if (!candidate) {
            for (const k of Object.keys(groupMap)) {
              if (k !== 'signature' && groupMap[k]?.nameKey) { candidate = groupMap[k].nameKey; break; }
            }
          }
          if (candidate) groupMap.signature.nameKey = candidate;
        }

        // Collect labels for UI inline list (only groups that include a signature)
        const inlineLabels = [];

        Object.keys(groupMap).forEach((groupKey) => {
          const conf = groupMap[groupKey];
          let label = '';
          let showInput = false;
          let showSignature = false;
          let nameKey = undefined;
          let signatureKey = undefined;

          if (groupKey === 'signature') {
            if (conf.nameKey) {
              const role = conf.nameKey.replace(/_name$/, '');
              label = `${toLabel(role)} Signature`;
              showInput = true;
              nameKey = conf.nameKey;
            } else {
              // Explicit fallback to Operator when only bare `signature` is present
              label = 'Operator Signature';
            }
            showSignature = true;
            signatureKey = conf.signature;
          } else if (groupKey === 'comments') {
            label = 'Comments';
            showInput = true;
            nameKey = conf.comments;
          } else if (groupKey === 'service') {
            label = 'Next Service Due (KM)';
            showInput = true;
            nameKey = conf.service;
          } else {
            label = toLabel(groupKey);
            showInput = !!conf.nameKey;
            showSignature = !!conf.signatureKey;
            nameKey = conf.nameKey;
            signatureKey = conf.signatureKey;
          }

          footerSignatureFields.push({
            key: groupKey,
            label,
            roleName: label,
            required: true,
            showInput,
            showSignature,
            nameKey,
            signatureKey,
            originalData: { groupKey, conf },
          });

          if ((signatureKey || groupKey === 'signature') && label) {
            inlineLabels.push(label);
          }
        });

        // Deduplicate and save inline labels for summary row
        const uniqueInlineLabels = Array.from(new Set(inlineLabels));
        setFooterGroupLabels(uniqueInlineLabels);
      } else if (Array.isArray(footerInputs)) {
        footerInputs.forEach((input, index) => {
          footerSignatureFields.push({
            key: `footer_${index}`,
            label: input.label || input.name || `Signature ${index + 1}`,
            roleName: input.roleName || input.label || `Signature ${index + 1}`,
            required: input.required !== false,
            showInput: input.showInput !== false,
            originalData: input,
          });
        });
        setFooterGroupLabels(
          footerInputs
            .map((i) => i.label || i.name)
            .filter(Boolean)
        );
      }

      // Only include options that actually require a signature capture
      const signatureOnlyFields = footerSignatureFields.filter(
        (field) => field.showSignature && (field.signatureKey || field.key === 'signature')
      );

      console.log('Footer signature fields converted (final):', signatureOnlyFields);
      setChecklistSignatureFields(signatureOnlyFields);
      
      // Don't automatically set signature fields - only show when selected from modal
      // This fixes issue #2 where fields were showing by default
      
    } catch (error) {
      console.error('Error processing footer inputs:', error);
      // Set default signature fields on error
      const defaultSignatureFields = [
        {
          key: 'driller',
          label: 'Driller',
          roleName: 'Driller',
          required: true,
          showInput: true,
          showSignature: true,
          nameKey: 'driller_name',
          signatureKey: 'driller_signature'
        },
        {
          key: 'mechanic',
          label: 'Mechanic',
          roleName: 'Mechanic',
          required: true,
          showInput: true,
          showSignature: true,
          nameKey: 'mechanic_name',
          signatureKey: 'mechanic_signature'
        }
      ];
      setChecklistSignatureFields(defaultSignatureFields);
      // Don't automatically set signature fields here either
      setFooterGroupLabels(['Driller', 'Mechanic']);
    }
  };

  const logCompleteChecklistData = async () => {
    try {
      console.log('=== COMPLETE CHECKLIST DATA LOG ===');
      
      // Get inspection step 1 data
      const inspectionStep1Raw = await AsyncStorage.getItem('inspectionStep1');
      const inspectionStep1 = inspectionStep1Raw ? JSON.parse(inspectionStep1Raw) : {};
      const checklistId = inspectionStep1.selectedChecklistId;
      
      console.log('Checklist ID:', checklistId);
      console.log('Inspection Step 1 Data:', inspectionStep1);
      
      // Get stored checklist data
      if (checklistId) {
        const storedChecklist = await AsyncStorage.getItem(`checklist_${checklistId}`);
        if (storedChecklist) {
          const checklistData = JSON.parse(storedChecklist);
          console.log('=== STORED CHECKLIST DATA ===');
          console.log('Checklist Title:', checklistData.title);
          console.log('Checklist ID:', checklistData.id);
          console.log('Document ID:', checklistData.document_id);
          console.log('Version:', checklistData.version);
          console.log('Version Date:', checklistData.version_date);
          console.log('Approved By:', checklistData.approved_by);
          
          // Log header inputs
          if (checklistData.header_inputs) {
            console.log('=== HEADER INPUTS ===');
            console.log('Header Inputs:', checklistData.header_inputs);
            console.log('Header Inputs Keys:', Object.keys(checklistData.header_inputs));
          } else {
            console.log('=== HEADER INPUTS ===');
            console.log('No header inputs found');
          }
          
          // Log footer inputs
          if (checklistData.footer_inputs) {
            console.log('=== FOOTER INPUTS ===');
            console.log('Footer Inputs:', checklistData.footer_inputs);
            console.log('Footer Inputs Keys:', Object.keys(checklistData.footer_inputs));
          } else {
            console.log('=== FOOTER INPUTS ===');
            console.log('No footer inputs found');
          }
          
          // Log checklist items
          if (checklistData.items) {
            console.log('=== CHECKLIST ITEMS ===');
            console.log('Number of sections:', checklistData.items.length);
            checklistData.items.forEach((section, index) => {
              console.log(`Section ${index + 1}:`, section.text);
              console.log(`Sub-items in section ${index + 1}:`, section.subItems?.length || 0);
            });
          } else {
            console.log('=== CHECKLIST ITEMS ===');
            console.log('No checklist items found');
          }
          
          // Log checklist structure
          if (checklistData.checklist) {
            console.log('=== CHECKLIST STRUCTURE ===');
            console.log('Checklist structure:', checklistData.checklist);
          }
          
          console.log('=== END CHECKLIST DATA ===');
        } else {
          console.log('No stored checklist data found for ID:', checklistId);
        }
      }
      
      // Also log current state
      console.log('=== CURRENT STATE ===');
      console.log('Checklist Meta:', checklistMeta);
      console.log('Checklist Signature Fields:', checklistSignatureFields);
      console.log('Signature Fields:', signatureFields);
      console.log('Selected Signature Types:', selectedSignatureTypes);
      console.log('=== END CURRENT STATE ===');
      
    } catch (error) {
      console.error('Error logging checklist data:', error);
    }
  };



  useEffect(() => {
    loadFaultyItems();
    loadPersistedState();
  }, []);

  const appendVoiceTextToAdditionalNote = useCallback((incomingText) => {
    const trimmedFinal = (incomingText || '').trim();
    if (!trimmedFinal) {
      return;
    }

    let nextValue = '';

    setAdditionalNote((prevValue) => {
      const currentValue = typeof prevValue === 'string' ? prevValue : '';
      const currentTrimmed = currentValue.trim();

      if (!currentTrimmed) {
        lastFinalTextRef.current = trimmedFinal;
        nextValue = trimmedFinal;
        return trimmedFinal;
      }

      if (lastFinalTextRef.current === trimmedFinal) {
        nextValue = currentValue;
        return currentValue;
      }

      if (currentTrimmed.endsWith(trimmedFinal)) {
        nextValue = currentValue;
        return currentValue;
      }

      const lastFewWords = currentTrimmed.split(' ').slice(-3).join(' ');
      const firstFewWords = trimmedFinal.split(' ').slice(0, 3).join(' ');
      if (lastFewWords && firstFewWords && lastFewWords === firstFewWords && lastFewWords.length > 5) {
        nextValue = currentValue;
        return currentValue;
      }

      if (trimmedFinal.length > 15 && currentTrimmed.includes(trimmedFinal)) {
        nextValue = currentValue;
        return currentValue;
      }

      const updated = currentTrimmed ? `${currentTrimmed} ${trimmedFinal}` : trimmedFinal;
      lastFinalTextRef.current = trimmedFinal;
      nextValue = updated;
      return updated;
    });

    if (nextValue) {
      setTimeout(() => {
        saveState(nextValue);
      }, 0);
    }
  }, []);

  const onSpeechResults = useCallback((event) => {
    if (!isMountedRef.current || isCleaningUpRef.current || ignoreVoiceEventsRef.current) {
      return;
    }

    hasReceivedFinalResultsRef.current = true;

    try {
      let payload = event?.value ?? event?.results ?? event?.text ?? event;
      let finalText = '';

      if (Array.isArray(payload)) {
        if (payload.length) {
          finalText = payload.join(' ');
        }
      } else if (typeof payload === 'string') {
        finalText = payload;
      } else if (payload && typeof payload === 'object') {
        const extracted = Object.values(payload)
          .flatMap((v) => (Array.isArray(v) ? v : [v]))
          .filter((v) => typeof v === 'string');
        if (extracted.length) {
          finalText = extracted.join(' ');
        } else {
          finalText = String(payload ?? '');
        }
      } else {
        finalText = String(payload ?? '');
      }

      if (finalText && finalText.trim()) {
        const trimmedFinal = finalText.trim();

        if (lastFinalTextRef.current === trimmedFinal) {
          setPartialText('');
          return;
        }

        appendVoiceTextToAdditionalNote(trimmedFinal);
        setPartialText('');
      }
    } catch (err) {
      console.warn('onSpeechResults parse error', err, event);
    }
  }, [appendVoiceTextToAdditionalNote]);

  const onSpeechError = useCallback((event) => {
    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }

    const message = event?.error?.message || event?.message || JSON.stringify(event);
    const lowerMessage = String(message).toLowerCase();
    const code = String(event?.error?.code ?? event?.code ?? '');
    const isSilenceError =
      lowerMessage.includes('no match') ||
      lowerMessage.includes('timeout') ||
      lowerMessage.includes('11') ||
      code.includes('11');

    if (isSilenceError) {
      setVoiceError(null);
      setIgnoreVoiceEvents(false);

      if (
        isMountedRef.current &&
        !isCleaningUpRef.current &&
        !hasStoppedRecordingRef.current &&
        isListeningRef.current &&
        !isRestartingRef.current
      ) {
        isRestartingRef.current = true;
        setTimeout(restartListening, 200);
      }
      return;
    }

    setVoiceError(`Speech recognition error: ${message}`);
    isListeningRef.current = false;
    setIsListening(false);
    setVoiceModalVisible(false);
    setIsStopping(false);
  }, [restartListening]);

  const checkMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.check(PermissionsAndroid.PERMISSIONS.RECORD_AUDIO);
      setMicPermissionGranted(granted);
      return granted;
    } catch (err) {
      console.warn('Permission check error:', err);
      return false;
    }
  };

  const requestMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.request(
        PermissionsAndroid.PERMISSIONS.RECORD_AUDIO,
        {
          title: 'Microphone Permission',
          message: 'The app needs microphone access to capture your note by voice.',
          buttonPositive: 'OK',
        }
      );
      const isGranted = granted === PermissionsAndroid.RESULTS.GRANTED;
      setMicPermissionGranted(isGranted);
      return isGranted;
    } catch (err) {
      console.warn('Permission request error:', err);
      return false;
    }
  };

  const restartListening = useCallback(async () => {
    if (
      !VoiceToText ||
      !isMountedRef.current ||
      isCleaningUpRef.current ||
      hasStoppedRecordingRef.current
    ) {
      isRestartingRef.current = false;
      return;
    }

    try {
      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      }
    } catch (restartError) {
      console.warn('Failed to auto-restart recording:', restartError);
    } finally {
      isRestartingRef.current = false;
    }
  }, []);

  const finalizeStop = useCallback(
    (shouldAddPartialText = false) => {
      setIgnoreVoiceEvents(true);
      isListeningRef.current = false;
      hasStoppedRecordingRef.current = true;
      pendingStopShouldAddRef.current = false;

      if (!isMountedRef.current || isCleaningUpRef.current) {
        return;
      }

      if (shouldAddPartialText && !hasReceivedFinalResultsRef.current) {
        setPartialText((currentPartial) => {
          if (currentPartial && currentPartial.trim()) {
            appendVoiceTextToAdditionalNote(currentPartial);
          }
          return '';
        });
      } else {
        setPartialText('');
      }

      setIsListening(false);
      setVoiceModalVisible(false);
      setIsStopping(false);

      if (stopFallbackTimeoutRef.current) {
        clearTimeout(stopFallbackTimeoutRef.current);
        stopFallbackTimeoutRef.current = null;
      }
    },
    [appendVoiceTextToAdditionalNote]
  );

  const safeStopRecording = useCallback(
    ({ shouldAddPartialText = false, showProcessing = false } = {}) => {
      pendingStopShouldAddRef.current = shouldAddPartialText;
      setIgnoreVoiceEvents(true);

      if (hasStoppedRecordingRef.current) {
        isListeningRef.current = false;
        if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
          setIsStopping(true);
        }
        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, showProcessing ? 200 : 0);
        if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
          if (stopFallbackTimeoutRef.current) {
            clearTimeout(stopFallbackTimeoutRef.current);
          }
          stopFallbackTimeoutRef.current = setTimeout(() => {
            finalizeStop(pendingStopShouldAddRef.current);
          }, 3000);
        }
        return;
      }

      if (!isListeningRef.current) {
        isListeningRef.current = false;
        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, showProcessing ? 200 : 0);
        if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
          if (stopFallbackTimeoutRef.current) {
            clearTimeout(stopFallbackTimeoutRef.current);
          }
          stopFallbackTimeoutRef.current = setTimeout(() => {
            finalizeStop(pendingStopShouldAddRef.current);
          }, 3000);
        }
        return;
      }

      if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
        setIsStopping(true);
        if (stopFallbackTimeoutRef.current) {
          clearTimeout(stopFallbackTimeoutRef.current);
        }
        stopFallbackTimeoutRef.current = setTimeout(() => {
          finalizeStop(pendingStopShouldAddRef.current);
        }, 3000);
      }

      try {
        hasStoppedRecordingRef.current = true;
        isListeningRef.current = false;

        if (VoiceToText && isVoiceModuleAvailable) {
          let stopFunc = null;
          if (typeof VoiceToText.stopListening === 'function') {
            stopFunc = VoiceToText.stopListening;
          } else if (typeof VoiceToText.stop === 'function') {
            stopFunc = VoiceToText.stop;
          } else if (typeof VoiceToText.cancelListening === 'function') {
            stopFunc = VoiceToText.cancelListening;
          } else if (typeof VoiceToText.cancel === 'function') {
            stopFunc = VoiceToText.cancel;
          }

          if (stopFunc) {
            try {
              const result = stopFunc();
              if (result && typeof result.then === 'function') {
                result.catch((stopError) => console.warn('⚠️ Error stopping recording:', stopError));
              }
            } catch (stopError) {
              console.warn('⚠️ Error stopping recording:', stopError);
            }
          }
        }

        if (lottieRef.current) {
          lottieRef.current.pause();
          lottieRef.current.reset();
        }

        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }

        const delay = showProcessing ? 550 : 400;
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, delay);
      } catch (e) {
        console.warn('⚠️ Error in safeStopRecording:', e);
        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, showProcessing ? 300 : 0);
      }
    },
    [finalizeStop, isVoiceModuleAvailable]
  );

  const startRecording = async () => {
    if (!isVoiceModuleAvailable) {
      setVoiceError('Speech engine unavailable. Please reinstall or relink the voice module.');
      return;
    }

    setVoiceError(null);

    if (Platform.OS === 'android' && !micPermissionGranted) {
      const granted = await requestMicrophonePermission();
      if (!granted) {
        setVoiceError('Microphone permission denied. Enable it in System Settings to use voice notes.');
        return;
      }
    }

    try {
      if (Platform.OS === 'ios' && !isRecognitionAvailable) {
        setVoiceError('Speech recognition not available on this iOS device.');
        return;
      }

      if (typeof VoiceToText.setRecognitionLanguage === 'function') {
        await VoiceToText.setRecognitionLanguage('en-US');
      }

      hasStoppedRecordingRef.current = false;
      hasReceivedFinalResultsRef.current = false;
      isRestartingRef.current = false;
      isListeningRef.current = true;
      setIgnoreVoiceEvents(false);
      setIsStopping(false);
      pendingStopShouldAddRef.current = false;
      if (stopFallbackTimeoutRef.current) {
        clearTimeout(stopFallbackTimeoutRef.current);
        stopFallbackTimeoutRef.current = null;
      }
      setPartialText('');
      recordingStartTimeRef.current = Date.now();
      lastFinalTextRef.current = '';

      setVoiceModalVisible(true);
      setIsListening(true);

      if (lottieRef.current) {
        lottieRef.current.play();
      }

      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      } else {
        throw new Error('startListening not available on VoiceToText');
      }
    } catch (e) {
      const message = e?.message || JSON.stringify(e);
      setVoiceError(`Failed to start voice note: ${message}`);
      setIsListening(false);
      setVoiceModalVisible(false);
      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }
    }
  };

  const stopRecording = async () => {
    if (isStopping) {
      return;
    }
    safeStopRecording({ showProcessing: true });
  };

  const cancelRecording = async () => {
    if (isStopping) {
      return;
    }

    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }

    try {
      safeStopRecording({ shouldAddPartialText: false, showProcessing: true });
    } catch (e) {
      if (isMountedRef.current && !isCleaningUpRef.current) {
        setVoiceError(`Failed to cancel recording: ${e.message || JSON.stringify(e)}`);
      }
    }
  };

  const onDone = async () => {
    if (isStopping) {
      return;
    }

    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }

    try {
      safeStopRecording({ shouldAddPartialText: true, showProcessing: true });
    } catch (err) {
      console.warn('onDone stop error', err);
    }
  };

  const loadPersistedState = async () => {
    try {
      const persistedState = await AsyncStorage.getItem('Summerystep3State');
      if (persistedState) {
        const state = JSON.parse(persistedState);
        setEquipmentStatus(state.equipmentStatus || null);
        setPriority(state.priority || null);
        setSafeToUse(state.safeToUse || null);
        setAdditionalNote(state.additionalNote || '');
        setOverallCondition(state.overallCondition || null);
      }
    } catch (error) {
      console.error('Error loading persisted state:', error);
      Alert.alert('Error', 'Failed to load persisted state.');
    }
  };

  const saveState = async (nextAdditionalNote = additionalNote) => {
    try {
      const state = {
        equipmentStatus,
        priority,
        safeToUse,
        additionalNote: nextAdditionalNote,
        overallCondition,
      };
      await AsyncStorage.setItem('Summerystep3State', JSON.stringify(state));
    } catch (error) {
      console.error('Error saving state:', error);
    }
  };

  const loadFaultyItems = async () => {
    try {
      const inspectionResultsRaw = await AsyncStorage.getItem('inspectionResults');
      if (!inspectionResultsRaw) {
        setFaultyItems(0);
        return;
      }
      const inspectionResults = JSON.parse(inspectionResultsRaw);
      const latestInspectionResult = inspectionResults[inspectionResults.length - 1] || {};
      const sections = latestInspectionResult.sections || [];
      const selections = latestInspectionResult.selections || {};

      const faultyCount = sections.flatMap((section) =>
        (section.subItems || []).filter((item) => 
          selections[item.key] === 'Faulty' || 
          selections[item.key] === 'Reject' || 
          selections[item.key] === 'Not Ok'
        )
      ).length;
      setFaultyItems(faultyCount);
    } catch (error) {
      console.error('Error loading faulty items:', error);
      setFaultyItems(0);
      Alert.alert('Error', 'Failed to load faulty items.');
    }
  };

  const statusOptions = ['Available', 'Maintenance Required', 'Breakdown'];
  const priorityOptions = [
    { label: 'Low', image: require('../../../Images/adminhome/inspreport/Priority1.png') },
    { label: 'Medium', image: require('../../../Images/adminhome/inspreport/Priority2.png') },
    { label: 'High', image: require('../../../Images/adminhome/inspreport/Priority3.png') },
    { label: 'Emergency', image: require('../../../Images/adminhome/inspreport/Priority4.png') },
  ];
  const safeOptions = ['Yes', 'No'];
  const conditionOptions = [
    'Good Condition',
    'Minor Repair required',
    'Major Repair required',
    'Cannot determine',
    'Not Applicable',
  ];

  const handleStatusSelect = (option) => {
    setEquipmentStatus(option);
    if (option !== 'Maintenance Required' && option !== 'Breakdown') {
      setPriority(null);
    }
    setErrors((prev) => ({ ...prev, equipmentStatus: '' }));
    saveState();
  };

  const handlePrioritySelect = (option) => {
    setPriority(option);
    saveState();
  };

  const handleSafeSelect = (option) => {
    setSafeToUse(option);
    setErrors((prev) => ({ ...prev, safeToUse: '' }));
    saveState();
  };

  const handleConditionSelect = (option) => {
    setOverallCondition(option);
    setModalVisible(false);
    setErrors((prev) => ({ ...prev, overallCondition: '' }));
    saveState();
  };

  const handleAdditionalNoteChange = (text) => {
    setAdditionalNote(text);
    saveState(text);
  };



  const validateSignatures = () => {
    const newErrors = {};
    let hasError = false;

    // Check if signature fields are required
    if (signatureFields.length > 0) {
      signatureFields.forEach(field => {
        if (field.required) {
          // Check name input if required
          if (field.showInput && field.nameKey) {
            const nameValue = signatureValues[field.nameKey];
            if (!nameValue || nameValue.trim() === '') {
              newErrors[field.nameKey] = `${field.roleName || field.label} name is required`;
              hasError = true;
            }
          }
          
          // Check signature if required
          if (field.showSignature && field.signatureKey) {
            const hasSignature = individualSignatureStatus[field.signatureKey] && individualSignatures[field.signatureKey];
            if (!hasSignature) {
              newErrors[field.signatureKey] = `${field.roleName || field.label} signature is required`;
              hasError = true;
            }
          }
        }
      });
    } else {
      // Check if at least one signature is provided (either main signature or individual signatures)
      const hasMainSignature = signatureStatus && signatureData;
      const hasIndividualSignatures = Object.keys(individualSignatureStatus).length > 0;
      
      if (!hasMainSignature && !hasIndividualSignatures) {
        newErrors.general = 'At least one signature is required';
        hasError = true;
      }
    }

    setSignatureErrors(newErrors);
    return !hasError;
  };

  const handleSignatureCountSelect = (count) => {
    console.log('handleSignatureCountSelect called with count:', count);
    setSignatureCount(count);
    setShowSignatureCountModal(false);
    
    // Generate signature fields based on count with proper structure
    const signatureOptions = [
      { 
        key: 'driller', 
        label: 'DRILLER\'S NAME', 
        roleName: 'Driller', 
        required: true,
        showInput: true,
        showSignature: true,
        nameKey: 'driller_name',
        signatureKey: 'driller_signature'
      },
      { 
        key: 'mechanic', 
        label: 'MECHANIC\'S NAME', 
        roleName: 'Mechanic', 
        required: true,
        showInput: true,
        showSignature: true,
        nameKey: 'mechanic_name',
        signatureKey: 'mechanic_signature'
      },
      { 
        key: 'supervisor', 
        label: 'SUPERVISOR\'S NAME', 
        roleName: 'Supervisor', 
        required: true,
        showInput: true,
        showSignature: true,
        nameKey: 'supervisor_name',
        signatureKey: 'supervisor_signature'
      },
      { 
        key: 'operator', 
        label: 'OPERATOR\'S NAME', 
        roleName: 'Operator', 
        required: true,
        showInput: true,
        showSignature: true,
        nameKey: 'operator_name',
        signatureKey: 'operator_signature'
      },
      { 
        key: 'inspector', 
        label: 'INSPECTOR\'S NAME', 
        roleName: 'Inspector', 
        required: true,
        showInput: true,
        showSignature: true,
        nameKey: 'inspector_name',
        signatureKey: 'inspector_signature'
      },
      { 
        key: 'engineer', 
        label: 'ENGINEER\'S NAME', 
        roleName: 'Engineer', 
        required: true,
        showInput: true,
        showSignature: true,
        nameKey: 'engineer_name',
        signatureKey: 'engineer_signature'
      },
      { 
        key: 'technician', 
        label: 'TECHNICIAN\'S NAME', 
        roleName: 'Technician', 
        required: true,
        showInput: true,
        showSignature: true,
        nameKey: 'technician_name',
        signatureKey: 'technician_signature'
      },
      { 
        key: 'manager', 
        label: 'MANAGER\'S NAME', 
        roleName: 'Manager', 
        required: true,
        showInput: true,
        showSignature: true,
        nameKey: 'manager_name',
        signatureKey: 'manager_signature'
      }
    ];
    
    const selectedFields = signatureOptions.slice(0, count);
    console.log('Selected signature fields:', selectedFields);
    setSignatureFields(selectedFields);
    
    // Reset signature values for new fields
    const newSignatureValues = {};
    selectedFields.forEach(field => {
      if (field.nameKey) newSignatureValues[field.nameKey] = '';
      if (field.signatureKey) newSignatureValues[field.signatureKey] = '';
    });
    setSignatureValues(newSignatureValues);
    
    // Reset individual signatures for new fields
    const newIndividualSignatures = {};
    const newIndividualSignatureStatus = {};
    selectedFields.forEach(field => {
      if (field.signatureKey) {
        newIndividualSignatures[field.signatureKey] = null;
        newIndividualSignatureStatus[field.signatureKey] = false;
      }
    });
    setIndividualSignatures(newIndividualSignatures);
    setIndividualSignatureStatus(newIndividualSignatureStatus);
    
    // Set selected signature types
    setSelectedSignatureTypes(selectedFields.map(field => field.key));
  };

  const handleSignatureTypeSelect = (type) => {
    // Only show fields for the chosen type. Never render duplicates or other types.
    setSelectedSignatureType(type);
    setShowSignatureTypeModal(false);

    // Get exactly one config for the selected type from checklist definitions
    const cfg = checklistSignatureFields.find((field) => field.key === type);

    if (cfg) {
      setSignatureFields([cfg]);
      setSelectedSignatureTypes([type]);

      // Reset values only for this field
      const newValues = {};
      if (cfg.nameKey) newValues[cfg.nameKey] = '';
      if (cfg.signatureKey) newValues[cfg.signatureKey] = '';
      setSignatureValues(newValues);
      // Clear individual signatures map except current
      setIndividualSignatures({});
      setIndividualSignatureStatus({});
    } else {
      // If not present in checklist fields, create a minimal fallback entry
      const fallback = { key: type, label: type.replace(/_/g, ' ').replace(/\b\w/g, c => c.toUpperCase()), roleName: type, required: true, showInput: true, showSignature: true, nameKey: `${type}_name`, signatureKey: `${type}_signature` };
      setSignatureFields([fallback]);
      setSelectedSignatureTypes([type]);
      setSignatureValues({ [fallback.nameKey]: '', [fallback.signatureKey]: '' });
      setIndividualSignatures({});
      setIndividualSignatureStatus({});
    }
  };

  // Local preview loader
  const openLocalPreview = async () => {
    try {
      // Load Step 1
      const step1Raw = await AsyncStorage.getItem('inspectionStep1');
      const step1 = step1Raw ? JSON.parse(step1Raw) : {};

      // Load latest Step 2 results
      const allResultsRaw = await AsyncStorage.getItem('inspectionResults');
      const allResults = allResultsRaw ? JSON.parse(allResultsRaw) : [];
      const latest = allResults.length > 0 ? allResults[allResults.length - 1] : {};

      const sections = latest.sections || [];
      const selections = latest.selections || {};
      const notes = latest.notes || {};

      // Build inspected items list (only those with a value)
      const inspected = sections.flatMap((section) => {
        const items = (section.subItems || []).filter((it) => selections[it.key] !== undefined && selections[it.key] !== null);
        return items.map((it) => ({
          name: it.itemName,
          section: section.name,
          selection: selections[it.key],
          note: notes[it.key] || '',
        }));
      });

      // Header inputs
      const headerInputValues = latest.headerInputValues || {};

      // Footer groups (labels only)
      const footerLabels = footerGroupLabels;

      setPreviewData({ step1, headerInputValues, inspected, footerLabels, checklistTitle: latest.checklistTitle || '', faultyItems: latest.faultyItems || 0 });
      setPreviewVisible(true);
    } catch (e) {
      console.error('Error preparing local preview:', e);
      Alert.alert('Error', 'Failed to prepare local preview.');
    }
  };

  const handleMultiSignatureTypeSelect = (type) => {
    setShowMultiSignatureModal(false);
    
    if (type === 'multiple') {
      // Show all available signature fields
      setSignatureFields(signatureFields);
      
      // Reset signature values for all fields
      const newSignatureValues = {};
      signatureFields.forEach(field => {
        newSignatureValues[field.key] = '';
      });
      setSignatureValues(newSignatureValues);
      
      // Set selected types to all available
      setSelectedSignatureTypes(signatureFields.map(field => field.key));
    } else {
      // Single selection
      const selectedField = signatureFields.find(field => field.key === type);
      
      if (selectedField) {
        setSignatureFields([selectedField]);
        setSelectedSignatureTypes([type]);
        
        // Reset signature values for the selected field
        const newSignatureValues = {};
        newSignatureValues[selectedField.key] = '';
        setSignatureValues(newSignatureValues);
      }
    }
  };

  const getSignatureTypeLabel = () => {
    if (selectedSignatureTypes.length > 1) {
      const selectedFields = checklistSignatureFields.filter(field => selectedSignatureTypes.includes(field.key));
      const names = selectedFields.map(field => field.label).join(' & ');
      return names;
    } else if (selectedSignatureTypes.length === 1) {
      const selectedField = checklistSignatureFields.find(field => field.key === selectedSignatureTypes[0]);
      return selectedField ? selectedField.label : 'Select Signature Type';
    }
    return 'Select Signature Type';
  };

  const getSignatureLabel = () => {
    if (selectedSignatureTypes.length > 1) {
      const selectedFields = signatureFields.filter(field => selectedSignatureTypes.includes(field.key));
      const names = selectedFields.map(field => field.roleName || field.label).join(' & ');
      return `${names} Signature`;
    } else if (selectedSignatureTypes.length === 1) {
      const selectedField = signatureFields.find(field => field.key === selectedSignatureTypes[0]);
      return selectedField ? `${selectedField.roleName || selectedField.label} Signature` : 'Inspector Signature';
    }
    return 'Inspector Signature';
  };

  const handleSignaturePress = () => {
    saveState();
    setErrors((prev) => ({ ...prev, signature: '' }));
    
    // Get the current signature type from selected signature types
    let signatureType = 'Inspector';
    if (selectedSignatureTypes.length === 1) {
      const signatureField = signatureFields.find(field => field.key === selectedSignatureTypes[0]);
      signatureType = signatureField ? (signatureField.roleName || signatureField.label) : 'Inspector';
    } else if (selectedSignatureTypes.length > 1) {
      signatureType = 'Multiple Signatures';
    }
    
    // Open signature modal for main signature
    setCurrentSignatureField(null); // Reset to indicate main signature
    setSignatureModalVisible(true);
    setCurrentSignatureType(signatureType);
  };

  const handleIndividualSignaturePress = (fieldKey) => {
    console.log('=== Individual Signature Press ===');
    console.log('Field key:', fieldKey);
    console.log('Available signature fields:', signatureFields);
    
    setCurrentSignatureField(fieldKey);
    setErrors((prev) => ({ ...prev, signature: '' }));
    
    // Find the signature field to get the correct label
    const signatureField = signatureFields.find(field => field.key === fieldKey);
    const signatureType = signatureField ? (signatureField.roleName || signatureField.label) : 'Inspector';
    
    console.log('Found signature field:', signatureField);
    console.log('Signature type:', signatureType);
    
    // Open signature modal for individual signature
    setSignatureModalVisible(true);
    setCurrentSignatureType(signatureType);
  };



  const handleIndividualSignatureComplete = (signatureData) => {
    if (currentSignatureField) {
      setIndividualSignatures(prev => ({
        ...prev,
        [currentSignatureField]: signatureData
      }));
      setIndividualSignatureStatus(prev => ({
        ...prev,
        [currentSignatureField]: true
      }));
      setCurrentSignatureField(null);
    }
  };

  // Signature modal functions
  const openSignatureModal = (type) => {
    setCurrentSignatureType(type);
    setSignatureModalVisible(true);
  };

  const closeSignatureModal = () => {
    setSignatureModalVisible(false);
    setCurrentSignatureType(null);
  };

  const handleSignatureComplete = (signature) => {
    console.log('=== Signature Complete ===');
    console.log('Signature data received:', signature ? signature.substring(0, 50) + '...' : 'null');
    console.log('Current signature field:', currentSignatureField);
    console.log('Current signature type:', currentSignatureType);
    
    if (signature && currentSignatureType) {
      if (currentSignatureField) {
        // Handle individual signature or footer signature
        console.log('Setting individual signature for field:', currentSignatureField);
        setIndividualSignatures(prev => ({
          ...prev,
          [currentSignatureField]: signature
        }));
        setIndividualSignatureStatus(prev => ({
          ...prev,
          [currentSignatureField]: true
        }));
        setCurrentSignatureField(null);
        console.log('Individual signature set successfully');
      } else {
        // Handle main signature
        console.log('Setting main signature');
        setSignatureStatus(true);
        setSignatureData(signature);
        console.log('Main signature set successfully');
      }
    } else {
      console.log('No signature data or signature type provided');
    }
    setSignatureModalVisible(false);
    setCurrentSignatureType(null);
  };

  const handleSignatureClear = () => {
    if (signatureRef.current) {
      signatureRef.current.clearSignature();
    }
  };

  const handleSignatureConfirm = () => {
    if (signatureRef.current) {
      signatureRef.current.readSignature();
    }
  };

  const handleSignatureData = (signature) => {
    console.log('Signature captured:', signature ? signature.substring(0, 50) + '...' : signature);
    handleSignatureComplete(signature);
  };



  const showPriority = equipmentStatus === 'Maintenance Required' || equipmentStatus === 'Breakdown';

  const shouldShowVoiceError =
    voiceError && !String(voiceError).toLowerCase().includes('unknown error: 11');

  const validateFields = () => {
    const newErrors = {
      overallCondition: '',
      equipmentStatus: '',
      safeToUse: '',
      signature: '',
    };
    let hasError = false;

    if (!overallCondition) {
      newErrors.overallCondition = 'Overall Condition is required';
      hasError = true;
    }

    if (!equipmentStatus) {
      newErrors.equipmentStatus = 'Equipment Status is required';
      hasError = true;
    }

    if (!safeToUse) {
      newErrors.safeToUse = 'Equipment Safe to Use is required';
      hasError = true;
    }

    // Validate signatures
    if (!validateSignatures()) {
      hasError = true;
    }

    setErrors(newErrors);
    return !hasError;
  };

  const requestStoragePermission = async () => {
    try {
      const permissions = Platform.OS === 'android'
        ? Platform.Version >= 33
          ? [PERMISSIONS.ANDROID.READ_MEDIA_IMAGES]
          : [PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE, PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE]
        : [];

      if (permissions.length === 0) {
        return true;
      }

      let allGranted = true;
      for (const permission of permissions) {
        const result = await check(permission);
        if (result === RESULTS.DENIED) {
          const requestResult = await request(permission);
          if (requestResult !== RESULTS.GRANTED) {
            allGranted = false;
            Alert.alert('Permission Denied', 'Storage permission is required to save the report.', [{ text: 'OK' }]);
          }
        } else if (result === RESULTS.BLOCKED) {
          allGranted = false;
          Alert.alert(
            'Permission Blocked',
            'Storage permission is blocked. Please enable it in Settings.',
            [{ text: 'OK', onPress: () => Linking.openSettings() }]
          );
        }
      }
      return allGranted;
    } catch (error) {
      console.error('Error requesting permission:', error);
      Alert.alert('Error', 'Failed to request permissions.');
      return false;
    }
  };

  // Upload signature to MongoDB
  const uploadSignatureToMongoDB = async (signatureData, userId) => {
    try {
      if (!signatureData || !signatureData.startsWith('data:image')) {
        throw new Error('Invalid signature data');
      }

      const base64Data = signatureData.split(',')[1];
      const buffer = Buffer.from(base64Data, 'base64');
      const tempPath = `${RNFS.TemporaryDirectoryPath}/signature_${Date.now()}.png`;
      await RNFS.writeFile(tempPath, buffer.toString('binary'), 'ascii');

      // Create FormData for signature upload
      const formData = new FormData();
      formData.append('signature', {
        uri: Platform.OS === 'android' ? `file://${tempPath}` : tempPath,
        type: 'image/png',
        name: `signature_${Date.now()}.png`,
      });

      // Upload to MongoDB
      const response = await InspectionAPI.uploadSignature(userId, formData);
      
      await RNFS.unlink(tempPath).catch((err) => console.warn('Error deleting temp signature file:', err));
      
      if (response.success) {
        console.log('Signature uploaded to MongoDB:', response.downloadUrl);
        return response.downloadUrl;
      } else {
        throw new Error('Failed to upload signature to MongoDB');
      }
    } catch (error) {
      console.error('Error uploading signature to MongoDB:', error);
      throw error;
    }
  };

  // Upload all individual signatures to MongoDB
  const uploadIndividualSignaturesToMongoDB = async (userId) => {
    try {
      const uploadedSignatures = {};
      
      for (const [key, signatureData] of Object.entries(individualSignatures)) {
        if (signatureData && individualSignatureStatus[key]) {
          try {
            const downloadUrl = await uploadSignatureToMongoDB(signatureData, userId);
            uploadedSignatures[key] = downloadUrl;
            console.log(`✅ Uploaded signature for ${key}:`, downloadUrl);
          } catch (uploadError) {
            console.error(`Failed to upload signature for ${key}:`, uploadError);
            // Keep the base64 data if upload fails
            uploadedSignatures[key] = signatureData;
          }
        }
      }
      
      return uploadedSignatures;
    } catch (error) {
      console.error('Error uploading individual signatures:', error);
      return individualSignatures; // Return original signatures if upload fails
    }
  };

  const generatePDFWithHtml = async (
    report,
    {
      signatureFieldsOverride,
      signatureValuesOverride,
      individualSignaturesOverride,
      individualSignatureStatusOverride,
      selectedSignatureTypesOverride,
    } = {},
  ) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};

      // Helpers to match in-app formatting in PDF
      const toTitleCase = (value) => {
        if (value === undefined || value === null) return '';
        return String(value)
          .replace(/_/g, ' ')
          .split(' ')
          .map((w) => (w ? w.charAt(0).toUpperCase() + w.slice(1).toLowerCase() : ''))
          .join(' ');
      };
      const toUpperUnderscore = (value) => {
        if (value === undefined || value === null) return '';
        return String(value).replace(/_/g, ' ').toUpperCase();
      };

      // Debug: Log signature fields for PDF generation
      console.log('=== PDF Generation Debug ===');
      console.log('signatureFields for PDF:', signatureFieldsOverride || signatureFields);
      console.log('signatureValues for PDF:', signatureValuesOverride || signatureValues);
      console.log('individualSignatureStatus for PDF:', individualSignatureStatusOverride || individualSignatureStatus);
      console.log('individualSignatures for PDF:', individualSignaturesOverride || individualSignatures);
      console.log('selectedSignatureTypes for PDF:', selectedSignatureTypesOverride || selectedSignatureTypes);
      
      // Debug: Log header input values
      console.log('=== Header Input Values Debug ===');
      console.log('step2.headerInputValues:', step2.headerInputValues);
      console.log('step1.selectedEquipment:', step1.selectedEquipment);
      
      // Use signature data from report instead of component state
      const reportSignatureFields =
        signatureFieldsOverride || step3.signatureFields || signatureFields;
      const reportSignatureValues =
        signatureValuesOverride || step3.signatureValues || signatureValues;
      const reportIndividualSignatures =
        individualSignaturesOverride || step3.individualSignatures || individualSignatures;
      const reportIndividualSignatureStatus =
        individualSignatureStatusOverride || step3.individualSignatureStatus || individualSignatureStatus;
      const reportSelectedSignatureTypes =
        selectedSignatureTypesOverride || step3.selectedSignatureTypes || selectedSignatureTypes;
      
      console.log('=== Using Report Signature Data ===');
      console.log('reportSignatureFields:', reportSignatureFields);
      console.log('reportSignatureValues:', reportSignatureValues);
      console.log('reportIndividualSignatures:', reportIndividualSignatures);
      console.log('reportIndividualSignatureStatus:', reportIndividualSignatureStatus);
      
      // Debug: Check if signatures exist
      console.log('=== Signature Debug ===');
      console.log('step3.signatureStatus:', step3.signatureStatus);
      console.log('step3.signatureData:', step3.signatureData);
      console.log('reportIndividualSignatures:', reportIndividualSignatures);
      console.log('reportIndividualSignatureStatus:', reportIndividualSignatureStatus);

      let checklistRows = '';
      const sections = step2.sections || [];
      const selections = step2.selections || {};
      const notes = step2.notes || {};

      // Build dynamic header info tables from Step 2 headerInputValues
      const headerPairs = Object.entries(step2.headerInputValues || {});
      const formattedHeaderPairs = headerPairs.map(([key, val]) => ({
        label: String(key).replace(/_/g, ' ').toUpperCase(),
        value: val ?? 'N/A',
      }));

      // Add Equipment Name and Equipment Model Number to header
      if (step1.selectedEquipment) {
        formattedHeaderPairs.push({
          label: 'EQUIPMENT NAME',
          value: step1.selectedEquipment.equipmentName || step1.selectedEquipment.vehicleNumber || 'N/A'
        });
        formattedHeaderPairs.push({
          label: 'EQUIPMENT MODEL NUMBER',
          value: step1.selectedEquipment.model || step1.selectedEquipment.vehicleNumber || 'N/A'
        });
      }

      const half = Math.ceil(formattedHeaderPairs.length / 2);
      const leftHeaders = formattedHeaderPairs.slice(0, half);
      const rightHeaders = formattedHeaderPairs.slice(half);

      const renderHeaderTable = (rows) => {
        if (rows.length === 0) return '<table></table>';
        return `
          <table>
            ${rows
              .map(
                (r) => `
                <tr>
                  <td>${r.label}</td>
                  <td>${r.value}</td>
                </tr>`
              )
              .join('')}
          </table>
        `;
      };

      const dynamicGeneralInfoHtml = formattedHeaderPairs.length > 0
        ? `
        <div class="general-info-section">
          <div class="info-box">
            ${renderHeaderTable(leftHeaders)}
          </div>
          <div class="info-box">
            ${renderHeaderTable(rightHeaders)}
          </div>
        </div>
      `
        : '';

      // Debug: Log the sections and selections for PDF generation
      console.log('=== PDF Checklist Items Debug ===');
      console.log('Sections:', sections);
      console.log('Selections:', selections);
      console.log('Notes:', notes);

      // Include ALL items that have been inspected (have a selection)
      const inspectedItems = sections.flatMap((section) =>
        (section.subItems || []).filter((item) => 
          selections[item.key] !== undefined && selections[item.key] !== null
        )
      );

      console.log('Inspected items:', inspectedItems);
      console.log('Inspected items count:', inspectedItems.length);

      if (inspectedItems.length === 0) {
        checklistRows += `
          <tr>
            <td colspan="6">No inspected items available</td>
          </tr>
        `;
      } else {
        // Sort items by classification (A, B, C) and then by their original order
        const sortedItems = inspectedItems.sort((a, b) => {
          // Find sections for both items
          const sectionA = sections.find(section => 
            section.subItems && section.subItems.some(subItem => subItem.key === a.key)
          );
          const sectionB = sections.find(section => 
            section.subItems && section.subItems.some(subItem => subItem.key === b.key)
          );
          
          // Get section names for comparison
          const sectionNameA = sectionA?.name || a.category || 'C';
          const sectionNameB = sectionB?.name || b.category || 'C';
          
          const order = { 'A': 1, 'B': 2, 'C': 3 };
          const aOrder = order[sectionNameA] || 4;
          const bOrder = order[sectionNameB] || 4;
          
          if (aOrder !== bOrder) {
            return aOrder - bOrder;
          }
          return (a.sortOrder || 0) - (b.sortOrder || 0);
        });

        console.log('Sorted items by classification:', sortedItems);

        // Group items by section for better organization
        const itemsBySection = {};
        sortedItems.forEach(item => {
          const itemSection = sections.find(section => 
            section.subItems && section.subItems.some(subItem => subItem.key === item.key)
          );
          const sectionName = itemSection?.name || item.category || 'Unknown';
          
          if (!itemsBySection[sectionName]) {
            itemsBySection[sectionName] = [];
          }
          itemsBySection[sectionName].push(item);
        });

        // Generate rows with section headers for non-A/B/C sections
        Object.entries(itemsBySection).forEach(([sectionName, sectionItems]) => {
          const isABC = /^[ABC]$/.test(sectionName);
          
          if (!isABC) {
            // Add section header row for non-A/B/C sections
            checklistRows += `
              <tr style="background-color: #f2f2f2; font-weight: bold;">
                <td style="text-align: left; padding: 6px 8px; font-weight: bold;">${toTitleCase(sectionName)}</td>
                <td style="text-align: center; padding: 6px 8px; font-weight: bold;"></td>
                <td style="text-align: center; padding: 6px 8px; font-weight: bold;"></td>
                <td style="text-align: center; padding: 6px 8px; font-weight: bold;"></td>
                <td style="text-align: center; padding: 6px 8px; font-weight: bold;"></td>
                <td style="text-align: left; padding: 6px 8px; font-weight: bold;"></td>
              </tr>
            `;
          }
          
          // Add items for this section
          sectionItems.forEach((item, index) => {
            const selection = selections[item.key] || 'N/A';
            const isOkay = selection === 'Okay' || selection === 'Ok';
            const isFaulty = selection === 'Faulty' || selection === 'Reject' || selection === 'Not Ok';
            const isNA = selection === 'N/A';
            const comment = (notes && notes[item.key]) ? notes[item.key] : '';
            
            // For A/B/C sections, show the class. For others, leave it empty
            const displayClass = isABC ? sectionName : '';
            
            console.log(`Item ${index + 1}: ${item.itemName} - Section: ${sectionName} - Classification: ${displayClass} - Selection: ${selection}`);
            
            checklistRows += `
              <tr>
                <td style="text-align: left; padding: 6px 8px;">${toUpperUnderscore(item.itemName || 'Unknown')}</td>
                <td style="text-align: center; padding: 6px 8px;">${displayClass}</td>
                <td style="text-align: center; padding: 6px 8px;">${isOkay ? '✓' : ''}</td>
                <td style="text-align: center; padding: 6px 8px;">${isFaulty ? 'X' : ''}</td>
                <td style="text-align: center; padding: 6px 8px;">${isNA ? '—' : ''}</td>
                <td style="text-align: left; padding: 6px 8px;">${comment || ''}</td>
              </tr>
            `;
          });
        });
      }

      // Generate signature HTML - check both main signature and individual signatures
      let signatureHtml = 'Not Provided';
      
      // Check for main signature first
      if (step3.signatureStatus && step3.signatureData) {
        signatureHtml = `<img src="${step3.signatureData}" alt="Inspector's Signature" style="width: 120px; height: 60px; object-fit: contain;"/>`;
      } 
      // Check for individual signatures if no main signature
      else if (reportIndividualSignatures && Object.keys(reportIndividualSignatures).length > 0) {
        const firstSignatureKey = Object.keys(reportIndividualSignatures)[0];
        const firstSignature = reportIndividualSignatures[firstSignatureKey];
        if (firstSignature && reportIndividualSignatureStatus[firstSignatureKey]) {
          signatureHtml = `<img src="${firstSignature}" alt="Signature" style="width: 120px; height: 60px; object-fit: contain;"/>`;
        }
      }
      
      console.log('Generated signatureHtml:', signatureHtml);

      // Generate dynamic signature section based on selected signature fields
      let signatureSectionHtml = '';
      
      // Debug: Check if we have signature fields
      console.log('Building signature section for PDF...');
      console.log('reportSignatureFields.length:', reportSignatureFields.length);
      console.log('step3.signatureFields:', step3.signatureFields);
      console.log('step3.signatureValues:', step3.signatureValues);
      
      if (reportSignatureFields.length > 0) {
        console.log('Using reportSignatureFields for PDF signature section');
        signatureSectionHtml = `
          <div class="signature-section">
            <table style="width: 100%; border-collapse: collapse; margin-top: 20px;">
        `;
        
        reportSignatureFields.forEach((field, index) => {
          console.log(`Processing signature field ${index}:`, field);
          const fieldValue = reportSignatureValues[field.nameKey] || step1.inspector || 'N/A';
          const signatureKey = field.signatureKey || field.key;
          const hasIndividualSignature = reportIndividualSignatureStatus[signatureKey] && reportIndividualSignatures[signatureKey];
          const signatureData = hasIndividualSignature ? reportIndividualSignatures[signatureKey] : null;
          
          console.log(`Field ${field.label}: value=${fieldValue}, hasSignature=${hasIndividualSignature}, signatureKey=${signatureKey}`);
          console.log(`Signature data for ${signatureKey}:`, signatureData);
          
          signatureSectionHtml += `
            <tr>
              <td style="border: 1px solid #222; padding: 8px; font-weight: bold; width: 30%;">${field.label}:</td>
              <td style="border: 1px solid #222; padding: 8px; width: 40%;">${fieldValue}</td>
              <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                ${hasIndividualSignature && signatureData
                  ? `<img src="${signatureData}" alt="Signature" style="width: 120px; height: 60px; object-fit: contain; border: 1px solid #ccc;"/>`
                  : 'Not Provided'}
              </td>
            </tr>
            <tr>
              <td style="border: 1px solid #222; padding: 8px; font-weight: bold;"></td>
              <td style="border: 1px solid #222; padding: 8px;"></td>
            </tr>
          `;
        });
        
        signatureSectionHtml += `
            </table>
          </div>
        `;
      } else if (step3.signatureFields && step3.signatureFields.length > 0) {
        // Use signature fields from step3 if available
        console.log('Using step3.signatureFields for PDF signature section');
        signatureSectionHtml = `
          <div class="signature-section">
            <table style="width: 100%; border-collapse: collapse; margin-top: 20px;">
        `;
        
        step3.signatureFields.forEach((field, index) => {
          console.log(`Processing step3 signature field ${index}:`, field);
          const fieldValue = step3.signatureValues[field.nameKey] || step1.inspector || 'N/A';
          const signatureKey = field.signatureKey || field.key;
          const hasIndividualSignature = step3.individualSignatureStatus[signatureKey] && step3.individualSignatures[signatureKey];
          const signatureData = hasIndividualSignature ? step3.individualSignatures[signatureKey] : null;
          
          console.log(`Field ${field.label}: value=${fieldValue}, hasSignature=${hasIndividualSignature}, signatureKey=${signatureKey}`);
          console.log(`Signature data for ${signatureKey}:`, signatureData);
          
          signatureSectionHtml += `
            <tr>
              <td style="border: 1px solid #222; padding: 8px; font-weight: bold; width: 30%;">${field.label}:</td>
              <td style="border: 1px solid #222; padding: 8px; width: 40%;">${fieldValue}</td>
              <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                ${hasIndividualSignature && signatureData
                  ? `<img src="${signatureData}" alt="Signature" style="width: 120px; height: 60px; object-fit: contain; border: 1px solid #ccc;"/>`
                  : 'Not Provided'}
              </td>
            </tr>
            <tr>
              <td style="border: 1px solid #222; padding: 8px; font-weight: bold;"></td>
              <td style="border: 1px solid #222; padding: 8px;"></td>
            </tr>
          `;
        });
        
        signatureSectionHtml += `
            </table>
          </div>
        `;
      } else {
        console.log('No reportSignatureFields, using fallback signature section');
        // Default signature section with footer inputs from checklist
        const footerInputs = checklistMeta?.footer_inputs || {};
        const footerInputKeys = Object.keys(footerInputs);
        
        if (footerInputKeys.length > 0) {
          signatureSectionHtml = `
            <div class="signature-section">
              <table style="width: 100%; border-collapse: collapse; margin-top: 20px;">
          `;
          
          footerInputKeys.forEach((key) => {
            const label = footerInputs[key] || key;
            
            // Skip footer items that only contain "Signature" text
            if (label.toLowerCase().trim() === 'signature') {
              return;
            }
            
            const value = reportSignatureValues[key] || step1.inspector || 'N/A';
            const hasIndividualSignature = reportIndividualSignatureStatus[key] && reportIndividualSignatures[key];
            const signatureData = hasIndividualSignature ? reportIndividualSignatures[key] : null;
            
            console.log(`Fallback field ${key}: value=${value}, hasSignature=${hasIndividualSignature}, signatureData=${signatureData}`);
            
            signatureSectionHtml += `
              <tr>
                <td style="border: 1px solid #222; padding: 8px; font-weight: bold; width: 30%;">${label}:</td>
                <td style="border: 1px solid #222; padding: 8px; width: 40%;">${value}</td>
                <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                  ${hasIndividualSignature && signatureData
                    ? `<img src="${signatureData}" alt="Signature" style="width: 120px; height: 60px; object-fit: contain; border: 1px solid #ccc;"/>`
                    : 'Not Provided'}
                </td>
              </tr>
              <tr>
                <td style="border: 1px solid #222; padding: 8px; font-weight: bold;"></td>
                <td style="border: 1px solid #222; padding: 8px;"></td>
              </tr>
            `;
          });
          
          signatureSectionHtml += `
              </table>
            </div>
          `;
        } else {
          // Fallback to default signature section
          signatureSectionHtml = `
            <div class="signature-section">
              <table style="width: 100%; border-collapse: collapse; margin-top: 20px;">
                <tr>
                  <td style="border: 1px solid #222; padding: 8px; font-weight: bold; width: 30%;">DRILLER'S NAME:</td>
                  <td style="border: 1px solid #222; padding: 8px; width: 40%;">${step1.inspector || 'N/A'}</td>
                  <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                    ${signatureHtml}
                  </td>
                </tr>
                <tr>
                  <td style="border: 1px solid #222; padding: 8px; font-weight: bold;">DRILLER'S SIGNATURE:</td>
                  <td style="border: 1px solid #222; padding: 8px;"></td>
                </tr>
                <tr>
                  <td style="border: 1px solid #222; padding: 8px; font-weight: bold;">MECHANIC'S NAME:</td>
                  <td style="border: 1px solid #222; padding: 8px; width: 40%;">${step1.inspector || 'N/A'}</td>
                  <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                    ${signatureHtml}
                  </td>
                </tr>
                <tr>
                  <td style="border: 1px solid #222; padding: 8px; font-weight: bold;">MECHANIC'S SIGNATURE:</td>
                  <td style="border: 1px solid #222; padding: 8px;"></td>
                </tr>
              </table>
            </div>
          `;
        }
      }

      console.log('Final signatureSectionHtml:', signatureSectionHtml);
      
      // If no signature section was generated, add a default one
      if (!signatureSectionHtml || signatureSectionHtml.trim() === '') {
        console.log('No signature section generated, adding default signature section');
        signatureSectionHtml = `
          <div class="signature-section">
            <table style="width: 100%; border-collapse: collapse; margin-top: 20px;">
              <tr>
                <td style="border: 1px solid #222; padding: 8px; font-weight: bold; width: 30%;">DRILLER'S NAME:</td>
                <td style="border: 1px solid #222; padding: 8px; width: 40%;">${step1.inspector || 'N/A'}</td>
                <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                  ${signatureHtml}
                </td>
              </tr>
              <tr>
                <td style="border: 1px solid #222; padding: 8px; font-weight: bold;">DRILLER'S SIGNATURE:</td>
                <td style="border: 1px solid #222; padding: 8px;"></td>
              </tr>
              <tr>
                <td style="border: 1px solid #222; padding: 8px; font-weight: bold;">MECHANIC'S NAME:</td>
                <td style="border: 1px solid #222; padding: 8px; width: 40%;">${step1.inspector || 'N/A'}</td>
                <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                  ${signatureHtml}
                </td>
              </tr>
              <tr>
                <td style="border: 1px solid #222; padding: 8px; font-weight: bold;">MECHANIC'S SIGNATURE:</td>
                <td style="border: 1px solid #222; padding: 8px;"></td>
              </tr>
            </table>
          </div>
        `;
      }

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8"/>
          <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
          <title>${checklistMeta?.document_name || checklistMeta?.title || 'Daily DD Rig Pre-start Checklist'}</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 20px;
              padding: 0;
              font-size: 12px;
              line-height: 1.4;
            }
            .header-section {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              margin-bottom: 20px;
            }
            .logo-section {
              flex: 1;
            }
            .logo-section img {
              width: 140px;
              height: 50px;
              object-fit: contain;
            }
            .document-info {
              flex: 1;
              margin-left: 20px;
            }
            .document-info table {
              width: 100%;
              border: 1px solid #222;
              border-collapse: collapse;
              font-size: 11px;
            }
            .document-info td {
              border: 1px solid #222;
              padding: 4px 6px;
              text-align: left;
            }
            .document-info td:first-child {
              font-weight: bold;
              background-color: #f5f5f5;
              width: 40%;
            }
            .title-section {
              text-align: center;
              margin: 20px 0;
            }
            .title-section h1 {
              margin: 0;
              font-size: 18px;
              font-weight: bold;
              color: #333;
            }
            .request-number {
              font-size: 14px;
              font-weight: bold;
              margin-top: 5px;
              color: #333;
            }
            .general-info-section {
              display: flex;
              justify-content: space-between;
              margin: 20px 0;
            }
            .info-box {
              width: 48%;
            }
            .info-box table {
              width: 100%;
              border: 1px solid #222;
              border-collapse: collapse;
              font-size: 11px;
            }
            .info-box td {
              border: 1px solid #222;
              padding: 4px 6px;
              text-align: left;
            }
            .info-box td:first-child {
              font-weight: bold;
              background-color: #f5f5f5;
              width: 40%;
            }
            .classification-rules {
              margin: 20px 0;
            }
            .classification-rules table {
              width: 100%;
              border: 1px solid #222;
              border-collapse: collapse;
              font-size: 11px;
            }
            .classification-rules td {
              border: 1px solid #222;
              padding: 6px 8px;
              text-align: left;
              font-weight: bold;
            }
            .checklist-table {
              width: 100%;
              border-collapse: collapse;
              margin: 20px 0;
              font-size: 11px;
            }
            .checklist-table th {
              background-color: #f2f2f2;
              border: 1px solid #222;
              padding: 8px 6px;
              text-align: center;
              font-weight: bold;
            }
            .checklist-table td {
              border: 1px solid #222;
              padding: 6px 8px;
            }
            .checklist-table th:nth-child(1), .checklist-table td:nth-child(1) {
              width: 35%;
              text-align: left;
            }
            .checklist-table th:nth-child(2), .checklist-table td:nth-child(2) {
              width: 10%;
              text-align: center;
            }
            .checklist-table th:nth-child(3), .checklist-table td:nth-child(3),
            .checklist-table th:nth-child(4), .checklist-table td:nth-child(4),
            .checklist-table th:nth-child(5), .checklist-table td:nth-child(5) {
              width: 12%;
              text-align: center;
            }
            .checklist-table th:nth-child(6), .checklist-table td:nth-child(6) {
              width: 19%;
              text-align: left;
            }
            .remarks-row td {
              text-align: left;
              font-weight: bold;
            }
            .signature-section {
              margin-top: 30px;
            }
            .signature-section table {
              width: 100%;
              border: none;
              font-size: 12px;
            }
            .signature-section td {
              border: none;
              padding: 8px;
              text-align: left;
            }
            .signature-section .label {
              font-weight: bold;
              width: 150px;
            }
            .signature-section .signature-cell {
              text-align: right;
              vertical-align: middle;
              padding-right: 30px;
            }
            .signature-section .signature-image {
              width: 120px;
              height: 60px;
              border: 1px solid #ccc;
            }
            .footer-inputs-section {
              margin-top: 30px;
            }
            .footer-inputs-section h3 {
              color: #015185;
              margin-bottom: 10px;
            }
            .footer-inputs-section table {
              width: 100%;
              border-collapse: collapse;
              font-size: 11px;
            }
            .footer-inputs-section td {
              border: 1px solid #222;
              padding: 6px 8px;
            }
            .footer-inputs-section td:first-child {
              font-weight: bold;
              background-color: #f5f5f5;
              width: 30%;
            }
          </style>
        </head>
        <body>
          <div class="header-section">
            <div class="logo-section">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo"/>
            </div>
            <div class="document-info">
              <table>
                <tr>
                  <td>Document Name</td>
                  <td>${checklistMeta?.document_name || checklistMeta?.title || 'HSE-FOR-8.1.1 DD Prestart Check'}</td>
                </tr>
                <tr>
                  <td>Document ID</td>
                  <td>${checklistMeta?.document_id || 'HSE-028'}</td>
                </tr>
                <tr>
                  <td>Version Date</td>
                  <td>${checklistMeta?.version_date || '05/08/2021'}</td>
                </tr>
                <tr>
                  <td>Version</td>
                  <td>${checklistMeta?.version || '2'}</td>
                </tr>
                <tr>
                  <td>Approved</td>
                  <td>${checklistMeta?.approved_by || 'HSM'}</td>
                </tr>
                <tr>
                  <td>NO</td>
                  <td>${step1.requestNumber || '0001'}</td>
                </tr>
              </table>
            </div>
          </div>

          <div class="title-section">
            <h1>${ checklistMeta.Header_title || checklistMeta?.document_name || checklistMeta?.title || 'CHECKLIST'} </h1>
          </div>

          ${dynamicGeneralInfoHtml || `
            <div class=\"general-info-section\"> 
              <div class=\"info-box\"> 
                <table>
                  <tr><td>DATE</td><td>${step2.headerInputValues?.date || new Date().toLocaleDateString()}</td></tr>
                  <tr><td>RIG</td><td>${step2.headerInputValues?.rig || step1.selectedEquipment?.equipmentName || step1.selectedEquipment?.rig || 'N/A'}</td></tr>
                  <tr><td>ODOMETER READING</td><td>${step2.headerInputValues?.odometer_reading || step2.headerInputValues?.meter_reading || step1.selectedEquipment?.meterReading || step1.meterReading || 'N/A'}</td></tr>
                  <tr><td>FUEL LEVEL</td><td>${step2.headerInputValues?.fuel_level || step1.selectedEquipment?.fuelLevel || 'N/A'}</td></tr>
                </table>
              </div>
              <div class=\"info-box\"> 
                <table>
                  <tr><td>SITE / PROJECT</td><td>${step1.project || 'N/A'}</td></tr>
                  <tr><td>INSPECTOR</td><td>${step1.inspector || 'N/A'}</td></tr>
                  <tr><td>COMPRESSOR No</td><td>${step1.selectedEquipment?.equipmentName || 'N/A'}</td></tr>
                  <tr><td>HOURS</td><td>${step1.selectedEquipment?.meterReading || step1.meterReading || 'N/A'} ${step1.selectedEquipment?.meterUnit || 'Hours'}</td></tr>
                </table>
              </div>
            </div>
          `}

          <div class="classification-rules">
            <table>
              <tr>
                <td>1. ALL DEFECTS UNDER "A" CLASSIFICATION TO BE ATTENDED IMMEDIATELY</td>
              </tr>
              <tr>
                <td>2. ALL DEFECTS UNDER "B" CLASSIFICATION TO BE ATTENDED WITHIN 48 HOURS</td>
              </tr>
              <tr>
                <td>3. ALL DEFECTS UNDER "C" CLASSIFICATION TO BE ATTENDED TO AS SOON AS POSSIBLE</td>
              </tr>
            </table>
          </div>

          <table class="checklist-table">
            <thead>
              <tr>
                <th>DESCRIPTION</th>
                <th>CLASS</th>
                <th>IN ORDER (✓)</th>
                <th>DEFECTIVE (X)</th>
                <th>N/A</th>
                <th>COMMENT</th>
              </tr>
            </thead>
            <tbody>
              ${checklistRows}
              <tr class="remarks-row">
                <td>REMARKS</td>
                <td colspan="5">${step3.additionalNote || 'N/A'}</td>
              </tr>
            </tbody>
          </table>

          ${signatureSectionHtml}
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `Daily_DD_Rig_Prestart_Checklist_${report.id}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  // Generate proper request number from MongoDB
  const generateMongoRequestNumber = async (userId, country, project) => {
    try {
      // Get total count from the new optimized endpoint
      console.log('🔢 Generating request number for:', { userId, country, project });
      
      const response = await InspectionAPI.getTotalCount();
      let nextNumber = 1;
      
      if (response.success && response.total !== undefined) {
        nextNumber = response.total + 1;
        console.log('📊 Total reports in database:', response.total);
        console.log('📈 Next number will be:', nextNumber);
      } else {
        console.log('⚠️ Count response not successful, trying fallback...');
        // Fallback to getAllReports
        const fallbackResponse = await InspectionAPI.getAllReports({ page: 1, limit: 1 });
        if (fallbackResponse.success && fallbackResponse.pagination) {
          nextNumber = (fallbackResponse.pagination.total || 0) + 1;
          console.log('📊 Total inspection reports (fallback):', fallbackResponse.pagination.total);
        }
      }
      
      // Generate country and project codes with proper mapping
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      
      // Generate project code based on project name
      let projectCode = 'XXX';
      if (project) {
        switch (project.toLowerCase()) {
          case 'lubumbashi':
            projectCode = 'LUB';
            break;
          case 'musompo':
            projectCode = 'MUS';
            break;
          case 'kansanshi':
            projectCode = 'KAN';
            break;
          case 'kalumbila':
            projectCode = 'KAL';
            break;
          case 'kimteto':
            projectCode = 'KIM';
            break;
          case 'kobold':
            projectCode = 'KOB';
            break;
          case 'fqm exploration':
            projectCode = 'FQM';
            break;
          case 'mimosa':
            projectCode = 'MIM';
            break;
          case 'ime':
            projectCode = 'IME';
            break;
          case 'kamoa':
            projectCode = 'KAM';
            break;
          default:
            // Fallback to first 3 characters
            projectCode = project.slice(0, 3).toUpperCase();
            break;
        }
      }
      
      const requestNumber = `${countryCode}-${projectCode}-IR-${String(nextNumber).padStart(3, '0')}`;
      console.log('✅ Generated request number:', requestNumber);
      return requestNumber;
    } catch (error) {
      console.error('❌ Error generating request number:', error);
      // Fallback for offline or error scenarios
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      
      // Generate project code based on project name
      let projectCode = 'XXX';
      if (project) {
        switch (project.toLowerCase()) {
          case 'lubumbashi':
            projectCode = 'LUB';
            break;
          case 'musompo':
            projectCode = 'MUS';
            break;
          case 'kansanshi':
            projectCode = 'KAN';
            break;
          case 'kalumbila':
            projectCode = 'KAL';
            break;
          case 'kimteto':
            projectCode = 'KIM';
            break;
          case 'kobold':
            projectCode = 'KOB';
            break;
          case 'fqm exploration':
            projectCode = 'FQM';
            break;
          case 'mimosa':
            projectCode = 'MIM';
            break;
          case 'ime':
            projectCode = 'IME';
            break;
          case 'kamoa':
            projectCode = 'KAM';
            break;
          default:
            projectCode = project.slice(0, 3).toUpperCase();
            break;
        }
      }
      
      const requestNumber = `${countryCode}-${projectCode}-IR-001`;
      console.log('🔄 Fallback request number:', requestNumber);
      return requestNumber;
    }
  };

  // Upload PDF to MongoDB
  const uploadPdfToMongoDB = async (pdfPath, userId) => {
    try {
      console.log('Uploading PDF to MongoDB...', pdfPath);
      
      // Create FormData
      const formData = new FormData();
      formData.append('pdf', {
        uri: Platform.OS === 'android' ? `file://${pdfPath}` : pdfPath,
        type: 'application/pdf',
        name: `inspection_${Date.now()}.pdf`,
      });
      
      // Upload to MongoDB inspection handler
      const response = await InspectionAPI.uploadPdf(userId, formData);
      
      if (response.success) {
        const { downloadUrl, fileName } = response;
        let normalizedUrl = downloadUrl;

        if (normalizedUrl && !/^https?:\/\//i.test(normalizedUrl)) {
          if (fileName && typeof InspectionAPI.getDownloadUrl === 'function') {
            normalizedUrl = InspectionAPI.getDownloadUrl(userId, fileName);
          } else {
            const baseUrl = 'https://api.titandrillingzm.com:6004';
            const startsWithSlash = normalizedUrl.startsWith('/');
            normalizedUrl = `${baseUrl}${startsWithSlash ? '' : '/'}${normalizedUrl}`;
          }
        }

        console.log('PDF uploaded successfully:', normalizedUrl);
        return {
          downloadUrl: normalizedUrl,
          fileName,
          raw: response,
        };
      } else {
        throw new Error('Failed to upload PDF to MongoDB');
      }
    } catch (error) {
      console.error('Error uploading PDF to MongoDB:', error);
      throw error;
    }
  };

  const handleGenerateReport = async () => {
    if (!validateFields()) {
      return;
    }

    const hasPermission = await requestStoragePermission();
    if (!hasPermission) return;

    setSyncModalVisible(true);
    try {
      // Check network status
      const netInfo = await NetInfo.fetch();
      const isOnline = netInfo.isConnected;
      
      console.log('🌐 Network status:', isOnline ? 'ONLINE' : 'OFFLINE');
      
      // Get user ID from AsyncStorage
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('AdminLoginScreen');
        return;
      }

      const inspectionStep1Raw = await AsyncStorage.getItem('inspectionStep1');
      const inspectionResultsRaw = await AsyncStorage.getItem('inspectionResults');

      if (!inspectionStep1Raw || !inspectionResultsRaw) {
        Alert.alert('Error', 'Inspection data is missing.');
        return;
      }

      const inspectionStep1 = JSON.parse(inspectionStep1Raw);
      const inspectionResults = JSON.parse(inspectionResultsRaw);
      const latestInspectionResult = inspectionResults[inspectionResults.length - 1] || {};

      if (!inspectionStep1.country || !inspectionStep1.project || !inspectionStep1.inspector) {
        Alert.alert('Error', 'Incomplete data from Step 1.');
        return;
      }
      if (!latestInspectionResult.checklistTitle) {
        Alert.alert('Error', 'Incomplete data from Step 2.');
        return;
      }

      // Generate proper request number (offline or online)
      let requestNumber;
      if (isOnline) {
        requestNumber = await generateMongoRequestNumber(
          userId,
          inspectionStep1.country,
          inspectionStep1.project
        );
      } else {
        // Offline mode - generate from cached data
        const countryCode = inspectionStep1.country === 'Zambia' ? 'ZAM' : inspectionStep1.country === 'DRC' ? 'DRC' : 'XXX';
        const projectCode = inspectionStep1.project ? inspectionStep1.project.slice(0, 3).toUpperCase() : 'XXX';
        const offlineReports = await OfflineDataHelper.getOfflineReports();
        const nextNumber = offlineReports.length + 1;
        requestNumber = `${countryCode}-${projectCode}-IR-${String(nextNumber).padStart(3, '0')}-OFFLINE`;
        console.log('📴 Generated offline request number:', requestNumber);
      }
      
      const reportId = Date.now().toString();
      setReportId(reportId);

      // Calculate faulty items count
      const sections = latestInspectionResult.sections || [];
      const selections = latestInspectionResult.selections || {};
      const faultyCount = sections.flatMap((section) =>
        (section.subItems || []).filter((item) => 
          selections[item.key] === 'Faulty' || 
          selections[item.key] === 'Reject' || 
          selections[item.key] === 'Not Ok'
        )
      ).length;

      // Get user permissions from AsyncStorage
      let userPermissions = [];
      let userCountries = [];
      let userProjects = [];
      try {
        const userDataStr = await AsyncStorage.getItem('userData');
        if (userDataStr) {
          const userData = JSON.parse(userDataStr);
          const inspectionModule = userData.modules?.find((module) => module.module === 'inspectionReport');
          userPermissions = inspectionModule?.permissions || [];
          userCountries = userData.countries || [];
          userProjects = userData.projects || [];
          console.log('📋 User permissions loaded:', userPermissions);
          console.log('🌍 User countries:', userCountries);
          console.log('🏗️ User projects:', userProjects);
        }
      } catch (permError) {
        console.warn('⚠️ Could not load user permissions:', permError);
      }

      // Resolve meter reading from available sources (step 1 input takes precedence)
      const resolvedMeterReading =
        inspectionStep1?.meterReading ??
        latestInspectionResult?.headerInputValues?.meter_reading ??
        latestInspectionResult?.headerInputValues?.odometer_reading ??
        inspectionStep1?.selectedEquipment?.meterReading ??
        inspectionStep1?.currentMeterReading ??
        null;

      // Prepare report data for MongoDB
      const reportData = {
        _id: reportId,
        userId: userId,
        requestNumber: requestNumber,
        status: 'completed',
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        step1: {
          ...inspectionStep1,
          requestNumber: requestNumber,
        },
        step2: {
          sections: latestInspectionResult.sections || [],
          selections: latestInspectionResult.selections || {},
          selectedFiles: latestInspectionResult.selectedFiles || {},
          numericInputs: latestInspectionResult.numericInputs || {},
          notes: latestInspectionResult.notes || {},
          checklistTitle: latestInspectionResult.checklistTitle,
          headerInputValues: latestInspectionResult.headerInputValues || {},
          footerInputValues: latestInspectionResult.footerInputValues || {},
          faultyItems: faultyCount,
          timestamp: new Date().toISOString(),
        },
        step3: {
          faultyItems: faultyCount,
          overallCondition,
          equipmentStatus,
          priority,
          safeToUse,
          additionalNote,
          signatureStatus,
          signatureData,
          signatureFields,
          signatureValues,
          selectedSignatureType,
          selectedSignatureTypes,
          individualSignatures,
          individualSignatureStatus,
          createdAt: new Date().toISOString(),
        },
        // Store user permissions with report for access control
        userPermissions: userPermissions,
        userCountries: userCountries,
        userProjects: userProjects,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        createdBy: userId,
        lastModifiedBy: userId,
      };

      // Clean the data to remove any circular references or invalid data
      // But preserve the original structure for signatureFields
      const cleanReportData = {
        ...reportData,
        step3: {
          ...reportData.step3,
          signatureFields: signatureFields.map(field => ({
            key: field.key,
            label: field.label,
            roleName: field.roleName,
            required: field.required,
            showInput: field.showInput,
            showSignature: field.showSignature,
            nameKey: field.nameKey,
            signatureKey: field.signatureKey
          }))
        }
      };
      
      console.log('=== Saving Report to MongoDB ===');
      console.log('Report data cleaned, size:', JSON.stringify(cleanReportData).length);

      // Handle signatures and PDF differently based on connection
      let pdfPath;
      let pdfDownloadUrl = null;
      let uploadedIndividualSignatures = {};
      
      if (isOnline) {
        // ONLINE MODE: Upload signatures to MongoDB first
        console.log('🌐 Online: Uploading individual signatures...');
        uploadedIndividualSignatures = await uploadIndividualSignaturesToMongoDB(userId);
        
        // Update report data with uploaded signature URLs BEFORE generating PDF
        console.log('📝 Updating report data with uploaded signatures before PDF generation...');
        console.log('Uploaded signatures:', Object.keys(uploadedIndividualSignatures));
        cleanReportData.step3.individualSignatures = uploadedIndividualSignatures;
        cleanReportData.step3.individualSignatureStatus = individualSignatureStatus;
        cleanReportData.step3.signatureFields = signatureFields;
        cleanReportData.step3.signatureValues = signatureValues;
        
        // Debug: Log what's being passed to PDF generation
        console.log('=== PDF Generation Debug ===');
        console.log('signatureFields:', signatureFields.length);
        console.log('individualSignatures keys:', Object.keys(uploadedIndividualSignatures));
        console.log('individualSignatureStatus:', individualSignatureStatus);
        
        // Build signature data for PDF (prefer fresh base64 captures)
        const pdfSignatureMap = { ...individualSignatures };
        Object.keys(uploadedIndividualSignatures || {}).forEach((key) => {
          if (!pdfSignatureMap[key]) {
            pdfSignatureMap[key] = uploadedIndividualSignatures[key];
          }
        });

        // Generate PDF with uploaded signatures
        pdfPath = await generatePDFWithHtml(
          cleanReportData,
          {
            signatureFieldsOverride: signatureFields,
            signatureValuesOverride: signatureValues,
            individualSignaturesOverride: pdfSignatureMap,
            individualSignatureStatusOverride: individualSignatureStatus,
            selectedSignatureTypesOverride: selectedSignatureTypes,
          },
        );
        console.log('✅ PDF generated at:', pdfPath);

        // Upload PDF to MongoDB
        try {
          console.log('📤 Uploading PDF to MongoDB...');
          const pdfUploadResult = await uploadPdfToMongoDB(pdfPath, userId);
          pdfDownloadUrl = pdfUploadResult?.downloadUrl || null;
          const uploadedPdfFileName = pdfUploadResult?.fileName || null;
          console.log('✅ PDF uploaded to MongoDB:', pdfDownloadUrl);
          
          // Update report data with PDF URL
          cleanReportData.step3.pdfDownloadUrl = pdfDownloadUrl;
          if (uploadedPdfFileName) {
            cleanReportData.step3.pdfFileName = uploadedPdfFileName;
          }
          
          // ALSO store local PDF path in AsyncStorage for offline viewing
          try {
            const pdfPathKey = `inspection_pdf_local_${requestNumber}`;
            await AsyncStorage.setItem(pdfPathKey, pdfPath);
            console.log('✅ Stored local PDF path in AsyncStorage:', pdfPathKey);
          } catch (storageError) {
            console.warn('⚠️ Failed to store local PDF path:', storageError);
          }
        } catch (pdfError) {
          console.error('❌ PDF upload failed:', pdfError);
          Toast.show({
            type: 'error',
            text1: 'Warning',
            text2: 'PDF upload failed, but report will be saved',
            position: 'top',
          });
        }
      } else {
        // OFFLINE MODE: Keep signatures as base64 and save PDF locally
        console.log('📴 Offline: Generating PDF with local signatures...');
        uploadedIndividualSignatures = individualSignatures; // Use base64 signatures
        cleanReportData.step3.individualSignatures = uploadedIndividualSignatures;
        cleanReportData.step3.individualSignatureStatus = individualSignatureStatus;
        cleanReportData.step3.signatureFields = signatureFields;
        cleanReportData.step3.signatureValues = signatureValues;
        
        // Build signature data for PDF
        const pdfSignatureMap = { ...individualSignatures };

        // Generate PDF
        pdfPath = await generatePDFWithHtml(
          cleanReportData,
          {
            signatureFieldsOverride: signatureFields,
            signatureValuesOverride: signatureValues,
            individualSignaturesOverride: pdfSignatureMap,
            individualSignatureStatusOverride: individualSignatureStatus,
            selectedSignatureTypesOverride: selectedSignatureTypes,
          },
        );
        console.log('📱 PDF generated locally at:', pdfPath);
        
        // Save PDF path for offline access
        cleanReportData.step3.pdfLocalPath = pdfPath;
        
        // Save PDF reference in OfflineDataHelper
        await OfflineDataHelper.saveOfflinePDF(reportId, pdfPath);
        
        // ALSO store in AsyncStorage with request number key
        try {
          const pdfPathKey = `inspection_pdf_local_${requestNumber}`;
          await AsyncStorage.setItem(pdfPathKey, pdfPath);
          console.log('✅ Stored offline PDF path in AsyncStorage:', pdfPathKey);
        } catch (storageError) {
          console.warn('⚠️ Failed to store offline PDF path:', storageError);
        }
      }

      // Save report (online to MongoDB, offline to local storage)
      let response;
      
      if (isOnline) {
        // ONLINE: Save to MongoDB
        console.log('📤 Sending report data to MongoDB:', {
          userId,
          reportId,
          requestNumber,
          dataSize: JSON.stringify(cleanReportData).length,
          step1Keys: Object.keys(cleanReportData.step1 || {}),
          step2Keys: Object.keys(cleanReportData.step2 || {}),
          step3Keys: Object.keys(cleanReportData.step3 || {}),
          reportDataStructure: {
            _id: cleanReportData._id,
            userId: cleanReportData.userId,
            requestNumber: cleanReportData.requestNumber,
            status: cleanReportData.status
          }
        });
        
        response = await InspectionAPI.createInspection(userId, cleanReportData);
        
        console.log('📡 MongoDB response:', response);
      } else {
        // OFFLINE: Save to local storage
        console.log('📱 Offline: Saving report to local storage...');
        const saved = await OfflineDataHelper.saveOfflineReport(cleanReportData);
        
        if (saved) {
          response = { 
            success: true, 
            message: 'Report saved offline. Will sync when online.' 
          };
          console.log('✅ Report saved offline successfully');
        } else {
          throw new Error('Failed to save report offline');
        }
      }
      
      if (response.success) {
        console.log('✅ Report saved successfully:', isOnline ? 'MongoDB' : 'Offline');
        console.log('📎 PDF path:', pdfPath);
        if (isOnline) {
          console.log('📎 PDF URL for notifications:', pdfDownloadUrl);
        }
        
        // Send notifications only if online AND PDF is uploaded
        if (isOnline && pdfDownloadUrl) {
          // Send notifications for new inspection report (with PDF URL)
          // Capture pdfDownloadUrl in closure to ensure it's available
          const pdfUrlForNotifications = pdfDownloadUrl;
          
          // Wait a bit to ensure MongoDB has the PDF accessible
          setTimeout(() => {
            (async () => {
              try {
                console.log('🔔 Sending notifications for new inspection report...');
                console.log('📎 PDF URL for email attachment:', pdfUrlForNotifications);
                
                // Get inspector email from AsyncStorage (current user)
                let inspectorEmail = null;
                try {
                  const userDataStr = await AsyncStorage.getItem('userData');
                  if (userDataStr) {
                    const userData = JSON.parse(userDataStr);
                    inspectorEmail = userData.email || null;
                    console.log('📧 Inspector email:', inspectorEmail);
                  }
                } catch (err) {
                  console.warn('⚠️ Could not fetch inspector email:', err);
                }
                
                const notificationData = {
                  inspectionData: {
                    _id: reportId,
                    requestNumber: requestNumber,
                    equipmentName: inspectionStep1.selectedEquipment?.equipmentName || 'N/A',
                    country: inspectionStep1.country,
                    project: inspectionStep1.project,
                    faultyItems: faultyCount,
                    equipmentStatus: equipmentStatus,
                    overallCondition: overallCondition,
                    safeToUse: safeToUse,
                  },
                  createdByName: inspectionStep1.inspector || 'Inspector',
                  inspectorEmail: inspectorEmail,
                  pdfUrl: pdfUrlForNotifications, // Include PDF URL for email attachment
                  pdfDownloadUrl: pdfUrlForNotifications, // Alternative key for backend compatibility
                };
                
                console.log('📤 Sending notification data:', JSON.stringify(notificationData, null, 2));
                
                // Call Inspection Notifications service
                const notifResponse = await fetch('https://api.titandrillingzm.com:6019/api/inspection-notifications/new-inspection', {
                  method: 'POST',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify(notificationData),
                  timeout: 15000, // Increased timeout for email with attachment
                });
                
                if (notifResponse.ok) {
                  const notifResult = await notifResponse.json();
                  console.log('✅ Inspection notifications sent successfully');
                  console.log(`  📱 Push: ${notifResult.results?.push?.sent || 0} device(s)`);
                  console.log(`  📧 Emails to Recipients: ${notifResult.results?.emails?.sent || 0}`);
                  console.log(`  📧 Inspector Email: ${notifResult.results?.inspectorEmail?.success ? 'Sent' : 'Failed'}`);
                  console.log(`  📎 PDF Attachment: ${notifResult.results?.pdfAttachment ? 'Included' : 'Not included'}`);
                  console.log(`  💾 MongoDB: ${notifResult.results?.mongodb?.saved || 0} notification(s)`);
                  console.log(`  👥 Recipients: ProAdmins=${notifResult.recipients?.proAdmins || 0}, Supervisors=${notifResult.recipients?.supervisors || 0}, Managers=${notifResult.recipients?.managers || 0}`);
                } else {
                  const errorText = await notifResponse.text();
                  console.error('❌ Failed to send inspection notifications:', errorText);
                }
              } catch (notifError) {
                console.error('❌ Error sending inspection notifications:', notifError);
                // Don't fail report creation if notifications fail
              }
            })();
          }, 1000); // Increased delay to ensure PDF is accessible
        } else if (isOnline && !pdfDownloadUrl) {
          console.log('⚠️ Online but PDF URL not available - notifications will be sent without PDF attachment');
          // Send notifications without PDF
          setTimeout(() => {
            (async () => {
              try {
                console.log('🔔 Sending notifications (without PDF)...');
                
                const userDataStr = await AsyncStorage.getItem('userData');
                let inspectorEmail = null;
                if (userDataStr) {
                  const userData = JSON.parse(userDataStr);
                  inspectorEmail = userData.email || null;
                }
                
                const notificationData = {
                  inspectionData: {
                    _id: reportId,
                    requestNumber: requestNumber,
                    equipmentName: inspectionStep1.selectedEquipment?.equipmentName || 'N/A',
                    country: inspectionStep1.country,
                    project: inspectionStep1.project,
                    faultyItems: faultyCount,
                    equipmentStatus: equipmentStatus,
                    overallCondition: overallCondition,
                    safeToUse: safeToUse,
                  },
                  createdByName: inspectionStep1.inspector || 'Inspector',
                  inspectorEmail: inspectorEmail,
                };
                
                await fetch('https://api.titandrillingzm.com:6019/api/inspection-notifications/new-inspection', {
                  method: 'POST',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify(notificationData),
                  timeout: 10000,
                });
              } catch (err) {
                console.error('❌ Error sending notifications:', err);
              }
            })();
          }, 500);
        } else {
          console.log('📴 Offline: Notifications will be sent when synced');
        }
        
        // Update equipment status and location details
        const selectedEquipment = inspectionStep1.selectedEquipment;
        if (selectedEquipment && equipmentStatus) {
          const equipmentUpdateTimestamp = new Date().toISOString();
          const coords = inspectionStep1.coordinates || selectedEquipment.lastKnownCoordinates || null;
          const locationText = inspectionStep1.gpsAddress || selectedEquipment.lastKnownLocation || null;
          const inspectionCountry = inspectionStep1.country || selectedEquipment.lastInspectionCountry || selectedEquipment.country || null;
          const inspectionProject = inspectionStep1.project || selectedEquipment.lastInspectionProject || selectedEquipment.project || null;
          const equipmentUpdatePayload = {
            status: equipmentStatus,
            updatedAt: equipmentUpdateTimestamp,
            lastInspectionAt: equipmentUpdateTimestamp,
            lastInspectionRequestNumber: requestNumber,
          };
          if (inspectionCountry) {
            equipmentUpdatePayload.country = inspectionCountry;
            equipmentUpdatePayload.lastInspectionCountry = inspectionCountry;
          }
          if (inspectionProject) {
            equipmentUpdatePayload.project = inspectionProject;
            equipmentUpdatePayload.lastInspectionProject = inspectionProject;
          }
          if (coords) {
            equipmentUpdatePayload.lastKnownCoordinates = coords;
            equipmentUpdatePayload.gpsCoordinates = coords;
            equipmentUpdatePayload.locationUpdatedAt = equipmentUpdateTimestamp;
          }
          if (locationText) {
            equipmentUpdatePayload.lastKnownLocation = locationText;
            equipmentUpdatePayload.gpsAddress = locationText;
            if (!equipmentUpdatePayload.locationUpdatedAt) {
              equipmentUpdatePayload.locationUpdatedAt = equipmentUpdateTimestamp;
            }
          }
          if (resolvedMeterReading !== null && resolvedMeterReading !== undefined && resolvedMeterReading !== '') {
            const parsedMeter = Number(resolvedMeterReading);
            if (!Number.isNaN(parsedMeter)) {
              equipmentUpdatePayload.meterReading = parsedMeter;
            }
          }
          if (selectedEquipment.meterUnit) {
            equipmentUpdatePayload.meterUnit = selectedEquipment.meterUnit;
          }

          if (isOnline) {
            try {
              const EquipmentAPI = require('../../ConfigMongo/EquipmentMongoConfig').default;
              await EquipmentAPI.updateEquipment(selectedEquipment.id, equipmentUpdatePayload);
              console.log('✅ Equipment status updated in MongoDB');
            } catch (equipError) {
              console.error('Equipment status update failed:', equipError);
            }
          } else {
            console.log('📴 Offline: Queuing equipment status/location update for sync');
          }

          try {
            const storedEquipment = await AsyncStorage.getItem('equipment');
            const equipmentList = storedEquipment ? JSON.parse(storedEquipment) : [];
            const equipmentIndex = equipmentList.findIndex((item) => item.id === selectedEquipment.id);
            if (equipmentIndex !== -1) {
              equipmentList[equipmentIndex] = {
                ...equipmentList[equipmentIndex],
                ...equipmentUpdatePayload,
                pendingSync: !isOnline || equipmentList[equipmentIndex]?.pendingSync,
              };
              await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
            }
          } catch (localUpdateError) {
            console.warn('⚠️ Unable to cache equipment status/location locally:', localUpdateError?.message);
          }
        }

        // Clean up local storage (keep temp data for offline mode)
        if (!isOnline) {
          // Offline: Keep PDF locally, don't delete it
          console.log('📱 Offline: Keeping PDF locally at:', pdfPath);
        } else {
          // Online: Can delete temp PDF after upload
          await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));
        }
        
        await AsyncStorage.removeItem('inspectionStep1');
        await AsyncStorage.removeItem('inspectionResults');
        await AsyncStorage.removeItem('selectedChecklistIds');
        await AsyncStorage.removeItem('Summerystep3State');

        // Show success modal
        setSavedPdfPath(pdfPath);
        setShowSuccessModal(true);
        
        // Show detailed success toast with storage information
        const storageInfo = isOnline 
          ? `✅ Saved to MongoDB\n📊 Report #${requestNumber}`
          : `📱 Saved Offline\n🔄 Auto-sync when online\n📊 Report #${requestNumber}`;
        
        Toast.show({
          type: 'success',
          text1: isOnline ? 'Report Saved Online' : 'Report Saved Offline',
          text2: storageInfo,
          position: 'top',
          visibilityTime: isOnline ? 4000 : 6000,
        });
        
        // Show alert with detailed information
        Alert.alert(
          isOnline ? '✅ Report Saved Successfully' : '📱 Report Saved Offline',
          isOnline 
            ? `Your inspection report has been saved to the database.\n\nReport Number: ${requestNumber}\nFaulty Items: ${faultyCount}\nStatus: ${equipmentStatus}\n\nThe report is now accessible from any device.`
            : `Your inspection report has been saved locally.\n\nReport Number: ${requestNumber}\nFaulty Items: ${faultyCount}\nStatus: ${equipmentStatus}\n\nThe report will automatically sync to the database when you're back online.`,
          [
            {
              text: 'OK',
              onPress: () => {
                // Continue to success modal
              }
            }
          ]
        );
      } else {
        throw new Error(response.error || `Failed to save report ${isOnline ? 'to MongoDB' : 'offline'}`);
      }
    } catch (error) {
      console.error('❌ Error generating report:', error);
      console.error('❌ Error details:', {
        message: error.message,
        name: error.name,
        stack: error.stack,
        userId,
        reportId,
        requestNumber
      });
      
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: error.message || 'Failed to generate or save report',
        position: 'top',
        visibilityTime: 4000,
      });
      
      Alert.alert('Error', error.message || 'Failed to generate or save report. Please try again.');
    } finally {
      setSyncModalVisible(false);
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    let isOnline = false;
    
    try {
      // Check network status
      const netInfo = await NetInfo.fetch();
      isOnline = netInfo.isConnected;
      
      console.log('🌐 Network status for PDF viewing:', isOnline ? 'ONLINE' : 'OFFLINE');
      
      // Get user ID from AsyncStorage
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('AdminLoginScreen');
        return;
      }

      let pdfUrl = null;
      
      if (isOnline) {
        // ONLINE MODE: Try to fetch from MongoDB first
        try {
          console.log('🌐 Online: Fetching report from MongoDB');
          const response = await InspectionAPI.getInspection(userId, reportId);
          
          if (response.success && response.data) {
            pdfUrl = response.data.step3?.pdfDownloadUrl;
            console.log('📄 PDF URL from MongoDB:', pdfUrl);
            
            if (!pdfUrl) {
              // Fallback to local PDF if MongoDB doesn't have URL yet
              console.log('⚠️ No PDF URL in MongoDB, checking local PDF...');
              
              // Try AsyncStorage first (with request number key)
              try {
                const requestNum = response.data.requestNumber || response.data.step1?.requestNumber;
                if (requestNum) {
                  const pdfPathKey = `inspection_pdf_local_${requestNum}`;
                  const storedPdfPath = await AsyncStorage.getItem(pdfPathKey);
                  if (storedPdfPath) {
                    const fileExists = await RNFS.exists(storedPdfPath);
                    if (fileExists) {
                      pdfUrl = Platform.OS === 'android' ? `file://${storedPdfPath}` : storedPdfPath;
                      console.log('✅ Using local PDF from AsyncStorage:', pdfUrl);
                      Toast.show({
                        type: 'info',
                        text1: 'Local PDF',
                        text2: 'Viewing locally stored PDF',
                        position: 'top',
                        visibilityTime: 3000,
                      });
                    }
                  }
                }
              } catch (asyncError) {
                console.warn('⚠️ AsyncStorage PDF lookup failed:', asyncError);
              }
              
              // Fallback to savedPdfPath
              if (!pdfUrl && savedPdfPath) {
                const fileExists = await RNFS.exists(savedPdfPath);
                if (fileExists) {
                  pdfUrl = Platform.OS === 'android' ? `file://${savedPdfPath}` : savedPdfPath;
                  console.log('✅ Using local PDF (not yet uploaded):', pdfUrl);
                  Toast.show({
                    type: 'info',
                    text1: 'Local PDF',
                    text2: 'Viewing locally stored PDF (not yet uploaded)',
                    position: 'top',
                    visibilityTime: 3000,
                  });
                }
              }
            }
          }
        } catch (mongoError) {
          console.error('⚠️ MongoDB fetch failed, falling back to local PDF:', mongoError);
          // Continue to offline logic below
        }
      }
      
      // OFFLINE MODE or fallback: Check local storage
      if (!pdfUrl) {
        console.log('📴 Checking local PDF storage...');
        
        // Priority 1: Check AsyncStorage with request number key
        try {
          const inspectionStep1Raw = await AsyncStorage.getItem('inspectionStep1');
          if (inspectionStep1Raw) {
            const inspectionStep1 = JSON.parse(inspectionStep1Raw);
            const requestNum = inspectionStep1.requestNumber;
            if (requestNum) {
              const pdfPathKey = `inspection_pdf_local_${requestNum}`;
              const storedPdfPath = await AsyncStorage.getItem(pdfPathKey);
              if (storedPdfPath) {
                const fileExists = await RNFS.exists(storedPdfPath);
                if (fileExists) {
                  pdfUrl = Platform.OS === 'android' ? `file://${storedPdfPath}` : storedPdfPath;
                  console.log('✅ Found PDF in AsyncStorage:', pdfUrl);
                }
              }
            }
          }
        } catch (asyncError) {
          console.warn('⚠️ AsyncStorage check failed:', asyncError);
        }
        
        // Priority 2: Check savedPdfPath (recently generated PDF)
        if (!pdfUrl && savedPdfPath) {
          const fileExists = await RNFS.exists(savedPdfPath);
          if (fileExists) {
            pdfUrl = Platform.OS === 'android' ? `file://${savedPdfPath}` : savedPdfPath;
            console.log('✅ Found PDF at savedPdfPath:', pdfUrl);
          }
        }
        
        // Priority 3: Check OfflineDataHelper
        if (!pdfUrl) {
          try {
            const offlinePdfPath = await OfflineDataHelper.getOfflinePDF(reportId);
            if (offlinePdfPath) {
              const fileExists = await RNFS.exists(offlinePdfPath);
              if (fileExists) {
                pdfUrl = Platform.OS === 'android' ? `file://${offlinePdfPath}` : offlinePdfPath;
                console.log('✅ Found PDF in OfflineDataHelper:', pdfUrl);
              }
            }
          } catch (offlineError) {
            console.warn('⚠️ OfflineDataHelper check failed:', offlineError);
          }
        }
        
        // Priority 4: Check AsyncStorage for cached report data
        if (!pdfUrl) {
          try {
            const reportRaw = await AsyncStorage.getItem(`report_${reportId}`);
            if (reportRaw) {
              const reportData = JSON.parse(reportRaw);
              const localPath = reportData.step3?.pdfLocalPath;
              if (localPath) {
                const fileExists = await RNFS.exists(localPath);
                if (fileExists) {
                  pdfUrl = Platform.OS === 'android' ? `file://${localPath}` : localPath;
                  console.log('✅ Found PDF in AsyncStorage report data:', pdfUrl);
                }
              }
            }
          } catch (storageError) {
            console.warn('⚠️ AsyncStorage report check failed:', storageError);
          }
        }
      }

      if (!pdfUrl) {
        const errorMessage = isOnline 
          ? 'PDF not available. Please try again later.' 
          : 'PDF not available offline. Please connect to the internet to view the PDF.';
        Alert.alert('Error', errorMessage);
        return;
      }

      console.log('🚀 Opening PDF:', pdfUrl);
      
      if (!isOnline) {
        Toast.show({
          type: 'info',
          text1: 'Offline Mode',
          text2: 'Viewing locally stored report',
          position: 'top',
          visibilityTime: 3000,
        });
      }
      
      navigation.replace('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.replace('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('❌ Error fetching PDF URL:', error);
      
      let errorMessage = 'Could not fetch the PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the PDF.';
      } else if (error.message?.includes('Network')) {
        errorMessage = 'Network error. Please check your connection.';
      } else if (error.message?.includes('404')) {
        errorMessage = 'Report not found in database.';
      }
      
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'top',
        visibilityTime: 4000,
      });
    } finally {
      setShowSuccessModal(false);
    }
  };

  return (
    <KeyboardAvoidingView
      style={styles.container}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <Modal
        animationType="fade"
        transparent={true}
        visible={modalVisible}
        onRequestClose={() => setModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>
              Overall Condition <Text style={styles.requiredAsterisk}>*</Text>
            </Text>
            {conditionOptions.map((option) => (
              <View key={option}>
                <TouchableOpacity
                  style={styles.modalRow}
                  onPress={() => handleConditionSelect(option)}
                >
                  <Text style={styles.modalOptionText}>{option}</Text>
                  <View
                    style={[
                      styles.radioCircle,
                      overallCondition === option && styles.radioCircleSelected,
                    ]}
                  >
                    {overallCondition === option && <View style={styles.radioDot} />}
                  </View>
                </TouchableOpacity>
                <View style={styles.modalDivider} />
              </View>
            ))}
          </View>
        </View>
      </Modal>

      <Modal
        animationType="fade"
        transparent={true}
        visible={showSuccessModal}
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.popupModalContent}>
            <Text style={styles.popupTitle}>Success</Text>
            <MaterialIcons name="check" size={44} color="#015185" style={{ alignSelf: 'center', marginVertical: 8 }} />
            <Text style={styles.popupSubtitle}>Inspection Report Generated and Saved</Text>
            <View style={{ flexDirection: 'row', justifyContent: 'center', marginTop: 18 }}>
              <TouchableOpacity
                style={styles.popupButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.reset({
                    index: 1,
                    routes: [
                      { name: 'MainApp' },
                      { name: 'Inspection_Report_Home' },
                    ],
                  });
                }}
              >
                <Text style={styles.popupButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.popupButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.popupButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        animationType="fade"
        transparent={true}
        visible={showSignatureCountModal}
        onRequestClose={() => setShowSignatureCountModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>
              Select Number of Signatures Required <Text style={styles.requiredAsterisk}>*</Text>
            </Text>
            {[1, 2, 3, 4, 5, 6, 7, 8].map((count) => (
              <View key={count}>
                <TouchableOpacity
                  style={styles.modalRow}
                  onPress={() => handleSignatureCountSelect(count)}
                >
                  <Text style={styles.modalOptionText}>{count} Signature{count > 1 ? 's' : ''} Required</Text>
                  <View
                    style={[
                      styles.radioCircle,
                      signatureCount === count && styles.radioCircleSelected,
                    ]}
                  >
                    {signatureCount === count && <View style={styles.radioDot} />}
                  </View>
                </TouchableOpacity>
                <View style={styles.modalDivider} />
              </View>
            ))}
          </View>
        </View>
      </Modal>

      <Modal
        animationType="fade"
        transparent={true}
        visible={showSignatureTypeModal}
        onRequestClose={() => setShowSignatureTypeModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>
                Select Signature Type <Text style={styles.requiredAsterisk}>*</Text>
              </Text>
              <TouchableOpacity
                style={styles.closeButton}
                onPress={() => setShowSignatureTypeModal(false)}
              >
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            
            <ScrollView style={styles.modalScrollView}>
              {console.log('checklistSignatureFields in modal:', checklistSignatureFields)}
              {console.log('selectedSignatureTypes in modal:', selectedSignatureTypes)}
              {checklistSignatureFields.length > 0 ? (
                <>
                  <Text style={{ fontSize: 14, color: '#0078D4', marginBottom: 15, textAlign: 'center', fontWeight: 'bold' }}>
                    Select Signature Types:
                  </Text>
              {checklistSignatureFields.map((field, index) => {
                const displayLabel = field.label && !/signature/i.test(field.label)
                  ? `${field.label} Signature`
                  : field.label || 'Signature';
                return (
                    <TouchableOpacity
                      key={field.key}
                      style={[
                        styles.modalRow,
                        selectedSignatureTypes.includes(field.key) && styles.selectedModalRow
                      ]}
                      onPress={() => {
                        const isSelected = selectedSignatureTypes.includes(field.key);
                        console.log(`Toggling ${field.key}, currently selected: ${isSelected}`);
                        if (isSelected) {
                          setSelectedSignatureTypes(prev => {
                            const newSelection = prev.filter(key => key !== field.key);
                            console.log('Removing from selection:', newSelection);
                            return newSelection;
                          });
                        } else {
                          setSelectedSignatureTypes(prev => {
                            const newSelection = [...prev, field.key];
                            console.log('Adding to selection:', newSelection);
                            return newSelection;
                          });
                        }
                    }}
                  >
                    <Text style={styles.modalOptionText}>{displayLabel}</Text>
                      <View
                        style={[
                          styles.radioCircle,
                          selectedSignatureTypes.includes(field.key) && styles.radioCircleSelected,
                        ]}
                      >
                        {selectedSignatureTypes.includes(field.key) && <View style={styles.radioDot} />}
                      </View>
                    </TouchableOpacity>
                );
              })}
                </>
              ) : (
                <>
                  <Text style={{ fontSize: 14, color: '#666', marginBottom: 10, textAlign: 'center' }}>
                    No footer inputs found. Showing default options:
                  </Text>
                  <TouchableOpacity
                    style={styles.modalRow}
                    onPress={() => {
                      const isSelected = selectedSignatureTypes.includes('driller');
                      console.log(`Toggling driller, currently selected: ${isSelected}`);
                      if (isSelected) {
                        setSelectedSignatureTypes(prev => {
                          const newSelection = prev.filter(key => key !== 'driller');
                          console.log('Removing driller from selection:', newSelection);
                          return newSelection;
                        });
                      } else {
                        setSelectedSignatureTypes(prev => {
                          const newSelection = [...prev, 'driller'];
                          console.log('Adding driller to selection:', newSelection);
                          return newSelection;
                        });
                      }
                    }}
                  >
                    <Text style={styles.modalOptionText}>Driller</Text>
                    <View
                      style={[
                        styles.radioCircle,
                        selectedSignatureTypes.includes('driller') && styles.radioCircleSelected,
                      ]}
                    >
                      {selectedSignatureTypes.includes('driller') && <View style={styles.radioDot} />}
                    </View>
                  </TouchableOpacity>
                  <TouchableOpacity
                    style={styles.modalRow}
                    onPress={() => {
                      const isSelected = selectedSignatureTypes.includes('mechanic');
                      console.log(`Toggling mechanic, currently selected: ${isSelected}`);
                      if (isSelected) {
                        setSelectedSignatureTypes(prev => {
                          const newSelection = prev.filter(key => key !== 'mechanic');
                          console.log('Removing mechanic from selection:', newSelection);
                          return newSelection;
                        });
                      } else {
                        setSelectedSignatureTypes(prev => {
                          const newSelection = [...prev, 'mechanic'];
                          console.log('Adding mechanic to selection:', newSelection);
                          return newSelection;
                        });
                      }
                    }}
                  >
                    <Text style={styles.modalOptionText}>Mechanic</Text>
                    <View
                      style={[
                        styles.radioCircle,
                        selectedSignatureTypes.includes('mechanic') && styles.radioCircleSelected,
                      ]}
                    >
                      {selectedSignatureTypes.includes('mechanic') && <View style={styles.radioDot} />}
                    </View>
                  </TouchableOpacity>
                  <TouchableOpacity
                    style={styles.modalRow}
                    onPress={() => {
                      const isSelected = selectedSignatureTypes.includes('operator');
                      console.log(`Toggling operator, currently selected: ${isSelected}`);
                      if (isSelected) {
                        setSelectedSignatureTypes(prev => {
                          const newSelection = prev.filter(key => key !== 'operator');
                          console.log('Removing operator from selection:', newSelection);
                          return newSelection;
                        });
                      } else {
                        setSelectedSignatureTypes(prev => {
                          const newSelection = [...prev, 'operator'];
                          console.log('Adding operator to selection:', newSelection);
                          return newSelection;
                        });
                      }
                    }}
                  >
                    <Text style={styles.modalOptionText}>Operator</Text>
                    <View
                      style={[
                        styles.radioCircle,
                        selectedSignatureTypes.includes('operator') && styles.radioCircleSelected,
                      ]}
                    >
                      {selectedSignatureTypes.includes('operator') && <View style={styles.radioDot} />}
                    </View>
                  </TouchableOpacity>
                </>
              )}
            </ScrollView>
            
            <View style={styles.modalFooter}>
              <TouchableOpacity
                style={[
                  styles.doneButton,
                  selectedSignatureTypes.length === 0 && styles.doneButtonDisabled
                ]}
                onPress={() => {
                  console.log('Done button pressed with selectedSignatureTypes:', selectedSignatureTypes);
                  if (selectedSignatureTypes.length > 0) {
                    setShowSignatureTypeModal(false);
                    
                    if (checklistSignatureFields.length > 0) {
                      // Update signature fields based on selection from checklist
                      const selectedFields = checklistSignatureFields.filter(field => 
                        selectedSignatureTypes.includes(field.key)
                      );
                      console.log('Selected fields from checklist:', selectedFields);
                      
                      // Remove duplicates based on key
                      const uniqueFields = selectedFields.filter((field, index, self) => 
                        index === self.findIndex(f => f.key === field.key)
                      );
                      console.log('Unique fields after deduplication:', uniqueFields);
                      console.log('Setting signatureFields to:', uniqueFields);
                      setSignatureFields(uniqueFields);
                      
                      // Reset signature values for selected fields
                      const newSignatureValues = {};
                      uniqueFields.forEach(field => {
                        if (field.nameKey) newSignatureValues[field.nameKey] = '';
                        if (field.signatureKey) newSignatureValues[field.signatureKey] = '';
                      });
                      console.log('Setting signature values:', newSignatureValues);
                      setSignatureValues(newSignatureValues);
                      
                      // Debug: Log the final state
                      console.log('=== Final Signature Selection ===');
                      console.log('Selected types:', selectedSignatureTypes);
                      console.log('Fields to be set:', uniqueFields);
                      console.log('Signature values to be set:', newSignatureValues);
                    } else {
                      // Handle default options
                      const defaultFields = [];
                      if (selectedSignatureTypes.includes('driller')) {
                        defaultFields.push({
                          key: 'driller',
                          label: 'Driller',
                          roleName: 'Driller',
                          required: true,
                          showInput: true,
                          showSignature: true,
                          nameKey: 'driller_name',
                          signatureKey: 'driller_signature'
                        });
                      }
                      if (selectedSignatureTypes.includes('mechanic')) {
                        defaultFields.push({
                          key: 'mechanic',
                          label: 'Mechanic',
                          roleName: 'Mechanic',
                          required: true,
                          showInput: true,
                          showSignature: true,
                          nameKey: 'mechanic_name',
                          signatureKey: 'mechanic_signature'
                        });
                      }
                      if (selectedSignatureTypes.includes('operator')) {
                        defaultFields.push({
                          key: 'operator',
                          label: 'Operator',
                          roleName: 'Operator',
                          required: true,
                          showInput: true,
                          showSignature: true,
                          nameKey: 'operator_name',
                          signatureKey: 'operator_signature'
                        });
                      }
                      
                      // Remove duplicates based on key
                      const uniqueDefaultFields = defaultFields.filter((field, index, self) => 
                        index === self.findIndex(f => f.key === field.key)
                      );
                      console.log('Setting default fields:', uniqueDefaultFields);
                      setSignatureFields(uniqueDefaultFields);
                      
                      // Reset signature values for selected fields
                      const newSignatureValues = {};
                      uniqueDefaultFields.forEach(field => {
                        if (field.nameKey) newSignatureValues[field.nameKey] = '';
                        if (field.signatureKey) newSignatureValues[field.signatureKey] = '';
                      });
                      console.log('Setting default signature values:', newSignatureValues);
                      setSignatureValues(newSignatureValues);
                      
                      // Debug: Log the final state
                      console.log('=== Final Default Signature Selection ===');
                      console.log('Selected types:', selectedSignatureTypes);
                      console.log('Default fields to be set:', uniqueDefaultFields);
                      console.log('Default signature values to be set:', newSignatureValues);
                    }
                  }
                }}
                disabled={selectedSignatureTypes.length === 0}
              >
                <Text style={[
                  styles.doneButtonText,
                  selectedSignatureTypes.length === 0 && styles.doneButtonTextDisabled
                ]}>
                  Done ({selectedSignatureTypes.length} selected)
                </Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        animationType="fade"
        transparent={true}
        visible={showMultiSignatureModal}
        onRequestClose={() => setShowMultiSignatureModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>
              Select Signature Types <Text style={styles.requiredAsterisk}>*</Text>
            </Text>
            <Text style={styles.modalSubtitle}>Select one or multiple signature types</Text>
            
            {signatureFields.map((field) => (
              <View key={field.key}>
                <TouchableOpacity
                  style={styles.modalRow}
                  onPress={() => {
                    const isSelected = selectedSignatureTypes.includes(field.key);
                    if (isSelected) {
                      setSelectedSignatureTypes(prev => prev.filter(key => key !== field.key));
                    } else {
                      setSelectedSignatureTypes(prev => [...prev, field.key]);
                    }
                  }}
                >
                  <Text style={styles.modalOptionText}>{field.roleName || field.label}</Text>
                  <View
                    style={[
                      styles.radioCircle,
                      selectedSignatureTypes.includes(field.key) && styles.radioCircleSelected,
                    ]}
                  >
                    {selectedSignatureTypes.includes(field.key) && <View style={styles.radioDot} />}
                  </View>
                </TouchableOpacity>
                <View style={styles.modalDivider} />
              </View>
            ))}
            
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={styles.modalCancelButton}
                onPress={() => setShowMultiSignatureModal(false)}
              >
                <Text style={styles.modalCancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.modalDoneButton}
                onPress={() => {
                  if (selectedSignatureTypes.length > 0) {
                    // Update signature fields based on selection
                    const selectedFields = signatureFields.filter(field => 
                      selectedSignatureTypes.includes(field.key)
                    );
                    setSignatureFields(selectedFields);
                    
                    // Reset signature values for selected fields
                    const newSignatureValues = {};
                    const newIndividualSignatures = {};
                    const newIndividualSignatureStatus = {};
                    
                    selectedFields.forEach(field => {
                      if (field.nameKey) newSignatureValues[field.nameKey] = '';
                      if (field.signatureKey) {
                        newSignatureValues[field.signatureKey] = '';
                        newIndividualSignatures[field.signatureKey] = null;
                        newIndividualSignatureStatus[field.signatureKey] = false;
                      }
                    });
                    
                    setSignatureValues(newSignatureValues);
                    setIndividualSignatures(newIndividualSignatures);
                    setIndividualSignatureStatus(newIndividualSignatureStatus);
                    
                    setShowMultiSignatureModal(false);
                  } else {
                    Alert.alert('Warning', 'Please select at least one signature type.');
                  }
                }}
              >
                <Text style={styles.modalDoneButtonText}>Done</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <ScrollView
        style={styles.mainContent}
        contentContainerStyle={styles.scrollContent}
        keyboardShouldPersistTaps="handled"
      >
        <View style={styles.rowContainer}>
          <Text style={styles.labelTextFault}>Faulty Items</Text>
          <Text style={styles.valueTextFault}>{faultyItems}</Text>
        </View>

        <Modal
          visible={voiceModalVisible}
          transparent
          animationType="fade"
          onRequestClose={() => {
            if (isListening) {
              cancelRecording();
            }
          }}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.modalContent}>
              {isStopping ? (
                <View style={styles.processingContainer}>
                  <ActivityIndicator size="large" color="#015185" />
                  <Text style={styles.processingText}>Finishing up...</Text>
                </View>
              ) : (
                <>
                  <View style={styles.lottieContainer}>
                    <LottieView
                      ref={lottieRef}
                      source={require('../../../Images/micanimation.json')}
                      autoPlay={false}
                      loop={true}
                      style={styles.lottieAnimation}
                      resizeMode="contain"
                    />
                  </View>

                  <Text style={styles.partial}>
                    {partialText
                      ? partialText
                      : isListening
                      ? 'Listening... Speak now'
                      : 'Preparing...'}
                  </Text>

                  {volume > 0 && (
                    <View style={styles.volumeIndicator}>
                      <View style={[styles.volumeBar, { width: `${Math.min(volume * 100, 100)}%` }]} />
                    </View>
                  )}

                  <View style={styles.modalButtons}>
                    <TouchableOpacity
                      style={[styles.modalButton, styles.cancelBtn]}
                      onPress={cancelRecording}
                      activeOpacity={0.7}
                      disabled={isStopping}
                    >
                      <Text style={[styles.modalButtonText, { color: '#015185' }]}>Cancel</Text>
                    </TouchableOpacity>
                    <TouchableOpacity
                      style={styles.modalButton}
                      onPress={onDone}
                      activeOpacity={0.7}
                      disabled={isStopping}
                    >
                      <Text style={styles.modalButtonText}>Done</Text>
                    </TouchableOpacity>
                  </View>
                </>
              )}
            </View>
          </View>
        </Modal>

        <View style={styles.rowContainer}>
          <Text style={styles.labelTextFault}>
            Overall Condition <Text style={styles.requiredAsterisk}>*</Text>
          </Text>
          <View style={styles.buttonContainer}>
            <TouchableOpacity style={styles.selectButton} onPress={() => setModalVisible(true)}>
              <Text style={styles.buttonText}>{overallCondition || 'Select'}</Text>
              <MaterialIcons name="chevron-right" size={20} color="#fff" style={styles.iconRight} />
            </TouchableOpacity>
          </View>
        </View>
        {errors.overallCondition ? (
          <Text style={styles.errorText}>{errors.overallCondition}</Text>
        ) : null}

        <Text style={styles.sectionLabelequip}>
          Equipment Status <Text style={styles.requiredAsterisk}>*</Text>
        </Text>
        <View style={styles.optionsRow}>
          {statusOptions.map((option) => {
            const isSelected = equipmentStatus === option;
            return (
              <TouchableOpacity
                key={option}
                style={[styles.statusOptionButton, isSelected ? styles.statusOptionButtonSelected : null]}
                onPress={() => handleStatusSelect(option)}
              >
                <Text style={isSelected ? styles.optionTextSelected : styles.optionText}>{option}</Text>
              </TouchableOpacity>
            );
          })}
        </View>
        {errors.equipmentStatus ? (
          <Text style={styles.errorText}>{errors.equipmentStatus}</Text>
        ) : null}

        {showPriority && (
          <>
            <Text style={styles.sectionLabelequip}>Priority</Text>
            <View style={styles.priorityContainer}>
              {priorityOptions.map((option) => {
                const isSelected = priority === option.label;
                return (
                  <View key={option.label} style={styles.priorityItem}>
                    <TouchableOpacity
                      style={[
                        styles.priorityButton,
                        isSelected ? styles.priorityButtonSelected : styles.priorityButtonUnselected,
                      ]}
                      onPress={() => handlePrioritySelect(option.label)}
                    >
                      <Image source={option.image} style={styles.priorityImage} />
                      <Text style={isSelected ? styles.priorityTextSelected : styles.priorityText}>
                        {option.label}
                      </Text>
                    </TouchableOpacity>
                  </View>
                );
              })}
            </View>
          </>
        )}

        <Text style={styles.sectionLabelequip}>
          Equipment Safe to Use <Text style={styles.requiredAsterisk}>*</Text>
        </Text>
        <View style={styles.safeOptionsRow}>
          {safeOptions.map((option) => {
            const isSelected = safeToUse === option;
            return (
              <TouchableOpacity
                key={option}
                style={[styles.optionButton, isSelected ? styles.optionButtonSelected : null, { marginRight: 20 }]}
                onPress={() => handleSafeSelect(option)}
              >
                <Text style={isSelected ? styles.optionTextSelected : styles.optionText}>{option}</Text>
              </TouchableOpacity>
            );
          })}
        </View>
        {errors.safeToUse ? (
          <Text style={styles.errorText}>{errors.safeToUse}</Text>
        ) : null}

        {/* Footer labels removed as per request */}

        <Text style={styles.voiceStatus}>Status: {isListening ? 'Listening...' : 'Stopped'}</Text>
        {shouldShowVoiceError ? <Text style={styles.voiceErrorText}>Error: {voiceError}</Text> : null}
        <View style={styles.inputContainer}>
          <TextInput
            style={styles.input}
            placeholder="Additional Note"
            value={additionalNote}
            onChangeText={handleAdditionalNoteChange}
            multiline={true}
            numberOfLines={4}
            textAlignVertical="top"
          />
          <TouchableOpacity
            style={[
              styles.inputIconContainer,
              isListening && styles.inputIconContainerActive,
              (!isListening && !isVoiceModuleAvailable) && styles.inputIconContainerDisabled,
            ]}
            onPress={isListening ? stopRecording : startRecording}
            disabled={isStopping || (!isListening && !isVoiceModuleAvailable)}
            activeOpacity={0.8}
          >
            <MaterialIcons name={isListening ? 'mic-off' : 'mic'} size={24} color="#fff" />
          </TouchableOpacity>
        </View>



        {/* Dynamic Signature Fields */}
        <View style={styles.signatureFieldsContainer}>
          <Text style={styles.sectionLabelequip}>Signature Information</Text>
          
          {/* Signature Count Selection */}
          <View style={styles.signatureCountContainer}>
            <Text style={styles.signatureCountLabel}>
              Signature Type <Text style={styles.requiredAsterisk}>*</Text>
            </Text>
            <TouchableOpacity 
              style={styles.signatureCountButton}
              onPress={async () => {
                // Log complete checklist data when modal opens
                await logCompleteChecklistData();
                // Clear current signature fields and selection when opening modal
                setSignatureFields([]);
                setSelectedSignatureTypes([]);
                setSignatureValues({});
                setShowSignatureTypeModal(true);
              }}
            >
              <Text style={styles.signatureCountButtonText}>{getSignatureTypeLabel()}</Text>
              <MaterialIcons name="chevron-down" size={20} color="#015185" />
            </TouchableOpacity>
          </View>

          {/* Dynamic Signature Fields - render all selected signature fields */}
          {signatureFields.length > 0 && (
            <>
              <Text style={styles.sectionLabelequip}>
                Signature Fields <Text style={styles.requiredAsterisk}>*</Text>
              </Text>
              {signatureFields.map((field, index) => {
            const uniqueKey = `${field.key}_${index}`;
            return (
              <View key={uniqueKey} style={[
                styles.signatureFieldContainer,
                index > 0 && { marginTop: 20 }
              ]}>
                {field.showInput && field.nameKey && (
                  <>
                    <Text style={styles.signatureFieldLabel}>
                      {field.roleName}'s Name <Text style={styles.requiredAsterisk}>*</Text>
                    </Text>
                    <TextInput
                      style={styles.signatureTextInput}
                      value={signatureValues[field.nameKey] || ''}
                      onChangeText={(text) => setSignatureValues((prev) => ({ ...prev, [field.nameKey]: text }))}
                      placeholder={`Enter ${field.roleName.toLowerCase()}'s name`}
                      placeholderTextColor="#888"
                    />
                    {signatureErrors[field.nameKey] && (
                      <Text style={styles.errorText}>{signatureErrors[field.nameKey]}</Text>
                    )}
                  </>
                )}

                {field.showInput && !field.nameKey && (
                  <>
                    <Text style={styles.signatureFieldLabel}>
                      {field.label} <Text style={styles.requiredAsterisk}>*</Text>
                    </Text>
                    <TextInput
                      style={styles.signatureTextInput}
                      value={signatureValues[field.key] || ''}
                      onChangeText={(text) => setSignatureValues((prev) => ({ ...prev, [field.key]: text }))}
                      placeholder={`Enter ${field.label.toLowerCase()}`}
                      placeholderTextColor="#888"
                    />
                    {signatureErrors[field.key] && (
                      <Text style={styles.errorText}>{signatureErrors[field.key]}</Text>
                    )}
                  </>
                )}

                <Text style={styles.individualSignatureLabel}>
                  {field.roleName || field.label} Signature <Text style={styles.requiredAsterisk}>*</Text>
                </Text>
                <TouchableOpacity
                  style={styles.individualSignatureBox}
                  onPress={() => handleIndividualSignaturePress(field.signatureKey || field.key)}
                >
                  <MaterialIcons name="add" size={40} color="#0078D4" />
                  <Text style={styles.individualSignatureText}>Tap to add signature</Text>
                </TouchableOpacity>

                {(individualSignatureStatus[field.signatureKey || field.key] && individualSignatures[field.signatureKey || field.key]) && (
                  <View style={styles.individualSignatureStatusContainer}>
                    <Text style={styles.individualSignatureStatus}>Signature Done</Text>
                    <Image
                      source={{ uri: individualSignatures[field.signatureKey || field.key] }}
                      style={styles.individualSignatureImage}
                      resizeMode="contain"
                      onError={() => {
                        Alert.alert('Error', 'Failed to load signature image.');
                      }}
                    />
                    <TouchableOpacity
                      style={styles.uploadAgainButton}
                      onPress={() => handleIndividualSignaturePress(field.signatureKey || field.key)}
                    >
                      <Text style={styles.uploadAgainText}>Upload again</Text>
                    </TouchableOpacity>
                  </View>
                )}
                
                {signatureErrors[field.signatureKey || field.key] && (
                  <Text style={styles.errorText}>{signatureErrors[field.signatureKey || field.key]}</Text>
                )}
              </View>
            );
              })}
            </>
          )}

          {signatureErrors.general && (
            <Text style={styles.errorText}>{signatureErrors.general}</Text>
          )}
        </View>

        {/* Main inspector signature removed as per request */}
      </ScrollView>

      <View style={styles.navButtonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(2)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.navButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleGenerateReport}>
          <Text style={styles.navButtonText}>Generate Report</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
      />

      {/* Signature Modal */}
      <Modal
        visible={signatureModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={closeSignatureModal}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>
                {currentSignatureType || 'Signature'} Signature
              </Text>
              <TouchableOpacity onPress={closeSignatureModal}>
                <MaterialIcons name="close" size={24} color="#333" />
              </TouchableOpacity>
            </View>
            
            <View style={styles.signatureCanvasContainer}>
              <SignatureCanvas
                ref={signatureRef}
                onOK={handleSignatureData}
                onEmpty={() => {
                  console.log('Signature canvas is empty')
                }}
                onBegin={() => console.log('User started drawing')}
                onEnd={() => console.log('User finished drawing')}
                descriptionText="Sign here"
                clearText="Clear"
                confirmText="Confirm"
                webStyle={canvasStyle}
                penColor="#000000"
                backgroundColor="#ffffff"
                minStrokeWidth={2}
                maxStrokeWidth={4}
                onError={(error) => {
                  console.error('SignatureCanvas WebView error:', error)
                  Alert.alert('Error', 'Failed to load signature canvas.')
                }}
              />
            </View>
            
            <View style={styles.modalButtons}>
              <TouchableOpacity 
                style={styles.modalCancelButton}
                onPress={closeSignatureModal}
              >
                <Text style={styles.modalCancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalClearButton}
                onPress={handleSignatureClear}
              >
                <Text style={styles.modalClearButtonText}>Clear</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalConfirmButton}
                onPress={handleSignatureConfirm}
              >
                <Text style={styles.modalConfirmButtonText}>Confirm</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </KeyboardAvoidingView>
  );
};

// Styles (unchanged, included for completeness)
const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  inlineFooterGroupsRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 12,
  },
  footerGroupsContainer: {
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 8,
    marginBottom: 12,
  },
  footerGroupRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 4,
  },
  footerGroupLabel: {
    fontSize: 14,
    color: '#333',
  },
  inlineFooterGroupsLabel: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
  inlineFooterGroupsText: {
    fontSize: 14,
    color: '#333',
    marginLeft: 4,
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingVertical: 10,
    paddingHorizontal: '5%',
    paddingBottom: 120,
  },
  rowContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    width: '100%',
    padding: 10,
    borderRadius: 5,
    marginBottom: 10,
  },
  labelTextFault: {
    fontSize: 16,
    fontWeight: '500',
    color: '#015185',
  },
  valueTextFault: {
    fontSize: 16,
    fontWeight: '600',
    color: '#FF0000',
  },
  buttonContainer: {
    alignItems: 'center',
  },
  selectButton: {
    backgroundColor: '#015185',
    paddingVertical: 4,
    paddingHorizontal: 10,
    borderRadius: 5,
    justifyContent: 'center',
    flexDirection: 'row',
    alignItems: 'center',
  },
  buttonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  iconRight: {
    marginLeft: 5,
  },
  sectionLabelequip: {
    fontSize: 16,
    fontWeight: '500',
    color: '#015185',
    marginVertical: 10,
    marginLeft: '0%',
  },
  optionsRow: {
    flexDirection: 'row',
    marginBottom: 8,
  },
  safeOptionsRow: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginBottom: 15,
  },
  statusOptionButton: {
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#ECF0F3',
    paddingHorizontal: 10,
    paddingVertical: 9,
    borderRadius: 5,
    marginHorizontal: 5,
  },
  statusOptionButtonSelected: {
    backgroundColor: '#015185',
    paddingHorizontal: 10,
    paddingVertical: 9,
    borderRadius: 5,
    marginHorizontal: 5,
  },
  optionButton: {
    paddingHorizontal: 15,
    paddingVertical: 8,
    borderRadius: 5,
    alignItems: 'center',
    justifyContent: 'center',
    minWidth: 80,
    backgroundColor: '#ECF0F3',
  },
  optionButtonSelected: {
    paddingHorizontal: 15,
    paddingVertical: 8,
    borderRadius: 5,
    alignItems: 'center',
    justifyContent: 'center',
    minWidth: 80,
    backgroundColor: '#015185',
  },
  optionText: {
    fontSize: 13,
    fontWeight: '600',
  },
  optionTextSelected: {
    fontSize: 13,
    color: '#fff',
    fontWeight: '600',
  },
  priorityContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: 15,
    backgroundColor: '#fff',
    borderRadius: 5,
    width: '100%',
  },
  priorityItem: {
    alignItems: 'center',
  },
  priorityButton: {
    padding: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    width: 85,
    height: 60,
    justifyContent: 'center',
    alignItems: 'center',
  },
  priorityButtonSelected: {
    backgroundColor: '#015185',
  },
  priorityButtonUnselected: {
    backgroundColor: '#FFFFFF',
  },
  priorityImage: {
    width: 24,
    height: 24,
  },
  priorityText: {
    fontSize: 10,
    color: '#333',
    marginTop: 3,
    textAlign: 'center',
  },
  priorityTextSelected: {
    fontSize: 10,
    color: '#fff',
    fontWeight: '600',
    marginTop: 3,
    textAlign: 'center',
  },
  inputContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#ECF0F3',
    borderRadius: 5,
    paddingHorizontal: 10,
    marginBottom: 20,
    width: '100%',
  },
  voiceStatus: {
    marginBottom: 6,
    marginLeft: 2,
    fontSize: 14,
    color: '#666',
  },
  voiceErrorText: {
    color: '#d32f2f',
    fontSize: 13,
    marginBottom: 6,
    marginLeft: 2,
    backgroundColor: '#ffebee',
    padding: 6,
    borderRadius: 4,
  },
  processingContainer: {
    width: '100%',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 24,
  },
  processingText: {
    marginTop: 20,
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    textAlign: 'center',
  },
  input: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    paddingVertical: 10,
  },
  inputIconContainer: {
    backgroundColor: '#015185',
    padding: 5,
    left: 8,
  },
  inputIconContainerActive: {
    backgroundColor: '#FF4B2B',
  },
  inputIconContainerDisabled: {
    opacity: 0.5,
  },
  signatureLabel: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    marginVertical: 10,
  },
  signatureContainer: {
    alignItems: 'center',
  },
  signatureStatus: {
    fontSize: 16,
    color: '#015185',
    textAlign: 'center',
    marginVertical: 5,
  },
  signatureImage: {
    width: width * 0.6,
    height: height * 0.15,
    alignSelf: 'center',
    marginTop: 10,
    borderRadius: 10,
    backgroundColor: '#F5F5F5',
  },
  uploadAgainButton: {
    backgroundColor: '#d3d3d3',
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 10,
    alignSelf: 'center',
  },
  uploadAgainText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  signatureBox: {
    height: height / 3.5,
    backgroundColor: '#F5F5F5',
    borderRadius: 10,
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    marginTop: 10,
  },
  navButtonContainer: {
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 15,
    paddingVertical: 10,
    backgroundColor: '#f0f0f0',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    paddingBottom: 30,
  },
  previewButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  navButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  modalRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  modalOptionText: {
    fontSize: 16,
    color: '#333',
  },
  modalDivider: {
    height: 1,
    backgroundColor: '#d3d3d3',
  },
  requiredAsterisk: {
    color: '#FF0000',
  },
  errorText: {
    color: '#FF0000',
    fontSize: 12,
    marginBottom: 10,
    marginLeft: 5,
  },
  radioCircle: {
    width: 24,
    height: 24,
    borderRadius: 12,
    borderWidth: 2,
    borderColor: '#015185',
    justifyContent: 'center',
    alignItems: 'center',
  },
  radioCircleSelected: {
    borderColor: '#015185',
  },
  radioDot: {
    width: 12,
    height: 12,
    borderRadius: 6,
    backgroundColor: '#015185',
  },
  popupModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
  },
  popupTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  popupSubtitle: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
  },
  popupButton: {
    backgroundColor: '#0078D4',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  popupButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  bottomicon: {
    marginTop: 2,
  },
  listeningModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  listeningModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  listeningModalText: {
    fontSize: 18,
    color: '#333',
    marginBottom: 15,
  },
  stopButton: {
    backgroundColor: '#FF0000',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  stopButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },

  signatureFieldsContainer: {
    marginTop: 20,
    paddingHorizontal: 10,
    backgroundColor: '#ECF0F3',
    borderRadius: 5,
    paddingVertical: 10,
  },
  signatureField: {
    marginBottom: 10,
  },
  signatureFieldLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 5,
  },
  signatureTextInput: {
    backgroundColor: '#fff',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 8,
    fontSize: 14,
    color: '#333',
    borderWidth: 1,
    borderColor: '#ccc',
  },
  signatureCountContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 8,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#ccc',
  },
  signatureCountLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
  },
  signatureCountButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    width: '60%',
  },
  signatureCountButtonText: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
  },
  signatureFieldContainer: {
    marginBottom: 20,
  },
  individualSignatureLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 5,
  },
  individualSignatureBox: {
    backgroundColor: '#fff',
    borderRadius: 5,
    padding: 10,
    alignItems: 'center',
    justifyContent: 'center',
    marginTop: 10,
  },
  individualSignatureText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: 'bold',
  },
  individualSignatureContainer: {
    alignItems: 'center',
  },
  individualSignatureStatus: {
    fontSize: 16,
    color: '#015185',
    textAlign: 'center',
    marginVertical: 5,
  },
  individualSignatureImage: {
    width: width * 0.6,
    height: height * 0.15,
    alignSelf: 'center',
    marginTop: 10,
    borderRadius: 10,
    backgroundColor: '#F5F5F5',
  },
  individualSignatureStatusContainer: {
    alignItems: 'center',
    marginTop: 10,
    paddingHorizontal: 10,
  },
  individualSignatureStatus: {
    fontSize: 14,
    color: '#015185',
    textAlign: 'center',
    marginBottom: 5,
    fontWeight: '600',
  },
  individualSignatureImage: {
    width: width * 0.4,
    height: height * 0.1,
    alignSelf: 'center',
    marginTop: 5,
    borderRadius: 5,
    backgroundColor: '#F5F5F5',
  },
  modalSubtitle: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 15,
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginTop: 20,
  },
  modalCancelButton: {
    backgroundColor: '#FF0000',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  modalCancelButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  modalDoneButton: {
    backgroundColor: '#0078D4',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  modalDoneButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  // Local preview styles
  previewModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '90%',
    maxHeight: '80%',
  },
  previewSectionTitle: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginTop: 10,
    marginBottom: 6,
  },
  previewRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 6,
  },
  previewLabel: {
    fontSize: 13,
    color: '#666',
    fontWeight: '600',
    marginRight: 8,
  },
  previewValue: {
    fontSize: 13,
    color: '#333',
    flexShrink: 1,
    textAlign: 'right',
  },
  previewEmpty: {
    fontSize: 13,
    color: '#888',
    fontStyle: 'italic',
  },
  previewItemRow: {
    marginBottom: 6,
  },
  previewItemName: {
    fontSize: 13,
    color: '#333',
    fontWeight: '600',
  },
  previewItemMeta: {
    fontSize: 12,
    color: '#666',
  },
  // Signature Modal Styles
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 20,
  },
  signatureCanvasContainer: {
    alignItems: 'center',
    marginBottom: 20,
    height: height * 0.4,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 10,
    overflow: 'hidden',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalCancelButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    marginRight: 5,
  },
  modalCancelButtonText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalClearButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#FF6B6B',
    marginHorizontal: 5,
  },
  modalClearButtonText: {
    color: '#FF6B6B',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalConfirmButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    backgroundColor: '#015185',
    marginLeft: 5,
  },
  modalConfirmButtonText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  closeButton: {
    position: 'absolute',
    top: 10,
    right: 10,
    padding: 5,
    borderRadius: 5,
  },
  selectedModalRow: {
    backgroundColor: '#f0f0f0',
  },
  modalScrollView: {
    maxHeight: 200,
  },
  modalFooter: {
    marginTop: 20,
    alignItems: 'center',
  },
  doneButton: {
    backgroundColor: '#0078D4',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  doneButtonDisabled: {
    backgroundColor: '#ccc',
  },
  doneButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  doneButtonTextDisabled: {
    color: '#999',
  },
});

export default Summerystep3;
