// Import validation patterns directly to avoid circular dependency
const EMAIL_VALIDATION = {
  email: /^[^\s@]+@[^\s@]+\.[^\s@]+$/,
  phone: /^\+?[\d\s\-\(\)]+$/,
  employeeNumber: /^[A-Z0-9]{3,10}$/
};

const EMAIL_CONFIG = {
  service: {
    name: 'Titan Drilling Email Service',
    from: 'noreply@titandrilling.com',
    replyTo: 'support@titandrilling.com',
    supportEmail: 'support@titandrilling.com'
  }
};

const EMAIL_STYLES = {
  colors: {
    primary: '#015185',
    secondary: '#3481BC',
    success: '#28a745',
    warning: '#ffc107',
    danger: '#dc3545',
    info: '#17a2b8',
    light: '#f8f9fa',
    dark: '#343a40'
  },
  borderRadius: '5px'
};

/**
 * Email utility functions for validation, formatting, and processing
 */
export class EmailUtils {
  /**
   * Validate email address format
   * @param {string} email - Email address to validate
   * @returns {boolean} - True if valid, false otherwise
   */
  static isValidEmail(email) {
    if (!email || typeof email !== 'string') {
      return false;
    }
    return EMAIL_VALIDATION.email.test(email.trim());
  }

  /**
   * Validate phone number format
   * @param {string} phone - Phone number to validate
   * @returns {boolean} - True if valid, false otherwise
   */
  static isValidPhone(phone) {
    if (!phone || typeof phone !== 'string') {
      return false;
    }
    return EMAIL_VALIDATION.phone.test(phone.trim());
  }

  /**
   * Validate employee number format
   * @param {string} employeeNumber - Employee number to validate
   * @returns {boolean} - True if valid, false otherwise
   */
  static isValidEmployeeNumber(employeeNumber) {
    if (!employeeNumber || typeof employeeNumber !== 'string') {
      return false;
    }
    return EMAIL_VALIDATION.employeeNumber.test(employeeNumber.trim().toUpperCase());
  }

  /**
   * Sanitize email content to prevent XSS
   * @param {string} content - HTML content to sanitize
   * @returns {string} - Sanitized HTML content
   */
  static sanitizeEmailContent(content) {
    if (!content || typeof content !== 'string') {
      return '';
    }

    // Remove potentially dangerous HTML tags and attributes
    const dangerousTags = ['script', 'iframe', 'object', 'embed', 'form', 'input', 'button'];
    const dangerousAttributes = ['onclick', 'onload', 'onerror', 'onmouseover', 'javascript:'];

    let sanitized = content;

    // Remove dangerous tags
    dangerousTags.forEach(tag => {
      const regex = new RegExp(`<${tag}[^>]*>.*?</${tag}>`, 'gi');
      sanitized = sanitized.replace(regex, '');
    });

    // Remove dangerous attributes
    dangerousAttributes.forEach(attr => {
      const regex = new RegExp(`${attr}=["'][^"']*["']`, 'gi');
      sanitized = sanitized.replace(regex, '');
    });

    return sanitized;
  }

  /**
   * Format date for email display
   * @param {string|Date} date - Date to format
   * @param {string} locale - Locale for formatting (default: 'en-US')
   * @returns {string} - Formatted date string
   */
  static formatDateForEmail(date, locale = 'en-US') {
    if (!date) {
      return 'N/A';
    }

    try {
      const dateObj = new Date(date);
      if (isNaN(dateObj.getTime())) {
        return 'Invalid Date';
      }

      return dateObj.toLocaleDateString(locale, {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
      });
    } catch (error) {
      console.error('Error formatting date for email:', error);
      return 'Invalid Date';
    }
  }

  /**
   * Format equipment status with color coding
   * @param {string} status - Equipment status
   * @returns {string} - HTML formatted status with color
   */
  static formatEquipmentStatus(status) {
    if (!status) {
      return '<span style="color: #666;">Unknown</span>';
    }

    const statusColors = {
      'Operational': '#28a745',
      'Maintenance Required': '#ffc107',
      'Out of Service': '#dc3545',
      'Under Review': '#17a2b8',
      'Pending': '#ffc107'
    };

    const color = statusColors[status] || '#666';
    return `<span style="color: ${color}; font-weight: bold;">${status}</span>`;
  }

  /**
   * Format priority level with color coding
   * @param {string} priority - Priority level
   * @returns {string} - HTML formatted priority with color
   */
  static formatPriority(priority) {
    if (!priority) {
      return '<span style="color: #666;">Not Set</span>';
    }

    const priorityColors = {
      'Low': '#28a745',
      'Medium': '#ffc107',
      'High': '#dc3545',
      'Critical': '#8B0000'
    };

    const color = priorityColors[priority] || '#666';
    return `<span style="color: ${color}; font-weight: bold;">${priority}</span>`;
  }

  /**
   * Generate email tracking pixel HTML
   * @param {string} emailId - Unique email identifier
   * @returns {string} - HTML for tracking pixel
   */
  static generateTrackingPixel(emailId) {
    if (!emailId) {
      return '';
    }

    return `
      <img src="${EMAIL_CONFIG.service.trackingUrl || 'https://example.com/track'}/${emailId}" 
           alt="" width="1" height="1" style="display:none;" />
    `;
  }

  /**
   * Generate unsubscribe link HTML
   * @param {string} email - Recipient email
   * @param {string} emailId - Email identifier
   * @returns {string} - HTML for unsubscribe link
   */
  static generateUnsubscribeLink(email, emailId) {
    if (!email || !emailId) {
      return '';
    }

    const unsubscribeUrl = `${EMAIL_CONFIG.service.unsubscribeUrl || 'https://example.com/unsubscribe'}?email=${encodeURIComponent(email)}&id=${emailId}`;
    
    return `
      <p style="font-size: 12px; color: #666; text-align: center; margin-top: 20px;">
        <a href="${unsubscribeUrl}" style="color: #666; text-decoration: underline;">
          Unsubscribe from these emails
        </a>
      </p>
    `;
  }

  /**
   * Generate email signature HTML
   * @param {Object} user - User information
   * @returns {string} - HTML for email signature
   */
  static generateEmailSignature(user) {
    if (!user) {
      return '';
    }

    return `
      <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee;">
        <p style="margin: 0; color: #333; font-size: 14px;">
          <strong>${user.name || 'Titan Drilling Team'}</strong><br>
          ${user.role ? `${user.role}<br>` : ''}
          ${user.employeeNumber ? `Employee #: ${user.employeeNumber}<br>` : ''}
          ${user.email ? `Email: ${user.email}<br>` : ''}
          ${user.phone ? `Phone: ${user.phone}` : ''}
        </p>
      </div>
    `;
  }

  /**
   * Generate table row HTML for email content
   * @param {string} label - Row label
   * @param {string} value - Row value
   * @param {boolean} isHighlighted - Whether to highlight the row
   * @returns {string} - HTML for table row
   */
  static generateTableRow(label, value, isHighlighted = false) {
    const highlightStyle = isHighlighted ? 'background-color: #f8f9fa;' : '';
    
    return `
      <tr style="${highlightStyle}">
        <td style="padding: 8px 12px; font-weight: bold; border-bottom: 1px solid #dee2e6; width: 40%;">
          ${label}
        </td>
        <td style="padding: 8px 12px; border-bottom: 1px solid #dee2e6; width: 60%;">
          ${value || 'N/A'}
        </td>
      </tr>
    `;
  }

  /**
   * Generate progress bar HTML for email content
   * @param {number} percentage - Progress percentage (0-100)
   * @param {string} label - Progress label
   * @param {string} color - Progress bar color
   * @returns {string} - HTML for progress bar
   */
  static generateProgressBar(percentage, label, color = EMAIL_STYLES.colors.primary) {
    const clampedPercentage = Math.max(0, Math.min(100, percentage));
    
    return `
      <div style="margin: 15px 0;">
        <div style="display: flex; justify-content: space-between; margin-bottom: 5px;">
          <span style="font-size: 14px; color: #333;">${label}</span>
          <span style="font-size: 14px; color: #666;">${clampedPercentage}%</span>
        </div>
        <div style="width: 100%; height: 8px; background-color: #e9ecef; border-radius: 4px; overflow: hidden;">
          <div style="width: ${clampedPercentage}%; height: 100%; background-color: ${color}; transition: width 0.3s ease;"></div>
        </div>
      </div>
    `;
  }

  /**
   * Generate button HTML for email content
   * @param {string} text - Button text
   * @param {string} url - Button URL
   * @param {string} color - Button color
   * @param {string} size - Button size (small, medium, large)
   * @returns {string} - HTML for button
   */
  static generateButton(text, url, color = EMAIL_STYLES.colors.primary, size = 'medium') {
    const sizeStyles = {
      small: 'padding: 8px 16px; font-size: 12px;',
      medium: 'padding: 12px 24px; font-size: 14px;',
      large: 'padding: 16px 32px; font-size: 16px;'
    };

    return `
      <a href="${url}" style="
        display: inline-block;
        background-color: ${color};
        color: white;
        text-decoration: none;
        border-radius: ${EMAIL_STYLES.borderRadius};
        font-weight: bold;
        text-align: center;
        ${sizeStyles[size] || sizeStyles.medium}
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
      ">
        ${text}
      </a>
    `;
  }

  /**
   * Generate alert box HTML for email content
   * @param {string} message - Alert message
   * @param {string} type - Alert type (success, warning, error, info)
   * @returns {string} - HTML for alert box
   */
  static generateAlertBox(message, type = 'info') {
    const alertColors = {
      success: EMAIL_STYLES.colors.success,
      warning: EMAIL_STYLES.colors.warning,
      error: EMAIL_STYLES.colors.danger,
      info: EMAIL_STYLES.colors.info
    };

    const color = alertColors[type] || alertColors.info;
    const icon = {
      success: '✅',
      warning: '⚠️',
      error: '❌',
      info: 'ℹ️'
    }[type] || 'ℹ️';

    return `
      <div style="
        background-color: ${color}15;
        border: 1px solid ${color};
        border-radius: ${EMAIL_STYLES.borderRadius};
        padding: 15px;
        margin: 15px 0;
        display: flex;
        align-items: center;
      ">
        <span style="font-size: 18px; margin-right: 10px;">${icon}</span>
        <span style="color: ${color}; font-weight: 500;">${message}</span>
      </div>
    `;
  }

  /**
   * Generate email preview text (for email clients that don't support HTML)
   * @param {string} htmlContent - HTML email content
   * @returns {string} - Plain text preview
   */
  static generatePreviewText(htmlContent) {
    if (!htmlContent) {
      return '';
    }

    // Remove HTML tags and decode HTML entities
    let text = htmlContent
      .replace(/<[^>]*>/g, ' ') // Remove HTML tags
      .replace(/&nbsp;/g, ' ') // Replace &nbsp; with space
      .replace(/&amp;/g, '&') // Replace &amp; with &
      .replace(/&lt;/g, '<') // Replace &lt; with <
      .replace(/&gt;/g, '>') // Replace &gt; with >
      .replace(/&quot;/g, '"') // Replace &quot; with "
      .replace(/&#39;/g, "'") // Replace &#39; with '
      .replace(/\s+/g, ' ') // Replace multiple spaces with single space
      .trim();

    // Limit to reasonable length for preview
    if (text.length > 150) {
      text = text.substring(0, 147) + '...';
    }

    return text;
  }

  /**
   * Validate email template data
   * @param {Object} data - Email template data
   * @param {Array} requiredFields - Array of required field names
   * @returns {Object} - Validation result with isValid boolean and errors array
   */
  static validateEmailData(data, requiredFields = []) {
    const errors = [];

    if (!data || typeof data !== 'object') {
      errors.push('Email data is required and must be an object');
      return { isValid: false, errors };
    }

    requiredFields.forEach(field => {
      if (!data[field]) {
        errors.push(`Required field '${field}' is missing`);
      }
    });

    return {
      isValid: errors.length === 0,
      errors
    };
  }

  /**
   * Generate unique email identifier
   * @param {string} prefix - Prefix for the identifier
   * @returns {string} - Unique identifier
   */
  static generateEmailId(prefix = 'email') {
    const timestamp = Date.now();
    const random = Math.random().toString(36).substring(2, 8);
    return `${prefix}_${timestamp}_${random}`;
  }

  /**
   * Check if email should be sent based on user preferences
   * @param {Object} user - User object with preferences
   * @param {string} emailType - Type of email being sent
   * @returns {boolean} - True if email should be sent
   */
  static shouldSendEmail(user, emailType) {
    if (!user || !user.emailPreferences) {
      return true; // Default to sending if no preferences set
    }

    const preferences = user.emailPreferences;
    
    // Check if user has opted out of all emails
    if (preferences.optOutAll) {
      return false;
    }

    // Check if user has opted out of specific email type
    if (preferences.optOutTypes && preferences.optOutTypes.includes(emailType)) {
      return false;
    }

    // Check if user has specific preferences for this email type
    if (preferences[emailType] === false) {
      return false;
    }

    return true;
  }
}

export default EmailUtils;
