import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Modal,
  FlatList,
  KeyboardAvoidingView,
  Platform,
  PermissionsAndroid,
  Animated,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DropDownPicker from 'react-native-dropdown-picker';
import { Dimensions } from 'react-native';
import Geolocation from '@react-native-community/geolocation';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import { Dropdown } from 'react-native-element-dropdown';
import InspectionAPI from '../../ConfigMongo/InspectionMongoConfig.js';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';
import EquipmentAPI from '../../ConfigMongo/EquipmentMongoConfig';
import OfflineDataHelper from './OfflineDataHelper';

const { width } = Dimensions.get('window');

// Custom Input Field Component
const CustomInputField = ({ label, value, onChangeText, isRequired = false, showError = false, errorMessage = '', ...props }) => (
  <View style={inputStyles.container}>
    <Text style={inputStyles.label}>
      {label}
      {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
    </Text>
    <TextInput
      style={[inputStyles.input, showError ? inputStyles.inputError : null]}
      value={value}
      onChangeText={onChangeText}
      {...props}
      placeholderTextColor="#888"
    />
    {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
  </View>
);

// SyncModal Component
const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.modalOverlay}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const InsInfostep1 = ({ setActiveStep, equipment }) => {
  const navigation = useNavigation();
  const route = useRoute();
  const [country, setCountry] = useState(null);
  const [countryOpen, setCountryOpen] = useState(false);
  const [countryItems] = useState([{ label: 'Zambia', value: 'Zambia' }, { label: 'DRC', value: 'DRC' }]);
  const [project, setProject] = useState(null);
  const [projectOpen, setProjectOpen] = useState(false);
  const [projectItems, setProjectItems] = useState([]);
  const [inspector, setInspector] = useState('');
  const [checklistModalVisible, setChecklistModalVisible] = useState(false);
  const [equipmentModalVisible, setEquipmentModalVisible] = useState(false);
  const [checklists, setChecklists] = useState([]);
  const [firebaseEquipment, setFirebaseEquipment] = useState([]);
  const [checklistSearchQuery, setChecklistSearchQuery] = useState('');
  const [isChecklistSearchActive, setIsChecklistSearchActive] = useState(false);
  const [selectedTab, setSelectedTab] = useState('ALL');
  const [selectedChecklistId, setSelectedChecklistId] = useState(null);
  const [checklistTitle, setChecklistTitle] = useState('');
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [meterReading, setMeterReading] = useState('');
  const [currentMeterReading, setCurrentMeterReading] = useState('');
  const [requestNumber, setRequestNumber] = useState('');
  const [equipmentSearchQuery, setEquipmentSearchQuery] = useState('');
  const [isEquipmentSearchActive, setIsEquipmentSearchActive] = useState(false);
  const [showErrors, setShowErrors] = useState(false);
  const [countryError, setCountryError] = useState(false);
  const [projectError, setProjectError] = useState(false);
  const [inspectorError, setInspectorError] = useState(false);
  const [equipmentError, setEquipmentError] = useState(false);
  const [checklistError, setChecklistError] = useState(false);
  const [coordinates, setCoordinates] = useState('');
  const [gpsAddress, setGpsAddress] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const equipmentSearchInputRef = useRef(null);
  const isMounted = useRef(true);
  const ADMIN_UID = 'cahj3jJqe5Z16pQdLGSWrNgm84o1';
  const [equipmentLoading, setEquipmentLoading] = useState(false); // Add loading state for equipment modal
  const [totalReportsCount, setTotalReportsCount] = useState(0); // Total inspection reports count

  // Fetch total inspection reports count
  const fetchTotalReportsCount = async () => {
    try {
      if (isOnline) {
        console.log('🔢 Fetching total reports count...');
        const response = await InspectionAPI.getTotalCount();
        if (response.success && response.total !== undefined) {
          setTotalReportsCount(response.total);
          console.log('📊 Total inspection reports:', response.total);
        } else {
          console.log('⚠️ Count response not successful, trying fallback...');
          // Fallback to getAllReports
          const fallbackResponse = await InspectionAPI.getAllReports({ page: 1, limit: 1 });
          if (fallbackResponse.success && fallbackResponse.pagination) {
            setTotalReportsCount(fallbackResponse.pagination.total || 0);
            console.log('📊 Total inspection reports (fallback):', fallbackResponse.pagination.total);
          }
        }
      } else {
        // Offline mode - get from local storage
        const savedInspections = await AsyncStorage.getItem('inspections');
        const inspections = savedInspections ? JSON.parse(savedInspections) : [];
        setTotalReportsCount(inspections.length);
      }
    } catch (error) {
      console.error('Error fetching total reports count:', error);
      console.error('Error details:', error.message);
      setTotalReportsCount(0);
      
      // Try fallback method
      try {
        console.log('🔄 Trying fallback method...');
        const fallbackResponse = await InspectionAPI.getAllReports({ page: 1, limit: 1 });
        if (fallbackResponse.success && fallbackResponse.pagination) {
          setTotalReportsCount(fallbackResponse.pagination.total || 0);
          console.log('📊 Total inspection reports (fallback):', fallbackResponse.pagination.total);
        }
      } catch (fallbackError) {
        console.error('Fallback method also failed:', fallbackError);
      }
    }
  };

  // Test function to verify request number generation logic
  const testRequestNumberGeneration = async () => {
    console.log('🧪 Testing request number generation...');
    
    // First, test the API endpoint directly
    try {
      console.log('🧪 Testing API endpoint directly...');
      const response = await fetch('https://api.titandrillingzm.com:6004/inspections/all?page=1&limit=10000');
      const data = await response.json();
      console.log('🧪 Direct API test result:', {
        status: response.status,
        success: data.success,
        dataLength: data.data ? data.data.length : 'no data',
        total: data.pagination ? data.pagination.total : 'no pagination'
      });
      
      if (data.data && data.data.length > 0) {
        console.log('🧪 Sample report from direct API:', {
          _id: data.data[0]._id,
          requestNumber: data.data[0].requestNumber,
          userId: data.data[0].userId
        });
      }
    } catch (directError) {
      console.error('🧪 Direct API test failed:', directError);
    }
    
    // Test scenarios based on your database
    const testCases = [
      { country: 'DRC', project: 'Lubumbashi', expectedPrefix: 'DRC-LUB-IR-' },
      { country: 'DRC', project: 'Musompo', expectedPrefix: 'DRC-MUS-IR-' },
      { country: 'Zambia', project: 'Kansanshi', expectedPrefix: 'ZAM-KAN-IR-' },
    ];
    
    for (const testCase of testCases) {
      console.log(`\n🧪 Testing ${testCase.country} - ${testCase.project}:`);
      console.log(`   Expected prefix: ${testCase.expectedPrefix}`);
    }
  };

  // Generate inspection request number based on total reports count
  const generateRequestNumber = async () => {
    try {
      // Generate country and project codes first
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      
      // Generate project code based on project name
      let projectCode = 'XXX';
      if (project) {
        switch (project.toLowerCase()) {
          case 'lubumbashi':
            projectCode = 'LUB';
            break;
          case 'musompo':
            projectCode = 'MUS';
            break;
          case 'kansanshi':
            projectCode = 'KAN';
            break;
          case 'kalumbila':
            projectCode = 'KAL';
            break;
          case 'kimteto':
            projectCode = 'KIM';
            break;
          case 'kobold':
            projectCode = 'KOB';
            break;
          case 'fqm exploration':
            projectCode = 'FQM';
            break;
          case 'mimosa':
            projectCode = 'MIM';
            break;
          case 'ime':
            projectCode = 'IME';
            break;
          case 'kamoa':
            projectCode = 'KAM';
            break;
          default:
            // Fallback to first 3 characters
            projectCode = project.slice(0, 3).toUpperCase();
            break;
        }
      }
      
      const prefix = `${countryCode}-${projectCode}-IR-`;
      let nextNumber = 1; // Default starting number
      
      if (isOnline && country && project) {
        try {
          console.log(`🔢 Using total reports count: ${totalReportsCount}`);
          
          // Use totalReportsCount + 1 for the next number
          nextNumber = totalReportsCount + 1;
          console.log(`📈 Total reports: ${totalReportsCount}, Next number: ${nextNumber}`);
          console.log(`✨ Generated number: ${String(nextNumber).padStart(3, '0')}`);
        } catch (error) {
          console.error('❌ Error using total count:', error.message);
          nextNumber = 1;
        }
      } else {
        console.log('⚠️ Offline or missing country/project, using 001');
        nextNumber = 1;
      }
      
      const newRequestNumber = `${prefix}${String(nextNumber).padStart(3, '0')}`;
      console.log(`✅ Generated request number: ${newRequestNumber}`);
      return newRequestNumber;
    } catch (error) {
      console.error('❌ Error generating request number:', error);
      // Fallback for offline or error scenarios
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      
      // Generate project code based on project name
      let projectCode = 'XXX';
      if (project) {
        switch (project.toLowerCase()) {
          case 'lubumbashi':
            projectCode = 'LUB';
            break;
          case 'musompo':
            projectCode = 'MUS';
            break;
          case 'kansanshi':
            projectCode = 'KAN';
            break;
          case 'kalumbila':
            projectCode = 'KAL';
            break;
          case 'kimteto':
            projectCode = 'KIM';
            break;
          case 'kobold':
            projectCode = 'KOB';
            break;
          case 'fqm exploration':
            projectCode = 'FQM';
            break;
          case 'mimosa':
            projectCode = 'MIM';
            break;
          case 'ime':
            projectCode = 'IME';
            break;
          case 'kamoa':
            projectCode = 'KAM';
            break;
          default:
            projectCode = project.slice(0, 3).toUpperCase();
            break;
        }
      }
      const newRequestNumber = `${countryCode}-${projectCode}-IR-001`;
      console.log(`🔄 Fallback request number: ${newRequestNumber}`);
      return newRequestNumber;
    }
  };

  // Generate inspection request number with explicit country and project parameters
  const generateRequestNumberWithParams = async (countryParam, projectParam) => {
    try {
      // Generate country and project codes first
      const countryCode = countryParam === 'Zambia' ? 'ZAM' : countryParam === 'DRC' ? 'DRC' : 'XXX';
      
      // Generate project code based on project name
      let projectCode = 'XXX';
      if (projectParam) {
        switch (projectParam.toLowerCase()) {
          case 'lubumbashi':
            projectCode = 'LUB';
            break;
          case 'musompo':
            projectCode = 'MUS';
            break;
          case 'kansanshi':
            projectCode = 'KAN';
            break;
          case 'kalumbila':
            projectCode = 'KAL';
            break;
          case 'kimteto':
            projectCode = 'KIM';
            break;
          case 'kobold':
            projectCode = 'KOB';
            break;
          case 'fqm exploration':
            projectCode = 'FQM';
            break;
          case 'mimosa':
            projectCode = 'MIM';
            break;
          case 'ime':
            projectCode = 'IME';
            break;
          case 'kamoa':
            projectCode = 'KAM';
            break;
          default:
            // Fallback to first 3 characters
            projectCode = projectParam.slice(0, 3).toUpperCase();
            break;
        }
      }
      
      const prefix = `${countryCode}-${projectCode}-IR-`;
      let nextNumber = 1; // Default starting number
      
      if (isOnline && countryParam && projectParam) {
        try {
          console.log(`🔢 Using total reports count: ${totalReportsCount}`);
          
          // Use totalReportsCount + 1 for the next number
          nextNumber = totalReportsCount + 1;
          console.log(`📈 Total reports: ${totalReportsCount}, Next number: ${nextNumber}`);
          console.log(`✨ Generated number: ${String(nextNumber).padStart(3, '0')}`);
        } catch (error) {
          console.error('❌ Error using total count:', error.message);
          nextNumber = 1;
        }
      } else {
        console.log('⚠️ Offline or missing country/project, using 001');
        nextNumber = 1;
      }
      
      const newRequestNumber = `${prefix}${String(nextNumber).padStart(3, '0')}`;
      console.log(`✅ Generated request number: ${newRequestNumber}`);
      return newRequestNumber;
    } catch (error) {
      console.error('❌ Error generating request number:', error);
      // Fallback for offline or error scenarios
      const countryCode = countryParam === 'Zambia' ? 'ZAM' : countryParam === 'DRC' ? 'DRC' : 'XXX';
      
      // Generate project code based on project name
      let projectCode = 'XXX';
      if (projectParam) {
        switch (projectParam.toLowerCase()) {
          case 'lubumbashi':
            projectCode = 'LUB';
            break;
          case 'musompo':
            projectCode = 'MUS';
            break;
          case 'kansanshi':
            projectCode = 'KAN';
            break;
          case 'kalumbila':
            projectCode = 'KAL';
            break;
          case 'kimteto':
            projectCode = 'KIM';
            break;
          case 'kobold':
            projectCode = 'KOB';
            break;
          case 'fqm exploration':
            projectCode = 'FQM';
            break;
          case 'mimosa':
            projectCode = 'MIM';
            break;
          case 'ime':
            projectCode = 'IME';
            break;
          case 'kamoa':
            projectCode = 'KAM';
            break;
          default:
            projectCode = projectParam.slice(0, 3).toUpperCase();
            break;
        }
      }
      const newRequestNumber = `${countryCode}-${projectCode}-IR-001`;
      console.log(`🔄 Fallback request number: ${newRequestNumber}`);
      return newRequestNumber;
    }
  };

  const saveInspectionData = async (data) => {
    try {
      await AsyncStorage.setItem('inspectionStep1', JSON.stringify(data));
      if (!isOnline) {
        const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
        syncQueue.push({ type: 'inspectionStep1', data });
        await AsyncStorage.setItem('syncQueue', JSON.stringify(syncQueue));
        Toast.show({ type: 'info', text1: 'Offline', text2: 'Data saved locally. Will sync when online.', position: 'top' });
      }
    } catch (error) {
      console.error('Error saving inspection data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save data locally.', position: 'top' });
    }
  };

  const syncDataToMongoDB = async () => {
    if (!isOnline) return;

    try {
      const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
      if (syncQueue.length === 0) return;

      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        Toast.show({ type: 'error', text1: 'Authentication Error', text2: 'User not authenticated. Please log in.', position: 'top' });
        return;
      }

      for (const item of syncQueue) {
        if (item.type === 'inspectionStep1') {
          // IMPORTANT: Check if report already exists before creating draft
          try {
            const existingReport = await InspectionAPI.getInspectionById(item.data.id);
            if (existingReport.success && existingReport.data) {
              console.log('⚠️ Draft report already exists in MongoDB, skipping:', item.data.id);
              continue; // Skip this item, don't create duplicate
            }
          } catch (checkError) {
            // Report doesn't exist, proceed with creation
            console.log('✅ Report not found in MongoDB, creating draft...');
          }
          
          // Save to MongoDB instead of Firebase
          await InspectionAPI.createInspection(userId, {
            _id: item.data.id,
            userId,
            requestNumber: item.data.requestNumber,
            status: 'draft',
            step1: item.data,
            step2: { sections: [] },
            step3: { signatureFields: [], selectedSignatureTypes: [] },
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
          });
          console.log('✅ Draft report created in MongoDB:', item.data.id);
        }
      }
      await AsyncStorage.setItem('syncQueue', '[]');
      Toast.show({ type: 'success', text1: 'Sync Complete', text2: 'Offline data synced to MongoDB.', position: 'top' });
    } catch (error) {
      console.error('Error syncing data to MongoDB:', error);
      Toast.show({ type: 'error', text1: 'Sync Error', text2: 'Failed to sync data. Will retry when online.', position: 'top' });
    }
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected);
      if (state.isConnected) {
        syncDataToMongoDB();
        fetchTotalReportsCount();
      }
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected);
      if (state.isConnected) {
        syncDataToMongoDB();
        fetchTotalReportsCount();
      }
    });

    return () => unsubscribe();
  }, []);

  // Update loadEquipmentData to accept showSyncModal argument
  // Load equipment data - FAST offline loading from JSON
  const loadEquipmentData = async (showSyncModal = true) => {
    // Don't show sync modal for equipment - modal has its own loading indicator
    try {
      let data = [];
      
      // Quick check: Try cache first (fastest)
      const cachedEquipment = await AsyncStorage.getItem('equipment');
      if (cachedEquipment) {
        try {
          data = JSON.parse(cachedEquipment);
          if (data && data.length > 0) {
            console.log('⚡ FAST: Loaded equipment from cache:', data.length);
            setFirebaseEquipment(data);
            setEquipmentLoading(false); // Hide loading immediately
            
            // Background refresh if online
            const netInfo = await NetInfo.fetch();
            if (netInfo.isConnected && isOnline) {
              console.log('🔄 Background: Refreshing from MongoDB...');
              // Refresh in background without blocking UI
              EquipmentAPI.getAllEquipment()
                .then(response => {
                  if (response.success && response.data) {
                    const freshData = response.data.filter((item, index, self) => 
                      self.findIndex((t) => t.id === item.id) === index
                    );
                    freshData.sort((a, b) => {
                      const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
                      const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
                      return dateB - dateA;
                    });
                    AsyncStorage.setItem('equipment', JSON.stringify(freshData));
                    setFirebaseEquipment(freshData);
                    console.log('✅ Background refresh complete:', freshData.length);
                  }
                })
                .catch(err => console.log('⚠️ Background refresh failed:', err.message));
            }
            return; // Exit early - data already loaded!
          }
        } catch (parseError) {
          console.log('⚠️ Cache parse failed, loading fresh...');
        }
      }
      
      // If no cache, check network status
      const netInfo = await NetInfo.fetch();
      
      if (netInfo.isConnected) {
        // Online: Try MongoDB with timeout
        console.log('🌐 Online: Loading equipment from MongoDB...');
        try {
          // Set a timeout for MongoDB request
          const timeoutPromise = new Promise((_, reject) => 
            setTimeout(() => reject(new Error('MongoDB timeout')), 5000)
          );
          
          const response = await Promise.race([
            EquipmentAPI.getAllEquipment(),
            timeoutPromise
          ]);
          
          if (response.success && response.data) {
            data = response.data;
            console.log(`✅ Loaded ${data.length} equipment items from MongoDB`);
            
            // Deduplicate and sort
            data = data.filter((item, index, self) => 
              self.findIndex((t) => t.id === item.id) === index
            );
            data.sort((a, b) => {
              const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
              const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
              return dateB - dateA;
            });
            
            // Cache for next time
            await AsyncStorage.setItem('equipment', JSON.stringify(data));
          } else {
            throw new Error('No equipment data from MongoDB');
          }
        } catch (mongoError) {
          console.log('⚠️ MongoDB failed or timeout, using offline data:', mongoError.message);
          // Load from offline JSON immediately
          data = await OfflineDataHelper.getEquipmentData();
        }
      } else {
        // Offline mode - load from JSON file IMMEDIATELY
        console.log('📴 OFFLINE: Loading from JSON file...');
        data = await OfflineDataHelper.getEquipmentData();
        console.log(`⚡ FAST: Loaded ${data.length} equipment from offline JSON`);
      }
      
      // Sort data
      data.sort((a, b) => {
        const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
        const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
        return dateB - dateA;
      });
      
      setFirebaseEquipment(data);
      
      if (!netInfo.isConnected) {
        Toast.show({ 
          type: 'info', 
          text1: 'Offline Mode', 
          text2: `Loaded ${data.length} equipment`, 
          position: 'top',
          visibilityTime: 2000
        });
      }
    } catch (error) {
      console.error('❌ Error loading equipment:', error);
      // Final fallback: OfflineDataHelper
      try {
        console.log('🔄 Final fallback: Loading from offline JSON...');
        const data = await OfflineDataHelper.getEquipmentData();
        setFirebaseEquipment(data);
        console.log(`✅ Loaded ${data.length} equipment from offline fallback`);
      } catch (offlineError) {
        console.error('❌ All loading methods failed:', offlineError);
        setFirebaseEquipment([]);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'No equipment data available',
          position: 'top',
        });
      }
    } finally {
      setEquipmentLoading(false); // Always hide loading
      setEquipmentSearchQuery('');
      setIsEquipmentSearchActive(false);
    }
  };

  useEffect(() => {
    const loadData = async () => {
      try {
        // Pre-load equipment and checklists data in background for instant modal opening
        console.log('⚡ Pre-loading offline data for fast access...');
        Promise.all([
          OfflineDataHelper.getEquipmentData(),
          OfflineDataHelper.getChecklistsData()
        ])
          .then(([equipmentData, checklistData]) => {
            console.log('✅ Pre-loaded:', equipmentData.length, 'equipment,', checklistData.length, 'checklists');
          })
          .catch(err => console.log('⚠️ Pre-load failed:', err.message));
        
        // Test the request number generation logic
        await testRequestNumberGeneration();
        
        // Fetch total reports count (skip if offline to avoid delays)
        const netInfo = await NetInfo.fetch();
        if (netInfo.isConnected) {
          await fetchTotalReportsCount();
        } else {
          console.log('📴 Offline: Skipping total count fetch');
          setTotalReportsCount(0);
        }

        const savedInspection = await AsyncStorage.getItem('inspectionStep1');
        let inspectionData = savedInspection ? JSON.parse(savedInspection) : {};

        // Load checklists if we have checklist params or if checklists are not loaded yet
        if (route.params?.checklistId || route.params?.checklistData || checklists.length === 0) {
          console.log('Loading checklists on component mount...');
          await loadChecklists();
        }

        const savedChecklistIds = await AsyncStorage.getItem('checklistIds');
        const parsedChecklistIds = savedChecklistIds ? JSON.parse(savedChecklistIds) : [];
        if (parsedChecklistIds.length > 0) setSelectedChecklistId(parsedChecklistIds[0]);

        if (route.params?.checklistTitle) {
          setChecklistTitle(route.params.checklistTitle);
          inspectionData.checklistTitle = route.params.checklistTitle;
        } else if (inspectionData.checklistTitle) setChecklistTitle(inspectionData.checklistTitle);

        if (route.params?.checklistId) {
          setSelectedChecklistId(route.params.checklistId);
          inspectionData.selectedChecklistId = route.params.checklistId;
        } else if (inspectionData.selectedChecklistId) setSelectedChecklistId(inspectionData.selectedChecklistId);

        // Handle checklist data from route params
        if (route.params?.checklistData) {
          const checklistData = route.params.checklistData;
          // Convert checklist data to the expected format
          const convertedChecklist = {
            id: checklistData.id,
            title: checklistData.title,
            description: checklistData.description || '',
            items: checklistData.sections ? checklistData.sections.map(section => ({
              text: section.sectionName,
              subItems: section.subItems.map(subItem => ({
                itemName: subItem.itemName,
                inputType: subItem.inputType || 'Ok/Reject/N/A',
                description: subItem.description || '',
                instruction: subItem.instruction || '',
                sortOrder: subItem.sortOrder || 1,
                isInspectionRequired: subItem.isInspectionRequired || false,
              }))
            })) : [],
            status: 'Available',
            isUsed: false,
            isUserChecklist: false,
          };
          
          // Add to checklists if not already present
          setChecklists(prevChecklists => {
            const existing = prevChecklists.find(c => c.id === convertedChecklist.id);
            if (!existing) {
              return [...prevChecklists, convertedChecklist];
            }
            return prevChecklists;
          });
          
          setSelectedChecklistId(checklistData.id);
          setChecklistTitle(checklistData.title);
          inspectionData.selectedChecklistId = checklistData.id;
          inspectionData.checklistTitle = checklistData.title;
        }

        if (equipment) {
          setSelectedEquipment(equipment);
          setMeterReading(equipment.meterReading?.toString() || '');
          inspectionData.selectedEquipment = equipment;
        } else if (inspectionData.selectedEquipment) {
          setSelectedEquipment(inspectionData.selectedEquipment);
          setMeterReading(inspectionData.selectedEquipment.meterReading?.toString() || '');
        }

        // Set country and project first
        setCountry(inspectionData.country || null);
        setProject(inspectionData.project || null);
        setInspector(inspectionData.inspector || '');
        setCurrentMeterReading(inspectionData.currentMeterReading || '');
        setCoordinates(inspectionData.coordinates || '');
        setGpsAddress(inspectionData.gpsAddress || '');

        // Generate request number using the loaded country and project values
        const newRequestNumber = await generateRequestNumberWithParams(inspectionData.country, inspectionData.project);
        setRequestNumber(newRequestNumber);

        await saveInspectionData(inspectionData);
      } catch (error) {
        console.error('Error loading data:', error);
        Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load initial data. Please try again.', position: 'top' });
      }
    };
    loadData();

    return () => {
      isMounted.current = false;
    };
  }, [equipment, route.params]);

  // Additional useEffect to ensure checklists are loaded when route params change
  useEffect(() => {
    if (route.params?.checklistId || route.params?.checklistData) {
      const handleChecklistFromParams = async () => {
        try {
          // Load checklists if not already loaded
          if (checklists.length === 0) {
            await loadChecklists();
          }
          
          // Handle checklist data from route params
          if (route.params?.checklistData) {
            const checklistData = route.params.checklistData;
            const convertedChecklist = {
              id: checklistData.id,
              title: checklistData.title,
              description: checklistData.description || '',
              items: checklistData.sections ? checklistData.sections.map(section => ({
                text: section.sectionName,
                subItems: section.subItems.map(subItem => ({
                  itemName: subItem.itemName,
                  inputType: subItem.inputType || 'Ok/Reject/N/A',
                  description: subItem.description || '',
                  instruction: subItem.instruction || '',
                  sortOrder: subItem.sortOrder || 1,
                  isInspectionRequired: subItem.isInspectionRequired || false,
                }))
              })) : [],
              status: 'Available',
              isUsed: false,
              isUserChecklist: false,
            };
            
            setChecklists(prevChecklists => {
              const existing = prevChecklists.find(c => c.id === convertedChecklist.id);
              if (!existing) {
                return [...prevChecklists, convertedChecklist];
              }
              return prevChecklists;
            });
            
            setSelectedChecklistId(checklistData.id);
            setChecklistTitle(checklistData.title);
          }
        } catch (error) {
          console.error('Error handling checklist from params:', error);
        }
      };
      
      handleChecklistFromParams();
    }
  }, [route.params?.checklistId, route.params?.checklistData]);

  useEffect(() => {
    const inspectionData = {
      id: Date.now().toString(),
      requestNumber,
      country,
      project,
      inspector,
      selectedEquipment,
      meterReading,
      currentMeterReading,
      selectedChecklistId,
      checklistTitle,
      coordinates,
      gpsAddress,
      createdAt: new Date().toISOString(),
    };
    saveInspectionData(inspectionData);
  }, [country, project, inspector, selectedEquipment, meterReading, currentMeterReading, selectedChecklistId, checklistTitle, coordinates, gpsAddress, requestNumber]);

  // Manual function to update request number - can be called anytime
  const updateRequestNumber = async () => {
    try {
      console.log('🔄 Updating request number...');
      console.log('📍 Current country:', country);
      console.log('📍 Current project:', project);
      
      if (!country || !project) {
        console.log('⚠️ Country or project not selected yet');
        return;
      }
      
      const newRequestNumber = await generateRequestNumber();
      console.log('✅ New request number generated:', newRequestNumber);
      setRequestNumber(newRequestNumber);
      
      // Also update the saved inspection data
      const savedInspection = await AsyncStorage.getItem('inspectionStep1');
      let inspectionData = savedInspection ? JSON.parse(savedInspection) : {};
      inspectionData.requestNumber = newRequestNumber;
      await AsyncStorage.setItem('inspectionStep1', JSON.stringify(inspectionData));
      
      Toast.show({
        type: 'success',
        text1: 'Request Number Updated',
        text2: `New number: ${newRequestNumber}`,
        position: 'top'
      });
    } catch (error) {
      console.error('❌ Error updating request number:', error);
      Toast.show({
        type: 'error',
        text1: 'Update Failed',
        text2: 'Could not update request number',
        position: 'top'
      });
    }
  };

  useEffect(() => {
    // Only update if both country and project are selected
    if (country && project) {
      console.log('🎯 Country/Project both selected, updating request number...');
      updateRequestNumber();
    }
  }, [country, project]);

  // Update request number when totalReportsCount changes
  useEffect(() => {
    if (country && project && totalReportsCount > 0) {
      console.log('🔄 Total reports count changed, updating request number...');
      updateRequestNumber();
    }
  }, [totalReportsCount]);

  useEffect(() => {
    if (country === 'Zambia') {
      setProjectItems([
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ]);
    } else if (country === 'DRC') {
      setProjectItems([
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ]);
    } else {
      setProjectItems([]);
    }
    if (!country || (project && !projectItems.some((item) => item.value === project))) setProject(null);
  }, [country]);

  // Test function to create sample custom checklist
  const createTestCustomChecklist = async () => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) return;

      const testChecklist = {
        checklistData: {
          id: `test_${Date.now()}`,
          title: 'Test Custom Checklist',
          description: 'This is a test custom checklist',
          items: [
            {
              text: 'Safety Section',
              subItems: [
                {
                  itemName: 'Check Safety Equipment',
                  inputType: 'Ok/Reject/N/A',
                  description: 'Verify all safety equipment is present',
                  instruction: 'Check each item carefully',
                  sortOrder: 1,
                  isInspectionRequired: true
                },
                {
                  itemName: 'Test Emergency Systems',
                  inputType: 'Ok/Not Ok',
                  description: 'Test emergency stop systems',
                  instruction: 'Press emergency stop button',
                  sortOrder: 2,
                  isInspectionRequired: true
                }
              ]
            }
          ]
        },
        userId: userId
      };

      const response = await fetch(`https://api.titandrillingzm.com:6004/checklists/custom`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(testChecklist),
      });

      const result = await response.json();
      console.log('🧪 Test checklist creation result:', result);
      
      if (result.success) {
        Toast.show({
          type: 'success',
          text1: 'Test Data',
          text2: 'Test custom checklist created successfully!',
        });
        // Reload checklists to show the new one
        loadChecklists();
      }
    } catch (error) {
      console.error('❌ Error creating test checklist:', error);
    }
  };

  const loadChecklists = async () => {
    setSyncModalVisible(true);
    try {
      // Quick check: Try cache first (fastest)
      const cachedChecklists = await AsyncStorage.getItem('checklists');
      if (cachedChecklists) {
        try {
          const data = JSON.parse(cachedChecklists);
          if (data && data.length > 0) {
            console.log('⚡ FAST: Loaded checklists from cache:', data.length);
            setChecklists(data);
            setSyncModalVisible(false);
            
            // Background refresh if online
            const netInfo = await NetInfo.fetch();
            if (netInfo.isConnected) {
              console.log('🔄 Background: Refreshing checklists from MongoDB...');
              // Continue with MongoDB refresh in background
            } else {
              // Offline - no need to refresh
              setChecklistSearchQuery('');
              setIsChecklistSearchActive(false);
              return;
            }
          }
        } catch (parseError) {
          console.log('⚠️ Cache parse failed, loading fresh...');
        }
      }
      
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        // Offline mode - load from OfflineDataHelper FAST
        console.log('📴 OFFLINE: Loading from JSON file...');
        const offlineChecklists = await OfflineDataHelper.getChecklistsData();
        setChecklists(offlineChecklists);
        console.log(`⚡ FAST: Loaded ${offlineChecklists.length} checklists from offline JSON`);
        Toast.show({ 
          type: 'info', 
          text1: 'Offline Mode', 
          text2: `${offlineChecklists.length} checklists loaded`, 
          position: 'top',
          visibilityTime: 2000
        });
        setSyncModalVisible(false);
        setChecklistSearchQuery('');
        setIsChecklistSearchActive(false);
        return;
      }

      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        Toast.show({ type: 'error', text1: 'Authentication Error', text2: 'Please log in to access checklists.', position: 'top' });
        setChecklists([]);
        return;
      }

      // Get user inspection reports to determine used checklists
      const inspectionsResponse = await InspectionAPI.getInspections(userId, { page: 1, limit: 1000 });
      const inspections = inspectionsResponse.success ? inspectionsResponse.data : [];
      const usedChecklistIds = new Set(inspections.map((i) => i.step1?.selectedChecklistId).filter(Boolean));

      // Fetch global checklists from MongoDB
      const checklistsResponse = await InspectionAPI.getChecklists({ page: 1, limit: 100 });
      let documentsArr = [];
      if (checklistsResponse.success && checklistsResponse.data) {
        documentsArr = checklistsResponse.data;
      }

      // Fetch ALL custom checklists from MongoDB (not just current user's)
      let customChecklistsArr = [];
      try {
        console.log('🌐 Fetching all custom checklists from API...');
        const customChecklistsResponse = await fetch(`https://api.titandrillingzm.com:6004/checklists/custom/all?limit=1000`);
        console.log('🌐 API Response status:', customChecklistsResponse.status);
        
        if (customChecklistsResponse.ok) {
          const customData = await customChecklistsResponse.json();
          console.log('🌐 API Response data:', customData);
          if (customData.success && customData.data && Array.isArray(customData.data)) {
            customChecklistsArr = customData.data;
            console.log('✅ Loaded ALL custom checklists for all users:', customChecklistsArr.length);
            console.log('📋 Sample custom checklist:', customChecklistsArr[0]);
          } else {
            console.log('⚠️ API returned success: false or no data');
            console.log('⚠️ Response:', customData);
            customChecklistsArr = [];
          }
        } else {
          console.log('❌ API request failed with status:', customChecklistsResponse.status);
          const errorText = await customChecklistsResponse.text();
          console.log('❌ Error response:', errorText);
          customChecklistsArr = [];
        }
      } catch (error) {
        console.error('❌ Error fetching all custom checklists:', error);
        customChecklistsArr = [];
        
        // Fallback to user-specific checklists if the "all" endpoint fails
        try {
          console.log('🔄 Fallback: Fetching user-specific custom checklists...');
          const fallbackResponse = await fetch(`https://api.titandrillingzm.com:6004/checklists/custom/${userId}?limit=1000`);
          console.log('🔄 Fallback API Response status:', fallbackResponse.status);
          
          if (fallbackResponse.ok) {
            const fallbackData = await fallbackResponse.json();
            console.log('🔄 Fallback API Response data:', fallbackData);
            if (fallbackData.success && fallbackData.data && Array.isArray(fallbackData.data)) {
              customChecklistsArr = fallbackData.data;
              console.log('✅ Fallback: Loaded user-specific custom checklists:', customChecklistsArr.length);
            } else {
              console.log('⚠️ Fallback API returned success: false or no data');
              console.log('⚠️ Fallback Response:', fallbackData);
            }
          } else {
            console.log('❌ Fallback API request failed with status:', fallbackResponse.status);
            const errorText = await fallbackResponse.text();
            console.log('❌ Fallback Error response:', errorText);
          }
        } catch (fallbackError) {
          console.error('❌ Error fetching user-specific custom checklists:', fallbackError);
        }
      }

      console.log('Loaded global checklists from MongoDB:', documentsArr.length);
      console.log('Loaded custom checklists from MongoDB:', customChecklistsArr.length);
      console.log('Sample global checklist data:', documentsArr[0]);
      console.log('Sample custom checklist data:', customChecklistsArr[0]);
      console.log('Full custom checklists array:', customChecklistsArr);

      // Map global checklists from MongoDB
      const globalChecklists = documentsArr.map((doc) => {
        const items = Object.entries(doc.checklist || {}).map(([sectionKey, sectionItems]) => ({
          text: `${sectionKey}`,
          subItems: Array.isArray(sectionItems)
            ? sectionItems.map((item, index) => ({
                itemName: typeof item === 'string' ? item : item.itemName || 'Unnamed Item',
                inputType: 'Ok/Reject/N/A',
                description: '',
                instruction: '',
                sortOrder: index + 1,
                isInspectionRequired: false,
              }))
            : [],
        }));
        return {
          id: doc.documentId,
          title: doc.documentName,
          description: '',
          items,
          status: 'Available',
          isUsed: usedChecklistIds.has(doc.documentId),
          isUserChecklist: false,
          version: doc.version || '',
          version_date: doc.versionDate || '',
          approved_by: doc.approvedBy || '',
          document_id: doc.documentId || '',
          document_name: doc.documentName || '',
          Header_title: doc.headerTitle || '',
          header_inputs: doc.header_inputs || {},
          footer_inputs: doc.footer_inputs || {},
          checklist: doc.checklist || {},
        };
      });

      console.log('Processed global checklists:', globalChecklists.length);
      console.log('First processed checklist:', globalChecklists[0]);

      // Map custom checklists from MongoDB
      const customChecklists = customChecklistsArr.map((doc) => {
        console.log('🔍 Processing custom checklist from backend:', doc);
        
        // Ensure items is an array and handle the structure properly
        let items = [];
        if (Array.isArray(doc.items)) {
          items = doc.items.map((item, index) => ({
            text: item.text || `Section ${index + 1}`,
            subItems: Array.isArray(item.subItems) ? item.subItems.map((subItem, subIndex) => ({
              itemName: subItem.itemName || 'Unnamed Item',
              inputType: subItem.inputType || 'Ok/Reject/N/A',
              description: subItem.description || '',
              instruction: subItem.instruction || '',
              sortOrder: subItem.sortOrder || subIndex + 1,
              isInspectionRequired: subItem.isInspectionRequired || false,
            })) : [],
          }));
        } else if (doc.items && typeof doc.items === 'object') {
          // Handle case where items might be an object instead of array
          items = Object.entries(doc.items).map(([key, value], index) => ({
            text: key || `Section ${index + 1}`,
            subItems: Array.isArray(value) ? value.map((subItem, subIndex) => ({
              itemName: subItem.itemName || subItem || 'Unnamed Item',
              inputType: subItem.inputType || 'Ok/Reject/N/A',
              description: subItem.description || '',
              instruction: subItem.instruction || '',
              sortOrder: subItem.sortOrder || subIndex + 1,
              isInspectionRequired: subItem.isInspectionRequired || false,
            })) : [],
          }));
        }
        
        const processedChecklist = {
          id: doc._id || doc.documentId,
          title: doc.title || 'Untitled Checklist',
          description: doc.description || '',
          items,
          status: 'Available',
          isUsed: usedChecklistIds.has(doc._id || doc.documentId),
          isUserChecklist: true, // Mark as custom checklist
          version: 'Custom',
          version_date: doc.createdAt || '',
          approved_by: 'User Created',
          document_id: doc._id || doc.documentId,
          document_name: doc.title || 'Untitled Checklist',
          Header_title: doc.title || 'Untitled Checklist',
          header_inputs: {},
          footer_inputs: {},
          checklist: doc.items || {},
          createdBy: doc.createdBy,
          createdAt: doc.createdAt,
          updatedAt: doc.updatedAt,
        };
        
        console.log('🔍 Processed custom checklist:', processedChecklist);
        return processedChecklist;
      });

      console.log('Processed custom checklists:', customChecklists.length);
      console.log('First processed custom checklist:', customChecklists[0]);
      console.log('All processed custom checklists:', customChecklists);

      // Merge global and custom checklists
      const allChecklists = [...globalChecklists, ...customChecklists];
      console.log('Total checklists (global + custom):', allChecklists.length);
      console.log('Final allChecklists array:', allChecklists);

      await AsyncStorage.setItem('checklists', JSON.stringify(allChecklists));
      setChecklists(allChecklists);
    } catch (error) {
      console.error('❌ Error loading checklists:', error);
      // Fallback to OfflineDataHelper
      try {
        console.log('🔄 Final fallback: Loading checklists from offline data...');
        const offlineChecklists = await OfflineDataHelper.getChecklistsData();
        setChecklists(offlineChecklists);
        Toast.show({ 
          type: 'info', 
          text1: 'Offline Data', 
          text2: `Loaded ${offlineChecklists.length} checklists from offline files`, 
          position: 'top' 
        });
      } catch (offlineError) {
        console.error('❌ Final fallback failed:', offlineError);
        setChecklists([]);
        Toast.show({ 
          type: 'error', 
          text1: 'Error', 
          text2: 'No checklist data available', 
          position: 'top' 
        });
      }
    } finally {
      setSyncModalVisible(false);
      setChecklistSearchQuery('');
      setIsChecklistSearchActive(false);
    }
  };

  const getFilteredEquipment = () => {
    let allEquipment = firebaseEquipment;

    if (selectedTab !== 'ALL') {
      allEquipment = allEquipment.filter((item) => {
        switch (selectedTab) {
          case 'VEHICLE': return ['Light Vehicle', 'Heavy Vehicle'].includes(item.equipmentCategory);
          case 'MACHINERY': return ['Machinery', 'Workshop Machinery'].includes(item.equipmentCategory);
          case 'CONTAINER': return item.equipmentCategory === 'Container';
          case 'RIG': return item.equipmentCategory === 'Rig';
          default: return true;
        }
      });
    }

    if (!equipmentSearchQuery.trim()) return allEquipment;
    return allEquipment.filter((item) =>
      [item.equipmentName, item.mainCategory, item.model].some((field) =>
        field?.toLowerCase().includes(equipmentSearchQuery.toLowerCase())
      )
    );
  };

  const handleSelectPress = async () => {
    setEquipmentModalVisible(true); // Open modal immediately
    setEquipmentLoading(true); // Show loading indicator
    
    // Load equipment IMMEDIATELY when modal opens
    try {
      await loadEquipmentData(false);
    } catch (error) {
      console.error('Error loading equipment:', error);
    }
  };

  const handleChecklistSelectPress = async () => {
    try {
      console.log('Opening checklist modal...');
      await loadChecklists();
      console.log('Checklists loaded, count:', checklists.length);
      setSelectedTab('ALL'); // Always show ALL by default
      setChecklistSearchQuery(''); // Clear search
      if (isMounted.current) {
        console.log('Setting modal visible to true');
        setChecklistModalVisible(true);
      }
    } catch (error) {
      console.error('Error opening checklist modal:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load checklist data. Please try again.', position: 'top' });
      if (isMounted.current) setChecklistModalVisible(true);
    }
  };

  const handleBackPress = () => {
    Alert.alert('Exit Step', 'Do you want to exit this step? Your data will be saved.', [
      { text: 'Cancel', style: 'cancel' },
      { text: 'OK', onPress: () => navigation.goBack(), style: 'default' },
    ], { cancelable: false });
  };

  const requestLocationPermission = async () => {
    try {
      if (Platform.OS === 'android') {
        const granted = await PermissionsAndroid.request(PermissionsAndroid.PERMISSIONS.ACCESS_FINE_LOCATION);
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } else {
        return new Promise((resolve) =>
          Geolocation.requestAuthorization(resolve, () => resolve(false))
        );
      }
    } catch (error) {
      console.error('Error requesting location permission:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to request location permission.', position: 'top' });
      return false;
    }
  };

  const fetchAddressFromCoordinates = async (latitude, longitude) => {
    const apiKey = 'AIzaSyDDEh8W8buXtPi7uhhwJx4e5iK7Gb-vbe8';
    const url = `https://maps.googleapis.com/maps/api/geocode/json?latlng=${latitude},${longitude}&key=${apiKey}`;

    try {
      const response = await fetch(url);
      const data = await response.json();
      if (data.status === 'OK' && data.results.length > 0) return data.results[0].formatted_address;
      throw new Error(data.status === 'ZERO_RESULTS' ? 'No address found' : 'Geocoding API error');
    } catch (error) {
      console.error('Error fetching address:', error.message);
      Toast.show({ type: 'error', text1: 'Address Fetch Failed', text2: error.message || 'Using coordinates only.', position: 'top' });
      return `Latitude: ${latitude}, Longitude: ${longitude} (Address unavailable)`;
    }
  };

  const handleGetLocation = async () => {
    let timeoutId;
    try {
      const hasPermission = await requestLocationPermission();
      if (!hasPermission) {
        Toast.show({ type: 'error', text1: 'Location Error', text2: 'Location permission denied.', position: 'top' });
        return;
      }
      setSyncModalVisible(true);
      timeoutId = setTimeout(() => {
        if (isMounted.current && syncModalVisible) {
          setSyncModalVisible(false);
          Toast.show({ type: 'error', text1: 'Error', text2: 'Location fetching timed out.', position: 'top' });
        }
      }, 15000);

      const position = await new Promise((resolve, reject) =>
        Geolocation.getCurrentPosition(resolve, reject, { enableHighAccuracy: true, timeout: 15000, maximumAge: 10000 })
      );
      const { latitude, longitude } = position.coords;
      const coords = `${latitude}, ${longitude}`;
      setCoordinates(coords);

      const address = await fetchAddressFromCoordinates(latitude, longitude);
      setGpsAddress(address);

      const savedInspection = await AsyncStorage.getItem('inspectionStep1');
      let inspectionData = savedInspection ? JSON.parse(savedInspection) : {};
      inspectionData.coordinates = coords;
      inspectionData.gpsAddress = address;

      if (selectedEquipment) {
        const timestamp = new Date().toISOString();
        const requestRef = inspectionData.requestNumber || requestNumber || selectedEquipment.lastInspectionRequestNumber || null;
        const updatedEquipmentDetails = {
          ...selectedEquipment,
          lastKnownCoordinates: coords,
          gpsCoordinates: coords,
          lastKnownLocation: address,
          gpsAddress: address,
          locationUpdatedAt: timestamp,
          lastInspectionAt: timestamp,
          lastInspectionCountry: country || selectedEquipment.lastInspectionCountry || null,
          lastInspectionProject: project || selectedEquipment.lastInspectionProject || null,
          lastInspectionRequestNumber: requestRef,
          updatedAt: timestamp,
        };
        if (country) {
          updatedEquipmentDetails.country = country;
        }
        if (project) {
          updatedEquipmentDetails.project = project;
        }
        setSelectedEquipment(updatedEquipmentDetails);
        inspectionData.selectedEquipment = updatedEquipmentDetails;
        inspectionData.lastInspectionAt = timestamp;
        if (country) inspectionData.lastInspectionCountry = country;
        if (project) inspectionData.lastInspectionProject = project;
        if (requestRef) inspectionData.lastInspectionRequestNumber = requestRef;

        // Persist GPS update to backend when online
        if (isOnline && updatedEquipmentDetails.id) {
          try {
            await EquipmentAPI.updateEquipment(updatedEquipmentDetails.id, {
              lastKnownCoordinates: coords,
              gpsCoordinates: coords,
              lastKnownLocation: address,
              gpsAddress: address,
              locationUpdatedAt: timestamp,
              lastInspectionAt: timestamp,
              lastInspectionCountry: country || updatedEquipmentDetails.lastInspectionCountry || null,
              lastInspectionProject: project || updatedEquipmentDetails.lastInspectionProject || null,
              lastInspectionRequestNumber: requestRef,
              updatedAt: timestamp,
            });
            console.log('✅ Equipment GPS details updated in MongoDB');
          } catch (remoteUpdateError) {
            console.warn('⚠️ Failed to update equipment GPS remotely:', remoteUpdateError?.message);
          }
        }

        try {
          const storedEquipment = await AsyncStorage.getItem('equipment');
          let equipmentList = storedEquipment ? JSON.parse(storedEquipment) : [];
          const index = equipmentList.findIndex((item) => item.id === updatedEquipmentDetails.id);
          if (index !== -1) {
            equipmentList[index] = {
              ...equipmentList[index],
              lastKnownCoordinates: coords,
              gpsCoordinates: coords,
              lastKnownLocation: address,
              gpsAddress: address,
              locationUpdatedAt: timestamp,
              lastInspectionAt: timestamp,
              lastInspectionCountry: country || equipmentList[index].lastInspectionCountry || null,
              lastInspectionProject: project || equipmentList[index].lastInspectionProject || null,
              lastInspectionRequestNumber: requestRef || equipmentList[index].lastInspectionRequestNumber || null,
              country: country || equipmentList[index].country,
              project: project || equipmentList[index].project,
              updatedAt: timestamp,
              pendingSync: !isOnline || equipmentList[index]?.pendingSync,
            };
            await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
          }
        } catch (storageError) {
          console.warn('Unable to persist equipment location locally:', storageError?.message);
        }
      }

      await saveInspectionData(inspectionData);

      Toast.show({ type: 'success', text1: 'Location Fetched', text2: `Coordinates: ${coords}\nAddress: ${address}`, position: 'top', visibilityTime: 4000 });
    } catch (error) {
      console.error('Error in handleGetLocation:', error);
      let errorMsg = 'An unexpected error occurred.';
      if (error && error.message) {
        errorMsg = error.message;
      } else if (typeof error === 'string') {
        errorMsg = error;
      }
      Toast.show({ type: 'error', text1: 'Location Error', text2: errorMsg, position: 'top' });
    } finally {
      if (timeoutId) clearTimeout(timeoutId);
      if (isMounted.current) setSyncModalVisible(false);
    }
  };

  const validateChecklistItems = (checklistId) => {
    console.log('Validating checklist:', checklistId);
    console.log('Available checklists:', checklists.length);
    console.log('Selected checklist:', checklists.find(c => c.id === checklistId));
    
    if (!checklistId) {
      Toast.show({ type: 'error', text1: 'Checklist Required', text2: 'Please select a checklist.', position: 'top' });
      return false;
    }

    const checklist = checklists.find((c) => c.id === checklistId);
    console.log('Found checklist:', checklist);
    
    if (!checklist) {
      Toast.show({ type: 'error', text1: 'Checklist Not Found', text2: 'The selected checklist could not be found. Please try selecting it again.', position: 'top' });
      return false;
    }

    // For user-created checklists
    if (checklist.isUserChecklist) {
      if (!checklist.items || !Array.isArray(checklist.items) || checklist.items.length === 0) {
        Toast.show({ type: 'error', text1: 'Invalid Checklist', text2: 'The selected checklist is empty or invalid.', position: 'top' });
        return false;
      }
      
      return checklist.items.every(
        (item) =>
          item &&
          typeof item.text === 'string' &&
          Array.isArray(item.subItems) &&
          item.subItems.length > 0 &&
          item.subItems.every(
            (subItem) =>
              subItem &&
              typeof subItem.itemName === 'string' &&
              typeof subItem.inputType === 'string'
          )
      );
    }

    // For global checklists
    if (!checklist.items || !Array.isArray(checklist.items) || checklist.items.length === 0) {
      Toast.show({ type: 'error', text1: 'Invalid Checklist', text2: 'The selected checklist is empty or invalid.', position: 'top' });
      return false;
    }

    return checklist.items.every(
      (item) =>
        item &&
        typeof item.text === 'string' &&
        Array.isArray(item.subItems) &&
        item.subItems.length > 0 &&
        item.subItems.every(
          (subItem) =>
            subItem &&
            typeof subItem.itemName === 'string' &&
            typeof subItem.inputType === 'string'
        )
    );
  };

  // In handleNextPress, after saving the inspection, update the global lastRequestNumber
  const handleNextPress = async () => {
    setShowErrors(true);
    
    // Validate all required fields
    const hasCountry = !!country;
    const hasProject = !!project;
    const hasInspector = !!inspector.trim();
    const hasEquipment = !!selectedEquipment;
    const hasChecklist = !!selectedChecklistId;
    
    setCountryError(!hasCountry);
    setProjectError(!hasProject);
    setInspectorError(!hasInspector);
    setEquipmentError(!hasEquipment);
    setChecklistError(!hasChecklist);

    // Check if all basic fields are filled
    if (!hasCountry || !hasProject || !hasInspector || !hasEquipment || !hasChecklist) {
      const missingFields = [];
      if (!hasCountry) missingFields.push('Country');
      if (!hasProject) missingFields.push('Project');
      if (!hasInspector) missingFields.push('Inspector');
      if (!hasEquipment) missingFields.push('Equipment');
      if (!hasChecklist) missingFields.push('Checklist');
      
      Toast.show({ 
        type: 'error', 
        text1: 'Required Fields Missing', 
        text2: `Please fill: ${missingFields.join(', ')}`, 
        position: 'top' 
      });
      return;
    }

    // Validate meter reading
    if (
      selectedEquipment &&
      meterReading &&
      !Number.isNaN(parseFloat(meterReading)) &&
      parseFloat(meterReading) < parseFloat(selectedEquipment.meterReading)
    ) {
      Toast.show({
        type: 'error',
        text1: 'Invalid Meter Reading',
        text2: `New meter reading (${meterReading}) must be greater than ${selectedEquipment.meterReading || '0'}.`,
        position: 'top',
      });
      return;
    }

    // Validate checklist structure
    console.log('Validating checklist:', selectedChecklistId);
    console.log('Available checklists:', checklists.length);
    console.log('Selected checklist:', checklists.find(c => c.id === selectedChecklistId));
    
    if (!validateChecklistItems(selectedChecklistId)) {
      setChecklistError(true);
      return;
    }

    setSyncModalVisible(true);
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        Alert.alert('Authentication Error', 'User not authenticated. Please log in.', [
          { text: 'OK', onPress: () => navigation.replace('AdminLoginScreen') }
        ]);
        return;
      }

      const inspectionData = {
        id: Date.now().toString(),
        requestNumber,
        country,
        project,
        inspector,
        selectedEquipment,
        meterReading,
        currentMeterReading,
        selectedChecklistId,
        checklistTitle,
        coordinates,
        gpsAddress,
        createdAt: new Date().toISOString(),
      };

      // Update equipment meter reading
      if (selectedEquipment && meterReading && parseFloat(meterReading) !== parseFloat(selectedEquipment.meterReading)) {
        const updatedEquipment = { 
          ...selectedEquipment, 
          meterReading: parseFloat(meterReading), 
          updatedAt: new Date().toISOString() 
        };
        
        if (isOnline) {
          // Try to update MongoDB, but don't block if it fails
          try {
            console.log('🌐 Online: Updating equipment meter reading in MongoDB...');
            await EquipmentAPI.updateEquipment(updatedEquipment.id, updatedEquipment);
            console.log('✅ Equipment meter reading updated in MongoDB');
          } catch (mongoError) {
            console.error('⚠️ MongoDB equipment update failed:', mongoError.message);
            console.log('📱 Continuing anyway - equipment will sync later');
            // Don't throw error - just continue with local update
          }
        } else {
          // Offline mode - save locally for later sync
          console.log('📴 Offline: Saving equipment meter reading locally...');
        }
        
        // Always update local storage regardless of online/offline
        try {
          const storedEquipment = await AsyncStorage.getItem('equipment');
          let equipmentList = storedEquipment ? JSON.parse(storedEquipment) : [];
          const index = equipmentList.findIndex((item) => item.id === updatedEquipment.id);
          
          if (index !== -1) {
            equipmentList[index] = { ...updatedEquipment, pendingSync: !isOnline };
            console.log('📝 Updated existing equipment in local storage');
          } else {
            equipmentList.push({ ...updatedEquipment, pendingSync: !isOnline });
            console.log('📝 Added new equipment to local storage');
          }
          
          await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
          console.log('✅ Equipment saved to local storage successfully');
        } catch (storageError) {
          console.error('⚠️ Error saving to local storage:', storageError.message);
          // Continue anyway - not critical
        }
        
        // Update state
        setFirebaseEquipment((prev) => prev.map((item) => (item.id === updatedEquipment.id ? updatedEquipment : item)));
        setSelectedEquipment(updatedEquipment);
        inspectionData.selectedEquipment = updatedEquipment;
        
        console.log('✅ Equipment meter reading updated:', parseFloat(meterReading));
      }

      await AsyncStorage.setItem('selectedChecklistIds', JSON.stringify([selectedChecklistId]));
      await saveInspectionData(inspectionData);

      setShowErrors(false);
      setCountryError(false);
      setProjectError(false);
      setInspectorError(false);
      setEquipmentError(false);
      setChecklistError(false);
      setActiveStep(2);
    } catch (error) {
      console.error('Error saving inspection data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save inspection data locally.', position: 'top' });
    } finally {
      setSyncModalVisible(false);
    }
  };

  const handleChecklistSearch = (query) => setChecklistSearchQuery(query);
  const handleEquipmentSearch = (query) => setEquipmentSearchQuery(query);

  const filteredChecklists = Array.isArray(checklists)
    ? (() => {
        let filtered = checklists;
        
        console.log('🔍 Filtering checklists - selectedTab:', selectedTab, 'total checklists:', checklists.length);
        console.log('🔍 All checklists:', checklists.map(c => ({ id: c.id, title: c.title, isUserChecklist: c.isUserChecklist })));
        
        // Filter by tab
        if (selectedTab === 'CUSTOM') {
          filtered = checklists.filter(checklist => checklist.isUserChecklist === true);
          console.log('🔍 Custom checklists after filtering:', filtered.length, filtered);
        } else if (selectedTab === 'Assign') {
          filtered = checklists.filter(checklist => checklist.isUsed === true);
          console.log('🔍 Assigned checklists after filtering:', filtered.length, filtered);
        }
        
        // Filter by search query
        if (checklistSearchQuery.trim()) {
          filtered = filtered.filter((checklist) =>
            (typeof checklist.title === 'string' ? checklist.title : '').toLowerCase().includes(checklistSearchQuery.toLowerCase())
          );
          console.log('🔍 After search filter:', filtered.length, filtered);
        }
        
        console.log('🔍 Final filtered checklists:', filtered.length, filtered);
        return filtered;
      })()
    : [];

  const toggleChecklistSelection = async (checklistId, title) => {
    console.log('toggleChecklistSelection called with:', checklistId, title);
    const selectedChecklist = checklists.find((c) => c.id === checklistId);
    console.log('Selected checklist data:', selectedChecklist);
    if (selectedChecklist) {
      // Save the full checklist object (with metadata) to AsyncStorage
      try {
        console.log('Saving checklist to AsyncStorage:', selectedChecklist);
        await AsyncStorage.setItem(`checklist_${checklistId}`, JSON.stringify(selectedChecklist));
        console.log('Checklist saved successfully');
        // Save footer_inputs separately for quick offline access in Step 3
        try {
          await AsyncStorage.setItem('selectedChecklistFooterInputs', JSON.stringify(selectedChecklist.footer_inputs || {}));
        } catch (e2) {
          console.warn('Unable to save selectedChecklistFooterInputs:', e2?.message);
        }
      } catch (e) {
        console.error('Error saving checklist metadata to AsyncStorage:', e);
      }
    }
    setSelectedChecklistId((prev) => {
      const newId = prev === checklistId ? null : checklistId;
      if (newId) {
        setChecklistTitle(title || 'Untitled Checklist');
        console.log('Selected checklistId:', newId, 'Title:', title);
      } else {
        setChecklistTitle('');
        console.log('Checklist deselected');
      }
      AsyncStorage.setItem('selectedChecklistIds', JSON.stringify(newId ? [newId] : []))
        .catch((error) => console.error('Error saving selected checklist:', error));
      return newId;
    });
    setChecklistModalVisible(false);
  };

  const selectEquipment = async (equipment) => {
    setSelectedEquipment(equipment);
    setMeterReading(equipment.meterReading?.toString() || '');
    setEquipmentModalVisible(false);
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available': return '#2ecc71';
      case 'Maintenance Required': return '#f1c40f';
      case 'In Maintenance': return '#e67e22';
      case 'Breakdown': return '#e74c3c';
      case 'Deploy': return '#3498db';
      default: return '#666';
    }
  };

  const renderChecklistItem = ({ item }) => {
    console.log('Rendering checklist item:', item?.id, item?.title);
    const isCustom = item?.isUserChecklist === true;
    const isSelected = selectedChecklistId === item?.id;
    
    return (
      <TouchableOpacity style={[styles.checklistItem, isSelected && styles.selectedChecklistItem]} onPress={() => toggleChecklistSelection(item?.id, item?.title)}>
        <View style={styles.checklistItemContent}>
          <MaterialIcons name={isSelected ? 'check-box' : 'check-box-outline-blank'} size={24} color={isSelected ? '#0078D4' : '#666'} />
          <View style={styles.checklistItemTextContainer}>
            <View style={styles.checklistTitleRow}>
              <Text style={styles.checklistItemText}>{item?.title || 'Untitled Checklist'}</Text>
              {isCustom && (
                <View style={styles.customBadge}>
                  <Text style={styles.customBadgeText}>CUSTOM</Text>
                </View>
              )}
            </View>
            <View style={styles.checklistDetailsRow}>
              {item?.document_id && (
                <Text style={styles.checklistItemSubText}>ID: {item.document_id}</Text>
              )}
              {isCustom && (
                <Text style={styles.checklistItemSubText}>• Created by User</Text>
              )}
              {item?.version && (
                <Text style={styles.checklistItemSubText}>• Version: {item.version}</Text>
              )}
            </View>
            {item?.description && (
              <Text style={styles.checklistDescription} numberOfLines={2}>
                {item.description}
              </Text>
            )}
            {item?.items && item.items.length > 0 && (
              <Text style={styles.checklistItemsCount}>
                {item.items.length} section{item.items.length !== 1 ? 's' : ''} • {item.items.reduce((total, section) => total + (section.subItems?.length || 0), 0)} inspection items
              </Text>
            )}
          </View>
        </View>
      </TouchableOpacity>
    );
  };

  const renderEquipmentItem = ({ item }) => (
    <View style={styles.equipmentRow}>
      <TouchableOpacity style={styles.equipmentInfo} onPress={() => selectEquipment(item)}>
        <View style={styles.equipmentDetailItemheader}>
          <Text style={styles.detailLabelheader}>{item.equipmentName || 'Tyre - Tyre'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Category</Text>
          <Text style={styles.detailValue}>{item.mainCategory || 'Sample Category'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Model</Text>
          <Text style={styles.detailValue}>{item.model || '326F'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Meter Reading</Text>
          <Text style={styles.detailValue}>{item.meterReading || '0'} KM</Text>
        </View>
        <View style={styles.equipmentDetailItemNoBorder}>
          <Text style={[styles.detailLabelstatus, { color: getStatusColor(item.status || 'Available') }]}>{item.status || 'Available'}</Text>
        </View>
      </TouchableOpacity>
      <TouchableOpacity style={styles.selectEquipmentButton} onPress={() => selectEquipment(item)}>
        <Text style={styles.selectEquipmentText}>{selectedEquipment && selectedEquipment.id === item.id ? 'Selected' : 'Select'}</Text>
      </TouchableOpacity>
    </View>
  );

  return (
    <>
      <KeyboardAvoidingView style={styles.container} behavior={Platform.OS === 'ios' ? 'padding' : 'height'} keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}>
        <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} keyboardShouldPersistTaps="handled">
          {checklists.length === 0 && (
            <View style={{flex: 1, justifyContent: 'center', alignItems: 'center', marginTop: 0}}>
              {/* <Text style={{color: '#888', fontSize: 18}}>No checklists available.</Text> */}
            </View>
          )}
          <View style={styles.headingContainer}><Text style={styles.headingText}>New Inspection</Text></View>
          <View style={styles.statsContainer}>
            {/* <View style={styles.statsCard}>
              <MaterialIcons name="assignment" size={24} color="#015185" />
              <View style={styles.statsTextContainer}>
                <Text style={styles.statsLabel}>Total Reports</Text>
                <Text style={styles.statsValue}>{totalReportsCount}</Text>
              </View>
            </View> */}
          </View>
          <View style={styles.requestRow}>
            <Text style={styles.requestLabel}>Request #</Text>
            <View style={styles.requestValueContainer}>
              <Text style={styles.requestValue}>{requestNumber}</Text>
              {country && project && (
                <TouchableOpacity 
                  style={styles.refreshButton} 
                  onPress={updateRequestNumber}
                >
                  <MaterialIcons name="refresh" size={16} color="#0078D4" />
                </TouchableOpacity>
              )}
            </View>
          </View>
          <View style={inputStyles.container}>
            <Text style={inputStyles.label}>Country<Text style={inputStyles.requiredError}> *</Text></Text>
            <Dropdown
              style={[inputStyles.input, countryError && showErrors ? inputStyles.inputError : null]}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.dropdownText}
              data={countryItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Country"
              value={country}
              onChange={item => setCountry(item.value)}
            />
            {countryError && showErrors && <Text style={inputStyles.errorText}>Country is required</Text>}
          </View>
          <View style={inputStyles.container}>
            <Text style={inputStyles.label}>Project<Text style={inputStyles.requiredError}> *</Text></Text>
            <Dropdown
              style={[inputStyles.input, !country && styles.disabledDropdown, projectError && showErrors ? inputStyles.inputError : null]}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.dropdownText}
              data={projectItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Project"
              value={project}
              onChange={item => setProject(item.value)}
              disable={!country}
            />
            {projectError && showErrors && <Text style={inputStyles.errorText}>Project is required</Text>}
          </View>
          <CustomInputField label="Inspector" value={inspector} onChangeText={setInspector} placeholder="Enter Inspector" isRequired showError={inspectorError && showErrors} errorMessage="Inspector is required" />
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Equipments</Text>
            <TouchableOpacity onPress={handleSelectPress} style={styles.selectButton}>
              <Text style={styles.selectText}>{selectedEquipment ? 'Change equipment' : 'Select'}</Text>
              <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
            </TouchableOpacity>
          </View>
          {equipmentError && showErrors && <Text style={[inputStyles.errorText, { marginHorizontal: 15 }]}>Equipment is required</Text>}
          {selectedEquipment && (
            <View style={styles.equipmentDetailsBox}>
              <View style={styles.vehicleEquipmentRow}><Text style={styles.vehicleEquipmentText}>Vehicle/Model: {selectedEquipment.vehicleNumber || selectedEquipment.model || 'N/A'}, Category: {selectedEquipment.mainCategory || 'N/A'}, Equipment Name: {selectedEquipment.equipmentName || 'N/A'}</Text></View>
              <View style={styles.meterReadingContainer}>
                <View style={styles.labelRow}><Text style={styles.meterReadingLabel}>Meter Reading</Text></View>
                <View style={styles.inputRow}>
                  <TextInput
                    style={styles.meterReadingInput}
                    value={meterReading}
                    onChangeText={setMeterReading}
                    onBlur={() => {
                      if (meterReading) {
                        const currentReading = parseFloat(selectedEquipment.meterReading) || 0;
                        const newReading = parseFloat(meterReading);
                        if (!isNaN(newReading) && newReading < currentReading) {
                          Toast.show({ type: 'error', text1: 'Invalid Reading', text2: `Meter reading must be greater than ${currentReading}`, position: 'top' });
                        }
                      }
                    }}
                    placeholder="Enter meter reading"
                    placeholderTextColor="#888"
                    keyboardType="numeric"
                  />
                  <Text style={styles.meterUnit}>KM</Text>
                </View>
                <View style={styles.currentMeterRow}>
                  <Text style={styles.currentMeterLabel}>Current Meter Reading: </Text>
                  <Text style={styles.currentMeterValue}>{selectedEquipment.meterReading || '0'} KM</Text>
                </View>
                {(gpsAddress || selectedEquipment.lastKnownLocation) && (
                  <View style={styles.locationRow}>
                    <Text style={styles.locationLabel}>Last Location:</Text>
                    <Text style={styles.locationValue}>{gpsAddress || selectedEquipment.lastKnownLocation}</Text>
                  </View>
                )}
              </View>
            </View>
          )}
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Checklist</Text>
            <TouchableOpacity onPress={handleChecklistSelectPress} style={styles.selectButton}>
              <Text style={styles.selectText}>{checklistTitle ? 'Change checklist' : 'Select'}</Text>
              <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
            </TouchableOpacity>
          </View>
          <View style={styles.selectedChecklistsBox}>
            {checklistTitle ? <Text style={styles.selectedChecklistText}>{checklistTitle}</Text> : <Text style={styles.noChecklistText}>No checklist selected</Text>}
          </View>
          {checklistError && showErrors && <Text style={[inputStyles.errorText, { marginHorizontal: 15 }]}>Checklist is required</Text>}
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>GPS Location</Text>
            <TouchableOpacity onPress={handleGetLocation} style={styles.selectButton}>
              <MaterialIcons name="location-on" size={16} color="#0078D4" />
              <Text style={styles.selectText}>Get Location</Text>
            </TouchableOpacity>
          </View>
          <View style={styles.gpsDetailsBox}>
            {coordinates && <View style={styles.gpsInfoRow}><Text style={styles.gpsLabel}>Coordinates: </Text><Text style={styles.gpsValue}>{coordinates}</Text></View>}
            {gpsAddress && <View style={styles.gpsInfoRow}><Text style={styles.gpsLabel}>GPS Address: </Text><Text style={styles.gpsValue}>{gpsAddress}</Text></View>}
            {!coordinates && !gpsAddress && <Text style={styles.noGpsText}>Click "Get Location" to fetch your current location</Text>}
          </View>
        </ScrollView>
        <View style={styles.buttonContainer}>
          <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
            <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
            <Text style={styles.cancelButtonText}>Cancel</Text>
          </TouchableOpacity>
          <TouchableOpacity style={styles.nextButton} onPress={handleNextPress}>
            <Text style={styles.nextButtonText}>Next</Text>
            <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
          </TouchableOpacity>
        </View>
        <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message="Processing" />
        <Toast />
      </KeyboardAvoidingView>
      {/* Render modals outside of KeyboardAvoidingView for iOS compatibility */}
      <Modal visible={checklistModalVisible} transparent animationType="fade" onRequestClose={() => setChecklistModalVisible(false)}>
        <View pointerEvents="box-none" style={styles.modalOverlay}>
          <View style={styles.checklistModalContent}>
            <View style={styles.checklistModalHeader}>
              <Text style={styles.checklistModalTitle}>Select Checklist</Text>
              <View style={{ flexDirection: 'row', alignItems: 'center' }}>
                <TouchableOpacity onPress={loadChecklists} style={[styles.modalCloseIcon, { marginRight: 10 }]}>
                  <MaterialIcons name="refresh" size={20} color="#0078D4" />
                </TouchableOpacity>
                <TouchableOpacity onPress={() => setChecklistModalVisible(false)} style={styles.modalCloseIcon}>
                  <MaterialIcons name="close" size={24} color="#666" />
                </TouchableOpacity>
              </View>
            </View>
            <View style={styles.checklistModalTabs}>
              <TouchableOpacity style={[styles.tabButton, selectedTab === 'ALL' && styles.activeTabButton]} onPress={() => setSelectedTab('ALL')}>
                <Text style={[styles.tabText, selectedTab === 'ALL' && styles.activeTabText]}>ALL</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.tabButton, selectedTab === 'CUSTOM' && styles.activeTabButton]} onPress={() => setSelectedTab('CUSTOM')}>
                <Text style={[styles.tabText, selectedTab === 'CUSTOM' && styles.activeTabText]}>
                  CUSTOM ({checklists.filter(c => c.isUserChecklist === true).length})
                </Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.tabButton, selectedTab === 'Assign' && styles.activeTabButton]} onPress={() => setSelectedTab('Assign')}>
                <Text style={[styles.tabText, selectedTab === 'Assign' && styles.activeTabText]}>Assign</Text>
              </TouchableOpacity>
            </View>
            <View style={styles.checklistSearchContainer}>
              {isChecklistSearchActive ? (
                <TextInput
                  style={styles.checklistSearchInput}
                  placeholder="Search checklists..."
                  placeholderTextColor="#888"
                  value={checklistSearchQuery}
                  onChangeText={handleChecklistSearch}
                  autoFocus
                />
              ) : (
                <Text style={styles.checklistSelectText}>Search Checklists</Text>
              )}
              <TouchableOpacity onPress={() => { setIsChecklistSearchActive(!isChecklistSearchActive); if (isChecklistSearchActive) setChecklistSearchQuery(''); }}>
                <MaterialIcons name={isChecklistSearchActive ? 'close' : 'search'} size={24} color="#666" />
              </TouchableOpacity>
            </View>
            <View style={styles.divider} />
            <FlatList
              data={filteredChecklists}
              keyExtractor={(item, idx) => (item?.id ? item.id.toString() : `checklist-${idx}`)}
              renderItem={renderChecklistItem}
              ListEmptyComponent={
                <View style={{ padding: 20, alignItems: 'center' }}>
                  <Text style={styles.checklistEmptyText}>
                    {selectedTab === 'CUSTOM' ? 'No custom checklists found' : 
                     selectedTab === 'Assign' ? 'No assigned checklists found' : 
                     'No checklists found'}
                  </Text>
                  <Text style={{ fontSize: 12, color: '#999', marginTop: 5 }}>
                    {selectedTab === 'CUSTOM' ? 'Create a custom checklist to get started' :
                     selectedTab === 'Assign' ? 'Checklists you have used before will appear here' :
                     `Total checklists: ${checklists.length}, Filtered: ${filteredChecklists.length}`}
                  </Text>
                  {checklistSearchQuery && (
                    <Text style={{ fontSize: 12, color: '#999', marginTop: 5 }}>
                      Search query: "{checklistSearchQuery}"
                    </Text>
                  )}
                </View>
              }
              style={styles.checklistList}
            />
            <View style={styles.divider} />
            <View style={styles.checklistModalFooter}>
              <TouchableOpacity onPress={() => { setChecklistModalVisible(false); navigation.navigate('CreatechecklistHeader', { checklist: null }); }}>
                <Text style={styles.createChecklistText}>Create New Checklist</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.checklistCloseButton} onPress={() => setChecklistModalVisible(false)}>
                <Text style={styles.checklistCloseText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      <Modal visible={equipmentModalVisible} transparent animationType="fade" onRequestClose={() => setEquipmentModalVisible(false)}>
        <View pointerEvents="box-none" style={styles.modalOverlay}>
          <View style={styles.equipmentModalContent}>
            <View style={styles.equipmentModalHeader}>
              <Text style={styles.equipmentModalTitle}>Select Equipments</Text>
              <TouchableOpacity onPress={() => setEquipmentModalVisible(false)} style={styles.modalCloseIcon}>
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            {equipmentLoading ? (
              <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', minHeight: 120 }}>
                <Text style={{ color: '#015185', fontSize: 16, marginBottom: 10 }}>Loading equipment...</Text>
                <Animated.View style={{ width: 40, height: 40 }}>
                  <MaterialIcons name="autorenew" size={40} color="#015185" style={{ opacity: 0.7 }} />
                </Animated.View>
              </View>
            ) : (
              <>
                <TouchableOpacity
                  style={[styles.equipmentSearchContainer, isEquipmentSearchActive && styles.equipmentSearchContainerActive]}
                  onPress={() => { setIsEquipmentSearchActive(true); equipmentSearchInputRef.current?.focus(); }}
                >
                  {isEquipmentSearchActive ? (
                    <TextInput
                      ref={equipmentSearchInputRef}
                      style={[styles.equipmentSearchInput, isEquipmentSearchActive && styles.equipmentSearchInputActive]}
                      placeholder="Search..."
                      placeholderTextColor="#888"
                      value={equipmentSearchQuery}
                      onChangeText={handleEquipmentSearch}
                      autoFocus
                    />
                  ) : (
                    <Text style={styles.equipmentSelectText}>Search Equipment</Text>
                  )}
                  <TouchableOpacity onPress={() => { setIsEquipmentSearchActive(!isEquipmentSearchActive); if (isEquipmentSearchActive) setEquipmentSearchQuery(''); equipmentSearchInputRef.current?.blur(); }}>
                    <MaterialIcons name={isEquipmentSearchActive ? 'close' : 'search'} size={24} color="#666" />
                  </TouchableOpacity>
                </TouchableOpacity>
                <View style={styles.divider} />
                <FlatList
                  data={getFilteredEquipment()}
                  keyExtractor={(item) => item.id.toString()}
                  renderItem={renderEquipmentItem}
                  ListEmptyComponent={<Text style={styles.equipmentEmptyText}>No equipment found</Text>}
                  style={styles.equipmentList}
                />
                <View style={styles.divider} />
                <View style={styles.equipmentModalFooter}>
                  <TouchableOpacity onPress={() => { setEquipmentModalVisible(false); navigation.navigate('Add_equipment'); }}>
                    <Text style={styles.addEquipmentText}>Add your own equipment</Text>
                  </TouchableOpacity>
                  <TouchableOpacity style={styles.equipmentCloseButton} onPress={() => setEquipmentModalVisible(false)}>
                    <Text style={styles.equipmentCloseText}>Close</Text>
                  </TouchableOpacity>
                </View>
              </>
            )}
          </View>
        </View>
      </Modal>
    </>
  );
};


const inputStyles = StyleSheet.create({
  container: { marginBottom: width * 0.04 },
  label: { fontSize: width * 0.038, color: '#015185', marginBottom: 4, fontWeight: '500' },
  requiredError: { color: 'red', fontWeight: 'bold' },
  input: { borderWidth: 1, borderColor: '#015185', borderRadius: 5, paddingHorizontal: 12, paddingVertical: 14, fontSize: width * 0.04, backgroundColor: '#fff' },
  inputError: { borderColor: 'red' },
  errorText: { color: 'red', fontSize: width * 0.032, marginTop: 2, marginLeft: 2 },
});

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  mainContent: { flex: 1, paddingHorizontal: width * 0.04, paddingVertical: width * 0.025 },
  scrollContent: { paddingBottom: 130 },
  headingContainer: { justifyContent: 'center', alignItems: 'center', paddingVertical: 10 },
  headingText: { fontSize: 20, fontWeight: '500', color: '#015185' },
  statsContainer: { flexDirection: 'row', justifyContent: 'center', alignItems: 'center', marginVertical: 10 },
  statsCard: { flexDirection: 'row', alignItems: 'center', backgroundColor: '#f0f8ff', paddingVertical: 12, paddingHorizontal: 20, borderRadius: 8, borderWidth: 1, borderColor: '#015185', minWidth: 180 },
  statsTextContainer: { marginLeft: 12 },
  statsLabel: { fontSize: 12, color: '#666', marginBottom: 2 },
  statsValue: { fontSize: 24, fontWeight: '700', color: '#015185' },
  requestRow: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', height: 50, backgroundColor: '#f0f0f0', paddingHorizontal: 15, marginBottom: 20 },
  requestLabel: { fontSize: 16, color: '#015185' },
  requestValueContainer: { flexDirection: 'row', alignItems: 'center' },
  requestValue: { fontSize: 16, color: '#015185', marginRight: 8 },
  refreshButton: { padding: 4, borderRadius: 12, backgroundColor: '#f0f8ff' },
  dropdownContainer: { borderWidth: 1, borderColor: '#015185', borderRadius: 5, backgroundColor: '#fff' },
  dropdownText: { fontSize: 16, color: '#333' },
  placeholderStyle: { fontSize: 16, color: '#888' },
  disabledDropdown: { backgroundColor: '#e0e0e0' },
  selectRow: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', width: '100%', height: 50, backgroundColor: '#f0f0f0', paddingHorizontal: 15, borderRadius: 5, marginTop: 5 },
  selectLabel: { fontSize: 16, color: '#015185', fontWeight: '600' },
  selectButton: { flexDirection: 'row', alignItems: 'center' },
  selectText: { fontSize: 16, color: '#0078D4', marginLeft: 5 },
  equipmentDetailsBox: { width: '100%', backgroundColor: '#f0f0f0', paddingVertical: 10, paddingHorizontal: 15, marginBottom: 10, borderRadius: 5, alignSelf: 'center' },
  gpsDetailsBox: { width: '100%', backgroundColor: '#f0f0f0', paddingVertical: 10, paddingHorizontal: 15, marginBottom: 10, borderRadius: 5, alignSelf: 'center' },
  gpsInfoRow: { flexDirection: 'row', flexWrap: 'wrap', marginBottom: 5 },
  gpsLabel: { fontSize: 16, color: '#333', fontWeight: '600' },
  gpsValue: { fontSize: 16, color: '#333', flex: 1 },
  noGpsText: { fontSize: 14, color: '#888', textAlign: 'center' },
  vehicleEquipmentRow: { paddingVertical: 10 },
  vehicleEquipmentText: { fontSize: 16, color: '#333' },
  meterReadingContainer: { padding: 12, backgroundColor: '#F9F9F9', borderRadius: 6, marginTop: 8 },
  labelRow: { marginBottom: 8 },
  meterReadingLabel: { fontSize: 14, color: '#333', fontWeight: '500' },
  inputRow: { flexDirection: 'row', alignItems: 'center', backgroundColor: '#fff', borderWidth: 1, borderColor: '#ddd', borderRadius: 4, paddingHorizontal: 12, paddingVertical: 8 },
  meterReadingInput: { flex: 1, fontSize: 16, color: '#333', padding: 0 },
  meterUnit: { color: '#666', fontSize: 14, marginLeft: 8 },
  currentMeterRow: { flexDirection: 'row', marginTop: 8, alignItems: 'center' },
  currentMeterLabel: { fontSize: 13, color: '#666' },
  currentMeterValue: { fontSize: 13, color: '#333', fontWeight: '500' },
  locationRow: { marginTop: 8 },
  locationLabel: { fontSize: 13, color: '#666', fontWeight: '500', marginBottom: 2 },
  locationValue: { fontSize: 13, color: '#333' },
  buttonContainer: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingHorizontal: 15, paddingVertical: 15, backgroundColor: '#F2F2F2', borderTopWidth: 1, borderTopColor: '#ddd', position: 'absolute', bottom: 0, left: 0, right: 0, elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: -2 }, shadowOpacity: 0.1, shadowRadius: 2, paddingBottom: 30 },
  cancelButton: { flexDirection: 'row', alignItems: 'center', justifyContent: 'center', paddingBottom: 30, paddingHorizontal: 15 },
  cancelButtonText: { color: '#015185', fontSize: 16, fontWeight: '600' },
  nextButton: { flexDirection: 'row', alignItems: 'center', justifyContent: 'center', paddingBottom: 30, borderRadius: 5 },
  nextButtonText: { color: '#015185', fontSize: 16, fontWeight: '600' },
  modalOverlay: { flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: 'rgba(0,0,0,0.5)' },
  checklistModalContent: { width: '90%', maxHeight: '70%', backgroundColor: '#fff', borderRadius: 10, padding: 20, elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: 2 }, shadowOpacity: 0.3, shadowRadius: 4 },
  checklistModalHeader: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginBottom: 15 },
  checklistModalTitle: { fontSize: 18, fontWeight: '600', color: '#015185' },
  modalCloseIcon: { padding: 5 },
  checklistModalTabs: { flexDirection: 'row', justifyContent: 'space-between', marginBottom: 10 },
  tabButton: { flex: 1, alignItems: 'center', paddingVertical: 10, borderBottomWidth: 2, borderBottomColor: 'transparent' },
  activeTabButton: { borderBottomColor: '#0078D4' },
  tabText: { fontSize: 16, color: '#666', fontWeight: '600' },
  activeTabText: { color: '#0078D4' },
  debugInfo: { backgroundColor: '#f0f0f0', padding: 8, marginVertical: 5, borderRadius: 4 },
  debugText: { fontSize: 12, color: '#666', textAlign: 'center', marginBottom: 8 },
  testButton: { backgroundColor: '#ff9800', padding: 8, borderRadius: 4, alignItems: 'center' },
  testButtonText: { color: '#fff', fontSize: 12, fontWeight: 'bold' },
  checklistSearchContainer: { flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingVertical: 10, borderWidth: 1, borderColor: '#ccc', borderRadius: 5, paddingHorizontal: 10 },
  checklistSelectText: { fontSize: 16, color: '#333', flex: 1 },
  checklistSearchInput: { flex: 1, height: 40, fontSize: 16, color: '#333' },
  divider: { height: 1, backgroundColor: '#eee', marginVertical: 10 },
  checklistList: { maxHeight: 200 },
  checklistItem: { paddingVertical: 12, paddingHorizontal: 8, marginVertical: 2, borderRadius: 8, backgroundColor: '#f9f9f9' },
  selectedChecklistItem: { backgroundColor: '#e3f2fd', borderWidth: 1, borderColor: '#0078D4' },
  checklistItemContent: { flexDirection: 'row', alignItems: 'flex-start' },
  checklistItemTextContainer: { flex: 1, marginLeft: 10 },
  checklistTitleRow: { flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', marginBottom: 4 },
  checklistItemText: { fontSize: 16, color: '#333', fontWeight: '600', flex: 1 },
  customBadge: { backgroundColor: '#ff9800', paddingHorizontal: 8, paddingVertical: 2, borderRadius: 12, marginLeft: 8 },
  customBadgeText: { fontSize: 10, color: '#fff', fontWeight: 'bold' },
  checklistDetailsRow: { flexDirection: 'row', flexWrap: 'wrap', marginBottom: 4 },
  checklistItemSubText: { fontSize: 12, color: '#888', marginRight: 8 },
  checklistDescription: { fontSize: 14, color: '#666', marginBottom: 4, fontStyle: 'italic' },
  checklistItemsCount: { fontSize: 12, color: '#0078D4', fontWeight: '500' },
  checklistEmptyText: { fontSize: 16, color: '#888', textAlign: 'center', marginVertical: 20 },
  checklistModalFooter: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingVertical: 10 },
  createChecklistText: { fontSize: 16, color: '#0078D4', fontWeight: '600' },
  checklistCloseButton: { padding: 10 },
  checklistCloseText: { fontSize: 16, color: '#3481BC', fontWeight: '600' },
  equipmentModalContent: { width: '90%', maxHeight: '70%', backgroundColor: '#fff', borderRadius: 10, padding: 20, elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: 2 }, shadowOpacity: 0.3, shadowRadius: 4 },
  equipmentModalHeader: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingVertical: 10 },
  equipmentModalTitle: { fontSize: 18, fontWeight: '600', color: '#015185' },
  equipmentSearchContainer: { flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingVertical: 10, borderWidth: 1, borderColor: '#ccc', borderRadius: 5, paddingHorizontal: 10, marginBottom: 10 },
  equipmentSearchContainerActive: { paddingVertical: 2 },
  equipmentSelectText: { fontSize: 16, color: '#333', flex: 1 },
  equipmentSearchInput: { flex: 1, height: 40, fontSize: 16, color: '#333' },
  equipmentSearchInputActive: { flex: 1, height: 40, fontSize: 16, color: '#333' },
  equipmentList: { maxHeight: 300, paddingHorizontal: 12 },
  equipmentRow: { marginVertical: 5, backgroundColor: '#F2F2F2', borderRadius: 5, borderWidth: 1, borderColor: '#eee', paddingRight: 10 },
  equipmentInfo: { flex: 1 },
  equipmentDetailItem: { flexDirection: 'row', justifyContent: 'space-between', paddingVertical: 5, borderBottomWidth: 1, borderBottomColor: '#ddd', paddingHorizontal: '5%', paddingRight: '10%' },
  equipmentDetailItemNoBorder: { flexDirection: 'row', justifyContent: 'space-between', paddingVertical: 5, paddingLeft: '5%' },
  equipmentDetailItemheader: { flexDirection: 'row', justifyContent: 'space-between', paddingVertical: 5, backgroundColor: '#E9E9E9', paddingHorizontal: '5%' },
  detailLabel: { fontSize: 14, color: '#666', fontWeight: '400', width: '40%', marginVertical: 3 },
  detailLabelstatus: { fontSize: 14, fontWeight: '400', width: '60%', marginVertical: 3 },
  detailLabelheader: { fontSize: 18, color: '#015185', fontWeight: '600', width: '100%' },
  detailValue: { fontSize: 16, color: '#015185', width: '60%', textAlign: 'right', fontWeight: '500' },
  selectEquipmentButton: { position: 'absolute', right: 10, bottom: 10, backgroundColor: '#0078D4', paddingVertical: 5, paddingHorizontal: 10, borderRadius: 5 },
  selectEquipmentText: { color: '#fff', fontSize: 14, fontWeight: '600' },
  equipmentEmptyText: { fontSize: 16, color: '#888', textAlign: 'center', marginVertical: 20 },
  equipmentModalFooter: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingVertical: 10 },
  addEquipmentText: { fontSize: 16, color: '#0078D4', fontWeight: '600' },
  equipmentCloseButton: { padding: 10 },
  equipmentCloseText: { fontSize: 16, color: '#3481BC', fontWeight: '600' },
  selectedChecklistsBox: { width: '94%', paddingVertical: 10, paddingHorizontal: 15, marginBottom: 10, alignSelf: 'center' },
  selectedChecklistText: { fontSize: 14, color: '#333', marginBottom: 5 },
  noChecklistText: { fontSize: 14, color: '#888' },
  bottomicon: { marginTop: 2 },
  syncModalContent: { width: '80%', backgroundColor: '#fff', borderRadius: 10, padding: 20, alignItems: 'center', elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: 2 }, shadowOpacity: 0.3, shadowRadius: 4 },
  syncModalText: { fontSize: 18, fontWeight: '600', color: '#015185', marginBottom: 10 },
  syncModalTextadd: { fontSize: 14, color: '#666', textAlign: 'center' },
  syncIcondiv: { alignItems: 'center' },
  syncIcon: { width: 30, height: 30, resizeMode: 'contain' },
});
export default InsInfostep1;
