import React, { useState, useEffect, useCallback, useRef, useMemo } from 'react';
import {
  View,
  Text,
  Image,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Modal,
  Animated,
  Dimensions,
  Alert,
  BackHandler,
  Linking,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';

// MongoDB API Configuration
const MONGODB_API_BASE = 'https://api.titandrillingzm.com:6003';


// SyncModal Component with customizable message
const SyncModal = ({ visible, onClose, message }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message || 'Processing...'}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Skeleton Component for Loading State
const Skeleton = () => {
  const deviceWidth = Dimensions.get('window').width;
  const boxSize = Math.floor((deviceWidth - 5 * 20) / 4); // Match real UI
  const minBoxSize = Math.max(boxSize, 70); // Ensure minimum box size
  const fadeGray = '#f5f6fa';
  const fadeGrayDark = '#e6e6e6';

  return (
    <ScrollView style={{ flex: 1, backgroundColor: '#fff' }}>
              <View style={{ paddingHorizontal: 5, paddingVertical: 10, alignItems: 'center', width: '100%' }}>


        {/* Skeleton for each section */}
        {[
          { title: 'Master Data', items: 3 },
          { title: 'Inspection and Compliances', items: 5 },
          { title: 'Maintenance & Premium Services', items: 3 },
          { title: 'Logbook', items: 2 },
          { title: 'HR', items: 3 },
        ].map((section, index) => {
          // Calculate how many rows and how many items per row
          const itemsPerRow = 4;
          const rows = Math.ceil(section.items / itemsPerRow);
          const itemsArray = Array.from({ length: section.items });
          let renderedRows = [];
          for (let row = 0; row < rows; row++) {
            const startIdx = row * itemsPerRow;
            const endIdx = startIdx + itemsPerRow;
            const rowItems = itemsArray.slice(startIdx, endIdx);
            // For the last row, if less than 4, add empty boxes to center
            const emptyBoxes = itemsPerRow - rowItems.length;
            renderedRows.push(
            <View
                key={row}
              style={{
                  flexDirection: 'row',
                  justifyContent: 'center',
                  alignItems: 'flex-start',
                  marginBottom: 18,
                  width: '100%',
                }}
              >
                {rowItems.map((_, idx) => (
                                  <View
                    key={idx}
                    style={{
                      alignItems: 'center',
                        marginHorizontal: 16,
                      width: minBoxSize,
                    }}
                  >
                    <View
                      style={{
                          backgroundColor: fadeGray,
                        borderRadius: 12,
                        width: minBoxSize,
                        height: minBoxSize,
                        justifyContent: 'center',
                        alignItems: 'center',
                        marginBottom: 8,
                      }}
                    />
                  <View
                    style={{
                      height: 13,
                      width: '80%',
                        backgroundColor: fadeGrayDark,
                      borderRadius: 4,
                      alignSelf: 'center',
                    }}
                  />
                </View>
                ))}
            </View>
            );
          }
          return (
            <View key={section.title} style={{ width: '100%' }}>
              {/* Heading skeleton */}
                <View
                  style={{
                  height: 24,
                  width: '60%',
                  backgroundColor: fadeGrayDark,
                      borderRadius: 4,
                      alignSelf: 'center',
                  marginBottom: 13,
                    }}
                  />
              {/* Render all rows for this section */}
              {renderedRows}
              {/* Button skeleton */}
              <View style={{ alignItems: 'center', marginVertical: 15, width: '100%' }}>
              <View
                style={{
                    width: 160,
                  height: 40,
                  borderRadius: 5,
                    backgroundColor: fadeGrayDark,
                    alignSelf: 'center',
                }}
              />
            </View>
          </View>
          );
        })}
      </View>
    </ScrollView>
  );
};

// Get screen dimensions outside component to prevent hooks order issues
const { width: screenWidth, height: screenHeight } = Dimensions.get('window');

const HomeScreen = React.memo(() => {
  // Use stable width and height from outside the component
  const width = screenWidth;
  const height = screenHeight;
  
  const navigation = useNavigation();
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [modalVisible, setModalVisible] = useState(false);
  const [userId, setUserId] = useState(null);
  const [userData, setUserData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [syncModalMessage, setSyncModalMessage] = useState('Processing...');
  const [dataLoaded, setDataLoaded] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [hasShownOfflineToast, setHasShownOfflineToast] = useState(false);
  const [hasShownOnlineToast, setHasShownOnlineToast] = useState(false);
  const [modules, setModules] = useState([]);
  const [selectedCountries, setSelectedCountries] = useState([]);
  const isMounted = useRef(true);

  // Valid country options
  const countryOptions = ['Zambia', 'DRC', 'Both Countries'];
  const memoizedCountryOptions = useMemo(() => countryOptions, []);

  // Module definitions with categories, navigation routes, and icons
  const moduleDefinitions = {
    Equipments: {
      category: 'Master Data',
      label: 'Equipments',
      navigation: 'Equipment_one',
      icon: require('../../Images/adminhome/icosn/eq101.png'),
    },
    Documents: {
      category: 'Master Data',
      label: 'Documents',
      navigation: 'Documenthome',
      icon: require('../../Images/adminhome/icosn/eq103.png'),
    },
    teamEmploy: {
      category: 'Master Data',
      label: 'Team/Employee',
      navigation: 'Team_employment_Home',
      icon: require('../../Images/adminhome/icosn/eq102.png'),
    },
    inspectionReport: {
      category: 'Inspection and Compliances',
      label: 'Inspection Report',
      navigation: 'Inspection_Report_Home',
      icon: require('../../Images/adminhome/icosn/ins0101.png'),
    },
    hseInspection: {
      category: 'Inspection and Compliances',
      label: 'HSE Inspection',
      navigation: 'HSEInspectionHome',
      icon: require('../../Images/adminhome/icosn/ins0101.png'),
    },
    dvirReport: {
      category: 'Inspection and Compliances',
      label: 'DVIR Inspection',
      navigation: 'DviranddotInHome',
      icon: require('../../Images/adminhome/icosn/eq103.png'),
    },
    checklist: {
      category: 'Inspection and Compliances',
      label: 'Check List',
      navigation: 'Checklist_home',
      icon: require('../../Images/adminhome/icosn/checklist1.png'),
    },
    incidentReport: {
      category: 'Inspection and Compliances',
      label: 'Incident Report',
      navigation: 'IncodentReport',
      icon: require('../../Images/adminhome/icosn/ins0101.png'),
    },
    dailyReport: {
      category: 'Inspection and Compliances',
      label: 'Daily Inspection',
      navigation: 'DailyIns_report_home',
      icon: require('../../Images/adminhome/icosn/solar_calendar-linear.png'),
    },
    requestMaintenance: {
      category: 'Maintenance & Premium Services',
      label: 'Request Maintenance',
      navigation: 'RequestmaintenanceHome',
      icon: require('../../Images/adminhome/icosn/RM01.png'),
    },
    jobcard: {
      category: 'Maintenance & Premium Services',
      label: 'Job Card',
      navigation: 'WorkOrdertab',
      icon: require('../../Images/adminhome/icosn/jobcard1.png'),
    },
    serviceSchedule: {
      category: 'Maintenance & Premium Services',
      label: 'Service Schedule',
      navigation: 'ServiceScheduleHome',
      icon: require('../../Images/adminhome/icosn/schdule01.png'),
    },

  
    fuelLog: {
      category: 'Logbook',
      label: 'Fuel Log',
      navigation: 'Fuel_Log_List',
      icon: require('../../Images/adminhome/icosn/fuel-log.png'),
    },
    logbook: {
      category: 'Logbook',
      label: 'Logbook',
      navigation: 'LogbookList',
      icon: require('../../Images/adminhome/icosn/logbook1.png'),
    },
    hrForm: {
      category: 'HR',
      label: 'HR Form',
      navigation: 'AdminCheckAllApli',
      icon: require('../../Images/adminhome/icosn/hrform01.png'),
    },
    hrApplications: {
      category: 'HR',
      label: 'HR Applications',
      navigation: 'AdminHrDocument',
      icon: require('../../Images/adminhome/icosn/hrdocuments01.png'),
    },
    teamKpi: {
      category: 'HR',
      label: 'Team KPI',
      navigation: 'HRReportingHome',
      icon: require('../../Images/adminhome/icosn/Hrreporting01.png'),
    },
  };


  // Check network status
  const checkNetworkStatus = useCallback(async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  }, []);

  // Show toast notification
  const showToast = useCallback((type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  }, []);

  // Retry mechanism for Firebase operations
  const withRetry = useCallback(async (operation, maxAttempts = 3, delay = 1000) => {
    for (let attempt = 1; attempt <= maxAttempts; attempt++) {
      try {
        return await operation();
      } catch (error) {
        if (attempt === maxAttempts) throw error;
        console.log(`Attempt ${attempt} failed, retrying after ${delay}ms...`);
        await new Promise((resolve) => setTimeout(resolve, delay));
      }
    }
  }, []);

  // Check Firebase connection
  const checkFirebaseConnection = useCallback(
    async (uid) => {
      if (!isMounted.current) return;
      try {
        if (!isOnline) {
          if (!hasShownOfflineToast) {
            showToast('info', 'Offline Mode', 'Operating in offline mode. Please connect to the internet.');
            setHasShownOfflineToast(true);
          }
          return;
        }
        setSyncModalVisible(true);
        setSyncModalMessage('Connecting to Firebase');
        const userRef = database().ref(`Globalusers/admins/${uid}`);
        await withRetry(() => userRef.once('value'), 2, 1000);
        if (isMounted.current) {
          showToast('success', 'Connected', 'Successfully connected to Firebase');
          setHasShownOnlineToast(true);
        }
      } catch (error) {
        console.error('Firebase connection error:', error);
        if (!isMounted.current) return;
        let errorMessage = 'Failed to connect to Firebase. Please try again.';
        if (error.message.includes('timeout')) {
          errorMessage = 'Firebase connection timed out. Using offline data.';
        } else if (error.code === 'PERMISSION_DENIED') {
          errorMessage = 'Permission denied. Please contact support.';
        } else if (error.code === 'NETWORK_ERROR' || !isOnline) {
          errorMessage = 'Offline mode. Data will sync when online.';
        }
        showToast('error', 'Connection Error', errorMessage);
      } finally {
        if (isMounted.current) {
          setSyncModalVisible(false);
          setSyncModalMessage('Processing...');
        }
      }
    },
    [showToast, withRetry, isOnline, hasShownOfflineToast]
  );

  // Check for app updates
  const checkForUpdates = useCallback(async () => {
    if (!isMounted.current || !isOnline) return;
    
    try {
      setSyncModalVisible(true);
      setSyncModalMessage('Checking for updates...');
      
      // Get ACTUAL running app version from device (dynamically from build files)
      const actualAppVersion = await DeviceInfo.getVersion();
      const actualBuildNumber = await DeviceInfo.getBuildNumber();
      const actualVersionWithBuild = `${actualAppVersion} (${actualBuildNumber})`;
      
      console.log('📱 Version Check Started:');
      console.log('  - Platform:', Platform.OS);
      console.log('  - ACTUAL running app version:', actualAppVersion);
      console.log('  - ACTUAL running build number:', actualBuildNumber);
      console.log('  - ACTUAL version with build:', actualVersionWithBuild);
      console.log('  - Source: DeviceInfo.getVersion() - DYNAMIC FROM BUILD FILES');
      console.log('  - Android: versionName from build.gradle');
      console.log('  - iOS: CFBundleShortVersionString from Info.plist');
      
      // Check Realtime Database for latest version
      const versionRef = database().ref('Android_IOS_Update_Version/Android_IOS_Update_V');
      const snapshot = await withRetry(() => versionRef.once('value'), 2, 1000);
      const versionData = snapshot.val();
      
      console.log('🔥 Firebase version data:', versionData);
      
      let firebaseVersion = '1.0.0'; // Default fallback
      let needsUpdate = false; // Initialize needsUpdate variable
      
      if (versionData) {
        // Get the correct Firebase version for the current platform
        const platformKey = Platform.OS === 'ios' ? 'IOS_version' : 'Android_version';
        firebaseVersion = versionData[platformKey] || '1.0.0';
        
        console.log('🔍 Version Analysis:');
        console.log('  - Platform:', Platform.OS);
        console.log('  - Platform Key:', platformKey);
        console.log('  - ACTUAL running version (device):', actualAppVersion);
        console.log('  - Firebase version:', firebaseVersion);
        console.log('  - Available Firebase fields:', Object.keys(versionData));
        console.log('  - Firebase path: Android_IOS_Update_Version/Android_IOS_Update_V');
        console.log('  - Version source: DYNAMIC from build files');
        
        // Store the Firebase version in state for the modal
        setLatestVersionFromFirebase(firebaseVersion);
        // Store the actual device version in state for display
        setActualDeviceVersion(actualVersionWithBuild);
        
        // NEW: SIMPLE VERSION MATCH CHECK - If versions match exactly, no update needed
        if (actualAppVersion === firebaseVersion) {
          console.log('✅ VERSIONS MATCH EXACTLY - NO UPDATE NEEDED');
          console.log('  - ACTUAL device version:', actualAppVersion);
          console.log('  - Firebase version:', firebaseVersion);
          console.log('  - Result: Versions are identical');
          console.log('  - Action: NO UPDATE MODAL - versions are the same');
          console.log('  - Platform:', Platform.OS);
          
          needsUpdate = false;
          setShouldShowUpdate(false);
          setShowUpdateModal(false);
          setRemindLaterClicked(false);
          
          console.log('✅ Update modal will NOT show - versions match exactly');
          console.log('✅ shouldShowUpdate set to false');
          console.log('✅ showUpdateModal set to false');
          console.log('✅ remindLaterClicked reset to false');
          
        } else {
          console.log('🔄 VERSIONS DO NOT MATCH - CHECKING IF UPDATE NEEDED');
          console.log('  - ACTUAL device version:', actualAppVersion);
          console.log('  - Firebase version:', firebaseVersion);
          console.log('  - Result: Versions are different');
          console.log('  - Platform:', Platform.OS);
          
          // VERSION COMPARISON - Compare ACTUAL device version vs Firebase version for the same platform
          needsUpdate = compareVersions(firebaseVersion, actualAppVersion) > 0;
          
          console.log('🔍 Update needed (semantic comparison):', needsUpdate);
          console.log('🔍 Firebase version is higher than device version:', needsUpdate);
          
          // FOR iOS: Always show modal if versions don't match (for testing)
          if (Platform.OS === 'ios') {
            console.log('🍎 iOS: Versions different, forcing modal to show');
            needsUpdate = true;
          }
        }
        
        // Log version comparison results based on the new logic
        if (actualAppVersion === firebaseVersion) {
          console.log('🔍 Version Comparison Results:');
          console.log('  - ACTUAL running version (device):', actualAppVersion);
          console.log('  - Firebase version:', firebaseVersion);
          console.log('  - Version comparison result: VERSIONS MATCH EXACTLY - NO UPDATE');
          console.log('  - Comparison method: EXACT MATCH CHECK');
          console.log('  - Version source: DYNAMIC FROM BUILD FILES');
        } else {
          console.log('🔍 Version Comparison Results:');
          console.log('  - ACTUAL running version (device):', actualAppVersion);
          console.log('  - Firebase version:', firebaseVersion);
          console.log('  - Version comparison result:', needsUpdate ? 'UPDATE NEEDED' : 'UP TO DATE');
          console.log('  - Comparison method: ACTUAL DEVICE vs Firebase for same platform');
          console.log('  - Version source: DYNAMIC FROM BUILD FILES');
        }
        
        if (needsUpdate) {
          console.log('🔄 UPDATE REQUIRED:');
          console.log('  - ACTUAL running version (device):', actualAppVersion);
          console.log('  - Firebase version:', firebaseVersion);
          console.log(`  - Platform: ${Platform.OS}`);
          
          // Set update flags for BOTH platforms
          setShouldShowUpdate(true);
          setShowUpdateModal(true);
          
          // Reset remind later state to ensure auto-show works
          setRemindLaterClicked(false);
          
          console.log('📱 Update Modal Activated:');
          console.log('  - shouldShowUpdate:', true);
          console.log('  - showUpdateModal:', true);
          console.log('  - remindLaterClicked:', false);
          console.log('  - latestVersionFromFirebase:', firebaseVersion);
          console.log('  - Auto-show will work every 5 seconds when closed');
          
          showToast('info', 'Update Available', `New version ${firebaseVersion} available for download`);
          
        } else {
          console.log('✅ APP IS UP TO DATE:');
          console.log('  - ACTUAL running version (device):', actualAppVersion);
          console.log('  - Firebase version:', firebaseVersion);
          console.log('  - No update needed');
          
          setShouldShowUpdate(false);
          setShowUpdateModal(false);
          setRemindLaterClicked(false); // Reset remind later if versions now match
          console.log('✅ Reset remind later state - versions now match');
        }
      } else {
        console.log('❌ No version data found in Firebase at path: Android_IOS_Update_Version/Android_IOS_Update_V');
        setShouldShowUpdate(false);
        setShowUpdateModal(false);
        setLatestVersionFromFirebase('1.0.0'); // Set default
        showToast('info', 'Version Check', 'No version data found in Firebase');
      }
      
      console.log('📱 Version Check Completed:');
      console.log('  - ACTUAL running version (device):', actualAppVersion);
      console.log('  - Firebase version:', firebaseVersion);
      console.log('  - Should show update:', needsUpdate);
      console.log('  - Platform:', Platform.OS);
      console.log('  - Firebase path verified: Android_IOS_Update_Version/Android_IOS_Update_V');
      console.log('  - Version source: DYNAMIC from build files');
      
    } catch (error) {
      console.error('❌ Error checking for updates:', error);
      showToast('error', 'Update Check Failed', 'Could not check for updates');
      setShouldShowUpdate(false);
      setShowUpdateModal(false);
    } finally {
      if (isMounted.current) {
        // Add a small delay before hiding sync modal to ensure update modal can show
        setTimeout(() => {
          setSyncModalVisible(false);
          setSyncModalMessage('Processing...');
        }, 500);
      }
    }
  }, [isOnline, showToast, withRetry]);

  // Update country in Firebase and AsyncStorage
  const updateCountryInDatabase = useCallback(
    async (userId, country) => {
      if (!isMounted.current || !userId || !countryOptions.includes(country)) {
        console.error('Invalid userId or country:', { userId, country });
        throw new Error('Invalid user or country.');
      }

      setLoading(true);
      setSyncModalVisible(true);
      setSyncModalMessage('Updating Country');

      try {
        if (!isOnline) throw new Error('Offline mode');

        const userRef = database().ref(`Globalusers/admins/${userId}`);
        await withRetry(() =>
          userRef.update({
            selectedCountry: country,
            lastCountryUpdate: new Date().toISOString(),
          })
        );

        await withRetry(() =>
          Promise.all([
            database().ref(`countries/Zambia/users/${userId}`).remove(),
            database().ref(`countries/DRC/users/${userId}`).remove(),
          ])
        );

        const userData = {
          userId,
          email: auth().currentUser?.email || 'Unknown',
          name: auth().currentUser?.displayName || 'Unknown',
          role: 'admin',
          lastLogin: new Date().toISOString(),
        };

        if (country === 'Both Countries') {
          await withRetry(() =>
            Promise.all([
              database().ref(`countries/Zambia/users/${userId}`).set(userData),
              database().ref(`countries/DRC/users/${userId}`).set(userData),
            ])
          );
        } else {
          await withRetry(() =>
            database().ref(`countries/${country}/users/${userId}`).set(userData)
          );
        }

        await AsyncStorage.setItem('selectedCountry', country);
        console.log(`Successfully updated country: ${country}`);
        showToast('success', 'Success', `Country updated to ${country}`);
      } catch (error) {
        console.error('Error updating country:', error);
        let errorMessage = 'Failed to update country.';
        if (error.message.includes('Offline') || !isOnline) {
          errorMessage = 'Offline mode. Changes will sync when online.';
          await AsyncStorage.setItem('selectedCountry', country);
          const newPendingActions = [...pendingActions, { type: 'updateCountry', userId, country }];
          setPendingActions(newPendingActions);
          await AsyncStorage.setItem('pendingActions', JSON.stringify(newPendingActions));
        } else if (error.code === 'PERMISSION_DENIED') {
          errorMessage = 'Permission denied. Please contact support.';
        }
        showToast('error', 'Error', errorMessage);
      } finally {
        if (isMounted.current) {
          setLoading(false);
          setSyncModalVisible(false);
          setSyncModalMessage('Processing...');
        }
      }
    },
    [showToast, withRetry, isOnline, pendingActions]
  );

  // Fetch modules from Firebase or AsyncStorage
  const fetchModules = useCallback(
    async (uid) => {
      if (!isMounted.current) return;
      try {
        if (!isOnline) {
          const storedModules = await AsyncStorage.getItem('modules');
          if (storedModules) {
            setModules(JSON.parse(storedModules));
          }
          return;
        }
        setSyncModalVisible(true);
        setSyncModalMessage('Fetching Modules');
        const userRef = database().ref(`Globalusers/admins/${uid}`);
        const snapshot = await withRetry(() => userRef.once('value'), 2, 1000);
        const userData = snapshot.val();
        
        if (userData && userData.modules) {
          const userModules = userData.modules;
          // Extract module names and ensure they exist in moduleDefinitions
          const moduleNames = userModules
            .map((mod) => mod.module)
            .filter((moduleName) => moduleDefinitions[moduleName]); // Only include valid modules
          
          if (moduleNames.length > 0) {
            setModules(moduleNames);
            await AsyncStorage.setItem('modules', JSON.stringify(moduleNames));
            console.log('Modules fetched from Realtime Database:', moduleNames);
          } else {
            // If no valid modules found, set default modules
            const defaultModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
            setModules(defaultModules);
            await AsyncStorage.setItem('modules', JSON.stringify(defaultModules));
            console.log('No valid modules found, using defaults:', defaultModules);
          }
        } else {
          // If no modules found, set default modules
          const defaultModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
          setModules(defaultModules);
          await AsyncStorage.setItem('modules', JSON.stringify(defaultModules));
          console.log('No modules found, using defaults:', defaultModules);
        }
      } catch (error) {
        console.error('Error fetching modules:', error);
        let errorMessage = 'Failed to fetch modules. Using local data.';
        if (error.message.includes('timeout')) {
          errorMessage = 'Module fetch timed out. Using offline data.';
        } else if (error.code === 'PERMISSION_DENIED') {
          errorMessage = 'Permission denied. Please contact support.';
        } else if (error.code === 'NETWORK_ERROR' || !isOnline) {
          errorMessage = 'Offline mode. Using local module data.';
        }
        showToast('error', 'Error', errorMessage);
        
        // Try to get stored modules
        const storedModules = await AsyncStorage.getItem('modules');
        if (storedModules) {
          setModules(JSON.parse(storedModules));
        } else {
          // Set default modules if none stored
          const defaultModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
          setModules(defaultModules);
          await AsyncStorage.setItem('modules', JSON.stringify(defaultModules));
        }
      } finally {
        if (isMounted.current) {
          setSyncModalVisible(false);
          setSyncModalMessage('Processing...');
        }
      }
    },
    [isOnline, showToast, withRetry]
  );

  // Sync pending actions when back online
  const syncPendingActions = useCallback(async () => {
    if (!isMounted.current || !isOnline) return;

    setSyncModalVisible(true);
    setSyncModalMessage('Syncing Pending Actions');

    try {
      const storedPending = await AsyncStorage.getItem('pendingActions');
      const actions = storedPending ? JSON.parse(storedPending) : [];

      if (actions.length === 0) return;

      for (const action of actions) {
        if (action.type === 'updateCountry') {
          await updateCountryInDatabase(action.userId, action.country);
        } else if (action.type === 'logNavigation') {
          await logNavigationAction(action.action);
        }
      }

      setPendingActions([]);
      await AsyncStorage.setItem('pendingActions', JSON.stringify([]));
      showToast('success', 'Success', 'Pending actions synced successfully');
    } catch (error) {
      console.error('Error syncing pending actions:', error);
      showToast('error', 'Error', 'Failed to sync pending actions');
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setSyncModalMessage('Processing...');
      }
    }
  }, [isOnline, updateCountryInDatabase, showToast]);

  // Log navigation action to Firebase
  const logNavigationAction = useCallback(
    async (action) => {
      if (!isMounted.current || !userId) return;

      try {
        if (!isOnline) {
          const newPendingActions = [...pendingActions, { type: 'logNavigation', action }];
          setPendingActions(newPendingActions);
          await AsyncStorage.setItem('pendingActions', JSON.stringify(newPendingActions));
          console.log('Offline: Navigation action queued for sync:', action);
          return;
        }

        const timestamp = new Date().toISOString();
        const actionData = {
          action,
          timestamp,
          selectedCountry: selectedCountry || 'None',
          userId,
        };

        await withRetry(() =>
          database()
            .ref(`masterData/${userId}/${timestamp.replace(/[:.]/g, '-')}`)
            .set(actionData)
        );
        console.log(`Logged navigation action: ${action}`);
      } catch (error) {
        console.error('Error logging navigation action:', error);
        if (isMounted.current) showToast('error', 'Error', 'Failed to log navigation action.');
      }
    },
    [userId, selectedCountry, showToast, isOnline, pendingActions]
  );

  // Handle country selection
  const handleCountrySelect = useCallback(
    async (country) => {
      if (!isMounted.current) return;
      try {
        if (!countryOptions.includes(country)) {
          showToast('error', 'Error', 'Invalid country selected.');
          return;
        }

        setSelectedCountry(country);
        setModalVisible(false);

        if (userId && isOnline) {
          await updateCountryInDatabase(userId, country);
        } else {
          await AsyncStorage.setItem('selectedCountry', country);
          const newPendingActions = [...pendingActions, { type: 'updateCountry', userId, country }];
          setPendingActions(newPendingActions);
          await AsyncStorage.setItem('pendingActions', JSON.stringify(newPendingActions));
          showToast('info', 'Offline', `Country set to ${country}. Will sync when online.`);
        }
      } catch (error) {
        console.error('Error selecting country:', error);
        showToast('error', 'Error', error.message);
      }
    },
    [userId, isOnline, showToast, updateCountryInDatabase, pendingActions]
  );

  // Handle TouchableOpacity presses
  const handlePress = useCallback(
    (action, actionName) => {
      if (!isMounted.current) return;
      if (!selectedCountry) {
        setModalVisible(true);
        showToast('error', 'Error', 'Please select a country first.');
        return;
      }
      if (actionName) {
        logNavigationAction(actionName).then(() => action());
      } else {
        action();
      }
    },
    [selectedCountry, logNavigationAction, showToast]
  );

  // Handle back button press only when HomeScreen is focused
  useFocusEffect(
    useCallback(() => {
      const backAction = () => {
        Alert.alert(
          'Exit App',
          'Are you sure you want to exit the app?',
          [
            {
              text: 'Cancel',
              style: 'cancel',
            },
            {
              text: 'Exit',
              style: 'destructive',
              onPress: () => BackHandler.exitApp(),
            },
          ]
        );
        return true; // Prevent default back action
      };

      const backHandler = BackHandler.addEventListener('hardwareBackPress', backAction);

      return () => backHandler.remove();
    }, [])
  );

  // Load data and set up listeners
  useEffect(() => {
    let authUnsubscribe;
    let netInfoUnsubscribe;
    let loadingTimeout;

    const loadData = async () => {
      try {
        const storedPending = await AsyncStorage.getItem('pendingActions');
        if (storedPending) setPendingActions(JSON.parse(storedPending));

        authUnsubscribe = auth().onAuthStateChanged(async (user) => {
          if (!isMounted.current) return;

          const uid = user ? user.uid : null;
          setUserId(uid);

          const onlineStatus = await checkNetworkStatus();
          setIsOnline(onlineStatus);

          if (!onlineStatus && !hasShownOfflineToast) {
            showToast('info', 'Offline Mode', 'Operating in offline mode. Please connect to the internet.');
            setHasShownOfflineToast(true);
          }

          try {
            const storedCountry = await AsyncStorage.getItem('selectedCountry');
            if (storedCountry && countryOptions.includes(storedCountry)) {
              setSelectedCountry(storedCountry);
            }

            // Always clear modules before fetching for the logged-in user
            await AsyncStorage.removeItem('modules');

            if (uid && onlineStatus) {
              await checkFirebaseConnection(uid);
              // Check for updates and wait for completion
              await checkForUpdates();
              const [zambiaSnapshot, drcSnapshot] = await Promise.all([
                withRetry(() => database().ref(`countries/Zambia/users/${uid}`).once('value'), 2, 1000),
                withRetry(() => database().ref(`countries/DRC/users/${uid}`).once('value'), 2, 1000),
              ]);
              const zambiaExists = zambiaSnapshot.exists;
              const drcExists = drcSnapshot.exists;

              let preSelectedCountry = storedCountry || 'Zambia'; // Default to Zambia if no data
              if (zambiaExists && drcExists) {
                preSelectedCountry = 'Both Countries';
              } else if (zambiaExists) {
                preSelectedCountry = 'Zambia';
              } else if (drcExists) {
                preSelectedCountry = 'DRC';
              }
              setSelectedCountry(preSelectedCountry);
              await AsyncStorage.setItem('selectedCountry', preSelectedCountry);

              // Always fetch modules from Firebase for the logged-in user
              await fetchModules(uid);
            } else {
              // If offline, only use modules from AsyncStorage if present
              const storedModules = await AsyncStorage.getItem('modules');
              if (storedModules) {
                setModules(JSON.parse(storedModules));
              }
            }
            
            // Set data loaded and loading to false regardless of success/failure
            setDataLoaded(true);
            setLoading(false);
            
          } catch (error) {
            console.error('Error loading data:', error);
            
            // Even on error, try to load stored data and set loading to false
            try {
              const storedModules = await AsyncStorage.getItem('modules');
              if (storedModules) {
                setModules(JSON.parse(storedModules));
              }
              const storedCountry = await AsyncStorage.getItem('selectedCountry');
              if (storedCountry && countryOptions.includes(storedCountry)) {
                setSelectedCountry(storedCountry);
              }
            } catch (storedDataError) {
              console.error('Error loading stored data:', storedDataError);
            }
            
            setDataLoaded(true);
            setLoading(false);
            
            let errorMessage = 'Failed to load user data. Using local data.';
            if (error.message.includes('timeout')) {
              errorMessage = 'Data loading timed out. Using offline data.';
            } else if (error.code === 'PERMISSION_DENIED') {
              errorMessage = 'Permission denied. Please contact support.';
            } else if (error.code === 'NETWORK_ERROR' || !onlineStatus) {
              errorMessage = 'Offline mode. Using local data.';
            }
            showToast('error', 'Error', errorMessage);
          }
        });

        netInfoUnsubscribe = NetInfo.addEventListener((state) => {
          if (!isMounted.current) return;

          const newOnlineStatus = state.isConnected;
          setIsOnline(newOnlineStatus);

          if (!newOnlineStatus && !hasShownOfflineToast) {
            showToast('info', 'Offline Mode', 'Operating in offline mode. Please connect to the internet.');
            setHasShownOfflineToast(true);
          } else if (newOnlineStatus && !hasShownOnlineToast) {
            showToast('success', 'Online', 'Now online, data synced');
            setHasShownOnlineToast(true);
            setHasShownOfflineToast(false);
            if (pendingActions.length > 0) {
              syncPendingActions().catch((error) => console.error('Sync pending actions failed:', error));
            }
            if (userId) {
              fetchModules(userId).catch((error) => console.error('Fetch modules failed:', error));
            }
          }
        });
      } catch (error) {
        console.error('Auth state listener error:', error);
        setDataLoaded(true);
        setLoading(false);
        showToast('error', 'Error', 'Failed to initialize authentication.');
      }
    };

    // Set a timeout to prevent infinite skeleton loading
    loadingTimeout = setTimeout(() => {
      if (isMounted.current && loading) {
        console.log('Loading timeout reached, forcing data loaded state');
        setDataLoaded(true);
        setLoading(false);
        showToast('info', 'Loading Timeout', 'Data loading took too long. Using cached data.');
      }
    }, 15000); // 15 seconds timeout

    loadData().catch((error) => {
      console.error('Load data failed:', error);
      // Ensure loading states are reset even on error
      setDataLoaded(true);
      setLoading(false);
    });

    return () => {
      isMounted.current = false;
      if (authUnsubscribe) authUnsubscribe();
      if (netInfoUnsubscribe) netInfoUnsubscribe();
      if (loadingTimeout) clearTimeout(loadingTimeout);
      setLoading(false);
    };
  }, [checkNetworkStatus, showToast, withRetry, checkFirebaseConnection, updateCountryInDatabase, syncPendingActions, fetchModules, checkForUpdates]);

  // Real-time listener for country and module changes
  useEffect(() => {
    if (!userId || !dataLoaded || !isOnline) return;

    const userRef = database().ref(`Globalusers/admins/${userId}`);

    const handleUserChange = debounce((snapshot) => {
      if (!isMounted.current) return;
      const data = snapshot.val();
      if (data?.selectedCountry && data.selectedCountry !== selectedCountry && countryOptions.includes(data.selectedCountry)) {
        setSelectedCountry(data.selectedCountry);
        AsyncStorage.setItem('selectedCountry', data.selectedCountry).catch((error) =>
          console.error('Error updating AsyncStorage:', error)
        );
        showToast('info', 'Country Updated', `Country synced to ${data.selectedCountry}`);
        console.log('Updated country from Realtime Database listener:', data.selectedCountry);
      }
      if (data?.modules) {
        const userModules = data.modules;
        // Extract module names and ensure they exist in moduleDefinitions
        const moduleNames = userModules
          .map((mod) => mod.module)
          .filter((moduleName) => moduleDefinitions[moduleName]); // Only include valid modules
        
        if (moduleNames.length > 0) {
          setModules(moduleNames);
          AsyncStorage.setItem('modules', JSON.stringify(moduleNames)).catch((error) =>
            console.error('Error updating AsyncStorage modules:', error)
          );
          showToast('info', 'Modules Updated', 'Module permissions synced from server');
          console.log('Updated modules from Realtime Database listener:', moduleNames);
        } else {
          // If no valid modules found, keep current modules but log warning
          console.warn('No valid modules found in real-time update, keeping current modules');
          // Optionally set default modules if none exist
          if (modules.length === 0) {
            const defaultModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
            setModules(defaultModules);
            AsyncStorage.setItem('modules', JSON.stringify(defaultModules)).catch((error) =>
              console.error('Error setting default modules:', error)
            );
          }
        }
      }
    }, 1000, { leading: false, trailing: true });

    userRef.on('value', handleUserChange);

    return () => {
      userRef.off('value', handleUserChange);
    };
  }, [userId, selectedCountry, dataLoaded, showToast, isOnline]);

  // Add this useEffect to sync selectedCountries with database value on load
  useEffect(() => {
    if (!userId || !dataLoaded) return;
    const userRef = database().ref(`Globalusers/admins/${userId}`);
    userRef.once('value').then((snapshot) => {
      const data = snapshot.val();
      if (data?.countries && Array.isArray(data.countries)) {
        setSelectedCountries(data.countries);
      } else {
        setSelectedCountries([]);
      }
    });
  }, [userId, dataLoaded]);

  // REMOVED: Auto-popup timer - Modal will only show when manually triggered or when "Remind Me Later" is clicked

  // REMOVED: Periodic version check - Only check once when screen loads

  // REMOVED: Auto-reminder timer - Modal will only show again when "Remind Me Later" is clicked

  // REMOVED: Android aggressive reminder timer - Modal will only show again when "Remind Me Later" is clicked

  // Handler for toggling country selection
  const handleToggleCountry = (country) => {
    setSelectedCountries((prev) => {
      if (country === 'Both Countries') {
        // Toggle both Zambia and DRC
        const hasBoth = prev.includes('Zambia') && prev.includes('DRC');
        if (hasBoth) {
          // Unselect both
          return prev.filter((c) => c !== 'Zambia' && c !== 'DRC' && c !== 'Both Countries');
        } else {
          // Select both
          return ['Zambia', 'DRC'];
        }
      } else {
        let newSelected;
        if (prev.includes(country)) {
          newSelected = prev.filter((c) => c !== country && c !== 'Both Countries');
        } else {
          newSelected = [...prev.filter((c) => c !== 'Both Countries'), country];
        }
        // If both Zambia and DRC are selected, show Both Countries as selected
        if (newSelected.includes('Zambia') && newSelected.includes('DRC')) {
          return ['Zambia', 'DRC'];
        }
        return newSelected;
      }
    });
  };

  // Handler for saving selected countries to database
  const handleSaveCountries = async () => {
    if (!userId) return;
    setSyncModalVisible(true);
    setSyncModalMessage('Updating Countries...');
    try {
              await database().ref(`Globalusers/admins/${userId}`).update({ countries: selectedCountries });
      showToast('success', 'Success', 'Countries updated successfully');
      setModalVisible(false);
    } catch (error) {
      showToast('error', 'Error', 'Failed to update countries');
    } finally {
      setSyncModalVisible(false);
      setSyncModalMessage('Processing...');
    }
  };

  // Handler for remind later functionality - Check versions and show popup again if needed
  const handleRemindLater = useCallback(async () => {
    console.log(`Remind later clicked on ${Platform.OS}`);
    console.log(`Platform: ${Platform.OS}, shouldShowUpdate: ${shouldShowUpdate}`);
    
    // Close the modal first
    setShowUpdateModal(false);
    
    // Get both versions and check if they still need update
    try {
      const actualAppVersion = await DeviceInfo.getVersion();
      const actualBuildNumber = await DeviceInfo.getBuildNumber();
      const actualVersionWithBuild = `${actualAppVersion} (${actualBuildNumber})`;
      
      console.log('🔄 REMIND LATER CLICKED - VERSION COMPARISON:');
      console.log('  - Platform:', Platform.OS);
      console.log('  - ACTUAL Device Version:', actualVersionWithBuild);
      console.log('  - Firebase Version:', latestVersionFromFirebase);
      console.log('  - Versions Match:', actualAppVersion === latestVersionFromFirebase ? '✅ YES' : '❌ NO');
      
      // Check if versions now match - if they do, don't show modal again
      if (actualAppVersion === latestVersionFromFirebase) {
        console.log('✅ VERSIONS NOW MATCH - NO NEED TO SHOW MODAL AGAIN');
        console.log('  - Response: Versions are the same, modal will not show again');
        
        // Reset all states since versions now match
        setShouldShowUpdate(false);
        setRemindLaterClicked(false);
        
        showToast('success', 'Up to Date', 'Your app is now up to date!');
        return;
      } else {
        console.log('🔄 VERSIONS STILL DIFFERENT - WILL SHOW MODAL AGAIN');
        console.log('  - Response: Will show modal again in 3 seconds');
        
        // Set remind later flag
        setRemindLaterClicked(true);
        
        // Show modal again after 3 seconds if versions still differ
        setTimeout(() => {
          if (isMounted.current && shouldShowUpdate) {
            console.log(`🔄 Remind later: Showing update modal again after 3 seconds`);
            console.log(`📱 ${Platform.OS} Debug - shouldShowUpdate: ${shouldShowUpdate}, latestVersion: ${latestVersionFromFirebase}`);
            setShowUpdateModal(true);
            showToast('info', 'Update Reminder', 'Update modal shown again - versions still differ');
          }
        }, 3000);
      }
      
    } catch (error) {
      console.error('❌ Error getting device version for comparison:', error);
      // Fallback: just set remind later flag
      setRemindLaterClicked(true);
      
      // Show modal again after 3 seconds as fallback
      setTimeout(() => {
        if (isMounted.current && shouldShowUpdate) {
          setShowUpdateModal(true);
          showToast('info', 'Update Reminder', 'Update modal shown again');
        }
      }, 3000);
    }
    
    console.log(`Remind later clicked on ${Platform.OS} - modal will show again in 3 seconds if versions differ`);
  }, [shouldShowUpdate, latestVersionFromFirebase, showToast]);

  // Enhanced Android-specific remind later handler - Check versions and show popup again if needed
  const handleAndroidRemindLater = useCallback(async () => {
    if (Platform.OS !== 'android') {
      console.log('This function is only for Android devices');
      return;
    }
    
    console.log('🤖 Android Remind Later clicked');
    console.log('🤖 Current state - shouldShowUpdate:', shouldShowUpdate, 'latestVersion:', latestVersionFromFirebase);
    
    // Close the modal first
    setShowUpdateModal(false);
    
    // Get both versions and check if they still need update
    try {
      const actualAppVersion = await DeviceInfo.getVersion();
      const actualBuildNumber = await DeviceInfo.getBuildNumber();
      const actualVersionWithBuild = `${actualAppVersion} (${actualBuildNumber})`;
      
      console.log('🤖 ANDROID REMIND LATER - VERSION COMPARISON:');
      console.log('  - Platform: Android');
      console.log('  - ACTUAL Device Version:', actualVersionWithBuild);
      console.log('  - Firebase Version:', latestVersionFromFirebase);
      console.log('  - Versions Match:', actualAppVersion === latestVersionFromFirebase ? '✅ YES' : '❌ NO');
      
      // Check if versions now match - if they do, don't show modal again
      if (actualAppVersion === latestVersionFromFirebase) {
        console.log('✅ ANDROID VERSIONS NOW MATCH - NO NEED TO SHOW MODAL AGAIN');
        console.log('  - Response: Versions are the same, modal will not show again');
        
        // Reset all states since versions now match
        setShouldShowUpdate(false);
        setRemindLaterClicked(false);
        
        showToast('success', 'Up to Date', 'Your Android app is now up to date!');
        return;
      } else {
        console.log('🔄 ANDROID VERSIONS STILL DIFFERENT - WILL SHOW MODAL AGAIN');
        console.log('  - Response: Will show modal again in 3 seconds');
        
        // Set remind later flag
        setRemindLaterClicked(true);
        
        // Show modal again after 3 seconds if versions still differ
        setTimeout(() => {
          if (isMounted.current && shouldShowUpdate) {
            console.log('🤖 Android: 3 seconds passed, showing update modal again');
            setShowUpdateModal(true);
            showToast('info', 'Update Reminder', 'Update modal shown again - versions still differ');
          }
        }, 3000);
      }
      
    } catch (error) {
      console.error('❌ Error getting Android device version for comparison:', error);
      // Fallback: just set remind later flag
      setRemindLaterClicked(true);
      
      // Show modal again after 3 seconds as fallback
      setTimeout(() => {
        if (isMounted.current && shouldShowUpdate) {
          setShowUpdateModal(true);
          showToast('info', 'Update Reminder', 'Update modal shown again');
        }
      }, 3000);
    }
    
    console.log('🤖 Android remind later: Modal will show again in 3 seconds if versions differ');
  }, [shouldShowUpdate, latestVersionFromFirebase, showToast]);

  // Initialize Firebase versions (utility function)
  const initializeFirebaseVersions = useCallback(async () => {
    try {
      console.log('🔄 Initializing Realtime Database versions...');
      setSyncModalVisible(true);
      setSyncModalMessage('Setting up version data...');
      
      const versionRef = database().ref('Android_IOS_Update_Version/Android_IOS_Update_V');
      
      // Get ACTUAL device versions for current platform
      const actualDeviceVersion = await DeviceInfo.getVersion();
      const actualBuildNumber = await DeviceInfo.getBuildNumber();
      const actualVersionWithBuild = `${actualDeviceVersion} (${actualBuildNumber})`;
      
      // Set Firebase versions higher than actual device version to trigger updates
      const androidFirebaseVersion = '2.2.0';  // Higher than actual device version
      const iosFirebaseVersion = '1.0.3';      // Higher than actual device version
      
      await versionRef.set({
        Android_version: androidFirebaseVersion,
        IOS_version: iosFirebaseVersion,
        lastUpdated: new Date().toISOString(),
        updatedBy: 'system',
        metadata: {
          actualDeviceVersion: actualVersionWithBuild,
          platform: Platform.OS,
          androidFirebaseVersion,
          iosFirebaseVersion,
          description: 'Firebase versions set higher than actual device version to trigger updates'
        }
      });
      
      console.log('📱 Firebase versions initialized:');
      console.log('  - Current Device:');
      console.log('    - Platform:', Platform.OS);
      console.log('    - ACTUAL device version:', actualVersionWithBuild);
      console.log('  - Android:');
      console.log('    - Firebase version:', androidFirebaseVersion);
      console.log('    - Update needed:', compareVersions(androidFirebaseVersion, actualDeviceVersion) > 0);
      console.log('  - iOS:');
      console.log('    - Firebase version:', iosFirebaseVersion);
      console.log('    - Update needed:', compareVersions(iosFirebaseVersion, actualDeviceVersion) > 0);
      console.log('  - Firebase path: Android_IOS_Update_Version/Android_IOS_Update_V');
      
      console.log('✅ Realtime Database versions initialized successfully');
      showToast('success', 'Success', 'Firebase versions initialized - Updates will be triggered');
      
      // Refresh the version check
      setTimeout(() => {
        checkForUpdates();
      }, 1000);
      
    } catch (error) {
      console.error('❌ Error initializing Realtime Database versions:', error);
      showToast('error', 'Error', 'Failed to initialize Firebase versions');
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setSyncModalMessage('Processing...');
      }
    }
  }, [checkForUpdates, showToast, compareVersions]);

  // Function to get ACTUAL running app version from device (dynamically from build files)
  const getCurrentAppVersion = useCallback(async () => {
    try {
      // Get ACTUAL running version from device (dynamically from build files)
      const actualVersion = await DeviceInfo.getVersion();
      const actualBuildNumber = await DeviceInfo.getBuildNumber();
      const versionWithBuild = `${actualVersion} (${actualBuildNumber})`;
      
      console.log('📱 Getting ACTUAL running app version:');
      console.log('  - Platform:', Platform.OS);
      console.log('  - Source: DeviceInfo.getVersion() - DYNAMIC FROM BUILD FILES');
      console.log('  - Android: versionName from build.gradle');
      console.log('  - iOS: CFBundleShortVersionString from Info.plist');
      console.log('  - ACTUAL version:', actualVersion);
      console.log('  - ACTUAL build number:', actualBuildNumber);
      console.log('  - Version with build:', versionWithBuild);
      
      return versionWithBuild;
    } catch (error) {
      console.error('❌ Error getting actual device version:', error);
      console.log('⚠️ Falling back to default version: 1.0.0');
      return '1.0.0';
    }
  }, []);

  // Function to get current version with build number
  const getCurrentVersionWithBuild = useCallback(async () => {
    try {
      const appVersionData = require('../../app-version.json');
      const platform = Platform.OS === 'ios' ? 'ios' : 'android';
      const currentAppVersion = appVersionData[platform]?.version || '1.0.0';
      const currentBuildNumber = appVersionData[platform]?.buildNumber || '1';
      
      console.log('📱 Getting current version with build:');
      console.log('  - Platform:', Platform.OS);
      console.log('  - Version:', currentAppVersion);
      console.log('  - Build number:', currentBuildNumber);
      
      return `${currentAppVersion} (${currentBuildNumber})`;
    } catch (error) {
      console.error('❌ Error getting version with build:', error);
      console.log('⚠️ Falling back to basic version');
      return getCurrentAppVersion();
    }
  }, [getCurrentAppVersion]);

  // Test function to manually trigger update modal (for testing)
  const testUpdateModal = useCallback(async () => {
    console.log('🧪 Manually triggering update modal for testing');
    const platform = Platform.OS === 'ios' ? 'ios' : 'android';
    
    // Get ACTUAL device version
    const actualVersion = await DeviceInfo.getVersion();
    const actualBuildNumber = await DeviceInfo.getBuildNumber();
    const actualVersionWithBuild = `${actualVersion} (${actualBuildNumber})`;
    
    console.log('🧪 Test Modal Setup:');
    console.log('  - Platform:', Platform.OS);
    console.log('  - ACTUAL device version:', actualVersionWithBuild);
    console.log('  - Version source: DYNAMIC FROM BUILD FILES');
    console.log('  - Android: versionName from build.gradle');
    console.log('  - iOS: CFBundleShortVersionString from Info.plist');
    
    // Use the actual Firebase version if available, otherwise set a test version
    if (latestVersionFromFirebase && latestVersionFromFirebase !== '1.0.0') {
      console.log('🧪 Using existing Firebase version:', latestVersionFromFirebase);
      setShouldShowUpdate(true);
      setShowUpdateModal(true);
    } else {
      // Set a test version that will trigger update
      const testVersion = platform === 'ios' ? '1.0.2' : '2.2.0';
      setLatestVersionFromFirebase(testVersion);
      setShouldShowUpdate(true);
      setShowUpdateModal(true);
      console.log('🧪 Set test version:', testVersion);
    }
    
    console.log('🧪 Test modal triggered:');
    console.log('  - Platform:', Platform.OS);
    console.log('  - ACTUAL device version:', actualVersionWithBuild);
    console.log('  - Firebase version:', latestVersionFromFirebase);
    console.log('  - Should show update:', true);
    console.log('  - Show modal:', true);
    console.log('  - Version source: DYNAMIC FROM BUILD FILES');
  }, [latestVersionFromFirebase]);

  // Function to set a different version in Firebase for testing
  const setTestVersion = useCallback(async (version) => {
    try {
      console.log(`🧪 Setting test version: ${version}`);
      setSyncModalVisible(true);
      setSyncModalMessage('Setting test version...');
      
      const versionRef = database().ref('Android_IOS_Update_Version/Android_IOS_Update_V');
      const platformKey = Platform.OS === 'ios' ? 'IOS_version' : 'Android_version';
      
      // Get ACTUAL device version for comparison
      const actualDeviceVersion = await DeviceInfo.getVersion();
      const actualBuildNumber = await DeviceInfo.getBuildNumber();
      const actualVersionWithBuild = `${actualDeviceVersion} (${actualBuildNumber})`;
      
      await versionRef.update({
        [platformKey]: version,
        lastUpdated: new Date().toISOString(),
        updatedBy: 'test',
        testInfo: {
          platform,
          actualDeviceVersion: actualVersionWithBuild,
          testVersion: version,
          willTriggerUpdate: compareVersions(version, actualDeviceVersion) > 0
        }
      });
      
      console.log(`✅ Test version ${version} set successfully in Firebase`);
      console.log(`📱 Platform: ${Platform.OS}, Key: ${platformKey}`);
      console.log(`📱 ACTUAL device version: ${actualVersionWithBuild}`);
      console.log(`📱 Test version: ${version}`);
      console.log(`📱 Will trigger update: ${compareVersions(version, actualDeviceVersion) > 0}`);
      console.log(`📱 Firebase path: Android_IOS_Update_Version/Android_IOS_Update_V`);
      
      showToast('success', 'Test Version Set', `Version set to ${version}`);
      
      // Refresh version check
      setTimeout(() => {
        checkForUpdates();
      }, 1000);
      
    } catch (error) {
      console.error('❌ Error setting test version:', error);
      showToast('error', 'Error', 'Failed to set test version');
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setSyncModalMessage('Processing...');
      }
    }
  }, [checkForUpdates, showToast, compareVersions]);

  // Function to set Android test version specifically
  const setAndroidTestVersion = useCallback(async () => {
    if (Platform.OS !== 'android') {
      showToast('info', 'Info', 'This function is only for Android devices');
      return;
    }
    
    try {
      console.log('🧪 Setting Android test version...');
      setSyncModalVisible(true);
      setSyncModalMessage('Setting Android test version...');
      
      const versionRef = database().ref('Android_IOS_Update_Version/Android_IOS_Update_V');
      
      // Get ACTUAL Android device version
      const actualAndroidVersion = await DeviceInfo.getVersion();
      const actualAndroidBuildNumber = await DeviceInfo.getBuildNumber();
      const actualAndroidVersionWithBuild = `${actualAndroidVersion} (${actualAndroidBuildNumber})`;
      
      // Set Android version to 2.3.0 (higher than actual device version) to trigger update
      const testVersion = '2.3.0';
      
      await versionRef.update({
        Android_version: testVersion,
        lastUpdated: new Date().toISOString(),
        updatedBy: 'android_test',
        testInfo: {
          platform: 'android',
          actualDeviceVersion: actualAndroidVersionWithBuild,
          testVersion: testVersion,
          willTriggerUpdate: compareVersions(testVersion, actualAndroidVersion) > 0,
          firebasePath: 'Android_IOS_Update_Version/Android_IOS_Update_V/Android_version'
        }
      });
      
      console.log('✅ Android test version set successfully');
      console.log('📱 Platform: Android');
      console.log('📱 ACTUAL device version:', actualAndroidVersionWithBuild);
      console.log('📱 Test Firebase version:', testVersion);
      console.log('📱 Will trigger update:', compareVersions(testVersion, actualAndroidVersion) > 0);
      console.log('📱 Firebase path: Android_IOS_Update_Version/Android_IOS_Update_V/Android_version');
      
      showToast('success', 'Android Test Version', `Version set to ${testVersion} - Update needed!`);
      
      // Refresh version check
      setTimeout(() => {
        checkForUpdates();
      }, 1000);
      
    } catch (error) {
      console.error('❌ Error setting Android test version:', error);
      showToast('error', 'Error', 'Failed to set Android test version');
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setSyncModalMessage('Processing...');
      }
    }
  }, [checkForUpdates, showToast, compareVersions]);

  // Comprehensive Android remind later test function
  const testAndroidRemindLater = useCallback(() => {
    if (Platform.OS !== 'android') {
      showToast('info', 'Info', 'This function is only for Android devices');
      return;
    }
    
    console.log('🧪 Testing Android remind later functionality comprehensively...');
    
    // Step 1: Set a test version that will trigger update
    setLatestVersionFromFirebase('2.6.0');
    setShouldShowUpdate(true);
    setShowUpdateModal(true);
    setRemindLaterClicked(false);
    
    showToast('success', 'Android Test', 'Modal shown! Click "Remind Me Later" to test');
    
    // Step 2: Set up a test scenario
    console.log('🧪 Android Remind Later Test Setup:');
    console.log('  - Test version: 2.6.0 (package.json: 2.1.0)');
    console.log('  - shouldShowUpdate: true');
    console.log('  - showUpdateModal: true');
    console.log('  - remindLaterClicked: false');
    console.log('  - Instructions: Click "Remind Me Later" button');
    console.log('  - Expected: Modal closes, then reappears every 3 seconds');
    
    // Step 3: Set up monitoring
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🧪 Android Test: 10 seconds passed, checking state...');
        console.log('  - Current shouldShowUpdate:', shouldShowUpdate);
        console.log('  - Current showUpdateModal:', showUpdateModal);
        console.log('  - Current remindLaterClicked:', remindLaterClicked);
        
        if (shouldShowUpdate && !showUpdateModal) {
          console.log('🧪 Android Test: Modal should be showing but is not visible');
          showToast('info', 'Test Issue', 'Modal should be visible but is not');
        } else if (shouldShowUpdate && showUpdateModal) {
          console.log('🧪 Android Test: Modal is visible as expected');
          showToast('success', 'Test Status', 'Modal is visible as expected');
        }
      }
    }, 10000);
    
  }, [shouldShowUpdate, showUpdateModal, remindLaterClicked]);

  // Function to check current Realtime Database data structure
  const checkFirestoreStructure = useCallback(async () => {
    try {
      console.log('🔍 Checking Realtime Database data structure...');
      setSyncModalVisible(true);
      setSyncModalMessage('Checking Realtime Database structure...');
      
      const versionRef = database().ref('Android_IOS_Update_Version/Android_IOS_Update_V');
      const snapshot = await versionRef.once('value');
      
      if (snapshot.exists()) {
        const data = snapshot.val();
        console.log('📊 Current Realtime Database data:', data);
        console.log('📊 Available fields:', Object.keys(data));
        
        const platformKey = Platform.OS === 'ios' ? 'IOS_version' : 'Android_version';
        const currentVersion = data[platformKey] || 'Not found';
        
        showToast('success', 'Realtime Database Data', `Current ${platformKey}: ${currentVersion}`);
        
        // Update local state
        setLatestVersionFromFirebase(currentVersion);
        
        // Check if update is needed
        const packageJson = require('../../package.json');
        const platform = Platform.OS === 'ios' ? 'ios' : 'android';
        const packageVersion = packageJson.appVersions[platform] || '1.0.0';
        
        if (currentVersion !== packageVersion) {
          setShouldShowUpdate(true);
          showToast('info', 'Update Available', 'Version mismatch detected');
        } else {
          setShouldShowUpdate(false);
          showToast('success', 'Up to Date', 'Versions match');
        }
        
      } else {
        console.log('❌ Document does not exist in Realtime Database');
        showToast('info', 'No Data', 'Version document not found in Realtime Database');
      }
      
    } catch (error) {
      console.error('❌ Error checking Realtime Database structure:', error);
      showToast('error', 'Error', 'Failed to check Realtime Database structure');
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setSyncModalMessage('Processing...');
      }
    }
  }, [showToast]);

  // Render category with dynamic modules
  const renderCategory = (category, navigationRoute) => {
    // Use the width from the component scope instead of calling Dimensions.get inside the function
    const boxSize = Math.floor((width - 5 * 10) / 4);
    // Ensure minimum box size for better text display
    const minBoxSize = Math.max(boxSize, 70);

    // Get modules for this category
    const categoryModules = modules
      .filter((mod) => moduleDefinitions[mod]?.category === category)
      .map((mod) => ({
        icon: moduleDefinitions[mod].icon,
        label: moduleDefinitions[mod].label,
        onPress: () => handlePress(() => navigation.navigate(moduleDefinitions[mod].navigation), mod),
      }));

    // Special handling for Inspection and Compliances category
    if (category === 'Inspection and Compliances') {
      // Always show HSE Inspection regardless of permissions
      const hseModule = {
        icon: moduleDefinitions.hseInspection.icon,
        label: moduleDefinitions.hseInspection.label,
        onPress: () => handlePress(() => navigation.navigate(moduleDefinitions.hseInspection.navigation), 'hseInspection'),
      };

      // Get other modules based on permissions
      const otherModules = categoryModules.filter(mod => mod.label !== 'HSE Inspection');
      
      // Combine HSE Inspection with other permitted modules
      const allModules = [hseModule, ...otherModules];

      if (allModules.length === 0) return null;

      const firstRow = allModules.slice(0, 4);
      const secondRow = allModules.slice(4);

      return (
        <>
          <Text style={styles.heading}>{category}</Text>
          <View style={{ alignItems: 'center', width: '100%', paddingHorizontal: 5 }}>
            <View style={{
              flexDirection: 'row',
              justifyContent: 'center',
              alignItems: 'flex-start',
              marginBottom: 12,
              width: '100%',
            }}>
              {firstRow.map((item) => (
                <TouchableOpacity
                  key={item.label}
                  style={{ alignItems: 'center', marginHorizontal: 8 }}
                  onPress={item.onPress}
                  activeOpacity={0.8}
                >
                  <View style={{
                    backgroundColor: '#035484',
                    borderRadius: 12,
                    width: minBoxSize,
                    height: minBoxSize,
                    justifyContent: 'center',
                    alignItems: 'center',
                    marginBottom: 8,
                  }}>
                    <Image source={item.icon} style={{
                      width: item.label === 'Daily Report' ? 42 : 48,
                      height: item.label === 'Daily Report' ? 42 : 48,
                      resizeMode: 'contain',
                    }} />
                  </View>
                  <Text style={{
                    color: '#29486A',
                    fontWeight: 'bold',
                    fontSize: 12,
                    textAlign: 'center',
                    width: minBoxSize,
                    lineHeight: 16,
                    flexWrap: 'wrap',
                    numberOfLines: 2,
                  }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                    {item.label}
                  </Text>
                </TouchableOpacity>
              ))}

              
            </View>
            {secondRow.length > 0 && (
              <View style={{
                flexDirection: 'row',
                justifyContent: 'center',
                alignItems: 'flex-start',
                marginBottom: 12,
                width: '100%',
              }}>
                {secondRow.map((item) => (
                  <TouchableOpacity
                    key={item.label}
                    style={{ alignItems: 'center', marginHorizontal: 8 }}
                    onPress={item.onPress}
                    activeOpacity={0.8}
                  >
                    <View style={{
                      backgroundColor: '#035484',
                      borderRadius: 12,
                      width: minBoxSize,
                      height: minBoxSize,
                      justifyContent: 'center',
                      alignItems: 'center',
                      marginBottom: 8,
                    }}>
                      <Image source={item.icon} style={{
                        width: item.label === 'Daily Report' ? 42 : 48,
                        height: item.label === 'Daily Report' ? 42 : 48,
                        resizeMode: 'contain',
                      }} />
                    </View>
                    <Text style={{
                      color: '#29486A',
                      fontWeight: 'bold',
                      fontSize: 12,
                      textAlign: 'center',
                      width: minBoxSize,
                      lineHeight: 16,
                      flexWrap: 'wrap',
                      numberOfLines: 2,
                    }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                      {item.label}
                    </Text>
                  </TouchableOpacity>
                ))}
              </View>
            )}
            <View style={styles.buttonContainer}>
              <TouchableOpacity
                style={styles.gradientButton}
                onPress={() => handlePress(() => navigation.navigate(navigationRoute), `View${category.replace(/ & /g, '').replace(/\s/g, '')}KPI`)}
                activeOpacity={0.8}
              >
                <LinearGradient colors={['#3481BC', '#025383']} style={styles.gradientButtonContent}>
                  <Text style={styles.buttonText}>View KPI</Text>
                </LinearGradient>
              </TouchableOpacity>
            </View>
          </View>
        </>
      );
    }

    // For other categories, show modules based on permissions
    if (categoryModules.length === 0) return null;

    const firstRow = categoryModules.slice(0, 4);
    const secondRow = categoryModules.slice(4);

    return (
      <>
        <Text style={styles.heading}>{category}</Text>
        <View style={{ alignItems: 'center', width: '100%', paddingHorizontal: 5 }}>
          <View style={{
            flexDirection: 'row',
            justifyContent: 'center',
            alignItems: 'flex-start',
            marginBottom: 12,
            width: '100%',
          }}>
            {firstRow.map((item, idx) => (
              <TouchableOpacity
                key={item.label}
                style={{ alignItems: 'center', marginHorizontal: 8 }}
                onPress={item.onPress}
                activeOpacity={0.8}
              >
                <View style={{
                  backgroundColor: '#035484',
                  borderRadius: 12,
                  width: minBoxSize,
                  height: minBoxSize,
                  justifyContent: 'center',
                  alignItems: 'center',
                  marginBottom: 8,
                }}>
                  <Image source={item.icon} style={{
                    width: item.label === 'Daily Report' ? 42 : 48,
                    height: item.label === 'Daily Report' ? 42 : 48,
                    resizeMode: 'contain'
                  }} />
                </View>
                <Text style={{
                  color: '#29486A',
                  fontWeight: 'bold',
                  fontSize: 12,
                  textAlign: 'center',
                  width: minBoxSize,
                  lineHeight: 16,
                  flexWrap: 'wrap',
                  numberOfLines: 2,
                }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                  {item.label}
                </Text>
              </TouchableOpacity>
            ))}
          </View>
          {secondRow.length > 0 && (
            <View style={{
              flexDirection: 'row',
              justifyContent: 'center',
              alignItems: 'flex-start',
              marginBottom: 12,
              width: '100%',
            }}>
              {secondRow.map((item, idx) => (
                <TouchableOpacity
                  key={item.label}
                  style={{ alignItems: 'center', marginHorizontal: 8 }}
                  onPress={item.onPress}
                  activeOpacity={0.8}
                >
                  <View style={{
                    backgroundColor: '#035484',
                    borderRadius: 12,
                    width: minBoxSize,
                    height: minBoxSize,
                    justifyContent: 'center',
                    alignItems: 'center',
                    marginBottom: 8,
                  }}>
                    <Image source={item.icon} style={{
                      width: item.label === 'Daily Report' ? 42 : 48,
                      height: item.label === 'Daily Report' ? 42 : 48,
                      resizeMode: 'contain'
                    }} />
                  </View>
                  <Text style={{
                    color: '#29486A',
                    fontWeight: 'bold',
                    fontSize: 12,
                    textAlign: 'center',
                    width: minBoxSize,
                    lineHeight: 16,
                    flexWrap: 'wrap',
                    numberOfLines: 2,
                  }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                    {item.label}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
          )}
          <View style={styles.buttonContainer}>
            <TouchableOpacity
              style={styles.gradientButton}
              onPress={() => handlePress(() => navigation.navigate(navigationRoute), `View${category.replace(/ & /g, '').replace(/\s/g, '')}KPI`)}
              activeOpacity={0.8}
            >
              <LinearGradient colors={['#3481BC', '#025383']} style={styles.gradientButtonContent}>
                <Text style={styles.buttonText}>View KPI</Text>
              </LinearGradient>
            </TouchableOpacity>
          </View>
        </View>
      </>
    );
  };

  // iOS-specific test function to manually trigger update modal (for testing)
  const testIOSUpdateModal = useCallback(async () => {
    if (Platform.OS !== 'ios') {
      console.log('This function is only for iOS devices');
      return;
    }
    
    console.log('🍎 iOS Update Modal Test - Manually triggering update modal');
    const platform = 'ios';
    
    // Get ACTUAL iOS device version
    const actualVersion = await DeviceInfo.getVersion();
    const actualBuildNumber = await DeviceInfo.getBuildNumber();
    const actualVersionWithBuild = `${actualVersion} (${actualBuildNumber})`;
    
    // Set a test version that will trigger update for iOS
    const testVersion = '1.0.3';
    setLatestVersionFromFirebase(testVersion);
    setShouldShowUpdate(true);
    setShowUpdateModal(true);
    
    // iOS-specific: Reset remind later state to ensure auto-show works
    setRemindLaterClicked(false);
    
    console.log('🍎 iOS Test Setup:');
    console.log('  - ACTUAL device version:', actualVersionWithBuild);
    console.log('  - Test Firebase version:', testVersion);
    console.log('  - shouldShowUpdate:', true);
    console.log('  - showUpdateModal:', true);
    console.log('  - remindLaterClicked:', false);
    console.log('  - Auto-show will work every 5 seconds when closed');
    console.log('  - Version source: DYNAMIC FROM BUILD FILES');
    console.log('  - iOS: CFBundleShortVersionString from Info.plist');
    
    showToast('info', 'iOS Test', 'Modal will auto-show every 5s when closed');
  }, []);

  // NEW: Aggressive iOS modal show function that forces visibility
  const forceShowIOSModalAggressive = useCallback(() => {
    if (Platform.OS !== 'ios') {
      console.log('This function is only for iOS devices');
      return;
    }
    
    console.log('🚀 iOS AGGRESSIVE MODAL SHOW - Using multiple approaches');
    
    // Approach 1: Set all states
    setLatestVersionFromFirebase('1.0.5');
    setShouldShowUpdate(true);
    setShowUpdateModal(true);
    setRemindLaterClicked(false);
    
    // Approach 2: Force multiple re-renders
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🚀 iOS AGGRESSIVE - First re-render');
        setShowUpdateModal(true);
        setShouldShowUpdate(true);
      }
    }, 50);
    
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🚀 iOS AGGRESSIVE - Second re-render');
        setShowUpdateModal(true);
        setShouldShowUpdate(true);
      }
    }, 150);
    
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🚀 iOS AGGRESSIVE - Third re-render');
        setShowUpdateModal(true);
        setShouldShowUpdate(true);
        showToast('success', 'iOS Modal', 'Modal should be visible now!');
      }
    }, 250);
    
    // Approach 3: Use a different modal state approach
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🚀 iOS AGGRESSIVE - Alternative approach');
        // Force the modal to show by changing the state multiple times
        setShowUpdateModal(false);
        setTimeout(() => {
          if (isMounted.current) {
            setShowUpdateModal(true);
            console.log('🚀 iOS AGGRESSIVE - Modal toggled to force visibility');
          }
        }, 100);
      }
    }, 300);
    
    console.log('🚀 iOS AGGRESSIVE MODAL SHOW - All approaches executed');
    showToast('info', 'Aggressive Show', 'Multiple approaches used to show modal!');
  }, []);

  // iOS-specific debug function to monitor update modal state
  const debugIOSUpdateState = useCallback(async () => {
    if (Platform.OS !== 'ios') {
      console.log('This function is only for iOS devices');
      return;
    }
    
    console.log('🍎 iOS Update State Debug:');
    console.log('  - Platform:', Platform.OS);
    console.log('  - shouldShowUpdate:', shouldShowUpdate);
    console.log('  - showUpdateModal:', showUpdateModal);
    console.log('  - remindLaterClicked:', remindLaterClicked);
    console.log('  - latestVersionFromFirebase:', latestVersionFromFirebase);
    
    // Get ACTUAL device version
    const actualVersion = await DeviceInfo.getVersion();
    const actualBuildNumber = await DeviceInfo.getBuildNumber();
    const actualVersionWithBuild = `${actualVersion} (${actualBuildNumber})`;
    console.log('  - ACTUAL device version:', actualVersionWithBuild);
    console.log('  - Version source: DYNAMIC FROM BUILD FILES');
    console.log('  - Android: versionName from build.gradle');
    console.log('  - iOS: CFBundleShortVersionString from Info.plist');
    
    // Check if update should be shown using semantic comparison
    const needsUpdate = compareVersions(latestVersionFromFirebase, actualVersion) > 0;
    
    console.log('  - Version comparison:');
    console.log('    - ACTUAL device version:', actualVersionWithBuild);
    console.log('    - Latest Firebase version:', latestVersionFromFirebase);
    console.log('    - Needs update (semantic):', needsUpdate);
    console.log('    - Should show update:', shouldShowUpdate);
    console.log('    - Modal visible:', showUpdateModal);
    console.log('    - Comparison method: ACTUAL DEVICE vs Firebase');
    console.log('    - Version source: DYNAMIC FROM BUILD FILES');
    
    showToast('info', 'iOS Debug', 'Check console for detailed state info');
  }, [shouldShowUpdate, showUpdateModal, remindLaterClicked, latestVersionFromFirebase, compareVersions]);

  // Simple iOS test function to force show update modal
  const forceShowIOSModal = useCallback(() => {
    if (Platform.OS !== 'ios') {
      console.log('This function is only for iOS devices');
      return;
    }
    
    console.log('🍎 iOS Force Show Modal - Manually triggering update modal');
    
    // Force set all the required states
    setLatestVersionFromFirebase('1.0.3');
    setShouldShowUpdate(true);
    setShowUpdateModal(true);
    setRemindLaterClicked(false);
    
    console.log('🍎 iOS Force Show Modal:');
    console.log('  - latestVersionFromFirebase: 1.0.3');
    console.log('  - shouldShowUpdate: true');
    console.log('  - showUpdateModal: true');
    console.log('  - remindLaterClicked: false');
    
    showToast('success', 'iOS Modal', 'Update modal forced to show');
  }, []);

  // REMOVED: iOS aggressive modal enforcement effects - causing auto-close/open issue

  // Comprehensive iOS debug function
  const comprehensiveIOSDebug = useCallback(async () => {
    if (Platform.OS !== 'ios') {
      console.log('This function is only for iOS devices');
      return;
    }
    
    console.log('🔍 COMPREHENSIVE iOS DEBUG - All current states:');
    console.log('  - Platform:', Platform.OS);
    console.log('  - shouldShowUpdate:', shouldShowUpdate);
    console.log('  - showUpdateModal:', showUpdateModal);
    console.log('  - remindLaterClicked:', remindLaterClicked);
    console.log('  - latestVersionFromFirebase:', latestVersionFromFirebase);
    console.log('  - isMounted.current:', isMounted.current);
    console.log('  - dataLoaded:', dataLoaded);
    console.log('  - isOnline:', isOnline);
    
    // Get ACTUAL device version
    const actualVersion = await DeviceInfo.getVersion();
    const actualBuildNumber = await DeviceInfo.getBuildNumber();
    const actualVersionWithBuild = `${actualVersion} (${actualBuildNumber})`;
    console.log('  - ACTUAL device version:', actualVersionWithBuild);
    console.log('  - Version source: DYNAMIC FROM BUILD FILES');
    console.log('  - Android: versionName from build.gradle');
    console.log('  - iOS: CFBundleShortVersionString from Info.plist');
    
    // Check if update should be shown using semantic comparison
    const needsUpdate = compareVersions(latestVersionFromFirebase, actualVersion) > 0;
    console.log('  - Version comparison logic:');
    console.log('    - ACTUAL device version:', actualVersionWithBuild);
    console.log('    - Comparison method: ACTUAL DEVICE vs Firebase');
    console.log('    - Version source: DYNAMIC FROM BUILD FILES');
    
    // Check if there are any conflicting states
    if (shouldShowUpdate && !showUpdateModal) {
      console.log('🚨 PROBLEM DETECTED: shouldShowUpdate is true but showUpdateModal is false');
      console.log('🚨 This means the modal should be showing but is not visible');
    }
    
    if (remindLaterClicked && shouldShowUpdate) {
      console.log('🚨 PROBLEM DETECTED: remindLaterClicked is true but shouldShowUpdate is also true');
      console.log('🚨 This might prevent the modal from showing');
    }
    
    showToast('info', 'iOS Debug', 'Check console for comprehensive state info');
  }, [shouldShowUpdate, showUpdateModal, remindLaterClicked, latestVersionFromFirebase, dataLoaded, isOnline]);

  // Comprehensive Android debug function
  const comprehensiveAndroidDebug = useCallback(async () => {
    if (Platform.OS !== 'android') {
      console.log('This function is only for Android devices');
      return;
    }
    
    console.log('🔍 COMPREHENSIVE ANDROID DEBUG - All current states:');
    console.log('  - Platform:', Platform.OS);
    console.log('  - shouldShowUpdate:', shouldShowUpdate);
    console.log('  - showUpdateModal:', showUpdateModal);
    console.log('  - remindLaterClicked:', remindLaterClicked);
    console.log('  - latestVersionFromFirebase:', latestVersionFromFirebase);
    console.log('  - isMounted.current:', isMounted.current);
    console.log('  - dataLoaded:', dataLoaded);
    console.log('  - isOnline:', isOnline);
    
    // Get ACTUAL device version
    const actualVersion = await DeviceInfo.getVersion();
    const actualBuildNumber = await DeviceInfo.getBuildNumber();
    const actualVersionWithBuild = `${actualVersion} (${actualBuildNumber})`;
    console.log('  - ACTUAL device version:', actualVersionWithBuild);
    console.log('  - Version source: DYNAMIC FROM BUILD FILES');
    console.log('  - Android: versionName from build.gradle');
    console.log('  - iOS: CFBundleShortVersionString from Info.plist');
    
    // Check if update should be shown using semantic comparison
    const needsUpdate = compareVersions(latestVersionFromFirebase, actualVersion) > 0;
    console.log('  - Version comparison logic:');
    console.log('    - ACTUAL device version:', actualVersionWithBuild);
    console.log('    - Latest Firebase version:', latestVersionFromFirebase);
    console.log('    - Needs update (semantic):', needsUpdate);
    console.log('    - Should show update (state):', shouldShowUpdate);
    console.log('    - Modal visible (state):', showUpdateModal);
    console.log('    - Comparison method: ACTUAL DEVICE vs Firebase');
    console.log('    - Version source: DYNAMIC FROM BUILD FILES');
    
    // Check if there are any conflicting states
    if (shouldShowUpdate && !showUpdateModal) {
      console.log('🚨 PROBLEM DETECTED: shouldShowUpdate is true but showUpdateModal is false');
      console.log('🚨 This means the modal should be showing but is not visible');
    }
    
    if (remindLaterClicked && shouldShowUpdate) {
      console.log('🚨 PROBLEM DETECTED: remindLaterClicked is true but shouldShowUpdate is also true');
      console.log('🚨 This might prevent the modal from showing');
    }
    
    showToast('info', 'Android Debug', 'Check console for comprehensive state info');
  }, [shouldShowUpdate, showUpdateModal, remindLaterClicked, latestVersionFromFirebase, dataLoaded, isOnline]);

  // iOS-specific force modal show function (bypasses all state checks)
  const forceIOSModalShow = useCallback(() => {
    if (Platform.OS !== 'ios') {
      console.log('This function is only for iOS devices');
      return;
    }
    
    console.log('🚀 iOS FORCE MODAL SHOW - Bypassing all state checks');
    
    // Force all states to show modal
    setLatestVersionFromFirebase('1.0.5');
    setShouldShowUpdate(true);
    setShowUpdateModal(true);
    setRemindLaterClicked(false);
    
    // Force a re-render multiple times to ensure it works
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🚀 iOS FORCE MODAL SHOW - First re-render');
        setShowUpdateModal(true);
      }
    }, 100);
    
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🚀 iOS FORCE MODAL SHOW - Second re-render');
        setShowUpdateModal(true);
      }
    }, 200);
    
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🚀 iOS FORCE MODAL SHOW - Third re-render');
        setShowUpdateModal(true);
        showToast('success', 'Force Show', 'Modal should be visible now!');
      }
    }, 300);
    
    console.log('🚀 iOS FORCE MODAL SHOW Results:');
    console.log('  - latestVersionFromFirebase: 1.0.5');
    console.log('  - shouldShowUpdate: true');
    console.log('  - showUpdateModal: true');
    console.log('  - remindLaterClicked: false');
    console.log('  - Platform: iOS');
    
    showToast('info', 'Force Show', 'All states forced, modal should show!');
  }, []);

  // Android-specific force modal show function (bypasses all state checks)
  const forceAndroidModalShow = useCallback(() => {
    if (Platform.OS !== 'android') {
      console.log('This function is only for Android devices');
      return;
    }
    
    console.log('🤖 ANDROID FORCE MODAL SHOW - Bypassing all state checks');
    
    // Force all states to show modal
    setLatestVersionFromFirebase('2.3.0');
    setShouldShowUpdate(true);
    setShowUpdateModal(true);
    setRemindLaterClicked(false);
    
    // Force a re-render multiple times to ensure it works
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🤖 ANDROID FORCE MODAL SHOW - First re-render');
        setShowUpdateModal(true);
      }
    }, 100);
    
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🤖 ANDROID FORCE MODAL SHOW - Second re-render');
        setShowUpdateModal(true);
      }
    }, 200);
    
    setTimeout(() => {
      if (isMounted.current) {
        console.log('🤖 ANDROID FORCE MODAL SHOW - Third re-render');
        setShowUpdateModal(true);
        showToast('success', 'Force Show', 'Modal should be visible now!');
      }
    }, 300);
    
    console.log('🤖 ANDROID FORCE MODAL SHOW Results:');
    console.log('  - latestVersionFromFirebase: 2.3.0');
    console.log('  - shouldShowUpdate: true');
    console.log('  - showUpdateModal: true');
    console.log('  - remindLaterClicked: false');
    console.log('  - Platform: Android');
    
    showToast('info', 'Force Show', 'All states forced, modal should show!');
  }, []);

  // Protection mechanism to prevent modal from being hidden on iOS
  useEffect(() => {
    if (Platform.OS === 'ios' && shouldShowUpdate && showUpdateModal) {
      console.log('🛡️ iOS Modal Protection - Modal is visible, protecting it from being hidden');
      
      // Set up a protection interval
      const protectionInterval = setInterval(() => {
        if (isMounted.current && shouldShowUpdate && !showUpdateModal) {
          console.log('🛡️ iOS Modal Protection - Modal was hidden, restoring it immediately');
          setShowUpdateModal(true);
          showToast('warning', 'Modal Protected', 'Modal was hidden, restored immediately');
        }
      }, 100); // Check every 100ms
      
      return () => clearInterval(protectionInterval);
    }
  }, [Platform.OS, shouldShowUpdate, showUpdateModal]);

  // Protection mechanism to prevent modal from being hidden on Android
  useEffect(() => {
    if (Platform.OS === 'android' && shouldShowUpdate && showUpdateModal) {
      console.log('🛡️ Android Modal Protection - Modal is visible, protecting it from being hidden');
      
      // Set up a protection interval
      const protectionInterval = setInterval(() => {
        if (isMounted.current && shouldShowUpdate && !showUpdateModal) {
          console.log('🛡️ Android Modal Protection - Modal was hidden, restoring it immediately');
          setShowUpdateModal(true);
          showToast('warning', 'Modal Protected', 'Modal was hidden, restored immediately');
        }
      }, 100); // Check every 100ms
      
      return () => clearInterval(protectionInterval);
    }
  }, [Platform.OS, shouldShowUpdate, showUpdateModal]);

    // REMOVED: iOS Auto-Trigger - Only check once when screen loads

  // REMOVED: Android Auto-Trigger - Only check once when screen loads

  // OPTIMIZED: Memoized country dropdown - Always call useMemo to maintain hook order
  const CountryDropdown = useMemo(() => (
    <TouchableOpacity
      style={{
        borderWidth: 2,
        borderColor: '#035484',
        paddingHorizontal: 7,
        paddingVertical: 3,
        minWidth: '40%',
        backgroundColor: '#fff',
        flexDirection: 'row',
        alignItems: 'center',
        justifyContent: 'space-between',
        shadowColor: '#000',
        shadowOpacity: 0.08,
        shadowRadius: 4,
        shadowOffset: { width: 0, height: 2 },
        elevation: 2,
        marginBottom: 10,
      }}
      onPress={() => setModalVisible(true)}
      activeOpacity={0.8}
    >
      <Text style={{ color: selectedCountries.length > 0 ? '#222' : '#888', fontSize: 16, flex: 1 }}>
        {selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')
          ? 'Both Countries'
          : selectedCountries.length > 0
            ? selectedCountries.join(', ')
            : 'Choose Country'}
      </Text>
      <MaterialIcons name="keyboard-arrow-down" size={28} color="#035484" />
    </TouchableOpacity>
  ), [selectedCountries]);

  // OPTIMIZED: Memoized country modal - Always call useMemo to maintain hook order
  const CountryModal = useMemo(() => (
    <Modal
      animationType="fade"
      transparent={true}
      visible={modalVisible}
      onRequestClose={() => setModalVisible(false)}
    >
      <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.10)', justifyContent: 'center', alignItems: 'center' }}>
        <View style={{ backgroundColor: '#fff', borderRadius: 14, width: 320, paddingBottom: 18, shadowColor: '#000', shadowOpacity: 0.12, shadowRadius: 12, shadowOffset: { width: 0, height: 4 }, elevation: 8 }}>
          <View style={{ backgroundColor: '#F6F7F9', borderTopLeftRadius: 14, borderTopRightRadius: 14, paddingVertical: 16, alignItems: 'center', borderBottomWidth: 1, borderBottomColor: '#eee' }}>
            <Text style={{ color: '#035484', fontWeight: 'bold', fontSize: 20 }}>Choose Countries</Text>
          </View>
          {memoizedCountryOptions.map((option, idx) => (
            <TouchableOpacity
              key={option}
              style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingVertical: 16, paddingHorizontal: 22, borderBottomWidth: idx !== memoizedCountryOptions.length - 1 ? 1 : 0, borderBottomColor: '#eee', backgroundColor: (option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? '#F6F7F9' : '#fff' }}
              onPress={() => handleToggleCountry(option)}
              activeOpacity={0.8}
            >
              <Text style={{ color: (option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? '#035484' : '#222', fontWeight: (option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? 'bold' : 'normal', fontSize: 18 }}>{option}</Text>
              <View style={{ width: 28, height: 28, borderRadius: 14, borderWidth: 2, borderColor: '#035484', alignItems: 'center', justifyContent: 'center', backgroundColor: '#fff' }}>
                {(option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? (
                  <View style={{ width: 16, height: 16, borderRadius: 8, backgroundColor: '#035484' }} />
                ) : null}
              </View>
            </TouchableOpacity>
          ))}
          <View style={{ flexDirection: 'row', justifyContent: 'center', marginTop: 10 }}>
            <TouchableOpacity
              style={{ backgroundColor: '#035484', borderRadius: 6, paddingVertical: 8, paddingHorizontal: 36, marginTop: 6, marginRight: 10 }}
              onPress={handleSaveCountries}
              activeOpacity={0.8}
            >
              <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Save</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={{ backgroundColor: '#888', borderRadius: 6, paddingVertical: 8, paddingHorizontal: 36, marginTop: 6 }}
              onPress={() => setModalVisible(false)}
              activeOpacity={0.8}
            >
              <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </Modal>
  ), [modalVisible, selectedCountries, memoizedCountryOptions, handleToggleCountry, handleSaveCountries]);

  // OPTIMIZED: Memoized HR section - Always call useMemo to maintain hook order
  const HRSection = useMemo(() => {
   
    const boxSize = Math.floor((width - 5 * 16) / 4);
    const items = [
      {
        icon: require('../../Images/adminhome/icosn/hrform01.png'),
        label: 'FirebaseFCMNotificaion',
        onPress: () => handlePress(() => navigation.navigate('FirebaseFCMNotificaion'), 'FirebaseFCMNotificaion'),
      },
    ];
    
    return (
      <>
        <Text style={styles.heading}>HR Apply Application</Text>
        <View style={{
          flexDirection: 'row',
          justifyContent: 'center',
          alignItems: 'flex-start',
          marginBottom: 18
        }}>
          {items.map((item, idx) => (
            <TouchableOpacity
              key={item.label}
              style={{ alignItems: 'center', marginHorizontal: 8 }}
              onPress={item.onPress}
              activeOpacity={0.8}
            >
              <View style={{
                backgroundColor: '#035484',
                borderRadius: 12,
                width: boxSize,
                height: boxSize,
                justifyContent: 'center',
                alignItems: 'center',
                marginBottom: 8
              }}>
                <Image source={item.icon} style={{ width: 48, height: 48, resizeMode: 'contain' }} />
              </View>
              <Text style={{
                color: '#29486A',
                fontWeight: 'bold',
                fontSize: 13,
                textAlign: 'center',
                width: boxSize
              }}>{item.label}</Text>
            </TouchableOpacity>
          ))}
        </View>
      </>
    );
  }, [navigation, handlePress, width]);

  // Conditional rendering based on loading state
  if (loading || !dataLoaded) {
    return <Skeleton />;
  }

  // If no modules loaded yet, show skeleton
  if (modules.length === 0) {
    return <Skeleton />;
  }

  return (
    <>
      <ScrollView style={styles.scrollContainer}>
        <View style={styles.container}>
          {/* Top bar with country dropdown */}
          {CountryDropdown}

          {/* Country Selection Modal (Multi-select) */}
          {CountryModal}

          {/* SyncModal for specific operations */}
          <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message={syncModalMessage} />

          {/* Version Update Modal - Platform Specific */}
          {Platform.OS === 'ios' ? (
            <>
              {console.log('🍎 RENDERING iOS MODAL - Platform:', Platform.OS, 'showUpdateModal:', showUpdateModal)}
              <IOSUpdateModal
                visible={showUpdateModal}
                currentVersion={actualDeviceVersion}
                latestVersion={latestVersionFromFirebase}
                updateUrl={updateUrl}
                onRemindLater={handleRemindLater}
                onClose={() => {
                  console.log(`🍎 iOS Modal closed`);
                  console.log(`🍎 iOS: shouldShowUpdate: ${shouldShowUpdate}, remindLaterClicked: ${remindLaterClicked}`);
                  
                  setShowUpdateModal(false);
                  // NO AUTO-SHOW - Modal will only show again when manually triggered
                }}
              />
            </>
          ) : (
            <>
              {console.log('🤖 RENDERING ANDROID MODAL - Platform:', Platform.OS, 'showUpdateModal:', showUpdateModal)}
              <VersionUpdateModal
                visible={showUpdateModal}
                currentVersion={actualDeviceVersion}
                latestVersion={latestVersionFromFirebase}
                updateUrl={updateUrl}
                onRemindLater={handleAndroidRemindLater}
                onClose={() => {
                  console.log(`🤖 Android Modal closed`);
                  console.log(`🤖 Android: shouldShowUpdate: ${shouldShowUpdate}, remindLaterClicked: ${remindLaterClicked}`);
                  
                  setShowUpdateModal(false);
                  // NO AUTO-SHOW - Modal will only show again when manually triggered
                }}
              />
            </>
          )}
          


          {/* Render Categories Dynamically */}
          {renderCategory('Master Data', 'masterdatakpi')}
          {renderCategory('Inspection and Compliances', 'InspectionKpi')}
          {renderCategory('Maintenance & Premium Services', 'MaintenanceKpiHome')}
          {renderCategory('Logbook', 'LogbookKPI')}
          {renderCategory('HR', 'HrModuleKpI')}

          {/* Quick Access: Speech to Text button (SpeetchTotext) */}
          <View style={{ width: '100%', alignItems: 'center', marginTop: 12, marginBottom: 18 }}>
            <TouchableOpacity
              style={[styles.initializeButton, { flexDirection: 'row', alignItems: 'center', paddingHorizontal: 18 }]}
              onPress={() => navigation.navigate('SpeechToText')}
              activeOpacity={0.8}
            >
              <MaterialIcons name="keyboard-voice" size={20} color="#fff" />
              <Text style={[styles.initializeButtonText, { marginLeft: 10, fontWeight: '700' }]}>SpeetchTotext</Text>
            </TouchableOpacity>
          </View>

          {/* Equipment MongoDB Upload button */}
          <View style={{ width: '100%', alignItems: 'center', marginTop: 12, marginBottom: 18 }}>
            <TouchableOpacity
              style={[styles.initializeButton, { flexDirection: 'row', alignItems: 'center', paddingHorizontal: 18, backgroundColor: '#0055A5' }]}
              onPress={() => navigation.navigate('UploadEquipmentToMongo')}
              activeOpacity={0.8}
            >
              <MaterialIcons name="cloud-upload" size={20} color="#fff" />
              <Text style={[styles.initializeButtonText, { marginLeft: 10, fontWeight: '700' }]}>Equipment Mongo</Text>
            </TouchableOpacity>
          </View>

          {/* Sixth Heading */}
          {/* {HRSection} */}
        </View>
      </ScrollView>
      <Toast />
    </>
  );
});

const styles = StyleSheet.create({
  scrollContainer: {
    flex: 1,
    backgroundColor: '#fff',
    width: '100%',
  },
  container: {
    paddingHorizontal: 5,
    paddingVertical: 10,
    alignItems: 'center',
    width: '100%',
  },
  heading: {
    fontSize: 24,
    fontWeight: 'bold',
    textAlign: 'center',
    marginBottom: 10,
    color: '#015185',
    alignSelf: 'center',
  },
  buttonContainer: {
    alignItems: 'center',
    marginVertical: 10,
    width: '100%',
  },
  gradientButton: {
    width: 160,
    borderRadius: 5,
    overflow: 'hidden',
    alignSelf: 'center',
  },
  gradientButtonContent: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    height: 40,
  },
  buttonText: {
    color: 'white',
    fontSize: 16,
    fontWeight: 'bold',
    marginRight: 5,
    textAlign: 'center',
    alignSelf: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  versionModalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.7)',
  },
  versionModalContainer: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 15,
    overflow: 'hidden',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
    elevation: 5,
  },
  versionModalHeader: {
    paddingVertical: 20,
    alignItems: 'center',
  },
  versionModalHeaderContent: {
    alignItems: 'center',
  },
  versionModalTitle: {
    color: '#fff',
    fontSize: 22,
    fontWeight: 'bold',
    marginTop: 10,
  },
  versionModalSubtitle: {
    color: '#fff',
    fontSize: 14,
    marginTop: 5,
  },
  versionModalBody: {
    padding: 20,
    paddingTop: 0,
  },
  versionInfoContainer: {
    marginBottom: 15,
  },
  versionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 5,
  },
  versionLabel: {
    color: '#555',
    fontSize: 14,
  },
  versionValue: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#333',
  },
  versionValueLatest: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#035484',
  },
  updateFeaturesContainer: {
    marginBottom: 20,
  },
  updateFeaturesTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    marginBottom: 10,
    color: '#333',
  },
  featureItem: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  featureText: {
    marginLeft: 8,
    fontSize: 14,
    color: '#555',
  },
  updateButton: {
    width: '100%',
    borderRadius: 2,
    overflow: 'hidden',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 12,
    marginBottom: 10,
  },
  updateButtonGradient: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    borderRadius: 4,
    padding:6
  },
  updateButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 8,
  },
  remindLaterButton: {
    width: '100%',
    borderRadius: 2,
    overflow: 'hidden',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 12,
  },
  remindLaterText: {
    color: '#035484',
    fontSize: 16,
    fontWeight: 'bold',
  },

  initializeButton: {
    backgroundColor: '#035484',
    borderRadius: 6,
    paddingVertical: 8,
    paddingHorizontal: 15,
    marginLeft: 10,
  },
  initializeButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  testButton: {
    backgroundColor: '#4CAF50',
    borderRadius: 6,
    paddingVertical: 8,
    paddingHorizontal: 15,
    marginLeft: 10,
  },
  testButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  testVersionButton: {
    backgroundColor: '#FF9800',
    borderRadius: 6,
    paddingVertical: 8,
    paddingHorizontal: 15,
    marginLeft: 10,
  },
  testVersionButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  latestVersionContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  loadingContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  loadingText: {
    fontSize: 14,
    color: '#666',
    fontStyle: 'italic',
  },
  refreshButton: {
    marginLeft: 10,
    padding: 5,
  },
  versionSource: {
    fontSize: 14,
    color: '#035484',
    fontWeight: '600',
  },
  
  // iOS Modal Styles
  iosModalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.7)',
  },
  iosModalContainer: {
    width: '85%',
    backgroundColor: '#fff',
    borderRadius: 20,
    overflow: 'hidden',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
    elevation: 8,
  },
  iosModalHeader: {
    paddingVertical: 25,
    alignItems: 'center',
  },
  iosModalHeaderContent: {
    alignItems: 'center',
  },
  iosModalTitle: {
    color: '#fff',
    fontSize: 24,
    fontWeight: 'bold',
    marginTop: 12,
  },
  iosModalSubtitle: {
    color: '#fff',
    fontSize: 16,
    marginTop: 6,
  },
  iosModalBody: {
    padding: 25,
    paddingTop: 0,
  },
  iosVersionInfoContainer: {
    marginBottom: 20,
  },
  iosVersionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 8,
    alignItems: 'center',
  },
  iosVersionLabel: {
    color: '#555',
    fontSize: 15,
    fontWeight: '500',
  },
  iosVersionValue: {
    fontSize: 17,
    fontWeight: 'bold',
    color: '#333',
  },
  iosVersionValueLatest: {
    fontSize: 17,
    fontWeight: 'bold',
    color: '#FF6B35',
  },
  iosLatestVersionContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  iosLoadingContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  iosLoadingText: {
    fontSize: 15,
    color: '#666',
    fontStyle: 'italic',
  },
  iosRefreshButton: {
    marginLeft: 12,
    padding: 6,
  },
  iosVersionSource: {
    fontSize: 15,
    color: '#FF6B35',
    fontWeight: '600',
  },
  iosUpdateFeaturesContainer: {
    marginBottom: 25,
  },
  iosUpdateFeaturesTitle: {
    fontSize: 19,
    fontWeight: 'bold',
    marginBottom: 12,
    color: '#333',
  },
  iosFeatureItem: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 10,
  },
  iosFeatureText: {
    marginLeft: 10,
    fontSize: 15,
    color: '#555',
  },
  iosModalActions: {
    gap: 12,
  },
  iosUpdateButton: {
    width: '100%',
    borderRadius: 8,
    overflow: 'hidden',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 15,
    marginBottom: 8,
  },
  iosUpdateButtonGradient: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    borderRadius: 8,
    padding: 8,
  },
  iosUpdateButtonText: {
    color: '#fff',
    fontSize: 18,
    fontWeight: 'bold',
    marginLeft: 10,
  },
  iosRemindLaterButton: {
    width: '100%',
    borderRadius: 8,
    overflow: 'hidden',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 15,
  },
  iosRemindLaterText: {
    color: '#FF6B35',
    fontSize: 17,
    fontWeight: 'bold',
  },
  

  
  
});

export default HomeScreen;