import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  ScrollView,
  RefreshControl,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import FontAwesome from 'react-native-vector-icons/FontAwesome';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import { MongoAPI } from '../../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.33;

// MongoDB API URL
const MONGODB_HRFORM_API_URL = 'https://api.titandrillingzm.com:6013';

// Helper function to show toast messages
const showToast = (type, title, message) => {
  Toast.show({
    type: type,
    text1: title,
    text2: message,
    position: 'bottom',
  });
};

const AdminHrDocument = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { updatedApplication } = route.params || {};

  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [applications, setApplications] = useState([]);
  const [filteredApplications, setFilteredApplications] = useState([]);
  const [searchQuery, setSearchQuery] = useState('');
  const [userUid, setUserUid] = useState('');
  const [selectedTab, setSelectedTab] = useState('All');
  const [isLoading, setIsLoading] = useState(false);
  
  // Permission states
  const [userCountries, setUserCountries] = useState([]);
  const [userProjects, setUserProjects] = useState([]);
  const [showPermissionModal, setShowPermissionModal] = useState(false);
  const [onlyMineView, setOnlyMineView] = useState(false);

  // Tab definitions
  const tabOptions = [
    { key: 'All', label: 'All' },
    { key: 'leave_application', label: 'Leave' },
    { key: 'funeral_grant_form', label: 'Funeral' },
    { key: 'salary_advance_application', label: 'Salary Advance' },
    { key: 'application_for_appeal', label: 'Appeal' },
    { key: 'grievance_record', label: 'Grievance' },
    { key: 'hospital_attendance_form', label: 'Hospital Attendance' },
  ];

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const [isRotating, setIsRotating] = useState(false);

  // Fetch user permissions from MongoDB
  const fetchUserPermissions = async (userUidVal) => {
    try {
      console.log('🔍 Fetching user permissions from MongoDB for UID:', userUidVal);
      const response = await MongoAPI.getUserById(userUidVal);
      
      if (response.success && response.data) {
        const userData = response.data;
        const countries = Array.isArray(userData.countries) ? userData.countries : [];
        const projects = Array.isArray(userData.projects) ? userData.projects : [];
        
        setUserCountries(countries);
        setUserProjects(projects);
        
        console.log('✅ User permissions loaded:', { countries, projects });
        return { countries, projects };
      } else {
        showToast('warning', 'Warning', 'Could not load user permissions.');
        return { countries: [], projects: [] };
      }
    } catch (error) {
      console.error('❌ Error fetching user permissions:', error);
      return { countries: [], projects: [] };
    }
  };

  // Fetch all HR applications from MongoDB
  const fetchApplications = async () => {
    setIsLoading(true);
    try {
      const userUidVal = await AsyncStorage.getItem('userUid');
      setUserUid(userUidVal);
      
      if (!userUidVal) {
        showToast('error', 'Error', 'Admin not authenticated.');
        setIsLoading(false);
        return;
      }

      // Fetch user permissions
      const { countries, projects } = await fetchUserPermissions(userUidVal);

      console.log('📋 Fetching HR applications from MongoDB...');

      const applicationTypes = [
        'leave_application',
        'funeral_grant_form',
        'salary_advance_application',
        'application_for_appeal',
        'grievance_record',
        'hospital_attendance_form',
      ];

      const allApplications = [];

      // Fetch applications for each type
      for (const type of applicationTypes) {
        try {
          const params = new URLSearchParams({
            countries: countries.join(','),
            projects: projects.join(','),
            onlyMineView: onlyMineView.toString(),
          });

          const response = await fetch(
            `${MONGODB_HRFORM_API_URL}/hr-applications/${type}/${userUidVal}?${params}`
          );

          if (response.ok) {
            const result = await response.json();
            if (result.success && result.data) {
              allApplications.push(...result.data);
            }
          }
        } catch (error) {
          console.error(`❌ Error fetching ${type}:`, error);
        }
      }

      console.log(`✅ Loaded ${allApplications.length} applications from MongoDB`);
      
      // Debug: Log application types to see what's coming from database
      const foundApplicationTypes = [...new Set(allApplications.map(app => app.applicationType))];
      console.log('🔍 Application types found:', foundApplicationTypes);

      setApplications(allApplications);
      setFilteredApplications(allApplications);
      setIsLoading(false);
    } catch (error) {
      console.error('❌ Error fetching applications:', error);
      showToast('error', 'Error', 'Failed to load applications.');
      setIsLoading(false);
    }
  };

  // Initial fetch
  useEffect(() => {
    fetchApplications();
  }, []);

  // Refetch when onlyMineView changes
  useEffect(() => {
    if (userUid) {
      fetchApplications();
    }
  }, [onlyMineView]);

  // Update applications list when an application is updated
  useEffect(() => {
    if (updatedApplication) {
      setApplications((prevApplications) =>
        prevApplications.map((app) =>
          app.applicationId === updatedApplication.applicationId &&
          app.teamMemberId === updatedApplication.teamMemberId
            ? updatedApplication
            : app
        )
      );
      setFilteredApplications((prevFiltered) =>
        prevFiltered.map((app) =>
          app.applicationId === updatedApplication.applicationId &&
          app.teamMemberId === updatedApplication.teamMemberId
            ? updatedApplication
            : app
        )
      );
    }
  }, [updatedApplication]);

  // Search filter and tab filter
  useEffect(() => {
    let filtered = applications;
    
    // Filter by tab
    if (selectedTab !== 'All') {
      filtered = filtered.filter(app => app.applicationType === selectedTab);
    }
    
    // Filter by search query
    if (searchQuery) {
      filtered = filtered.filter((app) => {
        const fullName = app.fullName || app.applicantName || '';
        const employeeNo = app.employeeNo || app.employeeNumber || '';
        const appType = app.applicationType || '';
        
        return (
          fullName.toLowerCase().includes(searchQuery.toLowerCase()) ||
          employeeNo.toLowerCase().includes(searchQuery.toLowerCase()) ||
          appType.toLowerCase().includes(searchQuery.toLowerCase())
        );
      });
    }
    
    setFilteredApplications(filtered);
  }, [searchQuery, applications, selectedTab]);

  // Refresh handler
  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    fetchApplications().then(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      showToast('success', 'Success', 'Data synced successfully.');
    });
  };

  // Start/stop rotation animation for loader and sync modal
  useEffect(() => {
    if (isLoading || syncModalVisible) {
      if (!isRotating) {
        rotateAnim.setValue(0);
        setIsRotating(true);
        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 1000,
            useNativeDriver: true,
          })
        ).start();
      }
    } else {
      setIsRotating(false);
      rotateAnim.stopAnimation();
      rotateAnim.setValue(0);
    }
  }, [isLoading, syncModalVisible]);

  // Format application type for display
  const formatApplicationType = (type) => {
    if (!type) return 'Unknown Application';
    
    switch (type) {
      case 'leave_application':
        return 'Leave Application';
      case 'funeral_grant_form':
        return 'Funeral Grant Form';
      case 'salary_advance_application':
        return 'Salary Advance Application';
      case 'application_for_appeal':
        return 'Application for Appeal';
      case 'grievance_record':
        return 'Grievance Record';
      case 'hospital_attendance_form':
        return 'Hospital Attendance Form';
      default:
        // If type exists but not in switch, format it nicely
        return type.split('_').map(word => 
          word.charAt(0).toUpperCase() + word.slice(1)
        ).join(' ');
    }
  };

  // Helper function to format dates
  const formatDate = (dateString) => {
    if (!dateString || dateString === 'N/A') return 'N/A';
    const date = new Date(dateString);
    return isNaN(date.getTime()) ? 'N/A' : date.toLocaleString();
  };

  // Get status icon and color
  const getStatusIcon = (status) => {
    switch (status) {
      case 'Approved':
        return { icon: 'check-circle', color: '#4CAF50' };
      case 'Rejected':
        return { icon: 'times-circle', color: '#F44336' };
      case 'Pending':
      default:
        return { icon: 'clock-o', color: '#FFA500' };
    }
  };

  // Filter applications by status
  const pendingApplications = filteredApplications.filter((app) => app.status === 'Pending');
  const approvedApplications = filteredApplications.filter((app) => app.status === 'Approved' || app.status === 'Rejected');

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>HR Application</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity 
              style={styles.iconButton} 
              onPress={() => setShowPermissionModal(true)}
              accessibilityLabel="View permissions"
            >
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity style={styles.iconButton} onPress={handleRefresh}>
              <Image
                source={require('../../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      {/* Search Bar */}
      <View style={styles.searchContainer}>
        <View style={styles.searchWrapper}>
          <Ionicons name="search" size={width * 0.05} color="#888" />
          <TextInput
            style={styles.searchInput}
            placeholder="Search by name, employee no, or application type"
            placeholderTextColor="#888"
            value={searchQuery}
            onChangeText={setSearchQuery}
            accessibilityLabel="Search applications"
          />
        </View>
        {Platform.OS === 'ios' ? (
          <TouchableOpacity
            style={styles.iosApplyNowButton}
            onPress={() => navigation.navigate('AdminHrAppliApply')}
            accessibilityLabel="Apply Now"
          >
            <Text style={styles.iosApplyNowText}>Apply Now</Text>
          </TouchableOpacity>
        ) : (
          <TouchableOpacity
            style={styles.applyNowButton}
            onPress={() => navigation.navigate('AdminHrAppliApply')}
            accessibilityLabel="Apply Now"
          >
            <LinearGradient
              colors={['#025383', '#3481BC']}
              style={styles.applyNowGradient}
              start={{ x: 0, y: 0 }}
              end={{ x: 1, y: 0 }}
            >
              <Text style={styles.applyNowText}>Apply Now</Text>
            </LinearGradient>
          </TouchableOpacity>
        )}
      </View>

      {/* Sticky Horizontal Tab Bar */}
      <View style={styles.stickyTabBarWrapper}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          style={styles.tabBar}
          contentContainerStyle={{ paddingHorizontal: width * 0.04, alignItems: 'center' }}
        >
          {/* Mine Toggle Button */}
          <TouchableOpacity
            style={[
              styles.mineToggleButton,
              onlyMineView && styles.mineToggleButtonActive,
            ]}
            onPress={() => setOnlyMineView(!onlyMineView)}
          >
            <MaterialIcons 
              name={onlyMineView ? "person" : "person-outline"} 
              size={width * 0.045} 
              color={onlyMineView ? "#fff" : "#666"} 
            />
            <Text
              style={[
                styles.mineToggleText,
                onlyMineView && styles.mineToggleTextActive,
              ]}
            >
              Mine
            </Text>
          </TouchableOpacity>
          
          {tabOptions.map((tab) => (
            <TouchableOpacity
              key={tab.key}
              style={[
                styles.tabButton,
                selectedTab === tab.key && styles.tabButtonActive,
              ]}
              onPress={() => setSelectedTab(tab.key)}
            >
              <Text
                style={[
                  styles.tabButtonText,
                  selectedTab === tab.key && styles.tabButtonTextActive,
                ]}
              >
                {tab.label}
              </Text>
            </TouchableOpacity>
          ))}
        </ScrollView>
      </View>

      {/* Applications List */}
      <ScrollView
        contentContainerStyle={[styles.listContent, { paddingTop: width * 0.16 }]}
        refreshControl={
          <RefreshControl refreshing={isRefreshing} onRefresh={handleRefresh} />
        }
      >
        {/* Pending Applications */}
        <View style={styles.applicationSection}>
          <Text style={styles.sectionTitle}>Pending Applications</Text>
          {pendingApplications.length > 0 ? (
            pendingApplications.map((item) => {
              const statusIcon = getStatusIcon(item.status);
              return (
                <TouchableOpacity
                  key={`${item.id || item.applicationId}`}
                  style={styles.applicationItem}
                  onPress={() =>
                    navigation.navigate('HrDocViewApprove', {
                      application: item,
                    })
                  }
                  accessibilityLabel={`View details for ${item.fullName || item.applicantName}'s ${formatApplicationType(item.applicationType)}`}
                >
                  <View style={styles.applicationRow}>
                    <FontAwesome
                      name={statusIcon.icon}
                      size={width * 0.05}
                      color={statusIcon.color}
                      style={styles.statusIcon}
                    />
                    <View style={styles.applicationDetails}>
                      <Text style={styles.applicationText}>
                        {item.fullName || item.applicantName || 'N/A'} - {formatApplicationType(item.applicationType)}
                      </Text>
                      <Text style={styles.applicationSubText}>
                        Employee No: {item.employeeNo || item.employeeNumber || 'N/A'}
                      </Text>
                      <Text style={styles.applicationSubText}>
                        Submitted: {formatDate(item.submittedAt || item.createdAt)}
                      </Text>
                      {item.country && item.project && (
                        <Text style={styles.applicationSubText}>
                          Location: {item.country} - {item.project}
                        </Text>
                      )}
                    </View>
                  </View>
                </TouchableOpacity>
              );
            })
          ) : (
            <Text style={styles.noDataText}>No pending applications</Text>
          )}

          {/* Approved/Rejected Applications */}
          <Text style={[styles.sectionTitle, { marginTop: width * 0.05 }]}>Approved/Rejected Applications</Text>
          {approvedApplications.length > 0 ? (
            approvedApplications.map((item) => {
              const statusIcon = getStatusIcon(item.status);
              return (
                <TouchableOpacity
                  key={`${item.id || item.applicationId}`}
                  style={styles.applicationItem}
                  onPress={() =>
                    navigation.navigate('HrDocViewApprove', {
                      application: item,
                    })
                  }
                  accessibilityLabel={`View details for ${item.fullName || item.applicantName}'s ${formatApplicationType(item.applicationType)}`}
                >
                  <View style={styles.applicationRow}>
                    <FontAwesome
                      name={statusIcon.icon}
                      size={width * 0.05}
                      color={statusIcon.color}
                      style={styles.statusIcon}
                    />
                    <View style={styles.applicationDetails}>
                      <Text style={styles.applicationText}>
                        {item.fullName || item.applicantName || 'N/A'} - {formatApplicationType(item.applicationType)}
                      </Text>
                      <Text style={styles.applicationSubText}>
                        Employee No: {item.employeeNo || item.employeeNumber || 'N/A'}
                      </Text>
                      <Text style={styles.applicationSubText}>Status: {item.status}</Text>
                      <Text style={styles.applicationSubText}>
                        Approved by: {item.approvedBy || item.approverName || 'N/A'}
                      </Text>
                      <Text style={styles.applicationSubText}>
                        Date: {formatDate(item.approvedAt || item.approvalDate)}
                      </Text>
                      {item.country && item.project && (
                        <Text style={styles.applicationSubText}>
                          Location: {item.country} - {item.project}
                        </Text>
                      )}
                      {item.status === 'Rejected' && item.approvalComments && (
                        <Text style={styles.applicationSubText}>
                          Reason: {item.approvalComments || item.rejectionReason || 'N/A'}
                        </Text>
                      )}
                    </View>
                  </View>
                </TouchableOpacity>
              );
            })
          ) : (
            <Text style={styles.noDataText}>No approved/rejected applications</Text>
          )}
        </View>

        <View style={{ height: width * 0.1 }} />
      </ScrollView>

      {/* Loader/Sync Modal (unified) */}
      <Modal visible={isLoading || syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
                resizeMode="contain"
              />
              <Text style={styles.syncModalTextadd}>Loading Data from Cloud</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Permission Modal */}
      <Modal
        visible={showPermissionModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowPermissionModal(false)}
      >
        <View style={styles.permissionModalOverlay}>
          <View style={styles.permissionModalContainer}>
            <View style={styles.permissionModalHeader}>
              <MaterialIcons name="lock" size={width * 0.08} color="#025383" />
              <Text style={styles.permissionModalTitle}>Your Permissions</Text>
              <TouchableOpacity onPress={() => setShowPermissionModal(false)}>
                <MaterialIcons name="close" size={width * 0.06} color="#666" />
              </TouchableOpacity>
            </View>

            <View style={styles.permissionModalContent}>
              {/* Countries */}
              <View style={styles.permissionSection}>
                <Text style={styles.permissionLabel}>
                  <MaterialIcons name="public" size={width * 0.05} color="#025383" /> Countries
                </Text>
                {userCountries.length > 0 ? (
                  <View style={styles.permissionItemsContainer}>
                    {userCountries.map((country, index) => (
                      <View key={index} style={styles.permissionChip}>
                        <Text style={styles.permissionChipText}>{country}</Text>
                      </View>
                    ))}
                  </View>
                ) : (
                  <Text style={styles.noPermissionText}>No countries assigned</Text>
                )}
              </View>

              {/* Projects */}
              <View style={styles.permissionSection}>
                <Text style={styles.permissionLabel}>
                  <MaterialIcons name="business" size={width * 0.05} color="#025383" /> Projects
                </Text>
                {userProjects.length > 0 ? (
                  <View style={styles.permissionItemsContainer}>
                    {userProjects.map((project, index) => (
                      <View key={index} style={styles.permissionChip}>
                        <Text style={styles.permissionChipText}>{project}</Text>
                      </View>
                    ))}
                  </View>
                ) : (
                  <Text style={styles.noPermissionText}>No projects assigned</Text>
                )}
              </View>

              {/* Info */}
              <View style={styles.permissionInfoBox}>
                <MaterialIcons name="info" size={width * 0.05} color="#025383" />
                <Text style={styles.permissionInfoText}>
                  {onlyMineView 
                    ? "You're viewing only your applications"
                    : "You can view applications from your assigned countries and projects"}
                </Text>
              </View>
            </View>

            <TouchableOpacity
              style={styles.permissionCloseButton}
              onPress={() => setShowPermissionModal(false)}
            >
              <LinearGradient
                colors={['#025383', '#3481BC']}
                style={styles.permissionCloseButtonGradient}
                start={{ x: 0, y: 0 }}
                end={{ x: 1, y: 0 }}
              >
                <Text style={styles.permissionCloseButtonText}>Close</Text>
              </LinearGradient>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f0f0f0',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
    marginLeft: width * 0.02,
  },
  applyNowButton: {
    marginLeft: 10,
    borderRadius: 20,
    overflow: 'hidden',
    alignSelf: 'center',
  },
  applyNowGradient: {
    paddingVertical: 10,
    paddingHorizontal: 28,
    borderRadius: 20,
    alignItems: 'center',
    justifyContent: 'center',
  },
  applyNowText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    letterSpacing: 0.5,
  },
  iosApplyNowButton: {
    backgroundColor: '#025383',
    paddingVertical: 12,
    paddingHorizontal: 24,
    borderRadius: 25,
    marginLeft: 10,
    alignSelf: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 3,
  },
  iosApplyNowText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    letterSpacing: 0.5,
    textAlign: 'center',
  },
  listContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.3,
  },
  applicationSection: {
    marginBottom: width * 0.05,
  },
  sectionTitle: {
    fontSize: width * 0.05,
    color: '#025383',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: width * 0.05,
  },
  applicationItem: {
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    marginBottom: width * 0.03,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.05,
    shadowRadius: 2,
    elevation: 1,
  },
  applicationRow: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
  },
  statusIcon: {
    marginRight: width * 0.03,
  },
  applicationDetails: {
    flex: 1,
  },
  applicationText: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#333',
  },
  applicationSubText: {
    fontSize: width * 0.035,
    color: '#666',
    marginTop: width * 0.01,
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#888',
    textAlign: 'center',
    fontStyle: 'italic',
  },
  tabBar: {
    height: width * 0.1,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    paddingVertical: width * 0.01,
    flexDirection: 'row',
  },
  tabButton: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.012,
    borderRadius: 16,
    marginRight: width * 0.025,
    backgroundColor: '#f4f6f8',
    minWidth: width * 0.18,
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 0,
  },
  tabButtonActive: {
    backgroundColor: '#025383',
    shadowColor: '#025383',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    borderWidth: 1.5,
    borderColor: '#025383',
    elevation: 2,
  },
  tabButtonText: {
    color: '#666',
    fontSize: width * 0.04,
    fontWeight: '600',
    textAlign: 'center',
  },
  tabButtonTextActive: {
    color: '#fff',
    fontWeight: '700',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  stickyTabBarWrapper: {
    position: 'absolute',
    top: TOTAL_HEADER_HEIGHT + width * 0.13,
    left: 0,
    right: 0,
    zIndex: 10,
    backgroundColor: '#fff',
    height: width * 0.16,
    justifyContent: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  mineToggleButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.012,
    borderRadius: 16,
    marginRight: width * 0.025,
    backgroundColor: '#f4f6f8',
    minWidth: width * 0.18,
    justifyContent: 'center',
    elevation: 0,
    borderWidth: 1.5,
    borderColor: '#e0e0e0',
  },
  mineToggleButtonActive: {
    backgroundColor: '#00B894',
    borderColor: '#00B894',
    shadowColor: '#00B894',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 2,
  },
  mineToggleText: {
    color: '#666',
    fontSize: width * 0.04,
    fontWeight: '600',
    marginLeft: width * 0.015,
  },
  mineToggleTextActive: {
    color: '#fff',
    fontWeight: '700',
  },
  permissionModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  permissionModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 12,
    width: width * 0.9,
    maxWidth: 400,
    padding: width * 0.05,
  },
  permissionModalHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: width * 0.04,
    paddingBottom: width * 0.03,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  permissionModalTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#025383',
    flex: 1,
    textAlign: 'center',
  },
  permissionModalContent: {
    marginBottom: width * 0.04,
  },
  permissionSection: {
    marginBottom: width * 0.04,
  },
  permissionLabel: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#333',
    marginBottom: width * 0.02,
    flexDirection: 'row',
    alignItems: 'center',
  },
  permissionItemsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  permissionChip: {
    backgroundColor: '#e8f4f8',
    borderRadius: 16,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.015,
    marginRight: width * 0.02,
    marginBottom: width * 0.02,
    borderWidth: 1,
    borderColor: '#025383',
  },
  permissionChipText: {
    fontSize: width * 0.035,
    color: '#025383',
    fontWeight: '600',
  },
  noPermissionText: {
    fontSize: width * 0.035,
    color: '#888',
    fontStyle: 'italic',
  },
  permissionInfoBox: {
    flexDirection: 'row',
    backgroundColor: '#f0f8ff',
    padding: width * 0.03,
    borderRadius: 8,
    borderLeftWidth: 4,
    borderLeftColor: '#025383',
    alignItems: 'center',
  },
  permissionInfoText: {
    fontSize: width * 0.035,
    color: '#333',
    marginLeft: width * 0.02,
    flex: 1,
  },
  permissionCloseButton: {
    borderRadius: 8,
    overflow: 'hidden',
  },
  permissionCloseButtonGradient: {
    paddingVertical: width * 0.035,
    alignItems: 'center',
  },
  permissionCloseButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
});

export default AdminHrDocument;