import React, { useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  FlatList,
  ActivityIndicator,
  Dimensions,
  Modal,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import LinearGradient from 'react-native-linear-gradient';

const { width, height } = Dimensions.get('window');

// MongoDB API URL
const MONGODB_HRFORM_API_URL = 'https://api.titandrillingzm.com:6013';

const ApplicationList = ({ applications, isLoading, applicationType, navigation, allowedCountries, allowedProjects, hrPermissions }) => {
  const [menuVisibleId, setMenuVisibleId] = useState(null);
  const [localApplications, setLocalApplications] = useState(applications);
  const [modalVisible, setModalVisible] = useState(false);
  const [modalLoading, setModalLoading] = useState(false);
  const [modalError, setModalError] = useState(null);
  const [userDetails, setUserDetails] = useState(null);
  const [statusModalVisible, setStatusModalVisible] = useState(false);
  const [selectedApplication, setSelectedApplication] = useState(null);

  // Filter applications based on allowed countries and projects
  React.useEffect(() => {
    if (!allowedCountries || !allowedProjects || (allowedCountries.length === 0 && allowedProjects.length === 0)) {
      setLocalApplications(applications);
    } else {
      setLocalApplications(applications.filter(app => {
        const countryOk = allowedCountries.length === 0 || allowedCountries.includes(app.country);
        const projectOk = allowedProjects.length === 0 || allowedProjects.includes(app.project);
        return countryOk && projectOk;
      }));
    }
  }, [applications, allowedCountries, allowedProjects]);

  const formatDate = (dateString) => {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    return `${day}/${month}/${year}`;
  };

  const formatDateTime = (dateString) => {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    return `${day}/${month}/${year} ${hours}:${minutes}`;
  };

  const getSummary = (item) => {
    switch (applicationType) {
      case 'leave_application':
        return `From: ${formatDate(item.dateFrom || item.leaveStartDate)} To: ${formatDate(item.dateTo || item.leaveEndDate)}`;
      case 'funeral_grant_form':
        return `Deceased: ${item.deceasedName || 'N/A'}, Relationship: ${item.relationshipToDeceased || item.deceasedRelationship || 'N/A'}`;
      case 'salary_advance_application':
        return `Amount: K ${item.advanceAmount || item.salaryAdvanceAmount || 'N/A'}, Recovery: ${item.repaymentPeriod || item.selectedRecoveryPeriod || 'N/A'}`;
      case 'application_for_appeal':
        return `Offence: ${item.appealOffence || 'N/A'}, Date: ${formatDate(item.incidentDate || item.appealOffenceDate)}`;
      case 'grievance_record':
        return `Complainants: ${item.grievanceComplaints?.length || 0}, Place: ${item.grievancePlaceOfWork || 'N/A'}`;
      case 'hospital_attendance_form':
        return `Clinics: ${item.selectedClinics?.join(', ') || 'N/A'}, Shift: ${item.selectedShift || 'N/A'}`;
      default:
        return 'N/A';
    }
  };

  const handleDelete = async (item) => {
    try {
      const applicationId = item.id || item.applicationId;
      if (!applicationId) {
        Toast.show({ type: 'error', text1: 'Error', text2: 'Application ID not found.', position: 'bottom' });
        return;
      }

      // Delete from MongoDB
      const response = await fetch(`${MONGODB_HRFORM_API_URL}/hr-applications/${applicationId}`, {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error('Failed to delete application');
      }

      const result = await response.json();
      
      if (result.success) {
        // Remove from local state
        setLocalApplications((prev) => prev.filter((app) => (app.id || app.applicationId) !== applicationId));
        Toast.show({ type: 'success', text1: 'Deleted', text2: 'Application deleted successfully.', position: 'bottom' });
      } else {
        throw new Error(result.message || 'Failed to delete application');
      }
      
      setMenuVisibleId(null);
    } catch (error) {
      console.error('❌ Error deleting application:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to delete application.', position: 'bottom' });
    }
  };

  const handleLockPress = async () => {
    setModalVisible(true);
    setModalLoading(true);
    setModalError(null);
    try {
      const userData = await AsyncStorage.getItem('userData');
      if (!userData) {
        setModalError('User not authenticated. Please log in again.');
        setModalLoading(false);
        return;
      }

      const user = JSON.parse(userData);
      
      // Use passed permissions or fallback to user data
      const country = allowedCountries && allowedCountries.length > 0
        ? allowedCountries.join(', ')
        : (Array.isArray(user.countries) && user.countries.length > 0
          ? user.countries.join(', ')
          : (user.country || 'N/A'));
      const project = allowedProjects && allowedProjects.length > 0
        ? allowedProjects.join(', ')
        : (Array.isArray(user.projects) && user.projects.length > 0
          ? user.projects.join(', ')
          : (user.project || 'N/A'));
      
      // Use passed HR permissions or fallback to user data
      let hrFormPermission = 'N/A';
      if (hrPermissions && hrPermissions.length > 0) {
        hrFormPermission = hrPermissions.join(', ');
      } else {
        // Fallback to user data
        const modules = Array.isArray(user.modules) ? user.modules : 
                        (user.modules ? Object.values(user.modules) : []);
        
        const hrModule = modules.find(m => m.module && (m.module === 'hrForm' || m.module.toLowerCase() === 'hrform'));
        if (hrModule && hrModule.permissions && Array.isArray(hrModule.permissions)) {
          hrFormPermission = hrModule.permissions.length > 0
            ? hrModule.permissions.join(', ')
            : 'No Permissions';
        } else {
          hrFormPermission = 'No Permissions';
        }
      }
      
      setUserDetails({ country, project, hrFormPermission });
    } catch (err) {
      console.error('❌ Error fetching user details:', err);
      setModalError('Failed to fetch user details.');
      setUserDetails(null);
    }
    setModalLoading(false);
  };

  const handleInfoPress = (item) => {
    setSelectedApplication(item);
    setStatusModalVisible(true);
  };

  const getStatusBreakdown = (item) => {
    const statuses = [];
    
    // Supervisor Status
    if (item.Supervisor) {
      if (item.Supervisor.status === 'Approved') {
        const timestamp = item.Supervisor.timestamp ? formatDateTime(item.Supervisor.timestamp) : 'N/A';
        statuses.push({
          role: 'Supervisor',
          status: 'Approved',
          name: item.Supervisor.name || 'N/A',
          timestamp: timestamp,
          color: '#4CAF50'
        });
      } else if (item.Supervisor.status === 'Rejected') {
        const timestamp = item.Supervisor.timestamp ? formatDateTime(item.Supervisor.timestamp) : 'N/A';
        statuses.push({
          role: 'Supervisor',
          status: 'Rejected',
          name: item.Supervisor.name || 'N/A',
          timestamp: timestamp,
          color: '#D81B60'
        });
      } else {
        statuses.push({
          role: 'Supervisor',
          status: 'Pending',
          name: 'Not yet reviewed',
          timestamp: 'N/A',
          color: '#F57C00'
        });
      }
    } else {
      statuses.push({
        role: 'Supervisor',
        status: 'Pending',
        name: 'Not yet reviewed',
        timestamp: 'N/A',
        color: '#F57C00'
      });
    }

    // HR Manager Status
    if (item.HrManager) {
      if (item.HrManager.status === 'Approved') {
        const timestamp = item.HrManager.timestamp ? formatDateTime(item.HrManager.timestamp) : 'N/A';
        statuses.push({
          role: 'HR Manager',
          status: 'Approved',
          name: item.HrManager.name || 'N/A',
          timestamp: timestamp,
          color: '#4CAF50'
        });
      } else if (item.HrManager.status === 'Rejected') {
        const timestamp = item.HrManager.timestamp ? formatDateTime(item.HrManager.timestamp) : 'N/A';
        statuses.push({
          role: 'HR Manager',
          status: 'Rejected',
          name: item.HrManager.name || 'N/A',
          timestamp: timestamp,
          color: '#D81B60'
        });
      } else {
        statuses.push({
          role: 'HR Manager',
          status: 'Pending',
          name: 'Not yet reviewed',
          timestamp: 'N/A',
          color: '#F57C00'
        });
      }
    } else {
      statuses.push({
        role: 'HR Manager',
        status: 'Pending',
        name: 'Not yet reviewed',
        timestamp: 'N/A',
        color: '#F57C00'
      });
    }

    // Finance Manager Status (only for salary advance applications)
    if (applicationType === 'salary_advance_application') {
      if (item.FinanceManager) {
        if (item.FinanceManager.status === 'Approved') {
          const timestamp = item.FinanceManager.timestamp ? formatDateTime(item.FinanceManager.timestamp) : 'N/A';
          statuses.push({
            role: 'Finance Manager',
            status: 'Approved',
            name: item.FinanceManager.name || 'N/A',
            timestamp: timestamp,
            color: '#4CAF50'
          });
        } else if (item.FinanceManager.status === 'Rejected') {
          const timestamp = item.FinanceManager.timestamp ? formatDateTime(item.FinanceManager.timestamp) : 'N/A';
          statuses.push({
            role: 'Finance Manager',
            status: 'Rejected',
            name: item.FinanceManager.name || 'N/A',
            timestamp: timestamp,
            color: '#D81B60'
          });
        } else {
          statuses.push({
            role: 'Finance Manager',
            status: 'Pending',
            name: 'Not yet reviewed',
            timestamp: 'N/A',
            color: '#F57C00'
          });
        }
      } else {
        statuses.push({
          role: 'Finance Manager',
          status: 'Pending',
          name: 'Not yet reviewed',
          timestamp: 'N/A',
          color: '#F57C00'
        });
      }
    }

    return statuses;
  };

  const renderItem = ({ item }) => {
    const itemId = item.id || item.applicationId;
    
    // Determine status display and approver information
    let statusText = item.status || 'Pending';
    let statusColor = '#F57C00';
    let approverInfo = '';
    
    if (statusText === 'Approved') {
      statusColor = '#4CAF50';
      // Check who approved
      if (item.Supervisor && item.Supervisor.status === 'Approved') {
        const timestamp = item.Supervisor.timestamp ? formatDateTime(item.Supervisor.timestamp) : 'N/A';
        approverInfo = `Approved by Supervisor: ${item.Supervisor.name || 'N/A'} (${timestamp})`;
      } else if (item.HrManager && item.HrManager.status === 'Approved') {
        const timestamp = item.HrManager.timestamp ? formatDateTime(item.HrManager.timestamp) : 'N/A';
        approverInfo = `Approved by HR Manager: ${item.HrManager.name || 'N/A'} (${timestamp})`;
      } else if (item.FinanceManager && item.FinanceManager.status === 'Approved') {
        const timestamp = item.FinanceManager.timestamp ? formatDateTime(item.FinanceManager.timestamp) : 'N/A';
        approverInfo = `Approved by Finance Manager: ${item.FinanceManager.name || 'N/A'} (${timestamp})`;
      } else {
        const timestamp = item.approvedAt ? formatDateTime(item.approvedAt) : 'N/A';
        approverInfo = `Approved by: ${item.approvedBy || 'Unknown'} (${timestamp})`;
      }
    } else if (statusText === 'Rejected') {
      statusColor = '#D81B60';
      // Check who rejected
      if (item.Supervisor && item.Supervisor.status === 'Rejected') {
        const timestamp = item.Supervisor.timestamp ? formatDateTime(item.Supervisor.timestamp) : 'N/A';
        approverInfo = `Rejected by Supervisor: ${item.Supervisor.name || 'N/A'} (${timestamp})`;
      } else if (item.HrManager && item.HrManager.status === 'Rejected') {
        const timestamp = item.HrManager.timestamp ? formatDateTime(item.HrManager.timestamp) : 'N/A';
        approverInfo = `Rejected by HR Manager: ${item.HrManager.name || 'N/A'} (${timestamp})`;
      } else if (item.FinanceManager && item.FinanceManager.status === 'Rejected') {
        const timestamp = item.FinanceManager.timestamp ? formatDateTime(item.FinanceManager.timestamp) : 'N/A';
        approverInfo = `Rejected by Finance Manager: ${item.FinanceManager.name || 'N/A'} (${timestamp})`;
      } else {
        const timestamp = item.rejectedAt ? formatDateTime(item.rejectedAt) : 'N/A';
        approverInfo = `Rejected by: ${item.rejectedBy || 'Unknown'} (${timestamp})`;
      }
    } else {
      // Pending status - show current approver status
      if (item.Supervisor && item.Supervisor.status) {
        if (item.Supervisor.status === 'Approved') {
          const timestamp = item.Supervisor.timestamp ? formatDateTime(item.Supervisor.timestamp) : 'N/A';
          approverInfo = `Supervisor Approved: ${item.Supervisor.name || 'N/A'} (${timestamp})`;
        } else if (item.Supervisor.status === 'Rejected') {
          const timestamp = item.Supervisor.timestamp ? formatDateTime(item.Supervisor.timestamp) : 'N/A';
          approverInfo = `Supervisor Rejected: ${item.Supervisor.name || 'N/A'} (${timestamp})`;
        } else {
          approverInfo = 'Pending Supervisor Approval';
        }
      } else if (item.HrManager && item.HrManager.status) {
        if (item.HrManager.status === 'Approved') {
          const timestamp = item.HrManager.timestamp ? formatDateTime(item.HrManager.timestamp) : 'N/A';
          approverInfo = `HR Manager Approved: ${item.HrManager.name || 'N/A'} (${timestamp})`;
        } else if (item.HrManager.status === 'Rejected') {
          const timestamp = item.HrManager.timestamp ? formatDateTime(item.HrManager.timestamp) : 'N/A';
          approverInfo = `HR Manager Rejected: ${item.HrManager.name || 'N/A'} (${timestamp})`;
        } else {
          approverInfo = 'Pending HR Manager Approval';
        }
      } else if (item.FinanceManager && item.FinanceManager.status) {
        if (item.FinanceManager.status === 'Approved') {
          const timestamp = item.FinanceManager.timestamp ? formatDateTime(item.FinanceManager.timestamp) : 'N/A';
          approverInfo = `Finance Manager Approved: ${item.FinanceManager.name || 'N/A'} (${timestamp})`;
        } else if (item.FinanceManager.status === 'Rejected') {
          const timestamp = item.FinanceManager.timestamp ? formatDateTime(item.FinanceManager.timestamp) : 'N/A';
          approverInfo = `Finance Manager Rejected: ${item.FinanceManager.name || 'N/A'} (${timestamp})`;
        } else {
          approverInfo = 'Pending Finance Manager Approval';
        }
      } else {
        approverInfo = 'Pending Approval';
      }
    }

    // Get name and employee number (support both MongoDB and old field names)
    const fullName = item.applicantName || item.fullName || 'Unknown';
    const employeeNo = item.employeeNumber || item.employeeNo || 'N/A';
    const submittedAt = item.createdAt || item.submittedAt;

    return (
      <View style={styles.applicationItem}>
        <TouchableOpacity
          style={{ flex: 1 }}
          onPress={() => navigation.navigate('ApplicationDetail', { application: item, applicationType })}
          accessibilityLabel={`View details for ${applicationType} application by ${fullName}`}
          accessibilityRole="button"
        >
          <View style={styles.applicationContent}>
            <View style={{ flex: 1 }}>
              <Text style={styles.applicationTitle}>{fullName}</Text>
              {item.email ? (
                <Text style={styles.applicationEmail}>{item.email}</Text>
              ) : null}
              <Text style={styles.applicationSubtitle}>Employee No: {employeeNo}</Text>
              <Text style={styles.applicationSubtitle}>Submitted: {formatDate(submittedAt)}</Text>
              <Text style={styles.applicationSubtitle}>{getSummary(item)}</Text>
              <Text style={[styles.applicationStatus, { color: statusColor }]}>Status: {statusText}</Text>
              {approverInfo ? (
                <Text style={[styles.applicationStatus, { 
                  color: statusText === 'Approved' ? '#4CAF50' : 
                         statusText === 'Rejected' ? '#D81B60' : '#015185', 
                  fontSize: width * 0.032, 
                  marginTop: 2,
                  fontStyle: 'italic'
                }]}>
                  {approverInfo}
                </Text>
              ) : null}
            </View>
            <View style={styles.actionButtons}>
              <TouchableOpacity
                onPress={() => handleInfoPress(item)}
                style={{ padding: width * 0.01, marginRight: width * 0.01 }}
                accessibilityLabel="Show approval status details"
                accessibilityRole="button"
              >
                <MaterialIcons name="info" size={width * 0.06} color="#015185" />
              </TouchableOpacity>
              <TouchableOpacity
                onPress={() => setMenuVisibleId(menuVisibleId === itemId ? null : itemId)}
                style={{ padding: width * 0.01 }}
                accessibilityLabel="Show options"
                accessibilityRole="button"
              >
                <MaterialIcons name="more-vert" size={width * 0.07} color="#333" />
              </TouchableOpacity>
            </View>
          </View>
        </TouchableOpacity>
        {menuVisibleId === itemId && (
          <View style={styles.menuDropdown}>
            <TouchableOpacity
              style={styles.menuItem}
              onPress={() => {
                setMenuVisibleId(null);
                navigation.navigate('ApplicationDetail', { application: item, applicationType });
              }}
            >
              <MaterialIcons name="visibility" size={width * 0.05} color="#015185" style={{ marginRight: 8 }} />
              <Text style={styles.menuText}>View</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.menuItem}
              onPress={() => handleDelete(item)}
            >
              <MaterialIcons name="delete" size={width * 0.05} color="#D81B60" style={{ marginRight: 8 }} />
              <Text style={[styles.menuText, { color: '#D81B60' }]}>Delete</Text>
            </TouchableOpacity>
          </View>
        )}
      </View>
    );
  };

  return (
    <View style={styles.listContainer}>
      {/* Header with lock icon */}
      <View style={styles.headerRow}>
        <Text style={styles.headerTitle}>Applications</Text>
        <TouchableOpacity style={styles.lockButton} onPress={handleLockPress} accessibilityLabel="Show user details" accessibilityRole="button">
          <MaterialIcons name="lock" size={20} color="#015185" />
        </TouchableOpacity>
      </View>
      {/* Modal for user details */}
      <Modal
        visible={modalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>User Details</Text>
            {modalLoading ? (
              <ActivityIndicator size="large" color="#015185" />
            ) : modalError ? (
              <Text style={styles.modalError}>{modalError}</Text>
            ) : userDetails ? (
              <View style={styles.modalDetailsBox}>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>Country:</Text> {userDetails.country}</Text>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>Project:</Text> {userDetails.project}</Text>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>HR Form Permission:</Text> {String(userDetails.hrFormPermission)}</Text>
              </View>
            ) : null}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setModalVisible(false)}
              accessibilityLabel="Close user details"
              accessibilityRole="button"
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
      {isLoading ? (
        <ActivityIndicator size="large" color="#015185" style={styles.loader} />
      ) : localApplications.length === 0 ? (
        <Text style={styles.noApplicationsText}>No applications found</Text>
      ) : (
        <FlatList
          data={localApplications}
          renderItem={renderItem}
          keyExtractor={(item) => item.id || item.applicationId || String(Math.random())}
          showsVerticalScrollIndicator={false}
          extraData={menuVisibleId}
        />
      )}
      
      {/* Status Breakdown Modal */}
      <Modal
        visible={statusModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setStatusModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.statusModalContent}>
            <Text style={styles.statusModalTitle}>Approval Status Breakdown</Text>
            {selectedApplication ? (
              <View style={styles.statusBreakdownContainer}>
                {getStatusBreakdown(selectedApplication).map((statusItem, index) => (
                  <View key={index} style={styles.statusBreakdownItem}>
                    <View style={styles.statusBreakdownHeader}>
                      <Text style={styles.statusBreakdownRole}>{statusItem.role}</Text>
                      <View style={[styles.statusBreakdownBadge, { backgroundColor: statusItem.color }]}>
                        <Text style={styles.statusBreakdownBadgeText}>{statusItem.status}</Text>
                      </View>
                    </View>
                    <Text style={styles.statusBreakdownName}>{statusItem.name}</Text>
                    <Text style={styles.statusBreakdownTimestamp}>{statusItem.timestamp}</Text>
                  </View>
                ))}
              </View>
            ) : null}
            <TouchableOpacity
              style={styles.statusModalCloseButton}
              onPress={() => setStatusModalVisible(false)}
              accessibilityLabel="Close status breakdown"
              accessibilityRole="button"
            >
              <Text style={styles.statusModalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  listContainer: {
    flex: 1,
    padding: width * 0.04,
  },
  headerRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: width * 0.03,
  },
  headerTitle: {
    fontSize: width * 0.055,
    fontWeight: 'bold',
    color: '#015185',
  },
  lockButton: {
    padding: width * 0.01,
  },
  applicationItem: {
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    marginBottom: width * 0.03,
    padding: width * 0.04,
    elevation: 3,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
  },
  applicationContent: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  applicationTitle: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#333',
  },
  applicationSubtitle: {
    fontSize: width * 0.035,
    color: '#666',
    marginTop: width * 0.01,
  },
  applicationStatus: {
    fontSize: width * 0.035,
    fontWeight: '600',
    marginTop: width * 0.01,
  },
  noApplicationsText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginTop: height * 0.2,
  },
  loader: {
    marginTop: height * 0.2,
  },
  menuDropdown: {
    position: 'absolute',
    top: width * 0.13,
    right: width * 0.04,
    backgroundColor: '#fff',
    borderRadius: 8,
    elevation: 6,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    zIndex: 100,
    paddingVertical: 4,
    minWidth: width * 0.32,
  },
  menuItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 16,
  },
  menuText: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  applicationEmail: {
    fontSize: width * 0.033,
    color: '#888',
    marginTop: 2,
    marginBottom: 2,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.4)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 24,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 5,
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    marginBottom: 16,
    color: '#015185',
  },
  modalDetailsBox: {
    marginBottom: 16,
    alignSelf: 'stretch',
  },
  modalDetail: {
    fontSize: 16,
    marginBottom: 8,
    color: '#222',
  },
  modalLabel: {
    fontWeight: 'bold',
    color: '#015185',
  },
  modalCloseButton: {
    marginTop: 8,
    backgroundColor: '#015185',
    borderRadius: 8,
    paddingVertical: 8,
    paddingHorizontal: 24,
  },
  modalCloseText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  modalError: {
    color: '#FF6B6B',
    fontSize: 16,
    marginBottom: 12,
    textAlign: 'center',
  },
  statusModalContent: {
    width: '90%',
    maxHeight: '80%',
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 24,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 5,
  },
  statusModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    marginBottom: 20,
    color: '#015185',
    textAlign: 'center',
  },
  statusBreakdownContainer: {
    width: '100%',
    marginBottom: 20,
  },
  statusBreakdownItem: {
    backgroundColor: '#F8F9FA',
    borderRadius: 12,
    padding: 16,
    marginBottom: 12,
    borderLeftWidth: 4,
    borderLeftColor: '#E0E0E0',
  },
  statusBreakdownHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  statusBreakdownRole: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#333',
  },
  statusBreakdownBadge: {
    paddingHorizontal: 12,
    paddingVertical: 4,
    borderRadius: 20,
  },
  statusBreakdownBadgeText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: 'bold',
  },
  statusBreakdownName: {
    fontSize: 14,
    color: '#666',
    marginBottom: 4,
  },
  statusBreakdownTimestamp: {
    fontSize: 12,
    color: '#888',
    fontStyle: 'italic',
  },
  statusModalCloseButton: {
    backgroundColor: '#015185',
    borderRadius: 8,
    paddingVertical: 12,
    paddingHorizontal: 24,
    minWidth: 100,
  },
  statusModalCloseText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
    textAlign: 'center',
  },
});

export default ApplicationList;