import React, { useEffect, useState, useMemo, useCallback, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  ActivityIndicator,
  Dimensions,
  Platform,
} from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import MapView, { Marker, PROVIDER_GOOGLE } from 'react-native-maps';
import Toast from 'react-native-toast-message';
import { useNavigation } from '@react-navigation/native';
import NetInfo from '@react-native-community/netinfo';
import EquipmentAPI from '../../ConfigMongo/EquipmentMongoConfig';

const { width, height } = Dimensions.get('window');

const DEFAULT_REGION = {
  latitude: -13.133897,
  longitude: 27.849332,
  latitudeDelta: 8,
  longitudeDelta: 8,
};

const statusColors = {
  Available: '#00B894',
  'Maintenance Required': '#FFD205',
  Breakdown: '#E74C3C',
  'In Maintenance': '#E67E22',
  Deploy: '#0055A5',
  Unknown: '#545d6b',
};

const MapViewEquipment = () => {
  const navigation = useNavigation();
  const [markers, setMarkers] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filterStatus, setFilterStatus] = useState('ALL');
  const [lastSync, setLastSync] = useState(null);
  const [refreshing, setRefreshing] = useState(false);
  const mapRef = useRef(null);

  const getStatusColor = (status) => statusColors[status] || statusColors.Unknown;

  const parseCoordinates = (coords) => {
    if (!coords) return null;
    if (typeof coords === 'string') {
      const [latStr, lonStr] = coords.split(',').map((v) => v.trim());
      const latitude = parseFloat(latStr);
      const longitude = parseFloat(lonStr);
      if (Number.isNaN(latitude) || Number.isNaN(longitude)) return null;
      return { latitude, longitude };
    }
    if (typeof coords === 'object' && coords.latitude && coords.longitude) {
      const latitude = parseFloat(coords.latitude);
      const longitude = parseFloat(coords.longitude);
      if (Number.isNaN(latitude) || Number.isNaN(longitude)) return null;
      return { latitude, longitude };
    }
    return null;
  };

  const buildMarker = useCallback((item) => {
    const coords =
      parseCoordinates(item.lastKnownCoordinates) ||
      parseCoordinates(item.gpsCoordinates) ||
      parseCoordinates(item.step1?.coordinates) ||
      parseCoordinates(item.coordinates);

    if (!coords) return null;

    const status = item.status || 'Unknown';

    return {
      id: item.id || item._id || Math.random().toString(),
      latitude: coords.latitude,
      longitude: coords.longitude,
      title: item.equipmentName || item.rigName || item.vehicleNumber || 'Equipment',
      description:
        item.gpsAddress ||
        item.lastKnownLocation ||
        `${coords.latitude.toFixed(4)}, ${coords.longitude.toFixed(4)}`,
      status,
      country: item.country || 'N/A',
      lastUpdated: item.updatedAt || item.locationUpdatedAt || item.lastInspectionAt || null,
    };
  }, []);

  const loadEquipmentLocations = useCallback(async ({ forceFresh = false } = {}) => {
    try {
      setLoading(true);

      let equipmentList = [];
      const storedEquipment = await AsyncStorage.getItem('equipment');

      if (storedEquipment && !forceFresh) {
        equipmentList = JSON.parse(storedEquipment);
      }

      const netInfo = await NetInfo.fetch();
      if (netInfo.isConnected) {
        try {
          const response = await EquipmentAPI.getAllEquipment();
          if (response.success && Array.isArray(response.data)) {
            equipmentList = response.data;
            await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
            setLastSync(new Date().toISOString());
          }
        } catch (err) {
          console.warn('Unable to refresh equipment from server:', err?.message);
          if (!equipmentList.length) {
            throw err;
          }
        }
      } else if (!equipmentList.length) {
        Toast.show({
          type: 'info',
          text1: 'Offline Mode',
          text2: 'Showing last cached equipment locations.',
          position: 'top',
        });
      }

      const markersData = equipmentList
        .map(buildMarker)
        .filter(Boolean);

      if (!markersData.length) {
        Toast.show({
          type: 'info',
          text1: 'No Locations',
          text2: 'No equipment with GPS coordinates yet.',
          position: 'top',
        });
      }

      setMarkers(markersData);
    } catch (error) {
      console.error('Error loading equipment locations:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load equipment locations.',
        position: 'top',
      });
    } finally {
      setLoading(false);
      setRefreshing(false);
    }
  }, [buildMarker]);

  useEffect(() => {
    loadEquipmentLocations();
  }, [loadEquipmentLocations]);

  const filteredMarkers = useMemo(() => {
    if (filterStatus === 'ALL') return markers;
    return markers.filter((marker) => marker.status === filterStatus);
  }, [markers, filterStatus]);

  const initialRegion = useMemo(() => {
    if (filteredMarkers.length === 0) {
      return DEFAULT_REGION;
    }
    const first = filteredMarkers[0];
    return {
      latitude: first.latitude,
      longitude: first.longitude,
      latitudeDelta: 3,
      longitudeDelta: 3,
    };
  }, [filteredMarkers]);

  const handleFilterChange = (status) => {
    setFilterStatus((prev) => (prev === status ? 'ALL' : status));
  };

  const handleFitToMarkers = () => {
    if (!mapRef.current || !filteredMarkers.length) return;
    mapRef.current.fitToCoordinates(
      filteredMarkers.map((marker) => ({
        latitude: marker.latitude,
        longitude: marker.longitude,
      })),
      {
        edgePadding: {
          top: height * 0.15,
          right: width * 0.1,
          bottom: height * 0.15,
          left: width * 0.1,
        },
        animated: true,
      },
    );
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={styles.header}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()}>
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Equipment Map View</Text>
          <TouchableOpacity onPress={() => loadEquipmentLocations({ forceFresh: true })}>
            <MaterialIcons name="refresh" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      <View style={styles.metaBar}>
        <View>
          <Text style={styles.metaTitle}>Total Equipments</Text>
          <Text style={styles.metaValue}>{markers.length}</Text>
        </View>
        <View>
          <Text style={styles.metaTitle}>With GPS</Text>
          <Text style={styles.metaValue}>{filteredMarkers.length}</Text>
        </View>
        {lastSync && (
          <View>
            <Text style={styles.metaTitle}>Last Sync</Text>
            <Text style={styles.metaValueSmall}>
              {new Date(lastSync).toLocaleString()}
            </Text>
          </View>
        )}
      </View>

      <View style={styles.filterBar}>
        {['Available', 'Maintenance Required', 'Breakdown', 'In Maintenance', 'Deploy'].map((status) => {
          const active = filterStatus === status;
          return (
            <TouchableOpacity
              key={status}
              style={[
                styles.filterChip,
                { backgroundColor: active ? getStatusColor(status) : '#f0f0f0' },
              ]}
              onPress={() => handleFilterChange(status)}
            >
              <Text style={[styles.filterChipText, active && styles.filterChipTextActive]}>
                {status}
              </Text>
            </TouchableOpacity>
          );
        })}
        <TouchableOpacity
          style={styles.filterChipOutline}
          onPress={handleFitToMarkers}
        >
          <MaterialIcons name="center-focus-strong" size={18} color="#015185" />
          <Text style={styles.filterChipOutlineText}>Fit</Text>
        </TouchableOpacity>
      </View>

      <View style={styles.mapWrapper}>
        {loading ? (
          <View style={styles.loadingContainer}>
            <ActivityIndicator size="large" color="#3481BC" />
            <Text style={styles.loadingText}>
              {refreshing ? 'Refreshing equipment...' : 'Loading equipment locations...'}
            </Text>
          </View>
        ) : (
          <MapView
            ref={mapRef}
            style={styles.map}
            initialRegion={initialRegion}
            provider={PROVIDER_GOOGLE}
          >
            {filteredMarkers.map((marker) => (
              <Marker
                key={marker.id}
                coordinate={{ latitude: marker.latitude, longitude: marker.longitude }}
                title={marker.title}
                description={`${marker.description}\nStatus: ${marker.status}`}
                pinColor={getStatusColor(marker.status)}
              />
            ))}
          </MapView>
        )}
      </View>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  metaBar: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.03,
    backgroundColor: '#f8f9fb',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e6ed',
  },
  metaTitle: {
    fontSize: width * 0.03,
    color: '#545d6b',
    fontWeight: '600',
  },
  metaValue: {
    fontSize: width * 0.05,
    color: '#015185',
    fontWeight: '700',
  },
  metaValueSmall: {
    fontSize: width * 0.03,
    color: '#015185',
    fontWeight: '600',
  },
  filterBar: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    alignItems: 'center',
    paddingHorizontal: width * 0.04,
    paddingTop: width * 0.02,
    paddingBottom: width * 0.03,
    gap: 8,
  },
  filterChip: {
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 16,
    backgroundColor: '#f0f0f0',
  },
  filterChipText: {
    fontSize: width * 0.03,
    color: '#333',
    fontWeight: '600',
  },
  filterChipTextActive: {
    color: '#fff',
  },
  filterChipOutline: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 16,
    borderWidth: 1,
    borderColor: '#015185',
    backgroundColor: '#fff',
    gap: 4,
  },
  filterChipOutlineText: {
    fontSize: width * 0.03,
    color: '#015185',
    fontWeight: '600',
  },
  mapWrapper: {
    flex: 1,
    borderTopWidth: 1,
    borderColor: '#e0e6ed',
    overflow: 'hidden',
  },
  header: {
    width: '100%',
    paddingTop: StatusBar.currentHeight || width * 0.12,
    paddingBottom: width * 0.04,
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '600',
  },
  map: {
    flex: 1,
    minHeight: height * 0.5,
  },
  loadingContainer: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
  },
  loadingText: {
    marginTop: 12,
    fontSize: width * 0.04,
    color: '#555',
  },
});

export default MapViewEquipment;