# Equipment Screen Sync Improvements

## Overview
Comprehensive online/offline sync improvements for `Equipment_one.js` with auto-sync, manual sync via cloud icon, and real-time status tracking.

---

## ✅ Key Features Implemented

### 1. **Smart Online/Offline Sync**
- ✅ Automatic network status detection
- ✅ Seamless online/offline switching
- ✅ Auto-sync when connection restored
- ✅ Local storage fallback
- ✅ Pending changes queue for offline edits

### 2. **Manual Sync via Cloud Icon**
- ✅ Cloud icon in header refreshes equipment data
- ✅ Only works when online (disabled offline)
- ✅ Shows sync status with toast notifications
- ✅ Updates last sync timestamp

### 3. **Visual Sync Status Bar**
- ✅ Real-time online/offline indicator
- ✅ Last sync timestamp display
- ✅ Auto-sync status badge
- ✅ Color-coded status (Green = Online, Orange = Offline)

### 4. **Toast Notifications**
- ✅ Success: "Equipment data synced successfully"
- ✅ Info: "Using cached equipment data" (offline)
- ✅ Warning: "Connection lost - Switching to offline mode"
- ✅ Success: "Connection restored" (auto-sync triggered)
- ✅ Error: "Cannot sync while offline" / "No offline data available"

---

## 🔄 How It Works

### Initial Load
```javascript
1. Check network status
2. Get user permissions from AsyncStorage
3. If ONLINE:
   - Fetch from MongoDB API
   - Filter by user permissions
   - Store locally with timestamp
   - Show equipment based on permissions
4. If OFFLINE:
   - Load from AsyncStorage
   - Filter by cached permissions
   - Show "Offline Mode" toast
   - Display cached equipment
```

### When Connection Restored
```javascript
1. Network listener detects online status
2. Show "Connection restored" toast
3. Auto-trigger equipment sync (after 1 second)
4. Fetch latest data from MongoDB
5. Update local storage with timestamp
6. Sync pending offline changes
7. Refresh UI
8. Show "Equipment data synced successfully" toast
```

### Manual Sync (Cloud Icon)
```javascript
1. User clicks cloud refresh icon in header
2. Check if online (show error if offline)
3. Show sync modal with rotating icon
4. Fetch latest equipment from MongoDB
5. Filter by user permissions
6. Update local storage
7. Update last sync timestamp
8. Refresh displayed equipment
9. Show success toast
```

### Pending Changes Sync
```javascript
1. When offline, edits are marked with pendingSync flag
2. When connection restored, auto-sync pending changes
3. Loop through pending items:
   - If pendingDelete: DELETE from MongoDB
   - If pendingSync: PUT to MongoDB
4. Clear pending flags after successful sync
5. Update local storage
6. Show "Pending changes synced" toast
```

---

## 📊 Data Storage Structure

### AsyncStorage Keys
- `equipment` - Array of equipment objects
- `equipmentLastSync` - ISO timestamp of last sync
- `userData` - User profile with permissions
- `userInfo` - Cached user permissions
- `selectedCountry` - Selected country filter

### Equipment Object
```json
{
  "id": "equip_123",
  "firebaseKey": "equip_123",
  "mainCategory": "Light Vehicle",
  "country": "DRC",
  "equipmentName": "Toyota Hilux",
  "equipmentNumber": "VH-001",
  "status": "Available",
  "meterReading": "45000",
  "meterUnit": "km",
  "createdAt": "2025-10-20T10:00:00Z",
  "updatedAt": "2025-10-24T10:30:00Z",
  "createdBy": "user123",
  "pendingSync": false,
  "pendingDelete": false,
  "lastSyncTime": "2025-10-24T10:30:00.000Z"
}
```

---

## 🎨 UI Components

### Sync Status Bar
```
┌──────────────────────────────────────────────────┐
│ 🌐 Online • Last sync: 2m ago  [Auto-Sync On]   │
└──────────────────────────────────────────────────┘
```

**Features:**
- Green background when online, orange when offline
- Pulsing dot indicator
- Last sync time (Just now, 5m ago, 2h ago, 3d ago)
- Auto-sync status badge
- Positioned between header and filter bar

### Cloud Icon (Manual Sync)
Located in header, next to "Equipments" title:
- ☁️ Cloud icon that user can tap
- Triggers manual sync when clicked
- Disabled when offline
- Shows rotating sync animation during sync
- Updates last sync time on completion

---

## 🔐 Permission-Based Filtering

### Equipment Permissions
Equipment is filtered based on user's permissions:

1. **`onlyMineView`** - See only equipment created by user OR in user's countries
2. **`view`** - See equipment in user's countries
3. **`all`** - See all equipment regardless of country
4. **`create`** - Can add new equipment
5. **`update`** - Can modify equipment and change status
6. **`delete`** - Can delete equipment

### Filtering Logic
```javascript
if (permission === 'onlyMineView') {
  // Show if created by user OR in user's countries
  return equipment.filter(item => 
    item.createdBy === userUid || 
    userCountries.includes(item.country)
  );
}

if (permission === 'view') {
  // Show if in user's countries
  return equipment.filter(item => 
    userCountries.includes(item.country)
  );
}

if (permission === 'all') {
  // Show all equipment
  return equipment;
}
```

---

## 🌐 Online/Offline Features

### Online Mode
- ✅ Fetch latest data from MongoDB
- ✅ Real-time updates
- ✅ Immediate sync of changes
- ✅ Full CRUD operations
- ✅ Status updates
- ✅ Delete operations

### Offline Mode
- ✅ Load from local cache
- ✅ View equipment data
- ✅ Queue status changes
- ✅ Queue delete operations
- ✅ Mark changes with `pendingSync` flag
- ✅ Auto-sync when back online

---

## 🔄 Auto-Sync Features

### Triggers for Auto-Sync
1. **Connection Restored**
   - Network listener detects online status
   - Waits 1 second (debounce)
   - Syncs equipment data
   - Syncs pending changes

2. **Screen Focus**
   - Uses `useFocusEffect` hook
   - Refreshes when user returns to screen
   - Checks for updates

3. **Pending Changes**
   - Separate effect monitors network
   - Syncs queued offline edits
   - Clears pending flags after sync

---

## 📱 User Experience Flow

### Scenario 1: Fresh Load (Online)
```
1. User navigates to Equipment screen
2. ✅ Shows sync modal "Processing Equipment Data"
3. 🌐 Fetches from MongoDB
4. 🔒 Filters by user permissions
5. 💾 Caches locally
6. 🔔 Toast: "Equipment data synced successfully"
7. 📊 Displays equipment list
8. ✅ Sync status shows "Just now"
```

### Scenario 2: Offline Mode
```
1. User opens app without internet
2. 📴 Detects offline status
3. 📦 Loads from AsyncStorage
4. 🔒 Applies cached permissions
5. 🔔 Toast: "Using cached equipment data"
6. 📊 Displays cached equipment
7. ⚠️ Sync status shows "Offline • Last sync: 2h ago"
8. ✅ All view features work
```

### Scenario 3: Connection Lost During Use
```
1. User is browsing equipment
2. 📴 Internet disconnects
3. 🔔 Toast: "No internet connection"
4. 🟠 Status bar changes to orange
5. 📦 Continues using cached data
6. ✏️ Edits are queued with pendingSync flag
```

### Scenario 4: Connection Restored
```
1. Internet reconnects
2. 🌐 Network listener detects
3. 🔔 Toast: "Connection restored"
4. 🔄 Auto-sync starts (1 second delay)
5. 📡 Fetches latest equipment
6. 🔄 Syncs pending changes
7. 💾 Updates local cache
8. 🔔 Toast: "Equipment data synced successfully"
9. 🟢 Status bar changes to green
10. ✅ Last sync updated to "Just now"
```

### Scenario 5: Manual Sync via Cloud Icon
```
1. User clicks cloud icon in header
2. ✅ Checks if online
3. 🔄 Shows sync modal
4. 📡 Fetches from MongoDB
5. 🔒 Applies permissions filter
6. 💾 Updates local storage
7. 🕒 Updates last sync timestamp
8. 📊 Refreshes equipment list
9. 🔔 Toast: "Equipment data synced successfully"
```

---

## 🛠️ Technical Implementation

### Network Status Monitoring
```javascript
useEffect(() => {
  const unsubscribe = NetInfo.addEventListener((state) => {
    const wasOffline = !isOnline;
    const isNowOnline = state.isConnected;
    
    setIsOnline(isNowOnline);
    
    if (wasOffline && isNowOnline) {
      // Auto-sync when connection restored
      setTimeout(() => {
        loadData(true, true);
      }, 1000);
    }
  });
  
  return () => unsubscribe();
}, [isOnline]);
```

### Last Sync Time Formatting
```javascript
const getLastSyncText = () => {
  if (!lastSyncTime) return 'Never synced';
  
  const diffInMinutes = Math.floor((now - syncDate) / (1000 * 60));
  
  if (diffInMinutes < 1) return 'Just now';
  if (diffInMinutes < 60) return `${diffInMinutes}m ago`;
  
  const diffInHours = Math.floor(diffInMinutes / 60);
  if (diffInHours < 24) return `${diffInHours}h ago`;
  
  const diffInDays = Math.floor(diffInHours / 24);
  return `${diffInDays}d ago`;
};
```

### Sync with Timestamp
```javascript
// Store with timestamp
await AsyncStorage.setItem('equipment', JSON.stringify(parsedData));
const syncTime = new Date().toISOString();
await AsyncStorage.setItem('equipmentLastSync', syncTime);
setLastSyncTime(syncTime);
```

---

## 🔍 Error Handling

### MongoDB Fetch Errors
```javascript
try {
  const response = await fetch('https://api.titandrillingzm.com:6001/...');
  // Process response
} catch (fetchError) {
  console.error('❌ Error fetching from MongoDB:', fetchError);
  
  // Fallback to cache
  const savedData = await AsyncStorage.getItem('equipment');
  if (savedData) {
    parsedData = JSON.parse(savedData);
    showToast('info', 'Offline Mode', 'Using cached equipment data');
  } else {
    showToast('error', 'Error', 'No offline data available');
  }
}
```

### Network Errors
```javascript
if (error.code === 'NETWORK_ERROR') {
  // Load from offline storage
  const savedData = await AsyncStorage.getItem('equipment');
  const parsedData = JSON.parse(savedData);
  
  // Apply permission filters
  const filteredData = filterEquipmentByPermission(parsedData, userInfo);
  
  showToast('info', 'Offline', 'Loading from local storage');
}
```

---

## 📈 Performance Optimizations

1. **Lazy Loading**
   - Initial load: 15 items
   - Scroll pagination: 10 items per page
   - Efficient memory usage

2. **Smart Caching**
   - Store filtered equipment locally
   - Timestamp-based cache validation
   - Only sync when needed

3. **Permission Filtering**
   - Filter once after fetch
   - Cache filtered results
   - Reuse filtered data

4. **Network Efficiency**
   - Debounce auto-sync (1 second)
   - Batch pending changes
   - Minimize API calls

5. **Memory Management**
   - Clean up listeners on unmount
   - Use `isMounted` ref
   - Cancel in-flight requests

---

## 🔄 Sync Workflow Diagram

```
┌─────────────┐
│ App Starts  │
└──────┬──────┘
       │
       ▼
┌─────────────────┐      Yes     ┌──────────────────┐
│  Online?        │─────────────▶│ Fetch from MongoDB│
└────────┬────────┘              └────────┬─────────┘
         │ No                             │
         │                                ▼
         │                      ┌──────────────────┐
         │                      │ Store Locally    │
         │                      │ + Timestamp      │
         │                      └────────┬─────────┘
         │                               │
         ▼                               │
┌──────────────────┐                     │
│ Load from Cache  │                     │
└────────┬─────────┘                     │
         │                               │
         └───────────────┬───────────────┘
                         │
                         ▼
                ┌────────────────┐
                │ Apply Permission│
                │ Filter          │
                └────────┬────────┘
                         │
                         ▼
                ┌────────────────┐
                │ Display Equipment│
                └────────────────┘
                         │
                         ▼
         ┌───────────────┴───────────────┐
         │                               │
         ▼                               ▼
┌─────────────────┐           ┌──────────────────┐
│ Connection Lost │           │ Connection Restored│
│ • Switch Offline│           │ • Auto-Sync       │
│ • Queue Changes │           │ • Sync Pending    │
└─────────────────┘           └──────────────────┘
```

---

## 📋 Summary

The improved Equipment screen now provides:
- ✅ Robust online/offline functionality
- ✅ Auto-sync when connection restored
- ✅ Manual sync via cloud icon
- ✅ Visual sync status indicators
- ✅ User-friendly toast notifications
- ✅ Permission-based equipment filtering
- ✅ Pending changes queue for offline edits
- ✅ Real-time network status monitoring
- ✅ Proper error handling with fallbacks
- ✅ Optimized performance

**Result:** Users can manage equipment seamlessly both online and offline, with automatic synchronization, clear visual feedback, and manual refresh capability via the cloud icon.

---

**Last Updated:** October 24, 2025  
**Version:** 2.0  
**Component:** Equipment_one.js

