import React, { useRef, useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  Alert,
} from 'react-native';
import SignatureCanvas from 'react-native-signature-canvas';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';

const { height, width } = Dimensions.get('window');

const DvirInspecSignature = ({ onSave, onGoBack, setActiveStep }) => {
  const signatureRef = useRef(null);
  const [isSignatureDone, setIsSignatureDone] = useState(false);
  const [signatureData, setSignatureData] = useState(null);

  const handleClear = () => {
    if (signatureRef.current) {
      signatureRef.current.clearSignature();
      setIsSignatureDone(false);
      setSignatureData(null);
      console.log('Signature cleared');
    } else {
      Alert.alert('Error', 'Signature canvas is not ready.');
    }
  };

  const handleDone = () => {
    if (signatureRef.current) {
      signatureRef.current.readSignature();
    } else {
      Alert.alert('Error', 'Signature canvas is not initialized.');
    }
  };

  const handleSave = () => {
    if (isSignatureDone && signatureData) {
      if (typeof onSave !== 'function') {
        Alert.alert('Error', 'Save function is not available.');
        console.log('onSave is not a function');
        return;
      }
      console.log('Saving signature data:', signatureData);
      onSave({ signatureStatus: true, signatureData });
      // Automatically navigate back after saving
      if (typeof onGoBack === 'function') {
        onGoBack(); // Use onGoBack for navigation
        console.log('Navigating back after save');
      } else {
        console.log('onGoBack is not a function, no navigation performed');
      }
    } else {
      Alert.alert('Warning', 'Please click "Done" to complete the signature first.');
      console.log('Save attempted but signature not done or no signature data');
    }
  };

  const handleOK = (signature) => {
    if (signature) {
      console.log('Signature captured via onOK:', signature);
      const processedSignature = signature.startsWith('data:image')
        ? signature
        : `data:image/png;base64,${signature}`;
      setSignatureData(processedSignature);
      setIsSignatureDone(true);
      Alert.alert('Info', 'Signature marked as complete. Click Save to proceed.');
      console.log('Processed signature data:', processedSignature);
    } else {
      Alert.alert('Warning', 'No signature detected. Please sign before completing.');
      console.log('No signature detected via onOK');
    }
  };

  return (
    <View style={styles.container}>
      <View style={styles.content}>
        <Text style={styles.label}>Inspector Signature</Text>
        <View style={styles.signatureContainer}>
          <SignatureCanvas
            ref={signatureRef}
            onOK={handleOK}
            descriptionText="Sign here, then click 'Done' to finalize"
            backgroundColor="#F5F5F5"
            penColor="#000000"
            canvasStyle={styles.signatureCanvas}
          />
        </View>

        <View style={styles.buttonContainer}>
          <TouchableOpacity onPress={handleClear} style={styles.clearButton}>
            <Text style={styles.clearButtonText}>Clear</Text>
          </TouchableOpacity>
          <TouchableOpacity
            onPress={handleDone}
            style={[styles.doneButton, isSignatureDone && styles.doneButtonDisabled]}
            disabled={isSignatureDone}
          >
            <Text style={styles.doneButtonText}>Done</Text>
          </TouchableOpacity>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.saveButton}>
            <TouchableOpacity
              onPress={handleSave}
              style={styles.saveButtonTouchable}
              disabled={!isSignatureDone}
            >
              <Text style={styles.saveButtonText}>Save</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  content: { flex: 1, alignItems: 'center', padding: 20 },
  label: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 20,
    alignSelf: 'flex-start',
  },
  signatureContainer: {
    width: width * 0.9,
    height: height * 0.3,
    backgroundColor: '#F5F5F5',
    borderRadius: 10,
    overflow: 'hidden',
    marginBottom: 30,
  },
  signatureCanvas: { width: '100%', height: '100%' },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: width * 0.9,
  },
  clearButton: {
    width: '30%',
    padding: 12,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#025383',
    alignItems: 'center',
    justifyContent: 'center',
  },
  clearButtonText: { color: '#025383', fontSize: 16, fontWeight: '600' },
  doneButton: {
    width: '30%',
    padding: 12,
    borderRadius: 5,
    backgroundColor: '#4CAF50',
    alignItems: 'center',
    justifyContent: 'center',
  },
  doneButtonDisabled: { backgroundColor: '#A9A9A9' },
  doneButtonText: { color: '#fff', fontSize: 16, fontWeight: '600' },
  saveButton: { width: '30%', borderRadius: 5, overflow: 'hidden' },
  saveButtonTouchable: { padding: 12, alignItems: 'center', justifyContent: 'center' },
  saveButtonText: { color: '#fff', fontSize: 16, fontWeight: '600' },
});

export default DvirInspecSignature;

