import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, ActivityIndicator, Platform, ToastAndroid, Modal } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import auth from '@react-native-firebase/auth';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import checklistData from './AllJsons.json'; // Path to your checklist JSON (ensure it contains the `checklist` array)
import { encode as b64encode } from 'base-64';
import { encode as utf8encode } from 'utf8';

const Checklist_library = () => {
  const [isUploading, setIsUploading] = useState(false);
  const [uploadStatus, setUploadStatus] = useState('Ready to upload');
  const [permissionsModalVisible, setPermissionsModalVisible] = useState(false);
  const [userPermissions, setUserPermissions] = useState({
    countries: [],
    projects: [],
    checklistPermissions: [],
  });
  const [firebaseCount, setFirebaseCount] = useState(0);
  const [localCount, setLocalCount] = useState(0);
  const [needsUpload, setNeedsUpload] = useState(false);

  // Validate checklist data structure
  const validateChecklistData = (data) => {
    if (!data || !Array.isArray(data.checklist)) {
      return { valid: false, error: 'Checklist data must contain an array under the "checklist" key' };
    }
    for (const form of data.checklist) {
      if (
        !form.formTitle ||
        !form.inspectionItems ||
        !Array.isArray(form.inspectionItems) ||
        !form.inspectionMatrix ||
        !Array.isArray(form.inspectionMatrix)
      ) {
        return { valid: false, error: `Invalid form structure in ${form.formTitle || 'unknown form'}` };
      }
      for (const item of form.inspectionMatrix) {
        if (!item.item || !Array.isArray(item.checks) || item.checks.length !== 31) {
          return { valid: false, error: `Invalid inspection matrix in ${form.formTitle}` };
        }
      }
    }
    return { valid: true };
  };

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  // Show toast notifications
  const showToast = (type, text1, text2) => {
    if (Platform.OS === 'android') {
      ToastAndroid.show(`${text1}: ${text2}`, ToastAndroid.LONG);
    } else {
      Toast.show({
        type,
        text1,
        text2,
        position: 'bottom',
      });
    }
  };

  // Fetch checklist count from Firebase
  const fetchFirebaseCount = async () => {
    try {
      const isOnline = await checkNetworkStatus();
      if (!isOnline) {
        setFirebaseCount(0);
        return;
      }

      const checklistRef = database().ref('/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson');
      const snapshot = await checklistRef.once('value');
      const data = snapshot.val();
      
      if (data && typeof data === 'object') {
        const count = Object.keys(data).length;
        setFirebaseCount(count);
        console.log('Firebase checklist count:', count);
      } else {
        setFirebaseCount(0);
      }
    } catch (error) {
      console.error('Error fetching Firebase count:', error);
      setFirebaseCount(0);
    }
  };

  // Fetch user permissions from Firebase
  const fetchUserPermissions = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      const userSnapshot = await database()
        .ref(`/Globalusers/admins/${userUid}`)
        .once('value');
      const userData = userSnapshot.val();

      if (!userData) {
        throw new Error('User data not found.');
      }

      const checklistModule = userData.modules?.find((mod) => mod.module === 'checklist');
      const permissions = checklistModule?.permissions || [];

      setUserPermissions({
        countries: userData.countries || [],
        projects: userData.projects || [],
        checklistPermissions: permissions,
      });

      if (!permissions.includes('upload')) {
        throw new Error('User lacks permission to upload checklists.');
      }
    } catch (error) {
      console.error('Error fetching user permissions:', error);
      showToast('error', 'Error', error.message || 'Failed to fetch user permissions.');
      setUserPermissions({
        countries: [],
        projects: [],
        checklistPermissions: [],
      });
    }
  };

  // Upload checklist data to Firebase
  const uploadToFirebase = async () => {
    setIsUploading(true);
    setUploadStatus('Uploading...');

    try {
      const user = auth().currentUser;
      if (!user || user.email !== '2@gmail.com') {
        throw new Error('Please log in with 2@gmail.com');
      }

      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User authentication data missing');
      }

      const isOnline = await checkNetworkStatus();
      if (!isOnline) {
        setUploadStatus('Failed: No internet connection');
        showToast('info', 'Offline', 'Data will be uploaded when online');
        await AsyncStorage.setItem('pendingChecklistUpload', JSON.stringify(checklistData));
        setIsUploading(false);
        return;
      }

      // Validate checklist data
      const validation = validateChecklistData(checklistData);
      if (!validation.valid) {
        throw new Error(validation.error);
      }

      console.log('Uploading checklist data:', checklistData.checklist.length, 'forms');

      // Firebase Realtime Database path - SAME PATH AS Daily_Ins_InfoStep1.js
      const dbRef = database().ref('/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson');

      // Convert array to object with unique IDs as keys (SAME FORMAT AS Daily_Ins_InfoStep1.js EXPECTS)
      const checklistObject = {};
      checklistData.checklist.forEach((checklist, index) => {
        // Generate unique ID based on form title and timestamp
        const timestamp = Date.now();
        const formTitleSlug = checklist.formTitle
          .replace(/[^a-zA-Z0-9]/g, '_')
          .toLowerCase()
          .substring(0, 20);
        const uniqueId = `${formTitleSlug}_${timestamp}_${index}`;
        checklistObject[uniqueId] = {
          ...checklist,
          id: uniqueId,
          uploadedAt: new Date().toISOString(),
          uploadedBy: userUid
        };
      });

      // Upload to Realtime Database
      await dbRef.set(checklistObject);
      console.log('Data uploaded to Realtime Database');

      // Upload to Firebase Storage
      const storageRef = storage().ref('DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson/AllJsons.json');
      const jsonString = JSON.stringify(checklistData.checklist);
      const base64String = b64encode(utf8encode(jsonString));
      await storageRef.putString(base64String, 'base64', {
        contentType: 'application/json',
      });
      const downloadUrl = await storageRef.getDownloadURL();
      console.log('Data uploaded to Storage, download URL:', downloadUrl);

      // Update Realtime Database with download URL
      await dbRef.update({ downloadUrl });

      // Cache locally
      await AsyncStorage.setItem('checklist_data', JSON.stringify(checklistData.checklist));

      // Refresh Firebase count after upload
      await fetchFirebaseCount();
      setNeedsUpload(false);

      setUploadStatus('Success: Data uploaded to Firebase');
      showToast('success', 'Success', 'Data uploaded to Realtime Database and Storage');
      await AsyncStorage.removeItem('pendingChecklistUpload');
    } catch (error) {
      console.error('Error uploading to Firebase:', error);
      setUploadStatus(`Failed: ${error.message}`);
      showToast('error', 'Error', `Failed to upload data: ${error.message}`);
      await AsyncStorage.setItem('pendingChecklistUpload', JSON.stringify(checklistData));
    } finally {
      setIsUploading(false);
    }
  };

  // Check for pending uploads and initialize on mount
  useEffect(() => {
    const initialize = async () => {
      await fetchUserPermissions();
      await fetchFirebaseCount();
      
      // Set local count
      const localCount = checklistData?.checklist?.length || 0;
      setLocalCount(localCount);
      
      // Check if upload is needed
      setNeedsUpload(localCount > firebaseCount);

      const pendingData = await AsyncStorage.getItem('pendingChecklistUpload');
      if (pendingData && (await checkNetworkStatus())) {
        setIsUploading(true);
        setUploadStatus('Syncing pending upload...');
        try {
          const data = JSON.parse(pendingData);
          const validation = validateChecklistData({ checklist: data });
          if (!validation.valid) {
            throw new Error(validation.error);
          }

          console.log('Syncing pending data:', data.length, 'forms');

          // Firebase Realtime Database path - SAME PATH AS Daily_Ins_InfoStep1.js
          const dbRef = database().ref('/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson');

          // Convert array to object with unique IDs as keys (SAME FORMAT AS Daily_Ins_InfoStep1.js EXPECTS)
          const checklistObject = {};
          data.forEach((checklist, index) => {
            // Generate unique ID based on form title and timestamp
            const timestamp = Date.now();
            const formTitleSlug = checklist.formTitle
              .replace(/[^a-zA-Z0-9]/g, '_')
              .toLowerCase()
              .substring(0, 20);
            const uniqueId = `${formTitleSlug}_${timestamp}_${index}`;
            checklistObject[uniqueId] = {
              ...checklist,
              id: uniqueId,
              uploadedAt: new Date().toISOString(),
              uploadedBy: userUid
            };
          });

          // Upload to Realtime Database
          await dbRef.set(checklistObject);
          console.log('Pending data uploaded to Realtime Database');

          // Upload to Firebase Storage
              const storageRef = storage().ref('/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson/AllJsons.json');
          const jsonString = JSON.stringify(data);
          const base64String = b64encode(utf8encode(jsonString));
          await storageRef.putString(base64String, 'base64', {
            contentType: 'application/json',
          });
          const downloadUrl = await storageRef.getDownloadURL();
          console.log('Pending data uploaded to Storage, download URL:', downloadUrl);

          // Update Realtime Database with download URL
          await dbRef.update({ downloadUrl });

          // Cache locally
          await AsyncStorage.setItem('checklist_data', JSON.stringify(data));

          setUploadStatus('Success: Pending data synced');
          showToast('success', 'Success', 'Pending data uploaded to Firebase');
          await AsyncStorage.removeItem('pendingChecklistUpload');
        } catch (error) {
          console.error('Error syncing pending upload:', error);
          setUploadStatus(`Failed to sync: ${error.message}`);
          showToast('error', 'Error', `Failed to sync pending data: ${error.message}`);
        } finally {
          setIsUploading(false);
        }
      }
    };

    initialize();

    const unsubscribe = auth().onAuthStateChanged((user) => {
      if (!user || user.email !== '2@gmail.com') {
        setUploadStatus('Please log in with amit.owninfotech@gmail.com');
      }
    });

    return () => unsubscribe();
  }, []);

  return (
    <View style={styles.container}>
      <LinearGradient colors={['#3481BC', '#025383']} style={styles.header}>
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => {}}>
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Checklist Upload</Text>
          <View style={styles.headerRight}>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={() => setPermissionsModalVisible(true)}
            >
              <MaterialIcons name="lock" size={24} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <View style={styles.mainContent}>
        <View style={styles.countContainer}>
          <Text style={styles.countText}>
            Local Checklists: {localCount}
          </Text>
          <Text style={styles.countText}>
            Firebase Checklists: {firebaseCount}
          </Text>
          {needsUpload && (
            <Text style={styles.needsUploadText}>
              ⚠️ {localCount - firebaseCount} checklists need to be uploaded
            </Text>
          )}
        </View>
        <Text style={styles.statusText}>Status: {uploadStatus}</Text>
        <TouchableOpacity
          style={[styles.uploadButton, isUploading && styles.disabledButton]}
          onPress={uploadToFirebase}
          disabled={isUploading}
        >
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <Text style={styles.buttonText}>
              {isUploading ? 'Uploading...' : 'Upload Checklist Data'}
            </Text>
            {isUploading && <ActivityIndicator color="#fff" style={styles.loader} />}
          </LinearGradient>
        </TouchableOpacity>
      </View>

      <Modal
        visible={permissionsModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setPermissionsModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.permissionsModalContent}>
            <Text style={styles.permissionsModalTitle}>User Permissions</Text>
            <View style={styles.permissionsSection}>
              <Text style={styles.permissionsLabel}>Countries</Text>
              <Text style={styles.permissionsValue}>
                {userPermissions.countries.length > 0
                  ? userPermissions.countries.join(', ')
                  : 'None'}
              </Text>
            </View>
            <View style={styles.permissionsSection}>
              <Text style={styles.permissionsLabel}>Projects</Text>
              <Text style={styles.permissionsValue}>
                {userPermissions.projects.length > 0
                  ? userPermissions.projects.join(', ')
                  : 'None'}
              </Text>
            </View>
            <View style={styles.permissionsSection}>
              <Text style={styles.permissionsLabel}>Checklist Permissions</Text>
              <Text style={styles.permissionsValue}>
                {userPermissions.checklistPermissions.length > 0
                  ? userPermissions.checklistPermissions.join(', ')
                  : 'None'}
              </Text>
            </View>
            <TouchableOpacity
              style={[styles.modalButton, styles.cancelButton]}
              onPress={() => setPermissionsModalVisible(false)}
            >
              <Text style={styles.modalButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    paddingTop: Platform.OS === 'ios' ? 50 : 30,
    paddingBottom: 20,
    paddingHorizontal: 16,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: 20,
    fontWeight: 'bold',
    marginLeft: 10,
  },
  iconButton: {
    padding: 5,
  },
  mainContent: {
    flex: 1,
    padding: 16,
    justifyContent: 'center',
    alignItems: 'center',
  },
  statusText: {
    fontSize: 16,
    color: '#333',
    marginBottom: 20,
    textAlign: 'center',
  },
  uploadButton: {
    borderRadius: 8,
    overflow: 'hidden',
  },
  disabledButton: {
    opacity: 0.6,
  },
  buttonGradient: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 24,
  },
  buttonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  loader: {
    marginLeft: 10,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  permissionsModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    alignItems: 'center',
  },
  permissionsModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#3481BC',
    marginBottom: 15,
  },
  permissionsSection: {
    width: '100%',
    marginBottom: 15,
  },
  permissionsLabel: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  permissionsValue: {
    fontSize: 16,
    color: '#333',
    textAlign: 'left',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 10,
    alignItems: 'center',
    borderRadius: 5,
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#f0f0f0',
  },
  modalButtonText: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  countContainer: {
    backgroundColor: '#f8f9fa',
    padding: 15,
    borderRadius: 8,
    marginBottom: 20,
    width: '100%',
  },
  countText: {
    fontSize: 16,
    color: '#025383',
    fontWeight: 'bold',
    marginBottom: 5,
    textAlign: 'center',
  },
  needsUploadText: {
    fontSize: 14,
    color: '#e74c3c',
    fontWeight: 'bold',
    textAlign: 'center',
    marginTop: 5,
  },
});

export default Checklist_library;





import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, ActivityIndicator, Platform, ToastAndroid, Modal } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import auth from '@react-native-firebase/auth';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import checklistData from './Checklistfile.json'; // Updated to import Checklistfile.json
import { encode as b64encode } from 'base-64';
import { encode as utf8encode } from 'utf8';

const Checklist_library = () => {
  const [isUploading, setIsUploading] = useState(false);
  const [uploadStatus, setUploadStatus] = useState('Ready to upload');
  const [permissionsModalVisible, setPermissionsModalVisible] = useState(false);
  const [userPermissions, setUserPermissions] = useState({
    countries: [],
    projects: [],
    checklistPermissions: [],
  });
  const [firebaseCount, setFirebaseCount] = useState(0);
  const [localCount, setLocalCount] = useState(0);
  const [needsUpload, setNeedsUpload] = useState(false);

  // Validate checklist data structure for Checklistfile.json format
  const validateChecklistData = (data) => {
    if (!data || !data.Checklistfile || !Array.isArray(data.Checklistfile.documents)) {
      return { valid: false, error: 'Checklist data must contain Checklistfile.documents array' };
    }
    
    for (const document of data.Checklistfile.documents) {
      if (
        !document.document_name ||
        !document.document_id ||
        !document.checklist ||
        typeof document.checklist !== 'object'
      ) {
        return { valid: false, error: `Invalid document structure in ${document.document_name || 'unknown document'}` };
      }
    }
    return { valid: true };
  };

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  // Show toast notifications
  const showToast = (type, text1, text2) => {
    if (Platform.OS === 'android') {
      ToastAndroid.show(`${text1}: ${text2}`, ToastAndroid.LONG);
    } else {
      Toast.show({
        type,
        text1,
        text2,
        position: 'bottom',
      });
    }
  };

  // Fetch checklist count from Firebase - Updated path
  const fetchFirebaseCount = async () => {
    try {
      const isOnline = await checkNetworkStatus();
      if (!isOnline) {
        setFirebaseCount(0);
        return;
      }

      const checklistRef = database().ref('/AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents');
      const snapshot = await checklistRef.once('value');
      const data = snapshot.val();
      
      if (data && typeof data === 'object') {
        const count = Object.keys(data).length;
        setFirebaseCount(count);
        console.log('Firebase checklist count:', count);
      } else {
        setFirebaseCount(0);
      }
    } catch (error) {
      console.error('Error fetching Firebase count:', error);
      setFirebaseCount(0);
    }
  };

  // Fetch user permissions from Firebase
  const fetchUserPermissions = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      const userSnapshot = await database()
        .ref(`/Globalusers/admins/${userUid}`)
        .once('value');
      const userData = userSnapshot.val();

      if (!userData) {
        throw new Error('User data not found.');
      }

      const checklistModule = userData.modules?.find((mod) => mod.module === 'checklist');
      const permissions = checklistModule?.permissions || [];

      setUserPermissions({
        countries: userData.countries || [],
        projects: userData.projects || [],
        checklistPermissions: permissions,
      });

      if (!permissions.includes('upload')) {
        throw new Error('User lacks permission to upload checklists.');
      }
    } catch (error) {
      console.error('Error fetching user permissions:', error);
      showToast('error', 'Error', error.message || 'Failed to fetch user permissions.');
      setUserPermissions({
        countries: [],
        projects: [],
        checklistPermissions: [],
      });
    }
  };

  // Upload checklist data to Firebase - Updated for new structure and path
  const uploadToFirebase = async () => {
    setIsUploading(true);
    setUploadStatus('Uploading...');

    try {
      const user = auth().currentUser;
      if (!user || user.email !== '2@gmail.com') {
        throw new Error('Please log in with 2@gmail.com');
      }

      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User authentication data missing');
      }

      const isOnline = await checkNetworkStatus();
      if (!isOnline) {
        setUploadStatus('Failed: No internet connection');
        showToast('info', 'Offline', 'Data will be uploaded when online');
        await AsyncStorage.setItem('pendingChecklistUpload', JSON.stringify(checklistData));
        setIsUploading(false);
        return;
      }

      // Validate checklist data
      const validation = validateChecklistData(checklistData);
      if (!validation.valid) {
        throw new Error(validation.error);
      }

      console.log('Uploading checklist data:', checklistData.Checklistfile.documents.length, 'documents');

      // Firebase Realtime Database path - Updated to new path
      const dbRef = database().ref('/AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents');

      // Convert array to object with unique IDs as keys
      const checklistObject = {};
      checklistData.Checklistfile.documents.forEach((document, index) => {
        // Generate unique ID based on document ID and timestamp
        const timestamp = Date.now();
        const documentIdSlug = document.document_id
          .replace(/[^a-zA-Z0-9]/g, '_')
          .toLowerCase()
          .substring(0, 20);
        const uniqueId = `${documentIdSlug}_${timestamp}_${index}`;
        checklistObject[uniqueId] = {
          ...document,
          id: uniqueId,
          uploadedAt: new Date().toISOString(),
          uploadedBy: userUid
        };
      });

      // Upload to Realtime Database
      await dbRef.set(checklistObject);
      console.log('Data uploaded to Realtime Database');

      // Upload to Firebase Storage
      const storageRef = storage().ref('AdminChecklistJson/GlobalChecklistJson/Checklistfile/Checklistfile.json');
      const jsonString = JSON.stringify(checklistData);
      const base64String = b64encode(utf8encode(jsonString));
      await storageRef.putString(base64String, 'base64', {
        contentType: 'application/json',
      });
      const downloadUrl = await storageRef.getDownloadURL();
      console.log('Data uploaded to Storage, download URL:', downloadUrl);

      // Update Realtime Database with download URL
      await dbRef.update({ downloadUrl });

      // Cache locally
      await AsyncStorage.setItem('checklist_data', JSON.stringify(checklistData.Checklistfile.documents));

      // Refresh Firebase count after upload
      await fetchFirebaseCount();
      setNeedsUpload(false);

      setUploadStatus('Success: Data uploaded to Firebase');
      showToast('success', 'Success', 'Data uploaded to Realtime Database and Storage');
      await AsyncStorage.removeItem('pendingChecklistUpload');
    } catch (error) {
      console.error('Error uploading to Firebase:', error);
      setUploadStatus(`Failed: ${error.message}`);
      showToast('error', 'Error', `Failed to upload data: ${error.message}`);
      await AsyncStorage.setItem('pendingChecklistUpload', JSON.stringify(checklistData));
    } finally {
      setIsUploading(false);
    }
  };

  // Check for pending uploads and initialize on mount
  useEffect(() => {
    const initialize = async () => {
      await fetchUserPermissions();
      await fetchFirebaseCount();
      
      // Set local count
      const localCount = checklistData?.Checklistfile?.documents?.length || 0;
      setLocalCount(localCount);
      
      // Check if upload is needed
      setNeedsUpload(localCount > firebaseCount);

      const pendingData = await AsyncStorage.getItem('pendingChecklistUpload');
      if (pendingData && (await checkNetworkStatus())) {
        setIsUploading(true);
        setUploadStatus('Syncing pending upload...');
        try {
          const data = JSON.parse(pendingData);
          const validation = validateChecklistData(data);
          if (!validation.valid) {
            throw new Error(validation.error);
          }

          console.log('Syncing pending data:', data.Checklistfile.documents.length, 'documents');

          // Firebase Realtime Database path - Updated to new path
          const dbRef = database().ref('/AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents');

          // Convert array to object with unique IDs as keys
          const checklistObject = {};
          data.Checklistfile.documents.forEach((document, index) => {
            // Generate unique ID based on document ID and timestamp
            const timestamp = Date.now();
            const documentIdSlug = document.document_id
              .replace(/[^a-zA-Z0-9]/g, '_')
              .toLowerCase()
              .substring(0, 20);
            const uniqueId = `${documentIdSlug}_${timestamp}_${index}`;
            checklistObject[uniqueId] = {
              ...document,
              id: uniqueId,
              uploadedAt: new Date().toISOString(),
              uploadedBy: userUid
            };
          });

          // Upload to Realtime Database
          await dbRef.set(checklistObject);
          console.log('Pending data uploaded to Realtime Database');

          // Upload to Firebase Storage
          const storageRef = storage().ref('AdminChecklistJson/GlobalChecklistJson/Checklistfile/Checklistfile.json');
          const jsonString = JSON.stringify(data);
          const base64String = b64encode(utf8encode(jsonString));
          await storageRef.putString(base64String, 'base64', {
            contentType: 'application/json',
          });
          const downloadUrl = await storageRef.getDownloadURL();
          console.log('Pending data uploaded to Storage, download URL:', downloadUrl);

          // Update Realtime Database with download URL
          await dbRef.update({ downloadUrl });

          // Cache locally
          await AsyncStorage.setItem('checklist_data', JSON.stringify(data.Checklistfile.documents));

          setUploadStatus('Success: Pending data synced');
          showToast('success', 'Success', 'Pending data uploaded to Firebase');
          await AsyncStorage.removeItem('pendingChecklistUpload');
        } catch (error) {
          console.error('Error syncing pending upload:', error);
          setUploadStatus(`Failed to sync: ${error.message}`);
          showToast('error', 'Error', `Failed to sync pending data: ${error.message}`);
        } finally {
          setIsUploading(false);
        }
      }
    };

    initialize();

    const unsubscribe = auth().onAuthStateChanged((user) => {
      if (!user || user.email !== '2@gmail.com') {
        setUploadStatus('Please log in with 2@gmail.com');
      }
    });

    return () => unsubscribe();
  }, []);

  return (
    <View style={styles.container}>
      <LinearGradient colors={['#3481BC', '#025383']} style={styles.header}>
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => {}}>
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Checklist Upload</Text>
          <View style={styles.headerRight}>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={() => setPermissionsModalVisible(true)}
            >
              <MaterialIcons name="lock" size={24} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <View style={styles.mainContent}>
        <View style={styles.countContainer}>
          <Text style={styles.countText}>
            Local Checklists: {localCount}
          </Text>
          <Text style={styles.countText}>
            Firebase Checklists: {firebaseCount}
          </Text>
          {needsUpload && (
            <Text style={styles.needsUploadText}>
              ⚠️ {localCount - firebaseCount} checklists need to be uploaded
            </Text>
          )}
        </View>
        <Text style={styles.statusText}>Status: {uploadStatus}</Text>
        <TouchableOpacity
          style={[styles.uploadButton, isUploading && styles.disabledButton]}
          onPress={uploadToFirebase}
          disabled={isUploading}
        >
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <Text style={styles.buttonText}>
              {isUploading ? 'Uploading...' : 'Upload Checklist Data'}
            </Text>
            {isUploading && <ActivityIndicator color="#fff" style={styles.loader} />}
          </LinearGradient>
        </TouchableOpacity>
      </View>

      <Modal
        visible={permissionsModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setPermissionsModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.permissionsModalContent}>
            <Text style={styles.permissionsModalTitle}>User Permissions</Text>
            <View style={styles.permissionsSection}>
              <Text style={styles.permissionsLabel}>Countries</Text>
              <Text style={styles.permissionsValue}>
                {userPermissions.countries.length > 0
                  ? userPermissions.countries.join(', ')
                  : 'None'}
              </Text>
            </View>
            <View style={styles.permissionsSection}>
              <Text style={styles.permissionsLabel}>Projects</Text>
              <Text style={styles.permissionsValue}>
                {userPermissions.projects.length > 0
                  ? userPermissions.projects.join(', ')
                  : 'None'}
              </Text>
            </View>
            <View style={styles.permissionsSection}>
              <Text style={styles.permissionsLabel}>Checklist Permissions</Text>
              <Text style={styles.permissionsValue}>
                {userPermissions.checklistPermissions.length > 0
                  ? userPermissions.checklistPermissions.join(', ')
                  : 'None'}
              </Text>
            </View>
            <TouchableOpacity
              style={[styles.modalButton, styles.cancelButton]}
              onPress={() => setPermissionsModalVisible(false)}
            >
              <Text style={styles.modalButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    paddingTop: Platform.OS === 'ios' ? 50 : 30,
    paddingBottom: 20,
    paddingHorizontal: 16,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: 20,
    fontWeight: 'bold',
    marginLeft: 10,
  },
  iconButton: {
    padding: 5,
  },
  mainContent: {
    flex: 1,
    padding: 16,
    justifyContent: 'center',
    alignItems: 'center',
  },
  statusText: {
    fontSize: 16,
    color: '#333',
    marginBottom: 20,
    textAlign: 'center',
  },
  uploadButton: {
    borderRadius: 8,
    overflow: 'hidden',
  },
  disabledButton: {
    opacity: 0.6,
  },
  buttonGradient: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 24,
  },
  buttonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  loader: {
    marginLeft: 10,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  permissionsModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    alignItems: 'center',
  },
  permissionsModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#3481BC',
    marginBottom: 15,
  },
  permissionsSection: {
    width: '100%',
    marginBottom: 15,
  },
  permissionsLabel: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  permissionsValue: {
    fontSize: 16,
    color: '#333',
    textAlign: 'left',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 10,
    alignItems: 'center',
    borderRadius: 5,
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#f0f0f0',
  },
  modalButtonText: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  countContainer: {
    backgroundColor: '#f8f9fa',
    padding: 15,
    borderRadius: 8,
    marginBottom: 20,
    width: '100%',
  },
  countText: {
    fontSize: 16,
    color: '#025383',
    fontWeight: 'bold',
    marginBottom: 5,
    textAlign: 'center',
  },
  needsUploadText: {
    fontSize: 14,
    color: '#e74c3c',
    fontWeight: 'bold',
    textAlign: 'center',
    marginTop: 5,
  },
});

export default Checklist_library;




































import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Dimensions,
  FlatList,
  ActivityIndicator,
  Modal,
  Animated,
  Image,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import NetInfo from '@react-native-community/netinfo';
import checklistData from './Checklistfile.json'; // Import Checklistfile.json for Inspections
import allJsonsData from './AllJsons.json'; // Import AllJsons.json for Daily Inspections

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const Checklist_library = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [checklists, setChecklists] = useState([]);
  const [filteredChecklists, setFilteredChecklists] = useState([]);
  const [showLoaderModal, setShowLoaderModal] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [userId, setUserId] = useState(null);
  const [isLoading, setIsLoading] = useState(true);
  const [activeTab, setActiveTab] = useState('inspections'); // 'inspections' or 'dailyInspections'
  const [counts, setCounts] = useState({
    inspections: { local: 0, firebase: 0, total: 0 },
    dailyInspections: { local: 0, firebase: 0, total: 0 }
  });

  // Animation setup for loader icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Start animation when loader modal is shown
  useEffect(() => {
    if (showLoaderModal) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [showLoaderModal]);

  // Check authentication and admin status
  useEffect(() => {
    const unsubscribe = auth().onAuthStateChanged(async user => {
      if (user) {
        setUserId(user.uid);
        try {
          await AsyncStorage.setItem('userEmail', user.email);
          const userSnapshot = await database()
            .ref('/Globalusers/admins')
            .orderByChild('email')
            .equalTo(user.email)
            .once('value');
          const userData = userSnapshot.val();
          if (!userData) {
            Toast.show({
              type: 'error',
              text1: 'Access Denied',
              text2: 'You are not authorized as an admin.',
            });
            navigation.replace('AdminLoginScreen');
            setUserId(null);
          }
        } catch (error) {
          console.error('Error verifying admin status:', error);
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Failed to verify admin status.',
          });
          navigation.replace('AdminLoginScreen');
          setUserId(null);
        }
      } else {
        setUserId(null);
        Toast.show({
          type: 'error',
          text1: 'Authentication Error',
          text2: 'Please log in to access checklists',
        });
        navigation.replace('AdminLoginScreen');
      }
    });
    return unsubscribe;
  }, [navigation]);

  // Network status listener
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener(state => {
      setIsOnline(state.isConnected);
    });
    return unsubscribe;
  }, []);

  // Load checklists from local JSON files and Firebase based on active tab
  useEffect(() => {
    if (!userId) return;

    const loadChecklists = async () => {
      try {
        setShowLoaderModal(true);
        setIsLoading(true);

        const storedDownloads = await AsyncStorage.getItem('downloadedChecklists');
        const downloadedItems = storedDownloads ? JSON.parse(storedDownloads) : {};

        let loadedChecklists = [];
        let localCount = 0;
        let firebaseCount = 0;

        if (activeTab === 'inspections') {
          // Load Inspections from Firebase only (no local data to avoid double rendering)
          if (isOnline) {
            try {
              console.log('Loading Firebase inspections...');
              const checklistRef = database().ref('AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents');
              const snapshot = await checklistRef.once('value');
              const firebaseData = snapshot.val();

              console.log('Firebase data received:', firebaseData);

              if (firebaseData && Object.keys(firebaseData).length > 0) {
                // Handle both array and object structures
                let firebaseDocuments = [];
                if (Array.isArray(firebaseData)) {
                  firebaseDocuments = firebaseData;
                } else {
                  // If it's an object, convert to array
                  firebaseDocuments = Object.values(firebaseData);
                }

                console.log('Firebase documents to process:', firebaseDocuments.length);

                const firebaseChecklists = firebaseDocuments.map((doc, index) => {
                  console.log('Processing Firebase document:', {
                    document_id: doc.document_id,
                    document_name: doc.document_name,
                    has_checklist: !!doc.checklist
                  });
                  
                  if (!doc.document_id || !doc.document_name || !doc.checklist) {
                    console.warn('Invalid Firebase checklist data for document_id:', doc.document_id);
                    return null;
                  }

                  let finalId = doc.document_id;
                  let suffix = 1;
                  while (loadedChecklists.some(c => c && c.id === finalId)) {
                    finalId = `${doc.document_id}-${suffix}`;
                    suffix++;
                  }

                  let sections = [];
                  
                  // Handle different checklist structures
                  if (doc.checklist.items && Array.isArray(doc.checklist.items)) {
                    sections = [{
                      sectionName: 'General Inspection',
                      subItems: doc.checklist.items.map((item, idx) => ({
                        key: `item_${idx}`,
                        itemName: item || `Item ${idx + 1}`,
                        inputType: 'Ok/Reject/N/A',
                        isInspectionRequired: false,
                      })),
                    }];
                  } else {
                    // Handle A, B, C structure like in local JSON
                    if (doc.checklist.A && Array.isArray(doc.checklist.A)) {
                      sections.push({
                        sectionName: 'Class A',
                        subItems: doc.checklist.A.map((item, idx) => ({
                          key: `A_${idx}`,
                          itemName: item || `Item A${idx + 1}`,
                          inputType: 'Ok/Reject/N/A',
                          isInspectionRequired: true,
                        })),
                      });
                    }
                    if (doc.checklist.B && Array.isArray(doc.checklist.B)) {
                      sections.push({
                        sectionName: 'Class B',
                        subItems: doc.checklist.B.map((item, idx) => ({
                          key: `B_${idx}`,
                          itemName: item || `Item B${idx + 1}`,
                          inputType: 'Ok/Reject/N/A',
                          isInspectionRequired: false,
                        })),
                      });
                    }
                    if (doc.checklist.C && Array.isArray(doc.checklist.C)) {
                      sections.push({
                        sectionName: 'Class C',
                        subItems: doc.checklist.C.map((item, idx) => ({
                          key: `C_${idx}`,
                          itemName: item || `Item C${idx + 1}`,
                          inputType: 'Ok/Reject/N/A',
                          isInspectionRequired: false,
                        })),
                      });
                    }
                    
                    // Handle other possible section names from the JSON
                    Object.keys(doc.checklist).forEach(sectionKey => {
                      if (sectionKey !== 'A' && sectionKey !== 'B' && sectionKey !== 'C' && sectionKey !== 'items') {
                        if (Array.isArray(doc.checklist[sectionKey])) {
                          sections.push({
                            sectionName: sectionKey,
                            subItems: doc.checklist[sectionKey].map((item, idx) => ({
                              key: `${sectionKey}_${idx}`,
                              itemName: item || `Item ${sectionKey}${idx + 1}`,
                              inputType: 'Ok/Reject/N/A',
                              isInspectionRequired: sectionKey === 'A',
                            })),
                          });
                        }
                      }
                    });
                  }

                  if (sections.length === 0) {
                    console.warn('No valid sections found for Firebase document_id:', doc.document_id);
                    return null;
                  }

                  const isDownloaded = !!downloadedItems[finalId];

                  return {
                    id: finalId,
                    uniqueId: `FIREBASE-CL-${doc.document_id}-${Date.now()}-${index}`,
                    title: doc.document_name || `Firebase Checklist ${index + 1}`,
                    document_id: doc.document_id,
                    document_name: doc.document_name || `Firebase Checklist ${index + 1}`,
                    description: 'Firebase General Checklist',
                    sections,
                    version_date: doc.version_date || 'N/A',
                    version: doc.version || '1.0',
                    approved_by: doc.approved_by || 'Unknown',
                    selected: false,
                    downloaded: isDownloaded,
                    type: 'inspections',
                    source: 'firebase',
                  };
                }).filter(checklist => checklist !== null);

                console.log('Firebase checklists processed:', firebaseChecklists.length);
                loadedChecklists = firebaseChecklists;
                firebaseCount = firebaseChecklists.length;
              } else {
                console.log('No Firebase data found or data is empty');
                Toast.show({
                  type: 'info',
                  text1: 'No Data',
                  text2: 'No inspection data found in Firebase',
                });
              }
            } catch (error) {
              console.error('Error loading Firebase inspections:', error);
              Toast.show({
                type: 'error',
                text1: 'Firebase Error',
                text2: `Failed to load Firebase inspections: ${error.message}`,
              });
            }
          } else {
            Toast.show({
              type: 'info',
              text1: 'Offline',
              text2: 'Cannot load inspections. Please check your internet connection.',
            });
          }

          // Update counts for inspections
          setCounts(prev => ({
            ...prev,
            inspections: {
              local: 0,
              firebase: firebaseCount,
              total: firebaseCount
            }
          }));

        } else {
          // Load Daily Inspections from Firebase only (no local data to avoid double rendering)
          if (isOnline) {
            try {
              console.log('Loading Firebase daily inspections...');
              const dailyChecklistRef = database().ref('DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson');
              const snapshot = await dailyChecklistRef.once('value');
              const firebaseData = snapshot.val();

              console.log('Firebase daily inspection data received:', firebaseData);

              if (firebaseData && Object.keys(firebaseData).length > 0) {
                // Process Firebase daily inspection data - handle flat structure
                const firebaseEntries = Object.entries(firebaseData);
                const firebaseChecklists = firebaseEntries.map(([entryKey, entryData], index) => {
                  console.log('Processing daily inspection entry:', {
                    key: entryKey,
                    formTitle: entryData.formTitle,
                    has_inspectionItems: !!entryData.inspectionItems
                  });

                  if (!entryData.formTitle || !entryData.inspectionItems || !Array.isArray(entryData.inspectionItems)) {
                    console.warn('Invalid daily inspection data for entry:', entryKey);
                    return null;
                  }

                  const finalId = entryKey;
                  const isDownloaded = !!downloadedItems[finalId];

                  // Create sections from inspection items
                  const sections = [{
                    sectionName: 'Daily Inspection Items',
                    subItems: entryData.inspectionItems.map((item, idx) => ({
                      key: `item_${idx}`,
                      itemName: item || `Item ${idx + 1}`,
                      inputType: 'Ok/Reject/N/A',
                      isInspectionRequired: true,
                    })),
                  }];

                  return {
                    id: finalId,
                    uniqueId: `FIREBASE-DAILY-${entryKey}-${Date.now()}`,
                    title: entryData.formTitle || `Daily Inspection ${index + 1}`,
                    document_id: entryData.document?.number || entryKey,
                    document_name: entryData.formTitle || `Daily Inspection ${index + 1}`,
                    description: 'Firebase Daily Inspection Checklist',
                    sections,
                    version_date: entryData.document?.versionDate || 'N/A',
                    version: entryData.document?.version || '1.0',
                    approved_by: entryData.document?.approvedBy || 'Unknown',
                    selected: false,
                    downloaded: isDownloaded,
                    type: 'dailyInspections',
                    entryKey: entryKey,
                    source: 'firebase',
                    // Additional data from the structure
                    days: entryData.days || [],
                    inspectionMatrix: entryData.inspectionMatrix || [],
                    machineDetails: entryData.machineDetails || {},
                    uploadedAt: entryData.uploadedAt,
                    uploadedBy: entryData.uploadedBy,
                  };
                }).filter(checklist => checklist !== null);

                loadedChecklists = firebaseChecklists;
                firebaseCount = firebaseChecklists.length;
                
                console.log('Firebase daily checklists processed:', firebaseChecklists.length);
              } else {
                console.log('No Firebase daily inspection data found');
                Toast.show({
                  type: 'info',
                  text1: 'No Data',
                  text2: 'No daily inspection data found in Firebase',
                });
              }
            } catch (error) {
              console.error('Error loading Firebase daily inspections:', error);
              Toast.show({
                type: 'error',
                text1: 'Firebase Error',
                text2: `Failed to load Firebase daily inspections: ${error.message}`,
              });
            }
          } else {
            Toast.show({
              type: 'info',
              text1: 'Offline',
              text2: 'Cannot load daily inspections. Please check your internet connection.',
            });
          }

          // Update counts for daily inspections
          setCounts(prev => ({
            ...prev,
            dailyInspections: {
              local: 0,
              firebase: firebaseCount,
              total: firebaseCount
            }
          }));
        }

        if (loadedChecklists.length > 0) {
          await AsyncStorage.setItem(`checklists_${activeTab}`, JSON.stringify(loadedChecklists));
        } else {
          Toast.show({
            type: 'info',
            text1: 'Info',
            text2: `No ${activeTab === 'inspections' ? 'inspections' : 'daily inspections'} found.`,
          });
        }

        const uniqueChecklists = Array.from(
          new Map(loadedChecklists.map(item => [item.id, item])).values()
        );

        setChecklists(uniqueChecklists);
        setFilteredChecklists(uniqueChecklists);
      } catch (error) {
        console.error('Error loading checklists:', error);
        let errorMessage = 'Failed to load checklists.';
        if (error.code === 'PERMISSION_DENIED') {
          errorMessage = 'Permission denied. Check Firebase security rules.';
        } else if (error.code === 'NETWORK_ERROR') {
          errorMessage = 'Network error. Check your internet connection.';
        }
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: errorMessage,
        });
      } finally {
        setShowLoaderModal(false);
        setIsLoading(false);
      }
    };

    loadChecklists();
  }, [userId, isOnline, activeTab]);

  // Handle checklist download
  const handleDownload = async checklist => {
    try {
      if (checklist.downloaded) {
        Toast.show({
          type: 'info',
          text1: 'Info',
          text2: `${checklist.title} is already downloaded.`,
        });
        return;
      }

      setShowLoaderModal(true);

      const stored = await AsyncStorage.getItem('downloadedChecklists');
      const downloadedItems = stored ? JSON.parse(stored) : {};
      downloadedItems[checklist.id] = true;
      await AsyncStorage.setItem(`checklist_${checklist.id}`, JSON.stringify(checklist));
      await AsyncStorage.setItem('downloadedChecklists', JSON.stringify(downloadedItems));

      if (isOnline) {
        await database()
          .ref(`users/${userId}/downloadedChecklists/${checklist.id}`)
          .set(checklist);
      }

      const updatedChecklists = checklists.map(item =>
        item.id === checklist.id ? { ...item, downloaded: true } : item
      );
      setChecklists(updatedChecklists);
      setFilteredChecklists(
        updatedChecklists.filter(
          item =>
            item.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
            item.description.toLowerCase().includes(searchQuery.toLowerCase())
        )
      );

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: `${checklist.title} downloaded successfully.`,
      });
    } catch (error) {
      console.error('Error downloading checklist:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to download checklist.',
      });
    } finally {
      setShowLoaderModal(false);
    }
  };

  // Handle search
  const handleSearch = text => {
    setSearchQuery(text);
    const filtered = checklists.filter(
      checklist =>
        checklist.title.toLowerCase().includes(text.toLowerCase()) ||
        checklist.description.toLowerCase().includes(text.toLowerCase())
    );
    setFilteredChecklists(filtered);
  };

  // Handle checklist selection
  const toggleSelection = id => {
    const updatedChecklists = checklists.map(checklist =>
      checklist.id === id ? { ...checklist, selected: !checklist.selected } : checklist
    );
    setChecklists(updatedChecklists);
    setFilteredChecklists(
      updatedChecklists.filter(
        item =>
          item.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
          item.description.toLowerCase().includes(searchQuery.toLowerCase())
      )
    );
    // Persist selection to AsyncStorage
    const selectedChecklists = updatedChecklists.filter(item => item.selected);
    AsyncStorage.setItem('selectedChecklists', JSON.stringify(selectedChecklists.map(item => item.id)));
  };

  // Handle tab change
  const handleTabChange = (tab) => {
    setActiveTab(tab);
    setSearchQuery('');
    setFilteredChecklists([]);
  };

  // Debug function to inspect data
  const debugData = async () => {
    try {
      console.log('=== DEBUG: Current Data ===');
      console.log('Active tab:', activeTab);
      console.log('Total checklists:', checklists.length);
      console.log('Local count:', counts[activeTab].local);
      console.log('Firebase count:', counts[activeTab].firebase);
      console.log('Online status:', isOnline);
      
      // Show first few checklists
      checklists.slice(0, 3).forEach((item, index) => {
        console.log(`Checklist ${index + 1}:`, {
          id: item.id,
          document_id: item.document_id,
          title: item.title,
          source: item.source,
          sections_count: item.sections?.length || 0
        });
      });

      // Check Firebase data
      if (isOnline) {
        console.log('=== Checking Firebase Data ===');
        const checklistRef = database().ref('AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents');
        const snapshot = await checklistRef.once('value');
        const firebaseData = snapshot.val();
        console.log('Firebase data exists:', !!firebaseData);
        if (firebaseData) {
          console.log('Firebase data keys:', Object.keys(firebaseData));
          console.log('Firebase data type:', typeof firebaseData);
          console.log('Is array:', Array.isArray(firebaseData));
        }
      }

      Toast.show({
        type: 'info',
        text1: 'Debug Info',
        text2: 'Check console for detailed information',
      });
    } catch (error) {
      console.error('Debug error:', error);
      Toast.show({
        type: 'error',
        text1: 'Debug Error',
        text2: error.message,
      });
    }
  };

  const renderActions = checklist => {
    // Don't show action buttons for daily inspections
    if (checklist.type === 'dailyInspections') {
      return null;
    }

    return (
      <View style={styles.actionsRow}>
        <TouchableOpacity
          style={styles.actionButton}
          onPress={() =>
            navigation.navigate('InsreportHeader', {
              checklistId: checklist.id,
              checklistTitle: checklist.title,
              checklistData: checklist,
            })
          }
        >
          <Text style={styles.actionButtonText}>Start Inspection</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.viewButton}
          onPress={() => navigation.navigate('ChecklistViewfile', { checklistId: checklist.id })}
        >
          <Text style={styles.viewButtonText}>View</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.downloadButton}
          onPress={() => handleDownload(checklist)}
        >
          <MaterialIcons
            name={checklist.downloaded ? 'check-circle' : 'file-download'}
            size={20}
            color={checklist.downloaded ? '#28a745' : '#015185'}
          />
        </TouchableOpacity>
      </View>
    );
  };

  const renderItem = ({ item, index }) => (
    <View style={styles.cardWrapper}>
      <View style={styles.cardRow}>
        <View style={styles.cardLeft}>
          <View style={styles.circleNumber}>
            <Text style={styles.circleNumberText}>{index + 1}</Text>
          </View>
          {/* Source indicator */}
          <View style={[styles.sourceIndicator, { backgroundColor: item.source === 'firebase' ? '#28a745' : '#007bff' }]}>
            <Text style={styles.sourceIndicatorText}>{item.source === 'firebase' ? 'F' : 'L'}</Text>
          </View>
        </View>
        <View style={styles.cardCenter}>
          <Text style={styles.cardTitle}>{item.title}</Text>
          <Text style={styles.cardSubtitle}>{item.description}</Text>
          <Text style={styles.cardDocumentId}>Document ID: {item.document_id}</Text>
          <Text style={styles.cardUniqueId}>Unique ID: {item.uniqueId}</Text>
          <Text style={styles.cardDetail}>Version: {item.version}</Text>
          <Text style={styles.cardDetail}>Date: {item.version_date}</Text>
          <Text style={styles.cardDetail}>Approved by: {item.approved_by}</Text>
          <Text style={styles.cardSource}>Source: {item.source === 'firebase' ? 'Firebase' : 'Local'}</Text>
          
          {/* Show additional info for daily inspections */}
          {item.type === 'dailyInspections' && (
            <View style={styles.dailyInspectionInfo}>
              <Text style={styles.dailyInfoText}>Items: {item.sections?.[0]?.subItems?.length || 0}</Text>
              {item.uploadedAt && (
                <Text style={styles.dailyInfoText}>Uploaded: {new Date(item.uploadedAt).toLocaleDateString()}</Text>
              )}
            </View>
          )}
          
          {renderActions(item)}
        </View>
        <View style={styles.cardRight}>
          <TouchableOpacity onPress={() => toggleSelection(item.id)}>
            {item.selected ? (
              <MaterialIcons name="check" size={22} color="#015185" style={styles.checkIcon} />
            ) : (
              <View style={styles.checkIcon} />
            )}
          </TouchableOpacity>
        </View>
      </View>
    </View>
  );

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient colors={['#3481BC', '#025383']} style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle} numberOfLines={1} ellipsizeMode="tail">
              Checklist Library
            </Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={debugData} style={styles.debugButton}>
              <MaterialIcons name="bug-report" size={width * 0.05} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity>
              <MaterialIcons name="cloud-download" size={width * 0.06} color={isOnline ? '#fff' : '#888'} />
            </TouchableOpacity>
          </View>
        </View>
        <Text style={styles.subHeaderText}>
          Tap on list to preview items • {activeTab === 'inspections' ? 'Inspections' : 'Daily Inspections'}: {counts[activeTab].total} total
        </Text>

        {/* Summary Display */}
        {/* <View style={styles.summaryContainer}>
          <Text style={styles.summaryText}>
            Local: {counts[activeTab].local} | Firebase: {counts[activeTab].firebase} | Online: {isOnline ? 'Yes' : 'No'}
          </Text>
        </View> */}

        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search checklists..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.searchIcon} />
          </View>
        </View>
      </LinearGradient>

      {/* Tabs */}
      <View style={styles.tabContainer}>
        <TouchableOpacity
          style={[styles.tab, activeTab === 'inspections' && styles.activeTab]}
          onPress={() => handleTabChange('inspections')}
        >
          <Text style={[styles.tabText, activeTab === 'inspections' && styles.activeTabText]}>
            Inspections
          </Text>
          <View style={styles.countContainer}>
            <Text style={[styles.countText, activeTab === 'inspections' && styles.activeCountText]}>
              {counts.inspections.total}
            </Text>
            <Text style={[styles.countSubText, activeTab === 'inspections' && styles.activeCountSubText]}>
              (Firebase)
            </Text>
          </View>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.tab, activeTab === 'dailyInspections' && styles.activeTab]}
          onPress={() => handleTabChange('dailyInspections')}
        >
          <Text style={[styles.tabText, activeTab === 'dailyInspections' && styles.activeTabText]}>
            Daily Inspections
          </Text>
          <View style={styles.countContainer}>
            <Text style={[styles.countText, activeTab === 'dailyInspections' && styles.activeCountText]}>
              {counts.dailyInspections.total}
            </Text>
            <Text style={[styles.countSubText, activeTab === 'dailyInspections' && styles.activeCountSubText]}>
              (Firebase)
            </Text>
          </View>
        </TouchableOpacity>
      </View>

      {/* Custom Loader Modal */}
      <Modal
        transparent={true}
        visible={showLoaderModal}
        animationType="fade"
        onRequestClose={() => setShowLoaderModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalText}>Please Wait</Text>
            <View style={styles.syncIconDiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.modalTextAdd}>Processing Checklist Data</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Offline Indicator */}
      {!isOnline && (
        <View style={styles.offlineIndicator}>
          <Text style={styles.offlineText}>Offline Mode - Data may not be up to date</Text>
        </View>
      )}

      {/* Checklist List */}
      {isLoading ? (
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#015185" />
          <Text style={styles.loadingText}>Loading {activeTab === 'inspections' ? 'Inspections' : 'Daily Inspections'}...</Text>
        </View>
      ) : (
        <FlatList
          data={filteredChecklists}
          renderItem={renderItem}
          keyExtractor={item => item.id}
          contentContainerStyle={styles.mainContent}
          ListEmptyComponent={
            <Text style={styles.defaultText}>
              No {activeTab === 'inspections' ? 'inspections' : 'daily inspections'} found
            </Text>
          }
        />
      )}

      {/* Toast Component */}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    marginTop: 20,
    flex: 1,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
    flex: 1,
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
    marginTop: 20,
  },
  debugButton: {
    marginRight: width * 0.025,
  },
  subHeaderText: {
    color: '#fff',
    fontSize: width * 0.035,
    paddingHorizontal: width * 0.04,
    paddingTop: 5,
    paddingBottom: 10,
  },
  summaryContainer: {
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    paddingVertical: 8,
    paddingHorizontal: 16,
    borderRadius: 8,
    marginHorizontal: 16,
    marginBottom: 12,
    alignItems: 'center',
  },
  summaryText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 16,
    marginTop: 8,
    marginBottom: 12,
    backgroundColor: '#fff',
    borderRadius: 25,
    paddingHorizontal: 16,
    borderWidth: 1,
    borderColor: '#E0E0E0',
    height: 44,
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 25,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    fontSize: 16,
    color: '#222',
    backgroundColor: 'transparent',
    paddingVertical: 0,
  },
  searchIcon: {
    marginLeft: 8,
  },
  tabContainer: {
    flexDirection: 'row',
    backgroundColor: '#f8f9fa',
    paddingHorizontal: 16,
    paddingVertical: 8,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  tab: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 16,
    alignItems: 'center',
    borderRadius: 8,
    marginHorizontal: 4,
  },
  activeTab: {
    backgroundColor: '#015185',
  },
  tabText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#666',
  },
  activeTabText: {
    color: '#fff',
  },
  countContainer: {
    flexDirection: 'row',
    alignItems: 'baseline',
    marginTop: 4,
  },
  countText: {
    fontSize: 14,
    fontWeight: 'bold',
    color: '#015185',
  },
  activeCountText: {
    color: '#fff',
  },
  countSubText: {
    fontSize: 12,
    color: '#666',
    marginLeft: 4,
  },
  activeCountSubText: {
    color: '#888',
  },
  mainContent: {
    paddingHorizontal: 0,
    paddingVertical: 0,
    paddingBottom: 30,
  },
  defaultText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginVertical: 20,
  },
  cardWrapper: {
    backgroundColor: '#fff',
    borderTopWidth: 2,
    borderTopColor: '#fff',
    borderBottomWidth: 2,
    borderBottomColor: '#fff',
    marginBottom: 0,
  },
  cardRow: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    borderRadius: 12,
    marginHorizontal: 16,
    marginTop: 18,
    paddingVertical: 18,
    paddingHorizontal: 12,
    minHeight: 120,
  },
  cardLeft: {
    alignItems: 'center',
    justifyContent: 'center',
    width: 40,
  },
  cardCenter: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
  },
  cardRight: {
    alignItems: 'center',
    justifyContent: 'center',
    width: 48,
  },
  circleNumber: {
    width: 32,
    height: 32,
    borderRadius: 16,
    backgroundColor: '#015185',
    alignItems: 'center',
    justifyContent: 'center',
  },
  circleNumberText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 18,
  },
  checkIcon: {
    marginBottom: 8,
  },
  cardTitle: {
    fontSize: 15,
    fontWeight: 'bold',
    color: '#222',
    textAlign: 'center',
    marginTop: 8,
  },
  cardSubtitle: {
    fontSize: 13,
    color: '#888',
    textAlign: 'center',
    marginBottom: 5,
  },
  cardUniqueId: {
    fontSize: 11,
    color: '#666',
    textAlign: 'center',
    marginBottom: 2,
  },
  cardDetail: {
    fontSize: 11,
    color: '#666',
    textAlign: 'center',
    marginBottom: 2,
  },
  actionsRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 4,
  },
  actionButton: {
    backgroundColor: '#fff',
    borderRadius: 4,
    paddingHorizontal: 10,
    paddingVertical: 4,
    borderWidth: 1,
    borderColor: '#015185',
    marginHorizontal: 4,
  },
  actionButtonText: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 13,
  },
  viewButton: {
    backgroundColor: '#015185',
    borderRadius: 4,
    paddingHorizontal: 10,
    paddingVertical: 4,
    marginHorizontal: 4,
  },
  viewButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 13,
  },
  downloadButton: {
    backgroundColor: '#fff',
    borderRadius: 4,
    paddingHorizontal: 10,
    paddingVertical: 4,
    marginHorizontal: 4,
    borderWidth: 1,
    borderColor: '#015185',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  modalTextAdd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncIconDiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  offlineIndicator: {
    backgroundColor: '#f8d7da',
    padding: 10,
    alignItems: 'center',
  },
  offlineText: {
    color: '#721c24',
    fontSize: 14,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 20,
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
  },
  sourceIndicator: {
    position: 'absolute',
    top: 0,
    right: 0,
    backgroundColor: '#007bff', // Default to blue for local
    borderRadius: 10,
    width: 20,
    height: 20,
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#fff',
  },
  sourceIndicatorText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: 'bold',
  },
  cardDocumentId: {
    fontSize: 11,
    color: '#666',
    textAlign: 'center',
    marginBottom: 2,
  },
  cardSource: {
    fontSize: 11,
    color: '#666',
    textAlign: 'center',
    marginTop: 2,
  },
  dailyInspectionInfo: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: 8,
    width: '100%',
  },
  dailyInfoText: {
    fontSize: 11,
    color: '#666',
  },
});

export default Checklist_library;