import React, { useEffect, useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  Image,
  Alert,
  Modal,
  ScrollView,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';

const Checklist_home = () => {
  const screenWidth = Dimensions.get('window').width * 0.8; // 80% of screen width
  const navigation = useNavigation();
  const [permissionsModalVisible, setPermissionsModalVisible] = useState(false);
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    checklistPermissions: [],
    checklistModule: null,
    role: null,
  });
  const [isLoading, setIsLoading] = useState(false);
  const [hasChecklistAccess, setHasChecklistAccess] = useState(false);
  const [isProAdmin, setIsProAdmin] = useState(false);

  useEffect(() => {
    const checkAuthAndChecklists = async () => {
      try {
        // Check user authentication from AsyncStorage (MongoDB-based)
        const userDataString = await AsyncStorage.getItem('userData');
        if (!userDataString) {
          console.log('❌ No user data found in AsyncStorage');
          Alert.alert('Error', 'User not authenticated. Please log in again.');
          navigation.replace('AdminLoginScreen');
          return;
        }

        // Parse user data from AsyncStorage
        const data = JSON.parse(userDataString);
        console.log('👤 User data from AsyncStorage:', data);

        // Extract checklist permissions and module info
        const checklistModule = data.modules?.find(mod => mod.module === 'checklist');
        const checklistPermissions = checklistModule?.permissions || [];
        
        console.log('🔍 Checklist Module Debug:', {
          allModules: data.modules,
          checklistModule: checklistModule,
          checklistPermissions: checklistPermissions,
          hasChecklistModule: !!checklistModule,
          moduleStatus: checklistModule?.status || 'not found'
        });

        // Check if user has checklist module access
        const hasAccess = !!checklistModule && checklistModule.status !== 'disabled';
        setHasChecklistAccess(hasAccess);

        // Check if user is ProAdmin
        const userRole = data.role || data.roles?.[0] || '';
        const isProAdminUser = userRole === 'ProAdmin';
        setIsProAdmin(isProAdminUser);

        console.log('🔍 User Role Check:', {
          role: userRole,
          isProAdmin: isProAdminUser
        });

        setUserData({
          countries: data.countries || [],
          projects: data.projects || [],
          checklistPermissions: checklistPermissions,
          checklistModule: checklistModule,
          role: userRole,
        });

        // Check for local checklists in AsyncStorage
        const cachedChecklists = await AsyncStorage.getItem('checklists');
        if (!cachedChecklists) {
          console.log('📋 No cached checklists found, fetching from MongoDB...');
          // Fetch checklists from MongoDB API
          try {
            const response = await fetch('https://api.titandrillingzm.com:6004/checklists', {
              method: 'GET',
              headers: {
                'Content-Type': 'application/json',
              },
            });

            if (response.ok) {
              const result = await response.json();
              console.log('✅ Checklists fetched from MongoDB:', result);
              
              if (result.success && result.data && result.data.length > 0) {
                // Store checklists in AsyncStorage
                await AsyncStorage.setItem('checklists', JSON.stringify(result.data));
                console.log('💾 Checklists cached in AsyncStorage');
              } else {
                console.warn('⚠️ No checklists found in MongoDB');
              }
            } else {
              console.warn('❌ Failed to fetch checklists from MongoDB:', response.status);
            }
          } catch (fetchError) {
            console.error('❌ Error fetching checklists from MongoDB:', fetchError);
          }
        } else {
          console.log('✅ Using cached checklists from AsyncStorage');
        }
      } catch (error) {
        console.error('❌ Error checking auth or checklists:', error);
        Alert.alert('Error', 'Failed to initialize. Please try again.');
      }
    };

    checkAuthAndChecklists();
  }, [navigation]);

  const scale = (size) => (Dimensions.get('window').width / 375) * size;

  return (
    <View style={styles.container}>
      {/* StatusBar with Gradient */}
      <StatusBar barStyle="light-content" backgroundColor="#015185" />

      {/* Gradient Header */}
      <LinearGradient colors={['#015185', '#3481BC']} style={styles.header}>
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.headerLeft}
            onPress={() => navigation.goBack()}
          >
            <MaterialIcons name="chevron-left" size={scale(28)} color="#fff" />
            <Text style={styles.headerText}>Checklist</Text>
            {/* Module Status Indicator */}
            {userData.checklistModule && (
              <View style={[
                styles.moduleStatusBadge,
                { backgroundColor: hasChecklistAccess ? '#4CAF50' : '#FF5722' }
              ]}>
                <MaterialIcons 
                  name={hasChecklistAccess ? "check-circle" : "block"} 
                  size={scale(16)} 
                  color="#fff" 
                />
                <Text style={styles.moduleStatusText}>
                  {hasChecklistAccess ? 'Active' : 'Disabled'}
                </Text>
              </View>
            )}
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.headerRight}
            onPress={() => setPermissionsModalVisible(true)}
          >
            <MaterialIcons name="lock" size={scale(24)} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Module Access Warning */}
      {!hasChecklistAccess && (
        <View style={styles.accessWarningContainer}>
          <View style={styles.accessWarning}>
            <MaterialIcons name="warning" size={scale(24)} color="#FF5722" />
            <Text style={styles.accessWarningText}>
              Checklist module is not available for your account. Please contact your administrator.
            </Text>
          </View>
        </View>
      )}

      {/* Buttons Container */}
      <View style={styles.buttonContainer}>
        {/* Button 1: Create Custom Checklist - Only for ProAdmin */}
        {hasChecklistAccess && isProAdmin ? (
          <TouchableOpacity
            onPress={() => navigation.navigate("Create_Custom_c")}
            activeOpacity={0.85}
            style={styles.cardButton}
          >
            <LinearGradient 
              colors={["#015185", "#3481BC"]} 
              style={styles.gradientButton}
              start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
            >
              <View style={styles.buttonContent}>
                <Image
                  source={require('../../../Images/adminhome/checklist/solar_list-check-bold.png')}
                  style={styles.logomenu}
                />
                <Text style={styles.buttonText}>Create Custom Checklist</Text>
              </View>
            </LinearGradient>
          </TouchableOpacity>
        ) : (
          <View style={[styles.cardButton, { position: 'relative' }]}> 
            <LinearGradient 
              colors={["#015185", "#3481BC"]} 
              style={styles.gradientButton}
              start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
            >
              <View style={styles.buttonContent}>
                <Image
                  source={require('../../../Images/adminhome/checklist/solar_list-check-bold.png')}
                  style={styles.logomenu}
                />
                <Text style={styles.buttonText}>Create Custom Checklist</Text>
              </View>
              {/* Overlay */}
              <View style={styles.disabledOverlay}>
                <MaterialIcons name="lock" size={40} color="#fff" style={{ opacity: 0.9 }} />
              </View>
            </LinearGradient>
            <Text style={styles.disabledTooltip}>
              {!hasChecklistAccess 
                ? 'Checklist module is not available for your account.' 
                : !isProAdmin
                ? 'Only ProAdmin can create custom checklists.'
                : 'You don\'t have permission to create checklists.'
              }
            </Text>
          </View>
        )}

        {/* Button 2: Checklist Library */}
        {hasChecklistAccess ? (
          <TouchableOpacity
            onPress={() => navigation.navigate("Checklist_library")}
            activeOpacity={0.85}
            style={styles.cardButton}
          >
            <LinearGradient 
              colors={["#015185", "#3481BC"]} 
              style={styles.gradientButton}
              start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
            >
              <View style={styles.buttonContent}>
                <Image
                  source={require('../../../Images/adminhome/checklist/tabler_copy-check.png')}
                  style={styles.logomenu}
                />
                <Text style={styles.buttonText}>Checklist Library</Text>
              </View>
            </LinearGradient>
          </TouchableOpacity>
        ) : (
          <View style={[styles.cardButton, { position: 'relative' }]}> 
            <LinearGradient 
              colors={["#015185", "#3481BC"]} 
              style={styles.gradientButton}
              start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
            >
              <View style={styles.buttonContent}>
                <Image
                  source={require('../../../Images/adminhome/checklist/tabler_copy-check.png')}
                  style={styles.logomenu}
                />
                <Text style={styles.buttonText}>Checklist Library</Text>
              </View>
              {/* Overlay */}
              <View style={styles.disabledOverlay}>
                <MaterialIcons name="block" size={40} color="#fff" style={{ opacity: 0.9 }} />
              </View>
            </LinearGradient>
            <Text style={styles.disabledTooltip}>Checklist module is not available for your account.</Text>
          </View>
        )}

        {/* Button 3: Database Upload/Download */}
        {/* <TouchableOpacity
          onPress={() => navigation.navigate("ChecklistUploadNavigation")}
          activeOpacity={0.85}
          style={styles.cardButton}
        >
          <LinearGradient 
            colors={["#015185", "#3481BC"]} 
            style={styles.gradientButton}
            start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
          >
            <View style={styles.buttonContent}>
              <MaterialIcons name="cloud-sync" size={44} color="#fff" />
              <Text style={styles.buttonText}>Database Sync</Text>
            </View>
          </LinearGradient>
        </TouchableOpacity> */}

        {/* Button 4: Downloaded Checklist (always disabled, modern overlay) */}
        <View style={[styles.cardButton, { position: 'relative' }]}> 
          <LinearGradient 
            colors={["#015185", "#3481BC"]} 
            style={styles.gradientButton}
            start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
          >
            <View style={styles.buttonContent}>
              <Image
                source={require('../../../Images/adminhome/checklist/subway_download-2.png')}
                style={styles.logomenu}
              />
              <Text style={styles.buttonText}>Downloaded Checklist</Text>
            </View>
            {/* Overlay */}
            <View style={styles.disabledOverlay}>
              <View style={styles.comingSoonBadge}>
                <Text style={styles.comingSoonText}>Coming Soon</Text>
              </View>
            </View>
          </LinearGradient>
          <Text style={styles.disabledTooltip}>This feature is coming soon!</Text>
        </View>
      </View>

      {/* Permissions Modal */}
      <Modal
        visible={permissionsModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionsModalVisible(false)}
      >
        <View style={styles.permissionModalOverlay}>
          <View style={styles.permissionModalContent}>
            <Text style={styles.permissionModalTitle}>User Permissions</Text>
            
            <Text style={styles.permissionModalLabel}>User Role:</Text>
            <Text style={styles.permissionModalValue}>
              {userData.role || 'Not assigned'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Checklist Module Status:</Text>
            <Text style={styles.permissionModalValue}>
              {userData.checklistModule ? 
                (hasChecklistAccess ? 'Module Active' : 'Module Disabled') : 
                'Checklist module not found in your account'
              }
            </Text>
            
            <Text style={styles.permissionModalLabel}>Create Custom Checklist:</Text>
            <Text style={styles.permissionModalValue}>
              {isProAdmin ? 'Allowed (ProAdmin)' : 'Not Allowed (ProAdmin only)'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Checklist Permissions:</Text>
            <Text style={styles.permissionModalValue}>
              {userData.checklistPermissions.length > 0 ? userData.checklistPermissions.join(', ') : 'None'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Countries:</Text>
            <Text style={styles.permissionModalValue}>
              {userData.countries.length > 0 ? userData.countries.join(', ') : 'None'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Projects:</Text>
            <Text style={styles.permissionModalValue}>
              {userData.projects.length > 0 ? userData.projects.join(', ') : 'None'}
            </Text>
            
            <TouchableOpacity
              style={styles.permissionCloseButton}
              onPress={() => setPermissionsModalVisible(false)}
            >
              <Text style={styles.permissionCloseButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    height: 100,
    justifyContent: 'center',
    alignItems: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    width: '100%',
    paddingHorizontal: 10,
    top: 20,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    padding: 10,
  },
  headerText: {
    color: '#fff',
    fontSize: 20,
    fontWeight: '500',
  },
  buttonContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  gradientButton: {
    borderRadius: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.12,
    shadowRadius: 6,
    elevation: 3,
    paddingVertical: 0,
  },
  buttonContent: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 28,
  },
  buttonText: {
    color: '#fff',
    fontSize: 18,
    fontWeight: '600',
    textAlign: 'center',
    marginTop: 10,
    letterSpacing: 0.2,
  },
  logomenu: {
    width: 44,
    height: 44,
    resizeMode: 'contain',
    marginBottom: 0,
  },
  cardButton: {
    width: Dimensions.get('window').width * 0.8,
    marginBottom: 28,
    borderRadius: 16,
    backgroundColor: '#fff',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.12,
    shadowRadius: 6,
    elevation: 3,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    overflow: 'visible',
    alignSelf: 'center',
  },
  disabledOverlay: {
    ...StyleSheet.absoluteFillObject,
    backgroundColor: 'rgba(0,0,0,0.35)',
    justifyContent: 'center',
    alignItems: 'center',
    borderRadius: 16,
    zIndex: 2,
  },
  disabledTooltip: {
    color: '#888',
    fontSize: 13,
    textAlign: 'center',
    marginTop: 6,
    marginBottom: 8,
    fontWeight: '500',
  },
  comingSoonBadge: {
    backgroundColor: '#fff',
    borderRadius: 12,
    paddingHorizontal: 16,
    paddingVertical: 6,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.10,
    shadowRadius: 2,
    elevation: 2,
  },
  comingSoonText: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 15,
    letterSpacing: 0.5,
  },
  permissionModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  permissionModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 24,
    width: '80%',
    alignItems: 'flex-start',
    elevation: 5,
  },
  permissionModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 16,
    alignSelf: 'center',
    width: '100%',
    textAlign: 'center',
  },
  permissionModalLabel: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
    marginTop: 10,
  },
  permissionModalValue: {
    fontSize: 14,
    color: '#333',
    marginBottom: 4,
  },
  permissionCloseButton: {
    alignSelf: 'center',
    marginTop: 18,
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 24,
    paddingVertical: 8,
  },
  permissionCloseButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  // New styles for module status and access warning
  moduleStatusBadge: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#4CAF50',
    borderRadius: 12,
    paddingHorizontal: 8,
    paddingVertical: 4,
    marginLeft: 8,
  },
  moduleStatusText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    marginLeft: 4,
  },
  accessWarningContainer: {
    paddingHorizontal: 16,
    paddingVertical: 12,
  },
  accessWarning: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#FFF3E0',
    borderRadius: 8,
    padding: 12,
    borderLeftWidth: 4,
    borderLeftColor: '#FF5722',
  },
  accessWarningText: {
    color: '#E65100',
    fontSize: 14,
    fontWeight: '500',
    marginLeft: 8,
    flex: 1,
  },
});

export default Checklist_home;