import React, { useState, useRef, useEffect } from 'react';
import { View, Text, StyleSheet, TextInput, Animated, Platform } from 'react-native';
import { Dimensions } from 'react-native';

const { width } = Dimensions.get('window');

const AnimatedInputField = ({ label, value, onChangeText, isRequired = false, isMultiline = false }) => {
  const animatedValue = useRef(new Animated.Value(value ? 1 : 0)).current;
  const [isFocused, setIsFocused] = useState(!!value);

  useEffect(() => {
    if (value) {
      setIsFocused(true);
      Animated.timing(animatedValue, {
        toValue: 1,
        duration: 0,
        useNativeDriver: true,
      }).start();
    } else {
      setIsFocused(false);
      Animated.timing(animatedValue, {
        toValue: 0,
        duration: 0,
        useNativeDriver: true,
      }).start();
    }
  }, [value, animatedValue]);

  const translateY = animatedValue.interpolate({
    inputRange: [0, 1],
    outputRange: [0, -15], // Reduced translateY to match smaller input height
  });

  const scale = animatedValue.interpolate({
    inputRange: [0, 1],
    outputRange: [1, 0.8],
  });

  const handleFocus = () => {
    setIsFocused(true);
    Animated.timing(animatedValue, {
      toValue: 1,
      duration: 200,
      useNativeDriver: true,
    }).start();
  };

  const handleBlur = () => {
    if (!value) {
      setIsFocused(false);
      Animated.timing(animatedValue, {
        toValue: 0,
        duration: 200,
        useNativeDriver: true,
      }).start();
    }
  };

  return (
    <View style={styles.formGroup}>
      <View style={[styles.inputContainer, { borderColor: isFocused ? '#015185' : '#707173' }]}>
        <Animated.Text
          style={[
            styles.inputLabel,
            {
              transform: [{ translateY }, { scale }],
              color: isFocused || value ? '#015185' : '#707173',
            },
          ]}
        >
          {label} {isRequired && <Text style={styles.requiredStar}>*</Text>}
        </Animated.Text>
        <TextInput
          style={[styles.input, isMultiline && styles.multilineInput]}
          value={value}
          onChangeText={onChangeText}
          onFocus={handleFocus}
          onBlur={handleBlur}
          multiline={isMultiline}
        />
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  formGroup: {
    marginBottom: width * 0.04, // Slightly reduced margin to match compact design
  },
  inputContainer: {
    borderWidth: 1,
    borderRadius: width * 0.02,
    padding: width * 0.02, // Reduced padding to make the container more compact
    position: 'relative',
    minHeight: width * 0.1, // Reduced minHeight to fit smaller input box
  },
  inputLabel: {
    position: 'absolute',
    left: width * 0.02,
    top: width * 0.015, // Adjusted top position to align with smaller input
    fontSize: width * 0.035, // Slightly reduced font size for label
    backgroundColor: '#fff',
  },
  requiredStar: {
    color: 'red',
  },
  input: {
    fontSize: width * 0.035, // Slightly reduced font size for input text
    color: '#333',
    paddingLeft: width * 0.02,
    height: width * 0.08, // Reduced height to make input box more compact
    ...Platform.select({
      android: {
        paddingVertical: 0, // Remove vertical padding on Android to control height
        lineHeight: width * 0.08, // Match the height to avoid text clipping
        textAlignVertical: 'center', // Center the text vertically on Android
      },
      ios: {
        paddingVertical: width * 0.005, // Minimal padding for iOS
      },
    }),
  },
  multilineInput: {
    height: width * 0.15, // Reduced height for multiline input
    textAlignVertical: 'top',
    paddingLeft: width * 0.04,
    ...Platform.select({
      android: {
        paddingVertical: width * 0.005, // Consistent padding for multiline on Android
      },
    }),
  },
});

export default AnimatedInputField;