import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StyleSheet,
  Modal,
  Animated,
  Platform,
  Dimensions,
} from 'react-native';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Toast from 'react-native-toast-message';
import { MongoAPI } from '../ConfigMongo/MongoDBConfig';

// Get screen dimensions for responsive design
const { width } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;

// SyncModal Component
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Processing Logout</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages
const showToast = (type, title, message) => {
  Toast.show({
    type: type,
    text1: title,
    text2: message,
    position: 'bottom',
  });
};

const Admin_Account = () => {
  const navigation = useNavigation();
  const [userInfo, setUserInfo] = useState({
    email: '',
    name: '',
    role: '',
    uid: '',
    employeeNumber: '',
    department: ''
  });
  const [isLogoutModalVisible, setIsLogoutModalVisible] = useState(false);
  const [isProcessingModalVisible, setIsProcessingModalVisible] = useState(false);
  const fadeAnim = useRef(new Animated.Value(0)).current;
  const [loading, setLoading] = useState(false);

  // Auto-refresh user data from MongoDB
  const refreshUserData = async () => {
    setLoading(true);
    try {
      console.log('🔄 Auto-refreshing user data from MongoDB...');
      
      // Get userUid from AsyncStorage
      let userUid = await AsyncStorage.getItem('userUid');
      
      if (!userUid) {
        const userData = await AsyncStorage.getItem('userData');
        if (userData) {
          const user = JSON.parse(userData);
          userUid = user._id || user.id || user.uid;
          if (userUid) {
            await AsyncStorage.setItem('userUid', userUid);
          }
        }
      }

      if (!userUid) {
        console.log('❌ No userUid found');
        showToast('error', 'Error', 'No user found. Please log in again.');
        navigation.replace('AdminLoginScreen');
        return;
      }

      console.log('🔍 Fetching fresh user data for UID:', userUid);

      // Try to fetch fresh data from MongoDB
      try {
        const userResult = await MongoAPI.getUserById(userUid);
        
        if (userResult && (userResult.success || userResult.user || userResult.data)) {
          const userData = userResult.user || userResult.data;
          console.log('✅ Fresh user data received from MongoDB');
          
          // Update state with fresh data
          setUserInfo({
            email: (userData.email || '').trim(),
            name: (userData.name || '').trim(),
            role: (userData.role || '').trim(),
            uid: userData._id || userData.uid || userUid,
            employeeNumber: (userData.employeeNumber || '').trim(),
            department: (userData.department || '').trim()
          });
          
          // Update AsyncStorage with fresh data
          await AsyncStorage.setItem('userData', JSON.stringify(userData));
          console.log('💾 Updated AsyncStorage with fresh user data');
          
        } else {
          throw new Error('Invalid response from MongoDB');
        }
      } catch (mongoError) {
        console.log('⚠️ MongoDB fetch failed, using cached data:', mongoError.message);
        
        // Fallback to cached data
        const cachedData = await AsyncStorage.getItem('userData');
        if (cachedData) {
          const userData = JSON.parse(cachedData);
          console.log('📦 Using cached user data');
          
          setUserInfo({
            email: (userData.email || '').trim(),
            name: (userData.name || '').trim(),
            role: (userData.role || '').trim(),
            uid: userData._id || userData.uid || userUid,
            employeeNumber: (userData.employeeNumber || '').trim(),
            department: (userData.department || '').trim()
          });
        } else {
          throw new Error('No cached data available');
        }
      }

      // Start fade animation
      Animated.timing(fadeAnim, {
        toValue: 1,
        duration: 500,
        useNativeDriver: true,
      }).start();
      
    } catch (error) {
      console.error('❌ Error loading user data:', error);
      setUserInfo(prev => ({
        ...prev,
        email: (prev.email || '').trim(),
        name: (prev.name || '').trim(),
        role: (prev.role || '').trim(),
        employeeNumber: (prev.employeeNumber || '').trim(),
      }));
      showToast('error', 'Error', 'Failed to load user data');
    }
    setLoading(false);
  };

  // Auto-refresh on screen focus
  useFocusEffect(
    React.useCallback(() => {
      console.log('📱 Screen focused - refreshing user data...');
      refreshUserData();
    }, [])
  );

  const handleLogout = async () => {
    setIsLogoutModalVisible(false);
    setIsProcessingModalVisible(true);

    try {
      console.log('🔄 Starting MongoDB logout process...');
      
      // Update login status in MongoDB
      if (userInfo.uid) {
        try {
          await MongoAPI.updateUser(userInfo.uid, {
            loginStatus: false,
            lastLogout: new Date().toISOString(),
          });
          console.log('✅ Updated logout status in MongoDB');
        } catch (mongoError) {
          console.error('⚠️ MongoDB logout update failed:', mongoError);
          // Continue logout even if MongoDB update fails
        }
      }

      // Clear AsyncStorage
      try {
        await AsyncStorage.removeItem('TeamMemberLogined');
        await AsyncStorage.removeItem('ProAdminLogined');
        await AsyncStorage.removeItem('userUid');
        await AsyncStorage.removeItem('userEmail');
        await AsyncStorage.removeItem('userData');
        await AsyncStorage.removeItem('Loginsuccess-01');
        await AsyncStorage.removeItem('adminPassword');
        console.log('✅ Cleared AsyncStorage data');
      } catch (storageError) {
        console.error('❌ AsyncStorage clear error:', storageError);
        showToast('error', 'Error', 'Failed to clear local storage');
      }

      showToast('success', 'Success', 'Logged out successfully');

      // Add slight delay for iOS to ensure toast is visible
      if (Platform.OS === 'ios') {
        await new Promise((resolve) => setTimeout(resolve, 300));
      }

      navigation.replace('LoginScreens');
    } catch (error) {
      console.error('❌ Logout error:', error);
      showToast('error', 'Error', 'Failed to logout. Please try again.');
    } finally {
      setIsProcessingModalVisible(false);
    }
  };

  const handleOptionSelect = (optionText) => {
    switch (optionText) {
      case 'View Profile':
        navigation.navigate('ViewProfilepage');
        break;
      case 'Notice Board':
        navigation.navigate('Notice_BoardAC');
        break;
      case 'Contact Us':
        navigation.navigate('Conatct_US_AC');
        break;
      case 'About Us':
        navigation.navigate('AboutUsAC');
        break;
      default:
        showToast('error', 'Error', 'Unknown option selected.');
    }
  };

  return (
    <Animated.View style={[styles.container, { opacity: fadeAnim }]}>
      <SyncModal visible={loading} onClose={() => {}} />
      {/* User Section */}
      <View style={styles.userSection}>
        <Text style={styles.userLabel}>{userInfo.role || 'Admin'}</Text>
        <Text style={styles.email}>{userInfo.email || 'Loading...'}</Text>
        <Text style={styles.nameText}>{userInfo.name || 'Unknown User'}</Text>
        <Text style={styles.uidText}>UID: {userInfo.uid || 'Unknown'}</Text>
        <Text style={styles.employeeText}>Emp: {userInfo.employeeNumber || 'N/A'}</Text>
        <Text style={styles.departmentText}>Dept: {userInfo.department || 'N/A'}</Text>
        <TouchableOpacity
          style={styles.viewProfileButton}
          onPress={() => handleOptionSelect('View Profile')}
          accessibilityRole="button"
          accessibilityLabel="View Profile"
          accessibilityHint="Navigate to view your profile"
        >
          <Text style={styles.viewProfileText}>View Profile</Text>
        </TouchableOpacity>
      </View>

      {/* Options List */}
      <View style={styles.optionsContainer}>
        {[
          { icon: 'tune', text: 'Notice Board' },
          { icon: 'contact-mail', text: 'Contact Us' },
          { icon: 'info', text: 'About Us' },
        ].map((option, index) => (
          <TouchableOpacity
            key={index}
            style={styles.optionItem}
            onPress={() => handleOptionSelect(option.text)}
            accessibilityRole="button"
            accessibilityLabel={option.text}
            accessibilityHint={`Navigate to ${option.text} page`}
          >
            <MaterialIcons name={option.icon} size={scale(24)} color="#666" style={styles.optionIcon} />
            <Text style={styles.optionText}>{option.text}</Text>
            <MaterialIcons name="chevron-right" size={scale(16)} color="#015185" style={styles.arrowIcon} />
          </TouchableOpacity>
        ))}
      </View>

      {/* Logout Button */}
      <TouchableOpacity
        style={styles.logoutButton}
        onPress={() => setIsLogoutModalVisible(true)}
        accessibilityRole="button"
        accessibilityLabel="Logout"
        accessibilityHint="Log out of your account"
      >
        <Text style={styles.logoutText}>LOGOUT</Text>
      </TouchableOpacity>

      {/* Logout Confirmation Modal */}
      <Modal
        visible={isLogoutModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setIsLogoutModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Logout Confirmation</Text>
            <Text style={styles.modalMessage}>Are you sure you want to logout?</Text>
            <View style={styles.modalButtons}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={() => setIsLogoutModalVisible(false)}
                accessibilityRole="button"
                accessibilityLabel="Cancel Logout"
                accessibilityHint="Cancel the logout action"
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.confirmButton]}
                onPress={handleLogout}
                accessibilityRole="button"
                accessibilityLabel="Confirm Logout"
                accessibilityHint="Confirm and proceed with logout"
              >
                <Text style={styles.modalButtonText}>Yes</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Processing Modal */}
      <SyncModal
        visible={isProcessingModalVisible}
        onClose={() => setIsProcessingModalVisible(false)}
      />
      <Toast />
    </Animated.View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  userSection: {
    backgroundColor: '#015185',
    padding: scale(20),
    alignItems: 'center',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  userLabel: {
    color: '#fff',
    fontSize: scale(18),
    fontWeight: 'bold',
  },
  email: {
    color: '#fff',
    fontSize: scale(14),
    marginVertical: scale(4),
    fontWeight: '600',
  },
  nameText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
    marginVertical: scale(4),
  },
  uidText: {
    color: '#fff',
    fontSize: scale(12),
    marginVertical: scale(2),
    opacity: 0.8,
  },
  employeeText: {
    color: '#fff',
    fontSize: scale(12),
    marginVertical: scale(2),
    opacity: 0.8,
  },
  departmentText: {
    color: '#fff',
    fontSize: scale(12),
    marginVertical: scale(2),
    opacity: 0.8,
  },
  viewProfileButton: {
    borderWidth: 1,
    borderColor: '#fff',
    borderRadius: scale(8),
    paddingVertical: scale(10),
    paddingHorizontal: scale(20),
    marginTop: scale(12),
  },
  viewProfileText: {
    color: '#fff',
    fontSize: scale(14),
    fontWeight: '600',
  },
  optionsContainer: {
    paddingHorizontal: scale(20),
    paddingVertical: scale(10),
  },
  optionItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: scale(15),
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  optionIcon: {
    marginRight: scale(15),
  },
  optionText: {
    flex: 1,
    fontSize: scale(16),
    color: '#333',
    fontWeight: '500',
  },
  arrowIcon: {
    width: scale(16),
    height: scale(16),
  },
  logoutButton: {
    alignItems: 'center',
    paddingVertical: scale(15),
    marginTop: scale(20),
  },
  logoutText: {
    fontSize: scale(16),
    fontWeight: 'bold',
    color: '#FF0000',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    width: '80%',
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  modalTitle: {
    fontSize: scale(18),
    fontWeight: 'bold',
    color: '#333',
    marginBottom: scale(10),
  },
  modalMessage: {
    fontSize: scale(16),
    color: '#555',
    marginBottom: scale(20),
    textAlign: 'center',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    paddingVertical: scale(10),
    borderRadius: scale(8),
    marginHorizontal: scale(5),
    alignItems: 'center',
  },
  cancelButton: {
    backgroundColor: '#888',
  },
  confirmButton: {
    backgroundColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: scale(18),
    marginBottom: scale(10),
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: scale(8),
  },
  syncModalTextadd: {
    fontSize: scale(14),
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default Admin_Account;