
import React, { useState, useEffect, useCallback, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Alert,
  Image,
  TextInput,
  Platform,
  Modal,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AntDesign from 'react-native-vector-icons/AntDesign';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import Toast from 'react-native-toast-message';
import { Dimensions } from 'react-native';
import { useSafeAreaInsets } from 'react-native-safe-area-context';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const Hsemodule43step2 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const insets = useSafeAreaInsets();
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [sections, setSections] = useState([]);
  const [expandedSections, setExpandedSections] = useState({});
  const [selections, setSelections] = useState({});
  const [selectedFiles, setSelectedFiles] = useState({});
  const [numericInputs, setNumericInputs] = useState({});
  const [notes, setNotes] = useState({});
  const [countInputs, setCountInputs] = useState({});
  const [hseId, setHseId] = useState(null);
  const [isLoading, setIsLoading] = useState(true);
  const [showNoteModal, setShowNoteModal] = useState(false);
  const [currentNoteItem, setCurrentNoteItem] = useState(null);
  const [noteText, setNoteText] = useState('');
  const [showDocModal, setShowDocModal] = useState(false);
  const [docModalItemKey, setDocModalItemKey] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  useEffect(() => {
    loadHseData();
  }, []);

  useFocusEffect(
    useCallback(() => {
      if (hseId) {
        loadNotes();
        loadSelections();
      }
    }, [hseId])
  );

  const deepCleanObject = (obj) => {
    if (typeof obj !== 'object' || obj === null) return obj;
    if (Array.isArray(obj)) return obj.map(deepCleanObject);
    const cleaned = {};
    Object.keys(obj).forEach((key) => {
      const value = obj[key];
      cleaned[key] = typeof value === 'object' && value !== null ? deepCleanObject(value) : String(value || '');
    });
    return cleaned;
  };

  const loadHseData = async () => {
    try {
      setIsLoading(true);
      const savedData = await AsyncStorage.getItem('hseStep1');
      if (!savedData) throw new Error('No Step 1 data found');
      const data = JSON.parse(savedData);
      setHseId(data.id);
      setSelectedChecklist(deepCleanObject(data.selectedChecklist));
      await loadChecklistData(data.selectedChecklist, data.id);
    } catch (error) {
      console.error('Error loading HSE data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load HSE data.', position: 'bottom' });
      setSections([]);
    } finally {
      setIsLoading(false);
    }
  };

  const loadNotes = async () => {
    try {
      if (!hseId) return;
      const savedNotes = JSON.parse(await AsyncStorage.getItem(`hseNotes_${hseId}`) || '{}');
      setNotes(savedNotes);
    } catch (error) {
      console.error('Error loading notes:', error);
    }
  };

  const loadSelections = async () => {
    try {
      if (!hseId) return;
      const savedResults = JSON.parse(await AsyncStorage.getItem(`hseResults_${hseId}`) || '{}');
      setSelections(savedResults.selections || {});
      setSelectedFiles(savedResults.selectedFiles || {});
      setNumericInputs(savedResults.numericInputs || {});
      setCountInputs(savedResults.countInputs || {});
    } catch (error) {
      console.error('Error loading selections:', error);
    }
  };

  const loadChecklistData = async (checklist, currentHseId) => {
    try {
      if (!checklist?.checklist) {
        setSections([]);
        return;
      }
      const sectionEntries = Object.entries(checklist.checklist);
      const dynamicSections = sectionEntries
        .map(([sectionKey, sectionValue], idx) => {
          if (Array.isArray(sectionValue) && sectionValue.length > 0) {
            const subItems = sectionValue.map((item, i) => {
              if (typeof item === 'object' && item !== null) {
                const itemKey = Object.keys(item)[0];
                const options = item[itemKey];
                return {
                  key: `${checklist.document_id}_${sectionKey}_${i}`,
                  itemName: itemKey,
                  category: sectionKey,
                  inputType: 'Safe/Unsafe/Not Applicable',
                  sortOrder: i,
                  isInspectionRequired: false,
                  description: '',
                  instruction: '',
                  options,
                };
              }
              return null;
            }).filter(item => item !== null);

            if (subItems.length > 0) {
              return {
                key: `${checklist.document_id}_${sectionKey}`,
                name: sectionKey,
                subItems,
              };
            }
          }
          return null;
        })
        .filter(section => section && Array.isArray(section.subItems) && section.subItems.length > 0);

      setSections(dynamicSections);

      const initialExpanded = {};
      dynamicSections.forEach((section) => {
        initialExpanded[section.key] = false;
      });
      setExpandedSections(initialExpanded);

      const savedResults = JSON.parse(await AsyncStorage.getItem(`hseResults_${currentHseId}`) || '{}');
      const initialSelections = {};
      const initialFiles = {};
      const initialNumericInputs = {};
      const initialCountInputs = {};
      dynamicSections
        .flatMap((section) => section.subItems)
        .forEach((item) => {
          initialSelections[item.key] = savedResults.selections?.[item.key] || null;
          initialFiles[item.key] = savedResults.selectedFiles?.[item.key] || null;
          initialNumericInputs[item.key] = savedResults.numericInputs?.[item.key] || '';
          initialCountInputs[item.key] = savedResults.countInputs?.[item.key] || { safe: '', unsafe: '', notApplicable: false };
        });

      setSelections(initialSelections);
      setSelectedFiles(initialFiles);
      setNumericInputs(initialNumericInputs);
      setCountInputs(initialCountInputs);
    } catch (error) {
      console.error('Error loading checklist data:', error);
      setSections([]);
    }
  };

  const getOptionsForInputType = (inputType) => {
    switch (inputType) {
      case 'Safe/Unsafe/Not Applicable':
        return ['Safe', 'Unsafe', 'Not Applicable'];
      case 'Okay/Faulty/N/A':
        return ['Okay', 'Faulty', 'N/A'];
      case 'Ok/Reject/N/A':
        return ['Ok', 'Reject', 'N/A'];
      case 'Ok/Not Ok':
        return ['Ok', 'Not Ok'];
      case 'Numeric':
        return [];
      default:
        return inputType.includes('/') ? inputType.split('/').map(opt => opt.trim()).filter(opt => opt) : ['Safe', 'Unsafe', 'Not Applicable'];
    }
  };

  const saveHseResults = async (selectionsToSave, filesToSave, numericInputsToSave, countInputsToSave) => {
    try {
      const savedNotes = JSON.parse(await AsyncStorage.getItem(`hseNotes_${hseId}`) || '{}');
      const hseResults = {
        selections: deepCleanObject(selectionsToSave),
        selectedFiles: deepCleanObject(filesToSave),
        numericInputs: deepCleanObject(numericInputsToSave),
        countInputs: deepCleanObject(countInputsToSave),
        notes: deepCleanObject(savedNotes),
        sections: deepCleanObject(sections),
        checklistTitle: selectedChecklist?.document_name || '',
        unsafeItems: calculateUnsafeItems(),
        timestamp: new Date().toISOString(),
      };
      await AsyncStorage.setItem(`hseResults_${hseId}`, JSON.stringify(hseResults));
    } catch (error) {
      console.error('Error saving HSE results:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save assessment data.', position: 'bottom' });
    }
  };

  // Helper to save only the current checklist's user input fields in AsyncStorage
  const saveAssessmentData = async (assessmentData) => {
    try {
      await AsyncStorage.setItem('hseStep2', JSON.stringify(assessmentData));
    } catch (error) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save assessment data.', position: 'bottom' });
    }
  };

  const handleSelectOption = async (itemKey, option) => {
    setSelections((prev) => {
      const updatedSelections = { ...prev, [itemKey]: option };
      saveHseResults(updatedSelections, selectedFiles, numericInputs, countInputs);
      return updatedSelections;
    });
    Toast.show({ type: 'success', text1: 'Selected', text2: `Selected: ${option}`, position: 'bottom' });
  };

  const handleNumericInput = async (itemKey, value) => {
    setNumericInputs((prev) => {
      const updatedNumericInputs = { ...prev, [itemKey]: value.replace(/[^0-9]/g, '') };
      setSelections((prevSelections) => {
        const updatedSelections = { ...prevSelections, [itemKey]: value ? value : null };
        saveHseResults(updatedSelections, selectedFiles, updatedNumericInputs, countInputs);
        return updatedSelections;
      });
      return updatedNumericInputs;
    });
  };

  const handleCountInput = async (itemKey, type, value) => {
    setCountInputs((prev) => {
      const updatedCountInputs = {
        ...prev,
        [itemKey]: { ...prev[itemKey], [type]: value.replace(/[^0-9]/g, '') },
      };
      saveHseResults(selections, selectedFiles, numericInputs, updatedCountInputs);
      return updatedCountInputs;
    });
  };

  const handleFilePick = (itemKey) => {
    setDocModalItemKey(itemKey);
    setShowDocModal(true);
  };

  const handleNotePress = async (itemKey) => {
    setCurrentNoteItem(itemKey);
    setNoteText(notes[itemKey] || '');
    setShowNoteModal(true);
  };

  const saveNote = async () => {
    if (currentNoteItem) {
      const updatedNotes = { ...notes, [currentNoteItem]: noteText };
      setNotes(updatedNotes);
      await AsyncStorage.setItem(`hseNotes_${hseId}`, JSON.stringify(updatedNotes));
      setShowNoteModal(false);
      setCurrentNoteItem(null);
      setNoteText('');
    }
  };

  const pickImage = async (itemKey, source) => {
    const options = { mediaType: 'photo', maxWidth: 500, maxHeight: 500, quality: 0.8 };
    const callback = async (response) => {
      if (response.didCancel) {
        console.log('User cancelled image picker');
      } else if (response.errorCode) {
        console.error('Image Picker Error:', response.errorMessage);
        Toast.show({ type: 'error', text1: 'Error', text2: response.errorMessage || 'Failed to pick image', position: 'bottom' });
      } else {
        const uri = response.assets[0].uri;
        setSelectedFiles((prev) => {
          const updatedFiles = { ...prev, [itemKey]: uri };
          saveHseResults(selections, updatedFiles, numericInputs, countInputs);
          return updatedFiles;
        });
      }
    };
    if (source === 'camera') {
      await launchCamera(options, callback);
    } else {
      await launchImageLibrary(options, callback);
    }
  };

  const calculateUnsafeItems = () => {
    return sections
      .flatMap((section) => section.subItems)
      .reduce((count, item) => {
        const selection = selections[item.key];
        if (selection && ['Unsafe', 'Faulty', 'Reject', 'Not Ok'].includes(selection) && item.inputType !== 'Numeric') {
          return count + 1;
        }
        return count;
      }, 0);
  };

  const validateSelections = () => {
    const allItems = sections.flatMap((section) => section.subItems);
    if (allItems.length === 0) {
      Toast.show({ type: 'error', text1: 'No Items', text2: 'The selected checklist has no valid items.', position: 'bottom' });
      return false;
    }
    const hasAtLeastOneSelection = allItems.some((item) => {
      const ci = countInputs[item.key] || {};
      return (
        (ci.safe && ci.safe.trim() !== '') ||
        (ci.unsafe && ci.unsafe.trim() !== '') ||
        ci.notApplicable
      );
    });
    if (!hasAtLeastOneSelection) {
      Toast.show({ type: 'error', text1: 'Incomplete Assessment', text2: 'Please make at least one selection.', position: 'bottom' });
      return false;
    }
    return true;
  };

  const handleNext = async () => {
    if (!validateSelections()) return;
    try {
      await saveHseResults(selections, selectedFiles, numericInputs, countInputs);
      const step2DataToSave = {
        id: hseId,
        checklistId: selectedChecklist?.document_id,
        checklistName: selectedChecklist?.document_name,
        sections: deepCleanObject(sections),
        selections: deepCleanObject(selections),
        countInputs: deepCleanObject(countInputs),
        notes: deepCleanObject(notes),
        selectedFiles: deepCleanObject(selectedFiles),
        numericInputs: deepCleanObject(numericInputs),
        completedAt: new Date().toISOString(),
      };
      await saveAssessmentData(step2DataToSave);
      Toast.show({ type: 'success', text1: 'Assessment Saved', text2: 'Assessment data saved successfully.', position: 'bottom' });
      setActiveStep(3);
    } catch (error) {
      console.error('Error saving HSE results:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save assessment data.', position: 'bottom' });
    }
  };

  const handleBackPress = () => {
    setActiveStep(1);
  };

  const toggleSection = (sectionKey) => {
    setExpandedSections((prev) => ({ ...prev, [sectionKey]: !prev[sectionKey] }));
  };

  const renderItem = (item, index) => {
    const selectedFileUri = selectedFiles[item.key];
    const numericValue = numericInputs[item.key];
    const itemNote = notes[item.key] || '';
    const options = getOptionsForInputType(item.inputType);
    const currentCountInputs = countInputs[item.key] || { safe: '', unsafe: '', notApplicable: false };
    const isNA = !!currentCountInputs.notApplicable;

    const handleSafeInput = (value) => {
      const newValue = value.replace(/[^0-9]/g, '').slice(0, 4);
      setCountInputs((prev) => {
        const updated = { ...prev, [item.key]: { ...prev[item.key], safe: newValue, notApplicable: '' } };
        saveHseResults(selections, selectedFiles, numericInputs, updated);
        return updated;
      });
    };

    const handleUnsafeInput = (value) => {
      const newValue = value.replace(/[^0-9]/g, '').slice(0, 4);
      setCountInputs((prev) => {
        const updated = { ...prev, [item.key]: { ...prev[item.key], unsafe: newValue, notApplicable: '' } };
        saveHseResults(selections, selectedFiles, numericInputs, updated);
        return updated;
      });
    };

    const handleNotApplicable = () => {
      setCountInputs((prev) => {
        const updated = { ...prev, [item.key]: { safe: '', unsafe: '', notApplicable: prev[item.key]?.notApplicable ? '' : 'checked' } };
        saveHseResults(selections, selectedFiles, numericInputs, updated);
        return updated;
      });
    };

    return (
      <View key={item.key} style={styles.centeredItem}>
        <View style={styles.itemContainer}>
          <Text style={styles.itemText}>
            {index + 1}. {item.itemName}
            {item.isInspectionRequired && <Text style={styles.requiredText}> (Required)</Text>}
          </Text>
          {item.description && <Text style={styles.detailText}>Description: {item.description}</Text>}
          {item.instruction && <Text style={styles.detailText}>Instruction: {item.instruction}</Text>}
          {item.inputType === 'Safe/Unsafe/Not Applicable' ? (
            <View style={styles.safeUnsafeContainer}>
              <View style={styles.inputRow}>
                <Text style={styles.inputLabel}>Safe:</Text>
                <TextInput
                  style={[styles.countInput, { width: 70, opacity: isNA ? 0.5 : 1 }]}
                  value={currentCountInputs.safe}
                  onChangeText={handleSafeInput}
                  placeholder="0"
                  keyboardType="numeric"
                  editable={!isNA}
                  maxLength={4}
                />
                <Text style={[styles.inputLabel, { marginLeft: 25 }]}>Unsafe:</Text>
                <TextInput
                  style={[styles.countInput, { width: 70, opacity: isNA ? 0.5 : 1 }]}
                  value={currentCountInputs.unsafe}
                  onChangeText={handleUnsafeInput}
                  placeholder="0"
                  keyboardType="numeric"
                  editable={!isNA}
                  maxLength={4}
                />
              </View>
              <TouchableOpacity style={styles.notApplicableContainer} onPress={handleNotApplicable}>
                <View style={[
                  styles.checkbox,
                  { backgroundColor: isNA ? '#015185' : '#fff' }
                ]}>
                  {isNA && <MaterialIcons name="check" size={16} color="#fff" />}
                </View>
                <Text style={styles.notApplicableText}>Not Applicable</Text>
              </TouchableOpacity>
            </View>
          ) : item.inputType === 'Numeric' ? (
            <TextInput
              style={styles.numericInput}
              value={numericValue}
              onChangeText={(text) => handleNumericInput(item.key, text)}
              placeholder="Enter number"
              keyboardType="numeric"
            />
          ) : (
            <View style={styles.optionsContainer}>
              {options.map((option) => {
                const isSelected = selections[item.key] === option;
                return (
                  <TouchableOpacity
                    key={option}
                    style={[
                      styles.optionButton,
                      isSelected && styles.selectedOptionButton,
                      isSelected && (option === 'Safe' || option === 'Okay' || option === 'Ok') && styles.safeButton,
                      isSelected && (option === 'Unsafe' || option === 'Faulty' || option === 'Reject' || option === 'Not Ok') && styles.unsafeButton,
                      isSelected && option === 'Not Applicable' && styles.naButton,
                      isSelected && !['Safe', 'Okay', 'Ok', 'Unsafe', 'Faulty', 'Reject', 'Not Ok', 'Not Applicable'].includes(option) && styles.customOptionButton,
                    ]}
                    onPress={() => handleSelectOption(item.key, option)}
                  >
                    <Text style={[styles.optionText, isSelected && styles.optionTextSelected]}>
                      {option}
                    </Text>
                  </TouchableOpacity>
                );
              })}
            </View>
          )}
          <View style={styles.iconsContainer}>
            <TouchableOpacity style={styles.iconButton} onPress={() => handleNotePress(item.key)}>
              <MaterialIcons name="note-add" size={24} color="#333" />
            </TouchableOpacity>
            <TouchableOpacity style={styles.iconButton} onPress={() => handleFilePick(item.key)}>
              <MaterialIcons name="camera-alt" size={24} color="#333" />
            </TouchableOpacity>
          </View>
          {itemNote && <Text style={styles.noteText}>Note: {itemNote}</Text>}
        </View>
        {selectedFileUri && (
          <Image
            source={{ uri: selectedFileUri }}
            style={styles.selectedImage}
            onError={(e) => console.error('Image error:', e.nativeEvent.error)}
          />
        )}
        <View style={styles.itemBorder} />
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <Text style={styles.checklistTitle}>{selectedChecklist?.document_name || 'HSE Assessment'}</Text>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <View style={styles.headingContainer}>
          <Text style={styles.headingText}>Tap on section to expand</Text>
        </View>
        {isLoading ? (
          <View style={styles.loadingContainer}>
            <Text style={styles.loadingText}>Loading checklist data...</Text>
          </View>
        ) : sections.length > 0 ? (
          sections.map((section) => (
            <View key={section.key}>
              <TouchableOpacity style={styles.sectionHeader} onPress={() => toggleSection(section.key)}>
                <View style={styles.sectionHeaderContent}>
                  <AntDesign
                    name={expandedSections[section.key] ? 'caretup' : 'caretdown'}
                    size={20}
                    color={expandedSections[section.key] ? 'red' : 'black'}
                  />
                  <Text style={styles.sectionHeaderText}>{section.name}</Text>
                </View>
              </TouchableOpacity>
              {expandedSections[section.key] && (
                <View style={styles.sectionContent}>
                  {section.subItems.length > 0 ? (
                    section.subItems.map((item, index) => renderItem(item, index))
                  ) : (
                    <Text style={styles.emptySectionText}>No assessment items</Text>
                  )}
                </View>
              )}
            </View>
          ))
        ) : (
          <View style={styles.emptyContainer}>
            <Text style={styles.emptySectionText}>No sections available</Text>
            <Text style={styles.emptySubText}>Please check your checklist data</Text>
          </View>
        )}
      </ScrollView>
      <Modal
        visible={showNoteModal}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setShowNoteModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Note</Text>
            <TextInput
              style={styles.noteTextInput}
              value={noteText}
              onChangeText={setNoteText}
              placeholder="Enter your note here..."
              multiline={true}
              numberOfLines={6}
              textAlignVertical="top"
            />
            <View style={styles.modalButtons}>
              <TouchableOpacity style={styles.modalButton} onPress={() => setShowNoteModal(false)}>
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.modalButton, styles.saveButton]} onPress={saveNote}>
                <Text style={[styles.modalButtonText, styles.saveButtonText]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      <Modal
        visible={showDocModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowDocModal(false)}
      >
        <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' }}>
          <View style={{ backgroundColor: '#fff', borderRadius: 10, padding: 24, width: '80%', alignItems: 'center' }}>
            <Text style={{ fontSize: 18, fontWeight: 'bold', marginBottom: 20 }}>Attach Document</Text>
            <TouchableOpacity
              style={{ width: '100%', padding: 12, backgroundColor: '#015185', borderRadius: 6, marginBottom: 12, alignItems: 'center' }}
              onPress={() => { setShowDocModal(false); pickImage(docModalItemKey, 'camera'); }}
            >
              <Text style={{ color: '#fff', fontSize: 16 }}>Take Picture</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={{ width: '100%', padding: 12, backgroundColor: '#015185', borderRadius: 6, marginBottom: 12, alignItems: 'center' }}
              onPress={() => { setShowDocModal(false); pickImage(docModalItemKey, 'gallery'); }}
            >
              <Text style={{ color: '#fff', fontSize: 16 }}>Choose from Gallery</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={{ width: '100%', padding: 12, backgroundColor: '#015185', borderRadius: 6, marginBottom: 12, alignItems: 'center' }}
              onPress={() => {
                setShowDocModal(false);
                setCurrentNoteItem(docModalItemKey);
                setNoteText(notes[docModalItemKey] || '');
                setShowNoteModal(true);
              }}
            >
              <Text style={{ color: '#fff', fontSize: 16 }}>Add Text</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={{ width: '100%', padding: 12, backgroundColor: '#eee', borderRadius: 6, alignItems: 'center' }}
              onPress={() => setShowDocModal(false)}
            >
              <Text style={{ color: '#015185', fontSize: 16 }}>Cancel</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
      <View style={[styles.buttonContainer, { paddingBottom: Math.max(insets.bottom, 10) }]}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message="Processing" />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  mainContent: { flex: 1 },
  scrollContent: { paddingBottom: 80 },
  headingContainer: { paddingVertical: 10, marginHorizontal: '5%' },
  headingText: { fontSize: 20, fontWeight: '500', color: '#015185' },
  sectionHeader: { paddingHorizontal: 12, paddingVertical: 6 },
  sectionHeaderContent: { flexDirection: 'row', alignItems: 'center', backgroundColor: '#F2F2F2', paddingHorizontal: 12, paddingVertical: 10, borderRadius: 5 },
  sectionHeaderText: { fontSize: 16, fontWeight: '600', color: '#333', marginLeft: 10 },
  sectionContent: { backgroundColor: '#fff' },
  centeredItem: { width: '90%', alignSelf: 'center', marginBottom: 15 },
  itemContainer: { paddingBottom: 15 },
  itemText: { fontSize: 16, fontWeight: '500', color: '#333', marginBottom: 8 },
  requiredText: { fontSize: 14, color: '#e74c3c', fontWeight: '600' },
  detailText: { fontSize: 14, color: '#666', marginBottom: 4 },
  optionsContainer: { flexDirection: 'row', flexWrap: 'wrap' },
  optionButton: { paddingHorizontal: 12, paddingVertical: 8, marginHorizontal: 3, marginVertical: 2, borderRadius: 5, minWidth: 60, backgroundColor: '#ECF0F3' },
  selectedOptionButton: { backgroundColor: '#333' },
  safeButton: { backgroundColor: '#015185' },
  unsafeButton: { backgroundColor: '#FF4B2B' },
  naButton: { backgroundColor: '#2193b0' },
  customOptionButton: { backgroundColor: '#6c757d' },
  optionText: { fontSize: 14, color: '#333' },
  optionTextSelected: { color: '#fff', fontWeight: '600' },
  numericInput: { flex: 1, height: 40, borderWidth: 1, borderColor: '#ccc', borderRadius: 5, paddingHorizontal: 10, fontSize: 14, marginRight: 10 },
  iconsContainer: { flexDirection: 'row' },
  iconButton: { padding: 10 },
  itemBorder: { width: '80%', alignSelf: 'center', borderBottomWidth: 1, borderBottomColor: '#ddd', marginTop: 10 },
  selectedImage: { width: 40, height: 40, marginTop: 5, alignSelf: 'center' },
  noteText: { fontSize: 14, color: '#333', marginTop: 8, marginBottom: 8, fontStyle: 'italic' },
  buttonContainer: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingHorizontal: 15, paddingVertical: 15, backgroundColor: '#F2F2F2', borderTopWidth: 1, borderTopColor: '#ddd', position: 'absolute', bottom: 0, left: 0, right: 0, elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: -2 }, shadowOpacity: 0.1, shadowRadius: 2, paddingBottom: 0 },
  cancelButton: { flexDirection: 'row', alignItems: 'center', justifyContent: 'center', paddingHorizontal: 15, paddingVertical: 10 },
  nextButton: { flexDirection: 'row', alignItems: 'center', justifyContent: 'center', paddingHorizontal: 15, paddingVertical: 10, borderRadius: 5 },
  cancelButtonText: { color: '#015185', fontSize: 16, fontWeight: '600' },
  nextButtonText: { color: '#015185', fontSize: 16, fontWeight: '600' },
  bottomicon: { marginTop: 2 },
  bottomicon1: { marginTop: 4 },
  emptySectionText: { fontSize: 16, color: '#888', textAlign: 'center', padding: 20 },
  checklistTitle: { fontSize: 20, fontWeight: 'bold', color: '#015185', textAlign: 'center', marginTop: 18, marginBottom: 4 },
  loadingContainer: { flex: 1, justifyContent: 'center', alignItems: 'center', padding: 20 },
  loadingText: { fontSize: 18, color: '#555' },
  emptyContainer: { flex: 1, justifyContent: 'center', alignItems: 'center', padding: 20 },
  emptySubText: { fontSize: 14, color: '#888', marginTop: 10, textAlign: 'center' },
  countInput: { height: 40, borderWidth: 1, borderColor: '#ccc', borderRadius: 5, paddingHorizontal: 12, fontSize: 16, backgroundColor: '#fff', textAlign: 'center' },
  safeUnsafeContainer: { marginTop: 10, flexDirection: 'row', flexWrap: 'wrap', alignItems: 'center' },
  inputRow: { flexDirection: 'row', alignItems: 'center', marginBottom: 8, marginRight: 15 },
  inputLabel: { width: 60, fontSize: 15, fontWeight: '500', color: '#333', marginRight: 15 },
  notApplicableContainer: { flexDirection: 'row', alignItems: 'center', marginTop: 8 },
  checkbox: { width: 22, height: 22, borderWidth: 2, borderColor: '#015185', borderRadius: 4, marginRight: 8, justifyContent: 'center', alignItems: 'center' },
  notApplicableText: { color: '#015185', fontSize: 14, fontWeight: '500' },
  modalOverlay: { flex: 1, backgroundColor: 'rgba(0, 0, 0, 0.5)', justifyContent: 'center', alignItems: 'center' },
  modalContent: { backgroundColor: '#fff', borderRadius: 10, padding: 20, width: '90%', maxWidth: 400 },
  modalTitle: { fontSize: 18, fontWeight: '600', color: '#015185', marginBottom: 15, textAlign: 'center' },
  noteTextInput: { borderWidth: 1, borderColor: '#ccc', borderRadius: 8, padding: 12, fontSize: 16, minHeight: 120, textAlignVertical: 'top', marginBottom: 20 },
  modalButtons: { flexDirection: 'row', justifyContent: 'space-between' },
  modalButton: { flex: 1, paddingVertical: 12, paddingHorizontal: 20, borderRadius: 6, marginHorizontal: 5, backgroundColor: '#f0f0f0' },
  saveButton: { backgroundColor: '#015185' },
  modalButtonText: { fontSize: 16, fontWeight: '600', color: '#333', textAlign: 'center' },
  saveButtonText: { color: '#fff' },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default Hsemodule43step2;