import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  FlatList,
  Dimensions,
  StatusBar,
  TextInput,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import LinearGradient from 'react-native-linear-gradient';
import HseChecklists from './HSE43_44_45_126.json';
import HseChecklists41_42 from './HSE41_42.json';
import HseChecklists101_024 from './HSE101_024/HSE101_024Step3.json';
import HseChecklists020 from './HSE_020/HSE_020Json.json';
import HseChecklists071 from './HSE_071/HSE_071Json.json';
import HseChecklists077_078 from './HSE_077_078/HSE_077_078Json.json';
import { HSE_156Json } from './HSE_156/HSE_156Json.js';
import HseChecklists129 from './HSE_129/HSE_129_Json.json';
import HseChecklists072 from './HSE_072/HSE_072Json.json';
import HseChecklist131 from './HSE_131/HSE_131Json.json';
import HseChecklist034 from './HSE_034/HSE_034Json.json';
import HseChecklist158 from './HSE_158/HSE_158Json.json';
import HseChecklist051 from './HSE_051/HSE_051json.json';
import HseChecklist122 from './HSE_122/HSE_122Json.json';
import HseChecklist123 from './HSE_123/HSE_123json.json';
import HseChecklist124 from './HSE_124/HSE_124json.json';
import HseChecklist125 from './HSE_125/HSE_125json.json';
import HSE_132Json from './HSE_132/HSE_132_Json.js';
import HSE_155_JSON from './HSE_155/HSE_155_json.js';
import HSE_077_JSON from './HSE_077/HSE_077_json.json';
import HSE_085_JSON from './HSE-085/HSE_085_json.json';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

const HseSelectChecklsit = () => {
  const navigation = useNavigation();
  const [checklists, setChecklists] = useState([]);
  const [searchQuery, setSearchQuery] = useState('');
  const [filteredChecklists, setFilteredChecklists] = useState([]);
  const searchTimeoutRef = useRef(null);

  useEffect(() => {
    // Load checklists from all JSON files
    const allChecklists = [...HseChecklists, ...HseChecklists41_42];
    
    // Process HSE101_024 checklists to match the expected format
    const processedHse101_024 = HseChecklists101_024.map((item, index) => {
      // Handle both possible structures in the JSON
      const checklistData = item.Checklist_024_101 || item.document;
      if (checklistData) {
        return {
          document_name: checklistData.document_name || checklistData.title,
          document_id: checklistData.document_id,
          version: checklistData.version,
          version_date: checklistData.date,
          approved_by: checklistData.approved_by,
          title: checklistData.title,
          // Add sections based on the structure
          checklist: {
            'SAFETY_FEATURES': item.safety_features || [],
            'PPE_REQUIREMENTS': item.personal_protective_equipment || {},
            'NO_GO_AREAS': item.no_go_areas || {},
            'INDUCTION_CATEGORIES': item.induction_categories || {},
            'VISITOR_SIGN_OFF': item.visitor_sign_off || {},
            'WORKSHOP_RULES': item.workshop_area_rules || {}
          }
        };
      }
      return null;
    }).filter(Boolean); // Remove null items
    
    // Process HSE-020 checklist to match the expected format
    const processedHse020 = {
      document_name: HseChecklists020.document.document_name,
      document_id: HseChecklists020.document.document_id,
      version: HseChecklists020.document.version,
      version_date: HseChecklists020.document.version_date,
      approved_by: HseChecklists020.document.approved_by,
      title: HseChecklists020.document.title,
      checklist: {
        'OBSERVATION_DETAILS': HseChecklists020.observation_details || {},
        'REASON_FOR_PTO_VFL': HseChecklists020.reason_for_pto_vfl || {},
        'EVALUATION_CRITERIA': HseChecklists020.evaluation_criteria || {},
        'COMMENDATION_SECTION': HseChecklists020.commendation_section || {},
        'CORRECTION_SECTION': HseChecklists020.correction_section || {},
        'ACTION_ITEMS': HseChecklists020.action_items || {},
        'SIGNATURES': HseChecklists020.signatures || {}
      }
    };

    // Process HSE-071 checklist to match the expected format
    const processedHse071 = {
      document_name: HseChecklists071.document.name,
      document_id: HseChecklists071.document.id,
      version: HseChecklists071.document.version,
      version_date: HseChecklists071.document.versionDate,
      approved_by: HseChecklists071.document.approvedBy,
      title: HseChecklists071.report.title,
      checklist: {
        'FIELDS': HseChecklists071.report.fields || {},
        'INCOMING_SHIFT_REQUIREMENTS': HseChecklists071.report.incomingShiftRequirements || [],
        'SHIFT_CONCERN': HseChecklists071.report.shiftConcern || [],
        'EQUIPMENT_CHECKLIST': HseChecklists071.report.equipmentChecklist || [],
        'STANDARD_PROCEDURES': HseChecklists071.report.standardProcedures || []
      }
    };

    // Process HSE-156 checklist to match the expected format
    const processedHse156 = {
      document_name: HSE_156Json.safetyHarnessChecklist.metadata.documentName,
      document_id: HSE_156Json.safetyHarnessChecklist.metadata.documentId,
      version: HSE_156Json.safetyHarnessChecklist.metadata.version,
      version_date: HSE_156Json.safetyHarnessChecklist.metadata.versionDate,
      approved_by: HSE_156Json.safetyHarnessChecklist.metadata.approvedBy,
      title: HSE_156Json.safetyHarnessChecklist.title,
      checklist: {
        'INSPECTION_REGISTER': HSE_156Json.safetyHarnessChecklist.inspectionRegister || [],
        'DEVIATIONS': HSE_156Json.safetyHarnessChecklist.deviations || [],
        'CHECK_ITEMS': HSE_156Json.safetyHarnessChecklist.inspectionRegister?.[0]?.checkItems || {}
      }
    };

    // Process HSE-077_078 checklist to match the expected format
    const processedHse077_078 = {
      document_name: HseChecklists077_078.document.document_name,
      document_id: HseChecklists077_078.document.document_id,
      version: HseChecklists077_078.document.version,
      version_date: HseChecklists077_078.document.version_date,
      approved_by: HseChecklists077_078.document.approved_by,
      title: HseChecklists077_078.form.form_type,
      checklist: {
        'FORM_FIELDS': HseChecklists077_078.form || {},
        'INSTRUCTIONS': HseChecklists077_078.instructions || {}
      }
    };

    // Process HSE-129 checklist to match the expected format
    const processedHse129 = {
      document_name: HseChecklists129.document.document_name,
      document_id: HseChecklists129.document.document_id,
      version: HseChecklists129.document.version,
      version_date: HseChecklists129.document.version_date,
      approved_by: HseChecklists129.document.approved_by,
      title: HseChecklists129.form.form_type,
      checklist: {
        'FORM_FIELDS': HseChecklists129.form || {},
        'DRILL_INSTRUCTION': HseChecklists129.form || {}
      }
    };

    // Process HSE-072 checklist to match the expected format
    const processedHse072 = {
      document_name: HseChecklists072.document.name,
      document_id: HseChecklists072.document.id,
      version: HseChecklists072.document.version,
      version_date: HseChecklists072.document.versionDate,
      approved_by: HseChecklists072.document.approvedBy,
      title: HseChecklists072.report.title,
      checklist: {
        'FIELDS': HseChecklists072.report.fields || {},
        'INCOMING_SHIFT_REQUIREMENTS': HseChecklists072.report.incomingShiftRequirements || [],
        'SHIFT_CONCERN': HseChecklists072.report.shiftConcern || [],
        'EQUIPMENT_CHECKLIST': HseChecklists072.report.equipmentChecklist || [],
        'STANDARD_PROCEDURES': HseChecklists072.report.standardProcedures || []
      }
    };

    // Process HSE-131 checklist to match the expected format
    const processedHse131 = {
      document_name: HseChecklist131.document.name,
      document_id: HseChecklist131.document.id,
      version: HseChecklist131.document.version,
      version_date: HseChecklist131.document.version_date,
      approved_by: HseChecklist131.document.approved_by,
      title: HseChecklist131.checklist.title,
      checklist: {
        'YEAR': HseChecklist131.year,
        'TABLE_HEADERS': HseChecklist131.checklist.headers || [],
        'INSPECTION_GUIDELINES': HseChecklist131.checklist.inspection_guidelines || {},
        'ROWS': HseChecklist131.checklist.rows || [],
        'CHECKED_BY': HseChecklist131.checklist.checked_by || {}
      }
    };

    // Process HSE-034 checklist to match the expected format
    const processedHse034 = {
      document_name: HseChecklist034.document.name,
      document_id: HseChecklist034.document.id,
      version: HseChecklist034.document.version,
      version_date: HseChecklist034.document.version_date,
      approved_by: HseChecklist034.document.approved_by,
      title: HseChecklist034.document.name,
      checklist: {
        'INITIATOR_TEAM': HseChecklist034.initiator_team || {},
        'SECTIONS': HseChecklist034.sections || [],
      }
    };

    // Process HSE-158 checklist to match the expected format
    const processedHse158 = {
      document_name: HseChecklist158.document.name,
      document_id: HseChecklist158.document.id,
      version: HseChecklist158.document.version,
      version_date: HseChecklist158.document.version_date,
      approved_by: HseChecklist158.document.approved_by,
      title: HseChecklist158.title,
      checklist: {
        'DESCRIPTION': HseChecklist158.description,
        'CREW_DETAILS': HseChecklist158.crew_details || {},
        'SECTIONS': HseChecklist158.sections || [],
        'LOG': HseChecklist158.log || {}
      }
    };

    // Process HSE-051 checklist to match the expected format
    const processedHse051 = {
      document_name: HseChecklist051.document.name,
      document_id: HseChecklist051.document.document_id,
      version: HseChecklist051.document.version,
      version_date: HseChecklist051.document.version_date,
      approved_by: HseChecklist051.document.approved,
      title: HseChecklist051.document.name,
      checklist: {
        'TASK_BEING_ANALYZED': HseChecklist051.task_being_analyzed || {},
        'REQUIRED_PROCEDURES': HseChecklist051.required_procedures || {},
        'JOB_STEPS': HseChecklist051.job_steps || []
      }
    };

    // Process HSE-122 checklist to match the expected format
    const processedHse122 = {
      document_name: HseChecklist122.document.name,
      document_id: HseChecklist122.document.document_id,
      version: HseChecklist122.document.version || '1.0',
      version_date: HseChecklist122.document.version_date,
      approved_by: HseChecklist122.document.approved_by || 'N/A',
      title: HseChecklist122.document.name,
      checklist: {
        'ENGINE': HseChecklist122.checks?.[0]?.items || [],
        'EVERY_2000_HRS': HseChecklist122.checks?.[1]?.items || [],
        'ELECTRICAL': HseChecklist122.checks?.[2]?.items || [],
        'MAST': HseChecklist122.checks?.[3]?.items || [],
        'DRILL_HEAD': HseChecklist122.checks?.[4]?.items || [],
        'HYDRAULICS': HseChecklist122.checks?.[5]?.items || [],
        'BEAN_PUMP': HseChecklist122.checks?.[6]?.items || [],
        'TYRES_AND_HUBS': HseChecklist122.checks?.[7]?.items || []
      }
    };

    // Process HSE-123 checklist to match the expected format
    const processedHse123 = {
      document_name: HseChecklist123.document_name,
      document_id: HseChecklist123.document_id,
      version: HseChecklist123.version,
      version_date: HseChecklist123.version_date,
      approved_by: HseChecklist123.approved_by,
      title: HseChecklist123.document_name,
      checklist: {
        '150000km': HseChecklist123.checklist['150000km'] || [],
        '300000km_or_12_month': HseChecklist123.checklist['300000km_or_12_month'] || [],
        '500000km': HseChecklist123.checklist['500000km'] || [],
        '50000km_or_18_months': HseChecklist123.checklist['50000km_or_18_months'] || [],
        'air_leaks': HseChecklist123.checklist['air_leaks'] || [],
        'ADDITIONAL_CHECKS': HseChecklist123.additional_checks || {}
      }
    };

    // Process HSE-124 checklist to match the expected format
    const processedHse124 = {
      document_name: HseChecklist124.document_name,
      document_id: HseChecklist124.document_id,
      version: HseChecklist124.version,
      version_date: HseChecklist124.version_date,
      approved_by: HseChecklist124.approved_by,
      title: HseChecklist124.document_name,
      checklist: {
        'DONKEY_ENGINE': HseChecklist124.checklist.donkey_engine || [],
        'EVERY_200_HRS': HseChecklist124.checklist.every_200_hrs || [],
        'ELECTRICAL': HseChecklist124.checklist.electrical || [],
        'MARST': HseChecklist124.checklist.marst || [],
        'DRILL_HEAD': HseChecklist124.checklist.drill_head || [],
        'CONTROL_PANEL': HseChecklist124.checklist.control_panel || []
      }
    };

    // Process HSE-125 checklist to match the expected format
    const processedHse125 = {
      document_name: HseChecklist125.document_name,
      document_id: HseChecklist125.document_id,
      version: HseChecklist125.version,
      version_date: HseChecklist125.version_date,
      approved_by: HseChecklist125.approved_by,
      title: HseChecklist125.document_name,
      checklist: {
        '5000KM': HseChecklist125.checklist['5000km'] || [],
        '300000KM_OR_12_MONTH': HseChecklist125.checklist['300000km_or_12_month'] || [],
        '50000KM': HseChecklist125.checklist['50000km'] || [],
        '100000KM': HseChecklist125.checklist['100000km'] || [],
        '5_CLICKS': HseChecklist125.checklist['5_clicks'] || [],
        '50000KM_OR_18_MONTHS': HseChecklist125.checklist['50000km_or_18_months'] || []
      }
    };

    // Process HSE-132 checklist to match the expected format
    const processedHse132 = {
      document_name: HSE_132Json.document_name,
      document_id: HSE_132Json.document_id,
      version: HSE_132Json.version,
      version_date: HSE_132Json.version_date,
      approved_by: HSE_132Json.approved_by,
      title: HSE_132Json.document_name,
      checklist: {
        'GENERAL_INSPECTION_DETAILS': HSE_132Json.sections?.[0]?.items || [],
        'ENVIRONMENTAL_REHABILITATION_CHECKLIST': HSE_132Json.sections?.[1]?.items || [],
        'PERSONNEL_INFORMATION': HSE_132Json.sections?.[2]?.items || []
      }
    };

    // Process HSE-155 checklist to match the expected format
    const processedHse155 = {
      document_name: HSE_155_JSON.document_name,
      document_id: HSE_155_JSON.document_id,
      version: HSE_155_JSON.version,
      version_date: HSE_155_JSON.version_date,
      approved_by: HSE_155_JSON.approved_by,
      title: HSE_155_JSON.checklist_type,
      checklist: {
        'WHEELS_AND_SUSPENSION': HSE_155_JSON.sections?.[0]?.subItems || [],
        'TRANSMISSION_AND_DRIVETRAIN': HSE_155_JSON.sections?.[1]?.subItems || [],
        'REAR_COMPONENTS': HSE_155_JSON.sections?.[2]?.subItems || [],
        'EXHAUST_AND_MOUNTINGS': HSE_155_JSON.sections?.[3]?.subItems || [],
        'DOORS_AND_WINDOWS': HSE_155_JSON.sections?.[4]?.subItems || [],
        'INTERIOR_AND_SAFETY': HSE_155_JSON.sections?.[5]?.subItems || [],
        'ENGINE_AND_ELECTRICAL': HSE_155_JSON.sections?.[6]?.subItems || [],
        'LIGHTING_AND_SIGNALS': HSE_155_JSON.sections?.[7]?.subItems || [],
        'OTHER_COMPONENTS': HSE_155_JSON.sections?.[8]?.subItems || []
      }
    };

    // Process HSE-077 checklist to match the expected format
    const processedHse077 = {
      document_name: HSE_077_JSON.document_name,
      document_id: HSE_077_JSON.document_id,
      version: HSE_077_JSON.version,
      version_date: HSE_077_JSON.version_date,
      approved_by: HSE_077_JSON.approved_by,
      title: HSE_077_JSON.checklist_type,
      checklist: {
        'CHAINS': HSE_077_JSON.sections?.[0]?.subItems || [],
        'HOOKS': HSE_077_JSON.sections?.[1]?.subItems || [],
        'SHACKLES': HSE_077_JSON.sections?.[2]?.subItems || [],
        'FIBRE_ROPE': HSE_077_JSON.sections?.[3]?.subItems || [],
        'STEEL_WIRE_ROPE': HSE_077_JSON.sections?.[4]?.subItems || [],
        'FITTINGS': HSE_077_JSON.sections?.[5]?.subItems || []
      }
    };

    // Process HSE-085 Lift Plan to match the expected format
    const processedHse085 = {
      document_name: HSE_085_JSON.document_name,
      document_id: HSE_085_JSON.document_id,
      version: HSE_085_JSON.version,
      version_date: HSE_085_JSON.version_date,
      approved_by: HSE_085_JSON.approved_by,
      title: HSE_085_JSON.title,
      checklist: {
        'FORM_FIELDS': HSE_085_JSON.form_fields || [],
        'CRANE_INFORMATION': (HSE_085_JSON.sections || []).find(s => s.key === 'crane_information')?.fields || [],
        'LOAD_DATA': (HSE_085_JSON.sections || []).find(s => s.key === 'load_data')?.fields || [],
        'RIGGING_DATA': (HSE_085_JSON.sections || []).find(s => s.key === 'rigging_data')?.fields || [],
        'CHECKLIST': HSE_085_JSON.checklist || [],
        'ATTACHMENTS': HSE_085_JSON.attachments || [],
        'CREW_FIELDS': HSE_085_JSON.crew_fields || []
      }
    };
    
    const allChecklistsData = [
      ...allChecklists,
      ...processedHse101_024,
      processedHse020,
      processedHse071,
      processedHse156,
      processedHse077_078,
      processedHse129,
      processedHse072,
      processedHse131,
      processedHse034,
      processedHse158,
      processedHse051,
      processedHse122,
      processedHse123,
      processedHse124,
      processedHse125,
      processedHse132,
      processedHse155,
      processedHse077,
      processedHse085,
    ];
    
    setChecklists(allChecklistsData);
    setFilteredChecklists(allChecklistsData); // Initialize filtered list
  }, []);

  // Filter checklists based on search query with debouncing
  useEffect(() => {
    // Clear any existing timeout
    if (searchTimeoutRef.current) {
      clearTimeout(searchTimeoutRef.current);
    }

    // Debounce search to prevent rapid updates
    searchTimeoutRef.current = setTimeout(() => {
      try {
        if (!searchQuery || searchQuery.trim() === '') {
          setFilteredChecklists(checklists);
        } else {
          const query = searchQuery.toLowerCase().trim();
          
          const filtered = checklists.filter(checklist => {
            try {
              // Safely check each field
              const documentId = checklist?.document_id ? String(checklist.document_id).toLowerCase() : '';
              const documentName = checklist?.document_name ? String(checklist.document_name).toLowerCase() : '';
              const title = checklist?.title ? String(checklist.title).toLowerCase() : '';
              const version = checklist?.version ? String(checklist.version).toLowerCase() : '';
              const approvedBy = checklist?.approved_by ? String(checklist.approved_by).toLowerCase() : '';
              const versionDate = checklist?.version_date ? String(checklist.version_date).toLowerCase() : '';
              
              // Check if query matches any field
              return (
                documentId.includes(query) ||
                documentName.includes(query) ||
                title.includes(query) ||
                version.includes(query) ||
                approvedBy.includes(query) ||
                versionDate.includes(query)
              );
            } catch (error) {
              console.warn('Error filtering checklist item:', error);
              return false;
            }
          });
          
          setFilteredChecklists(filtered);
        }
      } catch (error) {
        console.error('Error in search filter:', error);
        setFilteredChecklists(checklists);
      }
    }, 300); // 300ms debounce delay

    // Cleanup timeout on unmount
    return () => {
      if (searchTimeoutRef.current) {
        clearTimeout(searchTimeoutRef.current);
      }
    };
  }, [searchQuery, checklists]);

  const handleChecklistSelect = async (checklist) => {
    try {
      console.log('Selected checklist:', checklist);
      console.log('Checklist document_id:', checklist.document_id);
      
      // Save the selected checklist to AsyncStorage
      const savedHse = await AsyncStorage.getItem('hseStep1');
      let hseData = savedHse ? JSON.parse(savedHse) : {};
      hseData.selectedChecklist = checklist;
      await AsyncStorage.setItem('hseStep1', JSON.stringify(hseData));
      
      console.log('Saved checklist to AsyncStorage:', hseData);
      
      // Go back to previous screen (step 1)
      navigation.goBack();
    } catch (error) {
      console.error('Error saving selected checklist:', error);
      // Still go back even if saving fails
      navigation.goBack();
    }
  };

  const handleBackPress = () => {
    navigation.goBack();
  };

  const renderChecklistItem = ({ item }) => (
    <TouchableOpacity 
      style={styles.checklistItem} 
      onPress={() => handleChecklistSelect(item)}
      activeOpacity={0.7}
    >
      <View style={styles.checklistHeader}>
        <View style={styles.checklistTitleContainer}>
          <Text style={styles.checklistTitle}>{item.document_name}</Text>
          <View style={styles.checklistBadge}>
            <Text style={styles.checklistBadgeText}>{item.document_id || 'N/A'}</Text>
          </View>
        </View>
        <MaterialIcons name="chevron-right" size={24} color="#0078D4" />
      </View>
      
      <View style={styles.checklistDetails}>
        <View style={styles.detailRow}>
          <Text style={styles.detailLabel}>Document ID:</Text>
          <Text style={styles.detailValue}>{item.document_id || 'N/A'}</Text>
        </View>
        <View style={styles.detailRow}>
          <Text style={styles.detailLabel}>Version:</Text>
          <Text style={styles.detailValue}>{item.version || 'N/A'}</Text>
        </View>
        <View style={styles.detailRow}>
          <Text style={styles.detailLabel}>Date:</Text>
          <Text style={styles.detailValue}>{item.version_date || 'N/A'}</Text>
        </View>
        <View style={styles.detailRow}>
          <Text style={styles.detailLabel}>Approved by:</Text>
          <Text style={styles.detailValue}>{item.approved_by || 'N/A'}</Text>
        </View>
      </View>

      <View style={styles.checklistSections}>
        <Text style={styles.sectionsTitle}>Sections:</Text>
        <View style={styles.sectionsContainer}>
          {item.checklist && Object.keys(item.checklist).map((sectionKey, index) => {
            if (sectionKey !== 'TOTAL_OBSERVATIONS' && sectionKey !== 'BEHAVIOURAL' && sectionKey !== 'OPERATIONAL' && sectionKey !== 'ACTION_TO_BE_TAKEN') {
              return (
                <View key={index} style={styles.sectionChip}>
                  <Text style={styles.sectionChipText}>{sectionKey}</Text>
                </View>
              );
            }
            return null;
          })}
        </View>
      </View>
    </TouchableOpacity>
  );

  return (
    <View style={styles.container}>
      {/* Header (KPI-style) */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={{ width: 40, paddingLeft: width * 0.02 }}>
            <TouchableOpacity onPress={handleBackPress} style={[styles.backButton, { marginLeft: width * 0.01 }]}>
              <MaterialIcons name="chevron-left" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>

          <Text style={styles.headerTitle}>Select HSE Checklist</Text>

          <View style={{ width: 80, paddingRight: width * 0.02, flexDirection: 'row', justifyContent: 'flex-end' }}>
            <TouchableOpacity onPress={() => { setFilteredChecklists(checklists); }} style={{ padding: 6 }}>
              <MaterialIcons name="refresh" size={22} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity onPress={() => { /* reserved for lock/settings */ }} style={{ padding: 6, marginLeft: 8 }}>
              <MaterialIcons name="lock" size={22} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
        
        {/* Search Section */}
        <View style={styles.searchSection}>
          <View style={styles.searchContainer}>
            <MaterialIcons name="search" size={20} color="#666" style={styles.searchIcon} />
            <TextInput
              style={styles.searchInput}
              placeholder="Search by Document ID, Name, Title..."
              placeholderTextColor="#999"
              value={searchQuery}
              onChangeText={(text) => {
                try {
                  setSearchQuery(text);
                } catch (error) {
                  console.error('Error updating search query:', error);
                }
              }}
              autoCapitalize="none"
              autoCorrect={false}
              returnKeyType="search"
              enablesReturnKeyAutomatically={true}
            />
            {searchQuery.length > 0 && (
              <TouchableOpacity onPress={() => setSearchQuery('')} style={styles.clearButton}>
                <MaterialIcons name="close" size={18} color="#666" />
              </TouchableOpacity>
            )}
          </View>
          {searchQuery.length > 0 && (
            <Text style={styles.searchResultsText}>
              {filteredChecklists.length} result{filteredChecklists.length !== 1 ? 's' : ''} found
            </Text>
          )}
        </View>

        <View style={styles.infoSection}>
          <MaterialIcons name="info" size={24} color="#015185" />
          <Text style={styles.infoText}>
            Select a checklist to proceed with your HSE inspection. Available checklists: {searchQuery.length > 0 ? filteredChecklists.length : checklists.length}
          </Text>
        </View>

        <FlatList
          data={filteredChecklists}
          renderItem={renderChecklistItem}
          keyExtractor={(item, index) => `checklist-${item?.document_id || index}`}
          showsVerticalScrollIndicator={false}
          contentContainerStyle={styles.listContainer}
          ItemSeparatorComponent={() => <View style={styles.separator} />}
          scrollEnabled={false}
          removeClippedSubviews={Platform.OS === 'android'}
          maxToRenderPerBatch={10}
          updateCellsBatchingPeriod={50}
          windowSize={10}
        />
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    paddingTop: Platform.OS === 'android' ? StatusBar.currentHeight : 0,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 0,
    paddingTop: Platform.OS === 'ios' ? width * 0.06 : width * 0.04,
    height: '100%',
  },
  backButton: {
    padding: 8,
  },
  headerTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#fff',
    flex: 1,
    textAlign: 'center',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: 20,
    paddingTop: 20,
  },
  searchSection: {
    marginBottom: 15,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 25,
    paddingHorizontal: 15,
    paddingVertical: 12,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    elevation: 2,
  },
  searchIcon: {
    marginRight: 10,
  },
  searchInput: {
    flex: 1,
    fontSize: 16,
    color: '#333',
    paddingVertical: 0,
  },
  clearButton: {
    padding: 5,
    marginLeft: 10,
  },
  searchResultsText: {
    fontSize: 14,
    color: '#015185',
    textAlign: 'center',
    marginTop: 8,
    fontWeight: '500',
  },
  scrollContent: {
    paddingBottom: 20,
  },
  infoSection: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e3f2fd',
    padding: 15,
    borderRadius: 8,
    marginBottom: 15,
  },
  infoText: {
    marginLeft: 10,
    fontSize: 14,
    color: '#015185',
    flex: 1,
  },
  listContainer: {
    paddingBottom: 20,
  },
  headingContainer: {
    alignItems: 'center',
    marginBottom: 20,
  },
  headingText: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 10,
    marginBottom: 5,
  },
  subheadingText: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
  },
  checklistItem: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    marginBottom: 15,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  checklistHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
    marginBottom: 15,
  },
  checklistTitleContainer: {
    flex: 1,
    marginRight: 10,
  },
  checklistTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
    lineHeight: 22,
  },
  checklistBadge: {
    backgroundColor: '#0078D4',
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
    alignSelf: 'flex-start',
  },
  checklistBadgeText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
  },
  checklistDetails: {
    marginBottom: 15,
  },
  detailRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 5,
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
  },
  detailValue: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
  },
  checklistSections: {
    borderTopWidth: 1,
    borderTopColor: '#f0f0f0',
    paddingTop: 15,
  },
  sectionsTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 10,
  },
  sectionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  sectionChip: {
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 10,
    paddingVertical: 5,
    borderRadius: 15,
    marginRight: 8,
    marginBottom: 8,
  },
  sectionChipText: {
    fontSize: 12,
    color: '#666',
    fontWeight: '500',
  },
  separator: {
    height: 1,
    backgroundColor: '#f0f0f0',
    marginVertical: 10,
  },
});

export default HseSelectChecklsit;